/*
 * winUtil.c --
 *
 *	This module contains WIN32 routines not included in the Tcl/Tk
 *	libraries.  Adapted from bltWinUtil.c v2.4e.
 *
 * Copyright 1998 by Bell Labs Innovations for Lucent Technologies.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that the
 * copyright notice and warranty disclaimer appear in supporting documentation,
 * and that the names of Lucent Technologies any of their entities not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this software,
 * including all implied warranties of merchantability and fitness.  In no
 * event shall Lucent Technologies be liable for any special, indirect or
 * consequential damages or any damages whatsoever resulting from loss of use,
 * data or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or
 * performance of this software.
 *
 */

#include <pkg.h>
#include <X11/Xutil.h>
#include <X11/Xlib.h>
#include <windowsx.h>

#ifndef USHRT_MAX
#define	USHRT_MAX	0xFFFF
#endif /* USHRT_MAX */

typedef struct TkWindow *TkWindow;

#ifdef WIN32
/*
 * The TkWinDrawable is the internal implementation of an X Drawable (either
 * a Window or a Pixmap).  The following constants define the valid Drawable
 * types.
 */

#define TWD_BITMAP	1
#define TWD_WINDOW	2
#define TWD_WINDC	3

typedef struct {
    int type;
    HWND handle;
    TkWindow *winPtr;
} TkWinWindow;

typedef struct {
    int type;
    HBITMAP handle;
    Colormap colormap;
    int depth;
} TkWinBitmap;

typedef struct {
    int type;
    HDC hdc;
} TkWinDC;

typedef union {
    int type;
    TkWinWindow window;
    TkWinBitmap bitmap;
    TkWinDC winDC;
} TkWinDrawable;

/*
 * The TkWinDCState is used to save the state of a device context
 * so that it can be restored later.
 */

typedef struct TkWinDCState {
    HPALETTE palette;
} TkWinDCState;

extern HDC TkWinGetDrawableDC (Display *display, Drawable drawable, 
	TkWinDCState *state);
extern HDC TkWinReleaseDrawableDC (Drawable drawable, HDC dc, TkWinDCState *state);

#endif /*WIN32*/

static int tkpWinRopModes[] = {
    R2_BLACK,			/* GXclear */
    R2_MASKPEN,			/* GXand */
    R2_MASKPENNOT,		/* GXandReverse */
    R2_COPYPEN,			/* GXcopy */
    R2_MASKNOTPEN,		/* GXandInverted */
    R2_NOT,			/* GXnoop */
    R2_XORPEN,			/* GXxor */
    R2_MERGEPEN,		/* GXor */
    R2_NOTMERGEPEN,		/* GXnor */
    R2_NOTXORPEN,		/* GXequiv */
    R2_NOT,			/* GXinvert */
    R2_MERGEPENNOT,		/* GXorReverse */
    R2_NOTCOPYPEN,		/* GXcopyInverted */
    R2_MERGENOTPEN,		/* GXorInverted */
    R2_NOTMASKPEN,		/* GXnand */
    R2_WHITE			/* GXset */
};

static HPEN GCToPen _ANSI_ARGS_((HDC dc, GC gc));

static int
GetPlatformId(void)
{
    static int platformId = 0;

    if (platformId == 0) {
	OSVERSIONINFO opsysInfo;

	opsysInfo.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	if (GetVersionEx(&opsysInfo)) {
	    platformId = opsysInfo.dwPlatformId;
	}
    }
    return platformId;
}

/*
 *----------------------------------------------------------------------
 *
 * GCToPen --
 *
 *	Set up the graphics port from the given GC.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The current port is adjusted.
 *
 *----------------------------------------------------------------------
 */
static HPEN
GCToPen(dc, gc)
    HDC dc;
    GC gc;
{
    DWORD lineAttrs, lineStyle;
    DWORD dashArr[12];
    DWORD *dashPtr;
    int numValues, lineWidth;
    LOGBRUSH lb;
    HPEN pen;

    numValues = 0;
    lineWidth = (gc->line_width < 1) ? 1 : gc->line_width;
    if ((gc->line_style == LineOnOffDash) || 
	(gc->line_style == LineDoubleDash)) {
	unsigned char *valueArr = (unsigned char *)(gc->dash_offset);
	
	numValues = (int)gc->dashes;
	if (valueArr == NULL) {
	    dashArr[0] = numValues;
	    numValues = 1;
	} else {
	    register int i;

	    for (i = 0; i < numValues; i++) {
		dashArr[i] = (DWORD)valueArr[i];
	    }
	}
    }
    switch(numValues) {
    case 0:
	lineStyle = PS_SOLID;
	break;
    case 3:
	lineStyle = PS_DASHDOT;
	break;
    case 4:
	lineStyle = PS_DASHDOTDOT;
	break;
    case 2:
    default:
	/* PS_DASH style dash length is too long. */ 
	lineStyle = PS_DOT;
	break;
    }

    lb.lbStyle = BS_SOLID;
    lb.lbColor = gc->foreground;
    lb.lbHatch = 0;		/* Value is ignored with style is BS_SOLID */
    
    lineAttrs = 0;
    switch (gc->cap_style) {
    case CapNotLast:
    case CapButt:
	lineAttrs |= PS_ENDCAP_FLAT; 
	break;
    case CapRound:
	lineAttrs |= PS_ENDCAP_ROUND; 
	break;
    default:
	lineAttrs |= PS_ENDCAP_SQUARE; 
	break;
    }
    switch (gc->join_style) {
    case JoinMiter: 
	lineAttrs |= PS_JOIN_MITER; 
	break;
    case JoinRound:
	lineAttrs |= PS_JOIN_ROUND; 
	break;
    default:
	lineAttrs |= PS_JOIN_BEVEL; 
	break;
    }
    pen = NULL;
    SetBkMode(dc, TRANSPARENT);
    if ((lineStyle == PS_SOLID) ||
	(GetPlatformId() == VER_PLATFORM_WIN32_NT)) {
	/* 
	 * If the line style is solid or we're running on NT, first
	 * try to use a geometric pen.  
	 */
	if (numValues > 0) {
	    lineStyle = PS_USERSTYLE;
	    dashPtr = dashArr;
	} else {
	    dashPtr = NULL;
	}
	pen = ExtCreatePen(PS_GEOMETRIC | lineAttrs | lineStyle, lineWidth, 
	   &lb, numValues, dashPtr);
    } else {
	/* Windows 95: we'll sacrifice thick lines for dashes. */
	pen = ExtCreatePen(PS_COSMETIC | lineAttrs |lineStyle, 1, &lb, 0, NULL);
    }
    if (pen == NULL) {
	OutputDebugString("Can't create pen: trying solid");
	pen = CreatePen(PS_COSMETIC | PS_SOLID, 1, gc->foreground);
    }
    /* assert(pen != NULL); */
    return pen;
}

/*
 *----------------------------------------------------------------------
 *
 * DrawArc --
 *
 *	This procedure handles the rendering of drawn or filled
 *	arcs and chords.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Renders the requested arcs.
 *
 *----------------------------------------------------------------------
 */
static void
DrawArc(dc, arcMode, arcPtr, pen, brush)
    HDC dc;
    int arcMode;		/* Mode: either ArcChord or ArcPieSlice */
    XArc *arcPtr;
    HPEN pen;
    HBRUSH brush;
{
    int start, extent, clockwise;
    int xstart, ystart, xend, yend;
    double radian_start, radian_end, xr, yr;
    double dx, dy;

    start = arcPtr->angle1, extent = arcPtr->angle2;
    clockwise = (extent < 0);	/* Non-zero if clockwise */

    /*
     * Compute the absolute starting and ending angles in normalized radians.
     * Swap the start and end if drawing clockwise.
     */
    start = start % (64 * 360);
    if (start < 0) {
	start += (64 * 360);
    }
    extent = (start + extent) % (64 * 360);
    if (extent < 0) {
	extent += (64 * 360);
    }
    if (clockwise) {
	int tmp = start;
	start = extent;
	extent = tmp;
    }

#define XAngleToRadians(a) ((double)(a) / 64 * M_PI / 180);
    radian_start = XAngleToRadians(start);
    radian_end = XAngleToRadians(extent);

    /*
     * Now compute points on the radial lines that define the starting and
     * ending angles.  Be sure to take into account that the y-coordinate
     * system is inverted.
     */
    dx = arcPtr->width * 0.5;
    dy = arcPtr->height * 0.5;

    xr = arcPtr->x + dx;
    yr = arcPtr->y + dy;
    xstart = (int)((xr + cos(radian_start) * dx) + 0.5);
    ystart = (int)((yr + sin(-radian_start) * dy) + 0.5);
    xend = (int)((xr + cos(radian_end) * dx) + 0.5);
    yend = (int)((yr + sin(-radian_end) * dy) + 0.5);

    /*
     * Now draw a filled or open figure.  Note that we have to
     * increase the size of the bounding box by one to account for the
     * difference in pixel definitions between X and Windows.
     */

    if (brush == 0) {
	/*
	 * Note that this call will leave a gap of one pixel at the
	 * end of the arc for thin arcs.  We can't use ArcTo because
	 * it's only supported under Windows NT.
	 */
	Arc(dc, arcPtr->x, arcPtr->y, arcPtr->x + arcPtr->width + 1,
	    arcPtr->y + arcPtr->height + 1, xstart, ystart, xend, yend);
	/* FIXME: */
    } else {
	if (arcMode == ArcChord) {
	    Chord(dc, arcPtr->x, arcPtr->y, arcPtr->x + arcPtr->width + 1,
		arcPtr->y + arcPtr->height + 1, xstart, ystart, xend, yend);
	} else if (arcMode == ArcPieSlice) {
	    Pie(dc, arcPtr->x, arcPtr->y, arcPtr->x + arcPtr->width + 1,
		arcPtr->y + arcPtr->height + 1, xstart, ystart, xend, yend);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * XDrawArcs --
 *
 *	Draws multiple circular or elliptical arcs.  Each arc is
 *	specified by a rectangle and two angles.  The center of the
 *	circle or ellipse is the center of the rect- angle, and the
 *	major and minor axes are specified by the width and height.
 *	Positive angles indicate counterclock- wise motion, and
 *	negative angles indicate clockwise motion.  If the magnitude
 *	of angle2 is greater than 360 degrees, XDrawArcs truncates it
 *	to 360 degrees.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws an arc for each array element on the specified drawable.
 *
 *----------------------------------------------------------------------
 */
void
XDrawArcs(display, drawable, gc, arcArr, numArcs)
    Display *display;
    Drawable drawable;
    GC gc;
    XArc *arcArr;
    int numArcs;
{
    HPEN pen, oldPen;
    register int i;
    HDC dc;
    TkWinDCState state;
    register XArc *arcPtr;

    display->request++;
    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    pen = GCToPen(dc, gc);
    oldPen = SelectPen(dc, pen);
    for (arcPtr = arcArr, i = 0; i < numArcs; i++, arcPtr++) {
	DrawArc(dc, gc->arc_mode, arcPtr, pen, 0);
    }
    DeletePen(SelectPen(dc, oldPen));
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

/*
 *----------------------------------------------------------------------
 *
 * XFillArcs --
 *
 *	Draw a filled arc.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws a filled arc for each array element on the specified drawable.
 *
 *----------------------------------------------------------------------
 */
void
XFillArcs(display, drawable, gc, arcArr, numArcs)
    Display *display;
    Drawable drawable;
    GC gc;
    XArc *arcArr;
    int numArcs;
{
    HBRUSH brush, oldBrush;
    HPEN pen, oldPen;
    register int i;
    HDC dc;
    register XArc *arcPtr;
    TkWinDCState state;

    display->request++;
    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    pen = GCToPen(dc, gc);
    oldPen = SelectPen(dc, pen);
    brush = CreateSolidBrush(gc->foreground);
    oldBrush = SelectBrush(dc, brush);
    for (arcPtr = arcArr, i = 0; i < numArcs; i++, arcPtr++) {
	DrawArc(dc, gc->arc_mode, arcPtr, pen, brush);
    }
    DeleteBrush(SelectBrush(dc, oldBrush));
    DeletePen(SelectPen(dc, oldPen));
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

#ifndef XDrawSegments
/*
 * Needed for Bargraph, this func is now part of Tk8.3+
 */

typedef struct DrawInfo {
    HDC dc;
    int count;
    COLORREF color;
} DrawInfo;

static 
void CALLBACK
DrawDot(int x, int y,		/* Coordinates of point */
	LPARAM clientData)	/* Line information */
{
    DrawInfo *infoPtr = (DrawInfo *)clientData;

    infoPtr->count++;
    if (infoPtr->count & 0x1) {
	SetPixelV(infoPtr->dc, x, y, infoPtr->color);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * XDrawSegments --
 *
 *	Draws multiple, unconnected lines. For each segment, draws a
 *	line between (x1, y1) and (x2, y2).  It draws the lines in the
 *	order listed in the array of XSegment structures and does not
 *	perform joining at coincident endpoints.  For any given line,
 *	does not draw a pixel more than once. If lines intersect, the
 *	intersecting pixels are drawn multiple times.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws unconnected line segments on the specified drawable.
 *
 *----------------------------------------------------------------------
 */
void
XDrawSegments(display, drawable, gc, segArr, numSegments)
    Display *display;
    Drawable drawable;
    GC gc;
    XSegment *segArr;
    int numSegments;
{
    register int i;
    HDC dc;
    register XSegment *segPtr;
    TkWinDCState state;

    display->request++;
    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    if ((gc->line_style != LineSolid) && (gc->dashes == 1)) {
	/* Handle dotted lines specially */ 
	DrawInfo drawInfo;

	drawInfo.dc = dc;
	drawInfo.color = gc->foreground;
	for (segPtr = segArr, i = 0; i < numSegments; i++, segPtr++) {
	    drawInfo.count = 0;
	    LineDDA(segPtr->x1, segPtr->y1, segPtr->x2, segPtr->y2, 
		DrawDot, (LPARAM)&drawInfo);
	}
    } else {
	HPEN pen, oldPen;

	pen = GCToPen(dc, gc);
	oldPen = SelectPen(dc, pen);
	for (segPtr = segArr, i = 0; i < numSegments; i++, segPtr++) {
	    MoveToEx(dc, segPtr->x1, segPtr->y1, (LPPOINT)NULL);
	    LineTo(dc, segPtr->x2, segPtr->y2);
	}
	DeletePen(SelectPen(dc, oldPen));
    }
    TkWinReleaseDrawableDC(drawable, dc, &state);
}
#endif
