# -*- tcl -*-
# Generation (Conversion subsystem - Table of Contents)
# ------------------------------------------------------

package require tools
package require optchecker

namespace eval app-gen-toc {}

# ------------------------------------------------------
# Acceptable syntax for cmdline functionality

set ::app-gen-toc::help(cmdline) {

[call [cmd {@appname@}] [method gen-toc] [opt [arg options...]] [arg format] [arg iomap] [arg tocfile]]

Reads the file [arg tocfile] and writes a table of contents in the
output [arg format] to [const stdout]. Assumes that the [arg tocfile]
is written in the [syscmd doctoc] format. The resolution of symbolic
to actual filenames is guided by the [arg iomap]. It is assumed that
the map file is in the same format as generated by the subcommand
[method map].

The conversion can be influenced and manipulated by the options listed
below.

[list_begin definitions]
[lst_item "[option -varstring] [arg varname] [arg string]"]

This option is used to set format specific parameters to some
value. The parameter is specified by [arg varname], the new value by
[arg string].

[nl]

Currently only the HTML engine supports engine parameters.

[lst_item "[option -varfile] [arg varname] [arg file]"]

This option is used to set format specific parameters to some
value. The parameter is specified by [arg varname], the new value by
the contents of the file [arg file].

[nl]

Currently only the HTML engine supports engine parameters.

[lst_item "[option -subst] [arg varname] [arg marker] [arg path]"]

When used multiple times the data accumulates, in contrast to the
other options, where the last occurence is dominant.

[nl]

It declares that the data for the engine parameter [arg varname]
contains a variable link, that all locations for this link are marked
by the string [arg marker], and that the link has to refer to the
output file [arg path]. Based upon this information the formatter
will, for each output file in the [arg iomap], replace the marker with
a proper link to the [arg path], based upon the relative locations of
the current file and the file linked to, i.e. [arg path].

[nl]

An occurence of this option for a variable [arg varname] is ignored if
no engine parameters for [arg varname] were specified via either
[option -varfile] or [option -varstring].

[list_end]
}

proc ::app-gen-toc::help {topic} {
    variable help
    if {[info exists help($topic)]} {return $help($topic)}
    return {}
}

# ------------------------------------------------------
# Implementation of cmdline functionality.

proc ::app-gen-toc::run {argv} {
    set errstring "wrong#args: gen-toc ?-varstring var string? ?-varfile var file? ?-subst var mark link? format iomap tocfile"

    if {[llength $argv] < 3} {tools::usage $errstring}

    array set para {}
    set subst {}

    while {[string match -* [lindex $argv 0]]} {
	if {[string equal [lindex $argv 0] -varstring]} {
	    if {[llength $argv] < 6} {tools::usage $errstring}

	    set varname  [lindex $argv 1]
	    set varvalue [lindex $argv 2]
	    set argv     [lrange $argv 3 end]

	    set para($varname) $varvalue
	    continue
	}
	if {[string equal [lindex $argv 0] -varfile]} {
	    if {[llength $argv] < 6} {tools::usage $errstring}

	    set varname  [lindex $argv 1]
	    ::optcheck::infile [lindex $argv 2] "Parameter content file"
	    set varvalue [tools::getfile [lindex $argv 2]]
	    set argv     [lrange $argv 3 end]

	    set para($varname) $varvalue
	    continue
	}
	if {[string equal [lindex $argv 0] -subst]} {
	    if {[llength $argv] < 6} {tools::usage $errstring}

	    set var  [lindex $argv 1]
	    set mark [lindex $argv 2]
	    set link [lindex $argv 3]
	    set argv [lrange $argv 4 end]

	    lappend subst $var [list $mark $link]
	    continue
	}
	tools::usage $errstring
    }

    if {[llength $argv] != 3} {tools::usage $errstring}

    optcheck::infile [set mapfile [lindex $argv 1]] "Mapping file"
    optcheck::infile [set tocfile [lindex $argv 2]] "TOC file"

    set format  [lindex $argv 0]
    set tocdata [tools::getfile $tocfile]
    set iomap   [tools::readmap $mapfile]

    # Now we modify the iomap so that the actual file names resolve to
    # the filename relative to the generated file, which we find by
    # looking for the virtual input file _toc_. If no such file is
    # present we do not modify the map.

    # Before doing the above we also integrate the substitution
    # information into the parameters, if they exist.

    array set _ $iomap
    if {[info exists _(_toc_)]} {
	if {[llength $subst] > 0} {
	    array set __ {}
	    foreach {var val} $subst {
		foreach {mark link} $val break
		lappend __($var) $mark [set ref [::tools::link $_(_toc_) $link]]
	    }
	    foreach {var map} [array get __] {
		if {[info exists para($var)]} {
		    set para($var) [string map $map $para($var)]
		}
	    }
	}

	foreach key [array names _] {
	    if {[string equal _toc_ $key]} {continue}
	    set _($key) [tools::link $_(_toc_) $_($key)]
	}
	set iomap [array get _]
    }
    unset _

    package require dtglue
    puts stdout [dtglue::cvttoc $format $tocdata $iomap [array get para] $tocfile]
    return
}

# ------------------------------------------------------
package provide app-gen-toc 0.1
