
/*
 * bltSink.c --
 *
 * This module implements a data sink for the BLT toolkit.
 *
 *	Copyright 1991-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "bltInt.h"
#include <bltSink.h>

typedef struct Blt_DataSinkStruct DataSink;

void
Blt_SinkInit(DataSink *sinkPtr)
{
    sinkPtr->cursor = sinkPtr->nFilled = sinkPtr->nBytes = 0;
    sinkPtr->bytes = NULL;
    sinkPtr->chunk = 64;
}

void
Blt_SinkFree(DataSink *sinkPtr)
{
    if ((sinkPtr->bytes != NULL) && (sinkPtr->nBytes > 0)) {
	Blt_Free(sinkPtr->bytes);
    }
    Blt_SinkInit(sinkPtr);
}

int
Blt_SinkSetLength(DataSink *sinkPtr, size_t nBytes)
{
    if (sinkPtr->nBytes <= nBytes) {
	size_t size, wanted;
	unsigned char *bytes;

	wanted = nBytes + 1;
	size = sinkPtr->chunk; 

	/* 
	 * Double the buffer size until we have enough room or hit
	 * 64K.  After 64K, increase by multiples of 64K.
	 */
	while ((size <= wanted) && (size < (1<<16))) {
	    size += size;
	}    
	sinkPtr->chunk = size;
	while (size <= wanted) {
	    size += sinkPtr->chunk;
	}
	if (sinkPtr->bytes == NULL) {
 	    bytes = Blt_Malloc(size);
 	} else {
	    bytes = Blt_Realloc(sinkPtr->bytes, size);
	}
	if (bytes == NULL) {
	    return FALSE;
 	}
	sinkPtr->bytes = bytes;
	sinkPtr->nBytes = size;
    }
    sinkPtr->bytes[sinkPtr->nFilled] = '\0';
    return TRUE;
}

int
Blt_SinkAppendData(
    DataSink *sinkPtr,
    unsigned char *buffer, 	/* Array of bytes to append. */
    int nBytes)			/* # of bytes in above array. */
{
    if (nBytes == -1) {
	nBytes = strlen(buffer);
    }
    if (!Blt_SinkSetLength(sinkPtr, sinkPtr->nFilled + nBytes + 1)) {
	return FALSE;
    }
    memcpy(sinkPtr->bytes + sinkPtr->nFilled, buffer, nBytes);
    sinkPtr->nFilled += nBytes;
    sinkPtr->bytes[sinkPtr->nFilled] = '\0';
    return TRUE;
}

void
Blt_SinkVarAppend
TCL_VARARGS_DEF(DataSink *, arg1)
{
    DataSink *sinkPtr;
    va_list args;

    sinkPtr = TCL_VARARGS_START(DataSink, arg1, args);
    for (;;) {
	char *string;

	string = va_arg(args, char *);
	if (string == NULL) {
	    break;
	}
	Blt_SinkAppendData(sinkPtr, string, -1);
    }
}

void
Blt_SinkPrint
TCL_VARARGS_DEF(DataSink *, arg1)
{
    DataSink *sinkPtr;
    char *fmt;
    char string[BUFSIZ+4];
    int length;
    va_list args;

    sinkPtr = TCL_VARARGS_START(DataSink, arg1, args);
    fmt = va_arg(args, char *);
    length = vsnprintf(string, BUFSIZ, fmt, args);
    if (length > BUFSIZ) {
	strcat(string, "...");
    }
    va_end(args);
    Blt_SinkAppendData(sinkPtr, string, -1);
}

