
/*
 * bltPs.c --
 *
 * This module implements general PostScript conversion routines.
 *
 *	Copyright 1991-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include "bltInt.h"
#include <stdarg.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include "tkIntBorder.h"
#include "bltPicture.h"
#include "bltPsInt.h"
#include "tkDisplay.h"
#include "tkFont.h"

#define PS_MAXPATH	1500	/* Maximum number of components in a PostScript
				 * (level 1) path. */

#define MODE_MONOCHROME	0
#define MODE_GREYSCALE	1
#define MODE_COLOR	2

PostScript *
Blt_GetPostScript(interp, tkwin)
    Tcl_Interp *interp;
    Tk_Window tkwin;
{
    PostScript *psPtr;

    psPtr = Blt_Malloc(sizeof(PostScript));
    assert(psPtr);

    psPtr->xfontVarName = psPtr->xcolorVarName = NULL;
    psPtr->interp = interp;
    psPtr->tkwin = tkwin;
    psPtr->xcolorMode = MODE_COLOR;
    Tcl_DStringInit(&psPtr->dString);
    return psPtr;
}

void
Blt_FreePostScript(PostScript *psPtr)
{
    Tcl_DStringFree(&psPtr->dString);
    Blt_Free(psPtr);
}

char *
Blt_PostScriptValue(PostScript *psPtr)
{
    return Tcl_DStringValue(&psPtr->dString);
}

void
Blt_PostScriptToInterp(Tcl_Interp *interp, PostScript *psPtr)
{
    Tcl_DStringResult(interp, &psPtr->dString);
}

char *
Blt_PostScriptScratchBuffer(PostScript *psPtr)
{
    return psPtr->scratchArr;
}

Tcl_Interp *
Blt_PostScriptInterp(PostScript *psPtr)
{
    return psPtr->interp;
}

void
Blt_AppendToPostScript
TCL_VARARGS_DEF(PostScript *, arg1)
{
    va_list argList;
    PostScript *psPtr;
    char *string;

    psPtr = TCL_VARARGS_START(PostScript, arg1, argList);
    for (;;) {
	string = va_arg(argList, char *);
	if (string == NULL) {
	    break;
	}
	Tcl_DStringAppend(&(psPtr->dString), string, -1);
    }
}

void
Blt_FormatToPostScript
TCL_VARARGS_DEF(PostScript *, arg1)
{
    va_list argList;
    PostScript *psPtr;
    char *fmt;

    psPtr = TCL_VARARGS_START(PostScript, arg1, argList);
    fmt = va_arg(argList, char *);
    vsprintf(psPtr->scratchArr, fmt, argList);
    va_end(argList);
    Tcl_DStringAppend(&(psPtr->dString), psPtr->scratchArr, -1);
}

int
Blt_FileToPostScript(
    Tcl_Interp *interp, 
    Blt_PostScript ps, 
    char *fileName)
{
    Tcl_Channel channel;
    Tcl_DString dString;
    char *buf;
    char *libDir;
    int nBytes;

    buf = Blt_PostScriptScratchBuffer(ps);

    /*
     * Read in a standard prolog file from file and append it to the
     * PostScript output stored in the Tcl_DString in psPtr.
     */

    libDir = (char *)Tcl_GetVar(interp, "blt_library", TCL_GLOBAL_ONLY);
    if (libDir == NULL) {
	Tcl_AppendResult(interp, "couldn't find BLT script library:",
	    "global variable \"blt_library\" doesn't exist", (char *)NULL);
	return TCL_ERROR;
    }
    Tcl_DStringInit(&dString);
    Tcl_DStringAppend(&dString, libDir, -1);
    Tcl_DStringAppend(&dString, "/", -1);
    Tcl_DStringAppend(&dString, fileName, -1);
    fileName = Tcl_DStringValue(&dString);
    Blt_AppendToPostScript(ps, "\n% including file \"", fileName, "\"\n\n", 
	(char *)NULL);
    channel = Tcl_OpenFileChannel(interp, fileName, "r", 0);
    if (channel == NULL) {
	Tcl_AppendResult(interp, "couldn't open prologue file \"", fileName,
		 "\": ", Tcl_PosixError(interp), (char *)NULL);
	return TCL_ERROR;
    }
    for(;;) {
	nBytes = Tcl_Read(channel, buf, POSTSCRIPT_BUFSIZ);
	if (nBytes < 0) {
	    Tcl_AppendResult(interp, "error reading prologue file \"", 
		     fileName, "\": ", Tcl_PosixError(interp), 
		     (char *)NULL);
	    Tcl_Close(interp, channel);
	    Tcl_DStringFree(&dString);
	    return TCL_ERROR;
	}
	if (nBytes == 0) {
	    break;
	}
	buf[nBytes] = '\0';
	Blt_AppendToPostScript(ps, buf, (char *)NULL);
    }
    Tcl_DStringFree(&dString);
    Tcl_Close(interp, channel);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * XColorToPostScript --
 *
 *	Convert the a XColor (from its RGB values) to a PostScript
 *	command.  If a Tk color map variable exists, it will be
 *	consulted for a PostScript translation based upon the color
 *	name.
 *
 *	Maps an X color intensity (0 to 2^16-1) to a floating point
 *      value [0..1].  Many versions of Tk don't properly handle the 
 *	the lower 8 bits of the color intensity, so we can only 
 *	consider the upper 8 bits. 
 *
 * Results:
 *	The string representing the color mode is returned.
 *
 *----------------------------------------------------------------------
 */
static void
XColorToPostScript(
    Blt_PostScript ps,
    XColor *colorPtr)		/* Color value to be converted */
{
    /* 
     * Shift off the lower byte before dividing because some versions
     * of Tk don't fill the lower byte correctly.  
     */
    Blt_FormatToPostScript(ps, "%g %g %g",
	((double)(colorPtr->red >> 8) / 255.0),
	((double)(colorPtr->green >> 8) / 255.0),
	((double)(colorPtr->blue >> 8) / 255.0));
}

void
Blt_BackgroundToPostScript(PostScript *psPtr, XColor *colorPtr)
{
    /* If the color name exists in Tcl array variable, use that translation */
    if (psPtr->xcolorVarName != NULL) {
	CONST char *psColor;

	psColor = Tcl_GetVar2(psPtr->interp, psPtr->xcolorVarName,
	    Tk_NameOfColor(colorPtr), 0);
	if (psColor != NULL) {
	    Blt_AppendToPostScript(psPtr, " ", psColor, "\n", (char *)NULL);
	    return;
	}
    }
    XColorToPostScript(psPtr, colorPtr);
    Blt_AppendToPostScript(psPtr, " SetBgColor\n", (char *)NULL);
}

void
Blt_ForegroundToPostScript(PostScript *psPtr, XColor *colorPtr)
{
    /* If the color name exists in Tcl array variable, use that translation */
    if (psPtr->xcolorVarName != NULL) {
	CONST char *psColor;

	psColor = Tcl_GetVar2(psPtr->interp, psPtr->xcolorVarName,
	    Tk_NameOfColor(colorPtr), 0);
	if (psColor != NULL) {
	    Blt_AppendToPostScript(psPtr, " ", psColor, "\n", (char *)NULL);
	    return;
	}
    }
    XColorToPostScript(psPtr, colorPtr);
    Blt_AppendToPostScript(psPtr, " SetFgColor\n", (char *)NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * ReverseBits --
 *
 *	Convert a byte from a X image into PostScript image order.
 *	This requires not only the nybbles to be reversed but also
 *	their bit values.
 *
 * Results:
 *	The converted byte is returned.
 *
 *----------------------------------------------------------------------
 */
INLINE static unsigned char
ReverseBits(unsigned char byte)
{
    byte = ((byte >> 1) & 0x55) | ((byte << 1) & 0xaa);
    byte = ((byte >> 2) & 0x33) | ((byte << 2) & 0xcc);
    byte = ((byte >> 4) & 0x0f) | ((byte << 4) & 0xf0);
    return byte;
}

/*
 *----------------------------------------------------------------------
 *
 * ByteToHex --
 *
 *	Convert a byte to its ASCII hexidecimal equivalent.
 *
 * Results:
 *	The converted 2 ASCII character string is returned.
 *
 *----------------------------------------------------------------------
 */
INLINE static void
ByteToHex(unsigned char byte, char *string)
{
    static char hexDigits[] = "0123456789ABCDEF";

    string[0] = hexDigits[byte >> 4];
    string[1] = hexDigits[byte & 0x0F];
}

#ifdef WIN32
/*
 * -------------------------------------------------------------------------
 *
 * Blt_BitmapDataToPostScript --
 *
 *      Output a PostScript image string of the given bitmap image.
 *      It is assumed the image is one bit deep and a zero value
 *      indicates an off-pixel.  To convert to PostScript, the bits
 *      need to be reversed from the X11 image order.
 *
 * Results:
 *      None.
 *
 * Side Effects:
 *      The PostScript image string is appended.
 *
 * -------------------------------------------------------------------------
 */
void
Blt_BitmapDataToPostScript(
    PostScript *psPtr,
    Display *display,
    Pixmap bitmap,
    int width, int height)
{
    unsigned char byte;
    int x, y, bitPos;
    unsigned long pixel;
    int byteCount;
    char string[10];
    unsigned char *srcBits, *srcPtr;
    int bytesPerRow;

    srcBits = Blt_GetBitmapData(display, bitmap, width, height, &bytesPerRow);
    if (srcBits == NULL) {
        OutputDebugString("Can't get bitmap data");
	return;
    }
    Blt_AppendToPostScript(psPtr, "\t<", (char *)NULL);
    byteCount = bitPos = 0;	/* Suppress compiler warning */
    for (y = height - 1; y >= 0; y--) {
	srcPtr = srcBits + (bytesPerRow * y);
	byte = 0;
	for (x = 0; x < width; x++) {
	    bitPos = x % 8;
	    pixel = (*srcPtr & (0x80 >> bitPos));
	    if (pixel) {
		byte |= (unsigned char)(1 << bitPos);
	    }
	    if (bitPos == 7) {
		byte = ReverseBits(byte);
		ByteToHex(byte, string);
		string[2] = '\0';
		byteCount++;
		srcPtr++;
		byte = 0;
		if (byteCount >= 30) {
		    string[2] = '\n';
		    string[3] = '\t';
		    string[4] = '\0';
		    byteCount = 0;
		}
		Blt_AppendToPostScript(psPtr, string, (char *)NULL);
	    }
	}			/* x */
	if (bitPos != 7) {
	    byte = ReverseBits(byte);
	    ByteToHex(byte, string);
	    string[2] = '\0';
	    Blt_AppendToPostScript(psPtr, string, (char *)NULL);
	    byteCount++;
	}
    }				/* y */
    Blt_Free(srcBits);
    Blt_AppendToPostScript(psPtr, ">\n", (char *)NULL);
}

#else

/*
 * -------------------------------------------------------------------------
 *
 * Blt_BitmapDataToPostScript --
 *
 *      Output a PostScript image string of the given bitmap image.
 *      It is assumed the image is one bit deep and a zero value
 *      indicates an off-pixel.  To convert to PostScript, the bits
 *      need to be reversed from the X11 image order.
 *
 * Results:
 *      None.
 *
 * Side Effects:
 *      The PostScript image string is appended to interp->result.
 *
 * -------------------------------------------------------------------------
 */
void
Blt_BitmapDataToPostScript(
    Blt_PostScript ps,
    Display *display,
    Pixmap bitmap,
    int width, int height)
{
    XImage *imagePtr;
    int byteCount;
    int y, bitPos;

    imagePtr = XGetImage(display, bitmap, 0, 0, width, height, 1, ZPixmap);
    Blt_AppendToPostScript(ps, "\t<", (char *)NULL);
    byteCount = bitPos = 0;	/* Suppress compiler warning */
    for (y = 0; y < height; y++) {
	unsigned char byte;
	char string[10];
	int x;

	byte = 0;
	for (x = 0; x < width; x++) {
	    unsigned long pixel;

	    pixel = XGetPixel(imagePtr, x, y);
	    bitPos = x % 8;
	    byte |= (unsigned char)(pixel << bitPos);
	    if (bitPos == 7) {
		byte = ReverseBits(byte);
		ByteToHex(byte, string);
		string[2] = '\0';
		byteCount++;
		byte = 0;
		if (byteCount >= 30) {
		    string[2] = '\n';
		    string[3] = '\t';
		    string[4] = '\0';
		    byteCount = 0;
		}
		Blt_AppendToPostScript(ps, string, (char *)NULL);
	    }
	}			/* x */
	if (bitPos != 7) {
	    byte = ReverseBits(byte);
	    ByteToHex(byte, string);
	    string[2] = '\0';
	    Blt_AppendToPostScript(ps, string, (char *)NULL);
	    byteCount++;
	}
    }				/* y */
    Blt_AppendToPostScript(ps, ">\n", (char *)NULL);
    XDestroyImage(imagePtr);
}

#endif /* WIN32 */

#ifdef notdef
/*
 *----------------------------------------------------------------------
 *
 * NameOfAtom --
 *
 *	Wrapper routine for Tk_GetAtomName.  Returns NULL instead of
 *	"?bad atom?" if the atom can't be found.
 *
 * Results:
 *	The name of the atom is returned if found. Otherwise NULL.
 *
 *----------------------------------------------------------------------
 */
static char *
NameOfAtom(tkwin, atom)
    Tk_Window tkwin;
    Atom atom;
{
    char *result;

    result = Tk_GetAtomName(tkwin, atom);
    if ((result[0] == '?') && (strcmp(result, "?bad atom?") == 0)) {
	return NULL;
    }
    return result;
}
#endif

typedef struct {
    char *alias;
    char *fontName;
} FontMap;

static FontMap psFontMap[] =
{
    {"Arial", "Helvetica",},
    {"AvantGarde", "AvantGarde",},
    {"Courier New", "Courier",},
    {"Courier", "Courier",},
    {"Geneva", "Helvetica",},
    {"Helvetica", "Helvetica",},
    {"Monaco", "Courier",},
    {"New Century Schoolbook", "NewCenturySchlbk",},
    {"New York", "Times",},
    {"Palatino", "Palatino",},
    {"Symbol", "Symbol",},
    {"Times New Roman", "Times",},
    {"Times Roman", "Times",},
    {"Times", "Times",},
    {"Utopia", "Utopia",},
    {"ZapfChancery", "ZapfChancery",},
    {"ZapfDingbats", "ZapfDingbats",},
};

static int nFontNames = (sizeof(psFontMap) / sizeof(FontMap));

#ifdef notdef
#ifndef  WIN32
/*
 * -----------------------------------------------------------------
 *
 * XFontStructToPostScript --
 *
 *      Map X11 font to a PostScript font. Currently, only fonts whose
 *      FOUNDRY property are "Adobe" are converted. Simply gets the
 *      XA_FULL_NAME and XA_FAMILY properties and pieces together a
 *      PostScript fontname.
 *
 * Results:
 *      Returns the mapped PostScript font name if one is possible.
 *	Otherwise returns NULL.
 *
 * -----------------------------------------------------------------
 */
static char *
XFontStructToPostScript(
    Tk_Window tkwin,		/* Window to query for atoms */
    XFontStruct *fontPtr)	/* Font structure to map to name */
{
    Atom atom;
    char *fullName, *family, *foundry;
    char *src, *dest;
    size_t familyLen;
    char *start;
    static char string[200];	/* What size? */

    if (XGetFontProperty(fontPtr, XA_FULL_NAME, &atom) == False) {
	return NULL;
    }
    fullName = NameOfAtom(tkwin, atom);
    if (fullName == NULL) {
	return NULL;
    }
    family = foundry = NULL;
    if (XGetFontProperty(fontPtr, Tk_InternAtom(tkwin, "FOUNDRY"), &atom)) {
	foundry = NameOfAtom(tkwin, atom);
    }
    if (XGetFontProperty(fontPtr, XA_FAMILY_NAME, &atom)) {
	family = NameOfAtom(tkwin, atom);
    }
    /*
     * Try to map the font only if the foundry is Adobe
     */
    if ((foundry == NULL) || (family == NULL)) {
	return NULL;
    }
    src = NULL;
    familyLen = strlen(family);
    if (strncasecmp(fullName, family, familyLen) == 0) {
	src = fullName + familyLen;
    }
    if (strcmp(foundry, "Adobe") != 0) {
	FontMap *fp, *fend;

	if (strncasecmp(family, "itc ", 4) == 0) {
	    family += 4;	/* Throw out the "itc" prefix */
	}
	for (fp = psFontMap, fend = fp + nFontNames; fp < fend; fp++) {
	    if (strcasecmp(family, fp->alias) == 0) {
		family = fp->fontName;
	    }
	}
	if (fp == fend) {
	    family = "Helvetica";	/* Default to a known font */
	}
    }
    /*
     * PostScript font name is in the form <family>-<type face>
     */
    sprintf(string, "%s-", family);
    dest = start = string + strlen(string);

    /*
     * Append the type face (part of the full name trailing the family name)
     * to the the PostScript font name, removing any spaces or dashes
     *
     * ex. " Bold Italic" ==> "BoldItalic"
     */
    if (src != NULL) {
	while (*src != '\0') {
	    if ((*src != ' ') && (*src != '-')) {
		*dest++ = *src;
	    }
	    src++;
	}
    }
    if (dest == start) {
	--dest;			/* Remove '-' to leave just the family name */
    }
    *dest = '\0';		/* Make a valid string */
    return string;
}
#endif /* !WIN32 */
#endif

/*
 * -------------------------------------------------------------------
 * Routines to convert X drawing functions to PostScript commands.
 * -------------------------------------------------------------------
 */
void
Blt_ClearBackgroundToPostScript(Blt_PostScript ps)
{
    Blt_AppendToPostScript(ps, " 1.0 1.0 1.0 SetBgColor\n", (char *)NULL);
}

void
Blt_CapStyleToPostScript(
    Blt_PostScript ps, 
    int capStyle)
{
    /*
     * X11:not last = 0, butt = 1, round = 2, projecting = 3
     * PS: butt = 0, round = 1, projecting = 2
     */
    if (capStyle > 0) {
	capStyle--;
    }
    Blt_FormatToPostScript(ps, "%d setlinecap\n", capStyle);
}

void
Blt_JoinStyleToPostScript(
    Blt_PostScript ps, 
    int joinStyle)
{
    /*
     * miter = 0, round = 1, bevel = 2
     */
    Blt_FormatToPostScript(ps, "%d setlinejoin\n", joinStyle);
}

void
Blt_LineWidthToPostScript(
    Blt_PostScript ps, 
    int lineWidth)
{
    if (lineWidth < 1) {
	lineWidth = 1;
    }
    Blt_FormatToPostScript(ps, "%d setlinewidth\n", lineWidth);
}

void
Blt_LineDashesToPostScript(
    Blt_PostScript ps, 
    Blt_Dashes *dashesPtr)
{

    Blt_AppendToPostScript(ps, "[ ", (char *)NULL);
    if (dashesPtr != NULL) {
	unsigned char *vp;

	for (vp = dashesPtr->values; *vp != 0; vp++) {
	    Blt_FormatToPostScript(ps, " %d", *vp);
	}
    }
    Blt_AppendToPostScript(ps, "] 0 setdash\n", (char *)NULL);
}

void
Blt_LineAttributesToPostScript(
    Blt_PostScript ps,
    XColor *colorPtr,
    int lineWidth,
    Blt_Dashes *dashesPtr,
    int capStyle, 
    int joinStyle)
{
    Blt_JoinStyleToPostScript(ps, joinStyle);
    Blt_CapStyleToPostScript(ps, capStyle);
    Blt_ForegroundToPostScript(ps, colorPtr);
    Blt_LineWidthToPostScript(ps, lineWidth);
    Blt_LineDashesToPostScript(ps, dashesPtr);
    Blt_AppendToPostScript(ps, "/DashesProc {} def\n", (char *)NULL);
}

void
Blt_RectangleToPostScript(
    Blt_PostScript ps,
    double x, double y,
    int width, int height)
{
    Blt_FormatToPostScript(ps, "%g %g %d %d Box fill\n\n", x, y, width, height);
}

void
Blt_RegionToPostScript(
    Blt_PostScript ps,
    double x, double y,
    int width, int height)
{
    Blt_FormatToPostScript(ps, "%g %g %d %d Box\n\n", x, y, width, height);
}

void
Blt_PathToPostScript(
    Blt_PostScript ps,
    Point2D *screenPts,
    int nScreenPts)
{
    Point2D *pp, *pend;

    pp = screenPts;
    Blt_FormatToPostScript(ps, "newpath %g %g moveto\n", pp->x, pp->y);
    for (pp++, pend = screenPts + nScreenPts; pp < pend; pp++) {
	Blt_FormatToPostScript(ps, "%g %g lineto\n", pp->x, pp->y);
    }
}

void
Blt_PolygonToPostScript(
    Blt_PostScript ps,
    Point2D *screenPts,
    int nScreenPts)
{
    Blt_PathToPostScript(ps, screenPts, nScreenPts);
    Blt_FormatToPostScript(ps, "%g %g ", screenPts[0].x, screenPts[0].y);
    Blt_AppendToPostScript(ps, " lineto closepath Fill\n", (char *)NULL);
}

void
Blt_SegmentsToPostScript(
    Blt_PostScript ps,
    XSegment *segments,
    int nSegments)
{
    XSegment *sp, *send;

    for (sp = segments, send = sp + nSegments; sp < send; sp++) {
	Blt_FormatToPostScript(ps, "%d %d moveto\n", sp->x1, sp->y1);
	Blt_FormatToPostScript(ps, " %d %d lineto\n", sp->x2, sp->y2);
	Blt_AppendToPostScript(ps, "DashesProc stroke\n", (char *)NULL);
    }
}


void
Blt_RectanglesToPostScript(
    Blt_PostScript ps, 
    XRectangle *rectangles, 
    int nRectangles)
{
    XRectangle *rp, *rend;

    for (rp = rectangles, rend = rp + nRectangles; rp < rend; rp++) {
	Blt_RectangleToPostScript(ps, (double)rp->x, (double)rp->y, 
		(int)rp->width, (int)rp->height);
    }
}

#ifndef TK_RELIEF_SOLID
#define TK_RELIEF_SOLID		-1	/* Set the an impossible value. */
#endif /* TK_RELIEF_SOLID */

void
Blt_Draw3DRectangleToPostScript(
    Blt_PostScript ps,
    Tk_3DBorder border,		/* Token for border to draw. */
    double x, double y,		/* Coordinates of rectangle */
    int width, int height,	/* Region to be drawn. */
    int borderWidth,		/* Desired width for border, in pixels. */
    int relief)			/* Should be either TK_RELIEF_RAISED or
                                 * TK_RELIEF_SUNKEN;  indicates position of
                                 * interior of window relative to exterior. */
{
    Point2D points[7];
    TkBorder *borderPtr = (TkBorder *) border;
    XColor *lightPtr, *darkPtr;
    XColor *topPtr, *bottomPtr;
    XColor light, dark;
    int twiceWidth = (borderWidth * 2);

    if ((width < twiceWidth) || (height < twiceWidth)) {
	return;
    }
    if ((relief == TK_RELIEF_SOLID) ||
	(borderPtr->lightColor == NULL) || (borderPtr->darkColor == NULL)) {
	if (relief == TK_RELIEF_SOLID) {
	    dark.red = dark.blue = dark.green = 0x00;
	    light.red = light.blue = light.green = 0x00;
	    relief = TK_RELIEF_SUNKEN;
	} else {
	    Screen *screenPtr;

	    light = *borderPtr->bgColor;
	    screenPtr = Tk_Screen(ps->tkwin);
	    if (light.pixel == WhitePixelOfScreen(screenPtr)) {
		dark.red = dark.blue = dark.green = 0x00;
	    } else {
		dark.red = dark.blue = dark.green = 0xFF;
	    }
	}
	lightPtr = &light;
	darkPtr = &dark;
    } else {
	lightPtr = borderPtr->lightColor;
	darkPtr = borderPtr->darkColor;
    }


    /*
     * Handle grooves and ridges with recursive calls.
     */

    if ((relief == TK_RELIEF_GROOVE) || (relief == TK_RELIEF_RIDGE)) {
	int halfWidth, insideOffset;

	halfWidth = borderWidth / 2;
	insideOffset = borderWidth - halfWidth;
	Blt_Draw3DRectangleToPostScript(ps, border, (double)x, (double)y,
	    width, height, halfWidth, 
	    (relief == TK_RELIEF_GROOVE) ? TK_RELIEF_SUNKEN : TK_RELIEF_RAISED);
	Blt_Draw3DRectangleToPostScript(ps, border, 
  	    (double)(x + insideOffset), (double)(y + insideOffset), 
	    width - insideOffset * 2, height - insideOffset * 2, halfWidth,
	    (relief == TK_RELIEF_GROOVE) ? TK_RELIEF_RAISED : TK_RELIEF_SUNKEN);
	return;
    }
    if (relief == TK_RELIEF_RAISED) {
	topPtr = lightPtr;
	bottomPtr = darkPtr;
    } else if (relief == TK_RELIEF_SUNKEN) {
	topPtr = darkPtr;
	bottomPtr = lightPtr;
    } else {
	topPtr = bottomPtr = borderPtr->bgColor;
    }
    Blt_BackgroundToPostScript(ps, bottomPtr);
    Blt_RectangleToPostScript(ps, x, y + height - borderWidth, width, 
	borderWidth);
    Blt_RectangleToPostScript(ps, x + width - borderWidth, y, borderWidth, 
	height);
    points[0].x = points[1].x = points[6].x = x;
    points[0].y = points[6].y = y + height;
    points[1].y = points[2].y = y;
    points[2].x = x + width;
    points[3].x = x + width - borderWidth;
    points[3].y = points[4].y = y + borderWidth;
    points[4].x = points[5].x = x + borderWidth;
    points[5].y = y + height - borderWidth;
    if (relief != TK_RELIEF_FLAT) {
	Blt_BackgroundToPostScript(ps, topPtr);
    }
    Blt_PolygonToPostScript(ps, points, 7);
}

void
Blt_Fill3DRectangleToPostScript(
    Blt_PostScript ps,
    Tk_3DBorder border,		/* Token for border to draw. */
    double x, double y,		/* Coordinates of top-left of border area */
    int width, int height,	/* Dimension of border to be drawn. */
    int borderWidth,		/* Desired width for border, in pixels. */
    int relief)			/* Should be either TK_RELIEF_RAISED or
                                 * TK_RELIEF_SUNKEN;  indicates position of
                                 * interior of window relative to exterior. */
{
    TkBorder *borderPtr = (TkBorder *) border;

    /*
     * I'm assuming that the rectangle is to be drawn as a background.
     * Setting the pen color as foreground or background only affects
     * the plot when the colormode option is "monochrome".
     */
    Blt_BackgroundToPostScript(ps, borderPtr->bgColor);
    Blt_RectangleToPostScript(ps, x, y, width, height);
    Blt_Draw3DRectangleToPostScript(ps, border, x, y, width, height, 
	borderWidth, relief);
}

void
Blt_StippleToPostScript(
    Blt_PostScript ps, 
    Display *display, 
    Pixmap bitmap)
{
    int width, height;

    Tk_SizeOfBitmap(display, bitmap, &width, &height);
    Blt_FormatToPostScript(ps, "gsave\n  clip\n  %d %d\n", width, height);
    Blt_BitmapDataToPostScript(ps, display, bitmap, width, height);
    Blt_AppendToPostScript(ps, "  StippleFill\ngrestore\n", (char *)NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_PictureToPostScript --
 *
 *      Translates a picture into 3 component RGB PostScript output.
 *	Uses PS Language Level 2 operator "colorimage".
 *
 * Results:
 *      The dynamic string will contain the PostScript output.
 *
 *----------------------------------------------------------------------
 */
void
Blt_PictureToPostScript(
    PostScript *psPtr, 
    Blt_Picture pict,
    double x, double y)
{
    int width, height;
    int tmpSize;

    width = Blt_PictureWidth(pict);
    height = Blt_PictureHeight(pict);

    tmpSize = width;
    if (psPtr->xcolorMode == MODE_COLOR) {
	tmpSize *= 3;
    }
    Blt_FormatToPostScript(psPtr, "\n/tmpStr %d string def\n", tmpSize);
    Blt_AppendToPostScript(psPtr, "gsave\n", (char *)NULL);
    Blt_FormatToPostScript(psPtr, "  %g %g translate\n", x, y);
    Blt_FormatToPostScript(psPtr, "  %d %d scale\n", width, height);
    Blt_FormatToPostScript(psPtr, "  %d %d 8\n", width, height);
    Blt_FormatToPostScript(psPtr, "  [%d 0 0 %d 0 %d] ", width, -height,
	height);
    Blt_AppendToPostScript(psPtr, 
	"{\n    currentfile tmpStr readhexstring pop\n  } ",
	(char *)NULL);
    if (psPtr->xcolorMode != MODE_COLOR) {
	Blt_Picture greyscale;

	Blt_AppendToPostScript(psPtr, "image\n", (char *)NULL);
	greyscale = Blt_GreyscalePicture(pict);
	Blt_PictureToPsData(greyscale, 1, &(psPtr->dString), " ");
	Blt_FreePicture(greyscale);
    } else {
	Blt_AppendToPostScript(psPtr, 
		"false 3 colorimage\n", 
		(char *)NULL);
	Blt_PictureToPsData(pict, 3, &(psPtr->dString), " ");
    }
    Blt_AppendToPostScript(psPtr, 
	"\ngrestore\n\n", 
	(char *)NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_WindowToPostScript --
 *
 *      Converts a Tk window to PostScript.  If the window could not
 *	be "snapped", then a grey rectangle is drawn in its place.
 *
 * Results:
 *      None.
 *
 *----------------------------------------------------------------------
 */
void
Blt_WindowToPostScript(
    Blt_PostScript ps, 
    Tk_Window tkwin, 
    double x, double y)
{
    Blt_Picture picture;

    picture = Blt_DrawableToPicture(tkwin, Tk_WindowId(tkwin), 0, 0, 
	Tk_Width(tkwin), Tk_Height(tkwin), GAMMA);
    if (picture == NULL) {
	/* Can't grab window image so paint the window area grey */
	Blt_AppendToPostScript(ps, "% Can't grab window \"", Tk_PathName(tkwin),
		"\"\n", (char *)NULL);
	Blt_AppendToPostScript(ps, "0.5 0.5 0.5 SetBgColor\n", (char *)NULL);
	Blt_RectangleToPostScript(ps, x, y, Tk_Width(tkwin), Tk_Height(tkwin));
	return;
    }
    Blt_PictureToPostScript(ps, picture, x, y);
    Blt_FreePicture(picture);
}

/*
 * -------------------------------------------------------------------------
 *
 * Blt_PhotoToPostScript --
 *
 *      Output a PostScript image string of the given photo image.
 *	The photo is first converted into a picture and then
 *	translated into PostScript.
 *
 * Results:
 *      None.
 *
 * Side Effects:
 *      The PostScript output representing the photo is appended to
 *	the psPtr's dynamic string.
 *
 * -------------------------------------------------------------------------
 */
void
Blt_PhotoToPostScript(
    Blt_PostScript ps, 
    Tk_PhotoHandle photo, 
    double x, double y)		/* Origin of photo image */
{
    Blt_Picture picture;

    picture = Blt_PhotoToPicture(photo);
    Blt_PictureToPostScript(ps, picture, x, y);
    Blt_FreePicture(picture);
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_FontToPostScript --
 *
 *      Map the Tk font to a PostScript font and point size.
 *
 *	If a Tcl array variable was specified, each element should be
 *	indexed by the X11 font name and contain a list of 1-2
 *	elements; the PostScript font name and the desired point size.
 *	The point size may be omitted and the X font point size will
 *	be used.
 *
 *	Otherwise, if the foundry is "Adobe", we try to do a plausible
 *	mapping looking at the full name of the font and building a
 *	string in the form of "Family-TypeFace".
 *
 * Returns:
 *      None.
 *
 * Side Effects:
 *      PostScript commands are output to change the type and the
 *      point size of the current font.
 *
 * -----------------------------------------------------------------
 */

void
Blt_FontToPostScript(PostScript *psPtr, Blt_Font font) 
{
    Tcl_Interp *interp = psPtr->interp;

    /*
     * Use the font variable information if it exists.
     */
    if (psPtr->xfontVarName != NULL) {
	char *value;

	value = (char *)Tcl_GetVar2(interp, psPtr->xfontVarName, 
		Blt_NameOfFont(font), 0);
	if (value != NULL) {
	    char **argv = NULL;
	    int argc;
	    int newSize;
	    double pointSize;
	    char *fontName;

	    if (Tcl_SplitList(NULL, value, &argc, &argv) != TCL_OK) {
		return;
	    }

	    fontName = argv[0];
	    if ((argc != 2) || 
		(Tcl_GetInt(interp, argv[1], &newSize) != TCL_OK)) {
		Blt_Free(argv);
		return;
	    }
	    pointSize = (double)newSize;
	    Blt_FormatToPostScript(psPtr, "%g /%s SetFont\n", pointSize, 
		fontName);
	    Blt_Free(argv);
	    return;
	}
	/*FallThru*/
    }

    /*
     * Check to see if it's a PostScript font.  Tk_PostScriptFontName
     * silently generates a bogus PostScript font name, so we have to
     * check to see if this is really a PostScript font first before
     * we call it.
     */
    {
	CONST char *family;
	FontMap *fp, *fend;

	family = Blt_FontFamily(font);
	for (fp = psFontMap, fend = fp + nFontNames; fp < fend; fp++) {
	    if (strncasecmp(fp->alias, family, strlen(fp->alias)) == 0) {
		Tcl_DString dString;
		double pointSize;
		
		Tcl_DStringInit(&dString);
		pointSize = (double)Blt_PostscriptFontName(font, &dString);
		Blt_FormatToPostScript(psPtr, "%g /%s SetFont\n", pointSize,
				       Tcl_DStringValue(&dString));
		Tcl_DStringFree(&dString);
		return;
	    }
	}
    }
#ifdef notdef
#if !defined(WIN32) && !defined(MACOSX)
    {
	XFontStruct *fontPtr = (XFontStruct *)font;

	/* Can you believe what I have to go through to get an XFontStruct? */
	fontPtr = XLoadQueryFont(Tk_Display(psPtr->tkwin), 
				 Blt_NameOfFont(font));
	if (fontPtr != NULL) {
	    unsigned long fontProp;
	    double pointSize;
	    char *fontName;

	    if (XGetFontProperty(fontPtr, XA_POINT_SIZE, &fontProp) != False) {
		pointSize = (double)fontProp / 10.0;
	    }
	    fontName = XFontStructToPostScript(psPtr->tkwin, fontPtr);
	    XFreeFont(Tk_Display(psPtr->tkwin), fontPtr);
	    Blt_FormatToPostScript(psPtr, "%g /%s SetFont\n", pointSize,
		fontName);
	    return;
	}
    }
#endif /* !WIN32 */
#endif
    Blt_AppendToPostScript(psPtr, "12.0 /Helvetica-Bold SetFont\n");
}

static void
TextLayoutToPostScript(
    Blt_PostScript ps, 
    int x, int y, 
    TextLayout *textPtr)
{
    char *bp, *src, *dst, *end;
    int count;			/* Counts the # of bytes written to
				 * the intermediate scratch buffer. */
    TextFragment *fragPtr;
    int i;
    unsigned char c;
#if HAVE_UTF
    Tcl_UniChar ch;
#endif
    int limit;

    limit = POSTSCRIPT_BUFSIZ - 4; /* High water mark for the scratch
				   * buffer. */
    fragPtr = textPtr->fragments;
    for (i = 0; i < textPtr->nFrags; i++, fragPtr++) {
	if (fragPtr->count < 1) {
	    continue;
	}
	Blt_AppendToPostScript(ps, "(", (char *)NULL);
	count = 0;
	dst = Blt_PostScriptScratchBuffer(ps);
	src = fragPtr->text;
	end = fragPtr->text + fragPtr->count;
	while (src < end) {
	    if (count > limit) {
		/* Don't let the scatch buffer overflow */
		dst = Blt_PostScriptScratchBuffer(ps);
		dst[count] = '\0';
		Blt_AppendToPostScript(ps, dst, (char *)NULL);
		count = 0;
	    }
#if HAVE_UTF
	    /*
	     * INTL: For now we just treat the characters as binary
	     * data and display the lower byte.  Eventually this should
	     * be revised to handle international postscript fonts.
	     */
	    src += Tcl_UtfToUniChar(src, &ch);
	    c = (unsigned char)(ch & 0xff);
#else 
	    c = *src++;
#endif

	    if ((c == '\\') || (c == '(') || (c == ')')) {
		/*
		 * If special PostScript characters characters "\", "(",
		 * and ")" are contained in the text string, prepend
		 * backslashes to them.
		 */
		*dst++ = '\\';
		*dst++ = c;
		count += 2;
	    } else if ((c < ' ') || (c > '~')) {
		/* 
		 * Present non-printable characters in their octal
		 * representation.
		 */
		sprintf(dst, "\\%03o", c);
		dst += 4;
		count += 4;
	    } else {
		*dst++ = c;
		count++;
	    }
	}
	bp = Blt_PostScriptScratchBuffer(ps);
	bp[count] = '\0';
	Blt_AppendToPostScript(ps, bp, (char *)NULL);
	Blt_FormatToPostScript(ps, ") %d %d %d DrawAdjText\n",
	    fragPtr->width, x + fragPtr->x, y + fragPtr->y);
    }
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_TextToPostScript --
 *
 *      Output PostScript commands to print a text string. The string
 *      may be rotated at any arbitrary angle, and placed according
 *      the anchor type given. The anchor indicates how to interpret
 *      the window coordinates as an anchor for the text bounding box.
 *
 * Results:
 *      None.
 *
 * Side Effects:
 *      Text string is drawn using the given font and GC on the graph
 *      window at the given coordinates, anchor, and rotation
 *
 * -----------------------------------------------------------------
 */
void
Blt_TextToPostScript(
    Blt_PostScript ps,
    char *string,		/* String to convert to PostScript */
    TextStyle *tsPtr,		/* Text attribute information */
    double x, double y)		/* Window coordinates where to print text */
{
    TextLayout *textPtr;
    Point2D t;

    if ((string == NULL) || (*string == '\0')) { /* Empty string, do nothing */
	return;
    }
    textPtr = Blt_TextCreateLayout(string, tsPtr);
    {
	double angle;
	double rw, rh;
	
	angle = FMOD(tsPtr->angle, (double)360.0);
	Blt_GetBoundingBox(textPtr->width, textPtr->height, angle, &rw, &rh, 
	(Point2D *)NULL);
	/*
	 * Find the center of the bounding box
	 */
	t = Blt_AnchorPoint(x, y, rw, rh, tsPtr->anchor); 
	t.x += rw * 0.5;
	t.y += rh * 0.5;
    }

    /* Initialize text (sets translation and rotation) */
    Blt_FormatToPostScript(ps, "%d %d %g %g %g BeginText\n", textPtr->width, 
	textPtr->height, tsPtr->angle, t.x, t.y);

    Blt_FontToPostScript(ps, tsPtr->font);

    Blt_ForegroundToPostScript(ps, tsPtr->color);
    TextLayoutToPostScript(ps, 0, 0, textPtr);
    Blt_Free(textPtr);
    Blt_AppendToPostScript(ps, "EndText\n", (char *)NULL);
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_LineToPostScript --
 *
 *      Outputs PostScript commands to print a multi-segmented line.
 *      It assumes a procedure DashesProc was previously defined.
 *
 * Results:
 *      None.
 *
 * Side Effects:
 *      Segmented line is printed.
 *
 * -----------------------------------------------------------------
 */
void
Blt_LineToPostScript(
    Blt_PostScript ps, 
    XPoint *points, 
    int nPoints)
{
    int i;
    XPoint *pp;

    if (nPoints <= 0) {
	return;
    }
    pp = points;
    Blt_FormatToPostScript(ps, " newpath %d %d moveto\n", pp->x, pp->y);
    pp++;
    for (i = 1; i < (nPoints - 1); i++, pp++) {
	Blt_FormatToPostScript(ps, " %d %d lineto\n", pp->x, pp->y);
	if ((i % PS_MAXPATH) == 0) {
	    Blt_FormatToPostScript(ps,
		"DashesProc stroke\n newpath  %d %d moveto\n", pp->x, pp->y);
	}
    }
    Blt_FormatToPostScript(ps, " %d %d lineto\n", pp->x, pp->y);
    Blt_AppendToPostScript(ps, "DashesProc stroke\n", (char *)NULL);
}

void
Blt_BitmapToPostScript(
    Blt_PostScript ps,
    Display *display,
    Pixmap bitmap,		/* Bitmap to be converted to PostScript */
    double xScale, double yScale)
{
    int width, height;
    double sw, sh;

    Tk_SizeOfBitmap(display, bitmap, &width, &height);
    sw = (double)width * xScale;
    sh = (double)height * yScale;
    Blt_AppendToPostScript(ps, "  gsave\n", (char *)NULL);
    Blt_FormatToPostScript(ps, "    %g %g translate\n", sw * -0.5, sh * 0.5);
    Blt_FormatToPostScript(ps, "    %g %g scale\n", sw, -sh);
    Blt_FormatToPostScript(ps, "    %d %d true [%d 0 0 %d 0 %d] {", 
	width, height, width, -height, height);
    Blt_BitmapDataToPostScript(ps, display, bitmap, width, height);
    Blt_AppendToPostScript(ps, "    } imagemask\n  grestore\n", (char *)NULL);
}

void
Blt_2DSegmentsToPostScript(
    Blt_PostScript ps, 
    Segment2D *segments, 
    int nSegments)
{
    Segment2D *sp, *send;

    for (sp = segments, send = sp + nSegments; sp < send; sp++) {
	Blt_FormatToPostScript(ps, "%g %g moveto\n", sp->p.x, sp->p.y);
	Blt_FormatToPostScript(ps, "%g %g lineto\n", sp->q.x, sp->q.y);
	Blt_AppendToPostScript(ps, "DashesProc stroke\n", (char *)NULL);
    }
}

#ifdef notdef
/*
 *--------------------------------------------------------------
 *
 * Tk_PostscriptPhoto --
 *
 *	This procedure is called to output the contents of a
 *	photo image in PostScript, using a format appropriate for
 *	the requested postscript color mode (i.e. one byte per pixel
 *	in gray, and three bytes per pixel in color).
 *
 * Results:
 *	Returns a standard Tcl return value.  If an error occurs
 *	then an error message will be left in interp->result.
 *	If no error occurs, then additional PostScript will be
 *	appended to the interpreter's result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
int
Tk_PostscriptPhoto(interp, blockPtr, psInfo, width, height)
    Tcl_Interp *interp;
    Tk_PhotoImageBlock *blockPtr;
    Tk_PostscriptInfo psInfo;
    int width, height;
{
    TkPostscriptInfo *psInfoPtr = (TkPostscriptInfo *) psInfo;
    int colorLevel = psInfoPtr->colorLevel;
    static int codeIncluded = 0;

    unsigned char *pixelPtr;
    char buffer[256], cspace[40], decode[40];
    int bpc;
    int xx, yy, lineLen;
    float red, green, blue;
    int alpha;
    int bytesPerLine=0, maxWidth=0;

    unsigned char opaque = 255;
    unsigned char *alphaPtr;
    int alphaOffset, alphaPitch, alphaIncr;

    if (psInfoPtr->prepass) {
	codeIncluded = 0;
	return TCL_OK;
    }

    /*
     * Define the "TkPhoto" function, which is a modified version
     * of the original "transparentimage" function posted
     * by ian@five-d.com (Ian Kemmish) to comp.lang.postscript.
     * For a monochrome colorLevel this is a slightly different
     * version that uses the imagemask command instead of image.
     */

    if( !codeIncluded && (colorLevel != 0) ) {
	/*
	 * Color and gray-scale code.
	 */

	codeIncluded = !0;
	Tcl_AppendResult( interp,
		"/TkPhoto { \n",
		"  gsave \n",
		"  32 dict begin \n",
		"  /tinteger exch def \n",
		"  /transparent 1 string def \n",
		"  transparent 0 tinteger put \n",
		"  /olddict exch def \n",
		"  olddict /DataSource get dup type /filetype ne { \n",
		"    olddict /DataSource 3 -1 roll \n",
		"    0 () /SubFileDecode filter put \n",
		"  } { \n",
		"    pop \n",
		"  } ifelse \n",
		"  /newdict olddict maxlength dict def \n",
		"  olddict newdict copy pop \n",
		"  /w newdict /Width get def \n",
		"  /crpp newdict /Decode get length 2 idiv def \n",
		"  /str w string def \n",
		"  /pix w crpp mul string def \n",
		"  /substrlen 2 w log 2 log div floor exp cvi def \n",
		"  /substrs [ \n",
		"  { \n",
		"     substrlen string \n",
		"     0 1 substrlen 1 sub { \n",
		"       1 index exch tinteger put \n",
		"     } for \n",
		"     /substrlen substrlen 2 idiv def \n",
		"     substrlen 0 eq {exit} if \n",
		"  } loop \n",
		"  ] def \n",
		"  /h newdict /Height get def \n",
		"  1 w div 1 h div matrix scale \n",
		"  olddict /ImageMatrix get exch matrix concatmatrix \n",
		"  matrix invertmatrix concat \n",
		"  newdict /Height 1 put \n",
		"  newdict /DataSource pix put \n",
		"  /mat [w 0 0 h 0 0] def \n",
		"  newdict /ImageMatrix mat put \n",
		"  0 1 h 1 sub { \n",
		"    mat 5 3 -1 roll neg put \n",
		"    olddict /DataSource get str readstring pop pop \n",
		"    /tail str def \n",
		"    /x 0 def \n",
		"    olddict /DataSource get pix readstring pop pop \n",
		"    { \n",
		"      tail transparent search dup /done exch not def \n",
		"      {exch pop exch pop} if \n",
		"      /w1 exch length def \n",
		"      w1 0 ne { \n",
		"        newdict /DataSource ",
		          " pix x crpp mul w1 crpp mul getinterval put \n",
		"        newdict /Width w1 put \n",
		"        mat 4 x neg put \n",
		"        /x x w1 add def \n",
		"        newdict image \n",
		"        /tail tail w1 tail length w1 sub getinterval def \n",
		"      } if \n",
		"      done {exit} if \n",
		"      tail substrs { \n",
		"        anchorsearch {pop} if \n",
		"      } forall \n",
		"      /tail exch def \n",
		"      tail length 0 eq {exit} if \n",
		"      /x w tail length sub def \n",
		"    } loop \n",
		"  } for \n",
		"  end \n",
		"  grestore \n",
		"} bind def \n\n\n", (char *) NULL);
    } else if( !codeIncluded && (colorLevel == 0) ) {
	/*
	 * Monochrome-only code
	 */

	codeIncluded = !0;
	Tcl_AppendResult( interp,
		"/TkPhoto { \n",
		"  gsave \n",
		"  32 dict begin \n",
		"  /dummyInteger exch def \n",
		"  /olddict exch def \n",
		"  olddict /DataSource get dup type /filetype ne { \n",
		"    olddict /DataSource 3 -1 roll \n",
		"    0 () /SubFileDecode filter put \n",
		"  } { \n",
		"    pop \n",
		"  } ifelse \n",
		"  /newdict olddict maxlength dict def \n",
		"  olddict newdict copy pop \n",
		"  /w newdict /Width get def \n",
		"  /pix w 7 add 8 idiv string def \n",
		"  /h newdict /Height get def \n",
		"  1 w div 1 h div matrix scale \n",
		"  olddict /ImageMatrix get exch matrix concatmatrix \n",
		"  matrix invertmatrix concat \n",
		"  newdict /Height 1 put \n",
		"  newdict /DataSource pix put \n",
		"  /mat [w 0 0 h 0 0] def \n",
		"  newdict /ImageMatrix mat put \n",
		"  0 1 h 1 sub { \n",
		"    mat 5 3 -1 roll neg put \n",
		"    0.000 0.000 0.000 setrgbcolor \n",
		"    olddict /DataSource get pix readstring pop pop \n",
		"    newdict /DataSource pix put \n",
		"    newdict imagemask \n",
		"    1.000 1.000 1.000 setrgbcolor \n",
		"    olddict /DataSource get pix readstring pop pop \n",
		"    newdict /DataSource pix put \n",
		"    newdict imagemask \n",
		"  } for \n",
		"  end \n",
		"  grestore \n",
		"} bind def \n\n\n", (char *) NULL);
    }

    /*
     * Check that at least one row of the image can be represented
     * with a string less than 64 KB long (this is a limit in the
     * PostScript interpreter).
     */

    switch (colorLevel)
	{
	    case 0: bytesPerLine = (width + 7) / 8;  maxWidth = 240000;  break;
	    case 1: bytesPerLine = width;  maxWidth = 60000;  break;
	    case 2: bytesPerLine = 3 * width;  maxWidth = 20000;  break;
	}
    if (bytesPerLine > 60000) {
	Tcl_ResetResult(interp);
	sprintf(buffer,
		"Can't generate PostScript for images more than %d pixels wide",
		maxWidth);
	Tcl_AppendResult(interp, buffer, (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Set up the postscript code except for the image-data stream.
     */

    switch (colorLevel) {
	case 0: 
	    strcpy( cspace, "/DeviceGray");
	    strcpy( decode, "[1 0]");
	    bpc = 1;
	    break;
	case 1: 
	    strcpy( cspace, "/DeviceGray");
	    strcpy( decode, "[0 1]");
	    bpc = 8;
	    break;
	default:
	    strcpy( cspace, "/DeviceRGB");
	    strcpy( decode, "[0 1 0 1 0 1]");
	    bpc = 8;
	    break;
    }


    Tcl_AppendResult(interp,
	    cspace, " setcolorspace\n\n", (char *) NULL);

    sprintf(buffer,
	    "  /Width %d\n  /Height %d\n  /BitsPerComponent %d\n",
	    width, height,  bpc);
    Tcl_AppendResult(interp,
	    "<<\n  /ImageType 1\n", buffer,
	    "  /DataSource currentfile",
	    "  /ASCIIHexDecode filter\n", (char *) NULL);


    sprintf(buffer,
	    "  /ImageMatrix [1 0 0 -1 0 %d]\n", height);
    Tcl_AppendResult(interp, buffer,
	    "  /Decode ", decode, "\n>>\n1 TkPhoto\n", (char *) NULL);


    /*
     * Check the PhotoImageBlock information.
     * We assume that:
     *     if pixelSize is 1,2 or 4, the image is R,G,B,A;
     *     if pixelSize is 3, the image is R,G,B and offset[3] is bogus.
     */

    if (blockPtr->pixelSize == 3) {
	/*
	 * No alpha information: the whole image is opaque.
	 */

	alphaPtr = &opaque;
	alphaPitch = alphaIncr = alphaOffset = 0;
    } else {
	/*
	 * Set up alpha handling.
	 */

	alphaPtr = blockPtr->pixelPtr;
	alphaPitch = blockPtr->pitch;
	alphaIncr = blockPtr->pixelSize;
	alphaOffset = blockPtr->offset[3];
    }


    for (yy = 0, lineLen=0; yy < height; yy++) {
	switch (colorLevel) {
	    case 0: {
		/*
		 * Generate data for image in monochrome mode.
		 * No attempt at dithering is made--instead, just
		 * set a threshold.
		 * To handle transparecies we need to output two lines:
		 * one for the black pixels, one for the white ones.
		 */

		unsigned char mask=0x80;
		unsigned char data=0x00;
		for (xx = 0; xx< width; xx ++) {
		    int r, g, b;

		    pixelPtr = blockPtr->pixelPtr + (yy * blockPtr->pitch) 
			+ (xx *blockPtr->pixelSize);

		    red = pixelPtr[blockPtr->offset[0]];
		    green = pixelPtr[blockPtr->offset[1]];
		    blue = pixelPtr[blockPtr->offset[2]];

		    alpha = *(alphaPtr + (yy * alphaPitch)
			    + (xx * alphaIncr) + alphaOffset);

		    /*
		     * If pixel is less than threshold, then it is black.
		     */

		    if ((alpha != 0) && 
			(0.3086 * sp->Red + 0.6094 * sp->Green + 0.082 * sp->Blue < 128)) {
			data |= mask;
		    }
		    mask >>= 1;
		    if (mask == 0) {
			sprintf(buffer, "%02X", data);
			Tcl_AppendResult(interp, buffer, (char *) NULL);
			lineLen += 2;
			if (lineLen >= 60) {
			    lineLen = 0;
			    Tcl_AppendResult(interp, "\n", (char *) NULL);
			}
			mask=0x80;
			data=0x00;
		    }
		}
		if ((width % 8) != 0) {
		    sprintf(buffer, "%02X", data);
		    Tcl_AppendResult(interp, buffer, (char *) NULL);
		    mask=0x80;
		    data=0x00;
		}

		mask=0x80;
		data=0x00;
		for (xx = 0; xx< width; xx ++) {
		    pixelPtr = blockPtr->pixelPtr 
			+ (yy * blockPtr->pitch) 
			+ (xx *blockPtr->pixelSize);

		    red = pixelPtr[blockPtr->offset[0]];
		    green = pixelPtr[blockPtr->offset[1]];
		    blue = pixelPtr[blockPtr->offset[2]];

		    alpha = *(alphaPtr + (yy * alphaPitch)
			    + (xx * alphaIncr) + alphaOffset);
			    
		    /*
		     * If pixel is greater than threshold, then it is white.
		     */

		    if ((alpha != 0) && 
			    (  0.3086 * red 
				    + 0.6094 * green 
				    + 0.082 * blue >= 128)) {
			data |= mask;
		    }
		    mask >>= 1;
		    if (mask == 0) {
			sprintf(buffer, "%02X", data);
			Tcl_AppendResult(interp, buffer, (char *) NULL);
			lineLen += 2;
			if (lineLen >= 60) {
			    lineLen = 0;
			    Tcl_AppendResult(interp, "\n", (char *) NULL);
			}
			mask=0x80;
			data=0x00;
		    }
		}
		if ((width % 8) != 0) {
		    sprintf(buffer, "%02X", data);
		    Tcl_AppendResult(interp, buffer, (char *) NULL);
		    mask=0x80;
		    data=0x00;
		}
		break;
	    }
	    case 1: {

		/*
		 * Generate transparency data.
		 * We must prevent a transparent value of 0
		 * because of a bug in some HP printers.
		 */

		for (xx = 0; xx < width; xx ++) {
		    alpha = *(alphaPtr + (yy * alphaPitch) + (xx * alphaIncr) +
			      alphaOffset);
		    sprintf(buffer, "%02X", alpha | 0x01);
		    Tcl_AppendResult(interp, buffer, (char *) NULL);
		    lineLen += 2;
		    if (lineLen >= 60) {
			lineLen = 0;
			Tcl_AppendResult(interp, "\n", (char *) NULL);
		    }
		}


		/*
		 * Generate data in gray mode--in this case, take a 
		 * weighted sum of the red, green, and blue values.
		 */

		for (xx = 0; xx < width; xx ++) {
		    pixelPtr = blockPtr->pixelPtr + (yy * blockPtr->pitch) + 
			(xx *blockPtr->pixelSize);

		    red = pixelPtr[blockPtr->offset[0]];
		    green = pixelPtr[blockPtr->offset[1]];
		    blue = pixelPtr[blockPtr->offset[2]];

		    sprintf(buffer, "%02X", (int) floor(0.5 +
			    ( 0.3086 * red + 0.6094 * green + 0.0820 * blue)));
		    Tcl_AppendResult(interp, buffer, (char *) NULL);
		    lineLen += 2;
		    if (lineLen >= 60) {
			lineLen = 0;
			Tcl_AppendResult(interp, "\n", (char *) NULL);
		    }
		}
		break;
	    }
	    default: {
		/*
		 * Generate transparency data.
		 * We must prevent a transparent value of 0
		 * because of a bug in some HP printers.
		 */

		for (xx = 0; xx < width; xx ++) {
		    alpha = *(alphaPtr + (yy * alphaPitch)
			    + (xx * alphaIncr) + alphaOffset);
		    sprintf(buffer, "%02X", alpha | 0x01);
		    Tcl_AppendResult(interp, buffer, (char *) NULL);
		    lineLen += 2;
		    if (lineLen >= 60) {
			lineLen = 0;
			Tcl_AppendResult(interp, "\n", (char *) NULL);
		    }
		}


		/*
		 * Finally, color mode.  Here, just output the red, green,
		 * and blue values directly.
		 */

		for (xx = 0; xx < width; xx ++) {
		    pixelPtr = blockPtr->pixelPtr 
			+ (yy * blockPtr->pitch) 
			+ (xx *blockPtr->pixelSize);

		    sprintf(buffer, "%02X%02X%02X",
			    pixelPtr[blockPtr->offset[0]],
			    pixelPtr[blockPtr->offset[1]],
			    pixelPtr[blockPtr->offset[2]]);
		    Tcl_AppendResult(interp, buffer, (char *) NULL);
		    lineLen += 6;
		    if (lineLen >= 60) {
			lineLen = 0;
			Tcl_AppendResult(interp, "\n", (char *) NULL);
		    }
		}
		break;
	    }
	}
    }

    Tcl_AppendResult(interp, ">\n", (char *) NULL);
    return TCL_OK;
}
#endif
