/* 
 * tkgsTkWin3d.c --
 *
 *	This is a modified tkWin3d.c that uses TkGS instead of X. All original
 *	names have been prefixed by "TkGS".
 *	Original header follows.
 *
 *	This file contains the platform specific routines for
 *	drawing 3d borders in the Windows 95 style.
 *
 * Copyright (c) 1996 by Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * RCS: @(#) $Id: tkgsTkWin3d.c,v 1.2 2001/03/08 12:31:02 fbonnet Exp $
 */

#include <tkgs.h>

#include <tkgsTk.h>
#include <tkgsTk3d.h>

#include <drivers/win32/tkgsWin.h>

#include <tkWinInt.h>

/*
 * This structure is used to keep track of the extra colors used by
 * Windows 3d borders.
 */

typedef struct {
    TkGSTkBorder info;
    TkGS_Color light2Color; /* System3dLight */
    TkGS_Color dark2Color;  /* System3dDarkShadow */
} WinBorder;


/*
 *----------------------------------------------------------------------
 *
 * TkGSTkpGetBorder --
 *
 *	This function allocates a new TkGSTkBorder structure.
 *
 * Results:
 *	Returns a newly allocated TkGSTkBorder.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

TkGSTkBorder *
TkGSTkpGetBorder()
{
    WinBorder *borderPtr = (WinBorder *) ckalloc(sizeof(WinBorder));
    borderPtr->light2Color = NULL;
    borderPtr->dark2Color = NULL;
    return (TkGSTkBorder *) borderPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * TkGSTkpFreeBorder --
 *
 *	This function frees any colors allocated by the platform
 *	specific part of this module.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	May deallocate some colors.
 *
 *----------------------------------------------------------------------
 */

void
TkGSTkpFreeBorder(borderPtr)
    TkGSTkBorder *borderPtr;
{
    WinBorder *winBorderPtr = (WinBorder *) borderPtr;
    if (winBorderPtr->light2Color) {
	TkGS_FreeColor(winBorderPtr->light2Color);
    }
    if (winBorderPtr->dark2Color) {
	TkGS_FreeColor(winBorderPtr->dark2Color);
    }
}

/*
 *--------------------------------------------------------------
 *
 * TkGSTk_3DVerticalBevel --
 *
 *	This procedure draws a vertical bevel along one side of
 *	an object.  The bevel is always rectangular in shape:
 *			|||
 *			|||
 *			|||
 *			|||
 *			|||
 *			|||
 *	An appropriate shadow color is chosen for the bevel based
 *	on the leftBevel and relief arguments.  Normally this
 *	procedure is called first, then Tk_3DHorizontalBevel is
 *	called next to draw neat corners.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Graphics are drawn in drawable.
 *
 *--------------------------------------------------------------
 */

void
TkGSTk_3DVerticalBevel(drawable, border, x, y, width, height,
	leftBevel, relief)
    TkGS_Drawable drawable;	/* TkGS drawable in which to draw. */
    TkGSTk_3DBorder border;	/* Token for border to draw. */
    int x, y, width, height;	/* Area of vertical bevel. */
    int leftBevel;		/* Non-zero means this bevel forms the
				 * left side of the object;  0 means it
				 * forms the right side. */
    int relief;			/* Kind of bevel to draw.  For example,
				 * TK_RELIEF_RAISED means interior of
				 * object should appear higher than
				 * exterior. */
{
    TkGSTkBorder *borderPtr = (TkGSTkBorder *) border;
    TkGS_Color left, right;
    int half;
    TkGS_Color black = NULL; /* For solid drawing */

    if ((borderPtr->lightColor == NULL) && (relief != TK_RELIEF_FLAT)) {
	TkGSTkpGetShadows(borderPtr);
    }

    switch (relief) {
	case TK_RELIEF_RAISED:
	    left = (leftBevel)
		? borderPtr->lightColor
		: borderPtr->darkColor;
	    right = (leftBevel)
		? ((WinBorder *)borderPtr)->light2Color
		: ((WinBorder *)borderPtr)->dark2Color;
	    break;
	case TK_RELIEF_SUNKEN:
	    left = (leftBevel)
		? ((WinBorder *)borderPtr)->dark2Color
		: ((WinBorder *)borderPtr)->light2Color;
	    right = (leftBevel)
		? borderPtr->darkColor
		: borderPtr->lightColor;
	    break;
	case TK_RELIEF_RIDGE:
	    left = borderPtr->lightColor;
	    right = borderPtr->darkColor;
	    break;
	case TK_RELIEF_GROOVE:
	    left = borderPtr->darkColor;
	    right = borderPtr->lightColor;
	    break;
	case TK_RELIEF_FLAT:
	    left = right = borderPtr->bgColor;
	    break;
	case TK_RELIEF_SOLID:
	    /* TODO: improve this, eg something like TkGS_GetBlack */
	    black = TkGS_GetRGBColor(0,0,0);
	    left = right = black;
	    break;
    }
    half = width/2;
    if (leftBevel && (width & 1)) {
	half++;
    }
    TkGS_FillRectangle(drawable, left, x, y, (unsigned int) half, (unsigned int) height);
    TkGS_FillRectangle(drawable, right, x+half, y, (unsigned int) width-half, (unsigned int) height);

    if (black) {
	TkGS_FreeColor(black);
    }
}

/*
 *--------------------------------------------------------------
 *
 * TkGSTk_3DHorizontalBevel --
 *
 *	This procedure draws a horizontal bevel along one side of
 *	an object.  The bevel has mitered corners (depending on
 *	leftIn and rightIn arguments).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void
TkGSTk_3DHorizontalBevel(drawable, border, x, y, width, height,
	leftIn, rightIn, topBevel, relief)
    TkGS_Drawable drawable;	/* TkGS drawable in which to draw. */
    TkGSTk_3DBorder border;	/* Token for border to draw. */
    int x, y, width, height;	/* Bounding box of area of bevel.  Height
				 * gives width of border. */
    int leftIn, rightIn;	/* Describes whether the left and right
				 * edges of the bevel angle in or out as
				 * they go down.  For example, if "leftIn"
				 * is true, the left side of the bevel
				 * looks like this:
				 *	___________
				 *	 __________
				 *	  _________
				 *	   ________
				 */
    int topBevel;		/* Non-zero means this bevel forms the
				 * top side of the object;  0 means it
				 * forms the bottom side. */
    int relief;			/* Kind of bevel to draw.  For example,
				 * TK_RELIEF_RAISED means interior of
				 * object should appear higher than
				 * exterior. */
{
    TkGSTkBorder *borderPtr = (TkGSTkBorder *) border;
    int bottom, halfway, x1, x2, x1Delta, x2Delta;
    TkGS_Color topColor, bottomColor;
    TkGS_Color black = NULL; /* For solid drawing */

    if ((borderPtr->lightColor == NULL) && (relief != TK_RELIEF_FLAT)) {
	TkGSTkpGetShadows(borderPtr);
    }

    /*
     * Compute a color for the top half of the bevel and a color for the
     * bottom half (they're the same in many cases).
     */

    switch (relief) {
	case TK_RELIEF_RAISED:
	    topColor = (topBevel)
		? borderPtr->lightColor
		: borderPtr->darkColor;
	    bottomColor = (topBevel)
		? ((WinBorder *)borderPtr)->light2Color
		: ((WinBorder *)borderPtr)->dark2Color;
	    break;
	case TK_RELIEF_SUNKEN:
	    topColor = (topBevel)
		? ((WinBorder *)borderPtr)->dark2Color
		: ((WinBorder *)borderPtr)->light2Color;
	    bottomColor = (topBevel)
		? borderPtr->darkColor
		: borderPtr->lightColor;
	    break;
	case TK_RELIEF_RIDGE:
	    topColor = borderPtr->lightColor;
	    bottomColor = borderPtr->darkColor;
	    break;
	case TK_RELIEF_GROOVE:
	    topColor = borderPtr->darkColor;
	    bottomColor = borderPtr->lightColor;
	    break;
	case TK_RELIEF_FLAT:
	    topColor = bottomColor = borderPtr->bgColor;
	    break;
	case TK_RELIEF_SOLID:
	    /* TODO: improve this, eg something like TkGS_GetBlack */
	    black = TkGS_GetRGBColor(0,0,0);
	    topColor = bottomColor = black;
    }

    /*
     * Compute various other geometry-related stuff.
     */

    if (leftIn) {
	x1 = x+1;
    } else {
	x1 = x+height-1;
    }
    x2 = x+width;
    if (rightIn) {
	x2--;
    } else {
	x2 -= height;
    }
    x1Delta = (leftIn) ? 1 : -1;
    x2Delta = (rightIn) ? -1 : 1;
    halfway = y + height/2;
    if (topBevel && (height & 1)) {
	halfway++;
    }
    bottom = y + height;

    /*
     * Draw one line for each y-coordinate covered by the bevel.
     */

    for ( ; y < bottom; y++) {
	/*
	 * In some weird cases (such as large border widths for skinny
	 * rectangles) x1 can be >= x2.  Don't draw the lines
	 * in these cases.
	 */

	if (x1 < x2) {
	    TkGS_FillRectangle(drawable,
		(y < halfway) ? topColor : bottomColor,
		x1, y, (unsigned int) x2-x1, 1);
	}
	x1 += x1Delta;
	x2 += x2Delta;
    }

    if (black) {
	TkGS_FreeColor(black);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TkGSTkpGetShadows --
 *
 *	This procedure computes the shadow colors for a 3-D border
 *	and fills in the corresponding fields of the Border structure.
 *	It's called lazily, so that the colors aren't allocated until
 *	something is actually drawn with them.  That way, if a border
 *	is only used for flat backgrounds the shadow colors will
 *	never be allocated.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The lightColor and darkColor fields in borderPtr get filled in,
 *	if they weren't already.
 *
 *----------------------------------------------------------------------
 */

void
TkGSTkpGetShadows(borderPtr)
    TkGSTkBorder *borderPtr;	/* Information about border. */
{
    int tmp1, tmp2;
    unsigned short red, green, blue;
    int r, g, b;

    if (borderPtr->lightColor != None) {
	return;
    }

    /*
     * Handle the special case of the default system colors.
     */

    if (TkGS_GetColorType(borderPtr->bgColor) == TKGS_COLOR_SYSTEM
	&& ((TkGS_GetSystemColorValue(borderPtr->bgColor) == COLOR_3DFACE)
	     || (TkGS_GetSystemColorValue(borderPtr->bgColor) == COLOR_WINDOW))) {
	borderPtr->darkColor = TkGS_GetSystemColor(COLOR_3DSHADOW);
	borderPtr->lightColor = TkGS_GetSystemColor(COLOR_3DHIGHLIGHT);
	((WinBorder*)borderPtr)->dark2Color = 
		TkGS_GetSystemColor(COLOR_3DDKSHADOW);
	((WinBorder*)borderPtr)->light2Color = 
		TkGS_GetSystemColor(COLOR_3DLIGHT);
	return;
    } else {
	((WinBorder*)borderPtr)->dark2Color = TkGS_GetRGBColor(0,0,0);
	/* TODO: implement TkGS_DuplicateColor */
	((WinBorder*)borderPtr)->light2Color = borderPtr->bgColor;
	TkGS_IncrRefCount((TkGS_Obj *) borderPtr->bgColor);
    }
	
    /*
     * For now, only handle the case of a color display with lots of colors.
     * TODO: handle displays with fewer colors, if ever possible.
     * The shadow colors get computed using whichever formula results
     * in the greatest change in color:
     * 1. Lighter shadow is half-way to white, darker shadow is half
     *    way to dark.
     * 2. Lighter shadow is 40% brighter than background, darker shadow
     *    is 40% darker than background.
     */

    /*
     * This is a color display with lots of colors.  For the dark
     * shadow, cut 40% from each of the background color components.
     * But if the background is already very dark, make the
     * dark color a little lighter than the background by increasing
     * each color component 1/4th of the way to MAX_INTENSITY.
     *
     * For the light shadow, boost each component by 40% or half-way
     * to white, whichever is greater (the first approach works
     * better for unsaturated colors, the second for saturated ones).
     * But if the background is already very bright, instead choose a
     * slightly darker color for the light shadow by reducing each
     * color component by 10%.
     *
     * Compute the colors using integers, not using lightColor.red
     * etc.: these are shorts and may have problems with integer
     * overflow.
     */

    /*
     * Compute the dark shadow color
     */

    TkGS_GetRGBColorValues(borderPtr->bgColor, &red, &green, &blue);
    r = red; g = green; b = blue;

    if (r*0.5*r + g*1.0*g + b*0.28*b < MAX_INTENSITY*0.05*MAX_INTENSITY) {
	red = (MAX_INTENSITY + 3*r)/4;
	green = (MAX_INTENSITY + 3*g)/4;
	blue = (MAX_INTENSITY + 3*b)/4;
    } else {
	red = (60 * r)/100;
	green = (60 * g)/100;
	blue = (60 * b)/100;
    }

    /*
     * Allocate the dark shadow color
     */

    borderPtr->darkColor = TkGS_GetRGBColor(red, green, blue);

    /*
     * Compute the light shadow color
     */

    if (g > MAX_INTENSITY*0.95) {
	red = (90 * r)/100;
	green = (90 * g)/100;
	blue = (90 * b)/100;
    } else {
	tmp1 = (14 * r)/10;
	if (tmp1 > MAX_INTENSITY) {
	    tmp1 = MAX_INTENSITY;
	}
	tmp2 = (MAX_INTENSITY + r)/2;
	red = (tmp1 > tmp2) ? tmp1 : tmp2;
	tmp1 = (14 * g)/10;
	if (tmp1 > MAX_INTENSITY) {
	    tmp1 = MAX_INTENSITY;
	}
	tmp2 = (MAX_INTENSITY + g)/2;
	green = (tmp1 > tmp2) ? tmp1 : tmp2;
	tmp1 = (14 * b)/10;
	if (tmp1 > MAX_INTENSITY) {
	    tmp1 = MAX_INTENSITY;
	}
	tmp2 = (MAX_INTENSITY + b)/2;
	blue = (tmp1 > tmp2) ? tmp1 : tmp2;
    }

   /*
    * Allocate the light shadow color and its GC
    */

    borderPtr->lightColor = TkGS_GetRGBColor(red, green, blue);
}

/*
 *----------------------------------------------------------------------
 *
 * TkgSTkWinGetBorderPixels --
 *
 *	This routine returns the 5 COLORREFs used to draw a given
 *	3d border.  
 *
 * Results:
 *	Returns the colors in the specified array.
 *
 * Side effects:
 *	May cause the remaining colors to be allocated.
 *
 *----------------------------------------------------------------------
 */

COLORREF
TkGSTkWinGetBorderPixels(tkwin, border, which)
    Tk_3DBorder border;
    int which;			/* One of TK_3D_FLAT_GC, TK_3D_LIGHT_GC,
				 * TK_3D_DARK_GC, TK_3D_LIGHT2, TK_3D_DARK2 */
{
    WinBorder *borderPtr = (WinBorder *) border;
    
    if (borderPtr->info.lightColor == None) {
	TkGSTkpGetShadows(&borderPtr->info);
    }
    switch (which) {
	case TK_3D_FLAT_GC:
	    return WinTkGS_GetColorrefFromColor(borderPtr->info.bgColor);
	case TK_3D_LIGHT_GC:
	    if (borderPtr->info.lightColor == NULL) {
		/* TODO: improve this, eg something like TkGS_GetWhite */
		return RGB(255,255,255);
	    }
	    return WinTkGS_GetColorrefFromColor(borderPtr->info.lightColor);
	case TK_3D_DARK_GC:
	    if (borderPtr->info.darkColor == NULL) {
		/* TODO: improve this, eg something like TkGS_GetBlack */
		return RGB(0,0,0);
	    }
	    return WinTkGS_GetColorrefFromColor(borderPtr->info.darkColor);
	case TK_3D_LIGHT2:
	    return WinTkGS_GetColorrefFromColor(borderPtr->light2Color);
	case TK_3D_DARK2:
	    return WinTkGS_GetColorrefFromColor(borderPtr->dark2Color);
    }
    return 0;
}
