# TableList.tcl - itcl widget for displaying tabular information
#                 (based of tk listbox)
#
# usage: TableList $name [options]
#        (see public variables below for options)
#
# Copyright (C) 1994 Allan Brighton (allan@piano.sta.sub.org)
#


itcl_class TableList {
    inherit FrameWidget


    # format the given line with the given format string and return
    # the new string. The line is assumed to be a list of strings.
    # $formats is a list of format strings, one for each element.
    # If the line consists of only one char in the set "_-=", a line
    # is drawn using that char.

    method format_line_ {line} {
	eval $filtercmd line
	if {[string match {[-=_]} $line]} {
	    return [replicate $line $line_length_]
	}
	return [eval "format {$formats} $line"]
    }


    # format and return the table headings

    method format_headings_ {h} {
	return [eval "format {$hformats} $h"]
    }


    # setup the sizes_ array to hold the max width of each column.
    # if the -sizes option was specified, use it, otherwise scan
    # the info for the max widths.

    method get_col_sizes_ {} {
	if {$static_col_sizes && [info exists size_(1)]} {
	    return
	}
	for {set c 1} {$c <= $num_cols_} {incr c} {
	    set size_($c) $hsize_($c)
	}
	if {[llength $sizes] == $num_cols_} {
	    # use list of column sizes
	    set c 0
	    foreach i $sizes {
		incr c
		set size_($c) [max $size_($c) $i]
	    }
	} else {
	    # calculate max col size (loop through each row, col)
	    foreach j $disp_info_ {
		set c 0
		foreach i $j {
		    incr c
		    set size_($c) [max $size_($c) [string length $i]] 
		}
	    }
	}
    }


    # Calculate the print formats for table rows from the max 
    # column widths and options. The format string takes care of
    # column widths, show/hide column and column separators.
    #
    # If the format was set explicitly (formats_flag_ = 1), use it
    # otherwise, if the column widths are known (-sizes was set) use them,
    # otherwise calculate the max column widths from the info list.
    #
    # The formats list uses %n$-s type format strings to set left/right 
    # alignment, width and order all at once, for example:
    # "%-10s" for left justify, "%.0s" effectively hides the item...
    #
    # If the Precision option is set for a column, assume it is a floating
    # point value to be formatted like: %6.2f for example (precision = 2)

    method calculate_format_ {} {
	if {$formats_flag_} {
	    return
	}

	get_col_sizes_

	set hformats {}
	set formats {}
	set order_ {}
	foreach name [$this.config_file info public order -value] {
	    set col $ord_($name)
	    lappend order_ [expr $num_cols_-1]

	    set align [$this.config_file get_option $name Align]
	    set show [$this.config_file get_option $name Show]
	    set sep [$this.config_file get_option $name Separator]
	    set prec [$this.config_file get_option $name Precision]

	    if {"$align" == "Left"} {
		set a -
	    } else {
		set a {}
	    }

	    if {$prec == 0} {
		set p {}
		set t s
	    } else {
		set p .$prec
		set t f
	    }

	    if {!$show} {
		append formats [set f [format {%%%d$.0s} $col]]
		append hformats $f
	    } else {
		append formats [format {%%%d$%s%d%s%s %s } $col $a $size_($col) $p $t $sep]
		append hformats [format {%%%d$%s%ds %s } $col $a $size_($col) $sep]
	    }
	}
    }


    # set up the wildcard list to filter the info list
    
    method setup_matching_ {} {
	set m {}
	set match_all_ 1
	foreach name $headings {
	    set wildcard [$this.config_file get_option $name Wildcard]
	    if {"$wildcard" == ""} {
		set wildcard $match_any_
	    } elseif {"$wildcard" != "$match_any_"} {
		set match_all_ 0
	    }
	    append m "{$wildcard} "
	}
	set match_list_ [string trimright $m]
    }


    # apply the current wildcard filters to the info list
    # so that only the rows that match will be displayed

    method match_info_ {} {
	setup_matching_
	set info_cols_ [llength $headings]
	set total_rows_ [llength $info]
	if {$match_all_} {
	    set disp_info_ $info
	    set info_rows_ $total_rows_
	} else {
	    set disp_info_ {}
	    set info_rows_ 0
	    foreach i $info {
		if {[$match_proc_ $i]} {
		    lappend disp_info_ $i
		    incr info_rows_
		} 
	    }
	}
    }


    # sort the info list, if requested, based on the
    # sort keys

    method sort_info_ {} {
	if {"$sort_by" != ""} {
	    set disp_info_ [lsort -$sort_order -command "$this cmp_row_" $disp_info_]
	}
    }

    # compare the 2 given rows, ascii version (called by lsort above)
    # (see lsort man page for desc.)
    # Note the "1" prepended to the operands: this is in case you have
    # numbers that look like "0nnn" by are not octal (try [expr 0510 > 0328]).

    method cmp_row_ {r1 r2} {
	foreach i $sort_by {
	    if {"[set v1 1[lindex $r1 $i]]" > "1[set v2 [lindex $r2 $i]]"} {
		return 1
	    } elseif {$v1 < $v2} {
		return -1
	    }
	}
	return 0
    }


    # move the selected row down 1 row
    # and make the changes in the info list

    method move_down {} {
	set list [$listbox_ curselection]
	if {"$list" == ""} {
	    return
	}
	set rlist [lsort -decreasing $list]
	foreach i $rlist {
	    set s [$listbox_ get $i]
	    set r [lindex $disp_info_ $i]
	    $listbox_ delete $i
	    set disp_info_ [lremove $disp_info_ $i]
	    set j [expr $i+1]
	    $listbox_ insert $j $s
	    set disp_info_ [linsert $disp_info_ $j $r] 
	}
	select_rows [expr [lindex $list 0]+1] [expr [lindex $rlist 0]+1]
    }

    
    # move the selected row up 1 row
    # and make the changes in the info list

    method move_up {} {
	set list [$listbox_ curselection]
	if {"$list" == ""} {
	    return
	}
	foreach i $list {
	    set s [$listbox_ get $i]
	    set r [lindex $disp_info_ $i]
	    $listbox_ delete $i
	    set disp_info_ [lremove $disp_info_ $i]
	    set j [expr $i-1]
	    $listbox_ insert $j $s
	    set disp_info_ [linsert $disp_info_ $j $r] 
	}
	select_rows [expr [lindex $list 0]-1] [expr $i-1]
    }


    # make the table empty

    method clear {} {
	$listbox_ delete 0 end
	set info {}
	set disp_info_ {}
	set info_rows_ 0
 	set total_rows_ 0
   }

    # replace the contents of the given row with the new info
    # Note: this assumes that no 2 rows are exactly alike.
    # We can't use the row index here, since sorting and matching 
    # may mix things up too much.

    method set_row {oldrow newrow} {
	set index [lsearch -exact $info $oldrow]
	set info [lreplace $info $index $index $newrow]
	save_yview
	save_selection
	busy new_info
	restore_selection
	restore_yview
    }

    # save the current scroll position so it can be restored
    # later. 
    
    method save_yview {} {
	global tk_version
	if {$tk_version >= 4.0} {
	    set saved_yview_ [lindex [$listbox_ yview] 0]
	} else {
	    set saved_yview_ [lindex [$this.vscroll get] 2]
	}
    }


    # restore the previously saved scroll position 
    
    method restore_yview {} {
	global tk_version
	if {$tk_version >= 4.0} {
	    $listbox_ yview moveto $saved_yview_
	} else {
	    $listbox_ yview $saved_yview_
	}
    }

    
    # save a list of the currently selected rows so they can be restored
    # later. 
    
    method save_selection {} {
	set saved_selection_ [$listbox_ curselection]
    }


    # restore the previously saved row selection
    
    method restore_selection {} {
	set n [llength $saved_selection_]
	if {$n} {
	    select_rows [lindex $saved_selection_ 0] \
		[lindex $saved_selection_ [incr n -1]]
	}
    }


    # return true if the given row matches the current wildcard patterns

    method match_glob_ {row} {
	for {set i 0} {$i < $info_cols_} {incr i} {
	    if {![string match [lindex $match_list_ $i] [lindex $row $i]]} {
		return 0
	    }
	}
	return 1
    }


    # return true if the given row matches the current regexp

    method match_regexp_ {row} {
	for {set i 0} {$i < $info_cols_} {incr i} {
	    if {![regexp -- [lindex $match_list_ $i] [lindex $row $i]]} {
		return 0
	    }
	}
	return 1
    }


    # return true if the given row matches the current regexp (ignore case)

    method match_regexp_nocase_ {row} {
	for {set i 0} {$i < $info_cols_} {incr i} {
	    if {![regexp -nocase -- [lindex $match_list_ $i] [lindex $row $i]]} {
		return 0
	    }
	}
	return 1
    }


    # append a row to the table. 
    # Note: you will need to do a "new_info" when you are done appending
    # rows if you want any further matching, sorting or formatting to occur.

    method append_row {row} {
	$listbox_ insert end [format_line_ $row]
	lappend info $row
	lappend disp_info_ $row
	incr info_rows_
	incr total_rows_
    }


    # append a list of rows to the table

    method append_rows {rows} {
	foreach i $rows {
	    virtual append_row $i
	}
	new_info
    }

    
    # this method is called whenever the info list changes

    method new_info {} { 
	virtual match_info_
	virtual sort_info_
	set_info_
    }


    # update the display with the new info

    method set_info_ {} {
	calculate_format_
	set_headings_
	$listbox_ delete 0 end

	set n 0
	foreach i $disp_info_ {
	    $listbox_ insert end [format_line_ $i]
	}
    }

    
    # if using 2 line headings, split the table headings in 2 lines 
    # and return the results as a list of heading lines, otherwise 
    # just return a list whose only element is the single line of headings.

    method split_headings_ {} {
	if {$heading_lines == 1} {
	    return [list $headings]
	}
	set h1 {}
	set h2 {}
	foreach i $headings {
	    if {[llength $i] == 2} {
		lappend h1 [lindex $i 0]
		lappend h2 [lindex $i 1]
	    } else {
		lappend h1 {}
		lappend h2 $i
	    }
	}
	return [list $h1 $h2]
    }


    # set the table headings -
    # Note: headings may be 1 or 2 lines. The first and second lines
    # should be separated by spaces i.e.: {Sub Total} to put "Sub"
    # on the first line and "Total" on the second.
    # use the -heading_lines 2 option to enable this...

    method set_headings_ {} {
	$headbox_ delete 0 end
	foreach h [split_headings_] {
	    $headbox_ insert end [set s [format_headings_ $h]]
	}
	set line_length_ [string length $s]
    }


    # select the row with the given index

    method select_row {n} {
	global tk_version
	if {$tk_version >= 4.0} {
	    $listbox_ selection clear 0 end
	    $listbox_ selection set $n
	} else {
	    $listbox_ select from $n
	}
    }


    # select a range of rows

    method select_rows {from to} {
	global tk_version
	if {$tk_version >= 4.0} {
	    $listbox_ selection clear 0 end
	    $listbox_ selection set $from $to
	} else {
	    $listbox_ select from $from
	    $listbox_ select to $to
	}
    }


    # search for and highlight the first row containing the given value
    # in the named column

    method search {name value} {
	if {[set idx [lsearch -exact $headings $name]] < 0} {
	    return
	}
	set n 0
	foreach row $disp_info_ {
	    if {"[lindex $row $idx]" == "$value"} {
		select_row $n
		break
	    }
	    incr n
	}
    }


    # return a list of the selected rows
    # note: use disp_info_, since listbox contains formated lines

    method get_selected {} {
	set list {}
	foreach i [$listbox_ curselection] {
	    lappend list [lindex $disp_info_ $i]
	}
	return $list
    }


    # return the number of rows currently selected in the table

    method num_selected {} {
	return [llength [$listbox_ curselection]]
    }

    
    # clear the selection

    method clear_selection {} {
	global tk_version
	if {$tk_version >= 4.0} {
	    $listbox_ selection clear 0 end
	} else {
	    $listbox_ select clear
	}
    }


    # remove the selected rows from the table
    # and return them as a list of lists

    method remove_selected {} {
	set list [get_selected]
	set n 0
	foreach i [lsort -decreasing [$listbox_ curselection]] {
	    set disp_info_ [lremove $disp_info_ $i]
	    incr n
	}
	# note: hack: assumes matching is off...
	set info $disp_info_
	new_info
	if {$n} {
	    select_row $i
	}
	return $list
    }


    # pop up a window to change the layout of the table

    method layout_dialog {} {
	if {"$headings" == ""} {
	    return
	}
	busy {
	    set w [format {%s.tblcfg} [utilGetTopLevel $this]]
	    utilReUseWidget TableListConfig $w \
		    -table $this
	}
    }


    # return the total number of rows (before matching)

    method total_rows {} {
	return $total_rows_
    }


    # return the number of rows being displayed in the table (after matching)

    method info_rows {} {
	return $info_rows_
    }


    # pop up a dialog to sort the contents of the table based on given
    # criteria

    method sort_dialog {} {
	if {"$headings" == ""} {
	    return
	}
	busy {
	    set w [format {%s.tblsort} [utilGetTopLevel $this]]
	    utilReUseWidget TableListSort $w -table $this
	}
   }


   # pop up a dialog to print the contents of the table to a printer
   # or file

   method print_dialog {} {
	if {"$headings" == ""} {
	    return
	}
	busy {
	    set w [format {%s.tblprint} [utilGetTopLevel $this]]
	    if {[winfo exists $w]} {
		wm deiconify $w
	    } else {
		TableListPrint $w -table $this -printcmd $printcmd
	    }
	}
   }


    # print the table heading(s) given by h1 and h2 to the given fd and underline 
    # it (them) with the given underline char

    method print_heading_ {fd h1 h2 underline} {
	puts $fd "\n$h1"
	if {"$h2" != ""} {
	    puts $fd $h2
	}
	puts $fd "[replicate $underline [max [string length $h1] [string length $h2]]]\n"
    }
    

    # print the contents of the table to the open file descriptor

    method print {fd} {
	if {"$title" != ""} {
	    print_heading_ $fd "$title" "" "*"
	}

	if {$heading_lines == 1} {
	    print_heading_ $fd [format_headings_ $headings] "" "="
	} else {
	    set h [split_headings_] 
	    print_heading_ $fd [format_headings_ [lindex $h 0]] \
		[format_headings_ [lindex $h 1]] "="
	}
	
	foreach i $disp_info_ {
	    puts $fd [format_line_ $i]
	}
    }


    # return the contents of the table as a list of rows

    method get_contents {} {
	return $disp_info_
    }

    
    # set the option value for the given heading name
    # 
    # Options: 
    #  Show (bool)         - display or don't display the column
    #  Align (Left,Right)  - for left or right justify
    #  Separator           - set the separator string (goes after col)
    #  Wildcard            - only show rows where wildcard matches
    #  Precision           - number of places after the decimal for floating point values

    method set_option {name option value} {
	$this.config_file set_option $name $option $value
    }

    
    # same as set_option, but works on a list of column heading names

    method set_options {headings option value} {
	foreach name $headings {
	    $this.config_file set_option $name $option $value
	}
    }
   
    # return the option value for the given heading name.
    # See above for list of Options...

    method get_option {name option} {
	return [$this.config_file get_option $name $option]
    }

    
    # scroll to the end of the table and display the last set of rows

    method show_last_row {} {
	global tk_version
	if {$tk_version >= 4.0} {
	    $listbox_ yview moveto 1
	} else {
	    # need to update the scrollbar...
	    update idletasks
	    $listbox_ yview [expr $info_rows_ - [lindex [$this.vscroll get] 1]]
	}
    }


    # scroll both the heading box and the main listbox syncronously
    # (called for horizontal scrolling in listbox)

    method xview {args} {
	eval "$listbox_ xview $args"
	eval "$headbox_ xview $args"
    }


    # scroll the listbox vertically

    method yview {args} {
	eval "$listbox_ yview $args"
    }


    # load the named config file and update the display based on the new settings
    
    method load_config {file} {
	$this.config_file load $file
	virtual new_info
    }

    
    # get a name from the user and use it to save the current 
    # configuration to a file under the user's home directory

    method delete_config {file} {
	set s [file tail $file]
	exec rm -f $file
	$menubutton_.menu.load_pr delete $s
	$menubutton_.menu.delete_pr delete $s
    }

    
    # get a name from the user and use it to save the current 
    # configuration to a file under the user's home directory

    method save_dialog {} {
	set name [input_dialog "Please enter a name to use for this configuration:"]
	if {"$name" == ""} {
	    return
	}
	set file [utilGetConfigFilename $this $name]
	set s [file tail $file]
	if {![file exists $file]} {
	    $menubutton_.menu.load_pr add command -label $s \
		-command [list $this load_config $file]
	    $menubutton_.menu.delete_pr add command -label $s \
		-command [list $this delete_config $file]
	}
	$this.config_file save $file
    }


    # add the table config menu items to the given menu

    method make_table_menu {} {
	set m [menu $menubutton_.menu]
	$m add command -label "Print..." \
		-command "$this print_dialog"
	$m add command -label "Sort..." \
		-command "$this sort_dialog"
	$m add command -label "Configure..." \
		-command "$this layout_dialog"
	$m add command -label "Save Configuration..." \
		-command "$this save_dialog"
	# add a menu for the previously saved configurations
	$m add cascade -label "Load Configuration" \
		-menu $m.load_pr
	$m add cascade -label "Delete Configuration" \
		-menu $m.delete_pr
	menu $m.load_pr
	menu $m.delete_pr
	set dir [utilGetConfigFilename $this]
	foreach file [glob -nocomplain $dir/*] {
	    set s [file tail $file]
	    $m.load_pr add command -label $s \
		    -command [list $this load_config $file]
	    $m.delete_pr add command -label $s \
		    -command [list $this delete_config $file]
	}
    }

    #  create a new object of this class

    constructor {config} {
	FrameWidget::constructor
	global tk_version

	set heading_font [option get . headingFont Tk]
	set title_font [option get . titleFont Tk]
	set title_bg [option get . titleBackground Tk]

	# title and menubutton
	pack [frame $this.top] \
	    -side top -fill x

	if {"$menubar" == ""} {
	    set menubar $this.top
	}
	set menubutton_ [menubutton $menubar.table \
			     -text Table \
			     -menu $menubar.table.menu \
			     -state disabled]
	if {$show_menubutton} {
	    pack $menubutton_ -side left -padx 1m -ipadx 1m
	}
	make_table_menu

	pack [label $this.title \
		  -text $title \
		  -font $title_font] \
		-side top -fill x -expand 1 -in $this.top

	if {"$title_bg" != ""} {
	    $this.title config -bg $title_bg
	}

	# heading box

	if {$tk_version >= 4.0} {
	    set headbox_ [listbox $this.headbox \
			      -relief sunken \
			      -font $heading_font \
			      -setgrid 1 \
			      -height $heading_lines]
	} else {
	    set headbox_ [listbox $this.headbox \
			      -relief sunken \
			      -font $heading_font \
			      -setgrid 1 \
			      -geometry 1x$heading_lines]
	}

	# table
	set hscroll [scrollbar $this.hscroll \
		-relief sunken \
		-orient horizontal \
		-command "$this xview"]
	set vscroll [scrollbar $this.vscroll \
		-relief sunken \
		-command "$this.listbox yview"]
	set listbox_ [listbox $this.listbox \
		-yscroll "$vscroll set" \
		-xscroll "$hscroll set" \
		-relief sunken]

	# sync scrolling
	foreach i "$listbox_ $headbox_" {
	    bind $i <2> "$listbox_ scan mark %x %y; $headbox_ scan mark %x 0"
	    bind $i <B2-Motion> "$listbox_ scan dragto %x %y; $headbox_ scan dragto %x 0"
	}
	bind $headbox_ <B1-Motion> { }
	bind $headbox_ <Shift-B1-Motion> { }
	bind $headbox_ <Shift-1> { }
	bind $headbox_ <1> { }

	if {$show_vscroll} {
	    pack $vscroll -side right -fill y
	}
	pack $headbox_ -side top -fill x
	if {$show_hscroll} {
	    pack $hscroll -side bottom -fill x
	}
	pack $listbox_ -fill both -expand 1

	# create an object for reading the config file
	TableListConfigFile $this.config_file \
		-headings $headings \
		-use_regexp $use_regexp

	set initialized_ 1

	#  Explicitly handle config's that may have been ignored earlier
	foreach attr $config {
	    config -$attr [set $attr]
	}
    }


    # Destructor: called when this object is deleted

    destructor {
	$this.config_file delete
    }


    # -- public member variables --

    
    # title string
    public title {} {
	if {$initialized_} {
	    $this.title config -text $title
	}
    }

    # if true, add a "Table" menu button to the table for config and sorting
    public show_menubutton 1

    # if true, display the vertical scrollbar
    public show_vscroll 1

    # if true, display the horizontal scrollbar
    public show_hscroll 1

    # name of menubar frame in which to place table menubutton (optional)
    public menubar {}

    # field names for heading - Note: specify before "-info"
    public headings {} {
	if {$initialized_} {
	    if {"$headings" != ""} {
		$menubutton_ config -state normal
		# set number of cols, heading sizes, col order
		set num_cols_ 0
		foreach i $headings {
		    incr num_cols_
		    if {$heading_lines == 1} {
			set hsize_($num_cols_) [string length $i]
		    } else {
			set hsize_($num_cols_) \
			    [max [string length [lindex $i 0]] \
				 [string length [lindex $i 1]]]
		    }
		    set ord_($i) $num_cols_
		}
	    }
	    $this.config_file config -headings $headings
	}
    }

    # number of lines to display for the headings - may be 1 or 2
    # for 2: each column heading may be a list of 2 (top and bottom) strings.
    public heading_lines 1 {
	if {$initialized_} {
	    global tk_version
	    if {$tk_version >= 4.0} {
		$headbox_ config -height $heading_lines
	    } else {
		$headbox_ config -setgrid 1 -geometry 1x$heading_lines
	    }
	}
    }

    # set the number of rows to be displayed
    public row_lines {} {
	if {$initialized_} {
	    global tk_version
	    if {$tk_version >= 4.0} {
		$listbox_ config -height $row_lines
	    } else {
		if {[scan [utilGetConfigValue $listbox_ -geometry] {%dx%d} w h] == 2} {
		    $listbox_ config -geometry ${w}x$row_lines
		}
	    }
	}
    }

    # list of lists, one per line to display in table/list
    public info {} {
	if {$initialized_} {
	    new_info
	}
    }

    # list of column sizes
    # (if not specified, will be calculated)
    public sizes {}

    # if true, reuse the calculated column sizes rather than recalculate
    # for new info
    public static_col_sizes 0

    # list of printf formats for columns
    # (if not specified, will be calculated, see also -sizes)
    public formats {} {
	set formats_flag_ 1
	set hformats $formats
    }

    # print format string for the headings 
    # (set after -formats, defaults to same as $formats)
    # This might be different for headings if a column uses %f formats...
    public hformats {}
    
    # command to call to filter each row,
    # hook to modify the row before it is displayed
    # arg is is the name of the list holding the row (call by reference)
    public filtercmd "#"

    # default print command
    public printcmd {lpr}

    # choose the select mode: one of {single, browse, multiple, extended}
    public selectmode {single} {
	if {$initialized_} {
	    global tk_version
	    if {$tk_version >= 4.0} {
		$listbox_ config -selectmode $selectmode
	    } else {
		config -single_select [expr {"$selectmode" == "single"}]
	    }
	}
    }

    # tk3.6 flag: single selection mode
    public single_select {0} {
	if {$initialized_} {
	    if {$single_select} {
		tk_listboxSingleSelect $listbox_
	    }
	}
    }


    # flag: export selection ?
    public exportselection {1} {
	if {$initialized_} {
	    $listbox_ config -exportselection $exportselection
	}
    }


    # -- sort options --

    # list of col index: sort table based on given columns 
    # (empty means don't sort)
    public sort_by {} {
	if {$initialized_} {
	    $this.config_file config -sort_by $sort_by
	}
    }

    # set direction of sort: may be one of (increasing, decreasing) 
    public sort_order {increasing} {
	if {$initialized_} {
	    $this.config_file config -sort_order $sort_order
	}
    }

    # -- match options --

    # flag: if true, use regular exprs for matching, otherwise use wildcards
    public use_regexp {0} {
	if {$use_regexp} {
	    set match_any_ ".*"
	    set match_proc_ match_regexp_
	} else {
	    set match_any_ "*"
	    set match_proc_ match_glob_
	}
    }

    # flag: if true, ignore case in matching 
    # (only works when -use_regexp 1 was specified)
    public ignore_case 0 {
	if {$ignore_case} {
	    set match_any_ ".*"
	    set match_proc_ match_regexp_nocase_
	} else {
	    config -use_regexp $use_regexp
	}
    }


    # -- protected members --

    # flag: true after initialization
    protected initialized_ 0

    # flag: true if the -formats option was specified
    # so that we don't have to calculate the format string for a row
    protected formats_flag_ {0}

    # list of indexes in row for headings (indep. of viewing order)
    protected order_ {}

    # listbox widget
    protected listbox_

    # box for headings
    protected headbox_

    # list of glob expressions for matching rows to wildcards
    protected match_list_ {*}

    # flag: true if match_list_ should match all rows
    protected match_all_ {1}

    # string used to match any string
    protected match_any_ {*}

    # method to use for matching rows (match_glob_ or match_regexp_)
    protected match_proc_ {match_glob_}

    # number of rows in the info list (not including hidden rows)
    protected info_rows_ {0}

    # number of columns in table 
    protected info_cols_ 0

    # length of a line in the table
    protected line_length_ 0

    # total number of rows, including hidden rows
    protected total_rows_ {0}

    # max number of rows visible at one time in listbox
    protected max_visible_rows_ {0}

    # list of info to display (after filtering and sorting)
    protected disp_info_ {}

    # used for save_/restore_yview methods
    protected saved_yview_ 0

    # used for save_/restore_selection methods
    protected saved_selection_ {}

    # array(col) of col width
    protected size_
    
    # array(col) of heading width
    protected hsize_

    # number of columns
    protected num_cols_ 0

    # array(heading) of column order
    protected ord_

    # menubutton widget
    protected menubutton_
}


# make sure we are using fixed width fontts
option add *Listbox.font -adobe-courier-medium-r-*-*-*-120-*-*-*-*-*-* widgetDefault
option add *tableFont -adobe-courier-medium-r-*-*-*-120-*-*-*-*-*-* widgetDefault
option add *headingFont -adobe-courier-bold-r-*-*-*-120-*-*-*-*-*-* widgetDefault
option add *titleFont -adobe-courier-bold-r-*-*-*-120-*-*-*-*-*-* widgetDefault

