# FileReader.tcl - itcl class for reading and monitoring a file
#
# This class uses the addinput extension and a "tail -f" process 
# to arrange to be notified when ever there is more to read in the file.
# 
# Usage: ReadFile $obj -file $filename -command $cmdstring
#
# where command is called with 2 args: 
#  1. the line read from the file
#  2. a flag, set to 1 if the line is new and 0 if it was already in the
#     file or pipe.
#
# Deleting this object also kills the "tail -f" reader process.
#
# Methods:
#        $obj pause       - stop reading temporarily
#        $obj resume      - resume reading
#
# -------------------------------------------------------------------------
# Note: I tried using addinput directly on the file, rather than on a pipe
# from tail -f, but the callback is called constantly, since the file is
# apparently always ready for input...
# -------------------------------------------------------------------------
#
# Clean up is taken care of automatically by the destructor. This class
# also creates a dummy frame that is not displayed, but us used to ensure
# that the destructor is called when the main window is destroyed by catching
# the <Destroy> event.
#
# Author: Allan Brighton
#

itcl_class FileReader  {


    # constructor: handle public variables

    constructor {config} {

	if {"$command" == ""} {
	    set command "$this print_line_"
	}

	set initialized_ 1
	#  Explicitly handle config's that may have been ignored earlier
	foreach attr $config {
	    config -$attr [set $attr]
	}
    }


    # destructor: clean up, close file

    destructor {
	cleanup_
    }


    # configure public variables

    method config {config} {}


    # stop reading the file

    method pause {} {
	catch {removeinput $fd_}
	set paused_ 1
    }


    # resume reading 

    method resume {} {
	catch {removeinput $fd_}
	addinput $fd_ "$this reader_"
	set paused_ 0
    }

    
    # close the fd and kill the reader process, if it is still there
    
    method cleanup_ {} {
	catch {kill $pid_}
	catch {removeinput $fd_}
	catch {close $fd_}
	catch {destroy $frame_}
    }


    # This method is called when there is data to read.

    method reader_ {} {
	if {$paused_} {
	    # shouldn't get here
	    removeinput $fd_
	    return
	} 
	if {[gets $fd_ line] < 0} {
	    global argv
	    removeinput $fd_
	    error_dialog "[file tail $argv0]: error reading from file $file"
	    return
	}

	do_command_ $line 1
    }

    
    # process a line of input from the file by evaluating the 
    # command with the line and the arg as arguments

   method do_command_ {line new} {
       eval [concat $command [list $line $new]]
   }


    # dummy command for testing, called for each line read

    method print_line_ {line new} {
	puts "$line ($new)"
    }

    
    # start a new reader process on the file

    method start_reader_ {} {
	# first end any old readers
	cleanup_
	
	# read existing data first
	set fd_ [open $file]
	set n 0
	while {[gets $fd_ line] != -1} {
	    incr n
	    do_command_ $line 0
	}
	close $fd_

	# do a tail -f on the rest. since tail -0f doesn't work,
	# start reading at the last line and discard it...
	set fd_ [open "| tail -1f $file"]
	if {$n > 0} {
	    gets $fd_ line
	}
	set pid_ [pid $fd_]

	# setup notification
	addinput $fd_ "$this reader_"

	# create a dummy frame so we can be notified if the main window
	# (i.e.: the application) goes away. 
	set frame_ [frame .fr$pid_]
	bind $frame_ <Destroy> "catch {$this delete}"
    }


    # -- public variables --

    # file to read, close any previous file and start reading the new one
    public file {} {
	if {$initialized_} {
	    start_reader_
	}
    }

    # command to call for each line read from the file
    public command {}


    # -- protected variables --

    # message file handle (file desc)
    protected fd_ {}
    
    # process id of tail -f reader process
    protected pid_ {}

    # dummy frame used to catch application exit
    protected frame_ {}
    
    # set to 1 when reading is stopped
    protected paused_ {0}
    
    # set to 1 in constructor after all public vars have been set
    protected initialized_ 0
}


