# TextWin.tcl - class based on Tk text widget
#
# usage: TextWin name [options]
#
# see "public" variables below for options.
# 
# Author: Allan Brighton 


itcl_class TextWin {
    inherit FrameWidget

    constructor {config} {
	FrameWidget::constructor

	pack [text $this.text -yscroll "$this.vscroll set"] \
	    -side left -fill both -expand 1
	scrollbar $this.vscroll -command "$this.text yview"
	
	# set up the bindings for the text widget
	set_text_bindings $this.text

	if {$vscroll} {
	    pack $this.vscroll -side right -fill y
	} else {
	    bind $this.text <B2-Motion> { }
	}

	#  Explicitly handle config's that may have been ignored earlier
	set initialized_ 1
	foreach attr $config {
	    config -$attr [set $attr]
	}
    }

    
    # enforce the max chars limit, if specified, and
    # beep if the text is truncated

    method check_length {} {
	set end [$this.text index end]
	if {$max_chars > 0} {
	    $this.text delete "1.0 + $max_chars chars" end
	}
	if {"$end" != "[$this.text index end]"} {
	    blt_bell
	}
    }

    
    # if there are no args, return the contents of the text widget,
    # otherwise return the specified range (see text(n))
    
    method get {args} {
	if {[llength $args]} {
	    return [set value [eval $this.text get $args]]
	}
	return [set value [$this.text get 1.0 end]]
    }


    # return true if the contents of the text window has changed 
    # since it was last configured or read

    method changed {} {
	return [expr {"[$this.text get 1.0 end]" != "$value"}]
    }


    # pass these methods on to the text widget

    foreach i {index insert} {
	method $i {args} [format {return [eval "$this.text %s $args"]} $i]
    }

    
    
    # -- public variables --


    # set the value of the text to be displayed
    public value {} {
	if {$initialized_} {
	    $this.text delete 1.0 end
	    $this.text insert end $value
	}
    }

    # title for top level window
    public title {} {
	if {$initialized_} {
	    wm title $this $title
	}
    }

    
    # flag: if true, display a vertical scrollbar
    public vscroll {1} 

    # if non-zero, spcifies a max char count allowed for the window
    public max_chars {0} {
	if {$initialized_} {
	    if {$max_chars > 0} {
		foreach i {2 KeyPress Return} {
		    bind $this.text <$i> "+ $this check_length"
		}
	    }
	}	
    }

    # pass these on to the text widget unchanged

    foreach i {height width bg font state} {
	public $i {} [format {
	    if {$initialized_} {
		$this.text config -%s $%s
	    }
	} $i $i]
    }

    # -- protected variables --

    # set to 1 after widget creation
    protected initialized_ 0
}

