# TableListConfigFile.tcl - itcl class (no windows) for managing the config
#                           file for the TableList widget.
# 
# See also TableListConfig.tcl for the user interface to this class.
#
# Copyright (C) 1994 Allan Brighton (allan@piano.sta.sub.org)
#


itcl_class TableListConfigFile {

    
    # save the current information to the given file for later 
    # loading (with "load" below)

    method save {file} {
	if {[catch {set fd [open $file w]} msg]} {
	    warning_dialog "couldn't save layout information in file \"$file\": $msg"
	    return
	}
	
	# output the header and list of column headings 
	puts $fd $magic_
	puts $fd $headings
	puts $fd [list $sort_by $sort_order $sort_type]

	# make a list with all the headings and options in column order
	set list {}
	foreach i $order {
	    lappend list [list $i \
		    $options_($i,Show) \
		    $options_($i,Align) \
		    $options_($i,Wildcard) \
		    $options_($i,Separator)]
	}
	puts $fd $list
	close $fd
    }

    
    # load the user config file (created by save) if one exists,
    # otherwise set the default values. 

    method load {file} {
	global env

	if {![file exists $file]} {
	    error "file $file does not exist"
	    return
	}

	# read the file
	set fd [open $file]

	if {"[gets $fd]" != "$magic_"} {
	    set_defaults
	} else {
	    set saved_headings_ [gets $fd]
	    if {"$headings" != "" && "$headings" != "$saved_headings_"} {
		set_defaults
	    } else {
		# sorting options
		lassign [gets $fd] sort_by sort_order sort_type

		# headings and column options
		set order {}
		foreach i [gets $fd] {
		    lassign $i name show align wildcard sep
		    lappend order $name
		    set options_($name,Align) $align
		    set options_($name,Show) $show
		    set options_($name,Wildcard) $wildcard
		    set options_($name,Separator) $sep
		}
	    }
	}
	close $fd
    }

    
    # set the option value for the given heading name
    # 
    # Options: 
    #  Show (bool)         - display or don't display the column
    #  Align (Left,Right)  - for left or right justify
    #  Separator           - set the separator string (goes after col)
    #  Wildcard            - only show rows where wildcard matches

    method set_option {name option value} {
	set options_($name,$option) $value
    }

    
    # return the value of the given option for the given heading name
    # (see comments above for set_option...)

    method get_option {name option} {
	return $options_($name,$option)
    }


    # read option values from the named array

    method set_options {ar} {
	upvar $ar opts
	foreach i [array names opts] {
	    set options_($i) $opts($i)
	}
    }

    
    # write the current option values to the named array

    method get_options {ar} {
	upvar $ar opts
	foreach i [array names options_] {
	    set opts($i) $options_($i)
	}
    }


    # set the default values in case there is no config
    # file or it was out of date
    # 
    # If force is 1, set even values that are already set.

    method set_defaults {{force 0}} {
	set order [set saved_headings_ $headings]
	foreach name $headings {
	    if {$force || ![info exists options_($name,Align)]} {
		set options_($name,Align) Left
	    }
	    if {$force || ![info exists options_($name,Show)]} {
		set options_($name,Show) 1
	    }
	    if {$force || ![info exists options_($name,Wildcard)]} {
		set options_($name,Wildcard) $match_any_
	    }
	    if {$force || ![info exists options_($name,Separator)]} {
		set options_($name,Separator) {}
	    }
	}
    }

    
    # configure member variables

    method config {config} {}

    
    # constructor: load the config file if it exists, or set the default
    # values

    constructor {config} {
	set_defaults
	set initialized_ 1
    }


    # -- public member variables --


    # column headings from table
    public headings {} {
	if {$initialized_} {
	    if {"$headings" != "$saved_headings_"} {
		set_defaults
	    }
	}
    }

    # list of headings, in the order they (and the columns)
    # should be displayed
    public order {}

    # -- sort options --

    # list of col index: sort table based on given columns 
    # (empty means don't sort)
    public sort_by {}

    # set direction of sort: may be one of (increasing, decreasing) 
    public sort_order {increasing}

    # set type of sort key: may be one of (ascii, integer, real) 
    public sort_type {ascii}

    # -- match options --

    # flag: if true, use regular exprs for matching, otherwise use wildcards
    public use_regexp {0} {
	if {$use_regexp} {
	    set match_any_ ".*"
	} else {
	    set match_any_ "*"
	}
    }


    # -- protected members --

    # array(heading,opt) of option values
    protected options_

    # magic string for config files to keep track of versions
    protected magic_ {-- table list config file v1.3 --}

    # previous column headings - read from config_file
    protected saved_headings_ {}

    # set to 1 after initialization
    protected initialized_ 0

    # string used to match any string
    protected match_any_ {*}
}

