
/*
 * RXtRecorder.c --
 *      This module contains the functions to record the actions
 *	of an application suitable for replaying
 *
 * Copyright 1994 Jan Newmarch, University of Canberra.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The author
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <ctype.h>
#include <X11/keysym.h>
#include <X11/Intrinsic.h>

#include "funcs.h"

static long old_time;

/*
 *-----------------------------------------------------------------------------
 * PrintButtonEvent
 *
 *	Print all relevant fields of a button event
 *
 *-----------------------------------------------------------------------------
 */
static void
PrintButtonEvent(event)
    XButtonEvent *event;
{
    printf("\\\n    -x %d -y %d", event->x, event->y);
}

/*
 *-----------------------------------------------------------------------------
 * PrintKeyEvent
 *
 *	Print all relevant fields of a key event
 *
 *-----------------------------------------------------------------------------
 */
static void
PrintKeyEvent(display, event)
    Display *display;
    XKeyEvent *event;
{
#   define BUF_LEN 32
    char buf[BUF_LEN + 1];
    KeySym keysym;
    XComposeStatus status;
    int count;
    Modifiers modifiers;
    unsigned int state;

    printf(" \\\n    -x %d -y %d", event->x, event->y);

/*  This doesn't pick up osf... type keysyms, such as osfEnd, but only
    gives End 
    XtTranslateKey(display, event->keycode, event->state,
		&modifiers, &keysym);
*/

    keysym = XtGetActionKeysym((XEvent *) event, &modifiers);

    printf(" \\\n    -keysym \"%s\"", XKeysymToString(keysym));

    /* print state bits */
    /* ??? when I tried returned modifiers, always got useless
       (Shift | Lock), so just use original event->state
     */
    if ((state = event->state) != 0) {
	 printf(" \\\n    -modifiers { ");

	if (state & ShiftMask)
	    printf("Shift ");
	if (state & LockMask)
	    printf("Lock ");
	if (state & ControlMask)
	    printf("Control ");
	if (state & Mod1Mask)
	    printf("Mod1 ");
	if (state & Mod2Mask)
	    printf("Mod2 ");
	if (state & Mod3Mask)
	    printf("Mod3 ");
	if (state & Mod4Mask)
	    printf("Mod4 ");
	if (state & Mod5Mask)
	    printf("Mod5 ");

	printf("}");
    }
#   undef BUF_LEN
}


/*
 *-----------------------------------------------------------------------------
 * PrintActions
 *
 *	For an action, print it out as a tcl command
 *
 *-----------------------------------------------------------------------------
 */
static void
PrintActions(w, client_data, action_name, event, params, num_params)
    Widget w;
    XtPointer client_data;
    String action_name;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
    int n;
    long cur_time;
    struct timeval tp;

    gettimeofday(&tp, 0);
    cur_time = tp.tv_sec * 1000 + (tp.tv_usec / 1000);
    printf("sleep %ld\n", (cur_time - old_time));
    old_time = cur_time;

    printf("callActionProc {");

    RXt_PrintAlmostFullName(w);

    printf("} %s", action_name);
    putchar('(');

    for (n = 0; n < (int) *num_params - 1; n++) {
	printf("%s, ", params[n]);
    }
    if (*num_params > 0) {
        printf("%s", params[*num_params - 1]);
    }
    puts(") \\");

    printf("    -type ");
    switch (event->type) {
	case ButtonPress:
		printf("ButtonPress");
		PrintButtonEvent((XButtonEvent *) event);
		break;
	case ButtonRelease:
		printf("ButtonRelease");
		PrintButtonEvent((XButtonEvent *) event);
		break;
	case KeyPress:
		printf("KeyPress");
		PrintKeyEvent(XtDisplay(w), (XKeyEvent *) event);
		break;
	case MotionNotify:
		printf("MotionNotify");
		break;
	case EnterNotify:
		printf("EnterNotify");
		break;
	case LeaveNotify:
		printf("LeaveNotify");
		break;
	case FocusIn:
		printf("FocusIn");
		break;
	case FocusOut:
		printf("FocusOut");
		break;
	case KeymapNotify:
		printf("KeymapNotify");
		break;
	case Expose:
		printf(" Expose");
		break;
	case GraphicsExpose:
		printf("GraphicsExpose");
		break;
	case NoExpose:
		printf("NoExpose");
		break;
	case VisibilityNotify:
		printf("VisibilityNotify");
		break;
	case CreateNotify:
		printf("CreateNotify");
		break;
	case DestroyNotify:
		printf("DestroyNotify");
		break;
	case UnmapNotify:
		printf("UnmapNotify");
		break;
	case MapNotify:
		printf("MapNotify");
		break;
	case MapRequest:
		printf("MapRequest");
		break;
	case ReparentNotify:
		printf("ReparentNotify");
		break;
	case ConfigureNotify:
		printf("ConfigureNotify");
		break;
	case ConfigureRequest:
		printf("ConfigureRequest");
		break;
	case GravityNotify:
		printf("GravityNotify");
		break;
	case ResizeRequest:
		printf("ResizeRequest");
		break;
	case CirculateNotify:
		printf("CirculateNotify");
		break;
	case CirculateRequest:
		printf("CirculateRequest");
		break;
	case PropertyNotify:
		printf("PropertyNotify");
		break;
	case SelectionClear:
		printf("SelectionClear");
		break;
	case SelectionRequest:
		printf("SelectionRequest");
		break;
	case SelectionNotify:
		printf("SelectionNotify");
		break;
	case ColormapNotify:
		printf("ColormapNotify");
		break;
	case ClientMessage:
		printf("ClientMessage");
		break;
	case MappingNotify:
		printf("MappingNotify");
		break;
	default:
		printf("Unknown event type\n");
    }
    putchar('\n');
}

/*
 *-----------------------------------------------------------------------------
 * RXt_StartRecorder
 *
 *	start recording to standard output of all actions that occur
 *
 *-----------------------------------------------------------------------------
 */
void
RXt_StartRecorder(app_context)
    XtAppContext app_context;
{
    struct timeval tp;

    gettimeofday(&tp, 0);
    old_time = tp.tv_sec * 1000 + (tp.tv_usec / 1000);
    XtAppAddActionHook(app_context, PrintActions, NULL);
}
