/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

int	dirLoop = 0;

/*
 * otBuildDBfiles.c
 * 
 * Build headdb and histdb files for a project.
 *
 * otBuildDBfiles  [-p<proj>] [-t<num>] [-b<num>] [-d] [file]
 *
 * where:
 *	[-p<proj>]  - OT projectname
 *	[-t<num>]   - top CR number for the range of CRs to process
 *	[-b<num>]   - bottom CR number for the range of CRs to process
 *	[-d]	    - loop by files in d00-d99/d00-d99 directories
 *	[file]	    - one template file to process
 *
 * note:
 *	-t and -b switches are applicable for the loop by CR numbers only.
 *	when the -d switch is used, the -t and -b are ignored. 
 */

#define MAINPROGRAM

#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <pwd.h>
#include <tcl.h>
#include <tclInt.h>
#include "ot.h"
#include "otInt.h"

#ifndef LINT
static char RCSid_ot[] =
    "$RCSfile: otBuildDBfiles.c,v $ $Revision: 1.1.5.3 $ $Date: 1993/12/01 20:50:37 $";
#endif

/* command line options for getopt() */
#define optString "p:t:b:d"     

OTErr	bdbParseArgs();
void	bdbOneTemplateFile();
void	bdbCRnumLoop();
void	bdbDirLoop();


/*
 *			m a i n
 *
 */

main(argc, argv, envp)
int argc;
char ** argv;
char ** envp;
{
    OTErr err;
    OTPrivateCB *pvp;
    OTProject   *prjp;

    /*
     * Initialize TCL, control blocks, read environment variables and
     * preparse command line for project and template file names 
     */
    if ( err = otInitialInit(&argc, argv) ) {
	fprintf(stderr, "otInitialInit: %s\n", otGetPostedMessage() );
	exit(1);
    }

    if ( err = bdbParseArgs(argc, argv) ) {
	fprintf(stderr, "bdbParseArgs: %s\n", otGetPostedMessage() );
	exit(1);
    }

    if ( err = otReadProject( otCB->cb_project ) ) {
	fprintf(stderr, "otReadProject: %s\n", otGetPostedMessage() );
	exit(1);
    }

    pvp = otCB->cb_pcb;
    prjp = pvp->pcb_project;

    if ( err = otReadTopNumber(prjp, &pvp->pcb_topNumber) ) {
	fprintf(stderr, "%s\n", otGetPostedMessage() );
	exit(1);
    }

    if (otCB->cb_templFile)     /* process one template file */
	bdbOneTemplateFile();

    else { 			
	rcsCmdPath();		/* find RCS commands */

	if (!dirLoop)
	    bdbCRnumLoop(); 	/* CR number loop */
	else
	    bdbDirLoop();	/* loop by dirs */
    }

    exit(0);
}



void
bdbDirLoop()
{
    register char *cp;
    long  i, ti;
    int   iloop, oloop;
    char  crname[SHORTSTR];
    char  path[PATHLEN];
    char  rcspath[PATHLEN];
    char  filename[PATHLEN];
    char  rcsfname[PATHLEN];
    char  command[COMMAND];
    struct  stat stat_buf;
    DIR  * dirp;
    struct dirent *dp;
    OTPrivateCB *pvp;
    OTEnterCB   *entp;
    OTProject   *prjp;
    OTErr errCode = OT_SUCCESS;

    pvp = otCB->cb_pcb;
    entp = otCB->cb_ecb;
    prjp = pvp->pcb_project;
    entp->ecb_tStruct = 0;


    /* Get directory for the highest number */

    i = pvp->pcb_topNumber - 1;
    ti = i - ((i / 10000) * 10000);
    oloop = i / 10000;
    iloop = ti / 100;


    /* Loop through data directories */

    for (;  (oloop >= 00) && !errCode;  oloop--)  {

	for (;  (iloop >= 00) && !errCode;  iloop--)  {

	    sprintf(path, "%s/%s/d%02d/d%02d", prjp->proj_dir, prjp->name, 
		oloop, iloop);
	    sprintf(rcspath, "%s/RCS", path);
	    fprintf(stderr, "\n\n%s\n", path);

	    /* Go down the RCS directory */

	    if ( (dirp = opendir(rcspath)) == NULL) {
		otPutPostedMessage(OT_OPENDIR_LOCATION, "bdbDirLoop");
		fprintf(stderr, "\t\t%s\n", otGetPostedMessage() );
		continue;
	    }
	    
	    while (dp = readdir(dirp))  {
		if ( (strncmp(dp->d_name, "c", 1)) ||
		     (!strstr(dp->d_name, ",v")) || 
		     (strlen(dp->d_name) != 9) ) 
		    continue;

		strcpy(crname, dp->d_name);
		if (cp = strrchr(crname, ','))
		    *cp = 0;

		fprintf(stderr, "\n%s",crname);
		sprintf(filename, "%s/%s", path, crname);

		if (stat(filename, &stat_buf) != 0) {

		   /* get a file from the RCS */

		   sprintf(rcsfname, "%s/%s", rcspath, dp->d_name);

		   if (stat(rcsfname, &stat_buf) != 0) {
		      otPutPostedMessage(OT_STAT, rcsfname);
		      fprintf(stderr, "\t\t%s\n", otGetPostedMessage() );
		      continue;
		   }
		  
		   if (!pvp->pcb_rcsCmd[0]) {
		      sprintf(command,
			"no RCS commands found - can't check out %s", crname);
		      otPutPostedMessage(OT_GENERAL_ERROR, command);
		      fprintf(stderr, "\t\t%s\n", otGetPostedMessage() );
		      continue;
		   }

		   sprintf(command, "cd %s; %s/co -q %s\n", path, 
			pvp->pcb_rcsCmd, crname);
		   if (system(command)) {
		      sprintf(command,
		     	"system() error whicle checking out file %s", crname);
		      otPutPostedMessage(OT_GENERAL_ERROR, command);
		      fprintf(stderr, "\t\t%s\n", otGetPostedMessage() );
		      continue;
		   }
		}

		if (entp->ecb_tStruct) {
		    otFreeTemplate(entp->ecb_tStruct);
		    entp->ecb_tStruct = 0;
		}

		if ( errCode = otReadTemplateFromFilename(&entp->ecb_tStruct,
			filename, TRUE, TRUE) ) {
		    fprintf(stderr, "  otReadTemplateFromFilename: --> %s\n",
			otGetPostedMessage() );
		    continue;
		}

		if (errCode = otBuildQueueItem(entp->ecb_tStruct) ) {
		    fprintf(stderr, "  otBuildQueueItem: --> %s\n",
			otGetPostedMessage() );
		    continue;
		}
	    }

	    if (dirp)
		closedir(dirp);
	}

	iloop = 99; /* initialize for the next oloop loop */
    }
    otFreeTemplate(entp->ecb_tStruct);

    return;
}



void
bdbCRnumLoop()
{
    long  i, j;
    int   k, iloop, oloop;
    char  crname[SHORTSTR];
    char  filename[PATHLEN];
    char  rcsfname[PATHLEN];
    char  path[PATHLEN];
    char  command[COMMAND];
    struct  stat stat_buf;
    OTPrivateCB *pvp;
    OTEnterCB   *entp;
    OTQueryCB   *qryp;
    OTProject   *prjp;
    OTErr err;

    pvp = otCB->cb_pcb;
    entp = otCB->cb_ecb;
    qryp = otCB->cb_qcb;
    prjp = pvp->pcb_project;
    entp->ecb_tStruct = 0;

    if (qryp->qcb_hiRange)
	i = qryp->qcb_hiRange;
    else
	i = pvp->pcb_topNumber - 1;

    j = 0;
    if (qryp->qcb_loRange)
	j = qryp->qcb_loRange - 1;

    for ( ; i > j ; i-- ) {
/*
	oloop = i / 10000;
	iloop = i / 100;
*/
	k = i / 100;
	oloop = k / 100;
	iloop = k - oloop*100;

	sprintf(path, "%s/%s/d%02d/d%02d", prjp->proj_dir, prjp->name, 
	    oloop, iloop);
	sprintf(crname, "c%06d", i);

	sprintf(filename, "%s/%s", path, crname);

	DBUG_MIN((stderr, "%s\n", filename));
	fprintf(stderr, "\n%s", filename);

	if (stat(filename, &stat_buf) != 0) {
	    
	    /* get a file from the RCS */

	    sprintf(rcsfname, "%s/RCS/%s,v", path, crname);

	    if (stat(rcsfname, &stat_buf) != 0) {
		otPutPostedMessage(OT_STAT, rcsfname);
		fprintf(stderr, "\t\t%s\n", otGetPostedMessage() );
		continue;
	    }

	    if (!pvp->pcb_rcsCmd[0]) {
		sprintf(command, 
		    "no RCS commands found - can't check out %s", crname);
		otPutPostedMessage(OT_GENERAL_ERROR, command);
		fprintf(stderr, "\t\t%s\n", otGetPostedMessage() );
		continue;
	    }

	    sprintf(command, "cd %s; %s/co -q %s\n", path, pvp->pcb_rcsCmd, 
		crname);
	    if (system(command)) {
		sprintf(command, "system() error whicle checking out file %s", 
		    crname);
		otPutPostedMessage(OT_GENERAL_ERROR, command);
		fprintf(stderr, "\t\t%s\n", otGetPostedMessage() );
		continue;
	    }
	}

	if (entp->ecb_tStruct) {
	    otFreeTemplate(entp->ecb_tStruct);
	    entp->ecb_tStruct = 0;
	}

	if ( err = otReadTemplateFromFilename(&entp->ecb_tStruct, 
		filename, TRUE, TRUE) ) {
	    fprintf(stderr, "  otReadTemplateFromFilename: --> %s\n", 
		otGetPostedMessage() );
	    continue;
	}
	
	if (err = otBuildQueueItem(entp->ecb_tStruct) ) {
	    fprintf(stderr, "  otBuildQueueItem: --> %s\n", 
		otGetPostedMessage() );
	    continue;
	}

#ifdef NATA

    sprintf(command, "otqm -p %s", otCB->cb_project);
    if (system(command)) 
	fprintf(stderr, "otqm --> %s\n", otGetPostedMessage() );
	    
#endif

    }
    otFreeTemplate(entp->ecb_tStruct);

    return;
}



OTErr
bdbParseArgs(argc, argv)
int argc;
char * argv[];
{
    register int  i, opt;
    OTErr errCode = OT_SUCCESS;

    dirLoop = FALSE;
    otCB->cb_qcb->qcb_hiRange = 0;

    while ( (opt = getopt(argc, argv, optString)) != EOF )  {
	switch ( opt )  {

	case 'p':
	    otPutPostedMessage(OT_INTERNAL_ERROR,
			"otPreparseInput while extracting OT projectname");
	    errCode = OT_INTERNAL_ERROR;
	    break;

	case 't':
	    otCB->cb_qcb->qcb_hiRange = atol(optarg);
	    break;

	case 'b':
	    otCB->cb_qcb->qcb_loRange = atol(optarg);
	    break;

	case 'd':
	    dirLoop = TRUE;
	    break;

	case '?':
	    otPutPostedMessage(OT_GENERAL_ERROR, 
"usage:  otBuildDBfiles [-p<projectname>] [-t<topNum [-b<bottomNum>] [file]");
	    errCode = OT_GENERAL_ERROR;
	    break;
	}
    }

    for (i=optind;  (i < argc) && !errCode;  i++)    {
	if (otCB->cb_templFile) {
	    errCode = OT_ONE_FILENAME_ONLY;
	    otPutPostedMessage(OT_ONE_FILENAME_ONLY, optarg);
	}
	else {
	    struct stat stat_buf;

	    if ( !(errCode = otSetCBMember("templFile", argv[i])) &&
	          (strcmp( "-", otCB->cb_templFile)) )  {

	        if (stat(otCB->cb_templFile, &stat_buf) != 0) {
		    errCode = OT_STAT;
		    otPutPostedMessage(OT_STAT, otCB->cb_templFile);
		}
		else if (!(stat_buf.st_mode & S_IFREG)) {
		    errCode = OT_NOT_REGULAR_FILE;
		    otPutPostedMessage(OT_NOT_REGULAR_FILE, otCB->cb_templFile);
		}
	    }
	}
    }

    return errCode;
}



void
bdbOneTemplateFile()
{
    OTErr err;
    OTEnterCB   *entp;


    DBUG_MIN((stderr, "%s\n", otCB->cb_templFile));
    fprintf(stderr, "\n%s", otCB->cb_templFile);

    entp = otCB->cb_ecb;

    if ( err = otReadTemplateFromFilename(&entp->ecb_tStruct,
				otCB->cb_templFile, TRUE, TRUE) ) 	{
	fprintf(stderr, "  otReadTemplateFromFilename: --> %s\n",
	    otGetPostedMessage() );
	return;
    }

    if (err = otBuildQueueItem(entp->ecb_tStruct) ) {
	fprintf(stderr, "  otBuildQueueItem: --> %s\n",
	    otGetPostedMessage() );
    }

    otFreeTemplate(entp->ecb_tStruct);

    return;
}

