/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

#define MAINPROGRAM

static char * rcsid =
 "$RCSfile: ode2ot.c,v $";

#include <string.h>
/* #include <stdlib.h> */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <time.h>
#include <varargs.h>
#include <tcl.h>
#include <tclInt.h>
#include "ot.h"
#include "otInt.h"

/* command line options for getopt() */
#define optString "b:d:e:il:m:p:r"     /* for the "ode2ot" command */

#define CR_CHARS ", 	"
#define CR_BLANK_CHARS " 	"

OTErr	o2oParseArgs();	  /* Parse command line arguments */
OTErr	o2oChkArgs();	  /* Check passed command line arguments */
OTErr	o2oInitRun();	  /* Perform an initialization */
OTErr   o2oGetLastEntry(); /* Get the last processed bsubmit log entry's id */
OTErr   o2oGetExclList();  /* Get project's ode2ot exclusion list */
OTErr   o2oUpdateMarker(); /* Update marker file for the build */
OTErr   o2oProcessBsubmitLog(); /* Process the bsubmit log file */
OTErr	o2oPositionBsubmitLog(); /* Find where it left off the last time */
OTErr   o2oGetNextEntry(); /* Read next bsubmit log entry */
OTErr	o2oProcessOneEntry(); /* Process one bsubmit log entry */
OTErr	o2oInitBlogEntry(); /* Allocate and initialize blog_entry structure */
OTErr	o2oInitCurEntry(); /* Allocate and initialize "curr" structure */
OTErr	o2oAppendLine(); 
OTErr	o2oPutBlogError();  /* Acknowledge corrupted bsubmit log entry */
OTErr	o2oChkCRlist();	  /* Check blogCRlist */
OTErr	o2oProcessOneCR(); /* Process one CR from the blogCRlist */
OTErr	o2oMail();	   /* Send notification for illegal CR number */
OTErr	o2oCopyOpenBlogFile(); /* Open a copy of the blog file */
OTErr	o2oWriteBsubmitErr(); /* Write failed bsubmit.log entry to the bsubmitErr
				file */
int	o2oCompareIDs();  /* Compare bsubmit log entry IDs */
int	cmdBlogValue();   /* Bsubmit log TCL command */
int     o2oBlankCRlist(); /* Check for "all-blank" CRlist */

void	o2oPutErrorInCR(); /* Acknowledge illegal CR number(s) value */
void	o2oTclInitCmds(); /* Initialize TCL commands */
void	o2oClearBlogEntry(); /* Reset the blog_entry structure */
void    o2oCleanAfterRun();  /* Final cleanup */
void    o2oLogMsg();	  /* place an informational message in the log file */

typedef struct 		/* structure to hold bsubmit log entry's id */
{             /* structure to hold bsubmit.log entry's id */
    char    id_name[SHORTSTR];      /* blog person logname */
    char    id_date[SHORTSTR];     /* blog date */
    char    id_time[SHORTSTR];     /* blog time */
} BLogEntryID;

typedef struct 		/* structure to hold one bsubmit log entry */
{           
    char     blogName[SHORTSTR];
    char     blogDate[SHORTSTR];
    char     blogTime[SHORTSTR];
    char     blogFullName[NAMELEN];
    char     blogNfiles[SHORTSTR];
    char     blogCRlist[COMMAND];
    int	     blogCR;
    char     blogSetName[NAMELEN];
    char     blogSandbox[NAMELEN];
    char     blogByLine[NAMELEN];
    char     blogDTLine[NAMELEN];
    char   * blogFiles;
    char   * blogDefunctFiles;
    char   * blogTop;
    char   * blogBottmp;
    char   * blogBottom;
    char     blogBuildName[SHORTSTR]; 
    char     blogMarker[SHORTSTR]; 
    int	     blogFilesTotal;
    int	     blogDefunctFilesTotal;
    int	     blogTopTotal;
    int	     blogBottomTotal;
} BLogEntry;

typedef struct       /* structure to hold a copy of current bsubmit log entry */
{
    char   * curBsubm;
    int      curBsubmLen;
} CurEntry;


char     bsubmitLog[MAXPATHLEN] = "";  /* corresponds to the -l flag */
char	 rmTmpFile[PATHLEN] = "";
char     buildName[SHORTSTR] = "";     /* corresponds to the -b flag */
char     markerName[SHORTSTR] = "";    /* corresponds to the -m flag */
char	 o2oDir[PATHLEN] = "";		/* corresponds to the -d flag */
char	 bsubmitErr[PATHLEN] = "";	/* corresponds to the -e flag */

char    *excludeList = 0;       /* CRs to be excluded from ode2ot updates */
FILE    *o2o_lfp = 0;		/* ode2ot log file */
FILE    *blog_fp = 0;		/* bsubmit log file */
FILE	*bsubmErr_fp = 0;	/* file with failed bsubmit.log entries */
CurEntry   *curr = 0;		/* copy of the current bsubmit.log entry */
BLogEntry  *bl = 0;		/* current bsubmit log entry structure */
BLogEntryID *bl_id = 0; 	/* current blog entry's id */


/* 
 * ode2ot  -l<bsubmit.log file> -b<baseline> -m<marker> -d<directory> 
 *         -e<bsubmitErrors> [-p proj] [-r]
 *
 * The -l specifies the bsubmit.log file (full or relative path)
 * The -b specifies the baseline name
 * The -m specifies the build name (and marker for the +1BSUBMIT record)
 * The -d specifies the ode2ot administration location
 * The -p specifies the OT project name
 * The -e specifies the name of the file to put failed bsubmit.log file entries
 * The -r (restart from the top) indicate to process all entries from the 
 *        bsubmit.log file
 *
 *
 * Old-style 'bsubmit.log' files are missing the previous revision field.
 * They can also be read by bdelta, which determines the number of fields
 * and, in the absence of the previous revision, uses bstat to obtain it
 * or a heuristic rule to guess it.
 *
 * The output:
 * 
 * Corresponding  <o2oDir>/<markerName>   file will be updated/created with 
 * the record ID from the last processed entry from the bsubmit.log file. 
 * The contents of the file prior the run will be saved in the file  
 * <o2oDir>/<markerName>.bak
 *
 */

main ( argc, argv, envp )
int argc;
char ** argv;
char ** envp;
{
    register int i;

    BLogEntryID   prev_id; /* last entry processed during previous run */
    char argbuf[COMMAND];
    char command[COMMAND];
    char o2oUsage[COMMAND];
    char tmp[COMMAND];
    char *cp;

    bool entireFile = FALSE;  /* process an entire bsubmit.log file indicator */
    bool ignoreMarker = FALSE;  /* do not update the marker file */

    OTErr errCode = OT_SUCCESS;
    OTErr tmpErr = OT_SUCCESS;


    sprintf(o2oUsage, "usage: ode2ot %s", 
"-b<baseline> -m<build marker> -d<directory> -l<bsubmit log file> -e<bsubmit err file> [-p<OTproj>] [-r] [i]");

    if ( (argc == 1) || ((argc == 2) && (argv[1][1] == 'h')) ) {
	fprintf(stderr, "%s\n", o2oUsage);
	exit(1);
    }

    /* Save all command line arguments (otInitialInit removes OT projectname) */

    memset(argbuf, 0, COMMAND);
    for (i=0; i < argc; i++) {
	strcat(argbuf, argv[i]);
	strcat(argbuf, " ");
    }

    /* Initialize TCL, preparse command line for project, etc. */

    if ( errCode = otInitialInit(&argc, argv) ) {
	fprintf(stderr, "%s\n", otGetPostedMessage() );
	exit(1);
    }

    /* Parse command line args */

    if (errCode = o2oParseArgs(argc, argv, &entireFile, &ignoreMarker)) {
	otPutPostedMessage(OT_GENERAL_ERROR, o2oUsage);
	fprintf(stderr, "%s\n", otGetPostedMessage() );
	exit(1);
    }

    /* Check ode2ot input */

    if (errCode = o2oChkArgs(o2oUsage)) {
	fprintf(stderr, "%s\n", otGetPostedMessage() );
	exit(1);
    }

    if ( errCode = otReadProject(otCB->cb_project) ) {
	fprintf(stderr, "%s\n", otGetPostedMessage() );
	exit(1);
    }

    /* Initialize ode2ot */

    if (errCode = o2oInitRun(argc, argv, &prev_id, &entireFile, argbuf)) {
	fprintf(stderr, "%s\n", otGetPostedMessage() );
	exit(1);
    }

    /* Process entries from the bsubmit.log file */

    if (errCode = o2oProcessBsubmitLog(&prev_id, entireFile))  {
	if (cp = otGetPostedMessage())
	    fprintf(stderr, "\n%s\n", cp);
    }
    
    if (ignoreMarker == TRUE) {
	sprintf(tmp,
	"New value (%s %s %s) for the marker file '%s/%s' is ignored",
	bl_id->id_name, bl_id->id_date, bl_id->id_time, o2oDir, markerName);
	o2oLogMsg("%s", tmp);
    }
    else {
	if (tmpErr = o2oUpdateMarker()) 
	    fprintf(stderr, "%s\n", otGetPostedMessage() );
    }

    if (!errCode && !tmpErr) {
	o2oLogMsg("---------- ode2ot completed ----------", "");
	fprintf(stderr, "---------- ode2ot completed ----------\n");
    }
    else  {
	o2oLogMsg("----- ode2ot completed with error(s) -----", "");
	fprintf(stderr, "----- ode2ot completed with error(s) -----\n");
    }

    if ( errCode = otBringDownServer() ) {
	cp = otGetPostedMessage();
	fprintf(stderr, "\n%s\n", cp ? cp : "error in server" );
    }

    o2oCleanAfterRun();

    if (errCode || tmpErr)
	exit(1);
    else
        exit(0);
}


OTErr
o2oChkArgs(o2oUsage)
char * o2oUsage;
{
    register char *cp, *cp1;
    char tmp[MAXPATHLEN];
    char tmp1[COMMAND];
    struct stat stat_buf;

    OTErr errCode = OT_SUCCESS;

    /* Check baseline name */

    if (buildName[0] == 0) {
	otPutPostedMessage(OT_GENERAL_ERROR, 
	        "must specify a baseline name with -b<baseline>");
	errCode = OT_GENERAL_ERROR;
    }

    /* Check marker name */

    if (markerName[0] == 0) {
	otPutPostedMessage(OT_GENERAL_ERROR, 
	    "must specify a marker (live build name usially) with -m<marker>");
	errCode = OT_GENERAL_ERROR;
    }

    /* Check bsubmit.log file */

    if (bsubmitLog[0] == 0) {
	otPutPostedMessage(OT_GENERAL_ERROR,
	    "must specify a bsubmit log file with -l<bsubmit.log file>");
	errCode = OT_GENERAL_ERROR;
    }
    else {
	if (stat(bsubmitLog, &stat_buf) != 0)  {
	    otPutPostedMessage(OT_STAT, bsubmitLog);
	    errCode = OT_GENERAL_ERROR;
	}
    }

    /* Check bsubmit error file */

    if (bsubmitErr[0] == 0) {
	otPutPostedMessage(OT_GENERAL_ERROR,
"must specify a file to save failed bsubmit log entries with -e<bsubmitErrors>");
	errCode = OT_GENERAL_ERROR;
    }

    /* Check ode2ot administration directory */

    if (o2oDir[0] == 0) {
	otPutPostedMessage(OT_GENERAL_ERROR,
	    "must specify ode2ot administration location with --d<directory>");
	errCode = OT_GENERAL_ERROR;
    }
    else {
	if (stat(o2oDir, &stat_buf) != 0)  {
	    otPutPostedMessage(OT_NO_DIR, o2oDir);
	    errCode = OT_GENERAL_ERROR;
	}
	else if (bsubmitErr[0]) {
	    sprintf(tmp, "%s/%s", o2oDir, bsubmitErr);
	    if (stat(tmp, &stat_buf) == 0) {
		sprintf(tmp1, "file '%s' already exist", tmp);
	        otPutPostedMessage(OT_GENERAL_ERROR, tmp1);
		errCode = OT_GENERAL_ERROR;
	    }
	}
    }

    /* Check OT project */

    if (!otCB->cb_project) {
	otPutPostedMessage(OT_NEED_PROJECT);
	errCode = OT_NEED_PROJECT;
    }
    return errCode;
}


OTErr
o2oInitRun(argc, argv, prev_id, entireFile, argbuf)
int argc;
char ** argv;
BLogEntryID * prev_id;
bool *entireFile;
char * argbuf;
{
    register int i;
    char   wrkbuf[PATHLEN];
    char   tmp[COMMAND];
    struct stat stat_buf;
    FILE   *recid_fp;	/* file with the log entry id for the build */

    OTErr  errCode = OT_SUCCESS;


    if ( errCode = o2oCopyOpenBlogFile()) /* open a copy of the bsubmit file */
	return errCode;

    o2oTclInitCmds(); /* initialize special ode2ot Tcl commands */

    /* Allocate and initialize bl_id BLogEntryID structure to hold 
       current bsubmit log entry's id */

    if ( (bl_id = (BLogEntryID*) malloc(sizeof(BLogEntryID)) ) == NULL ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "()");
	return OT_MALLOC_LOCATION;
    }
    bl_id->id_name[0] = 0;
    bl_id->id_date[0] = 0;
    bl_id->id_time[0] = 0;


    /* Open ode2ot.log file under o2oDir and log the command line */

    sprintf(otCB->cb_pcb->pcb_logPid, "%ld", (long) getpid());

    sprintf(wrkbuf, "%s/ode2ot.log", o2oDir);
    if ( (o2o_lfp = fopen(wrkbuf, "a")) == NULL) {
	otPutPostedMessage(OT_LOG_APPEND, wrkbuf);
	return OT_LOG_APPEND;
    }

    chmod(wrkbuf, 0666);

    o2oLogMsg("========== ode2ot started ==========", "");
    fprintf(stderr, "========== ode2ot started ==========\n");

    o2oLogMsg("command line  '%s'", argbuf);
    fprintf(stderr, "command line  '%s'\n", argbuf);


    /* Initialize previous run prev_id BLogEntryID structure */

    prev_id->id_name[0] = 0;
    prev_id->id_date[0] = 0;
    prev_id->id_time[0] = 0;

    /* If the file with a record_id for the build does not exist - 
       process the entire bsubmit log file */

    sprintf(wrkbuf, "%s/%s", o2oDir, markerName);
    if ( (recid_fp = fopen(wrkbuf, "r")) == NULL) 
	*entireFile = TRUE;
	    
    if (*entireFile == FALSE)  {
	/* parse the last processed bsubmit log entry for the build */
	if (errCode = (o2oGetLastEntry(recid_fp, prev_id)) ) 
	    sprintf(tmp, "corrupted bsubmit log record id (%s)", wrkbuf);
	    otPutPostedMessage(OT_GENERAL_ERROR, tmp);
    }

    if (recid_fp)
        fclose(recid_fp);

    if (errCode)
	return errCode;

    sprintf(wrkbuf, "marker build name  '%s'", markerName);
    o2oLogMsg("%s", wrkbuf);
    fprintf(stderr, "%s\n", wrkbuf);

    sprintf(wrkbuf, "previous run blog's entry id  '%s;%s;%s'", 
		prev_id->id_name, prev_id->id_date, prev_id->id_time);
    o2oLogMsg("%s", wrkbuf);
    fprintf(stderr, "%s\n", wrkbuf);


    /* Get CR numbers to be excluded from the ode2ot update */

    if (errCode = o2oGetExclList())
	return errCode;

    if (excludeList && *excludeList) {
	o2oLogMsg("exclusion list: '%s'", excludeList);   
	fprintf(stderr, "exclusion list: '%s'\n", excludeList);
    }
    else {
	o2oLogMsg("no exclusion list", "");
	fprintf(stderr, "no exclusion list\n");
    }
    
    return errCode;
}



OTErr
o2oProcessBsubmitLog(prev_id, entireFile)
BLogEntryID * prev_id;
bool  entireFile;
{
    char   tmp[COMMAND];

    int	   stat = FALSE;
    int	   endBLog = FALSE;
    OTErr  tmpErr = OT_SUCCESS;
    OTErr  errCode = OT_SUCCESS;


  DBUG_MIN((stderr, "o2oProcessBsubmitLog: entireFile = %d\n", entireFile));

    if (! entireFile) {

	if (errCode = o2oPositionBsubmitLog(prev_id, &stat)) 
	    return errCode;

	if (!stat)  {
	    sprintf(tmp,
	    	"couldn't locate bsubmit entry (%s;%s;%s) in (%s) log file",
	         prev_id->id_name, prev_id->id_date, prev_id->id_time, 
		 bsubmitLog);
	    otPutPostedMessage(OT_GENERAL_ERROR, tmp);
	    return OT_GENERAL_ERROR;
	}

	o2oLogMsg("last processed blog entry id is located", "");
	fprintf(stderr, "last processed blog entry id is located\n");
    }

    /* Allocate space and initialize BLogEntry entry (bl) */

    if ( (bl = (BLogEntry *) malloc(sizeof(BLogEntry)) ) == NULL ) {
        otPutPostedMessage(OT_MALLOC_LOCATION, "o2oProcessBsubmitLog()");
	return OT_MALLOC_LOCATION;
    }

    if (errCode = o2oInitBlogEntry())
	return errCode;

    if (errCode = o2oInitCurEntry())
	return errCode;

    while (!endBLog) {

        stat = TRUE;
	if ( (tmpErr = o2oGetNextEntry(&stat, &endBLog)) && 
					(tmpErr != OT_GENERAL_ERROR) ) 
	    return tmpErr;

	if (tmpErr)
	    errCode = tmpErr;

	if ( !stat && (tmpErr = o2oWriteBsubmitErr()) )
	    return tmpErr;

	if (stat && !endBLog)  {
	    if ((tmpErr = o2oProcessOneEntry()) && (tmpErr != OT_GENERAL_ERROR))
		return tmpErr;

	    if (tmpErr) {
		errCode = tmpErr;
		if (tmpErr = o2oWriteBsubmitErr())
		    return tmpErr;
	    }
	}

	/* Update bl_id structure with just processed bsubmit log entry's id */

	if ( bl->blogName[0] && bl->blogDate[0] && bl->blogTime[0] ) {
	    strcpy(bl_id->id_name, bl->blogName);
	    strcpy(bl_id->id_date, bl->blogDate);
	    strcpy(bl_id->id_time, bl->blogTime);
	}
    }

    DBUG_MIN((stderr, "o2oProcessBsubmitLog: allocated memory FilesTotal = %d DefunctFilesTotal = %d TopTotal = %d BottomTotal = %d\n", bl->blogFilesTotal, bl->blogDefunctFilesTotal, bl->blogTopTotal, bl->blogBottomTotal));

    return errCode;
}


OTErr
o2oPositionBsubmitLog(prev_id, stat)
BLogEntryID * prev_id;
int	*stat;
{
    enum parse_states { log_id, log_entry, null_state } ;

    enum parse_states log_state;
    char tmp[COMMAND];
    char line[MAXPATHLEN], *lp = line;
    char *cp;
    BLogEntryID  curr_id;

    OTErr  errCode = OT_SUCCESS;

    DBUG_MIN((stderr, "prev_id = |%s;%s;%s|\n", prev_id->id_name, prev_id->id_date, prev_id->id_time));

  /*
   * bsubmit.log format and assumed state transitions
   *
   *						   Start state: null_state
   *  Submission Log *****		           null_state -> log_id
   *
   *    Submitted by <NAME>; User name: <UNAME>    log_id
   *    Date: <DATE>; Time: <TIME>		   log_id
   *    ......
   *    List of files and revisions:		   log_id  -> log_entry
   *    List of defunct files:			   log_id  -> log_entry
   *	......
   *    Detailed description:                      log_entry
   *    End Log *****				   log_entry  -> null_state
   *
   * This machine is adopted from 'bdelta' and represents a primitive kind of
   * input validation.
   */

    curr_id.id_name[0] = 0;
    curr_id.id_date[0] = 0;
    curr_id.id_time[0] = 0;

    log_state = null_state;

    while (!errCode && !(*stat) && 
			(lp = fgets(line, MAXPATHLEN, blog_fp)) != NULL ) {

	DBUG_MED((stderr, "state %d >> %s", log_state, lp));

	switch ( log_state ) {

	    case null_state:

		if ( strstr(lp, "Submission Log *****") == lp ) 
		    log_state = log_id;
		break;

	    case log_id:

		if (strstr(lp, "  List of files and revisions:") == lp ) 
		    log_state = log_entry;
		else if (strstr(lp, "  List of defunct files:") == lp ) 
		    log_state = log_entry;
		else if ( strstr(lp, "End Log *****") == lp ) {
		    log_state = null_state;
		    *stat = o2oCompareIDs(prev_id, &curr_id);
		}
		else if ( strstr(lp, "  Submitted by ") == lp ) {
		    if ( cp=strrchr(lp, '\n'))
			*cp = 0;
		    if ( (cp=strrchr(lp, ':')) && cp++ && cp++)
			strcpy(curr_id.id_name, cp);
		}
		else if ( strstr(lp, "  Date: ") ) {
		    sscanf(lp, "  Date: %[0-9/:]; Time: %s", curr_id.id_date,
			curr_id.id_time);
		}

		break;

	    case log_entry:
		if ( strstr(lp, "End Log *****") == lp ) {
		    log_state = null_state;
		    *stat = o2oCompareIDs(prev_id, &curr_id);
		}
		break;

	    default:
		sprintf(tmp, "state machine transitions error (%d)", 
			log_state);		
		otPutPostedMessage(OT_INTERNAL_ERROR, tmp);
		errCode = OT_INTERNAL_ERROR;
		break;
	}
    }
    return errCode;
}



OTErr
o2oGetNextEntry(stat, end)
int *stat;
int *end;
{
    enum parse_states { log_top, file_list, defunct_list, log_bottom, null_state } ;

    enum parse_states log_state;
    register char *cp, *cp1;
    int  filesRemain, topRemain, bottomRemain, slen, i;

    char tmp[COMMAND];
    char line[MAXPATHLEN], *lp = line;
    int  gotit;

    OTErr errCode = OT_SUCCESS;

  /*
   * bsubmit.log format and assumed state transitions
   *
   *						   Start state: null_state
   *  Submission Log *****		           null_state -> log_top
   *
   *    Submitted by <NAME>; User name: <UNAME>    log_top
   *    Date: <DATE>; Time: <TIME>		   log_top
   *    Number of files: <N>; Defect number: <CR>  log_top
   *    Set name: <SBNAME>; Sandbox <SB>	   log_top
   *    
   *    List of files and revisions:		   log_top    -> file_list
   *    List of defunct files:		   log_top/file_list  -> defunct_list
   *	
   *    Detailed description:           file_list/defunct_list  -> log_bottom
   *
   *    End Log *****				   log_bottom -> null_state
   *
   * This machine is adopted from 'bdelta' and represents a primitive kind of
   * input validation.
   */

    gotit = FALSE;		/* TRUE when bl structure is filled in */
    log_state = null_state;

    o2oClearBlogEntry();
    *curr->curBsubm = 0;	

    while (!errCode && !(gotit) &&
        		(lp = fgets(line, MAXPATHLEN, blog_fp)) != NULL ) {

      DBUG_MED((stderr, "state %d >> %s", log_state, lp));

      switch ( log_state ) {

	case null_state:

	    if (errCode=o2oAppendLine(&curr->curBsubm, &curr->curBsubmLen, lp) )
		return errCode;

	    if ( strstr(lp, "Submission Log *****") == lp )
	        log_state = log_top;
	    break;

	case log_top:

	    if (errCode=o2oAppendLine(&curr->curBsubm, &curr->curBsubmLen, lp) )
		return errCode;

	    if (strstr(lp, "  List of files and revisions:") == lp ) {
	        log_state = file_list;
		if (errCode=o2oAppendLine(&bl->blogTop, &bl->blogTopTotal, lp) )
		    return errCode;
	    }
	    else if (strstr(lp, "  List of defunct files:") == lp ) {
		log_state = defunct_list;
		if (errCode=o2oAppendLine(&bl->blogTop, &bl->blogTopTotal, lp) )
		    return errCode;
	    }
	    else if (strstr(lp, "Submission Log *****") == lp ) {
		strcpy(tmp, "unexpected 'Submission Log' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  Detailed description:") == lp ) {
		log_state = log_bottom; 
		if (errCode = o2oAppendLine(&bl->blogBottmp, 
						&bl->blogBottomTotal, lp) )
		    return errCode;
		strcpy(tmp, "unexpected 'Detailed description' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "End Log *****") == lp ) {
		strcpy(tmp, "unexpected 'End Log' line");
		errCode = o2oPutBlogError(tmp);
	        log_state = null_state;
		gotit = TRUE;
	    }
	    else if ( strstr(lp, "  Submitted by ") == lp ) {
	        if (errCode=o2oAppendLine(&bl->blogTop, &bl->blogTopTotal, lp) )
		    return errCode;
		if ( cp=strrchr(lp, '\n')) 
		    *cp = 0;
		strcpy(bl->blogByLine, lp);
		if ( (cp=strrchr(lp, ':')) && cp++ && cp++)
		    strcpy(bl->blogName, cp);
		if ( cp=strrchr(lp, ';')) {
		    *cp = 0;
		    if ((cp = lp + 15) && *cp)
			strcpy(bl->blogFullName, cp);
		}
	    }
	    else if ( strstr(lp, "  Date: ") ) {
	        if (errCode=o2oAppendLine(&bl->blogTop, &bl->blogTopTotal, lp) )
		    return errCode;
		if ( cp=strrchr(lp, '\n')) 
		    *cp = 0;
		strcpy(bl->blogDTLine, lp);
		sscanf(lp, "  Date: %[0-9/:]; Time: %s", bl->blogDate, 
								bl->blogTime);
	    }
	    else if ( strstr(lp, "  Number of files: ") ) {
	        if (errCode=o2oAppendLine(&bl->blogTop, &bl->blogTopTotal, lp) )
		    return errCode;
		if ( cp=strrchr(lp, '\n'))
		    *cp = 0;
		if ( (cp=strrchr(lp, ':')) && cp++ && cp++ && *cp)
		    strcpy(bl->blogCRlist, cp);
		if (cp=strrchr(lp, ';')) {
		    *cp = 0;
		    if ( (cp=strrchr(lp, ':')) && cp++ && cp++ && *cp ) 
			strcpy(bl->blogNfiles, cp);
		}
	    }
	    else if ( strstr(lp, "  Set name: ") ) {
	        if (errCode=o2oAppendLine(&bl->blogTop, &bl->blogTopTotal, lp) )
		    return errCode;
		if ( cp=strrchr(lp, '\n'))
		    *cp = 0;
		if ( (cp=strrchr(lp, ':')) && cp++ && cp++ && *cp)
		    strcpy(bl->blogSandbox, cp);
		if ( cp=strrchr(lp, ';')) {
		    *cp = 0;
		    if ( (cp=strrchr(lp, ':')) && cp++ && cp++ && *cp)
			strcpy(bl->blogSetName, cp);
		}
	    }
            break;

	case file_list:

	    if (errCode=o2oAppendLine(&curr->curBsubm, &curr->curBsubmLen, lp) )
		return errCode;

	    if ( strstr(lp, "  List of defunct files:") == lp ) {
		log_state = defunct_list;
		if (errCode=o2oAppendLine(&bl->blogTop, &bl->blogTopTotal, lp))
		    return errCode;
	    }
	    else if ( strstr(lp, "  Detailed description:") == lp ) {
		log_state = log_bottom;
		if (errCode = o2oAppendLine(&bl->blogBottmp, 
						&bl->blogBottomTotal, lp) )
		    return errCode;
	    }
	    else if ( strstr(lp, "End Log *****") == lp ) {
		strcpy(tmp, "unexpected 'End Log' line");
		errCode = o2oPutBlogError(tmp);
		log_state = null_state;
		gotit = TRUE;
	    }
	    else if (strstr(lp, "Submission Log *****") == lp ) {
		strcpy(tmp, "unexpected 'Submission Log' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  Submitted by ") == lp ) {
		strcpy(tmp, "unexpected 'Submitted by' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  Date: ") ) {
		strcpy(tmp, "unexpected 'Date and Time' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  Number of files: ") ) {
		strcpy(tmp, "unexpected 'Number of files' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  Set name: ") ) {
		strcpy(tmp, "unexpected 'Set name' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else {
	        if (errCode=o2oAppendLine(&bl->blogTop, &bl->blogTopTotal, lp) )
	            return errCode;
	        if (errCode = o2oAppendLine(&bl->blogFiles, 
						&bl->blogFilesTotal, lp) )
		    return errCode;
	    }
            break;

	case defunct_list:

	    if (errCode=o2oAppendLine(&curr->curBsubm, &curr->curBsubmLen, lp) )
		return errCode;

	    if ( strstr(lp, "  Detailed description:") == lp ) {
		log_state = log_bottom;
		if (errCode = o2oAppendLine(&bl->blogBottmp, 
						&bl->blogBottomTotal, lp) )
		    return errCode;
	    }
	    else if ( strstr(lp, "End Log *****") == lp ) {
		strcpy(tmp, "unexpected 'End Log' line");
		errCode = o2oPutBlogError(tmp);
		log_state = null_state;
		gotit = TRUE;
	    }
	    else if (strstr(lp, "Submission Log *****") == lp ) {
		strcpy(tmp, "unexpected 'Submission Log' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  Submitted by ") == lp ) {
		strcpy(tmp, "unexpected 'Submitted by' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  Date: ") ) {
		strcpy(tmp, "unexpected 'Date and Time' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  Number of files: ") ) {
		strcpy(tmp, "unexpected 'Number of files' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  Set name: ") ) {
		strcpy(tmp, "unexpected 'Set name' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else {
	        if (errCode=o2oAppendLine(&bl->blogTop, &bl->blogTopTotal, lp) )
	            return errCode;
	        if (errCode = o2oAppendLine(&bl->blogDefunctFiles, 
					&bl->blogDefunctFilesTotal, lp) )
		    return errCode;
	    }
            break;

	case log_bottom:

	    if (errCode=o2oAppendLine(&curr->curBsubm, &curr->curBsubmLen, lp) )
		return errCode;

	    if ( strstr(lp, "End Log *****") == lp ) {
		log_state = null_state;
		gotit = TRUE;
	    }
	    else if (strstr(lp, "Submission Log *****") == lp ) {
		strcpy(tmp, "unexpected 'Submission Log' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  Submitted by ") == lp ) {
		strcpy(tmp, "unexpected 'Submitted by' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  Date: ") ) {
		strcpy(tmp, "unexpected 'Date and Time' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  Number of files: ") ) {
		strcpy(tmp, "unexpected 'Number of files' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  Set name: ") ) {
		strcpy(tmp, "unexpected 'Set name' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  List of files and revisions:") == lp ) {
		strcpy(tmp, "unexpected 'List of files' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else if ( strstr(lp, "  List of defunct files:") == lp ) {
		strcpy(tmp, "unexpected 'List of defunct files' line");
		errCode = o2oPutBlogError(tmp);
	    }
	    else {
	        if (errCode = o2oAppendLine(&bl->blogBottmp, 
					&bl->blogBottomTotal, lp) )
	        return errCode;
	    }
            break;

	default:

	    sprintf(tmp, "state machine transitions error (%d)", log_state);
	    otPutPostedMessage(OT_INTERNAL_ERROR, tmp);
	    errCode = OT_INTERNAL_ERROR;
	    break;
       }
    }
    if (!lp) {
	*end =  TRUE;
	return errCode;
    }

    if ( !gotit || errCode || !bl->blogName[0] || !bl->blogDate[0] ||
			      !bl->blogTime[0] ) {
        *stat = FALSE;

	sprintf(tmp, "%s %s %s (CR='%s') -- NOT processed", bl->blogName, 
		bl->blogDate, bl->blogTime, bl->blogCRlist);
	o2oLogMsg("***** %s", tmp);
	fprintf(stderr, "***** %s\n", tmp);
    }

    else {		/* if (gotit)  */

	/* strip trailing new lines from the bl->blogFiles */

	if ( bl->blogFiles[0] ) {
	    i = strlen(bl->blogFiles) -1 ;
	    while ( (i >= 0)  && (bl->blogFiles[i] == '\n') ) {
		bl->blogFiles[i--] = 0;
	    }
	}

	/* strip trailing new lines from the bl->blogDefunctFiles */

	if ( bl->blogDefunctFiles[0] ) {
	    i = strlen(bl->blogDefunctFiles) -1 ;
	    while ( (i >= 0)  && (bl->blogDefunctFiles[i] == '\n') ) {
		bl->blogDefunctFiles[i--] = 0;
	    }
	}

	sprintf(tmp, "%s %s %s (CR='%s')", bl->blogName, bl->blogDate, 
					bl->blogTime, bl->blogCRlist);
	o2oLogMsg(" - %s", tmp);
	fprintf(stderr, "submission: %s\n", tmp);
    }

    return errCode;
}



OTErr
o2oProcessOneEntry()
{
    register char *cp, *cp1;
    int  sargc, len, result, scanFlags;
    char tmp[LONGVALUE]; 
    char list[COMMAND];
    char *sargv[2];
    OTProject *pjp;
    Tcl_Interp *interp;

    OTErr tmpErr = OT_SUCCESS;
    OTErr errCode = OT_SUCCESS;
    interp = otCB->cb_pcb->pcb_interp;
    cp = 0;

    DBUG_MIN((stderr, "o2oProcessOneEntry: %s %s %s (CR='%s')\n", bl->blogName, bl->blogDate, bl->blogTime, bl->blogCRlist));

    pjp = otCB->cb_pcb->pcb_project;

    /* Skip "all-blank" CR submissions */

    if (o2oBlankCRlist())  {
	sprintf(tmp, "\t -- SKIPPED (all-blank CR)");
	o2oLogMsg("%s", tmp);
	fprintf(stderr, "%s\n", tmp);
	return errCode;
    }

    /* Make sure blogCRlist contains digits, "," and white space chars only */

    if (errCode = o2oChkCRlist()) {
	o2oPutErrorInCR();
	return errCode;
    }

    /* Convert bl->blogBottmp to escape unmatched curly braces, etc. */

#ifdef NATA
    DBUG_MIN((stderr, "bl->blogBottmp |%s|\n", bl->blogBottmp));

    if (*bl->blogBottmp && bl->blogBottmp[0]) {
	len = Tcl_ScanElement(bl->blogBottmp, &scanFlags);
	if ( !(bl->blogBottom = malloc(len) ) ) {
	    otPutPostedMessage(OT_MALLOC_LOCATION, "o2oProcessOneEntry()");
	    return OT_MALLOC_LOCATION;
	}
	result = Tcl_ConvertElement(bl->blogBottmp, bl->blogBottom, scanFlags);

        DBUG_MIN((stderr, "bl->blogBottom |%s|\n", bl->blogBottom));
    }
#endif

#ifdef NATA
commented out the following line if Tcl_ConvertElement will be used
#endif
    bl->blogBottom = bl->blogBottmp;

    /* Get one CR number from the blog entry (blogCRlist) */

    strcpy(list, bl->blogCRlist);
    cp = list;
    while (cp && *cp) {   /*  while (cp && *cp && !errCode) */
	for (; cp && *cp && !isdigit(*cp) ; cp++)
	    ;
	for (cp1=cp; cp && *cp && isdigit(*cp) ; cp++)
	    ;
	if (cp && *cp)
	    *cp++ = 0;

	if ( (cp != cp1) && ((bl->blogCR = atoi(cp1)) > 0) )  {

	    if (tmpErr = o2oProcessOneCR()) {
		char *perr;
		perr = otGetPostedMessage();

		if (tmpErr == OT_TEMPLATE_UNCHANGED) {
		    sprintf(tmp, "%s %d: %s", pjp->object_name, bl->blogCR, 
		    	    perr ? perr : "" );
		} else {
		    sprintf(tmp, "Error updating %s %d: %s", 
		    	    pjp->object_name, bl->blogCR, perr ? perr : "" );
		}	
		o2oLogMsg("***** %s", tmp);
		fprintf(stderr, "***** %s\n", tmp);

		/* Have to call "unlock" for all errors because from the
		   otRequestInitiateUpdate() return it is impossible to
		   distinguish the failure to lock CR from other error
		*/
		   
		if (otRemoteTcl("unlock\n"))  {
		  /*		   
		     Don't show "unsuccessful" ulnock results, because it's
		     confusing - ode2ot complains that it can't release CR
		     which wasn't locked in the first place. I had a case when
		     Tim B. was proving that ode2ot has a bug because it 
		     displayed the following:
		        ***** Error updating CR 8680: No such file or directory
		        ***** RCS lock is not released CR 8680: No such file 
				or directory

		     while he was able to update that CR manually 
                  */
		  /*
		    sprintf(tmp, 
		        "RCS lock is not released %s %d: %s", pjp->object_name, 
		         bl->blogCR, interp->result ? interp->result : "" );
		    o2oLogMsg("***** %s", tmp);
		    fprintf(stderr, "***** %s\n", tmp);
		  */
		}

		if (tmpErr != OT_TEMPLATE_UNCHANGED) {	

		    /* don't overwrite fatal errors with OT_GENERAL_ERROR */
		    if ( (!errCode) || (tmpErr != OT_GENERAL_ERROR) ) 
		        errCode = tmpErr;
		}
	    }
	} 
	else  {
	    /* if entries like "238,," should be considerate as an error, 
	       remove the "if (cp != cp1)" condition below
	    */
	    if (cp != cp1) {
	        o2oPutErrorInCR();
	        if (!errCode)
		    errCode = OT_GENERAL_ERROR;
	    }
	}
    }

#ifdef NATA
/* this code is needed if Tcl_ScanElement and Tcl_ConvertElement are used */
    if (bl->blogBottom) {
        free(bl->blogBottom);
        bl->blogBottom = 0;
    }
#endif

    return errCode;
}


OTErr
o2oProcessOneCR()
{
    char *cp;
    char tmp[COMMAND];
    char tmpfile[PATHLEN];
    char deltamsg[LONGVALUE];
    char *subject_line;
    OTFilen fname;
    OTTemplate * tp;
    OTProject *pjp;
    OTPrivateCB *pcp;
    Tcl_Interp *interp;

    char *newCR = 0;
    char *updString = 0;
    OTErr errCode = OT_SUCCESS;

    interp = otCB->cb_pcb->pcb_interp;
    pcp = otCB->cb_pcb;
    pjp = pcp->pcb_project;


    DBUG_MIN((stderr, "o2oProcessOneCR: CR='%d'\n", bl->blogCR));

    /* return if the CR is in the exclusion list */

    sprintf(tmp, "%d", bl->blogCR);

    if (excludeList && *excludeList && 
	    	(otCompList(tmp, excludeList) == OT_SUCCESS) ) {
	DBUG_MIN((stderr, "o2oProcessOneCR: %s is in the exclusion list\n", tmp));
	sprintf(tmp, "\t%s # %d -- in the exclusion list", pjp->object_name, 			bl->blogCR);
	o2oLogMsg("%s", tmp);
	fprintf(stderr, "%s\n", tmp);
	return OT_SUCCESS;
    }

    sprintf(tmp, "\t%s # %d", pjp->object_name, bl->blogCR); /* log CR number */
    o2oLogMsg("%s", tmp);
    fprintf(stderr, "%s\n", tmp);

   /*
    *    Mimic otDoUpdate 
    */
    
    otCB->cb_operation = UPDATE;
    otCB->cb_pcb->pcb_CRNumber = otCB->cb_CRNumber = bl->blogCR;

    /* Create a temporary file to hold the edited version of the template */

    tmpfile[0] = 0;
    tmpnam(tmpfile);
    strcpy(rmTfile, tmpfile);

    if (otCB->cb_ecb->ecb_tStruct) {
	otFreeTemplate(otCB->cb_ecb->ecb_tStruct);
	otCB->cb_ecb->ecb_tStruct = 0;
    }

    if (otCB->cb_ecb->ecb_origStruct) {
	otFreeTemplate(otCB->cb_ecb->ecb_origStruct);
	otCB->cb_ecb->ecb_origStruct = 0;
    }

    if ( errCode = otInitTemplate() ) 
	return OT_GENERAL_ERROR;
    
    otCB->cb_ecb->ecb_origStruct = otCB->cb_ecb->ecb_blankStruct;
    otCB->cb_ecb->ecb_blankStruct = 0;

    /*
     * Is this really necessary, since otDupTemplate() will be called
     * later by otRequestInitiateUpdate()?
     */
    if ( errCode = otDupTemplate( otCB->cb_ecb->ecb_origStruct,
	&(otCB->cb_ecb->ecb_tStruct) ) ) 
	return OT_GENERAL_ERROR;

    if ( errCode = otRequestInitiateUpdate() ) {
	DBUG_MIN((stderr, "o2oProcessOneCR: otRequestInitiateUpdate returned an error: %s\n", interp->result));
	return OT_GENERAL_ERROR;
    }

    /* 
     * Generate and return the bsubmit update script
     */
    if (errCode = otTclEvalProc("generateBsubmitUpdateScript")) {
	if (errCode == OT_TCL_NOPROC) 
	    strcpy(tmp, 
  		"no 'generateBsubmitUpdateScript()' TCL procedure defined");
	else 
	    sprintf(tmp, "%s", interp->result);

	otPutPostedMessage(OT_GENERAL_ERROR, tmp);

	DBUG_MIN((stderr, "o2oProcessOneCR: error in generateBsubmitUpdateScript: %s\n", interp->result));

	return OT_GENERAL_ERROR;
    }

    if ( otCopyString( interp->result, &updString) ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "o2oProcessOneCR()");
	return OT_GENERAL_ERROR;
    }

    DBUG_MIN((stderr, "o2oProcessOneCR: generateBsubmitUpdateScript returned:\n\t%s\n", updString));

    /* Send the bsubmit update string to update the object on a server */
	
    errCode = otRemoteTcl(updString);
    DBUG_MIN((stderr, "sent >> %s\n", updString));

    free(updString);
    *updString = 0;

    if (errCode) {
	DBUG_MIN((stderr, "o2oProcessOneCR: otRemoteTcl returned: %s\n", interp->result));
	otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	return OT_GENERAL_ERROR;
    }
    else if (*interp->result) {		/* return tcl results to the user */
	fprintf(stderr, "%s %d: %s\n", pjp->object_name, bl->blogCR, interp->result);
    }

    /* Call cmdCommit to do: validation, postProcess, store an object under 
       RCS, build queue item and start otqm */

    if (errCode = otRemoteTcl("commit\n")) {
	otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	return errCode;
    } else if ( *interp->result ) {
	/* return informational PostProcess messages to the user */
	fprintf(stderr, "%s %d: %s\n", 
		pjp->object_name, bl->blogCR, interp->result);
    }
    DBUG_MIN((stderr, "sent >> commit\n"));

    if (errCode = otRemoteTcl("tclString\n")) {
	otPutPostedMessage(OT_GENERAL_ERROR, interp->result);
	return OT_GENERAL_ERROR;
    } 

    if ( otCopyString(interp->result, &newCR) ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "o2oProcessOneCR()");
	return OT_GENERAL_ERROR;
    }

    DBUG_MIN((stderr, "received >> %s\n", newCR));
    errCode = otTclEval(newCR);
    free(newCR);
    *newCR = 0;

    if (errCode)
    	return OT_GENERAL_ERROR;
    
    /*
     * Call otNotify 
     */

    strcpy(fname.dir, tmpfile);
    cp = strrchr(fname.dir, '/');
    cp++;
    *cp = 0;

    cp = strrchr(tmpfile, '/');
    cp++;
    strcpy(fname.name, cp);

    tp = otCB->cb_ecb->ecb_tStruct;

    if ( errCode = otWriteTemplateToFilename( tp, tmpfile ) )
	return OT_GENERAL_ERROR;	

    if (errCode = otFormDeltamsg(tmpfile, deltamsg, FALSE)) 
	return OT_GENERAL_ERROR;

    /* Have to refresh the tp pointer because otFormDeltamsg() calls
       otFreeTemplate(otCB->cb_ecb->ecb_tStruct) --> frees the ecb_tStruct
       and then calls otReadTemplateFromFilename(&(otCB->cb_ecb->ecb_tStruct)..)
       which allocates new template structure for the otCB->cb_ecb->ecb_tStruct
    */
    tp = otCB->cb_ecb->ecb_tStruct;

    /* pass "otCB->cb_ecb->ecb_origStruct" instead of NULL, so that the status
       change will be reflected in the notification mail Subject line (CR #550)
    */
    subject_line = otGetSubjectLine(otCB->cb_ecb->ecb_origStruct, 
				tp, fname, pcp->pcb_operation, pjp);

    otNotify(pjp, tp, fname, fname, otCB->cb_operation,	pcp->pcb_notify,
	deltamsg, subject_line);

    otCleanup();
    return errCode;
}



OTErr
o2oChkCRlist()
{
    register char *cp;
    int i;

    for (i=0, cp=bl->blogCRlist; cp[i]; i++) {
	if (!isdigit(cp[i]) && !strchr(CR_CHARS, cp[i]) )
	    return OT_GENERAL_ERROR;
    }
    return OT_SUCCESS;
}



o2oBlankCRlist()
{
    register char *cp;
    int i;

    if (!bl->blogCRlist[0])
	return TRUE;
    
    cp = bl->blogCRlist;

    if (cp[strlen(cp) - 1] == '.')
	cp[strlen(cp) - 1] = 0;

    for (i=0; cp[i]; i++) {
	if (!strchr(CR_BLANK_CHARS, cp[i]) )
	    return FALSE;
    }
    return TRUE;
}



int
o2oCompareIDs(prev, curr)
BLogEntryID * prev;
BLogEntryID * curr;
{
    int stat = FALSE;

    if ( (!strcmp(prev->id_name, curr->id_name)) &&
	 (!strcmp(prev->id_date, curr->id_date)) &&
	 (!strcmp(prev->id_time, curr->id_time)) ) 
	
	stat = TRUE;

    curr->id_name[0] = 0;
    curr->id_date[0] = 0;
    curr->id_time[0] = 0;

    return stat;
}



OTErr
o2oParseArgs(argc, argv, entireFile, ignoreMarker)
int argc;
char * argv[];
bool * entireFile;
bool * ignoreMarker;
{
    register int  opt;
    OTErr errCode = OT_SUCCESS;

    while ( (opt = getopt(argc, argv, optString)) != EOF )  {

	switch ( opt )  {

	case 'b':
	    if (optarg[0] == '-') {
		otPutPostedMessage(OT_GENERAL_ERROR, 
			"the -b switch requires baseline name");
		errCode = OT_GENERAL_ERROR;
	    }
	    else {
		if (buildName[0] == 0)
		    strcpy(buildName, optarg);
		else {
		    otPutPostedMessage(OT_GENERAL_ERROR, 
			    "no multiple -b switch occurence is allowed");
		    errCode = OT_GENERAL_ERROR;
		} 
	    }
	    break;

	case 'd':
	    if (optarg[0] == '-') {
		otPutPostedMessage(OT_GENERAL_ERROR,
		    "the -d switch requires directory path");
		errCode = OT_GENERAL_ERROR;
	    }
	    else {
		if (o2oDir[0] == 0)
		    strcpy(o2oDir, optarg);
		else {
		    otPutPostedMessage(OT_GENERAL_ERROR,
			"no multiple -d switch occurence is allowed");
		    errCode = OT_GENERAL_ERROR;
		}
	    }
	    break;

	case 'e':
	    if (optarg[0] == '-') {
		otPutPostedMessage(OT_GENERAL_ERROR,
	      "the -e switch requires filename for failed bsubmit log enrties");
		errCode = OT_GENERAL_ERROR;
	    }
	    else {
		if (bsubmitErr[0] == 0)
		    strcpy(bsubmitErr, optarg);
		else {
		    otPutPostedMessage(OT_GENERAL_ERROR,
			"no multiple -e switch occurence is allowed");
		    errCode = OT_GENERAL_ERROR;
		}
	    }
	    break;

        case 'i':       /* do not update marker file with the new valus */
	    *ignoreMarker = TRUE;
	    break;


	case 'l':
	    if (optarg[0] == '-') {
	        otPutPostedMessage(OT_GENERAL_ERROR, 
			"the -l switch requires bsubmit.log file");
		errCode = OT_GENERAL_ERROR;
	    }
	    else {
		if (bsubmitLog[0] == 0)
		    strcpy(bsubmitLog, optarg);
		else {
		    otPutPostedMessage(OT_GENERAL_ERROR, 
			    "no multiple -l switch occurence is allowed");
		    errCode = OT_GENERAL_ERROR;
		}
	    }
	    break;


	case 'm':
	    if (optarg[0] == '-') {
		otPutPostedMessage(OT_GENERAL_ERROR,
		    "the -m switch requires marker name (live build)");
		errCode = OT_GENERAL_ERROR;
	    }
	    else {
		if (markerName[0] == 0)
		    strcpy(markerName, optarg);
		else {
		    otPutPostedMessage(OT_GENERAL_ERROR,
			"no multiple -m switch occurence is allowed");
		    errCode = OT_GENERAL_ERROR;
		}
	    }
	    break;

	case 'p':
    	    otPutPostedMessage(OT_INTERNAL_ERROR, 
		    "otPreparseInput while extracting OT projectname");
	    errCode = OT_INTERNAL_ERROR;
	    break;

        case 'r':       /* process all entries from the bsubmit.log file */
	    *entireFile = TRUE;
	    break;

	case '?':
	    errCode = OT_GENERAL_ERROR;
	    break;
	}
    }

    if (optind != argc)		/* there are remaining args */
	errCode = OT_GENERAL_ERROR;

    return errCode;
}


OTErr
o2oGetLastEntry(recid_fp, prev_id)
FILE   *recid_fp;   
BLogEntryID * prev_id;
{
    register int i, ch;
    OTErr  errCode = OT_SUCCESS;
	    
    /* Parse the last processed bsubmit log entry's id for the build.
	The log entry id consist of three pieces separated by semicolon:
	name, date and time.
    */

    for (i=0; ((ch=getc(recid_fp)) != EOF) && (ch != '\n') && (ch != ';'); i++)
	prev_id->id_name[i] = ch;
    prev_id->id_name[i] = 0;

    if (ch != ';')
	return OT_GENERAL_ERROR;

    for (i=0; ((ch=getc(recid_fp)) != EOF) && (ch != '\n') && (ch != ';'); i++)
	prev_id->id_date[i] = ch;
    prev_id->id_date[i] = 0;

    if (ch != ';')
	return OT_GENERAL_ERROR;

    for (i=0; ((ch=getc(recid_fp)) != EOF) && (ch != '\n') && (ch != ';'); i++)
	prev_id->id_time[i] = ch;
    prev_id->id_time[i] = 0;

    if (!prev_id->id_name[0] || !prev_id->id_date[0] || !prev_id->id_time[0])
	errCode = OT_GENERAL_ERROR;

    return errCode;
}



OTErr
o2oGetExclList()
{
    char *list;
    FILE *fp;
    struct stat stat_buf;
    register int ch, i;
    char wrkbuf[PATHLEN];
    char tmp[COMMAND];
    char line[NAMELEN];
    char *cp;

    OTErr   errCode = OT_SUCCESS;

    sprintf(wrkbuf, "%s/ode2ot.exclude", o2oDir);
    if (stat(wrkbuf, &stat_buf) != 0)  {
	/* there is no CR exclusion list file for the project */
	return errCode;
    }
    
    if ( (list = (char *)calloc(stat_buf.st_size + 10, sizeof(char))) == 0 ) {
        otPutPostedMessage(OT_MALLOC_LOCATION, "readExclList()");
	return OT_MALLOC_LOCATION;
    }

    if ( ! (fp = fopen(wrkbuf, "r")) ) {
	sprintf(tmp, "can't open file (%s)", wrkbuf);
	otPutPostedMessage(OT_GENERAL_ERROR, tmp);
	return OT_GENERAL_ERROR;
    }

    while ( (ch = getc(fp)) != EOF) {
	for (i=0;  (ch != '\n') && (ch != EOF);  i++) {
	    line[i] = ch;
	    ch = getc(fp);
	}
	line[i] = 0;
	otTrimWs(&line[0]);
	if (!line[0])
	    continue;
	strcat(list, line);
	strcat(list, ",");
    }

    fclose(fp);

    if (strlen(list)) {
        cp = list + strlen(list) - 1;
        if (*cp == ',')
            *cp = 0;
	excludeList = list;
    }
    return errCode;
}



OTErr
o2oUpdateMarker()
{
    char   tmp[COMMAND];
    char   wrkbuf[COMMAND];
    struct stat stat_buf;
    FILE   *fp;

    OTErr  errCode = OT_SUCCESS;

    if (!bl_id->id_name[0] || !bl_id->id_date[0] || !bl_id->id_time[0]) {
	sprintf(tmp, 
       "Build's marker file '%s/%s' is not updated (blog entry id '%s %s %s')", 
    	    o2oDir, markerName, bl_id->id_name, bl_id->id_date, bl_id->id_time);
	o2oLogMsg("***** %s", tmp);
	otPutPostedMessage(OT_GENERAL_ERROR, tmp);
	return OT_GENERAL_ERROR;
    }
    
    sprintf(tmp, "Last processed blog entry id  '%s;%s;%s'", bl_id->id_name, 
		bl_id->id_date, bl_id->id_time);
    o2oLogMsg("%s", tmp);
    fprintf(stderr, "%s\n", tmp);


    /* If the marker file for a build exist, move it into markerName.bak */

    sprintf(tmp, "%s/%s", o2oDir, markerName);
    if ((stat(tmp, &stat_buf) == 0) && (stat_buf.st_mode & S_IFREG)) {
        sprintf(wrkbuf, "%s.bak", tmp);
	if (rename(tmp, wrkbuf)) {
	    sprintf(wrkbuf, 
		"can't move blog's marker file '%s' into '%s.bak'", tmp, tmp);
	    o2oLogMsg("%s", wrkbuf);
	    otPutPostedMessage(OT_GENERAL_ERROR, wrkbuf);
	    return OT_GENERAL_ERROR;
    	}
    }

    /* Put the last processed blog entry id into the marker file */

    if ( (fp = fopen(tmp, "w")) == NULL ) {
	sprintf(wrkbuf, "can't create marker file (%s)", tmp);
	o2oLogMsg("%s", wrkbuf);
	otPutPostedMessage(OT_GENERAL_ERROR, wrkbuf);
	errCode = OT_GENERAL_ERROR;
    }
    else {
	sprintf(wrkbuf, "%s;%s;%s\n", 
	    bl_id->id_name, bl_id->id_date, bl_id->id_time);
	fputs(wrkbuf, fp);
    }
    if (fp)
	fclose(fp);

    return errCode;
}



OTErr
o2oAppendLine(cpp, totlen, line)
char ** cpp;
int  *totlen;
char *line;
{
    OTErr errCode = OT_SUCCESS;

    if ( (*totlen - strlen(*cpp) - 1 ) <= strlen(line) ) {
	if ( !(*cpp = realloc( *cpp, *totlen + LONGVALUE )) ) {
	    otPutPostedMessage(OT_MALLOC_LOCATION, "o2oAppendLine()" );
	    (void)free(*cpp);
	    return OT_MALLOC_LOCATION;
	}
	else 
	    *totlen = *totlen + LONGVALUE;
    }
    strcat(*cpp, line);

    return errCode;
}


OTErr
o2oPutBlogError(msg)
char  *msg;
{
    char tmp[COMMAND];

    sprintf(tmp, "%s in the blog entry '%s %s %s' (CR='%s')", msg, 
    		bl->blogName, bl->blogDate, bl->blogTime, bl->blogCRlist);

    o2oLogMsg("** %s ", tmp);
    fprintf(stderr, "** %s\n", tmp);

    return OT_GENERAL_ERROR;
}


void
o2oPutErrorInCR()
{
    char tmp[COMMAND];
    char tmp1[COMMAND];
    OTProject *pjp;

    pjp = otCB->cb_pcb->pcb_project;

    sprintf(tmp, "Illegal value for %s number(s)", pjp->object_name);
    sprintf(tmp1, "%s '%s' in the blog entry '%s %s %s'", tmp,
		bl->blogCRlist, bl->blogName, bl->blogDate, bl->blogTime);
    o2oLogMsg("***** %s ", tmp1);
    fprintf(stderr, "***** %s\n", tmp1);

    if (o2oMail(tmp) != OT_SUCCESS) {
	sprintf(tmp, 
	       "Failed to send error notification mail (blog entry '%s %s %s')",
		bl->blogName, bl->blogDate, bl->blogTime);
	o2oLogMsg("***** %s ", tmp);
    }
    return;
}



void
o2oTclInitCmds()
{
    Tcl_Interp *interp;

    interp = otCB->cb_pcb->pcb_interp;

    Tcl_CreateCommand(interp, "blogName", cmdBlogValue, (ClientData)"blogName",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "blogDate", cmdBlogValue, (ClientData)"blogDate",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "blogTime", cmdBlogValue, (ClientData)"blogTime",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "blogCRlist", cmdBlogValue, 
	(ClientData)"blogCRlist", (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "blogCR", cmdBlogValue, (ClientData)"blogCR",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "blogFiles", cmdBlogValue, 
	(ClientData)"blogFiles", (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "blogDefunctFiles", cmdBlogValue,
	(ClientData)"blogDefunctFiles", (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "blogTop", cmdBlogValue, (ClientData)"blogTop",
	(Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "blogBottom", cmdBlogValue, 
	(ClientData)"blogBottom", (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "blogBuildName", cmdBlogValue,
	(ClientData)"blogBuildName", (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "blogMarker", cmdBlogValue,
	(ClientData)"blogMarker", (Tcl_CmdDeleteProc *)0);

    Tcl_CreateCommand(interp, "blogNfiles", cmdBlogValue, 
	(ClientData)"blogNfiles", (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "blogSetName", cmdBlogValue, 
	(ClientData)"blogSetName", (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "blogSandbox", cmdBlogValue, 
	(ClientData)"blogSandbox", (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "blogByLine", cmdBlogValue, 
	(ClientData)"blogByLine", (Tcl_CmdDeleteProc *)0);
    Tcl_CreateCommand(interp, "blogDTLine", cmdBlogValue, 
	(ClientData)"blogDTLine", (Tcl_CmdDeleteProc *)0);

    return;
}



int
cmdBlogValue(clientData, interp, argc, argv)
ClientData *clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    char *blog_val;
    char tmp[SHORTSTR];
    char errStr[NAMELEN];

    if (argc != 1) {
	sprintf(errStr, "command '%s' takes no arguments\n", argv[0]);
	Tcl_SetResult(interp, errStr, TCL_VOLATILE);
	return TCL_ERROR;
    }

   DBUG_MED((stderr, "cmdBlogValue: argc = %d, argv[0] = %s\n", argc, argv[0]));

    if (!bl) {
	sprintf(errStr, "there is no BLogEntry structure\n");
	Tcl_SetResult(interp, errStr, TCL_VOLATILE);
	return TCL_ERROR;
    }

    blog_val = 0;

    if ( !strcmp(argv[0], "blogName") ) {
	if ( !bl->blogName[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogName;
    }
    else if ( !strcmp(argv[0], "blogDate") ) {
	if ( !bl->blogDate[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogDate;
    }
    else if ( !strcmp(argv[0], "blogTime") ) {
	if ( !bl->blogTime[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogTime;
    }
    else if ( !strcmp(argv[0], "blogCRlist") ) {
	if ( !bl->blogCRlist[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogCRlist;
    }
    else if ( !strcmp(argv[0], "blogCR") ) {
	if ( !bl->blogCR )
	    blog_val = "";
	else {
	    sprintf(tmp, "%d", bl->blogCR);
	    blog_val = tmp;
	}
    }
    else if ( !strcmp(argv[0], "blogFiles") ) {
	if ( !bl->blogFiles || !bl->blogFiles[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogFiles;
    }
    else if ( !strcmp(argv[0], "blogDefunctFiles") ) {
	if ( !bl->blogDefunctFiles || !bl->blogDefunctFiles[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogDefunctFiles;
    }
    else if ( !strcmp(argv[0], "blogTop") ) {
	if ( !bl->blogTop || !bl->blogTop[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogTop;
    }
    else if ( !strcmp(argv[0], "blogBottom") ) {
	if ( !bl->blogBottom || !bl->blogBottom[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogBottom;
    }
    else if ( !strcmp(argv[0], "blogBuildName") ) {
	if ( !bl->blogBuildName || !bl->blogBuildName[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogBuildName;
    }
    else if ( !strcmp(argv[0], "blogMarker") ) {
	if ( !bl->blogMarker || !bl->blogMarker[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogMarker;
    }
    else if ( !strcmp(argv[0], "blogNfiles") ) {
	if ( !bl->blogNfiles[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogNfiles;
    }
    else if ( !strcmp(argv[0], "blogSetName") ) {
	if ( !bl->blogSetName[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogSetName;
    }
    else if ( !strcmp(argv[0], "blogSandbox") ) {
	if ( !bl->blogSandbox[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogSandbox;
    }
    else if ( !strcmp(argv[0], "blogByLine") ) {
	if ( !bl->blogByLine[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogByLine;
    }
    else if ( !strcmp(argv[0], "blogDTLine") ) {
	if ( !bl->blogDTLine[0] )
	    blog_val = "";
	else
	    blog_val = bl->blogDTLine;
    }
    else {
	sprintf(errStr, "unknown command '%s'\n", argv[0]);
	Tcl_SetResult(interp, errStr, TCL_VOLATILE);
	return TCL_ERROR;
    }

    if ( !blog_val)
	Tcl_SetResult(interp, "", TCL_VOLATILE);
    else
	Tcl_SetResult(interp, blog_val, TCL_VOLATILE);
    return TCL_OK;
}



OTErr
o2oMail(msg)
char *msg;
{
    char subject[COMMAND];
    char maillist[COMMAND];
    char mailcmd[NAMELEN];
    char command[COMMAND];
    char tmp[COMMAND];
    struct stat stat_buf;
    FILE *mail_fp;

    OTErr errCode = OT_SUCCESS;
    OTProject * proj = otCB->cb_pcb->pcb_project;


    DBUG_MIN((stderr, "o2oMail: CRlist is '%s' blogID is '%s %s %s'\n", bl->blogCRlist, bl->blogName, bl->blogDate, bl->blogTime));

    maillist[0] = 0;
    sprintf(subject, "ode2ot: %s", msg);

    if (proj->blogMlist[0] && (proj->blogMlist[0] != '#') ) { 
	strcpy(maillist, proj->blogMlist); 

	if (proj->blogSubmitter && bl->blogName[0]) {
	    sprintf(tmp, "%s %s", maillist, bl->blogName);
	    strcpy(maillist, tmp);
	}

	if (!stat("/usr/bin/mailx", &stat_buf))
	    strcpy(mailcmd, "mailx");
	else
	    strcpy(mailcmd, "Mail");

	sprintf(command, "%s -s '%s' %s", mailcmd, subject, maillist);

	if ( (mail_fp = popen(command, "w")) == NULL ) {
	    sprintf(tmp, "error during popen() in o2oMail()");
	    otPutPostedMessage(OT_GENERAL_ERROR, tmp);
	    return OT_POPEN;
	}

	fprintf(mail_fp, "%s: '%s'\n%s '%s' (build marker %s)\n\n\n", 
	    msg, bl->blogCRlist,
	    "in the following bsubmit log entry for the baseline", 
	    buildName, markerName);
	fprintf(mail_fp, "%s\n", bl->blogTop);
	fprintf(mail_fp, "%s\n", bl->blogBottmp);
	fputc(EOF, mail_fp);
	pclose(mail_fp);
    }
    return errCode;
}



OTErr
o2oCopyOpenBlogFile()
{
    char  command[COMMAND];
    char  tmp[COMMAND];
    char  tmpname[PATHLEN];
    char *tmpfile;

    OTErr errCode = OT_SUCCESS;

    /* Copy bsubmit log file into a temporaty file */

    tmpfile = tmpname; 
    tmpfile = tempnam("", "");

    DBUG_MED((stderr, "o2oCopyOpenBlogFile: temporary file is %s\n", tmpfile));

    sprintf(command, "cp %s %s", bsubmitLog, tmpfile);
    if (system(command)) {
	sprintf(tmp, "can't copy bsubmit log file (%s) to temporary file (%s)",
		bsubmitLog, tmpfile);
	otPutPostedMessage(OT_GENERAL_ERROR, tmp);
	return OT_GENERAL_ERROR;
    }
    strcpy(rmTmpFile, tmpfile);

    /* Open a copy of the bsubmit log file */

    if ( (blog_fp = fopen(tmpfile, "r" )) == NULL ) {
	sprintf(tmp, "can't open file (%s)", tmpfile);
	otPutPostedMessage(OT_GENERAL_ERROR, tmp);
	errCode = OT_GENERAL_ERROR;
    }

    return errCode;
}



OTErr
o2oWriteBsubmitErr()
{
    char fname[MAXPATHLEN];
    char tmp[MAXPATHLEN];

    OTErr errCode = OT_SUCCESS;

    if (!bsubmErr_fp) {
	sprintf(fname, "%s/%s", o2oDir, bsubmitErr);
	if ( (bsubmErr_fp = fopen(fname, "w")) == NULL ) {
	    sprintf(tmp, "can't open bsubmit error file (%s)", fname);
	    otPutPostedMessage(OT_GENERAL_ERROR, tmp);
	    return OT_GENERAL_ERROR;
	}
    }

    fprintf(bsubmErr_fp, "%s", curr->curBsubm);
    fflush(bsubmErr_fp);

    return errCode;
}



void
o2oCleanAfterRun()
{

    if (excludeList)
        free(excludeList);

    if (bl) {
	if (bl->blogFiles)
	    free(bl->blogFiles);
	if (bl->blogDefunctFiles)
	    free(bl->blogDefunctFiles);
    	if (bl->blogTop)
	    free(bl->blogTop);
    	if (bl->blogBottmp)
	    free(bl->blogBottmp);
    	if (bl->blogBottom)
	    free(bl->blogBottom);
        free(bl);
    }

    if (curr) {
	if (curr->curBsubm)
	    free(curr->curBsubm);
	free(curr);
    }

    if (blog_fp != 0)
        fclose(blog_fp);

    if (rmTmpFile[0]) {
	unlink(rmTmpFile);        /* temp file to remove */
	strcpy(rmTmpFile, "");
    }

    if (o2o_lfp != 0)
        fclose(o2o_lfp);

    if (bsubmErr_fp != 0)
	fclose(bsubmErr_fp);

    return;
}


void
o2oClearBlogEntry()
{

    bl->blogName[0] = 0;
    bl->blogDate[0] = 0;
    bl->blogTime[0] = 0;
    bl->blogFullName[0] = 0;
    bl->blogNfiles[0] = 0;
    bl->blogCRlist[0] = 0;
    bl->blogCR = 0;
    bl->blogSandbox[0] = 0;
    bl->blogSetName[0] = 0;
    bl->blogByLine[0] = 0;
    bl->blogDTLine[0] = 0;
    *bl->blogFiles = 0;
    *bl->blogDefunctFiles = 0;
    *bl->blogTop = 0;
    *bl->blogBottmp = 0;
    bl->blogBottom = 0;

    return;
}



OTErr
o2oInitBlogEntry()
{
    char * cp;
    OTErr errCode = OT_SUCCESS;

    if ( !(cp = malloc(LONGVALUE)) ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "o2oInitBlogEntry()");
	return OT_MALLOC_LOCATION;
    }
    bl->blogFiles = cp;
    bl->blogFilesTotal = LONGVALUE;

    if ( !(cp = malloc(LONGVALUE)) ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "o2oInitBlogEntry()");
	return OT_MALLOC_LOCATION;
    }
    bl->blogDefunctFiles = cp;
    bl->blogDefunctFilesTotal = LONGVALUE;

    if ( !(cp = malloc(LONGVALUE)) ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "o2oInitBlogEntry()");
	return OT_MALLOC_LOCATION;
    }
    bl->blogTop = cp;
    bl->blogTopTotal = LONGVALUE; 

    if ( !(cp = malloc(LONGVALUE)) ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "o2oInitBlogEntry()");
	return OT_MALLOC_LOCATION;
    }
    bl->blogBottmp = cp;
    bl->blogBottomTotal = LONGVALUE;

    o2oClearBlogEntry();
    strcpy(bl->blogBuildName, buildName);
    strcpy(bl->blogMarker, markerName);

    return errCode;
}



OTErr
o2oInitCurEntry()
{

    if ( (curr = (CurEntry *) malloc(sizeof(CurEntry)) ) == NULL ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "o2oInitCurEntry()");
	return OT_MALLOC_LOCATION;
    }

    if ( !(curr->curBsubm = malloc(LONGVALUE)) ) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "o2oInitCurEntry()");
	return OT_MALLOC_LOCATION;
    }
    curr->curBsubmLen = LONGVALUE;
    *curr->curBsubm = 0;

    return OT_SUCCESS;
}



/*  o2oLogMsg()  --  place a message to the log file
 *
 */
void
o2oLogMsg(msg, arg)
char * msg;     /* message to send */
char * arg;     /* an optional arg */
{
    long timer;
    struct tm * timeptr;

    timer = time(0L);
    timeptr = localtime(&timer);
    fprintf(o2o_lfp, "%s (%d/%d/%d %d:%d %s): ", otCB->cb_pcb->pcb_logPid,
	timeptr->tm_mon +1, timeptr->tm_mday, timeptr->tm_year, 
	timeptr->tm_hour, timeptr->tm_min, otCB->cb_pcb->pcb_uName);
    fprintf(o2o_lfp, msg, arg);
    fprintf(o2o_lfp, "\n");
    fflush(o2o_lfp);

    return;
}

