/*
 *=============================================================================
 *                                  tSippRLE.c
 *-----------------------------------------------------------------------------
 * Tcl commands to manipulate and render to Utah Raster Toolkit RLE files.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1995 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * Our own functions are used to manage comments and other aspects of the RLE
 * header to prevent memory lose that occurs in the RLE library.
 *-----------------------------------------------------------------------------
 * $Id: tSippRLE.c,v 5.9 1996/05/12 23:06:51 markd Exp $
 *=============================================================================
 */

#include "tSippInt.h"

#ifdef TSIPP_HAVE_RLE

#include "rle.h"

/*
 * Check if this is the 3.1 beta release of URT.
 */
#ifdef RLE_INIT_MAGIC
#   define URT_3_1b
#endif


/*
 * Type used to hold a file handle "fileNNN" or RLE file handle "rleNNN".
 */
typedef char fileHandle_t [10];

/*
 * RLE file table entry.  Note that name is redundent with RLE 3.1, as its
 * in the RLE header.
 */
typedef struct {
    fileHandle_t  handle;
    FILE         *filePtr;
    char         *name;
    int           perms;
    bool          eof;
    rle_hdr       rleHeader;
} rleFile_t;

/*
 * Client-data for pixel setting call back used to hold a single row of data.
 * The data will be outputed when the last pixel is written.
 */
typedef struct {
    rleFile_t    *rleFilePtr;
    rle_hdr      *rleHdrPtr;
    int           xMax;
    rle_pixel   **rowPtr;
} renderData_t;

/*
 * Internal prototypes.
 */
static rleFile_t *
RLEHandleToPtr _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                            char        *handle));

static void
RLEInitHeader _ANSI_ARGS_((Tcl_Interp *interp,
                           FILE       *filePtr,
                           char       *fileName,
                           rle_hdr    *rleHdrPtr));

static void
RLEFreeHeader _ANSI_ARGS_((rle_hdr *rleHdrPtr));

static void
RLEResetHeader _ANSI_ARGS_((rle_hdr *rleHdrPtr));

static bool
RLEReadError _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                          rleFile_t    *rleFilePtr));

static bool
RLEWriteError _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                           rleFile_t    *rleFilePtr));

static bool
ReadRLEHeader _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                           rleFile_t   *rleFilePtr));

static void
CopyRLEHeader _ANSI_ARGS_((rleFile_t *srcFilePtr,
                           rleFile_t *destFilePtr,
                           bool       mergeComments));

static bool
CloseRLEFile _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                          char        *command,
                          rleFile_t   *rleFilePtr));

static void *
RLEOutputStart _ANSI_ARGS_((tSippGlob_t         *tSippGlobPtr,
                            tSippOutputParms_t  *outputParmsPtr,
                            char                *handle,
                            char               **comments));

static bool
RLEOutputLine _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                           void_pt       clientData,
                           int           y,
                           u_char       *rowPtr));

static bool
RLEOutputBitMap _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                             void_pt       clientData,
                             Sipp_bitmap  *bitMapPtr));

static bool
RLEOutputEnd _ANSI_ARGS_((tSippGlob_t        *tSippGlobPtr,
                          tSippOutputParms_t *outputParmsPtr,
                          void_pt             clientData));

static bool
ParseImageComment _ANSI_ARGS_((tSippGlob_t        *tSippGlobPtr,
                               rleFile_t          *rleFilePtr,
                               tSippOutputParms_t *outputParmsPtr));

static bool
RLEToRLECopy _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                          char         *srcHandle,
                          rleFile_t    *srcFilePtr,
                          char         *destHandle));

static bool
RLEToOtherImage _ANSI_ARGS_((tSippGlob_t          *tSippGlobPtr,
                             tSippOutputParms_t   *outputParmsPtr,
                             rleFile_t            *srcFilePtr,
                             void                 *destClientData,
                             tSippStorageClass_t  *storageClassPtr));

static bool
RLECopyImage _ANSI_ARGS_((tSippGlob_t          *tSippGlobPtr,
                          char                 *srcHandle,
                          char                 *destHandle,
                          tSippStorageClass_t  *destClassPtr,
                          bool                  clear));

static rleFile_t *
RLECommentCmdSetup _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                                char        *handle,
                                bool         writeComment));

static int
RLECommentPut _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                           char         *handle,
                           char         *name,
                           char         *value));

static int
RLECommentGet _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                           char         *handle,
                           char         *name,
                           char        **valuePtr));

static int
RLECommentGetVec _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                              char         *handle,
                              char       ***argvPtr));

static int
RLECommentDelete _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                              char         *handle,
                              char         *name));

static void
RLECleanUp _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr));

/*=============================================================================
 * RLEHandleToPtr --
 *   Utility procedure to convert a RLE file handle to a RLE header pointer.
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A RLE handle.
 * Returns:
 *   A pointer to the RLE file entry, or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
static rleFile_t *
RLEHandleToPtr (tSippGlobPtr, handle)
    tSippGlob_t *tSippGlobPtr;
    char        *handle;
{
    rleFile_t *rleFilePtr;

    rleFilePtr = (rleFile_t *)
        Tcl_HandleXlate (tSippGlobPtr->interp,
                         tSippGlobPtr->rleTblPtr, handle);
    if (rleFilePtr == NULL)
        return NULL;
    return rleFilePtr;
}

/*=============================================================================
 * RLEInitHeader --
 *   Initialize an RLE header for a just opened file.  Care is taken to set up
 * the RLE header so it doesn't leak memory.
 * 
 * Parameters:
 *   o interp (I) - Pointer to the Tcl interpreter, used to get argv0.
 *   o filePtr (I) - Pointer to the file that was opened.
 *   o fileName (I) - Name the file was opened.
 *   o rleHdrPtr (O) - Pointer to the RLE header to initialize.
 *-----------------------------------------------------------------------------
 */
static void
RLEInitHeader (interp, filePtr, fileName, rleHdrPtr)
    Tcl_Interp *interp;
    FILE       *filePtr;
    char       *fileName;
    rle_hdr    *rleHdrPtr;
{
#ifdef URT_3_1b
    char *programName;

    /*
     * Get the program name to store in the header.
     */
    programName = Tcl_GetVar (interp, "argv0", TCL_GLOBAL_ONLY);
    if (programName == NULL)
        programName = "tsipp";

    /*
     * Init header. Make sure magic check doesn't succeed.
     */
    rleHdrPtr->is_init = 0;
    *rleHdrPtr = *rle_hdr_init (NULL);

    /*
     * Delete the names init put in and put in our own.
     */
    sfree ((char *) rleHdrPtr->cmd);
    rleHdrPtr->cmd = NULL;
    sfree ((char *) rleHdrPtr->file_name);
    rleHdrPtr->file_name = NULL;
    rle_names (rleHdrPtr, programName, fileName, 0);
#else
    *rleHdrPtr = rle_dflt_hdr;
    rleHdrPtr->bg_color = NULL;
#endif
    rleHdrPtr->rle_file = filePtr;

    /*
     * Indicate that an image is zero sized until specified by reading,
     * rendering or copying.  ?????
     */
    rleHdrPtr->xmin =  0;
    rleHdrPtr->xmax = -1;
    rleHdrPtr->ymin =  0;
    rleHdrPtr->ymax = -1;
}

/*=============================================================================
 * RLEFreeHeader --
 *   Free all dynamic memory associated with an RLE header before the file is
 * closed.  
 * 
 * Parameters:
 *   o rleHdrPtr (I) - Pointer to the RLE header.  The header structure is
 *     not freed, only the dynamic contents.
 *-----------------------------------------------------------------------------
 */
static void
RLEFreeHeader (rleHdrPtr)
    rle_hdr *rleHdrPtr;
{
    RLEResetHeader (rleHdrPtr);

#ifdef URT_3_1b
    if (rleHdrPtr->cmd != NULL)
        sfree ((char *) rleHdrPtr->cmd);
    if (rleHdrPtr->file_name != NULL)
        sfree ((char *) rleHdrPtr->file_name);
#endif
}

/*=============================================================================
 * RLEResetHeader --
 *   Free all dynamic memory associated with an RLE header.  This does not
 * free the header itself or change the file name in the header.
 * 
 * Parameters:
 *   o rleHdrPtr (I) - Pointer to the RLE header.
 *-----------------------------------------------------------------------------
 */
static void
RLEResetHeader (rleHdrPtr)
    rle_hdr *rleHdrPtr;
{
    if (rleHdrPtr->cmap != NULL) {
        sfree (rleHdrPtr->cmap);
        rleHdrPtr->cmap = NULL;
    }
    if (rleHdrPtr->bg_color != NULL) {
        sfree (rleHdrPtr->bg_color);
        rleHdrPtr->bg_color = NULL;
    }
    if (rleHdrPtr->comments != NULL)
        TSippCommentFree ((char ***) &rleHdrPtr->comments);
}

/*=============================================================================
 * RLEReadError --
 *   Handle a RLE read error.  This is a hack, since the RLE library doesn't
 * report I/O errors.  This should be called when ferror reports an error.
 * 
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.  The
 *     error message is returned in interp->result.
 *   o rleFilePtr (I) - RLE file that the error occured on.
 * Globals:
 *   o errno (I) - Should contain the error number of the error that occured.
 * Returns:
 *   Always returns FALSE.
 *-----------------------------------------------------------------------------
 */
static bool
RLEReadError (tSippGlobPtr, rleFilePtr)
    tSippGlob_t *tSippGlobPtr;
    rleFile_t    *rleFilePtr;
{
    Tcl_AppendResult (tSippGlobPtr->interp, "read error on RLE file: ",
                      Tcl_PosixError (tSippGlobPtr->interp),
                      (char *) NULL);

    clearerr (rleFilePtr->rleHeader.rle_file);
    return FALSE;
}

/*=============================================================================
 * RLEWriteError --
 *   Handle a RLE write error.  This is a hack, since the RLE library doesn't
 * report I/O errors.  This should be called when ferror reports an error.
 * 
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.  The
 *     error message is returned in interp->result.
 *   o rleFilePtr (I) - RLE file that the error occured on.
 * Globals:
 *   o errno (I) - Should contain the error number of the error that occured.
 * Returns:
 *   Always returns FALSE.
 *-----------------------------------------------------------------------------
 */
static bool
RLEWriteError (tSippGlobPtr, rleFilePtr)
    tSippGlob_t *tSippGlobPtr;
    rleFile_t    *rleFilePtr;
{
    Tcl_AppendResult (tSippGlobPtr->interp, "write error on RLE file: ",
                      Tcl_PosixError (tSippGlobPtr->interp),
                      (char *) NULL);

    clearerr (rleFilePtr->rleHeader.rle_file);
    return FALSE;
}

/*=============================================================================
 * ReadRLEHeader --
 *   Read the header of a RLE file.   If there is no data left in the file,
 * the EOF is set.  Care is taken not to lose any memory.  After a header
 * is read, the comments are copied and converted to the standard dynamic
 * layout.  All old dynamic data in the header is cleaned up before the read.
 * 
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o rleFilePtr (I) - Pointer to the RLE file structure containing the
 *     header.  The filePtr field must be initialized.
 * Returns:
 *   TRUE if all is OK, FALSE if an error occured with error message in
 *   tSippGlobPtr->interp->result.
 *-----------------------------------------------------------------------------
 */
static bool
ReadRLEHeader (tSippGlobPtr, rleFilePtr)
    tSippGlob_t *tSippGlobPtr;
    rleFile_t   *rleFilePtr;
{
    int    status;
    char **oldArrayPtr;

    RLEResetHeader (&rleFilePtr->rleHeader);
    rleFilePtr->rleHeader.rle_file = rleFilePtr->filePtr;

    status = rle_get_setup (&rleFilePtr->rleHeader);

    switch (status) {
      case RLE_SUCCESS:
        rleFilePtr->eof = FALSE;
        break;
      case RLE_EMPTY:
      case RLE_EOF:
        rleFilePtr->eof = TRUE;
        break;
      case RLE_NOT_RLE:
        Tcl_AppendResult (tSippGlobPtr->interp, rleFilePtr->name,
                          " is an invalid RLE file", (char *) NULL);
        return FALSE;
      case RLE_NO_SPACE:
        panic ("rle_get_setup: malloc failed");
    }
    
    /*
     * Since toolkit doesn't check I/O errors, we will.
     */
    if (ferror (rleFilePtr->rleHeader.rle_file)) {
        return RLEReadError (tSippGlobPtr,
                             rleFilePtr);
    }

    /*???Check for useable RLE file (i.e. 3 colors, etc).
     * Handle color maps (limited).
     */

    /*
     * Convert comments to standard malloced string layout.  This is really
     * dangerous.  It knows how rle_get_setup works.
     */
    if (rleFilePtr->rleHeader.comments != NULL) {
        oldArrayPtr = (char **) rleFilePtr->rleHeader.comments;
        rleFilePtr->rleHeader.comments = 
            (CONST_DECL char **) TSippCommentCopy (oldArrayPtr);
        sfree (oldArrayPtr [0]);
        sfree (oldArrayPtr);
    }

    return TRUE;
}

/*=============================================================================
 * CopyRLEHeader --
 *   Copy the header of one file to another.  All dynamic fields are duplicated
 * rather than sharing a pointer, making them independent.
 * 
 * Parameters:
 *   o srcFilePtr (I) - Pointer to the RLE file structure containing the
 *     source header.
 *   o destFilePtr (I) - Pointer to the RLE file structure containing the
 *     source header.
 *   o mergeComments (I) - If TRUE, then the comments will be merged, with
 *     the comments in destFile overriding srcFile.  If FALSE, the comments
 *     will be copied.
 *-----------------------------------------------------------------------------
 */
static void
CopyRLEHeader (srcFilePtr, destFilePtr, mergeComments)
    rleFile_t *srcFilePtr;
    rleFile_t *destFilePtr;
    bool       mergeComments;
{
    int    sizeBytes;
    char **saveComments = NULL, *holdCmd, *holdFName;

    /*
     * Cleanup the header then copy all of the static fields.  The only thing
     * that changes is the stdio file pointer.  Then copy the dynamic fields.
     * Save the comments instead of releasing them if comments are being
     * merged.  The comment pointer saving code just copies a NULL if
     * mergeComments is FALSE;
     */
    if (mergeComments) {
        saveComments = (char **) destFilePtr->rleHeader.comments;
        destFilePtr->rleHeader.comments = NULL;
    }

    RLEResetHeader (&destFilePtr->rleHeader);

    /*
     * Copy the header.  We don't even bother with rle_hdr_cp, as it loses
     * too much memory.
     */
#ifdef URT_3_1b
    holdCmd = (char *) destFilePtr->rleHeader.cmd;
    holdFName = (char *) destFilePtr->rleHeader.file_name;

    destFilePtr->rleHeader = srcFilePtr->rleHeader;

    destFilePtr->rleHeader.cmd = holdCmd;
    destFilePtr->rleHeader.file_name = holdFName;
    destFilePtr->rleHeader.comments = (CONST_DECL char **) saveComments;
#else
    destFilePtr->rleHeader = srcFilePtr->rleHeader;
    destFilePtr->rleHeader.comments = saveComments;
#endif
    destFilePtr->rleHeader.rle_file = destFilePtr->filePtr;

    /*
     * Copy the color map
     */
    if (srcFilePtr->rleHeader.cmap != NULL) {
        sizeBytes = sizeof (rle_map) * srcFilePtr->rleHeader.cmaplen;
        destFilePtr->rleHeader.cmap = (rle_map *) smalloc (sizeBytes);
        memcpy (destFilePtr->rleHeader.cmap,
                srcFilePtr->rleHeader.cmap,
                sizeBytes);
    }

    /*
     * Copy the background color array.
     */
    if (srcFilePtr->rleHeader.bg_color != NULL) {
        sizeBytes = sizeof (int) * srcFilePtr->rleHeader.ncolors;
        destFilePtr->rleHeader.bg_color = (int *) smalloc (sizeBytes);
        memcpy (destFilePtr->rleHeader.bg_color,
                srcFilePtr->rleHeader.bg_color,
                sizeBytes);
    }

    /*
     * Copy or merge the comments.
     */
    if (mergeComments)
        TSippCommentMerge ((char **) srcFilePtr->rleHeader.comments,
                           (char ***) &destFilePtr->rleHeader.comments);
    else
        destFilePtr->rleHeader.comments = (CONST_DECL char **) 
            TSippCommentCopy ((char **) srcFilePtr->rleHeader.comments);
}

/*=============================================================================
 * CloseRLEFile --
 *   Close an RLE file, deleting the handle entry.
 * 
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o command (I) - The command that called this function (argv [0]).
 *   o rleFilePtr (I) - Pointer to the file entry to close.
 * Returns:
 *   TRUE if all is ok, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
CloseRLEFile (tSippGlobPtr, command, rleFilePtr)
    tSippGlob_t *tSippGlobPtr;
    char         *command;
    rleFile_t    *rleFilePtr;
{
    int  result;

    result = TSippCallCmd (tSippGlobPtr,
                           "close", 
                           rleFilePtr->handle,
                           (char *) NULL);

    RLEFreeHeader (&rleFilePtr->rleHeader);

    sfree (rleFilePtr->name);
    Tcl_HandleFree (tSippGlobPtr->rleTblPtr, rleFilePtr);

    return (result == TCL_OK);
}

/*=============================================================================
 * SippRLEOpen --
 *   Implements the command:
 *     SippRLEOpen filename [access]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.  We cheat by calling
 * the Tcl "open" command executor to actually do the open, that way we get the
 * full semantics of the open command.
 *-----------------------------------------------------------------------------
 */
static int
SippRLEOpen (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t   *tSippGlobPtr = (tSippGlob_t *) clientData;
    fileHandle_t   fileHandle, rleHandle;
    rleFile_t     *rleFilePtr;
    FILE          *filePtr;
    int            perms;
    char          *errorMsg;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 2) || (argc > 3)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " filename [access]", (char *) NULL);
        return TCL_ERROR;
    }
    
    /*
     * Use the open command to actually open the file.  If it succeeds, save
     * the handle, which will latter be stored in the RLE file table entry.
     */
    if (TSippCallCmd (tSippGlobPtr,
                      "open", 
                      argv [1],
                      argv [2],    /* Could be NULL */
                      (char *) NULL) == TCL_ERROR)
        return TCL_ERROR;

    strcpy (fileHandle, interp->result);
    Tcl_ResetResult (interp);

    if (Tcl_GetChannel (interp, fileHandle, &perms) == NULL)
        return TCL_ERROR;  /* Should never happen */

    /*
     * Make sure that the file will only be opened for reading or writting,
     * but not both.  This is done after the header is setup so that we 
     * can use standard error close path.
     */
    if ((perms & TCL_READABLE) && (perms & TCL_WRITABLE)) {
        TSippCallCmd (tSippGlobPtr,
                      "close", 
                      fileHandle,
                      (char *) NULL);
        Tcl_ResetResult (interp);
        Tcl_AppendResult (interp, "RLE files may be opened for reading or ",
                          "writing, but not both", (char *) NULL);
        return TCL_ERROR;
    }

    if (Tcl_GetOpenFile (interp,
                         fileHandle,
                         ((perms & TCL_WRITABLE) != 0), 0,
                         (ClientData *) &filePtr) != TCL_OK)
        return TCL_ERROR;  /* Should never happen */

    /*
     * Allocate and initialize the RLE file structure and the RLE header.
     */
    rleFilePtr = (rleFile_t *) Tcl_HandleAlloc (tSippGlobPtr->rleTblPtr,
                                               rleHandle);
    strcpy (rleFilePtr->handle, fileHandle);
    rleFilePtr->name = strdup (argv [1]);
    rleFilePtr->perms = perms;
    rleFilePtr->filePtr = filePtr;
    rleFilePtr->eof = TRUE;  /* Assume writing at the end. */

    RLEInitHeader (interp,
                   filePtr,
                   argv [1],
                   &rleFilePtr->rleHeader);

    /*
     * If read access, get the header.  Must stash the handle in case an
     * error gets generated.
     */
    if (rleFilePtr->perms & TCL_READABLE) {
        if (!ReadRLEHeader (tSippGlobPtr, rleFilePtr))
            goto errorExit;
    }

    strcpy (interp->result, rleHandle);
    return TCL_OK;

    /*
     * Close the file and clean up the without losing an error message.
     */
  errorExit:
    errorMsg = (char *) alloca (strlen (interp->result) + 1);
    strcpy (errorMsg, interp->result);
    Tcl_ResetResult (interp);

    CloseRLEFile (tSippGlobPtr, argv [0], rleFilePtr);

    Tcl_ResetResult (interp);
    Tcl_SetResult (interp, errorMsg, TCL_VOLATILE);

    return TCL_ERROR;
}

/*=============================================================================
 * SippRLEClose --
 *   Implements the command:
 *     SippRLEClose rlehandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippRLEClose (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t *tSippGlobPtr = (tSippGlob_t *) clientData;
    rleFile_t   *rleFilePtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " rlehandle", (char *) NULL);
        return TCL_ERROR;
    }

    rleFilePtr = RLEHandleToPtr (tSippGlobPtr, argv [1]);
    if (rleFilePtr == NULL)
        return TCL_ERROR;

    if (CloseRLEFile (tSippGlobPtr, argv [0], rleFilePtr))
        return TCL_OK;
    else
        return TCL_ERROR;
}

/*=============================================================================
 * SippRLEInfo --
 *   Implements the command:
 *     SippRLEInfo rlehandle attribute
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippRLEInfo (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t *tSippGlobPtr = (tSippGlob_t *) clientData;
    rleFile_t   *rleFilePtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " rlehandle attribute", (char *) NULL);
        return TCL_ERROR;
    }

    rleFilePtr = RLEHandleToPtr (tSippGlobPtr, argv [1]);
    if (rleFilePtr == NULL)
        return TCL_ERROR;

    if (STREQU (argv [2], "EOF")) {
        interp->result = rleFilePtr->eof ? "1" : "0";
        return TCL_OK;
    }

    if (rleFilePtr->eof)
        TSippAtEOF (tSippGlobPtr->interp, argv [1]);

    if (STREQU (argv [2], "XSIZE")) {
        sprintf (interp->result, "%d", (rleFilePtr->rleHeader.xmax -
                                        rleFilePtr->rleHeader.xmin) + 1);
        return TCL_OK;
    }
    if (STREQU (argv [2], "YSIZE")) {
        sprintf (interp->result, "%d", (rleFilePtr->rleHeader.ymax -
                                        rleFilePtr->rleHeader.ymin) + 1);
        return TCL_OK;
    }

    Tcl_AppendResult (interp, "unknown attribute \"", argv [2], 
                      "\", expected one of \"EOF\", \"XSIZE\". or \"YSIZE\"",
                      (char *) NULL);
    return TCL_ERROR;
}

/*=============================================================================
 * RLEOutputStart --
 *   Start output to an RLE file.  This routine is referenced by a pointer in
 * the RLE image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o handle (I) - The RLE handle for the file to write.
 *   o comments (I) - If the source image has comments associated with it, they
 *     are passed here.  If the destination object can store them, it possible.
 *     NULL if no comments available.  Should contain standard comments when
 *     rendering.
 *
 * Returns:
 *   A pointer to be passed back into the other RLE output routines or NULL
 *   an a message in tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static void *
RLEOutputStart (tSippGlobPtr, outputParmsPtr, handle, comments)
    tSippGlob_t         *tSippGlobPtr;
    tSippOutputParms_t  *outputParmsPtr;
    char                *handle;
    char               **comments;
{
    rleFile_t     *rleFilePtr;
    renderData_t  *renderDataPtr;

    rleFilePtr = RLEHandleToPtr (tSippGlobPtr, handle);
    if (rleFilePtr == NULL)
        return NULL;

    if ((rleFilePtr->perms & TCL_WRITABLE) == 0) {
        TSippNotWritable (tSippGlobPtr->interp, handle);
        return NULL;
    }

    if (comments != NULL) {
        TSippCommentMerge (comments,
                           (char ***) &rleFilePtr->rleHeader.comments);
    }

    /*
     * Set up the RLE header for a 24 bit image and write it.
     */
    rleFilePtr->rleHeader.ncolors = 3;  /* Red, Green and Blue */
    rleFilePtr->rleHeader.xmin = 0;
    rleFilePtr->rleHeader.xmax = outputParmsPtr->imgData.xSize - 1;
    rleFilePtr->rleHeader.ymin = 0;
    rleFilePtr->rleHeader.ymax = outputParmsPtr->imgData.ySize - 1;

    RLE_SET_BIT (rleFilePtr->rleHeader, RLE_RED);
    RLE_SET_BIT (rleFilePtr->rleHeader, RLE_GREEN);
    RLE_SET_BIT (rleFilePtr->rleHeader, RLE_BLUE);

    if (rleFilePtr->rleHeader.bg_color == NULL)
        rleFilePtr->rleHeader.bg_color = (int *) smalloc (3 * sizeof (int));
    rleFilePtr->rleHeader.bg_color [RLE_RED] =
        tSippGlobPtr->backgroundColor.red * 255;
    rleFilePtr->rleHeader.bg_color [RLE_GREEN] =
        tSippGlobPtr->backgroundColor.grn * 255;
    rleFilePtr->rleHeader.bg_color [RLE_BLUE] =
        tSippGlobPtr->backgroundColor.blu * 255;

    rleFilePtr->rleHeader.background = 2;        /* Use background color */

    /*
     * Write the header to file.
     */
    rle_put_setup (&rleFilePtr->rleHeader);
    if (ferror (rleFilePtr->rleHeader.rle_file)) {
        RLEWriteError (tSippGlobPtr,
                       rleFilePtr);
        return NULL;
    }


    /*
     * Setup the clientdata to pass back to the other rendering routines.
     */
    renderDataPtr = (renderData_t *) smalloc (sizeof (renderData_t));

    renderDataPtr->rleFilePtr = rleFilePtr;
    renderDataPtr->rleHdrPtr  = &rleFilePtr->rleHeader;
    renderDataPtr->xMax       = renderDataPtr->rleHdrPtr->xmax;

    rle_row_alloc (&rleFilePtr->rleHeader, &renderDataPtr->rowPtr);

    return renderDataPtr;
}

/*=============================================================================
 * RLEOutputLine --
 *   Output a rendered line to an RLE file.  This routine is referenced by a
 * pointer in the RLE image storage class table.
 *
 * Parameters:
 *   o clientData (I) - Actually a pointer to the RLE rendering clientdata.
 *   o y (I) - The scan line that was just rendered.
 *   o rowPtr (I) - The pixels for the scanline that was just rendered.
 * Returns:
 *   TRUE if all is ok, FALSE and an error message in tSippGlobPtr->interp->
 * result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
RLEOutputLine (tSippGlobPtr, clientData, y, rowPtr)
    tSippGlob_t  *tSippGlobPtr;
    void_pt       clientData;
    int           y;
    u_char       *rowPtr;
{
    renderData_t  *renderDataPtr = (renderData_t *) clientData;
    int            x;
    
    for (x = 0; x <= renderDataPtr->xMax; x++) {
        renderDataPtr->rowPtr [RLE_RED] [x]   = rowPtr [TSIPP_RED];
        renderDataPtr->rowPtr [RLE_GREEN] [x] = rowPtr [TSIPP_GREEN];
        renderDataPtr->rowPtr [RLE_BLUE] [x]  = rowPtr [TSIPP_BLUE];
        rowPtr += 3;
    }

    rle_putrow (renderDataPtr->rowPtr, renderDataPtr->xMax,
                renderDataPtr->rleHdrPtr);
    if (ferror (renderDataPtr->rleHdrPtr->rle_file)) {
        return RLEWriteError (tSippGlobPtr,
                              renderDataPtr->rleFilePtr);
    }

    return TRUE;
}

/*=============================================================================
 * RLEOutputBitMap --
 *   Output a SIPP bit map to an RLE file.  This routine is referenced by a
 * pointer in the RLE image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o clientData (I) - Actually a pointer to the RLE rendering clientdata.
 *   o bitMapPtr (I) - Pointer to the SIPP bit map structure.
 * Returns:
 *   TRUE if all is ok, FALSE and an error message in tSippGlobPtr->interp->
 * result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
RLEOutputBitMap (tSippGlobPtr, clientData, bitMapPtr)
    tSippGlob_t *tSippGlobPtr;
    void_pt      clientData;
    Sipp_bitmap *bitMapPtr;
{
    renderData_t *renderDataPtr = (renderData_t *) clientData;
    rle_pixel   **rowPtr        = renderDataPtr->rowPtr;
    rle_hdr      *rleHdrPtr     = renderDataPtr->rleHdrPtr;
    int           xMax          = renderDataPtr->xMax;

    rle_pixel     lineColor [3], backgroundColor [3];
    int           x, y;
    u_char       *bitRowPtr;
    unsigned      bit;

    lineColor [RLE_RED]   = tSippGlobPtr->lineColor.red * 255;
    lineColor [RLE_GREEN] = tSippGlobPtr->lineColor.grn * 255;
    lineColor [RLE_BLUE]  = tSippGlobPtr->lineColor.blu * 255;

    backgroundColor [RLE_RED]   = tSippGlobPtr->backgroundColor.red * 255;
    backgroundColor [RLE_GREEN] = tSippGlobPtr->backgroundColor.grn * 255;
    backgroundColor [RLE_BLUE]  = tSippGlobPtr->backgroundColor.blu * 255;

    /*
     * Loop setting pixels based on the bit map and outputting rows to the
     * file.
     */

    for (y = rleHdrPtr->ymax - 1; y >= 0 ; y--) {
        bitRowPtr = &bitMapPtr->buffer [y * bitMapPtr->width_bytes];
        
        for (x = 0; x <= xMax; x++) {
            bit = (bitRowPtr [x >> 3] >> (7-(x & 7))) & 1;
            if (bit) {
                rowPtr [RLE_RED] [x]   = lineColor [RLE_RED];
                rowPtr [RLE_GREEN] [x] = lineColor [RLE_GREEN];
                rowPtr [RLE_BLUE] [x]  = lineColor [RLE_BLUE];
            } else {
                rowPtr [RLE_RED] [x]   = backgroundColor [RLE_RED];
                rowPtr [RLE_GREEN] [x] = backgroundColor [RLE_GREEN];
                rowPtr [RLE_BLUE] [x]  = backgroundColor [RLE_BLUE];
            }
        }
        rle_putrow (rowPtr, xMax, rleHdrPtr);
        if (ferror (rleHdrPtr->rle_file)) {
            return RLEWriteError (tSippGlobPtr,
                                  renderDataPtr->rleFilePtr);
        }
    }
    return TRUE;
}

/*=============================================================================
 * RLEOutputEnd --
 *   Finish up output to an RLE file.  This routine is referenced by a pointer
 * in the RLE image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o clientData (I) - Actually a pointer to the RLE rendering clientdata.
 * Returns:
 *   TRUE if all is ok, FALSE and an error message in
 *  tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
RLEOutputEnd (tSippGlobPtr, outputParmsPtr, clientData)
    tSippGlob_t         *tSippGlobPtr;
    tSippOutputParms_t  *outputParmsPtr;
     void_pt             clientData;
{
    renderData_t  *renderDataPtr = (renderData_t *) clientData;
    
    rle_puteof (renderDataPtr->rleHdrPtr);
    
    rle_row_free (renderDataPtr->rleHdrPtr, renderDataPtr->rowPtr);
    sfree (renderDataPtr);

    fflush (renderDataPtr->rleHdrPtr->rle_file);
    if (ferror (renderDataPtr->rleHdrPtr->rle_file)) {
        return RLEWriteError (tSippGlobPtr,
                              renderDataPtr->rleFilePtr);
    }

   /*
    * Reset the output file header, since we will start with a new image.
    */
   RLEResetHeader (renderDataPtr->rleHdrPtr);

   return TRUE;
}

/*=============================================================================
 * ParseImageComment --
 *   Parse the TSIPP_IMAGE comment stored in an RLE file header.  This
 * initializes fields in the outputParms structure.  If the comment is not,
 * found, then output parameters are unchanged.  If the comment is found, but
 * is not, valid, an error is returned.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o rleFilePtr (I) - Pointer to the RLE file structure containing the
 *     header.  The filePtr field must be initialized.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 * Returns:
 *   TRUE if all is OK, FALSE if an error occured with error message in
 *   tSippGlobPtr->interp->result.
 *-----------------------------------------------------------------------------
 */
static bool
ParseImageComment (tSippGlobPtr, rleFilePtr, outputParmsPtr)
    tSippGlob_t        *tSippGlobPtr;
    rleFile_t          *rleFilePtr;
    tSippOutputParms_t *outputParmsPtr;
{
    char  *infoList, **infoArgv = NULL, *errorMsg;
    int    infoArgc;
    Color  backgroundColor, lineColor;

    infoList = TSippCommentGet ((char **) rleFilePtr->rleHeader.comments,
                                "TSIPP_IMAGE");
    if (infoList == NULL)
        return TRUE;

    if (Tcl_SplitList (tSippGlobPtr->interp, infoList,
                       &infoArgc, &infoArgv) != TCL_OK)
        goto errorExit;
    
    if (infoArgc != 5) {
        tSippGlobPtr->interp->result = "wrong number of elements in list";
        goto errorExit;
    }

    /*
     * Convert mode.
     */
    if (STREQU (infoArgv [0], "FLAT"))
        outputParmsPtr->imgData.mode = FLAT;
    else if (STREQU (infoArgv [0], "GOURAUD"))
        outputParmsPtr->imgData.mode = GOURAUD;
    else if (STREQU (infoArgv [0], "PHONG"))
        outputParmsPtr->imgData.mode = PHONG;
    else if (STREQU (infoArgv [0], "LINE"))
        outputParmsPtr->imgData.mode = LINE;
    else if (STREQU (infoArgv [0], "UNKNOWN24"))
        outputParmsPtr->imgData.mode = MODE_UNKNOWN24;
    else {
        Tcl_AppendResult (tSippGlobPtr->interp, "invalid mode: ",
                          infoArgv [0], (char *) NULL);
        goto errorExit;
    }

    /*
     * Convert over-sampling factor.
     */
    if (!TSippConvertPosUnsigned (tSippGlobPtr, infoArgv [1],
                                  &outputParmsPtr->imgData.overSampling))
        goto errorExit;

        
    /*
     * Convert field (interlace).
     */
    if (STREQU (infoArgv [2], "BOTH"))
        outputParmsPtr->imgData.field = BOTH;
    else if (STREQU (infoArgv [2], "ODD"))
        outputParmsPtr->imgData.field = ODD;
    else if (STREQU (infoArgv [2], "EVEN"))
        outputParmsPtr->imgData.field = EVEN;
    else  {
        Tcl_AppendResult (tSippGlobPtr->interp, "invalid field: ",
                          infoArgv [2], (char *) NULL);
        goto errorExit;
    }

    /*
     * Convert background and line colors.
     */
    if (!TSippConvertColor (tSippGlobPtr, infoArgv [3], &backgroundColor))
        goto errorExit;

    outputParmsPtr->imgData.backgroundColor [TSIPP_RED] =
        backgroundColor.red * 255;
    outputParmsPtr->imgData.backgroundColor [TSIPP_GREEN] =
        backgroundColor.grn * 255;
    outputParmsPtr->imgData.backgroundColor [TSIPP_BLUE] =
        backgroundColor.blu * 255;

    if (!TSippConvertColor (tSippGlobPtr, infoArgv [4], &lineColor))
        goto errorExit;

    outputParmsPtr->imgData.lineColor [TSIPP_RED]   = lineColor.red * 255;
    outputParmsPtr->imgData.lineColor [TSIPP_GREEN] = lineColor.grn * 255;
    outputParmsPtr->imgData.lineColor [TSIPP_BLUE]  = lineColor.blu * 255;

    sfree (infoArgv);
    return TRUE;

    /*
     * Handle error return.  Takes message in interp and adds more info.
     */
  errorExit:
    if (infoArgv == NULL)
        sfree (infoArgv);

    errorMsg = strdup (tSippGlobPtr->interp->result);
    Tcl_ResetResult (tSippGlobPtr->interp);
    Tcl_AppendResult (tSippGlobPtr->interp, "invalid format for TSIPP_IMAGE ",
                      "comment: ", errorMsg, (char *) NULL);
    sfree (errorMsg);
    return FALSE;
}

/*=============================================================================
 * RLEToRLECopy --
 *   Copy an image from an RLE file to another RLE file.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o srcHandle (I) - The handle for the source RLE file.  Used for error
 *     messages.
 *   o srcFilePtr (I) - Pointer to the RLE file structure for the source file.
 *   o destHandle (I) - The handle for the destination RLE file.
 * Returns:
 *   TRUE is all is OK, or FALSE if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
RLEToRLECopy (tSippGlobPtr, srcHandle, srcFilePtr, destHandle)
    tSippGlob_t *tSippGlobPtr;
    char        *srcHandle;
    rleFile_t   *srcFilePtr;
    char        *destHandle;
{
    rleFile_t *destFilePtr;

    destFilePtr = RLEHandleToPtr (tSippGlobPtr, destHandle);
    if (destFilePtr == NULL)
        return FALSE;

    if ((destFilePtr->perms & TCL_WRITABLE) == 0) {
        TSippNotWritable (tSippGlobPtr->interp, destHandle);
        return FALSE;
    }

    /*
     * Copy the header and data.
     */
    CopyRLEHeader (srcFilePtr, destFilePtr, TRUE);  /* Merge comments */
    
    rle_put_setup (&destFilePtr->rleHeader);
    if (ferror (destFilePtr->rleHeader.rle_file)) {
        return RLEWriteError (tSippGlobPtr,
                              destFilePtr);
    }

    rle_cp (&srcFilePtr->rleHeader, &destFilePtr->rleHeader);
    fflush (destFilePtr->filePtr);
    if (ferror (srcFilePtr->rleHeader.rle_file)) {
        return RLEReadError (tSippGlobPtr,
                             srcFilePtr);
    }
    if (ferror (destFilePtr->rleHeader.rle_file)) {
        return RLEWriteError (tSippGlobPtr,
                              destFilePtr);
    }


    /*
     * Reset the write header and read in the next input header, if available.
     */
    RLEResetHeader (&destFilePtr->rleHeader);
    if (!ReadRLEHeader (tSippGlobPtr, srcFilePtr))
        return FALSE;
    
    return TRUE;
}

/*=============================================================================
 * RLEToOtherImage --
 *   Copy an image from an RLE file to some other image type.  This assumes
 * all setup has been done, it just copies the scanlines.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o srcFilePtr (I) - Pointer to the RLE file structure for the source file.
 *   o destClientData (I) - The clientdata returned by the output start
 *     function for the image destination.
 *   o storageClassPtr (I) - Pointer to the image storage class description
 *     for the target class.
 * Returns:
 *   TRUE if all is ok, FALSE and an error message in tSippGlobPtr->interp->
 * result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
RLEToOtherImage (tSippGlobPtr, outputParmsPtr, srcFilePtr, destClientData,
                 storageClassPtr)
    tSippGlob_t          *tSippGlobPtr;
    tSippOutputParms_t   *outputParmsPtr;
    rleFile_t            *srcFilePtr;
    void                 *destClientData;
    tSippStorageClass_t  *storageClassPtr;
{
    int           x, xSize = outputParmsPtr->imgData.xSize;
    int           y, ySize = outputParmsPtr->imgData.ySize;
    int           redIdx, greenIdx, blueIdx;
    bool          status = TRUE;
    rle_pixel   **inRowPtr;
    u_char       *outRowPtr, *outFillPtr;
    outputLine_t *outputLine= storageClassPtr->outputLine;

    /*
     * Set up the in and out row buffers.
     */
    rle_row_alloc (&srcFilePtr->rleHeader, &inRowPtr);
    outRowPtr = (u_char *) alloca (outputParmsPtr->imgData.xSize *
                                   sizeof (u_char) * 3);

    /*
     * Set up indices for copying the color channels.  Normally there will be
     * three color channel, but this will handle a monochrome or even something
     * strange with 2 colors.
     */
    redIdx = 0;
    if (srcFilePtr->rleHeader.ncolors >= 2)
        greenIdx = 1;
    else
        greenIdx = srcFilePtr->rleHeader.ncolors - 1;
    if (srcFilePtr->rleHeader.ncolors >= 2)
        blueIdx = 2;
    else
        blueIdx = srcFilePtr->rleHeader.ncolors - 1;

    /*
     * Copy the image.
     */
    for (y = ySize - 1; y >= 0; y--) {
        rle_getrow (&srcFilePtr->rleHeader, inRowPtr);
        if (ferror (srcFilePtr->rleHeader.rle_file)) {
            status = RLEReadError (tSippGlobPtr,
                                   srcFilePtr);
            break;
        }

        outFillPtr = outRowPtr;

        for (x = 0; x < xSize; x++) {
            outFillPtr [TSIPP_RED]   = inRowPtr [redIdx] [x];
            outFillPtr [TSIPP_GREEN] = inRowPtr [greenIdx] [x];
            outFillPtr [TSIPP_BLUE]  = inRowPtr [blueIdx] [x];
            outFillPtr += 3;
        }
        status = (*outputLine) (tSippGlobPtr, 
                                destClientData,
                                y,
                                outRowPtr);
        if (!status)
            break;
    }

    rle_row_free (&srcFilePtr->rleHeader, inRowPtr);

    return status;
}

/*=============================================================================
 * RLEToOtherBitMap --
 *   Copy an image from an RLE file that we know from the comments was done
 * in line mode to some other image type.  This assumes all setup has been
 * done.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o srcFilePtr (I) - Pointer to the RLE file structure for the source file.
 *   o destClientData (I) - The clientdata returned by the output start
 *     function for the image destination.
 *   o storageClassPtr (I) - Pointer to the image storage class description
 *     for the target class.
 * Returns:
 *   TRUE if all is ok, FALSE and an error message in tSippGlobPtr->interp->
 * result if an error occurs.
 *
 * Notes:
 *   Color indexing for less that 3 channels is still done to handle images
 * that have been post processed, but the results probably will not be very
 * nice.
 *-----------------------------------------------------------------------------
 */
static bool
RLEToOtherBitMap (tSippGlobPtr, outputParmsPtr, srcFilePtr, destClientData,
                  storageClassPtr)
    tSippGlob_t          *tSippGlobPtr;
    tSippOutputParms_t   *outputParmsPtr;
    rleFile_t            *srcFilePtr;
    void                 *destClientData;
    tSippStorageClass_t  *storageClassPtr;
{
    int            x, xSize = outputParmsPtr->imgData.xSize;
    int            y, ySize = outputParmsPtr->imgData.ySize;
    int            redIdx, greenIdx, blueIdx;
    bool           status1, status2;
    rle_pixel    **inRowPtr;
    Sipp_bitmap   *bitMapPtr;
    u_char         backgroundColor [3], *bitMapRowPtr;

    /*
     * Set up the input row buffer and the bitmap.  Stash the background color.
     */
    rle_row_alloc (&srcFilePtr->rleHeader, &inRowPtr);
    bitMapPtr = sipp_bitmap_create (xSize, ySize);

    backgroundColor [TSIPP_RED] =
        outputParmsPtr->imgData.backgroundColor [TSIPP_RED];
    backgroundColor [TSIPP_GREEN] =
        outputParmsPtr->imgData.backgroundColor [TSIPP_GREEN];
    backgroundColor [TSIPP_BLUE] =
        outputParmsPtr->imgData.backgroundColor [TSIPP_BLUE];

    /*
     * Set up indices for copying the color channels.  Normally there will be
     * three color channel, but this will handle a monochrome or even something
     * strange with 2 colors.
     */
    redIdx = 0;
    if (srcFilePtr->rleHeader.ncolors >= 2)
        greenIdx = 1;
    else
        greenIdx = srcFilePtr->rleHeader.ncolors - 1;
    if (srcFilePtr->rleHeader.ncolors >= 2)
        blueIdx = 2;
    else
        blueIdx = srcFilePtr->rleHeader.ncolors - 1;

    /*
     * Read the image and fill in the bitmap.
     */
    status1 = TRUE;
    for (y = ySize - 1; y >= 0; y--) {
        rle_getrow (&srcFilePtr->rleHeader, inRowPtr);
        if (ferror (srcFilePtr->rleHeader.rle_file)) {
            status1 = RLEReadError (tSippGlobPtr,
                                    srcFilePtr);
            break;
        }

        bitMapRowPtr = bitMapPtr->buffer + (y * bitMapPtr->width_bytes);

        for (x = 0; x < xSize; x++) {
            if (inRowPtr [redIdx] [x]   != backgroundColor [TSIPP_RED] ||
                inRowPtr [greenIdx] [x] != backgroundColor [TSIPP_GREEN] ||
                inRowPtr [blueIdx] [x]  != backgroundColor [TSIPP_BLUE])

                bitMapRowPtr [x >> 3] |= 1 << (7 - (x & 7));
        }
    }

    status2 = (*storageClassPtr->outputBitMap) (tSippGlobPtr,
                                                destClientData,
                                                bitMapPtr);

    rle_row_free (&srcFilePtr->rleHeader, inRowPtr);
    sipp_bitmap_destruct (bitMapPtr);

    return status1 && status2;
}

/*=============================================================================
 * RLECopyImage --
 *   Copy an image from an RLE file to another image store.  This routine is
 * referenced by a pointer in the RLE image store class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o srcHandle (I) - The RLE handle for the file to read.
 *   o destHandle (I) - The handle for the destination image store.
 *   o destClassPtr (I) - Pointer to the image storage class description
 *     for the destination class.
 *   o clear (I) - If TRUE, clear target image before copying, if FALSE,
 *     don't clear.  Ignored if the ISO does not support this.
 * Returns:
 *   TRUE is all is OK, or FALSE if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
RLECopyImage (tSippGlobPtr, srcHandle, destHandle, destClassPtr, clear)
    tSippGlob_t         *tSippGlobPtr;
    char                *srcHandle;
    char                *destHandle;
    tSippStorageClass_t *destClassPtr;
    bool                 clear;
{
    rleFile_t           *srcFilePtr;
    tSippOutputParms_t   outputParms;
    bool                 status1, status2, status3;
    void                *destClientData;

    srcFilePtr = RLEHandleToPtr (tSippGlobPtr, srcHandle);
    if (srcFilePtr == NULL)
        return FALSE;

    if ((srcFilePtr->perms & TCL_READABLE) == 0) {
        TSippNotReadable (tSippGlobPtr->interp, srcHandle);
        return FALSE;
    }
    if (srcFilePtr->eof) {
        TSippAtEOF (tSippGlobPtr->interp, srcHandle);
        return FALSE;
    }

    /*
     * If the target is an RLE file, optimize the copy.
     */
    if (destClassPtr->copyImage == RLECopyImage)
        return RLEToRLECopy (tSippGlobPtr, srcHandle, srcFilePtr, destHandle);

    /*
     * Set up the general output parameters assoicated with this image.
     * Default them first, then try and get better values from the TSIPP_IMAGE
     * comment.
     */
    outputParms.imgData.xSize = (srcFilePtr->rleHeader.xmax -
                                 srcFilePtr->rleHeader.xmin) + 1;
    outputParms.imgData.ySize = (srcFilePtr->rleHeader.ymax -
                                 srcFilePtr->rleHeader.ymin) + 1;

    outputParms.imgData.mode = MODE_UNKNOWN24;
    outputParms.imgData.overSampling = 1;
    outputParms.imgData.field = BOTH;
    
    outputParms.imgData.backgroundColor [TSIPP_RED] =
        srcFilePtr->rleHeader.bg_color [RLE_RED];
    outputParms.imgData.backgroundColor [TSIPP_GREEN] =
        srcFilePtr->rleHeader.bg_color [RLE_GREEN];
    outputParms.imgData.backgroundColor [TSIPP_BLUE] =
        srcFilePtr->rleHeader.bg_color [RLE_BLUE];
        
    outputParms.imgData.lineColor [TSIPP_RED]   = 255;
    outputParms.imgData.lineColor [TSIPP_GREEN] = 255;
    outputParms.imgData.lineColor [TSIPP_BLUE]  = 255;

    if (!ParseImageComment (tSippGlobPtr, srcFilePtr, &outputParms))
        return FALSE;

    /*
     * Initialize other output parms fields.
     */
    outputParms.tSippGlobPtr = tSippGlobPtr;
    outputParms.argv = NULL;   
    outputParms.update = 0;
    outputParms.clear= clear;

    outputParms.bitMapOutput = (outputParms.imgData.mode == LINE) &&
                               (destClassPtr->bitMapOptimal);
    outputParms.scanDirection = TSIPP_SCAN_BOTTOM_UP;

    /*
     * Copy to other image source  If this is from a line image and the target
     * stores bitmaps in an optimal manner, output a bitmap.
     */
    destClientData =
        (*destClassPtr->outputStart) (tSippGlobPtr,
                                      &outputParms,
                                      destHandle,
                                     (char **) srcFilePtr->rleHeader.comments);
    if (destClientData == NULL)
        return FALSE;

    if ((outputParms.imgData.mode == LINE) && (destClassPtr->bitMapOptimal)) {
        status1 = RLEToOtherBitMap (tSippGlobPtr, 
                                    &outputParms,
                                    srcFilePtr,
                                    destClientData,
                                    destClassPtr);
    } else {
        status1 = RLEToOtherImage (tSippGlobPtr, 
                                   &outputParms,
                                   srcFilePtr,
                                   destClientData,
                                   destClassPtr);
    }

    status2 = (*destClassPtr->outputEnd) (tSippGlobPtr,
                                          &outputParms,
                                          destClientData);

    /*
     * Read in the next input header, if available.
     */
    status3 = ReadRLEHeader (tSippGlobPtr, srcFilePtr);

    return status1 && status2 && status3;
}

/*=============================================================================
 * RLECommentCmdSetup --
 *   Utility procedure to set up for one of the comments commands.  If converts
 * the rlehandle.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A RLE handle.
 *   o writeComment (I) - TRUE if the command is going to write the comment
 *     area.
 * Returns:
 *   A pointer to the RLE file entry, or NULL if an error occured.  Errors
 * returned in tSippGlobPtr->interp->result.
 *-----------------------------------------------------------------------------
 */
static rleFile_t *
RLECommentCmdSetup (tSippGlobPtr, handle, writeComment)
    tSippGlob_t *tSippGlobPtr;
    char        *handle;
    bool         writeComment;
{
    rleFile_t *rleFilePtr;

    rleFilePtr = RLEHandleToPtr (tSippGlobPtr, handle);
    if (rleFilePtr == NULL)
        return NULL;

    /*
     * Check access.  You can read something that is write only, as the
     * header is stashed in memory.
     */
    if (writeComment && ((rleFilePtr->perms & TCL_WRITABLE) == 0)) {
        TSippNotWritable (tSippGlobPtr->interp, handle);
        return NULL;
    }
    if (!writeComment && rleFilePtr->eof &&
        ((rleFilePtr->perms & TCL_WRITABLE) == 0)) {
        TSippAtEOF (tSippGlobPtr->interp, handle);
        return NULL;
    }

    return rleFilePtr;
}

/*=============================================================================
 * RLECommentPut --
 *   Store a comment in the comment table.  This routine is referenced by a
 * pointer in the RLE image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - The handle of the image.
 *   o name (I) - Name of the comment to store.
 *   o value (I) - Value of the comment to store.  If NULL, just store the
 *     comment name.
 * Returns:
 *   TRUE is all is OK, or FALSE if an error occurs.
 *-----------------------------------------------------------------------------
 */
static int
RLECommentPut (tSippGlobPtr, handle, name, value)
    tSippGlob_t  *tSippGlobPtr;
    char         *handle;
    char         *name;
    char         *value;
{
    rleFile_t   *rleFilePtr;

    rleFilePtr = RLECommentCmdSetup (tSippGlobPtr,
                                     handle,
                                     TRUE);  /* writeComment */
    if (rleFilePtr == NULL)
        return FALSE;

    TSippCommentPut ((char ***) &rleFilePtr->rleHeader.comments,
                     name,
                     value);
    return TRUE;
}

/*=============================================================================
 * RLECommentGet --
 *   Get a comment from the comment table.  This routine is referenced by a
 * pointer in the RLE image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - The handle of the image.
 *   o name (I) - Name of the comment to store.
 *   o valuePtr (O) - Value of the comment is returned here, or NULL if its
 *     not found.
 * Returns:
 *   TRUE is all is OK, or FALSE if an error occurs.
 *-----------------------------------------------------------------------------
 */
static int
RLECommentGet (tSippGlobPtr, handle, name, valuePtr)
    tSippGlob_t  *tSippGlobPtr;
    char         *handle;
    char         *name;
    char        **valuePtr;
{
    rleFile_t   *rleFilePtr;

    rleFilePtr = RLECommentCmdSetup (tSippGlobPtr,
                                     handle,
                                     FALSE);  /* writeComment */
    if (rleFilePtr == NULL)
        return FALSE;

    *valuePtr = TSippCommentGet ((char **) rleFilePtr->rleHeader.comments,
                                 name);
    return TRUE;
}

/*=============================================================================
 * RLECommentGetVec --
 *   Get the comment table.  This routine is referenced by a pointer in the
 * RLE image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - The handle of the image.
 *   o name (I) - Name of the comment to store.
 *   o argvPtr (O) - The comment vector is returned here, it might be NULL or
 *     empty.
 * Returns:
 *   TRUE is all is OK, or FALSE if an error occurs.
 *-----------------------------------------------------------------------------
 */
static int
RLECommentGetVec (tSippGlobPtr, handle, argvPtr)
    tSippGlob_t  *tSippGlobPtr;
    char         *handle;
    char       ***argvPtr;
{
    rleFile_t   *rleFilePtr;
    char        *comStr;

    rleFilePtr = RLECommentCmdSetup (tSippGlobPtr,
                                     handle,
                                     FALSE);  /* writeComment */
    if (rleFilePtr == NULL)
        return FALSE;

    *argvPtr =  (char **) rleFilePtr->rleHeader.comments;
    return TRUE;
}

/*=============================================================================
 * RLECommentDelete --
 *    Delete a comment from the image or all image comments.  This routine is
 * referenced by a pointer in the RLE image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - The handle of the image.
 *   o name (I) - Name of the comment to delete.  Ignored if not found.  If
 *     NULL, delete all image comments.
 * Returns:
 *    TRUE is all is OK, or FALSE if an error occurs.
 *-----------------------------------------------------------------------------
 */
static int
RLECommentDelete (tSippGlobPtr, handle, name)
    tSippGlob_t  *tSippGlobPtr;
    char         *handle;
    char         *name;
{
    rleFile_t *  rleFilePtr;

    rleFilePtr = RLECommentCmdSetup (tSippGlobPtr,
                                     handle,
                                     TRUE);  /* writeComment */
    if (rleFilePtr == NULL)
        return FALSE;

    if (name == NULL)
        TSippCommentFree ((char ***) &rleFilePtr->rleHeader.comments);
    else
        TSippCommentDelete ((char ***) &rleFilePtr->rleHeader.comments, 
                            name);

    return TRUE;
}

/*=============================================================================
 * TSippRLEInit --
 *   Initialized the Utah Raster Toolkit RLE commands and the RLE file
 *   table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippRLEInit (tSippGlobPtr)
    tSippGlob_t *tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippRLEOpen",       (Tcl_CmdProc *) SippRLEOpen},
        {"SippRLEClose",      (Tcl_CmdProc *) SippRLEClose},
        {"SippRLEInfo",       (Tcl_CmdProc *) SippRLEInfo},
        {NULL,                NULL}
    };

    static tSippStorageClass_t storageClass = {
        "rlefile",                  /* handlePrefix  */
        7,                          /* prefixSize    */
        NULL,                       /* identFunc     */
        TSIPP_SCAN_BOTTOM_UP,       /* scanDirection */
        FALSE,                      /* bitMapOptimal */
        RLEOutputStart,
        RLEOutputLine,
        RLEOutputBitMap,
        RLEOutputEnd,
        RLECopyImage,
        RLECommentPut,
        RLECommentGet,
        RLECommentGetVec,
        RLECommentDelete
    };

    tSippGlobPtr->rleTblPtr =
        Tcl_HandleTblInit ("rlefile", sizeof (rleFile_t), 4);
    tSippGlobPtr->rleCleanUpProc = RLECleanUp;

    TSippInitCmds (tSippGlobPtr, cmdTable);
    TSippAddStorageClass (tSippGlobPtr, &storageClass);
}

/*=============================================================================
 * RLECleanUp --
 *   Close all RLE files and release all associated resources.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static void
RLECleanUp (tSippGlobPtr)
    tSippGlob_t *tSippGlobPtr;
{
    int         walkKey = -1;
    rleFile_t *rleFilePtr;

    while (TRUE) {
        rleFilePtr = Tcl_HandleWalk (tSippGlobPtr->rleTblPtr, &walkKey);
        if (rleFilePtr == NULL)
            break;
        if (!CloseRLEFile (tSippGlobPtr, "RLECleanUp", rleFilePtr))
            Tcl_ResetResult (tSippGlobPtr->interp);
    }
    Tcl_HandleTblRelease (tSippGlobPtr->rleTblPtr);
    tSippGlobPtr->rleTblPtr = NULL;
}

#else /* TSIPP_HAVE_RLE */

/*=============================================================================
 * SippRLEIsNotHere --
 *   Implements the a command that returns a error indicates that the Utah
 * Raster Toolkit is not available.  If this module is compiled with
 * TSIPP_HAVE_RLE is not defined, all RLE commands are bound to this command.
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippRLEIsNotHere (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    Tcl_AppendResult (interp, "The Utah Raster Toolkit RLE library has not ",
                      "been linked into\n", "this program. ", argv [0],
                      " is not available", (char *) NULL);
    return TCL_ERROR;
}

/*=============================================================================
 * TSippRLEInit --
 *   Initialization when the Utah Raster Toolkit is not available.  Bind all
 *   commands to a routine that returns an error.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippRLEInit (tSippGlobPtr)
    tSippGlob_t *tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippRLEOpen",       (Tcl_CmdProc *) SippRLEIsNotHere},
        {"SippRLEClose",      (Tcl_CmdProc *) SippRLEIsNotHere},
        {"SippRLEPutCom",     (Tcl_CmdProc *) SippRLEIsNotHere},
        {"SippRLEGetCom",     (Tcl_CmdProc *) SippRLEIsNotHere},
        {"SippRLEDelCom",     (Tcl_CmdProc *) SippRLEIsNotHere},
        {"SippRLEInfo",       (Tcl_CmdProc *) SippRLEIsNotHere},
        {NULL,                NULL}
    };

    tSippGlobPtr->rleTblPtr = NULL;
    tSippGlobPtr->rleCleanUpProc = NULL;

    TSippInitCmds (tSippGlobPtr, cmdTable);
}

#endif /* TSIPP_HAVE_RLE */
