/*
 *=============================================================================
 *                              tSippComment.c
 *-----------------------------------------------------------------------------
 * General commands for manipulating image comments (really image environment
 * variables).  This calls functions in the image storage class table to
 * actually operate on the 
 *-----------------------------------------------------------------------------
 * Copyright 1992-1995 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * Our own functions are used to manage comments and other aspects of the RLE
 * header to prevent memory lose that occurs in the RLE library.
 *-----------------------------------------------------------------------------
 * $Id: tSippComment.c,v 1.2 1995/08/15 08:38:59 markd Exp $
 *=============================================================================
 */

#include "tSippInt.h"

/*
 * Internal prototypes.
 */
static int
FindComment _ANSI_ARGS_((char  **comments,
                         char   *name,
                         int    *numComPtr));
static char *
CommentKeyedList _ANSI_ARGS_((char  **comments));


/*=============================================================================
 * FindComment --
 *   Find a comment in the comment array.
 * 
 * Parameters:
 *   o comments (I) - Pointer to the comment array.
 *   o name (I) - The name of the comment to look for.
 *   o numComPtr (O) - The length of the comment array, excluding the NULL,
 *     is returned here.  If NULL, the the length is not returned.
 * Return:
 *   The index of the comment, or -1 if its not found.
 *-----------------------------------------------------------------------------
 */
static int
FindComment (comments, name, numComPtr)
    char  **comments;
    char   *name;
    int    *numComPtr;
{
    int    idx, nameLen, numCom;

    if (comments == NULL) {
        if (numComPtr != NULL)
            *numComPtr = 0;
        return -1;
    }

    nameLen = strlen (name);
    for (idx = 0; comments [idx] != NULL;  idx++) {
        if (STRNEQU (name, comments [idx], nameLen) &&
            ((comments [idx][nameLen] == '=') ||
             (comments [idx][nameLen] == '\0')))
            break;
    }

    if (numComPtr != NULL) {
        for (numCom = idx; comments [numCom] != NULL;  numCom++)
            continue;
        *numComPtr = numCom;
    }

    return (comments [idx] == NULL) ? -1 : idx;
   
}

/*=============================================================================
 * CommentKeyedList --
 *   Return a Tcl keyed list of all the comments in a comment array.
 *
 * Parameters:
 *   o comments (I) - The comment array.
 * Returns:
 *   A pointer to a dynamic keyed list.  This is returned even if the array is
 * empty.
 *-----------------------------------------------------------------------------
 */
static char *
CommentKeyedList (comments)
    char  **comments;
{
    int    idx, numCom;
    char **elements, *keyedList, *sepPtr, *pair [2];

    if (comments == NULL)
        return strdup ("");  /* Empty list */

    /*
     * Determine length of the list and allocate and array to hold the
     * keyedlist elements.
     */
    for (numCom = 0; comments [numCom] != NULL; numCom++)
        continue;
    elements = (char **) alloca (numCom * sizeof (char *));

    for (idx = 0; idx < numCom; idx++) {
        /*
         * Temporarily split the string into name and optional value.
         */
        pair [0] = comments [idx];

        sepPtr = strchr (comments [idx], '=');
        if (sepPtr != NULL) {
            *sepPtr = '\0';
            pair [1] = sepPtr + 1;
        } else {
            pair [1] = "";
        }
        /*
         * Add to keyed list and put back `='.
         */
        elements [idx] = Tcl_Merge (2, pair);
        if (sepPtr != NULL)
            *sepPtr = '=';
    }
    
    /* 
     * Create the keyed list and cleanup the dynamic entries.
     */
    keyedList = Tcl_Merge (numCom, elements);
    
    for (idx = 0; idx < numCom; idx++)
        sfree (elements [idx]);

    return keyedList;
}

/*=============================================================================
 * TSippCommentPut --
 *   Add a comment to a comment array.
 *
 * Parameters:
 *   o commentsPtr (I/O) - Pointer to comment array.
 *   o name (I) - The name of the comment.
 *   o value (I) - The value associated with the comment.  NULL if no value is
 *     to be associated with the comment.
 *-----------------------------------------------------------------------------
 */
void
TSippCommentPut (commentsPtr, name, value)
    char  ***commentsPtr;
    char    *name;
    char    *value;
{
    int    comLen, numCom, idx;
    char  *newCom;
    char **comments = *commentsPtr;

    /*
     * Build up the new comment.
     */
    comLen = strlen (name) + 1;
    if (value != NULL) 
        comLen += strlen (value) + 1;
    newCom = smalloc (comLen);
    strcpy (newCom, name);
    if (value != NULL) {
        strcat (newCom, "=");
        strcat (newCom, value);
    }

    /*
     * Search for comment and either allocate space for a new entry or replace
     * an existing one.  One one element is added at a time, an there is always
     * room at the end.
     */
    idx = FindComment (comments, name, &numCom);
    if (idx < 0) {
        if (comments == NULL) {
            comments = (char **) smalloc (2 * sizeof (char **));
        } else {
            comments =  (char **) srealloc (comments,
                                            (numCom + 2) * sizeof (char **));
        }

        comments [numCom]     = newCom;
        comments [numCom + 1] = NULL;

    } else {
        sfree (comments [idx]);
        comments [idx] = newCom;
    }

    *commentsPtr = comments;
}

/*=============================================================================
 * TSippCommentGet --
 *   Find an comment's value in a comment array.
 *
 * Parameters:
 *   o comments (I) - The comment array.
 *   o name (I) - The name of the comment to find.
 * Returns:
 *   A pointer to the value or NULL if the comment is not found.  If the
 * comment exists, but has no value, a pointer to a empty string is returned.
 *-----------------------------------------------------------------------------
 */
char *
TSippCommentGet (comments, name)
    char  **comments;
    char    *name;
{
    int idx, len;

    idx = FindComment (comments, name, NULL);
    if (idx < 0)
        return NULL;

    len = strlen (name);
    if (comments [idx] [len] == '=')
        len++;

    return &(comments [idx] [len]);
}

/*=============================================================================
 * TSippCommentDelete --
 *   Delete a comment from a comment array.  If the comment is not found,
 * nothing is done.
 *
 * Parameters:
 *   o commentsPtr (I/O) - Pointer to comment array.
 *   o name (I) - The name of the comment to find.
 *-----------------------------------------------------------------------------
 */
void
TSippCommentDelete (commentsPtr, name)
    char  ***commentsPtr;
    char    *name;
{
    char **comments = *commentsPtr;
    int    idx, numCom;

    idx = FindComment (comments, name, &numCom);
    if (idx < 0)
        return;

    /*
     * Free entry and overlay with the last entry.
     */
    sfree (comments [idx]);

    comments [idx] = comments [numCom - 1];
    comments [numCom - 1] = NULL;
}

/*=============================================================================
 * TSippCommentFree --
 *   Free up all memory that is associated with a comment array and set the
 * array to NULL.
 * 
 * Parameters:
 *   o commentsPtr (I/O) - Pointer to comment array.
 *-----------------------------------------------------------------------------
 */
void
TSippCommentFree (commentsPtr)
    char  ***commentsPtr;
{
    int    idx;
    char **comments = *commentsPtr;

    if (comments == NULL)
        return;

    for (idx = 0; comments [idx] != NULL; idx++) {
        sfree (comments [idx]);
    }
    sfree (comments);

    *commentsPtr = NULL;
}

/*=============================================================================
 * TSippCommentCopy --
 *   Copy a comment array and all its strings.
 * 
 * Parameters:
 *   o comments (I/O) - The comment array.
 * Returns:
 *   A pointer to the new array, or NULL of in input array was null.
 *-----------------------------------------------------------------------------
 */
char **
TSippCommentCopy (comments)
    char  **comments;
{
    int    idx, numCom;
    char **newComments;

    if (comments == NULL)
        return NULL;

    for (numCom = 0; comments [numCom] != NULL; numCom++)
        continue;

    newComments = (char **) smalloc ((numCom + 1)  * sizeof (char *));

    for (idx = 0; idx < numCom; idx++)
        newComments [idx] = strdup (comments [idx]);
    newComments [idx] = NULL;

    return newComments;
}

/*=============================================================================
 * TSippCommentMerge --
 *   Merge two sets of comments together.  The result will contain all unique
 * comments between the old and new set.  If a comment name is in both the old
 * and new, the new overrides.
 * 
 * Parameters:
 *   o oldComments (I) - Pointer to old comment array.
 *   o newCommentsPtr (I/O) - Pointer to new comment array.  The modified array
 *     will be returned here.
 * Returns:
 *   A pointer to the new array, or NULL of in input array was null.
 *-----------------------------------------------------------------------------
 */
void
TSippCommentMerge (oldComments, newCommentsPtr)
    char   **oldComments;
    char  ***newCommentsPtr;
{
    char **newComments = *newCommentsPtr;
    int    oldIdx;
    char  *sepPtr, *pair [2];

    if (oldComments == NULL)
        return;

    /*
     * Run through the old comments and check if each one is in the
     * new array, if not, add it.
     */

    for (oldIdx = 0; oldComments [oldIdx] != NULL; oldIdx++) {
        /*
         * Temporarily split the string into name and optional value.
         */
        pair [0] = oldComments [oldIdx];

        sepPtr = strchr (oldComments [oldIdx], '=');
        if (sepPtr != NULL) {
            *sepPtr = '\0';
            pair [1] = sepPtr + 1;
        } else {
            pair [1] = "";
        }

        /*
         * If its not in the new list, add it.  Put back the '='.
         */
        if (FindComment (newComments, pair [0], NULL) < 0)
            TSippCommentPut (&newComments,
                             pair [0],
                             pair [1]);
        if (sepPtr != NULL)
            *sepPtr = '=';
    }

    *newCommentsPtr = newComments;

    return;
}

/*=============================================================================
 * CommentCmdSetup --
 *   Utility procedure to set up for one of the comments commands.  If converts
 * the handle to and image store class and validates the comment name.  Also
 * make sure rendering is not in progress.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A RLE handle.
 *   o name (I) - The name of the comment.  If NULL, then it is not validated
 * Returns:
 *   A pointer to the image store class, or NULL if an error occured.  Errors
 * returned in tSippGlobPtr->interp->result.
 *-----------------------------------------------------------------------------
 */
static tSippStorageClass_t *
CommentCmdSetup (tSippGlobPtr, handle, name)
    tSippGlob_t    *tSippGlobPtr;
    char           *handle;
    char           *name;
{
    tSippStorageClass_t *storageClassPtr;

    if (tSippGlobPtr->rendering) {
        TSippNotWhileRendering (tSippGlobPtr->interp);
        return NULL;
    }

    storageClassPtr = TSippFindStorageClass (tSippGlobPtr,
                                             handle);
    if (storageClassPtr == NULL)
        return NULL;

    if (storageClassPtr->commentDelete == NULL) {
        Tcl_AppendResult (tSippGlobPtr->interp, "image comments are not ",
                          "supported for \"",
                          storageClassPtr->handlePrefix, "\" images\"",
                          (char *) NULL);
        return NULL;
    }

    if ((name != NULL)  && (strpbrk (name, " \f\n\r\t\v=") != NULL)) {
        Tcl_AppendResult (tSippGlobPtr->interp, "name may not contain the ",
                          "`=' or whitespace characters", (char *) NULL);
        return NULL;
    }
    return storageClassPtr;

    
}

/*=============================================================================
 * SippCommentPut --
 *   Implements the command:
 *     SippCommentPut handle name [value]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippCommentPut (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t         *tSippGlobPtr = (tSippGlob_t *) clientData;
    tSippStorageClass_t *storageClassPtr;

    if ((argc < 3) || (argc > 4)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " handle name [value]", (char *) NULL);
        return TCL_ERROR;
    }

    storageClassPtr = CommentCmdSetup (tSippGlobPtr,
                                       argv [1],
                                       argv [2]);
    if (storageClassPtr == NULL)
        return TCL_ERROR;

    if (!storageClassPtr->commentPut (tSippGlobPtr,
                                      argv [1],
                                      argv [2],
                                      argv [3]))
        return TCL_ERROR;
    return TCL_OK;
}

/*=============================================================================
 * SippCommentGet --
 *   Implements the command:
 *     SippCommentGet handle [name] [retvar | {}]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippCommentGet (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t          *tSippGlobPtr = (tSippGlob_t *) clientData;
    tSippStorageClass_t  *storageClassPtr;
    char                 *comStr, **comVec;

    if ((argc < 2) || (argc > 4)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " handle [name] [retvar | {}]", (char *) NULL);
        return TCL_ERROR;
    }

    storageClassPtr = CommentCmdSetup (tSippGlobPtr,
                                       argv [1],
                                       argv [2]);
    if (storageClassPtr == NULL)
        return TCL_ERROR;

    /*
     * If no name, return a keyed list of all comments.
     */
    if (argc == 2) {
        if (!storageClassPtr->commentGetVec (tSippGlobPtr,
                                             argv [1],
                                             &comVec))
            return TCL_ERROR;
        Tcl_SetResult (interp,
                       CommentKeyedList (comVec),
                       TCL_DYNAMIC);
        return TCL_OK;
    }

    /*
     * Search for a single comment.  If comment was not found, either return
     * false or leave result an empty string.  If its was found, either return
     * it in the variable or as the command result.
     */
    if (!storageClassPtr->commentGet (tSippGlobPtr,
                                      argv [1],
                                      argv [2],
                                      &comStr))
        return TCL_ERROR;

    if (comStr == NULL) {
        if (argc == 4)
            interp->result = "0";
    } else {
        if (argc == 4) {
            if (argv [3][0] != '\0') {
                if (Tcl_SetVar (interp, argv [3], comStr, 
                                TCL_LEAVE_ERR_MSG) == NULL)
                    return TCL_ERROR;
            }
            interp->result = "1";
        } else {
            Tcl_SetResult (interp, comStr, TCL_STATIC);
        }
    }
    return TCL_OK;
}

/*=============================================================================
 * SippCommentDelete --
 *   Implements the command:
 *     SippCommentDelete handle [name]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippCommentDelete (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_t          *tSippGlobPtr = (tSippGlob_t *) clientData;
    tSippStorageClass_t  *storageClassPtr;

    if ((argc < 2) || (argc > 3)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " handle [name]", (char *) NULL);
        return TCL_ERROR;
    }

    storageClassPtr = CommentCmdSetup (tSippGlobPtr,
                                       argv [1],
                                       argv [2]);
    if (storageClassPtr == NULL)
        return TCL_ERROR;

    if (argc == 2) {
        if (!storageClassPtr->commentDelete (tSippGlobPtr,
                                             argv [1],
                                             NULL))
            return TCL_ERROR;
    } else {
        if (!storageClassPtr->commentDelete (tSippGlobPtr,
                                             argv [1],
                                             argv [2]))
            return TCL_ERROR;
    }
    return TCL_OK;
}


/*=============================================================================
 * TSippCommentInit --
 *   Initialized the comment commands, including creating the object table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippCommentInit (tSippGlobPtr)
    tSippGlob_t    *tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippCommentPut",         (Tcl_CmdProc *) SippCommentPut},
        {"SippCommentGet",         (Tcl_CmdProc *) SippCommentGet},
        {"SippCommentDelete",      (Tcl_CmdProc *) SippCommentDelete},
        {NULL,                     NULL}
    };

    TSippInitCmds (tSippGlobPtr, cmdTable);
}
