# file:saveAsMapfile maped
# Args - maped - global array data
# Side effects -
# 1) Open File Selection Box to select file to save
# 2) Wait to select file
# 3) save file.
proc file:saveAsMapfile {maped} {
    upvar #0 $maped data
    msg:set info {Select File Name for your Map Data.}
    set file [FileSelectionBox $maped .fs [set fmt [option get . mapFormat {}]] \
	    [lsearch -regexp $fmt $data(curFormat)] "Save Map File"]
    if {[lindex $file 0] != {}} {
	set data(curFileName) [lindex $file 0]
	set data(curFormat) [lindex $file 1]
    } else {
	msg:del
	return; # cancel
    }
    if {$data(curFormat) == {Client Side Map}} {
	msg:set info {Select/Input your Map Name.}
	if {[set name \
		[NameSelectionBox $maped .ns $data(curFileName)]] == {}} {
	    msg:del
	    return; # cancel 
	} else {
	    set data(curName) $name
	}
    }
    file:saveMapFile $maped
}

proc file:saveMapFile {maped} {
    upvar #0 $maped data
    if {$data(curFileName) == {} || \
	    ($data(curFormat) == {Client Side Map} && $data(curName) == {})} {
	file:saveAsMapfile $maped
    }
    msg:set info "Saveing $data(curFileName) as $data(curFormat) format ..."
    # cleate bakup file if current map file exists.
    set bakfile "[file rootname $data(curFileName)].bak"
    if {[file isfile $data(curFileName)] && \
	    [file readable $data(curFileName)]} {
	if {[catch [list open $bakfile w] bak]} {msg:set error $bak; return}
	if {[catch [list open $data(curFileName) r] fp]} {
	    msg:set error $fp
	    return
	}
	while {[gets $fp line] >= 0} {puts $bak $line}
	catch [list close $fp]; catch [list close $bak]
    }
    # write new map file
    if {[catch [list open $data(curFileName) w] fp]} {
	msg:set error $fp
	return
    }
    # if Client Side Map then we must keep outside of <map> ... </map>
    if {$data(curFormat) == {Client Side Map} && \
	    [file isfile $bakfile]} {
	set endHTML {}
	if {[catch [list open $bakfile r] bak]} {
	    msg:set error $bak
	    return
	}
	set mapnam {}
	while {[gets $bak line] >= 0} {
	    if {[regexp -nocase {<map name=\"(.*)\">} $line line mapnam] && \
		    $mapnam == $data(curName)} {
		file:skipTo $bak </map>; break
	    } elseif {[regexp -nocase {</html>} $line dummy]} {
		set endHTML $line; break
	    } else {
		puts $fp $line
	    }
	}
	puts $fp "<MAP NAME=\"$data(curName)\">"
	canvas:writeFileBody $maped $fp
	puts $fp "</MAP>"	
	if {$endHTML != {}} {puts $fp $endHTML}
	while {[gets $bak line] >=0} {
	    puts $fp $line
	}
	catch [list close $bak]
    } elseif {$data(curFormat) == {Client Side Map}} {
	if {$data(allowNetscapeBUG)} {
	    puts $fp "<MAP NAME=$data(curName)>"
	} else {
	    puts $fp "<MAP NAME=\"$data(curName)\">"
	}
	canvas:writeFileBody $maped $fp
	puts $fp "</MAP>"
    } else {
	canvas:writeFileBody $maped $fp
    }	
    catch [list close $fp]
    msg:del
}

proc file:skipTo {fp fmt} {
    while {[gets $fp line] >= 0} {
	if {[regexp -nocase $fmt $line] == 1} {
	    return $line
	}
    }
    return {}
}
# file:saveAsImageFile maped
# args - maped - global array data
# side effect -
# save current image file if exists.
proc file:saveAsImageFile {maped} {
    upvar #0 $maped data
    msg:set info {Select/Input Image File Name.}
    set file [FileSelectionBox $maped .fs {PPM} 0 "Save Image File"]
    if {$file != {}} {
	msg:set info "Saving [lindex $file 0] as [lindex $file 1] format ..."
	switch [lindex $file 1] {
	    PPM {canvas:saveImage [lindex $file 0] gif}
	}
    }
    msg:del
}    

proc file:loadFromArgs {maped arg} {
    upvar #0 $maped data
    if {[set format [lindex $arg 1]] == {}} {
	set format [lindex [lindex [option get . mapFormat {}] 0] 0]
    }
    set file [lindex $arg 0]
    file:load $maped $file $format
}    

# file:loadMapfile maped
# Args - maped - global array data
# Side effects -
# 1) Open File Selection Box to select load file.
# 2) Wait to select file
# 3) load file.
proc file:loadMapfile {maped} {
    upvar #0 $maped data
    msg:set info {Select/Input Map File Name.}
    set file [FileSelectionBox $maped .fs \
	    [set allfmt [option get . mapFormat {}]] \
	    [lsearch -regexp $allfmt $data(curFormat)] "Load Map File"]
    if {$file != {}} {
	set format [lindex $file 1]
	set file [lindex $file 0]
	file:load $maped $file $format
    } else {
	msg:del
    }
}

proc file:load {maped file format} {
    upvar #0 $maped data
    if {[file isfile $file]} {
	msg:set info "Loading [lindex $file 0] as [lindex $file 1] format ..."
	switch $format {
	    {CERN Httpd} {file:loadCERNMap $file}
	    {NCSA Httpd} {file:loadNCSAMap $file}
	    {NetSite} {file:loadNetSiteMap $file}
	    {Client Side Map} {file:loadClientSideMap $maped $file}
	    default {msg:set error "wrong # format: shound be $allfmt."}
	}	    
	canvas:adjScrRegion
	set data(curFileName) $file
	set data(curFormat) $format
	canvas:lower Image
	msg:del
    }
    return $file
}
proc file:initLoadCmd {} {
    foreach i {default rectangle polygon point rect poly area} {
	if {[info commands $i] != {}} {
	    rename $i {}
	}
    }
}

proc file:loadCERNMap {file} {
    file:initLoadCmd
    proc default {URL args} {
	canvas:setDefaultURL $URL
    }
    proc circle {coords r URL args} {
	if {[scan $coords {(%d,%d)} x y] == 2} {
	    canvas:lower [canvas:drawCircle $x $y $r $URL]
	} else {
	    tkerror "wrong # args: should be \"circle (x,y) r URL\""
	}
    }
    proc rectangle {coords1 coords2 URL args} {
	if {[scan $coords1 {(%d,%d)} x1 y1] == 2 && \
		[scan $coords2 {(%d,%d)} x2 y2] == 2} {
	    canvas:lower [canvas:drawRectangle $x1 $y1 $x2 $y2 $URL]
	} else {
	    tkerror "wrong # args: should be \"rectangle (x1,y1) (x2,y2) URL\""
	}
    }
    proc polygon {args} {
	foreach i [lrange $args 0 [expr [llength $args] - 2]] {
	    if {[scan $i {(%d,%d)} x y] == 2} {
		lappend coords $x $y
	    } else {
		tkerror "wrong # args: should be \"polygon (x1,y1) ... (xn,yn) URL\""
		return
	    }
	}
	canvas:lower [canvas:drawPolygon $coords [lrange $args end end]]
    }
    proc point {coords URL args} {
	if {[scan $coords {(%d,%d)} x y] == 2} {
	    foreach i [canvas:drawPoint $x $y $URL] {
		canvas:lower $i
	    }
	} else {
	    tkerror "wrong # args: should be \"point (x,y) URL\""
	}
    }
    canvas:delItems
    source $file
}

proc file:loadNCSAMap {file} {
    file:initLoadCmd
    proc default {URL args} {
	canvas:setDefaultURL $URL
    }
    proc circle {URL coords r args} {
	if {[scan $coords {%d,%d} x y] == 2} {
	    canvas:lower [canvas:drawCircle $x $y $r $URL]
	} else {
	    tkerror "wrong # args: should be \"circle URL x,y r\""
	}
    }
    proc rect {URL coords1 coords2 args} {
	if {[scan $coords1 {%d,%d} x1 y1] == 2 && \
		[scan $coords2 {%d,%d} x2 y2] == 2} {
	    canvas:lower [canvas:drawRectangle $x1 $y1 $x2 $y2 $URL]
	} else {
	    tkerror "wrong # args: should be \"rect URL x1,y1 x2,y2\""
	}
    }
    proc poly {URL args} {
	foreach i $args {
	    if {[scan $i {%d,%d} x y] == 2} {
		lappend coords $x $y
	    } else {
		tkerror "wrong # args: should be \"poly URL x1,y1 ... xn,yn\""
		return
	    }
	}
	canvas:lower [canvas:drawPolygon $coords $URL]
    }
    proc point {URL coords args} {
	if {[scan $coords {%d,%d} x y] == 2} {
	    foreach i [canvas:drawPoint $x $y $URL] {
		canvas:lower $i
	    }
	} else {
	    tkerror "wrong # args: should be \"point URL x,y\""
	}
    }
    canvas:delItems
    source $file
}

proc file:loadNetSiteMap {file} {
    file:initLoadCmd
    proc default {URL args} {
	canvas:setDefaultURL $URL
    }
    proc circle {URL cx cy rx ry args} {
	if {[catch [list set cx [expr int($cx)]]] || \
		[catch [list set cy [expr int($cy)]]] || \
		[catch [list set rx [expr int($rx)]]] || \
		[catch [list set ry [expr int($ry)]]]} {
	    tkerror "wrong # args: should be \"circle URL cx cy rx ry"
	} else {
	    canvas:lower [canvas:drawCircle $cx $cy \
		    [expr hypot($rx-$cx,$ry-$cy)] $URL]
	}
    }
    proc rect {URL x1 y1 x2 y2 args} {
	if {[catch [list set x1 [expr int($x1)]]] || \
		[catch [list set y1 [expr int($y1)]]] || \
		[catch [list set x2 [expr int($x2)]]] || \
		[catch [list set y2 [expr int($y2)]]]} {
	    tkerror "wrong # args: should be \"rect URL x1 y1 x2 y2\""
	} else {
	    canvas:lower [canvas:drawRectangle $x1 $y1 $x2 $y2 $URL]
	}
    }
    proc poly {URL args} {
	if {[llength $args] % 2 == 1} {
	    tkerror "wrong # args: should be \"poly URL x1 y1 ... xn yn\""
	    return
	}
	foreach i $args {
	    if {[catch [list set i [expr int($i)]]]} {
		tkerror "wrong # args: should be \"polygon URL x1,y1 ... xn,yn\""
		return
	    } else {
		lappend coords $i
	    }
	}
	canvas:lower [canvas:drawPolygon $coords $URL]
    }
    canvas:delItems
    source $file
}

proc file:loadClientSideMap {maped file} {
    upvar #0 $maped data
    proc area {shape args} {
	if {[regexp -nocase -- {coords=.*} $shape shape]} {
	    set args [concat $shape $args]
	    set shape {rect}
	} elseif {![regexp -nocase -- {shape=(.*)} $shape i shape]} {
	    tkerror {wrong # args: should be AREA [SHAPE=shape] \
		    COORDS="x,y,..." [HREF="refarence"] [NOHREF] [ALT="alt"]}
	    return
	}
	set nohref 0; set href {}; set alt {}; set coords {}
	set curAttrib {}
	foreach i [split $args {= }] {
	    if {[regexp -nocase -- {^coords$} $i dummy]} {
		# COORDS attribute
		set curAttrib coords
	    } elseif {[regexp -nocase -- {^href$} $i dummy]} {
		# HREF attribute
		set curAttrib href
	    } elseif {[regexp -nocase -- {^nohref$} $i dummy]} {
		set nohref 1; # NOHREF attribute
	    } elseif {[regexp -nocase -- {^alt$} $i i alt]} {
		# ALT attribute
		set curAttrib alt
	    } elseif {$curAttrib != {}} {
		lappend $curAttrib $i
	    } else {
		tkerror {wrong $i token: should be AREA [SHAPE=shape] \
			COORDS="x,y,..." [HREF="refarence"] [NOHREF] [ALT="alt"]}
		return
	    }
	}
	regexp {\\"(.*)\\"} [join $href] dummy href; # strip double quote
	regexp {\\"(.*)\\"} [join $alt] dummy alt; # strip double quote
	regexp {\\"(.*)\\"} [join $coords] dummy coords; # strip double quote
	if {$nohref == 1 || $href == "\{\}"} {set href {}}
	if {$alt == "\{\}"} {set alt {}}
	switch -exact [string tolower $shape] {
	    rect {
		if {[scan $coords {%d,%d,%d,%d} x1 y1 x2 y2] != 4} {
		    tkerror "wrong # coords in $coords"
		    return
		} else {
		    canvas:lower [canvas:drawRectangle $x1 $y1 $x2 $y2 \
			    $href $alt]
		}
	    }
	    circle {
		if {[scan $coords {%d,%d,%d} x y r] != 3} {
		    tkerror "wrong # coords in $coords"
		    return
		} else {
		    canvas:lower [canvas:drawCircle $x $y $r $href $alt]
		}
	    }
	    polygon {
		foreach i [split $coords {,}] {
		    if {[catch [list set i [expr int($i)]]]} {
			tkerror "wrong # coords in $i at $coords"
			return
		    } else {
			lappend newCoords $i
		    }
		}
		canvas:lower [canvas:drawPolygon $newCoords $href $alt]
	    }
	}
    }
    if {[set name [NameSelectionBox $maped .ns $file]] == {}} {
	return
    }
    set data(curName) $name
    if {[catch [list open $file] fp]} {
	msg:set error $fp
	return
    }
    set mapnam {}
    if {$data(allowNetscapeBUG)} {
	while {[gets $fp line] >= 0} {
	    if {[regexp -nocase -- {<map name=(.*)>} $line line mapnam] && \
		    $mapnam == $data(curName)} {
		# serach specific name
		break
	    }
	}
    } else {
	while {[gets $fp line] >= 0} {
	    if {[regexp -nocase -- {<map name="(.*)">} $line line mapnam] && \
		    $mapnam == $data(curName)} {
		# serach specific name
		break
	    }
	}
    }
    set buffer {}
    while {[gets $fp line] >= 0 && \
	    ![regexp -nocase -- {</map>} $line line]} {
	set buffer [concat $buffer $line]
    }
    catch [list close $fp]
    canvas:delItems; # remove old map items
    if {$data(DEBUG)} {puts stdout "buffer = $buffer"}
    foreach i [split [join $buffer] {<>}] {
	if {$i != {} && [regexp -nocase -- {area .*} $i i]} {
	    eval area [lrange $i 1 end]
	}
    }
}		    

# file:loadImagefile maped
# Args - maped - global array data
# Side effects -
# 1) Open File Selection Box to select image file to load.
# 2) Wait to select file
# 3) load image file.
proc file:loadImagefile {maped} {
    upvar #0 $maped data
    set format [list GIF PPM XBM]
    if {[lsearch -exact [image types] pixmap] >= 0} {
	lappend format XPM
    }
    set tmpPath [pwd]
    set file [FileSelectionBox $maped .fs $format 0 "Load Image File"]
    catch [list cd $tmpPath]
    msg:set info "Loading [lindex $file 0] as [lindex $file 1] format ..."
    switch [lindex $file 1] {
	PPM -
	GIF {canvas:loadImage [lindex $file 0] photo}
	XBM {canvas:loadImage [lindex $file 0] bitmap}
	XPM {canvas:loadImage [lindex $file 0] pixmap}
    }
    if {$data(defaultURL) != ""} {
	canvas:setDefaultURL $data(defaultURL)
    }
    canvas:adjScrRegion
    msg:del
}    

# FileSelectionBox: Motif like File Selection Box
proc FileSelectionBox {maped w \
	{format {}} {idx 0} {title FileSelectionBox} {filter "*"}} {
    upvar #0 $maped data

    # 0) Global Resource definition
    option add *[string trimleft $w*Frame.relief .] raised widgetDefault
    option add *[string trimleft $w*Frame.borderWidth .] 1 widgetDefault
    option add *[string trimleft $w*Label.anchor .] w widgetDefault
    option add *[string trimleft $w*Entry.relief .] sunken widgetDefault
    option add *[string trimleft $w*Entry.borderWidth .] 2 widgetDefault
    option add *[string trimleft $w*Listbox.relief .] sunken widgetDefault
    option add *[string trimleft $w*Listbox.borderWidth .] 2 widgetDefault
    option add *[string trimleft $w*Listbox.setGrid .] true widgetDefault
    option add *[string trimleft $w*Scrollbar.Orient .] vertical widgetDefault
    option add *[string trimleft $w*Menu.anchor .] w widgetDefault
    option add *[string trimleft $w*Menubutton.relief .] raised widgetDefault

    # 1) Set up TopLevelWidget
    toplevel $w -class Dialog
    wm title $w $title;    # Window Title
    wm iconname $w $title; # Icon Name

    # 2) Create Frames
    frame $w.top; # top frame which contain filter entry
    frame $w.mid1; # frame mid1 contain listboxes.
    frame $w.mid2; # frame mid2 contain selection entry
    frame $w.bottom; # frame bottom contain buttons.
    pack $w.top -side top -fill x -ipady 1m; # pack them
    pack $w.mid1 -side top -fill both -expand true -ipady 1m
    pack $w.mid2 $w.bottom -side top -fill x -ipady 1m

    # 3) create contents of frame top
    option add *[string trimleft $w.top.label.text .] {Filter} widgetDefault
    label $w.top.label; # title of filter entry
    entry $w.top.filter; # filter fields.
    pack $w.top.label $w.top.filter -side top -anchor w -fill x -padx 2m

    # 4) create listboxes
    option add *[string trimleft $w.mid1*Frame.relief .] flat widgetDefault
    # 4-0) create drive selector (for win only)
    if {[string match {[A-Z]:\\*} [pwd]]} {
	if {[info commands pwd.old] == {}} {
	    nename pwd pwd.orig
	    proc pwd {} {
		set drv [lindex [set pwd [split [pwd.old] {:}]] 1]
		regsub -all -- {\\} [lindex $pwd 0] {/} pwd
		return [list $pwd $drv]
	    }
	}
	set data(curDrive) [lindex [pwd] 1]
	frame $w.mid1.drvf; # contain drive menu
	option add *[string trimleft $w.mid1.drvf.label.text .] \
		{Drive:} widgetDefault
	label $w.mid1.drvf.label
	menubutton $w.mid1.drvf.menu -menu $w.mid1.drvf.menu.dm \
		-textvariable [subst $maped](curDrive)
	menu $w.mid1.drvf.menu.dm
	if {![info exists env(LASTDRIVE)]} {set env(LASTDRIVE) Z}
	if {[info exists env(DRIVELETTERS)]} {
	    set drives $env(DRIVELETTERS)
	} else {
	    set drives {A B C D E F G H I J K L M N O P Q R S T U V W X Y Z}
	}
	foreach i $drives {
	    $w.mid1.drvf.menu.dm add radiobutton -label $i -underline 0 \
		    -variable [subst $maped](curDrive) -value $i
	    if {$i == $env(LASTDRIVE)} {
		break
	    }
	}
	pack $w.mid1.drvf -fill x -expand true -side top -padx 2m
	pack $w.mid1.drvf.label -side left
	pack $w.mid1.drvf.menu -side left -fill x -anchor w
    }
    # 4-1) create directory listbox
    frame $w.mid1.left; # contain directory listbox
    frame $w.mid1.right; # contain file listbox
    pack $w.mid1.left $w.mid1.right -side left -padx 2m -fill both -expand true

    option add *[string trimleft $w.mid1.left.label.text .] \
	    {Directories} widgetDefault
    label $w.mid1.left.label
    listbox $w.mid1.left.l -yscrollcommand [list $w.mid1.left.scr set]
    scrollbar $w.mid1.left.scr -command [list $w.mid1.left.l yview]
    pack $w.mid1.left.label -side top -anchor w -fill x
    pack $w.mid1.left.scr -side right -fill y
    pack $w.mid1.left.l -fill both -expand true

    # 4-1) create file listbox
    option add *[string trimleft $w.mid1.right.label.text .] \
	    {Files} widgetDefault
    label $w.mid1.right.label;
    listbox $w.mid1.right.l -yscrollcommand [list $w.mid1.right.scr set]
    scrollbar $w.mid1.right.scr -command [list $w.mid1.right.l yview]
    pack $w.mid1.right.label -side top -fill x -anchor w 
    pack $w.mid1.right.scr -side right -fill y 
    pack $w.mid1.right.l -fill both -expand true

    # 5) create selection entry
    option add *[string trimleft $w.mid2.label.text .] \
	    {Selection:} widgetDefault
    label $w.mid2.label; entry $w.mid2.selection
    pack $w.mid2.label -anchor w -padx 2m
    pack $w.mid2.selection -fill x -anchor w -padx 2m
    if {$format != ""} {
	set data(tmpFormat) [lindex [lindex $format $idx] 0]
	option add *[string trimleft $w.mid2*Frame.relief .] flat widgetDefault
	frame $w.mid2.format
	option add *[string trimleft $w.mid2.format.label.text .] \
		Format: widgetDefault
	label $w.mid2.format.label
	menubutton $w.mid2.format.menu -menu $w.mid2.format.menu.fm \
		-textvariable [subst $maped](tmpFormat)
	menu $w.mid2.format.menu.fm
	foreach i $format {
	    if {[set idx [lindex $i 1]] == {}} {set idx 0}
	    $w.mid2.format.menu.fm add radiobutton \
		    -label [lindex $i 0] -underline $idx \
		    -variable [subst $maped](tmpFormat) -value [lindex $i 0]
	}
	pack $w.mid2.format -fill x -expand true -anchor w -padx 2m
	pack $w.mid2.format.label -side left
	pack $w.mid2.format.menu -side left -fill x -expand true
    }	

    # 6) create buttons
    option add *[string trimleft $w.bottom*Frame.relief .] flat widgetDefault
    option add *[string trimleft $w.bottom.fr1.ok.text .] OK widgetDefault
    frame $w.bottom.fr1
    button $w.bottom.fr1.ok \
	    -command [list $w:okProc $w.mid2.selection $maped]
    option add *[string trimleft $w.bottom.fr2.filter.text .] \
	    Filter widgetDefault
    frame $w.bottom.fr2
    button $w.bottom.fr2.filter \
	    -command [list $w:refreshFilter $maped $w $w.top.filter \
	    $w.mid1.left.l $w.mid1.right.l $w.mid2.selection]
    option add *[string trimleft $w.bottom.fr3.cancel.text .] \
	    Cancel widgetDefault
    frame $w.bottom.fr3
    button $w.bottom.fr3.cancel \
	    -command [list set [subst $maped](selectedFile) {}]
    pack $w.bottom.fr1.ok $w.bottom.fr2.filter $w.bottom.fr3.cancel \
	    -padx 2m -pady 1m -ipadx 1m -ipady 1m
    pack $w.bottom.fr1 $w.bottom.fr2 $w.bottom.fr3 \
	    -side left -expand true

    # 7) define usefull methods
    # callback for ok button ; return selected file
    proc $w:okProc {entry varname} {
	upvar #0 $varname data
	if {[info commands pwd.orig] != {}} {
	    return [set data(selectedFile) $data(curDrive):[$entry get]]
	} else {
	    return [set data(selectedFile) [$entry get]]
	}
    }
    # $w:refreshFilter: change filter entry, and update everything.
    proc $w:refreshFilter {maped w filterw dirw filew selectw} {
	upvar #0 $maped data
	if {[info command pwd.orig] != {}} {
	    set newPath $data(curDrive):[file dirname [$filterw get]]
	} else {
	    set newPath [file dirname [$filterw get]]
	}
	if {![catch [list cd $newPath] msg]} {
	    $w:exchangeEntry $selectw $newPath
	    return [$w:changeLists $dirw $filew \
		    "$newPath/[file tail [$filterw get]]"]
	} else {
	    tkerror $msg
	}
    }
    # $w:changeFilter: change filter entry
    proc $w:changeFilter {w filterw dirw} {
	return [$w:exchangeEntry $filterw "[lindex [pwd] 0]/[$dirw get \
		[$dirw curselection]]/[file tail [$filterw get]]"]
    }    
    # $w:changeSelection: change selection entry
    proc $w:changeSelection {w filew selectw} {
	if {[set index [$filew curselection]] != ""} {
	    if {[info command pwd.orig] != {}} {
		set pwd "[lindex [set pwd [pwd]] 1]:[lindex $pwd 0]"
	    } else {
		set pwd [pwd]
	    }
	    $w:exchangeEntry $selectw "$pwd/[$filew get $index]"
	}
	return [$selectw get]
    }
    
    # $w:exchangeEntry: set contents of item to entry w
    proc $w:exchangeEntry {w item} {
	$w delete 0 end
	$w insert 0 $item
	$w xview moveto [lindex [$w xview] 1]
	return $item
    }
    # $w:changeLists: update listboxes with given args
    proc $w:changeLists {dirw filew path} {
	$dirw delete 0 end
	$filew delete 0 end
	$dirw insert end {..}
	foreach i [lsort [glob -nocomplain -- [file dirname $path]/*]] {
	    if {[file isdirectory $i]} {
		$dirw insert end [file tail $i]
	    } elseif {[file isfile $i] && \
		    [string match [file tail $path] [set i [file tail $i]]]} {
		$filew insert end $i
	    }
	}
    }

    # 8) initialize filter
    if {[file dirname $filter] == "" || \
	    [string index [file dirname $filter] 0] != "/"} {
	if {[info commands pwd.orig] != {}} {
	    set filter "[lindex [set pwd [pwd]] 1]:[lindex $pwd 0]/$filter"
	} else {
	    set filter "[pwd]/$filter"
	}
    }
    $w.top.filter insert 0 $filter
    $w.mid2.selection insert 0 [file dirname $filter]
    $w:refreshFilter $maped $w $w.top.filter $w.mid1.left.l \
	    $w.mid1.right.l $w.mid2.selection

    # 9) Set up callbacks
    bindtags $w.mid1.left.l [list Listbox $w.mid1.left.l $w all]
    bindtags $w.mid1.right.l [list Listbox $w.mid1.right.l $w all]
    bind $w.top.filter <Return> [list $w:refreshFilter $maped $w %W \
	    $w.mid1.left.l $w.mid1.right.l $w.mid2.selection]
    bind $w.mid1.left.l <1> [list $w:changeFilter $w $w.top.filter %W]
    bind $w.mid1.left.l <Double-1> [list $w:refreshFilter $maped $w \
	    $w.top.filter %W $w.mid1.right.l $w.mid2.selection]
    bind $w.mid1.right.l <1> [list $w:changeSelection $w %W $w.mid2.selection]
    bind $w.mid1.right.l <Double-1> \
	    [list $w:okProc $w.mid2.selection $maped]
    bind $w.mid2.selection <Return> \
	    [list $w:okProc $w.mid2.selection $maped]
    
    # 10) Event Loop
    set oldFocus [focus]; # store current focus
    grab set $w;          # grab this window
    focus $w;             # set focus
    tkwait variable [subst $maped](selectedFile); # wait user selection ....
    destroy $w;           # delete this window
    focus $oldFocus;      # restore focus
    if {$format != {}} {
	return [list $data(selectedFile) $data(tmpFormat)]
    } else {
	return $data(selectedFile)
    }
}

proc NameSelectionBox {maped w file {title NameSelectionBox}} {
    upvar #0 $maped data
    # 0) Global Resource definition
    option add *[string trimleft $w*Frame.relief .] raised widgetDefault
    option add *[string trimleft $w*Frame.borderWidth .] 1 widgetDefault
    option add *[string trimleft $w*Label.anchor .] w widgetDefault
    option add *[string trimleft $w*Entry.relief .] sunken widgetDefault
    option add *[string trimleft $w*Entry.borderWidth .] 2 widgetDefault
    option add *[string trimleft $w*Listbox.relief .] sunken widgetDefault
    option add *[string trimleft $w*Listbox.borderWidth .] 2 widgetDefault
    option add *[string trimleft $w*Listbox.setGrid .] true widgetDefault
    option add *[string trimleft $w*Scrollbar.Orient .] vertical widgetDefault

    # 1) Set up TopLevelWidget
    toplevel $w -class Dialog
    wm title $w $title;    # Window Title
    wm iconname $w $title; # Icon Name

    # 2) Create Frames
    frame $w.top; # top frame which contain name listbox
    frame $w.mid; # contents selection entry
    frame $w.bottom; # frame bottom contain buttons.
    pack $w.top -side top -fill both -expand true -ipady 1m -ipadx 2m
    pack $w.mid $w.bottom -side top -fill x -ipady 1m -ipadx 2m

    # 3) Create name listbox parts
    option add *[string trimleft $w.top.label.text .] Names: widgetDefault
    label $w.top.label
    listbox $w.top.l -yscrollcommand [list $w.top.scr set]
    scrollbar $w.top.scr -command [list $w.top.l yview]
    pack $w.top.label -side top -fill x -anchor w 
    pack $w.top.scr -side right -fill y 
    pack $w.top.l -fill both -expand true

    # 4) Create Selection entry
    option add *[string trimleft $w.mid.label.text .] Names: widgetDefault
    label $w.mid.label
    entry $w.mid.entry
    pack $w.mid.label -side top -fill x -anchor w
    pack $w.mid.entry -side top -fill x
    
    # 5) Create buttons
    option add *[string trimleft $w.bottom*Frame.relief .] flat widgetDefault
    option add *[string trimleft $w.bottom.fr1.ok.text .] OK widgetDefault
    frame $w.bottom.fr1
    button $w.bottom.fr1.ok -command [list $w:okProc $maped $w.mid.entry]
    option add *[string trimleft $w.bottom.fr2.cancel.text .] \
	    Cancel widgetDefault
    frame $w.bottom.fr2
    button $w.bottom.fr2.cancel \
	    -command [list set [subst $maped](selectedName) {}]
    pack $w.bottom.fr1.ok $w.bottom.fr2.cancel \
	    -padx 2m -pady 1m -ipadx 1m -ipady 1m
    pack $w.bottom.fr1 $w.bottom.fr2 -side left -expand true

    # 6) define usefull methods
    proc $w:changeSelection {list entry} {
	if {[set index [$list curselection]] != ""} {
	    $entry delete 0 end
	    $entry insert 0 [$list get $index]
	    $entry xview moveto [lindex [$entry xview] 1]
	}	    
	return [$entry get]
    }
    proc $w:okProc {maped entry} {
	upvar #0 $maped data
	set data(selectedName) [$entry get]
	return $data(selectedName)
    }

    # 7) Set up bindings
    bindtags $w.top.l [list Listbox $w.top.l $w all]
    bind $w.top.l <1> [list $w:changeSelection $w.top.l $w.mid.entry]
    bind $w.top.l <Double-1> [list $w:okProc $maped $w.mid.entry]
    bind $w.mid.entry <Return> [list $w:okProc $maped $w.mid.entry]

    # 8) initialize listbox
    if {[file isfile $file]} {
	if {[catch [list open $file] fp]} {
	    msg:set error $fp
	    return {}
	}
	while {[gets $fp line] >= 0} {
	    if {[regexp -nocase -- {<MAP NAME=\"(.*)\">} $line line name]} {
		# "list up all names in the file
		$w.top.l insert end $name
	    }
	}
	catch [list close $fp]
    }
    if {$data(curName) != {} && \
	    [lsearch -exact [$w.top.l get 0 end] $data(curName)] == -1} {
	$w.top.l insert end $data(curName)
    }
    # 9) Event Loop    
    set oldFocus [focus]; # store current focus
    grab set $w;          # grab this window
    focus $w;             # set focus
    tkwait variable [subst $maped](selectedName); # wait user selection ....
    destroy $w;           # delete this window
    focus $oldFocus;      # restore focus
    puts stdout "return : $data(selectedName)"
    return $data(selectedName)
}

#
# Copy Procs for installer.
#
proc file:copy {srcFName distFName {mode {}}} {
    if {[regexp {[A-Z]:\\.*} [pwd] dummy] == 1} {
	set cpProc "copy"
    } else {
	set cpProc "cp"
    }
    catch [list exec $cpProc $srcFName $distFName]
}

proc file:copyWithRegsub {srcFName distFName regDat} {
    upvar $regDat data
    set len [llength [set regs [array get data]]]
    if {[catch [list open $srcFName r] srcFP]} {
	tkerror $srcFP; return
    }
    if {[catch [list open $distFName w] distFP]} {
	tkerror $distFP; return
    }
    while {[gets $srcFP line] >= 0} {
	for {set i 0} {$i < $len} {incr i 2} {
	    regsub -all -- [lindex $regs $i] $line \
		    [lindex $regs [expr $i + 1]] line
	}
	puts $distFP $line
    }
    catch [list close $srcFP]
    catch [list close $distFP]
}
    

