# applets.tcl,v 1.1 1995/11/17 00:42:00 steve Exp
#
#	PASTIME Project
#	Cooperative Research Centre for Advanced Computational Systems
#	COPYRIGHT NOTICE AND DISCLAIMER.
#
#	Copyright (c) 1995 ANU and CSIRO
#	on behalf of the participants in
#	the CRC for Advanced Computational Systems (ACSys)
#
# This software and all associated data and documentation ("Software")
# was developed for research purposes and ACSys does not warrant that 
# it is error free or fit for any purpose.  ACSys disclaims all liability
# for all claims, expenses, losses, damages and costs any user may incur 
# as a result of using, copying or modifying the Software.
#
# You may make copies of the Software but you must include all of this
# notice on any copy.
###
# tcl-handler creates a content-type handler for application/x-tcl documents
#
# We must be extremely careful not to perform substitutions or evaluate the 
# Applet code in the main (trusted) interpreter or otherwise the security of 
# the browser (and the user's entire environment) may be compromised.
#
# Facilities are also provided for Applets to interact with the main interpreter
# in a safe (restricted) manner.
#
# By default, applets are local to a hyperpage (unless they are the target
# of a <FORM SCRIPT=applet> element, in which case they are local to that form).
# All applets local to a hyperpage are destroyed when that page is reset.
# Applets may declare themselves to be local to a window, or global to the browser.
#
# All procedures and variables in this module are prefixed by "App"

# Problem: disambiguating applet inclusions: eg. 
# <html><body><a rel=embed href="applet.tcl"><a rel=embed href="applet.tcl"></body</html>
# The same applet source is used in two separate applet instances.
# We solve this by assigning a unique identifier to every document downloaded,
# and using that in creating applets.
#

proc Applet_handler {data win {start {}}} {
    set win [string trim $win]
    upvar #0 AppData$win app
    upvar #0 PR$win var

    # Store the Applet code in a variable for later evaluation
    # in a slave interpreter

    append app($var(id)) $data
}

proc Applet_evaluate {win} {
    upvar #0 AppData$win app
    upvar #0 PR$win var
    upvar #0 PS$win altvar

    if {!$var(eof)} return; # Wait until all data is loaded

    # Check that win really refers to a window
    if {[info exists altvar(win)]} {
	set win $altvar(win)
    }

    upvar #0 AppPage$win page
    upvar #0 AppForms$win forms
    upvar #0 AppFormsMap$win formmap

    # Create a slave interpreter for this applet
    # Safe-Tcl doesn't support Tk within safe interpreters (yet)
    # so we need to add in this functionality via aliases
    set slave [interp_create_safe_tk $var(id)]
    #set slave [interp create -safe -tk $var(id)]

    # Configure the slave interpreter

    $slave alias applet Applet_command $slave $win
    $slave alias exit Applet_destroy $slave

    $slave alias HMmap_reply HMmap_reply

    # Just for debugging
    $slave alias puts Applet_puts $slave

    # Expose some TclX commands to the slave
    $slave alias getclock getclock
    $slave alias convertclock convertclock
    $slave alias fmtclock fmtclock
    $slave alias random random

    # Expose some BLT commands to the slave
    $slave alias blt_table Applet_blt_table $slave

    # Allow the applet access to the hyperpage from which it was loaded
    interp_add_toplevel $slave [SurfIt_hyperpage $win]
    $slave alias master[SurfIt_hyperpage $win] interp_tk_method $slave master[SurfIt_hyperpage $win]

    # Attach the applet
    if {[info exists altvar(form)]} {
	# Attach at the form level, and record which form
	set forms($slave) $altvar(form)
	set formmap($altvar(form)) $slave	;# Map from form ID to slave
    } else {
	# Attach at the hyperpage level
	set page($slave) {}
    }

    # Initialise the global array for this Applet's state
    upvar #0 AppState$slave appstate
    set appstate(loadID) 0
    if {[info exists altvar(index)]} {
	set appstate(embedindex) $altvar(index)
    } else {
	set appstate(embedindex) [$win index {end - 1 char}]
    }
    set appstate(url) $var(url)

    # Callback to the application so that it can set up
    # controls for this applet

    Applet_created $win $slave

    # Send the applet code to the interpreter for evaluation

    $slave eval $app($var(id))
    unset app($var(id))

    upvar #0 $forms($slave) formstate
    # If this applet is attached to a form and form items have already
    # been processed then invoke HMapplet_item for each item
    if {[info exists formstate(Applet_formitems)]} {
	foreach item $formstate(Applet_formitems) {
	    eval HMapplet_item $win $forms($slave) $item
	}
    }
    # If this applet is attached to a form and the </FORM> has already
    # been seen then invoke HMend_form
    if {[info exists formstate(Applet_formready)]} {
	HMapplet_/form $win $forms($slave)
    }
    unset altvar	;# No longer needed
}

# Application should override this

proc Applet_created {win slave} {
    puts "==> Applet \"$slave\" created for window \"$win\""
}

# Register the handler for application/x-tcl content-type

catch {
    PRregister_type application/x-tcl .tcl Applet_handler {
	{PRdata $data}
	{PRfile {[$read_handler $read_handler_state]}}
	{PRfd {[$read_handler $read_handler_state]}}
	{PRimage {[error "incompatible data source"]}}
    } {} Applet_evaluate
}

# Destroy all applets associated with a window

proc Applet_destroy_window {win} {
    upvar #0 AppWin$win app

    Applet_destroy_page $win	;# Also destroy hyperpage & form applets

    foreach a [array names app] {
	if {[interp exists $a]} {
	    catch {$a eval terminate}
	    interp_delete $a
	}
    }
    catch {unset app}
}

# Destroy all applets associated with a hyperpage

proc Applet_destroy_page {win} {
    upvar #0 AppPage$win app
    upvar #0 AppForms$win forms
    upvar #0 AppFormsMap$win formmap

    foreach a [array names app] {
	if {[interp exists $a]} {
	    catch {$a eval terminate}
	    interp_delete $a
	}
    }
    catch {unset app}

    foreach a [array names forms] {
	if {[interp exists $a]} {
	    catch {$a eval terminate}
	    interp_delete $a
	}
    }
    catch {unset forms}
    catch {unset formmap}
}

# Destroy an applet attached to a form

proc Applet_destroy_form {win form_id} {
    upvar #0 AppForms$win forms
    upvar #0 AppFormsMap$win formmap

    if {[info exists formmap($form_id)]} {
	catch {$formmap($form_id) eval terminate}
	interp_delete $formmap($form_id)
	unset form($formmap($form_id))
	unset formmap($form_id)
    }
}

### Callins

# Notify all applets on a hyperpage that an anchor has been activated.
# Should applets attached to forms be notified? Yes, but probably only
# of anchors activated within their form (not implemented)

proc Applet_anchor_activation {win url} {
    upvar #0 AppPage$win page
    upvar #0 AppWin$win app

    foreach a [array names app] {
	if {[interp exists $a]} {
	    catch {$a eval anchor_activation $url}
	}
    }
    foreach a [array names page] {
	if {[interp exists $a]} {
	    catch {$a eval anchor_activation $url}
	}
    }
}

# Notify all applets when a new page has been loaded
# NB. Only hyperwindow level applets need be informed, since
# lower level applets attached to the previous page have 
# been terminated

proc Applet_pageloaded {win url} {
    upvar #0 AppWin$win app

    foreach a [array names app] {
	if {[interp exists $a]} {
	    puts "sending pageloaded to applet $a"
	    $a eval pageloaded $url
	}
    }
}

### Callbacks/Callins for form management

# When the form is finished, let the applet know the index
# of the end of the form.  NB race condition exists - applet
# may not have been created when this procedure is called

proc HMapplet_/form {win form_id} {
    upvar #0 AppFormsMap$win formmap

    if {[catch {set formmap($form_id)} slave]} {
	upvar #0 $form_id form
	# HMend_form will be invoked later
	set form(Applet_formready) 1
    }

    upvar #0 AppState$slave appstate
    set appstate(formendindex) [$win index {end - 1 char}]

    # Call into slave so that it knows that the form is ready
    catch {$slave eval HMend_form}
}

# HMapplet_item is called for each input element in the form.

proc HMapplet_item {win form_id type name value item} {
    upvar #0 AppFormsMap$win formmap
    upvar #0 $form_id form

    if {[catch {set formmap($form_id)} slave]} {
	# HMapplet_item will be invoked later
	lappend form(Applet_formitems) [list $type $name $value $item]
    }

    # Define a target alias in the slave for this window and all children
    Applet_win_install $slave $item

    upvar #0 AppState$slave appstate
    lappend appstate(formitems) [list $type $name $value $item]
    catch {$slave eval HMapplet_item \{$type\} \{$name\} \{$value\} \{[interp_unfix_all_pathnames $slave $item]\}}
}

### Override HTML renderer window installation procedure so that
### all applets attached to this hyperpage can manipulate the window
### and its children

proc HMwin_install {win item} {
    surfit_win_install $win $item

    # If an applet is attached to the hyperpage define a target alias 
    # in the slave for this window
    upvar #0 AppPage$win app

    foreach a [array names app] {
	if {[interp exists $a]} {
	    Applet_win_install $a $item
	}
    }
}

proc Applet_win_install {slave win} {
    set winlist $win
    while {$winlist != {}} {
	# Remove the head of the list, and then expand the list with the
	# children of the (just removed) head
	set path [lindex $winlist 0]
	set winlist [lrange $winlist 1 end]
	append winlist [winfo children $path]

	set path [interp_unfix_all_pathnames $slave $path]
	$slave alias $path interp_tk_method $slave $path
    }
}

###
### Slave alias target procedures
###

#
# The "applet" command allows the applet to interact 
# with the master interpreter
#

proc Applet_command {slave win method args} {
    upvar #0 AppState$slave appstate

    set hyperpage [SurfIt_hyperpage $win]
    switch $method {
	browserversion {
	    if {$args != {}} {error "too many arguments"}

	    global surfit
	    return $surfit(version)
	}
	embedindex {
	    if {$args != {}} {error "too many arguments"}

	    # The index at which the applet was embedded
	    return $appstate(embedindex)
	}
	embedwindow {
	    if {$args != {}} {error "too many arguments"}

	    # The Text widget in which the applet is embedded
	    return master$win
	}
	flush {
	    if {[llength $args] != 1} {error "wrong # arguments"}

	    Cache_expire_document $args
	    return {}
	}
	formendindex {
	    if {$args != {}} {error "too many arguments"}
	    upvar #0 AppForms$win forms
	    if {![info exists forms($slave)]} {error "slave is not attached to a form"}
	    if {[info exists appstate(formendindex)]} {
		return $appstate(formendindex)
	    } else {
		# Hasn't been set yet
		return {}
	    }
	}
	formitems {
	    if {$args != {}} {error "too many arguments'}
	    upvar #0 AppForms$win forms
	    if {![info exists forms($slave)]} {error "slave is not attached to a form"}
	    if {[info exists appstate(formitems)]} {
		# Map master widget pathnames to slave pathnames
		return [interp_unfix_all_pathnames $slave $appstate(formitems)]
	    } else {
		# Hasn't been set yet
		return {}
	    }
	}
	level {
	    if {[llength $args] > 1} {error "too many arguments"}

	    # Report or change at which level the applet is attached

	    upvar #0 AppForms$win forms
	    upvar #0 AppPage$hyperpage page
	    upvar #0 AppWin$hyperpage appwin
	    global AppBrowser

	    # Find where the applet is currently attached
	    if {[info exists forms($slave)]} {set former form} \
	    elseif {[info exists page($slave)]} {set former hyperpage} \
	    elseif {[info exists appwin($slave)]} {set former hyperwindow} \
	    elseif {[info exists AppBrowser($slave)]} {set former browser} \
	    else {error "*** internal error: cannot find slave interpreter"}

	    if {$args == {}} {return $former}

	    ### Change applet level

	    if {$former == "form"} {error "cannot reattach from form level"}
	    switch $args {
		form {error "cannot demote applet"}
		hyperpage {error "cannot demote applet"}
		hyperwindow {if {$former == "browser"} {error "cannot demote applet"}}
		browser {}
		default {error "unknown level \"$args\""}
	    }
	    if {$former == $args} return	;# No change

	    # Remove applet from previous level
	    if {$former == "hyperpage"} {unset page($slave)} \
	    else {unset appwin($slave)}

	    # Reattach at the new level
	    if {$args == "hyperwindow"} {set appwin($slave) {}} \
	    else {set AppBrowser($slave) {}}

	    # Callback to application to signal that things have changed
	    Applet_levelChange $slave $former $args
	}
	loaddata {
	    if {[llength $args] != 3} {error "wrong # arguments"}
	    Applet_loadData $win $slave [lindex $args 0] [lindex $args 1] [lindex $args 2]
	}
	loadurl {
	    # Args should be url [type]
	    if {[llength $args] == 1} {set type {}} \
	    elseif {[llength $args] != 2} {error "wrong # arguments"} \
	    else {set type [lindex $args 1]}

	    SurfIt_loadURL [lindex $args 0] $win $type
	}
	newpage {
	    if {$args != {}} {error "too many arguments"}

	    # Callback to application to clear the hyperpage
	    Applet_new_page $appstate(url) $win
	}
	parsehtml {
	    HMparse_html [lindex $args 0] "HMrender $win"
	}
	default {error "unknown method \"$method\""}
    }
}

# Application should override this
proc Applet_levelChange {slave old new} {
    puts "==> applet changed from $old level to $new level"
}

#
# Self destruct
#

proc Applet_destroy {slave args} {

    # This is wrong, since the Applet is still on the 
    # execution stack.
    #interp delete $slave

    # Instead, arrange for the slave interpeter to be removed later
    after idle interp_delete $slave
}

# Interim data load handler: We want to allow nonblocking I/O, but
# we'll batch up the document until it has been downloaded and then
# deliver it to the applet.  These procedures currently only support text data.
# 'callback' is the procedure to invoke when the data has been downloaded.
# 'data' is the (global) variable which is to receive the data.

proc Applet_loadData {win slave url data callback} {
    global feedback
    # Allocate global variable for this download
    upvar #0 AppState$slave appstate
    set state PR[incr appstate(loadID)]
    upvar #0 $state var

    if {[set result [loadData $url $state data_ret]] == {}} {return 0};

    # Provide visual cues for the user
    set hyperpage [SurfIt_hyperpage $win]
    set feedback($hyperpage) "Applet $slave is downloading \"$url\""
    update

    set var(handlerDelayed) 0
    if {$result == "PRfd"} {
	set appstate(data) {}
	fileevent $var(fd) readable "Applet_data_handler $win $slave $data \{$callback\} $var(read_handler) $state"
	return 0
    }
    # Data is immediately available
    if {$result == "PRfile"} {
	# Read the data into memory
	set data_ret [$var(read_handler) $state]
    }
    $slave eval global $data \; set $data \{$data_ret\}
    $slave eval $callback
}

proc Applet_data_handler {win slave data callback read_handler dataState} {
    global feedback
    upvar #0 AppState$slave appState
    upvar #0 $dataState var

    append appState(data) [$read_handler $dataState]

    if {$var(eof)} {
	# Update user feedback
	set hyperpage [SurfIt_hyperpage $win]
	set feedback($hyperpage) "Applet $slave download finished"
	update

	$slave eval global $data \; set $data \{$appState(data)\}
	$slave eval $callback
    }
}

proc Applet_blt_table {slave args} {
    interp_evaluate_list blt_table [interp_fix_all_pathnames $slave $args]
}

# Normally the application would override this procedure

proc Applet_new_page {base win} {
    $win delete 0.0 end
}

proc Applet_puts {slave args} {
    if {[llength $args] > 1} {
	set filed [lindex $args 0]
	set text [lindex $args 1]
	if {$filed == "stdout" || $filed == "stderr"} {
	    puts $filed "Applet $slave: $text"
	} else {error "applet not permitted to use file descriptor \"$filed\""}
    } else {puts "Applet $slave: [lindex $args 0]"}
}
