
#include "funcs.h"

/*
 * RXtResources.c --
 *	This module contains the functions to implement resource
 *	handling such as getValues, setValues, and resources
 *
 * Copyright 1993 Jan Newmarch, University of Canberra.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The author
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdio.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#ifdef MOTIF
#include <Xm/Xm.h>
#include <Xm/List.h>
#endif /* MOTIF */

#include <tcl.h>

extern Boolean RXt_verbose;
extern Widget RXt_GetTopShell();
/*
 *--------------------------------------------------------------
 *
 * RXt_ConvertValue --
 *
 *	convert a value from one format to another
 *
 * Results:
 *
 *	new value (of whatever type) stored in "new_value"
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Boolean
RXt_ConvertValue(w, fromType, fromValue, fromSize,
		   toType, toValue, toSize)
    Widget w;
    char *fromType;
    char *fromValue;
    unsigned int fromSize;
    char *toType;
    XtArgVal *toValue;
    unsigned int toSize;
{
    XrmValue from, to;

    if (strcmp(fromType, toType) == 0) {
	*toValue = (XtArgVal) fromValue;
	return TRUE;
    }

    from.size = fromSize;
    from.addr = fromValue;
    to.size = toSize;
    to.addr = XtMalloc(toSize);

    if (!XtConvertAndStore(w, fromType, &from, toType, &to))  {
	fprintf(stderr, "failed conversion %s to %s\n",
				fromType, toType);
	return FALSE;
    }
    *toValue = 0;
    _XtCopyToArg(to.addr, toValue, to.size);
/*
    XtFree(to.addr);
*/

   return TRUE;
}


/*
 *--------------------------------------------------------------
 *
 * RXt_MaybeSetStringTableSize --
 *
 *	Hack for Motif XmStringTables.
 *	Motif *sometimes* NULL terminates its XmStringTables
 *	most times doesnt.
 *	XmStringGet...() sometimes returns False on running off
 *	the end of a table, sometimes crashes
 *	_XmStringIsXmString() sometimes returns False on running off
 *	the end of a table, sometimes crashes
 *	i.e. there is no internal way of finding the size of one
 *	of these things. So we have to check if we have one and
 *	then look up the resource that tells us the size :-(
 *
 * Results:
 *
 * Side effects:
 *
 *	Sets global vbl RXt_XmStringTableSize
 *--------------------------------------------------------------
 */

static void
RXt_MaybeSetStringTableSize(w, resource)
    Widget w;
    String resource;
{
    extern int RXt_XmStringTableSize;
    WidgetClass class;
    Arg arg;
    char *size_resource;

#ifdef MOTIF
    class = XtClass(w);
    if (class == xmListWidgetClass) {
	if (strcmp (resource, XmNitems) == 0) {
	    size_resource = XmNitemCount;
	} else
	if (strcmp (resource, XmNselectedItems) == 0) {
	    size_resource = XmNselectedItemCount;
	} else
	return;
    } else
    return;

    /* Sigh, we have a StringTable, find its size */
    XtSetArg(arg, size_resource, &RXt_XmStringTableSize);
    XtGetValues(w, &arg, 1);
#endif /* MOTIF */
}

/*
 *--------------------------------------------------------------
 *
 * RXt_CovertResourceToString --
 *	convert a single resource to a string
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Boolean
RXt_ConvertResourceToString(w, resource, resource_size, resource_type, value)
    Widget w;
    char *resource;
    Cardinal resource_size;
    String resource_type;
    char **value;
{
    XrmValue from, converted;
    Arg args[1];

    RXt_MaybeSetStringTableSize(w, resource);

    from.size = resource_size;
    from.addr = (XtPointer) XtMalloc(from.size);
    XtSetArg(args[0], resource, from.addr);
    XtGetValues(w, args, 1);
    
    /* special case - same type */
    if (strcmp(resource_type, XtRString) == 0) {
	*value =  *(char **) from.addr;
	return TRUE;
    }
    converted.addr = NULL;
    if (!XtConvertAndStore(w, resource_type, &from,
			XtRString, &converted)) {
	return FALSE;
    }
    bzero(value, sizeof(XtArgVal));
    bcopy(converted.addr, value, converted.size);
    
    XtFree(from.addr);
    return TRUE;
}

/*
 *--------------------------------------------------------------
 *
 * RXt_ConvertValueToString --
 *
 *	converts from the internal widget form to a string for Tcl
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Boolean
RXt_ConvertValueToString(w, resources, num_resources,
		resource, new_value)
    Widget w;
    XtResourceList resources;
    int num_resources;
    char *resource;
    char **new_value;
{
    int n;

    for (n = 0; n < num_resources; n++) {
	if (strcmp(resource, resources->resource_name) == 0) {
	    break;
	}
	resources++;
    }
    if (n == num_resources) {
	return FALSE;
    }

    return RXt_ConvertResourceToString(w, resource,
		resources->resource_size, resources->resource_type, new_value);

}
		
		
#ifdef MOTIF
/*
 *--------------------------------------------------------------
 *
 * RXt_ConvertSubValueToString --
 *
 *	converts from the internal widget form to a string for Tcl
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static Boolean
RXt_ConvertSubValueToString(w, sec_resources, num_sec_resources,
		resource, new_value)
    Widget w;
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
    char *resource;
    char **new_value;
{
    int n, length;

    for (n = 0; n < num_sec_resources; n++) {
       	if (RXt_ConvertValueToString(w, sec_resources[n]->resources, 
            sec_resources[n]->num_resources, resource,
       	    new_value)) {
	    return True;
	}
    }
    return False;
}

/*
 *--------------------------------------------------------------
 *
 * RXt_FreeSecondaryResources --
 *
 *	Motif gives us a copy of these, so we have to tidy up
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
RXt_FreeSecondaryResources(sec_resources, num_sec_resources)
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
{
    int n;

    for (n = 0; n < num_sec_resources; n++) {
	XtFree((char *) sec_resources[n]->resources);
	XtFree((char *) sec_resources[n]);
    }

    if (num_sec_resources > 0) {
        XtFree((char *) sec_resources);
    }
}
#endif /* MOTIF */

/*
 *--------------------------------------------------------------
 *
 *--------------------------------------------------------------
 */
void RXt_PrintFullName(w)
    Widget w;
{
    if (XtParent(w) == NULL) {
	printf("%s", XtName(w));
	return;
    }
    RXt_PrintFullName(XtParent(w));
    printf(".%s", XtName(w));
}

/*
 *--------------------------------------------------------------
 *
 * RXt_PrintAlmostFullName
 *
 * Side effects:
 *	prints to stdout the path less the toplevel widget
 *
 *--------------------------------------------------------------
 */
void RXt_PrintAlmostFullName(w)
    Widget w;
{
    Widget parent;

    parent = XtParent(w);
    if (parent == NULL) {
	return;
    }
    if (XtParent(parent) == NULL) {
	printf("%s", XtName(w));
	return;
    }
    RXt_PrintAlmostFullName(parent);
    printf(".%s", XtName(w));
}

/*
 *--------------------------------------------------------------
 *
 * RXt_GetValues --
 *
 *	get the resource values out of a widget
 *
 * Results:
 *	a tcl result
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

int
RXt_GetValues(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Widget toplevel = (Widget) clientData;
    Widget w;
    WidgetClass class;
    XtResourceList resources, constraint_resources;
#ifdef MOTIF
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
#endif
    Cardinal num_resources;
    Cardinal num_constraint_resources;
    int num_values = 0;
    XrmValue from, converted;
    char *new_value;
#   define MAXARGS 100
    char *resource;
    Widget parent;

    if (argc > MAXARGS*2) {
	return TCL_ERROR;
    }

    w = XtNameToWidget(RXt_GetTopShell(), argv[1]);
    if(w == NULL) {
	fprintf(stderr,"no such widget: %s\n",argv[1]);
	sprintf(interp->result, "no such widget: %s\n", argv[1]);
	return TCL_ERROR;
    }

    if (RXt_verbose) {
        printf("getting values on ");
        RXt_PrintFullName(w);
        printf("\n");
    }

    parent = XtParent(w);
    class = XtClass(w);

    XtInitializeWidgetClass(class); /* make sure w gets all resources */
    XtGetResourceList(class, &resources, &num_resources);

#ifdef MOTIF
    num_sec_resources = XmGetSecondaryResourceData(class, &sec_resources);
#endif
  
    if (parent != NULL && XtIsConstraint(parent)) {
	XtGetConstraintResourceList(XtClass(parent), 
	    &constraint_resources, &num_constraint_resources);
    } else {
	num_constraint_resources = 0;
    }

    argv += 2;
    while (argc > 2) {
	if (argv[0][0] != '-') {
	    sprintf(interp->result, "illegal getValues option \"%50s\"", argv[0]);
	    return TCL_ERROR;
	}
	resource = argv[0]+1;

        if (RXt_ConvertValueToString(w, resources, num_resources,
		resource, &new_value)) {
	    /* store new_value in variable in argv[1] now */
#	    ifdef DEBUG
	    fprintf(stderr, "Got value: %s\n", (char *) new_value);
#	    endif
	    if (Tcl_SetVar(interp, argv[1], (char *) new_value,
			TCL_LEAVE_ERR_MSG) == NULL)
		fprintf(stderr, "%s\n", interp->result);
	} else
        if (num_constraint_resources > 0 &&
	    RXt_ConvertValueToString(w, constraint_resources, 
	            num_constraint_resources, resource,
        	    &new_value)) {
	    /* store in a variable now */
	    if (Tcl_SetVar(interp, argv[1], (char *) new_value,
			TCL_LEAVE_ERR_MSG) == NULL)
		fprintf(stderr, "%s\n", interp->result);
	} else 
#ifdef MOTIF
        if (num_sec_resources > 0 &&
	    RXt_ConvertSubValueToString(w, sec_resources, 
	            num_sec_resources, resource,
        	    &new_value)) {
	    /* store in a variable now */
	    if (Tcl_SetVar(interp, argv[1], (char *) new_value,
			TCL_LEAVE_ERR_MSG) == NULL)
		fprintf(stderr, "%s\n", interp->result);
	} else
#endif
	{
	    sprintf(interp->result, "Conversion from %s to String failed\n",
					resource);
	    return TCL_ERROR;
	}
	argc -= 2;
	argv += 2;
    }
#ifdef MOTIF
    if (num_sec_resources > 0)
	RXt_FreeSecondaryResources(sec_resources, num_sec_resources);
#endif
    return TCL_OK;
}

