#
# Tcl Library for OA.
#
 
# 
# infotree.tcl,v 1.5 1995/02/05 09:21:20 del Exp
#
# Set up a Information Tree dialog.
#
# Global Variables:
#
#   global InfoTree(fname)			Currently selected file name.
#
#   global InfoTree(displayed)			List of currently expanded dirs
#						and subdirs.
#
#   global InfoTree(ypos)			The y position of the selected
#						item in the tree list.
#
#   global InfoTree(workdir)			The top level working directory.
#
#   global InfoTree(modfile)			The module (file names) file.
#
#   global InfoTree(global_bitmapdir)		Usually
# 						/usr/include/X11/bitmaps.
#
#   global InfoTree(local_bitmapdir)		Where the bitmaps specific to
#						the OA package are installed.
#
#   global InfoTree(rootname)			The root name of the toplevel
#						window.  Can be null to make
#						InfoTree have its own top level
#						"." window.
#
#   global InfoTree(wmname)			The name by which the window
#						manager sees us (wm).  Can be
#						a toplevel or ".".
#
#   global InfoTree_mtitle    			List of names of modules
#						(files).
#
#   global InfoTree_dtitle    			List of names of directories.
#
#   global InfoTree_dcontents 			List of files in a directory.
#
#   global InfoTree_dsubmenus 			List of submenus
#						(subdirectories) in a directory.
#
#

#
# Constants for user configuration
#
set InfoTree(workdir) "$env(HOME)/.infotree"
set InfoTree(modfile) "$env(HOME)/.infotree/.modules"

proc infotree_setup {{rootname ""}} {
#
# This creates the InfoTree toplevel window and all of the child
# windows.
#
  global InfoTree

  set InfoTree(rootname) $rootname
  # Set the name by which the window manager sees us.
  if {$InfoTree(rootname) != ""} {
    set InfoTree(wmname) $InfoTree(rootname)
  } else {
    set InfoTree(wmname) "."
  }
  #
  # Set up InfoTree directory
  #
  if {! [file isdirectory $InfoTree(workdir)]} {
    exec mkdir -p $InfoTree(workdir)
    exec mkdir -p $InfoTree(workdir)/general
    set modules [open $InfoTree(modfile) w+] 
    puts $modules "#D\tgeneral\tGeneral Information"
    close $modules
  }

  if {$InfoTree(rootname) != ""} {
    toplevel .infotree
  }
  frame $InfoTree(rootname).down
  frame $InfoTree(rootname).up
  frame $InfoTree(rootname).up.left -relief raised
  frame $InfoTree(rootname).up.right -relief raised

  pack $InfoTree(rootname).up -side top -fill x
  pack $InfoTree(rootname).down -side bottom -fill x
  pack $InfoTree(rootname).up.left -side left -fill both
  pack $InfoTree(rootname).up.right -side left -fill both -expand 1

  label $InfoTree(rootname).lfname -text "File Name" -anchor w
  entry $InfoTree(rootname).tfname -relief sunken -textvariable InfoTree(fname)

  bind_motifentry $InfoTree(rootname).tfname

  pack $InfoTree(rootname).lfname -in $InfoTree(rootname).up.left \
    -side top -fill x -pady 3
  pack $InfoTree(rootname).tfname -in $InfoTree(rootname).up.right \
    -side top -fill both -expand 1 -pady 3

  button $InfoTree(rootname).ok -text "OK" \
    -command {infotree_ok}
  button $InfoTree(rootname).newFile -text "New File" \
    -command {infotree_newfile}
  button $InfoTree(rootname).newDir -text "New Directory" \
    -command {infotree_newdir}
  button $InfoTree(rootname).delete -text "Delete" \
    -command {infotree_delete}

  # We want the Quit button to do different things depending on whether
  # we have a parent or not (this app can run contained or standalone).
  if {$InfoTree(rootname) != ""} {
    button $InfoTree(rootname).quit -text "Quit" \
      -command {wm withdraw $InfoTree(rootname)}
  } else {
    button $InfoTree(rootname).quit -text "Quit" \
      -command {exit}
  }
 
  pack $InfoTree(rootname).ok $InfoTree(rootname).newFile \
    $InfoTree(rootname).newDir $InfoTree(rootname).delete \
    $InfoTree(rootname).quit \
    -in $InfoTree(rootname).down -side left \
    -ipadx 2 -ipady 2 -padx 4 -pady 4 -fill both -expand 1
 
  # Create a scroll bar and three list boxes.
  scrollbar $InfoTree(rootname).scroll -relief sunken \
    -command {infotree_scroll}
  listbox $InfoTree(rootname).namelist \
    -yscroll {$InfoTree(rootname).scroll set} -relief sunken \
    -geometry 40x20 -setgrid yes
  listbox $InfoTree(rootname).infolist \
    -yscroll {$InfoTree(rootname).scroll set} -relief sunken \
    -geometry 10x20 -setgrid yes
  listbox $InfoTree(rootname).filelist \
    -yscroll {$InfoTree(rootname).scroll set} -relief sunken \
    -geometry 20x20 -setgrid yes

  # Set up a selection rule so that only one item can be selected
  # at a time.
  tk_listboxSingleSelect $InfoTree(rootname).namelist
  tk_listboxSingleSelect $InfoTree(rootname).infolist
  tk_listboxSingleSelect $InfoTree(rootname).filelist

  # Set up key bindings for the lists so that they all select as
  # if the code list was pointed to.
  bind $InfoTree(rootname).filelist <Double-Button-1> \
    {infotree_act_on_code %y}
  bind $InfoTree(rootname).namelist <Double-Button-1> \
    {infotree_act_on_code %y}
  bind $InfoTree(rootname).infolist <Double-Button-1> \
    {infotree_act_on_code %y}
  bind $InfoTree(rootname).filelist <1> {infotree_select_code %y}
  bind $InfoTree(rootname).namelist <1> {infotree_select_code %y}
  bind $InfoTree(rootname).infolist <1> {infotree_select_code %y}

  # Pack the scroll bar and the 3 lists side by side.
  pack $InfoTree(rootname).scroll -side right -fill y -pady 2 -padx 2
  pack $InfoTree(rootname).namelist $InfoTree(rootname).infolist \
    $InfoTree(rootname).filelist \
    -side left -fill both -expand 1

  # Window manager stuff.
  wm title $InfoTree(wmname) "Information Tree"
  wm iconname $InfoTree(wmname) "Info"
  wm iconbitmap $InfoTree(wmname) @$InfoTree(local_bitmapdir)/tree32.xbm
  wm iconmask $InfoTree(wmname) @$InfoTree(local_bitmapdir)/tree32_mask.xbm
  wm minsize $InfoTree(wmname) 1 1

  # This window gets withdrawn by default.  If you want to run it stand
  # alone, you must do "infotree_setup; infotree_run" to de-iconify the
  # window.
  wm withdraw $InfoTree(wmname)

  set InfoTree(displayed) {}
  infotree_search_tree
  infotree_fill_listbox

  #
  # A new toplevel for creating new files.
  #
  toplevel .newfile
  frame .newfile.left
  frame .newfile.right
  frame .newfile.down -relief groove -border 2

  pack .newfile.down -side bottom -fill x -expand 1
  pack .newfile.left -side left -fill y
  pack .newfile.right -side left -fill both -expand 1
 
  label .newfile.lnewname -text "File Name" -anchor w
  label .newfile.lnewdesc -text "File Description" -anchor w
 
  entry .newfile.tnewname -relief sunken -textvariable InfoTree(newname)
  entry .newfile.tnewdesc -relief sunken -textvariable InfoTree(newdesc) \
    -width 40
 
  bind_motifentry .newfile.tnewname
  bind_motifentry .newfile.tnewdesc

  pack .newfile.lnewname .newfile.lnewdesc -in .newfile.left \
    -side top -fill x -pady 3

  pack .newfile.tnewname .newfile.tnewdesc -in .newfile.right \
    -side top -fill both -expand 1 -pady 3
 
  button .newfile.ok -text "OK" \
    -command infotree_do_newfile
  button .newfile.quit -text "Quit" \
    -command { wm withdraw .newfile }
 
  pack .newfile.ok .newfile.quit -in .newfile.down -side left \
    -ipadx 2 -ipady 2 -padx 4 -pady 4 -fill both -expand 1
 
  wm withdraw .newfile
  wm title .newfile "Create a New File"
  wm group .newfile $InfoTree(wmname)
  # Doesn't do decorations in olwm so leave this line out.
  # wm transient .newfile $InfoTree(wmname)
  wm iconname .newfile "New File"
  wm iconbitmap .newfile @$InfoTree(local_bitmapdir)/tree32.xbm
  wm iconmask .newfile @$InfoTree(local_bitmapdir)/tree32_mask.xbm
  wm minsize .newfile 1 1

  #
  # A new toplevel for creating new directories.
  #
  toplevel .newdir
  frame .newdir.left
  frame .newdir.right
  frame .newdir.down -relief groove -border 2

  pack .newdir.down -side bottom -fill x -expand 1
  pack .newdir.left -side left -fill y
  pack .newdir.right -side left -fill both -expand 1

  label .newdir.lnewname -text "Directory Name" -anchor w
  label .newdir.lnewdesc -text "Directory Description" -anchor w

  entry .newdir.tnewname -relief sunken -textvariable InfoTree(newdirname)
  entry .newdir.tnewdesc -relief sunken -textvariable InfoTree(newdirdesc) \
    -width 40

  bind_motifentry .newdir.tnewname
  bind_motifentry .newdir.tnewdesc

  pack .newdir.lnewname .newdir.lnewdesc -in .newdir.left \
    -side top -fill x -pady 3

  pack .newdir.tnewname .newdir.tnewdesc -in .newdir.right \
    -side top -fill both -expand 1 -pady 3

  button .newdir.ok -text "OK" \
    -command infotree_do_newdir
  button .newdir.quit -text "Quit" \
    -command { wm withdraw .newdir }

  pack .newdir.ok .newdir.quit -in .newdir.down -side left \
    -ipadx 2 -ipady 2 -padx 4 -pady 4 -fill both -expand 1

  wm withdraw .newdir
  wm title .newdir "Create a New Directory"
  wm group .newdir $InfoTree(wmname)
  # Doesn't do decorations in olwm
  # wm transient .newdir $InfoTree(wmname)
  wm iconname .newdir "New Dir"
  wm iconbitmap .newdir @$InfoTree(local_bitmapdir)/tree32.xbm
  wm iconmask .newdir @$InfoTree(local_bitmapdir)/tree32_mask.xbm
  wm minsize .newdir 1 1
}

proc infotree_scroll index {
#
# To support scrolling 3 listboxes simultaneously
#
  $InfoTree(rootname).namelist yview $index
  $InfoTree(rootname).infolist yview $index
  $InfoTree(rootname).filelist yview $index
}

proc infotree_run {} {
#
# This procedure maps the InfoTree window onto the screen.
#
  global InfoTree

  wm deiconify $InfoTree(wmname)
}

proc infotree_search_tree {} {
#
# This searches the modules file and creates the tree structures needed
# to run the program.
#
  global InfoTree
  global InfoTree_mtitle
  global InfoTree_dtitle
  global InfoTree_dcontents
  global InfoTree_dsubmenus

  # Trash everything to start a clean slate.
  catch {unset InfoTree_mtitle}
  catch {unset InfoTree_dtitle}
  catch {unset InfoTree_dcontents}
  catch {unset InfoTree_dsubmenus}

  # Open the modules file and cycle through it looking for #Ds and #Ms
  set modules [open $InfoTree(modfile)]
  while {[gets $modules line] >= 0} {
    # Split and parse the line
    if {$line != {}} {
      set text [split $line "\t"]
 
      # #D describes a directory title.
 
      if {[lindex $text 0] == "#D"} {
        set dname [lindex $text 1]
        set InfoTree_dtitle($dname) [lindex $text 2]
        set layers [split $dname "/"]
        # puts "$dname is called $InfoTree_dtitle($dname)"
        if {[llength $layers] > 1} {
          set pname [file dirname $dname]
          if [info exists InfoTree_dsubmenus($pname)] {
            lappend InfoTree_dsubmenus($pname) $dname
          } else {
            set InfoTree_dsubmenus($pname) $dname
          }
        }
      }

      # #M means this is a module title
 
      if {[lindex $text 0] == "#M"} {
        set mname [lindex $text 1]
        set InfoTree_mtitle($mname) [lindex $text 2]
        set pname [file dirname $mname]
        if [info exists InfoTree_dcontents($pname)] {
          lappend InfoTree_dcontents($pname) $mname
        } else {
          set InfoTree_dcontents($pname) $mname
        }
      }
    }
  }
  # No more lines in the modules file
  close $modules
}
  
proc infotree_fill_listbox {} {
#
# Do this to update the display of the listbox.
#
  global InfoTree
  global InfoTree_dtitle

  $InfoTree(rootname).namelist delete 0 end
  $InfoTree(rootname).infolist delete 0 end
  $InfoTree(rootname).filelist delete 0 end

  foreach dname [array names InfoTree_dtitle] {
    if {[file dirname $dname] == "."} {
      $InfoTree(rootname).namelist insert end $InfoTree_dtitle($dname)
      $InfoTree(rootname).infolist insert end "Directory"
      $InfoTree(rootname).filelist insert end $dname
      if {[lsearch $InfoTree(displayed) $dname] != -1} {
        infotree_fill_dir $dname
      }
    }
  }
}

proc infotree_fill_dir {dname} {
#
# This is a recursive procedure.  It should only be called by itself or
# by infotree_fill_listbox.
#
  global InfoTree_mtitle
  global InfoTree_dtitle
  global InfoTree_dcontents
  global InfoTree_dsubmenus
  global InfoTree

  # Display all of the modules in this subdir.

  if [info exists InfoTree_dcontents($dname)] {
    foreach mname $InfoTree_dcontents($dname) {
      $InfoTree(rootname).namelist insert end $InfoTree_mtitle($mname)
      $InfoTree(rootname).infolist insert end "File"
      $InfoTree(rootname).filelist insert end $mname
    }  
  }  

  # Recurse through each subdir, only displaying sublevels of those
  # that are marked for display.

  if [info exists InfoTree_dsubmenus($dname)] {
    foreach subdir $InfoTree_dsubmenus($dname) {
      $InfoTree(rootname).namelist insert end $InfoTree_dtitle($subdir)
      $InfoTree(rootname).infolist insert end "Directory"
      $InfoTree(rootname).filelist insert end $subdir
      if {[lsearch $InfoTree(displayed) $subdir] != -1} {
        infotree_fill_dir $subdir
      }
    }
  }
}

proc infotree_select_code {yposition} {
#
# Do this when a code is clicked on.
#
  global InfoTree

  set InfoTree(ypos) $yposition

  # This does the actual selection
  $InfoTree(rootname).filelist select from \
    [$InfoTree(rootname).filelist nearest $yposition]
  set code [selection get]

  # This will update the "File Name" entry box.
  set InfoTree(fname) [lindex $code 0]

  return $code
}

proc infotree_act_on_code {yposition} {
#
# Do this when a code is double-clicked on.
#
  global InfoTree

  # Do the selection bit.
  set code [infotree_select_code $yposition]

  # Act on the item selected.
  infotree_ok

  # Do the selection bit again so that the item remains selected after the
  # redraw.
  set code [infotree_select_code $yposition]
}

proc infotree_ok {} {
#
# Do this when you click OK.
#
  global InfoTree

  if [file isdirectory $InfoTree(workdir)/$InfoTree(fname)] {
    # If the module is displayed, remove it from the display.  If it is
    # not displayed, add it to the display.
    set pos [lsearch $InfoTree(displayed) $InfoTree(fname)]
    if {$pos == -1} {
      lappend InfoTree(displayed) $InfoTree(fname)
    } else {
      set InfoTree(displayed) [lreplace $InfoTree(displayed) $pos $pos]
    }
    # Redraw the listbox.
    infotree_fill_listbox
  } else {
    exec notebook $InfoTree(workdir)/$InfoTree(fname) &
  }
}

proc infotree_newfile {} {
#
# Do this to add a new file.
#
  global InfoTree

  if [file isdirectory $InfoTree(workdir)/$InfoTree(fname)] {
    set InfoTree(newname) "$InfoTree(fname)/"
  } else {
    set InfoTree(newname) "$InfoTree(fname)"
  }
  wm deiconify .newfile
}

proc infotree_do_newfile {} {
#
# This gets called by the new file dialog when OK is pressed.
#
  global InfoTree

  set modules [open $InfoTree(modfile) a+] 
  set module_line [format "#M\t%s\t%s" $InfoTree(newname) $InfoTree(newdesc)]
  puts $modules $module_line
  close $modules

  wm withdraw .newfile
  exec notebook $InfoTree(workdir)/$InfoTree(newname) &
  infotree_search_tree
  infotree_fill_listbox
}

proc infotree_newdir {} {
#
# Do this to add a new directory.
#
  global InfoTree

  if [file isdirectory $InfoTree(workdir)/$InfoTree(fname)] {
    set InfoTree(newdirname) "$InfoTree(fname)/"
  } else {
    set InfoTree(newdirname) [format "%s/" [file dirname $InfoTree(fname)] ]
  }
  wm deiconify .newdir
}

proc infotree_do_newdir {} {
#
# This gets called by the new directory dialog when OK is pressed.
#
  global InfoTree

  set modules [open $InfoTree(modfile) a+] 
  set module_line \
    [format "#D\t%s\t%s" $InfoTree(newdirname) $InfoTree(newdirdesc)]
  puts $modules $module_line
  close $modules

  wm withdraw .newdir
  exec mkdir -p $InfoTree(workdir)/$InfoTree(newdirname)
  infotree_search_tree
  infotree_fill_listbox
}

proc infotree_delete {} {
#
# This gets called when you hit the delete button.
#
  global InfoTree
  set title "Confirm Delete"
  set mess "This will delete the directory\n\n$InfoTree(fname)\n
and all of its contents.\n\nAre you sure?"

  if [file isdirectory $InfoTree(workdir)/$InfoTree(fname)] {
    if {[tk_dialog .confirmdelete $title $mess question 0 Yes No] == 0} {
      infotree_delete_tree $InfoTree(fname)
      infotree_write_stuff
      infotree_search_tree
      infotree_fill_listbox
    }
  } else {
    infotree_delete_file $InfoTree(fname)
    infotree_write_stuff
    infotree_search_tree
    infotree_fill_listbox
  }
}

proc infotree_delete_file {mname} {
  global InfoTree
  global InfoTree_mtitle

  # puts stderr "deleting file $mname"

  # Remove the file from the structure.
  unset InfoTree_mtitle($mname)

  # .. and delete it.
  if [file exists $InfoTree(workdir)/$mname] {
    exec rm -f $InfoTree(workdir)/$mname
  }
}

proc infotree_delete_tree {dname} {
#
# This deletes a directory and all of its trees from the structure.
#
  global InfoTree
  global InfoTree_mtitle
  global InfoTree_dtitle
  global InfoTree_dcontents
  global InfoTree_dsubmenus

  # puts stderr "deleting directory $dname"

  # Trash the directory.
  unset InfoTree_dtitle($dname)

  # Trash all of its subdirectories
  if [info exists InfoTree_dsubmenus($dname)] {
    foreach subdir $InfoTree_dsubmenus($dname) {
      infotree_delete_tree $subdir
    }
    unset InfoTree_dsubmenus($dname)
  }

  # Trash all of its files
  if [info exists InfoTree_dcontents($dname)] {
    foreach mname $InfoTree_dcontents($dname) {
      infotree_delete_file $mname
    }
    unset InfoTree_dcontents($dname)
  }

  # Delete the directory and all of its contents.
  if [file isdirectory $InfoTree(workdir)/$dname] {
    exec rm -rf $InfoTree(workdir)/$dname
  }

}

proc infotree_write_stuff {} {
#
# For writing out the internal tree structures.
#
  global InfoTree
  global InfoTree_mtitle
  global InfoTree_dtitle

  set modules [open $InfoTree(modfile) "w"]

  foreach item [array names InfoTree_dtitle] {
    puts $modules [format "#D\t%s\t%s" $item $InfoTree_dtitle($item)]
  }
  foreach item [array names InfoTree_mtitle] {
    puts $modules [format "#M\t%s\t%s" $item $InfoTree_mtitle($item)]
  }

  close $modules
}
