/*
 *=============================================================================
 *                               tSippGeom.c
 *-----------------------------------------------------------------------------
 * Vector and matrix operations.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1993 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippGeom.c,v 4.0 1993/11/27 21:29:38 markd Rel $
 *============================================================================
 */

#include "tSippInt.h"
#include "patchlevel.h"

/*
 * Internal prototypes.
 */
typedef void (*rotateFunc_pt) _ANSI_ARGS_((Transf_mat *matrix,
                                           double      angle));

     

static bool
MatrixAxisRotate _ANSI_ARGS_((tSippGlob_pt   tSippGlobPtr,
                              int            argc,
                              char         **argv,
                              rotateFunc_pt  rotateFunc));

void
ReturnVector _ANSI_ARGS_((tSippGlob_pt   tSippGlobPtr,
                          Vector        *vectorPtr));

/*=============================================================================
 * ReturnVector --
 *   Return a vector or point into the interp->result with no lose of
 * precision.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o vectorPTr (I) - A pointer to the vector to format.
 *-----------------------------------------------------------------------------
 */
void
ReturnVector (tSippGlobPtr, vectorPtr)
    tSippGlob_pt   tSippGlobPtr;
    Vector        *vectorPtr;
{
    char  *bufPtr;

    /*
     * Allocate a dynamic buffer only if there is a chance of the result not
     * fitting into the standard Tcl result.
     */
    if (TCL_RESULT_SIZE < (TSIPP_DOUBLE_STR_SIZE * 3)) {
        Tcl_SetResult (tSippGlobPtr->interp,
                       smalloc (TSIPP_DOUBLE_STR_SIZE * 3),
                       TCL_DYNAMIC);
    }

    bufPtr = TSippFormatDouble (vectorPtr->x, tSippGlobPtr->interp->result);
    *bufPtr = ' ';
    bufPtr = TSippFormatDouble (vectorPtr->y, bufPtr + 1);
    *bufPtr = ' ';
    TSippFormatDouble (vectorPtr->z, bufPtr + 1);

}

/*=============================================================================
 * SippMkVector --
 *   Implements the command:
 *     SippMkVector point0 point1
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippMkVector (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Vector        point0, point1, vector;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " point0 point1", (char *) NULL);
        return TCL_ERROR;
    }                     
 
    if (!TSippConvertVertex (tSippGlobPtr, argv [1], &point0))
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [2], &point1))
        return TCL_ERROR;

    vector.x = point1.x - point0.x;
    vector.y = point1.y - point0.y;
    vector.z = point1.z - point0.z;

    ReturnVector (tSippGlobPtr, &vector);
    return TCL_OK;

}

/*=============================================================================
 * SippVecNegate --
 *   Implements the command:
 *     SippVecNegate vector
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippVecNegate (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Vector        vector;

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " vector", (char *) NULL);
        return TCL_ERROR;
    }                     
 
    if (!TSippConvertVertex (tSippGlobPtr, argv [1], &vector))
        return TCL_ERROR;

    VecNegate (vector);
    ReturnVector (tSippGlobPtr, &vector);
    return TCL_OK;

}

/*=============================================================================
 * SippVecDot --
 *   Implements the command:
 *     SippVecDot vector0 vector1
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippVecDot (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Vector        vector0, vector1;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " vector0 vector1", (char *) NULL);
        return TCL_ERROR;
    }                     
 
    if (!TSippConvertVertex (tSippGlobPtr, argv [1], &vector0))
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [2], &vector1))
        return TCL_ERROR;

    TSippFormatDouble (VecDot (vector0, vector1), interp->result);
    return TCL_OK;

}

/*=============================================================================
 * SippVecLen --
 *   Implements the command:
 *     SippVecLen vector
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippVecLen (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Vector        vector;

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " vector", (char *) NULL);
        return TCL_ERROR;
    }                     
 
    if (!TSippConvertVertex (tSippGlobPtr, argv [1], &vector))
        return TCL_ERROR;

    TSippFormatDouble (VecLen (vector), interp->result);
    return TCL_OK;

}

/*=============================================================================
 * SippVecAdd --
 *   Implements the command:
 *     SippVecAdd vector0 vector1
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippVecAdd (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Vector        vector0, vector1, result;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " vector0 vector1", (char *) NULL);
        return TCL_ERROR;
    }                     
 
    if (!TSippConvertVertex (tSippGlobPtr, argv [1], &vector0))
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [2], &vector1))
        return TCL_ERROR;

    VecAdd (result, vector0, vector1);
    ReturnVector (tSippGlobPtr, &result);
    return TCL_OK;

}

/*=============================================================================
 * SippVecSub --
 *   Implements the command:
 *     SippVecSub vector0 vector1
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippVecSub (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Vector        vector0, vector1, result;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " vector0 vector1", (char *) NULL);
        return TCL_ERROR;
    }                     
 
    if (!TSippConvertVertex (tSippGlobPtr, argv [1], &vector0))
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [2], &vector1))
        return TCL_ERROR;

    VecSub (result, vector0, vector1);
    ReturnVector (tSippGlobPtr, &result);
    return TCL_OK;

}

/*=============================================================================
 * SippVecMult --
 *   Implements the command:
 *     SippVecMult number vector
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippVecMult (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Vector        vector, result;
    double        number;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " number vector", (char *) NULL);
        return TCL_ERROR;
    }                     

    if (Tcl_GetDouble (interp, argv [1], &number) != TCL_OK)
        return TCL_ERROR;

    if (!TSippConvertVertex (tSippGlobPtr, argv [2], &vector))
        return TCL_ERROR;

    VecScalMul (result, number, vector);
    ReturnVector (tSippGlobPtr, &result);
    return TCL_OK;

}

/*=============================================================================
 * SippVecCross --
 *   Implements the command:
 *     SippVecCross vector0 vector1
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippVecCross (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Vector        vector0, vector1, result;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " vector0 vector1", (char *) NULL);
        return TCL_ERROR;
    }                     
 
    if (!TSippConvertVertex (tSippGlobPtr, argv [1], &vector0))
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [2], &vector1))
        return TCL_ERROR;

    VecCross (result, vector0, vector1);
    ReturnVector (tSippGlobPtr, &result);
    return TCL_OK;

}

/*=============================================================================
 * SippMatIdent --
 *   Implements the command:
 *     SippMatIdent
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippMatIdent (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;

    if (argc != 1) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          (char *) NULL);
        return TCL_ERROR;;
    }
    Tcl_SetResult (tSippGlobPtr->interp, TSippFormatMatrix (&ident_matrix),
                   TCL_DYNAMIC);
    return TCL_OK;
    
}

/*=============================================================================
 * SippMatTranslate --
 *   Implements the command:
 *     SippMatTranslate matrix vector
 * Notes:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippMatTranslate (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Transf_mat    matrix;
    Vector        vector;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " matrix vector", (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertMatrix (tSippGlobPtr, argv [1], &matrix))
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [2], &vector))
        return TCL_ERROR;

    mat_translate (&matrix, vector.x, vector.y, vector.z);

    Tcl_SetResult (tSippGlobPtr->interp, TSippFormatMatrix (&matrix),
                   TCL_DYNAMIC);
    return TCL_OK;

}

/*=============================================================================
 * MatrixAxisRotate --
 *   Process parameters for the commands to rotate an matrix around an axis
 * and call one of the standard functions on it.
 *
 * These commands have the arguments: matrix angle
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the Tcl SIPP globals.
 *   o argc, argv (I) - The arguments to the command.
 *   o rotateFunc (I) - The rotate function that is to be called.
 * Returns:
 *   TRUE if all is OK,  FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
static bool
MatrixAxisRotate (tSippGlobPtr, argc, argv, rotateFunc)
    tSippGlob_pt    tSippGlobPtr;
    int             argc;
    char          **argv;
    rotateFunc_pt   rotateFunc;
{
    Transf_mat  matrix;
    double      angle;

    if (argc != 3) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " matrix angle", (char *) NULL);
        return FALSE;
    }
    if (!TSippConvertMatrix (tSippGlobPtr, argv [1], &matrix))
        return FALSE;
    if (!TSippConvertAngleRad (tSippGlobPtr, argv [2], &angle))
        return FALSE;
 
    (*rotateFunc) (&matrix, angle);
    Tcl_SetResult (tSippGlobPtr->interp, TSippFormatMatrix (&matrix),
                   TCL_DYNAMIC);
    return TRUE;

}

/*=============================================================================
 * SippMatRotateX --
 *   Implements the command:
 *     SippMatRotateX matrix angle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippMatRotateX (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (MatrixAxisRotate ((tSippGlob_pt) clientData, argc, argv, mat_rotate_x))
        return TCL_OK;
    else
        return TCL_ERROR;;
    
}

/*=============================================================================
 * SippMatRotateY --
 *   Implements the command:
 *     SippMatRotateY matrix angle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippMatRotateY (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (MatrixAxisRotate ((tSippGlob_pt) clientData, argc, argv, mat_rotate_y))
        return TCL_OK;
    else
        return TCL_ERROR;;
    
}

/*=============================================================================
 * SippMatRotateZ --
 *   Implements the command:
 *     SippMatRotateZ matrix angle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippMatRotateZ (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (MatrixAxisRotate ((tSippGlob_pt) clientData, argc, argv, mat_rotate_z))
        return TCL_OK;
    else
        return TCL_ERROR;;
    
}

/*=============================================================================
 * SippMatRotate --
 *   Implements the command:
 *     SippMatRotate matrix point vector angle
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippMatRotate (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt   tSippGlobPtr = (tSippGlob_pt) clientData;
    Transf_mat     matrix;
    Vector         point, vector;
    double         angle;

    if (argc != 5) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " matrix point vector angle", (char *) NULL);
        return TCL_ERROR;
    }
    
    if (!TSippConvertMatrix (tSippGlobPtr, argv [1], &matrix))
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [2], &point))
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [3], &vector))
        return TCL_ERROR;
    if (!TSippConvertAngleRad (tSippGlobPtr, argv [4], &angle))
        return TCL_ERROR;

    mat_rotate (&matrix, &point, &vector, angle);
    Tcl_SetResult (tSippGlobPtr->interp, TSippFormatMatrix (&matrix),
                   TCL_DYNAMIC);
    return TCL_OK;

}

/*=============================================================================
 * SippMatScale --
 *   Implements the command:
 *     SippMatScale matrix factor|{xfactor yfactor zfactor}
 * Notes:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippMatScale (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Transf_mat    matrix;
    Vector        scale;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " matrix factor|{xfactor yfactor zfactor}",
                           (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertMatrix (tSippGlobPtr, argv [1], &matrix))
        return TCL_ERROR;

    /*
     * Scale can be a list or a single factor.  If it contains any white space
     * assume its a list.
     */
    if (strpbrk (argv [2], " \f\t\n\r\v") == NULL) {
        if (Tcl_GetDouble (interp, argv [2], &scale.x) != TCL_OK)
            return TCL_ERROR;
        scale.y = scale.z = scale.x;
    } else {
        if (!TSippConvertVertex (tSippGlobPtr, argv [2], &scale))
            return TCL_ERROR;
    } 

    mat_scale (&matrix, scale.x, scale.y, scale.z);

    Tcl_SetResult (tSippGlobPtr->interp, TSippFormatMatrix (&matrix),
                   TCL_DYNAMIC);
    return TCL_OK;

}

/*=============================================================================
 * SippMatMirrorPlane --
 *   Implements the command:
 *     SippMatMirrorPlane matrix point normal
 * Notes:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippMatMirrorPlane (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Transf_mat    matrix;
    Vector        point, normal;

    if (argc != 4) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " matrix point normal", (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertMatrix (tSippGlobPtr, argv [1], &matrix))
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [2], &point))
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [3], &normal))
        return TCL_ERROR;

    mat_mirror_plane (&matrix, &point, &normal);

    Tcl_SetResult (tSippGlobPtr->interp, TSippFormatMatrix (&matrix),
                   TCL_DYNAMIC);
    return TCL_OK;

}

/*=============================================================================
 * SippMatMult --
 *   Implements the command:
 *     SippMatMult matrix0 matrix1
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippMatMult (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Transf_mat    matrix0, matrix1, resultMat;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " matrix0 matrix1", (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertMatrix (tSippGlobPtr, argv [1], &matrix0))
        return TCL_ERROR;
    if (!TSippConvertMatrix (tSippGlobPtr, argv [2], &matrix1))
        return TCL_ERROR;

    mat_mul (&resultMat, &matrix0, &matrix1);
    
    Tcl_SetResult (interp, TSippFormatMatrix (&resultMat), TCL_DYNAMIC);
    return TCL_OK;

}

/*=============================================================================
 * SippPointTransform --
 *   Implements the command:
 *     SippPointTransform point matrix
 * Notes:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippPointTransform (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    Vector        point, result;
    Transf_mat    matrix;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " point matrix", (char *) NULL);
        return TCL_ERROR;
    }
    if (!TSippConvertVertex (tSippGlobPtr, argv [1], &point))
        return TCL_ERROR;
    if (!TSippConvertMatrix (tSippGlobPtr, argv [2], &matrix))
        return TCL_ERROR;

    point_transform (&result, &point, &matrix);

    ReturnVector (tSippGlobPtr, &result);
    return TCL_OK;

}

/*=============================================================================
 * TSippGeomInit --
 *   Initialized the vector and matrix commands.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippGeomInit (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippMkVector",        (Tcl_CmdProc *) SippMkVector},
        {"SippVecNegate",       (Tcl_CmdProc *) SippVecNegate},
        {"SippVecDot",          (Tcl_CmdProc *) SippVecDot},
        {"SippVecLen",          (Tcl_CmdProc *) SippVecLen},
        {"SippVecAdd",          (Tcl_CmdProc *) SippVecAdd},
        {"SippVecSub",          (Tcl_CmdProc *) SippVecSub},
        {"SippVecMult",         (Tcl_CmdProc *) SippVecMult},
        {"SippVecCross",        (Tcl_CmdProc *) SippVecCross},
        {"SippMatIdent",        (Tcl_CmdProc *) SippMatIdent},
        {"SippMatTranslate",    (Tcl_CmdProc *) SippMatTranslate},
        {"SippMatRotateX",      (Tcl_CmdProc *) SippMatRotateX},
        {"SippMatRotateY",      (Tcl_CmdProc *) SippMatRotateY},
        {"SippMatRotateZ",      (Tcl_CmdProc *) SippMatRotateZ},
        {"SippMatRotate",       (Tcl_CmdProc *) SippMatRotate},
        {"SippMatScale",        (Tcl_CmdProc *) SippMatScale},
        {"SippMatMirrorPlane",  (Tcl_CmdProc *) SippMatMirrorPlane},
        {"SippMatMult",         (Tcl_CmdProc *) SippMatMult},
	{"SippPointTransform",  (Tcl_CmdProc *) SippPointTransform},
        {NULL,                  NULL}
    };

    TSippInitCmds (tSippGlobPtr, cmdTable);

}
