/*
 * oratcl.c --
 *
 * Oracle interface to Tcl
 *
 * Copyright 1993 Tom Poindexter and U S WEST Advanced Technologies, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  
 * Tom Poindexter and U S WEST make no representations about the suitability 
 * of this software for any purpose.  It is provided "as is" without express or
 * implied warranty.  By use of this software the user agrees to 
 * indemnify and hold harmless Tom Poindexter and U S WEST from any 
 * claims or liability for loss arising out of such use.
 *
 *-----------------------------------------------------------------------------
 * Version 1.0 July, 1993
 * Tom Poindexter, Boulder Colorado
 * tpoind@advtech.uswest.com  or tpoindex@nyx.cs.du.edu   
 *-----------------------------------------------------------------------------
 * Version 2.0 November, 1993
 * Tom Poindexter, Boulder Colorado
 * tpoind@advtech.uswest.com  or tpoindex@nyx.cs.du.edu   
 * -make changes to support Tcl 7.0 internal calls, init call changed
 * -remove dependence on TclX by #include tcl.h, tclUnix.h 
 * -change malloc/free to challoc/ckfree in case TCL_MEM_DEBUG
 *-----------------------------------------------------------------------------
 * Version 2.1 February, 1994
 * Tom Poindexter, Boulder Colorado
 * tpoind@advtech.uswest.com  or tpoindex@nyx.cs.du.edu   
 * -change Oratcl_Init to int and return ok
 * -add "colprec" and "colscales" to oracols (thanks-Dan R. Schenck)
 * -add optional repeat commands to orafetch command
 * -don't call get_lda_err from failed OraLogon, may not have good lda
 * -change way cur handles are made, don't use argv[1] anymore
 *-----------------------------------------------------------------------------
 *
 */

/* support for version 6 or version 7; if not 7 assume 6  */
#ifndef VERSION7
#ifndef VERSION6
#define VERSION6
#endif
#endif

#include "tcl.h"
#include "tclUnix.h"

#include <string.h>
#include <ctype.h>


/* define oracle cursor (and logon) structures */

struct cda_def
{
   short          v2_rc;        /* v2 return code */
   unsigned short ft;    	/* function type */
   unsigned long  rpc;          /* rows processed count */
   unsigned short peo;          /* parse error offset */
   unsigned char  fc;           /* function code */
   unsigned char  fill1;        /* filler  */
   unsigned short rc;           /* v7 return code */
   unsigned char  wrn;          /* warning flags */
   unsigned char  flg;          /* error flags */
   unsigned int   d0;           /* cursor number */
   struct {                     /* rowid structure */
     struct {
        unsigned long   d1;
        unsigned short  d2;
        unsigned char   d3;
        } rd;
     unsigned long  d4;         /* rba of datablock */
     unsigned short d5;         /* sequence number of row in block */
     } rid;
   unsigned int   ose;          /* os dependent error code */
   unsigned char  sysparm[27];  /* private, reserved fill */
};


typedef struct cda_def cda_def;
/* this fails with some compilers: typedef struct cda_def lda_def; */
#define lda_def cda_def 


/* _ANSI_ARGS_ should be defined by tcl.h; ignore if not defined */
#ifndef _ANSI_ARGS_
#define _ANSI_ARGS_() ()
#endif 

/* prototype malloc */

char * malloc _ANSI_ARGS_((unsigned int numbytes));

/* prototypes for oci function  */

#ifdef VERSION7

/* version 7 has a few extra functions */

int odescr _ANSI_ARGS_((struct cda_def *, int, unsigned long *, short *, char *,
         unsigned long *, unsigned long *, short *, short *, short *));

int oflng _ANSI_ARGS_((struct cda_def *, int, char *, long, int, 
	 unsigned long *, long));

int oparse _ANSI_ARGS_((struct cda_def *, char *, long, int, unsigned long));

#else

/* fake version 7 odescr and oparse */

#define odescr(p1,p2,p3,p4,p5,p6,p7,p8,p9,p10) \
	  odsc(p1,p2,p3,NULL,NULL,p4,p5,p6,p7)

#define oparse(p1,p2,p3,p4,p5)  osql3(p1,p2,(int) p3)

#endif


int obndrv _ANSI_ARGS_((struct cda_def *, char *, int, char *, int, int,
         int, short *, char *, int, int));

int obndrn _ANSI_ARGS_((struct cda_def *, int, char *, int, int,
         int, short *, char *, int, int));

int obreak _ANSI_ARGS_((struct lda_def *));

int ocan _ANSI_ARGS_((struct cda_def *));

int oclose _ANSI_ARGS_((struct cda_def *));

int ocof _ANSI_ARGS_((struct lda_def *));

int ocom _ANSI_ARGS_((struct lda_def *));

int ocon _ANSI_ARGS_((struct lda_def *));

int odefin _ANSI_ARGS_((struct cda_def *, int, char *, int, int, int, 
	 short *, char *, int, int, unsigned short *, unsigned short *));

int odsc _ANSI_ARGS_((struct cda_def *, int, short *, unsigned short *, short *,
         short *, char *, short *, short *));

int oerhms _ANSI_ARGS_((struct lda_def *, short, char *, int));

int oermsg _ANSI_ARGS_((short, char *));

int oexec _ANSI_ARGS_((struct cda_def *));

int oexn _ANSI_ARGS_((struct cda_def *, int, int));

int ofen _ANSI_ARGS_((struct cda_def *, int));

int ofetch _ANSI_ARGS_((struct cda_def *));

int ologof _ANSI_ARGS_((struct lda_def *));

int olon _ANSI_ARGS_((struct lda_def *, char *, int, char *, int, int));

int oopen _ANSI_ARGS_((struct cda_def *, struct lda_def *, char *, int,
         int, char *, int));

int oopt _ANSI_ARGS_((struct cda_def *, int, int));

int orlon _ANSI_ARGS_((struct lda_def *, char *, char *, int, char *, int));

int orol _ANSI_ARGS_((struct lda_def *));

int osql3 _ANSI_ARGS_((struct cda_def *, char *, int));


/* various limits for arrays of structures, default buffer sizes */
#define ORATCLLDAS      25	/* default number of ldas available */
#define ORATCLCURS      25	/* default number of curs available */
#define ORA_HOST_AREA	256	/* Oracle host data area length */
#define ORA_BUFF_SIZE	4096	/* conversion buffer size for various needs*/
#define ORA_MSG_SIZE	1000	/* oracle error message max size*/
#define ORA_CACHE       10      /* number of rows to fetch each time */

/* parse_cols defines */
#define NO_MORE_COLUMNS 1007	/* oracle error code # during odescr */
#define EXT_STRING_TYPE 5	/* oracle external data type for string */

/* version 7 db / oci 2.0 oparse() has some options */
#define IMMED_PARSE     0 
#define DEFER_PARSE     1
#define DEF_FLAG        DEFER_PARSE	/* defer sending to server */

#define VER6_BEHAVIOR   0
#define NORM_BEHAVIOR   1
#define VER7_BEHAVIOR   2
#define LNG_FLAG        NORM_BEHAVIOR	/* normal treatment of data types */

/* append_cols defines */
#define OUT_OF_SEQUENCE 1002	/* oracle error # fetch out of sequence */
#define NO_DATA_FOUND   1403	/* oracle error # no data for this fetch */
#define NULL_VALUE      1405	/* oracle error # column returned null */
#define COL_TRUNCATED   1406	/* oracle error # column truncated */
#define NUMBER_TYPE     2 	/* oracle internal data type for number */

/* pl exec error */
#define PL_EXEC_ERR     6503	/* oracle error # pl block error */

/* db long column max size */
#ifdef VERSION6
#define MAX_LONG_SIZE   65536 		/* v6 */
#else
#define MAX_LONG_SIZE   2147483647 	/* v7 */
#endif

#define LONG_BUFF_SIZE  32768   /* chunks to get via oflng() */




/* ColBufs holds information and data of SELECT columns */

struct ColBufs {
    struct ColBufs  *next_buf;	/* pointer to next ColBufs in list */
    short       dbtype;		/* column type */
    char        dbname[256];	/* column name */
#ifdef VERSION7
    unsigned long 
#else
    unsigned short   
#endif
                dbname_len,	/* column name length */
                disp_size,	/* column display size */
                dbsize;		/* column internal size */

    short       prec;		/* precision of numeric */
    short       scale;   	/* scale of numeric */
    short       nullok;         /* if null ok */
    long        col_len;        /* length of column data */
    char       *col_data;       /* pointer to acutal column data */
    unsigned short rlen[ORA_CACHE];       /* actual column length */
    unsigned short rcode[ORA_CACHE];      /* actual column code */
};

typedef struct ColBufs ColBufs;


/* OraTclLdas - oracle logon struct et.al. */

typedef struct OraTclLdas {	/* struct for lda entries */
    int         lda_in_use;	/* if this entry is opened */
    lda_def    *lda;		/* Oracle Logon Data Area */
    char       *hda;            /* Oracle Host Data Area */
} OraTclLdas;


/* OraTclCurs - oracle cursor + ColBufs head list ptr */

typedef struct OraTclCurs {	/* struct for cursor entries */
    int         cur_in_use;	/* if this entry is opened */
    int         lda_num;	/* entry in lda table for this cursor */
    cda_def    *cda;		/* Oracle Cursor Area */
    ColBufs    *col_list;       /* list of select columns */
    int         cache_size;     /* cache size of this cursor */
    int         row_num;        /* current row in cache to fetch */
    int         fetch_end;      /* true when fetches exhausted */
    long        fetch_cnt;      /* total number of rows fetched so far */
} OraTclCurs;

/* static tables for lda & cda */
static OraTclLdas   OraLdas[ORATCLLDAS];  
static OraTclCurs   OraCurs[ORATCLCURS];  

/* quick fetch flag */
static int  fetch_ok;

static char *OraHandlePrefix = "oratcl";  /* prefix used to identify handles*/
/* lda handles are:  prefix , lda index          e.g.  oratcl0    */
/* cur handles are:  lda handle , '.', cur index e.g.  oratcl0.0  */
/* a cursor handle could be passed to a proc that needs an lda handle */

static char *OraMsgArray = "oramsg";  /* array to place errors & messages */

/* prototypes for all TCL visible functions */

extern Tcl_CmdProc  Tcl_OraLogon;
extern Tcl_CmdProc  Tcl_OraLogoff;
extern Tcl_CmdProc  Tcl_OraOpen;
extern Tcl_CmdProc  Tcl_OraClose;
extern Tcl_CmdProc  Tcl_OraSql;
extern Tcl_CmdProc  Tcl_OraFetch;
extern Tcl_CmdProc  Tcl_OraCols;
extern Tcl_CmdProc  Tcl_OraCancel;
extern Tcl_CmdProc  Tcl_OraPLexec;
extern Tcl_CmdProc  Tcl_OraCommit;
extern Tcl_CmdProc  Tcl_OraRoll;
extern Tcl_CmdProc  Tcl_OraAutocom;
extern Tcl_CmdProc  Tcl_OraWrlong;
extern Tcl_CmdProc  Tcl_OraRdlong;



/* 
 *----------------------------------------------------------------------
 * get_ora_lda_handle
 *    authenticate an lda handle string 
 *  return: OraLdas index number or -1 on error;
 */

static int
get_ora_lda_handle (handle) 
    char *handle;
{
    int prefix_len = strlen(OraHandlePrefix);
    int lda_h;

    if ( (strlen(handle) > prefix_len) &&
	 (strncmp(handle,OraHandlePrefix,prefix_len) == 0)  &&
	 (isdigit(*(handle + prefix_len))) 
       ) {

	 lda_h = atoi((handle + prefix_len));
	 if (lda_h < 0 || lda_h >= ORATCLLDAS) {
	   return (-1);
	 }
	 if (OraLdas[lda_h].lda_in_use) {
	   return (lda_h);
	 } else {
	   return (-1);
	 }
    } 

    return (-1);
}


/* 
 *----------------------------------------------------------------------
 * get_ora_handle
 *    authenticate a cursor handle string 
 *  return: OraCurs index number or -1 on error;
 */

static int
get_ora_handle (handle) 
    char *handle;
{
    int prefix_len = strlen(OraHandlePrefix);
    int lda_h      = get_ora_lda_handle(handle);
    int cur_h;
    char *p;

    if (lda_h == -1) {
	return (-1);
    }

    if ( ((p = strchr(handle,'.')) != NULL) && (isdigit(*(p+1))) ) {

	 cur_h = atoi((p + 1));
	 if (cur_h < 0 || cur_h >= ORATCLCURS) {
	   return (-1);
	 }
	 if (OraCurs[cur_h].cur_in_use) {
	   return (cur_h);
	 } else {
	   return (-1);
	 }
    } 

    return (-1);
}



/*
 *----------------------------------------------------------------------
 * get_lda_err
 *   get the oracle return code and error message for an lda error
 */

static void
get_lda_err (interp, hand)
    Tcl_Interp *interp;
    int         hand;
{

    char buf[ORA_MSG_SIZE];

    sprintf(buf,"%d",OraLdas[hand].lda->rc);
    Tcl_SetVar2(interp,OraMsgArray,"rc",buf,TCL_GLOBAL_ONLY);

    if (OraLdas[hand].lda->rc == 0) {
	buf[0] = '\0';
    } else {
        oerhms (OraLdas[hand].lda, OraLdas[hand].lda->rc, 
                  buf, ORA_MSG_SIZE);
    }
    Tcl_SetVar2(interp,OraMsgArray,"errortxt",buf,TCL_GLOBAL_ONLY);
}



/*
 *----------------------------------------------------------------------
 * get_ora_err
 *   get the oracle return code and error message
 */

static void
get_ora_err (interp, hand)
    Tcl_Interp *interp;
    int         hand;
{

    char buf[ORA_MSG_SIZE];


    sprintf(buf,"%d",OraCurs[hand].cda->rc);
    Tcl_SetVar2(interp,OraMsgArray,"rc",buf,TCL_GLOBAL_ONLY);

    sprintf(buf,"%d",OraCurs[hand].cda->ft);
    Tcl_SetVar2(interp,OraMsgArray,"sqlfunc",buf,TCL_GLOBAL_ONLY);

    sprintf(buf,"%d",OraCurs[hand].cda->fc);
    Tcl_SetVar2(interp,OraMsgArray,"ocifunc",buf,TCL_GLOBAL_ONLY);

    if (OraCurs[hand].cda->rc == 0) {
	buf[0] = '\0';
    } else {
        oerhms (OraLdas[OraCurs[hand].lda_num].lda, OraCurs[hand].cda->rc, 
	        buf, (int) ORA_MSG_SIZE);
    }

    Tcl_SetVar2(interp,OraMsgArray,"errortxt",buf,TCL_GLOBAL_ONLY);
	    

}




/*
 *----------------------------------------------------------------------
 * clear_msg --
 *
 * clears all error and message elements in the global array variable
 *
 */

static void
clear_msg(interp)
    Tcl_Interp *interp;
{
    /* index "rc" - return code from last oratcl call */
    Tcl_SetVar2(interp, OraMsgArray, "rc",       "", TCL_GLOBAL_ONLY);

    /* index "sqlfunc" - sql function code of last sql */
    Tcl_SetVar2(interp, OraMsgArray, "sqlfunc",  "", TCL_GLOBAL_ONLY);

    /* index "ocifunc" - oci function code of last call */
    Tcl_SetVar2(interp, OraMsgArray, "ocifunc",  "", TCL_GLOBAL_ONLY);

    /* index "errortxt" - text from last rc */
    Tcl_SetVar2(interp, OraMsgArray, "errortxt", "", TCL_GLOBAL_ONLY);

    /* index "rows" - rows processed count from oexec */
    Tcl_SetVar2(interp, OraMsgArray, "rows",     "", TCL_GLOBAL_ONLY);

    /* index "collengths" & "coltypes" only meaningful after oracols command */
    /* same for "colprecs" and "colscales"                                   */
    Tcl_SetVar2(interp, OraMsgArray, "collengths","", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, OraMsgArray, "coltypes",  "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, OraMsgArray, "colprecs",  "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, OraMsgArray, "colscales", "", TCL_GLOBAL_ONLY);


}



/*
 *----------------------------------------------------------------------
 * ora_prologue
 *
 * does most of standard command prologue, assumes handle is argv[1]
 * returns: cursor handle index  or -1 on failure
 * 
 */

static int
ora_prologue (interp, argc, argv, num_args, err_msg)
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
    int         num_args;
    char       *err_msg;
{
    int         hand;

    clear_msg(interp);

    /* check number of minimum args*/

    if (argc < num_args) {
	Tcl_AppendResult (interp, "wrong # args: ", argv[0],
			  err_msg, (char *) NULL);
	return (-1);
    }

    /* parse the handle */
    hand = get_ora_handle(argv[1]);

    if (hand == -1) {
	Tcl_AppendResult (interp, argv[0], ": handle ", argv[1],
			 " not open ", (char *) NULL);
	return (-1);
    }

    /* clear oramsg array for new messages & errors */
    Tcl_SetVar2(interp, OraMsgArray, "handle",  argv[1], TCL_GLOBAL_ONLY);

    return (hand);
}



/*
 *----------------------------------------------------------------------
 * free_cols
 *   free all ColBufs in a list
 *
 */

static int
free_cols (col_list_ptr)
    ColBufs  *col_list_ptr;
{
    ColBufs  *next;

    while (col_list_ptr != NULL) {
        next = col_list_ptr->next_buf;
	if (col_list_ptr->col_data != NULL) {
	    ckfree(col_list_ptr->col_data);
	}
	ckfree(col_list_ptr);
	col_list_ptr = next;
    }
}


/*
 *----------------------------------------------------------------------
 * free_curs
 *   free all cursors belonging to an lda
 *
 */

static int
free_curs (lda_owner)
    int  lda_owner;
{
    int  i;

    for (i = 0; i < ORATCLCURS; i++) {
	if (OraCurs[i].cur_in_use && (OraCurs[i].lda_num == lda_owner)) {
	    OraCurs[i].cur_in_use = 0;
	    OraCurs[i].lda_num    = 0;
	    oclose(OraCurs[i].cda);
	    ckfree(OraCurs[i].cda);
	    OraCurs[i].cda        = NULL;
	    free_cols(OraCurs[i].col_list);
	    OraCurs[i].col_list   = NULL;;
	}
    }
}




/*
 *----------------------------------------------------------------------
 * alloc_col
 *   return a new ColBufs with nulled pointers and zeroed fields
 */

static ColBufs *
alloc_col()
{
    ColBufs *new_col;

    if ((new_col = (ColBufs *) ckalloc(sizeof (ColBufs))) != NULL) {
	new_col->next_buf      = NULL;
	new_col->dbsize        = 0;
	new_col->dbtype        = 0;
	new_col->dbname[0]     = '\0';
	new_col->dbname_len    = sizeof new_col->dbname;
	new_col->disp_size     = 0;
	new_col->prec          = 0;
	new_col->scale         = 0;
	new_col->nullok        = 0;
	new_col->col_len       = 0;
	new_col->col_data      = NULL;
	new_col->rlen[0]       = 0;
	new_col->rcode[0]      = 0;
    }

    return new_col;

}


/*
 *----------------------------------------------------------------------
 * parse_columns
 *   parse result columns, allocate memory for fetches
 *   return -1 on error, 1 if ok
 */

static int
parse_columns (interp, hand)
    Tcl_Interp *interp;
    int         hand;
{
    ColBufs    *new_col_head;
    ColBufs    *new_col;
    ColBufs    *last_col;
    int         pos = 0;
    long        long_size;
    int         i;

    /* start by allocating column structures, and keep in linked list */
    
    new_col_head = alloc_col();
    if (new_col_head == NULL) {
	return -1;
    }
    new_col      = new_col_head;
    last_col     = new_col_head;

    /* get max long size */
    long_size = atol(Tcl_GetVar2(interp, OraMsgArray, "maxlong",
                     TCL_GLOBAL_ONLY));

    if (long_size < 0 ) {
	long_size = 0;
    }
    if (long_size > MAX_LONG_SIZE) {
	long_size = MAX_LONG_SIZE;
    }
    
    /* start with initial cache size of ORA_CACHE; may be reduced to 1 */
    OraCurs[hand].cache_size = ORA_CACHE;

    while (1) {		/* loop breaks when no more result columns */

        /* get oracle description of column data */
	odescr(OraCurs[hand].cda,(pos+1),&(new_col->dbsize),&(new_col->dbtype),
          new_col->dbname, &(new_col->dbname_len), &(new_col->disp_size),
          &(new_col->prec), &(new_col->scale), &(new_col->nullok)   );


       
	if (OraCurs[hand].cda->rc == NO_MORE_COLUMNS) {
	    last_col->next_buf = NULL;
	    ckfree(new_col);
	    break;
	}

	if (OraCurs[hand].cda->rc != 0) {
	    get_ora_err(interp, hand);
	    free_cols(new_col_head);
	    return -1;
	}

	/* trim trailing blanks from dbname */
        for (i = (sizeof new_col->dbname) - 1; i >= 0; i--) {
	    if (new_col->dbname[i] != ' ') {
		new_col->dbname[i+1] = '\0';
		break;
	    }
	}

	/* check for long (type 8) or long raw (type 24) */
	/* if long or long raw found, then set display size to long_size */
	/* and reduce cache to 1 for this cursor */

	if (new_col->dbtype == 8 || new_col->dbtype == 24) {
	    new_col->disp_size = long_size; 
	    OraCurs[hand].cache_size = 1;
	}

	pos++;
	last_col           = new_col;
	new_col            = alloc_col();
	if (new_col == NULL) {
	    free_cols(new_col_head);
	    return -1;
	}
	last_col->next_buf = new_col;

    }
     
    if (pos > 0) {
	OraCurs[hand].col_list = new_col_head;
    } else {
	/* no results, just return */
	OraCurs[hand].col_list = NULL;
	return 1;
    }

    /* now define the fetch areas */

    for (new_col = new_col_head, pos=1; new_col != NULL; 
				   new_col = new_col->next_buf, pos++) {
	new_col->disp_size += 1;   /* add room for null terminator */

	new_col->col_data=ckalloc(new_col->disp_size *OraCurs[hand].cache_size);
	if (new_col->col_data == NULL) {
	    free_cols(new_col_head);
	    OraCurs[hand].col_list = NULL;
	    return -1;
	}

	odefin(OraCurs[hand].cda, pos, new_col->col_data, 
	       (int) new_col->disp_size, EXT_STRING_TYPE, 0, (short *) 0, 
	       NULL, 0, 0, new_col->rlen, new_col->rcode);

    }

    return (1);

}


/*
 *----------------------------------------------------------------------
 *
 * append_cols
 *      appends column results to tcl result
 */

static void
append_cols (interp, hand)
    Tcl_Interp  *interp;
    int          hand;
{
    int      i;
    ColBufs *col;
    char    *data_ptr;
    char     buf[ORA_MSG_SIZE];
    char     buf2[ORA_MSG_SIZE];

    i = OraCurs[hand].row_num;

    if (i < 0 || i >= OraCurs[hand].cache_size) {	/* quick sanity check */
	return;
    }

    strncpy(buf2,Tcl_GetVar2(interp,OraMsgArray,"nullvalue",TCL_GLOBAL_ONLY),
	    (sizeof buf2) - 1);

    for (col = OraCurs[hand].col_list; col != NULL; col = col->next_buf) {

        /* get pointer to next row column data buffer */
	data_ptr = col->col_data + (i * col->disp_size);

        /* null column? except long (8) & long raw (24) may report null     */
	/*              in error, so check rlen for those types, for other  */
	/*              types, just check rcode                             */

	if ( ( (col->dbtype == 8 || col->dbtype == 24) && col->rlen[i] == 0 )
	     ||
	     ( (col->dbtype != 8 && col->dbtype != 24) && 
	        col->rcode[i] == NULL_VALUE ) ) {

	    if (strlen(buf2) > 0) {
		/* return user defined nullvalue */
		data_ptr = buf2;
	    } else {
		/* "nullvalue" null, use default of 0 for number, "" all else */
		if (col->dbtype == NUMBER_TYPE) {
		    data_ptr = "0";
		} else {
		    data_ptr = "";
		}
	    }

	} else {

	    /* make sure data is null terminated, just in case */
	    if (col->disp_size > 0) {
		*(data_ptr + (col->disp_size) - 1) = '\0';
	    } else {
		data_ptr = "";
	    }
	}

        Tcl_AppendElement(interp, data_ptr);
    }


    OraCurs[hand].row_num += 1;
    OraCurs[hand].fetch_cnt += 1;
    sprintf(buf,"%ld",OraCurs[hand].fetch_cnt);
    Tcl_SetVar2(interp, OraMsgArray, "rows", buf, TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, OraMsgArray, "rc",   "0", TCL_GLOBAL_ONLY);

    fetch_ok = 1;

}



/*
 *----------------------------------------------------------------------
 * Tcl_KillOra --
 *   perform all cleanup upon any command deletion
 *
 */

void
Tcl_KillOra (clientData)
    ClientData clientData;
{
    int i;


    for (i = 0; i < ORATCLLDAS; i++) {
	if (OraLdas[i].lda_in_use) {
	    free_curs(i);
	    OraLdas[i].lda_in_use = 0;
	    ologof(OraLdas[i].lda);
	    ckfree(OraLdas[i].lda);
	    ckfree(OraLdas[i].hda);
	}
    }

}


/*
 *----------------------------------------------------------------------
 * Oratcl_Init --
 *   perform all initialization for the Oracle - Tcl interface.
 *   adds additional commands to interp, creates message array
 *
 *   a call to Tcl_InitOra should exist in Tcl_CreateInterp or
 *   Tcl_CreateExtendedInterp.
 */

int
Oratcl_Init (interp)
    Tcl_Interp *interp;
{
    int i;


    /*
     * Initialize oratcl structures 
     */

    for (i = 0; i < ORATCLLDAS; i++) {
	OraLdas[i].lda_in_use    = 0;
	OraLdas[i].lda           = NULL;
	OraLdas[i].hda           = NULL;
    }

    for (i = 0; i < ORATCLCURS; i++) {
	OraCurs[i].cur_in_use    = 0;
	OraCurs[i].lda_num       = 0;
	OraCurs[i].cda           = NULL;
	OraCurs[i].col_list      = NULL;
	OraCurs[i].cache_size    = ORA_CACHE;
	OraCurs[i].row_num       = 0;
	OraCurs[i].fetch_end     = 1;
	OraCurs[i].fetch_cnt     = 0;
    }


    /*
     * Initialize the new Tcl commands
     */

    Tcl_CreateCommand (interp, "oralogon",   Tcl_OraLogon  , (ClientData)NULL,
		      Tcl_KillOra);
    Tcl_CreateCommand (interp, "oralogoff",  Tcl_OraLogoff,  (ClientData)NULL,
		      Tcl_KillOra);
    Tcl_CreateCommand (interp, "oraopen",    Tcl_OraOpen,    (ClientData)NULL,
		      Tcl_KillOra);
    Tcl_CreateCommand (interp, "oraclose",   Tcl_OraClose,   (ClientData)NULL,
		      Tcl_KillOra);
    Tcl_CreateCommand (interp, "orasql",     Tcl_OraSql,     (ClientData)NULL,
		      Tcl_KillOra);
    Tcl_CreateCommand (interp, "orafetch",   Tcl_OraFetch,   (ClientData)NULL,
		      Tcl_KillOra);
    Tcl_CreateCommand (interp, "oracols",    Tcl_OraCols,    (ClientData)NULL,
		      Tcl_KillOra);
    Tcl_CreateCommand (interp, "oracancel",  Tcl_OraCancel,  (ClientData)NULL,
		      Tcl_KillOra);
    Tcl_CreateCommand (interp, "oraplexec",  Tcl_OraPLexec,  (ClientData)NULL,
		      Tcl_KillOra);
    Tcl_CreateCommand (interp, "oracommit",  Tcl_OraCommit,  (ClientData)NULL,
		      Tcl_KillOra);
    Tcl_CreateCommand (interp, "oraroll",    Tcl_OraRoll,    (ClientData)NULL,
		      Tcl_KillOra);
    Tcl_CreateCommand (interp, "oraautocom", Tcl_OraAutocom, (ClientData)NULL,
		      Tcl_KillOra);
    Tcl_CreateCommand (interp, "orawritelong",Tcl_OraWrlong, (ClientData)NULL,
		      Tcl_KillOra);
    Tcl_CreateCommand (interp, "orareadlong",Tcl_OraRdlong,  (ClientData)NULL,
		      Tcl_KillOra);

    /*
     * Initialize oramsg global array, inital null elements
     */
    
    /* index "handle" - the last handle that set any other elements */
    Tcl_SetVar2(interp, OraMsgArray, "handle",   "", TCL_GLOBAL_ONLY);

    /* index "nullvalue" - what to return if column is null */
    /* user should set this value if something else is wanted */
    Tcl_SetVar2(interp, OraMsgArray, "nullvalue", "",TCL_GLOBAL_ONLY);

    /* index "maxlong" - the maximum length of a long column to return */
    /* in a select or oralong , can be set by user                 */
    Tcl_SetVar2(interp, OraMsgArray, "maxlong", "32768",TCL_GLOBAL_ONLY);

    /* other indices - correspond to error and message handler arguments */
    clear_msg(interp);

    return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraLogon --
 *    Implements the oralogon command:
 *    usage: oralogon connect_str
 *       connect_str should be a valid oracle logon string, consisting of:
 *         name
 *         name/password
 *         name@d:dbname
 *         name/password@d:dbname
 *	                
 *    results:
 *	handle - a character string of newly open logon handle
 *      TCL_OK - connect successful
 *      TCL_ERROR - connect not successful - error message returned
 */

int
Tcl_OraLogon (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{

    int        hand = -1;
    int        i;
    char       buf[ORA_MSG_SIZE];


    /* can't use ora_prologue, oralogon creates a logon handle */

    clear_msg(interp);

    if (argc < 2) {
	Tcl_AppendResult (interp, "wrong # args: ", argv[0],
			  " connect_str ", (char *) NULL);
	return TCL_ERROR;
    }

    /* find an unused handle */

    for (i = 0; i < ORATCLLDAS; i++) {
	if (OraLdas[i].lda_in_use == 0) {
            hand = i;
	    break;
	}
    }

    if (hand == -1) {
	Tcl_AppendResult (interp, argv[0], ": no lda handles available",
			  (char *) NULL);
	return TCL_ERROR;
    }


    if ( (OraLdas[hand].lda = (lda_def *) ckalloc(sizeof (lda_def))) == NULL) {
	Tcl_AppendResult (interp, argv[0], ": oralogon failed in malloc(lda)",
			  (char *) NULL);
	return TCL_ERROR;
    }
    if ( (OraLdas[hand].hda = ckalloc(ORA_HOST_AREA)) == NULL) {
        ckfree(OraLdas[hand].lda);
	Tcl_AppendResult (interp, argv[0], ": oralogon failed in malloc(hda)",
			  (char *) NULL);
	return TCL_ERROR;
    }


    if (orlon(OraLdas[hand].lda,OraLdas[hand].hda,argv[1], -1,NULL,-1) != 0) {
	/* logon failed, cleanup */
	/* can't use get_lda_err(), may not have connected to server*/
        sprintf(buf,"%d",OraLdas[hand].lda->rc);
        Tcl_SetVar2(interp,OraMsgArray,"rc",buf,TCL_GLOBAL_ONLY);
        Tcl_SetVar2(interp,OraMsgArray,"errortxt",
           "logon failed: invalid id/password or unable to connect to server",
	    TCL_GLOBAL_ONLY);

        ckfree(OraLdas[hand].lda);
        ckfree(OraLdas[hand].hda);
	Tcl_AppendResult (interp, argv[0], ": oralogon failed in orlon",
			  (char *) NULL);
	return TCL_ERROR;
    }

    OraLdas[i].lda_in_use = 1;	/* handle ok, set in use flag */

    /* construct logon handle and return */
    sprintf(buf,"%s%d",OraHandlePrefix,hand);

    Tcl_SetVar2(interp, OraMsgArray, "handle",      buf, TCL_GLOBAL_ONLY);
    clear_msg(interp);

    Tcl_SetResult(interp,buf,TCL_VOLATILE);

    return TCL_OK;

}




/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraLogoff --
 *    Implements the oralogoff command:
 *    usage: oralogon lda_handle
 *       lda_handle should be a valid, open lda handle from oralogon
 *	                
 *    results:
 *	null string
 *      TCL_OK - logoff successful
 *      TCL_ERROR - logoff not successful - error message returned
 */

int
Tcl_OraLogoff (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{

    int        hand;
    int        i;
    char       buf[ORA_MSG_SIZE];


    /* can't use ora_prologue, oralogoff just uses an lda handle */

    clear_msg(interp);

    if (argc < 2) {
	Tcl_AppendResult (interp, "wrong # args: ", argv[0],
			  " lda_handle ", (char *) NULL);
	return TCL_ERROR;
    }

    if ((hand = get_ora_lda_handle(argv[1])) == -1) {
	Tcl_AppendResult (interp, argv[0], ": lda_handle ", argv[1],
			  " not valid ", (char *) NULL);
	return TCL_ERROR;
    }

    /* close and free all open cursors through this connection */
    free_curs(hand);

    /* logoff and free data areas */
    OraLdas[hand].lda_in_use = 0;
    if (ologof(OraLdas[hand].lda) != 0) {
	/* logoff failed, get error msg */
	get_lda_err(interp,hand);
        ckfree(OraLdas[hand].lda);
        ckfree(OraLdas[hand].hda);

	return TCL_ERROR;
    }

    ckfree(OraLdas[hand].lda);
    ckfree(OraLdas[hand].hda);

    return TCL_OK;
}




/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraOpen --
 *    Implements the oraopen command:
 *    usage: oralogon lda_handle
 *       lda_handle should be a valid, open lda handle from oralogon
 *	                
 *    results:
 *	cur_handle
 *      TCL_OK - oopen successful
 *      TCL_ERROR - oopen not successful - error message returned
 */

int
Tcl_OraOpen (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{

    int        lda_hand;
    int        hand = -1;
    int        i;
    char       buf[ORA_MSG_SIZE];


    /* can't use ora_prologue, oraopen just uses an lda handle */

    clear_msg(interp);

    if (argc < 2) {
	Tcl_AppendResult (interp, "wrong # args: ", argv[0],
			  " lda_handle ", (char *) NULL);
	return TCL_ERROR;
    }

    if ((lda_hand = get_ora_lda_handle(argv[1])) == -1) {
	Tcl_AppendResult (interp, argv[0], ": lda_handle ", argv[1],
			  " not valid ", (char *) NULL);
	return TCL_ERROR;
    }


    /* find an unused cursor handle */

    for (i = 0; i < ORATCLCURS; i++) {
	if (OraCurs[i].cur_in_use == 0) {
            hand = i;
	    break;
	}
    }

    if (hand == -1) {
	Tcl_AppendResult (interp, argv[0], ": no cursor handles available",
			  (char *) NULL);
	return TCL_ERROR;
    }


    if ( (OraCurs[hand].cda = (cda_def *) ckalloc(sizeof (cda_def))) == NULL) {
	Tcl_AppendResult (interp, argv[0], ": oraopen failed in malloc(cda)",
			  (char *) NULL);
	return TCL_ERROR;
    }
    
    if (oopen(OraCurs[hand].cda,OraLdas[lda_hand].lda,NULL,-1,-1,NULL,-1) !=0) {
	/* open failed, cleanup */
	get_ora_err(interp,hand);
        ckfree(OraCurs[hand].cda);
	Tcl_AppendResult (interp, argv[0], ": oraopen failed in oopen",
			  (char *) NULL);
	return TCL_ERROR;
    }

    /* cursor open ok */
    OraCurs[hand].cur_in_use = 1;
    OraCurs[hand].lda_num    = lda_hand;
    OraCurs[hand].col_list   = NULL;
    OraCurs[hand].cache_size = ORA_CACHE;
    OraCurs[hand].row_num    = 0;
    OraCurs[hand].fetch_end  = 1;
    OraCurs[hand].fetch_cnt  = 0;


    /* construct and return a cursor handle */
    sprintf(buf,"%s%d.%d",OraHandlePrefix,lda_hand,hand);  

    Tcl_SetVar2(interp, OraMsgArray, "handle",      buf, TCL_GLOBAL_ONLY);
    clear_msg(interp);

    Tcl_SetResult(interp,buf,TCL_VOLATILE);

    return TCL_OK;

}



/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraClose --
 *    Implements the oraclose command:
 *    usage: oraclose cur_handle
 *	                
 *    results:
 *	null string
 *      TCL_OK - cursor closed successfully
 *      TCL_ERROR - wrong # args, or cur_handle not opened
 */

Tcl_OraClose (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;

    if ((hand = ora_prologue(interp,argc, argv, 2," cur_handle")) == -1) {
	return TCL_ERROR;
    }

    OraCurs[hand].cur_in_use = 0;
    OraCurs[hand].lda_num    = 0;
    oclose(OraCurs[hand].cda);
    ckfree(OraCurs[hand].cda);
    OraCurs[hand].cda        = NULL;
    free_cols(OraCurs[hand].col_list);
    OraCurs[hand].col_list   = NULL;
    OraCurs[hand].cache_size = ORA_CACHE;
    OraCurs[hand].row_num    = 0;
    OraCurs[hand].fetch_end  = 1;
    OraCurs[hand].fetch_cnt  = 0;

    return TCL_OK;

}



/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraSql --
 *    Implements the orasql command:
 *    usage: orasql cur_handle sql_string 
 *	                
 *    results:
 *	return code from oexec
 *      TCL_OK - handle is opened, sql executed ok
 *      TCL_ERROR - wrong # args, or handle not opened,  bad sql stmt
 */

Tcl_OraSql (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;
    char    buf[ORA_MSG_SIZE];


    if ((hand = ora_prologue(interp,argc, argv,3," cur_handle sql_str")) ==-1) {
	return TCL_ERROR;
    }

    /* cancel any pending usage of cursor */
    ocan(OraCurs[hand].cda);

    /* set up sql for execution */
    oparse(OraCurs[hand].cda, argv[2], ((long) -1), DEF_FLAG, LNG_FLAG);


    if (OraCurs[hand].cda->rc != 0) {
	get_ora_err(interp,hand);
	Tcl_AppendResult (interp, argv[0], ": oparse failed (bad sql ?) ", 
			  (char *) NULL);
	return TCL_ERROR;
    }

    /* clear any previous results */
    free_cols(OraCurs[hand].col_list);
    OraCurs[hand].col_list   = NULL;
    OraCurs[hand].cache_size = ORA_CACHE;
    OraCurs[hand].row_num    = 0;
    OraCurs[hand].fetch_end  = 1;
    OraCurs[hand].fetch_cnt  = 0;

    if (parse_columns(interp, hand) != -1) {

        oexec(OraCurs[hand].cda);

        /* set msg array variable "rows" to rpc (rows processed count) */
        /* in case of insert, update, delete,                          */
        sprintf(buf,"%d",OraCurs[hand].cda->rpc);
        Tcl_SetVar2(interp, OraMsgArray, "rows", buf, TCL_GLOBAL_ONLY);

	/* cause first orafetch to fetch */
	OraCurs[hand].row_num   = OraCurs[hand].cache_size;
	OraCurs[hand].fetch_end = 0;        /* try to fetch */
	OraCurs[hand].fetch_cnt = 0;        /* start fetch cnt at zero */

    } else {
	get_ora_err(interp,hand);
	Tcl_AppendResult (interp, argv[0], ": parse_columns failed ", 
			  (char *) NULL);
	return TCL_ERROR;
    }

    get_ora_err(interp,hand);

    sprintf(buf,"%d",OraCurs[hand].cda->rc);

    Tcl_SetResult(interp, buf, TCL_VOLATILE);

    return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraFetch --
 *    Implements the orafetch command:
 *    usage: orafetch cur_handle 
 *	                
 *    results:
 *	next row from latest results as tcl list, or null list
 *	sets message array element "rc" with value of fetch rcode
 *      TCL_OK - handle is opened
 *      TCL_ERROR - wrong # args, or handle not opened
 */

Tcl_OraFetch (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int      hand;
    char     buf[ORA_MSG_SIZE];
    char    *data_ptr;
    long     old_rpc;
    ColBufs *col;
    int      i;
    int      Tcl_OraFetchAll();


    if ((hand = ora_prologue(interp, argc, argv, 2, 
	 " cur_handle  ?commands?")) == -1) {
	return TCL_ERROR;
    }

    /* if there's a third argument, let next proc do the work */
    if (argc >= 3) {
	return (Tcl_OraFetchAll(clientData, interp, argc, argv));
    }

    fetch_ok = 0;

    /* check if already exahusted */

    if (OraCurs[hand].fetch_end && 
	OraCurs[hand].fetch_cnt >= OraCurs[hand].cda->rpc &&
	OraCurs[hand].fetch_cnt >  0) {

	sprintf(buf,"%d",NO_DATA_FOUND);
        Tcl_SetVar2(interp, OraMsgArray, "rc",   buf, TCL_GLOBAL_ONLY);
        sprintf(buf,"%ld",OraCurs[hand].cda->rpc);
        Tcl_SetVar2(interp, OraMsgArray, "rows", buf, TCL_GLOBAL_ONLY);
	return TCL_OK;
    }

    /* check to see if next set of results */

    old_rpc = OraCurs[hand].cda->rpc;

    if (OraCurs[hand].row_num >= OraCurs[hand].cache_size  ||
	OraCurs[hand].fetch_cnt >= OraCurs[hand].cda->rpc) {

        for (col = OraCurs[hand].col_list; col != NULL; col = col->next_buf) {
	    for (i = 0; i < OraCurs[hand].cache_size; i++) {
                col->rcode[i] = 0;
                col->rlen[i] = 0;
	    }
	}

	ofen(OraCurs[hand].cda, OraCurs[hand].cache_size);

	if (OraCurs[hand].cda->rc == OUT_OF_SEQUENCE) {
	    OraCurs[hand].fetch_cnt = 0;
	    OraCurs[hand].cda->rpc  = 0;
	    OraCurs[hand].fetch_end = 1;
	    OraCurs[hand].row_num   = OraCurs[hand].cache_size;
	    sprintf(buf,"%d",OUT_OF_SEQUENCE);
	    Tcl_SetVar2(interp, OraMsgArray, "rc", buf, TCL_GLOBAL_ONLY);
	    sprintf(buf,"%ld",OraCurs[hand].cda->rpc);
	    Tcl_SetVar2(interp, OraMsgArray, "rows", buf, TCL_GLOBAL_ONLY);
	    return TCL_OK;
	}

	if (OraCurs[hand].cda->rc == NO_DATA_FOUND) {
	    OraCurs[hand].fetch_end = 1;
	    if (old_rpc == OraCurs[hand].cda->rpc) {
	        sprintf(buf,"%d",NO_DATA_FOUND);
                Tcl_SetVar2(interp, OraMsgArray, "rc", buf, TCL_GLOBAL_ONLY);
                sprintf(buf,"%ld",OraCurs[hand].cda->rpc);
                Tcl_SetVar2(interp, OraMsgArray, "rows", buf, TCL_GLOBAL_ONLY);
	        return TCL_OK;
	    }
	}
        OraCurs[hand].row_num = 0;
    }

    append_cols(interp, hand);

    return TCL_OK;
}





/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraFetchAll --
 *    Implements the orafetch with iterative commands:
 *    usage: orafetch cur_handle tcl_stmts
 *	                
 *    results:
 *	fetch all rows from existing query, execute tcl_stmt on each row
 *	sets message array element "rc" with value of final fetch rcode
 *      TCL_OK - handle is opened
 *      TCL_ERROR - wrong # args, or handle not opened, tcl_stmt failed,
 *                  or wrong number of columns
 */

static int
Tcl_OraFetchAll (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int      hand;
    char     buf[ORA_MSG_SIZE];
    char    *eval_ptr;
    int      colsCnt;
    char   **colsPtr;
    int      i;
    char    *s;
    char    *s2;
    char    *p;
    ColBufs *col;
    int      max_col = 0;

#define NUMSTRS 300		/* number of individual strings to allow*/
#define SUBCHAR '@'		/* substitution character */

    int      num_str = 0;	/* number of strings to concat */
    char    *str[NUMSTRS];      /* strings to pass to concat   */

    int      colnum  = 0; 	/* number of colstr's used */

    struct {			
      int      column; 		/* column number index */
      int      strnum;          /* str array index */
    } colstr[NUMSTRS];

    Tcl_DString evalStr;	/* tcl dynamic string for building eval */
    int        inum;
    int        icol;
    int        tcl_rc;


    if ((hand = ora_prologue(interp, argc, argv, 3, 
			    " cur_handle  commands")) == -1) {
	return TCL_ERROR;
    }

    /* check if already exahusted or nothing to fetch */

    if (OraCurs[hand].fetch_end && 
	OraCurs[hand].fetch_cnt >= OraCurs[hand].cda->rpc &&
	OraCurs[hand].fetch_cnt >  0) {

	sprintf(buf,"%d",NO_DATA_FOUND);
        Tcl_SetVar2(interp, OraMsgArray, "rc",   buf, TCL_GLOBAL_ONLY);
        sprintf(buf,"%ld",OraCurs[hand].cda->rpc);
        Tcl_SetVar2(interp, OraMsgArray, "rows", buf, TCL_GLOBAL_ONLY);
	return TCL_OK;
    }

    if (OraCurs[hand].col_list == NULL) {   
	return TCL_OK;			/* nothing to fetch */
    }

    /* parse tcl_stmt for all '@' substitutions */

    s  = argv[2];			/* start of tcl_stmt */
    s2 = s;

    while ( (p = strchr(s2,SUBCHAR)) != NULL) {

      if (isdigit(*(p+1))) {		/* it's a substituion value ! */

	  if (s != p) {			/* subchar not at the front of string?*/
	      str[num_str] = s;		/* save the current front of string */
	      *p = '\0';  		/* terminate the literal string     */
	      num_str++;		/* into str array and               */
	  }

          i = atoi(p + 1);		/* the column number */

	  max_col = (i > max_col) ? i : max_col;/* sanity check, max col num */

	  colstr[colnum].column = i;		/* save column number */
	  colstr[colnum].strnum = num_str;	/* and str positiion  */
	  str[num_str] = "";			/* sanity check       */
	  colnum++;
	  num_str++;
	  p++;				/* skip past col number */
	  while (isdigit(*p)) {
	      p++;
	  }
	  s  = p;			/* next point to look for subchar */
	  s2 = p;			


      } else {				/* it's a subchar without number */
	  s2 = p + 1;			/* ignore and keep searching     */
      }

    }

    if (num_str == 0) {    		/* check if no substitutions */
	str[num_str] = argv[2];	
	num_str = 1;	
    } else {		
	if (strlen(s) > 0) {		/* remainder of tcl_stmt string */
            str[num_str] = s;
	    num_str++;
	}
    }


    /* make sure there are enough columns in result */
    i = 0;
    for (col = OraCurs[hand].col_list; col != NULL; col = col->next_buf) {
	i++;
    }
    if (max_col > i) {
	Tcl_AppendResult (interp, argv[0], ": @column number execeeds results", 
			  (char *) NULL);
	return TCL_ERROR;
    }


    /* loop until fetch exhausted */

    if (Tcl_OraFetch(clientData, interp, 2, argv) == TCL_ERROR) {
	Tcl_AppendResult (interp, ": ", argv[0], ": orafetch failed", 
			  (char *) NULL);
	return TCL_ERROR;
    }

    while (fetch_ok) {

        /* split the result columns left in interp->result */
	if (Tcl_SplitList(interp,interp->result,&colsCnt,&colsPtr) == 
							    TCL_ERROR) {
	   Tcl_AppendResult (interp, argv[0], ": split columns failed", 
			      (char *) NULL);
	   return TCL_ERROR;
	}

        /* build eval string from literals and columns */
        Tcl_DStringInit(&evalStr);
	eval_ptr = "";
	for (inum=0, icol=0; inum < num_str; inum++) {
	    if (inum == colstr[icol].strnum) {
	        eval_ptr = Tcl_DStringAppend(&evalStr, " ", -1);
		if (colstr[icol].column == 0) {    /* col 0 is entire result */
		    eval_ptr = Tcl_DStringAppendElement(&evalStr,
							interp->result);
                } else {
		    if (colstr[icol].column > colsCnt) {/* another sanity chk*/
			Tcl_AppendResult (interp, argv[0], 
			       ": column sanity failed!",(char *) NULL);
	                Tcl_DStringFree(&evalStr); 	/* free eval string */
	                ckfree((char *) colsPtr); 	/* free split array */
			return TCL_ERROR;
		    }
		    eval_ptr = Tcl_DStringAppendElement(&evalStr,
					  colsPtr[colstr[icol].column - 1]);
		}
	        eval_ptr = Tcl_DStringAppend(&evalStr, " ", -1);
	        icol++;
	    } else {
	        eval_ptr = Tcl_DStringAppend(&evalStr, str[inum], -1);
	    }
	}

        tcl_rc = Tcl_Eval(interp, eval_ptr);	/* do it! */

	Tcl_DStringFree(&evalStr); 	/* free eval string */

	ckfree((char *) colsPtr); 	/* free split array */

	switch (tcl_rc) {		/* check on eval return code */
	  case TCL_ERROR:
	    Tcl_AppendResult (interp, ": eval failed while in ", argv[0], 
			      (char *) NULL);
	    return TCL_ERROR;
	    break;

	  case TCL_BREAK:		/* return sooner */
	    return TCL_OK;
	    break;

	  default:
	    break;
	}

	Tcl_ResetResult(interp);	/* reset interp result for next */
 
        /* next fetch */
        if (Tcl_OraFetch(clientData, interp, 2, argv) == TCL_ERROR) {
	    Tcl_AppendResult (interp, ": ", argv[0], ": orafetch failed", 
			      (char *) NULL);
	    return TCL_ERROR;
        }

    }

    return TCL_OK;

}




/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraCols --
 *    Implements the oracols command:
 *    usage: oracols cur_handle 
 *	                
 *    results:
 *	latest column names as tcl list
 *      also set oramsg(collengths) and oramsg(coltypes) as tcl list
 *      TCL_OK - handle is opened
 *      TCL_ERROR - wrong # args, or handle not opened,
 */

Tcl_OraCols (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;
    int     i = 0;
    ColBufs *col;
    char   *type_ptr;
    char    len_buf[ORA_BUFF_SIZE];
    char    typ_buf[ORA_BUFF_SIZE];
    char    prec_buf[ORA_BUFF_SIZE];
    char    scale_buf[ORA_BUFF_SIZE];
    char    buf2[ORA_MSG_SIZE];

    if ((hand = ora_prologue(interp, argc, argv, 2, " cur_handle")) == -1) {
	return TCL_ERROR;
    }


    len_buf[0]   = '\0';   /* initial null buffers */
    typ_buf[0]   = '\0';   
    prec_buf[0]  = '\0';
    scale_buf[0] = '\0';
    buf2[0]      = '\0';   

    for (col = OraCurs[hand].col_list; col != NULL; col = col->next_buf) {

        i++;

        sprintf(buf2, "%s", col->dbname);
	Tcl_AppendElement(interp,buf2);

        /* get the display length and append to "collengths" */
	/* remember that we added one to disp_size in parse, so subtract it */
	sprintf(buf2, (i>1) ? " %u" : "%u", (int) col->disp_size - 1);
	strcat(len_buf,buf2);

	/* get the column type and append to "coltypes" */
	switch (col->dbtype) {
	    case 1:
#ifdef VERSION7
		type_ptr = "varchar2";
#else
		type_ptr = "char";
#endif
		break;
	    case 2:
		type_ptr = "number";
		break;
	    case 8:
		type_ptr = "long";
		break;
	    case 11:
		type_ptr = "rowid";
		break;
	    case 12:
		type_ptr = "date";
		break;
	    case 23:
		type_ptr = "raw";
		break;
	    case 24:
		type_ptr = "long_raw";
		break;
	    case 96:
		type_ptr = "char";
		break;
	    case 105:
		type_ptr = "mlslabel";
		break;
	    case 106:
		type_ptr = "raw_mlslabel";
		break;
	    default:
		type_ptr = "unknown";
		break;
	}
	sprintf(buf2, (i>1) ? " %s" : "%s",type_ptr);
	strcat(typ_buf,buf2);

        /* format precision and scale for numeric data */
	if (col->dbtype == 2) {
	  sprintf(buf2, (i>1) ? " %d" : "%d", col->prec);
	} else {
	  sprintf(buf2, (i>1) ? " %s" : "%s", "{}");
	}
	strcat(prec_buf,buf2);

	if (col->dbtype == 2) {
	  sprintf(buf2, (i>1) ? " %d" : "%d", col->scale);
	} else {
	  sprintf(buf2, (i>1) ? " %s" : "%s", "{}");
	}
	strcat(scale_buf,buf2);


    }

    if (i > 0) {
        Tcl_SetVar2(interp, OraMsgArray, "collengths", len_buf,TCL_GLOBAL_ONLY);
        Tcl_SetVar2(interp, OraMsgArray, "coltypes",   typ_buf,TCL_GLOBAL_ONLY);
        Tcl_SetVar2(interp, OraMsgArray, "colprecs",  prec_buf,TCL_GLOBAL_ONLY);
        Tcl_SetVar2(interp, OraMsgArray, "colscales",scale_buf,TCL_GLOBAL_ONLY);
    }


    return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraCancel --
 *    Implements the oracancel command:
 *    usage: oracancel  cur_handle 
 *	                
 *    results:
 *	null string
 *      TCL_OK - handle is opened
 *      TCL_ERROR - wrong # args, or handle not opened,
 */

Tcl_OraCancel (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;


    if ((hand = ora_prologue(interp, argc, argv, 2, " cur_handle")) == -1) {
	return TCL_ERROR;
    }

    ocan(OraCurs[hand].cda);

    get_ora_err(interp, hand);

    free_cols(OraCurs[hand].col_list);
    OraCurs[hand].col_list   = NULL;
    OraCurs[hand].cache_size = ORA_CACHE;
    OraCurs[hand].row_num    = 0;
    OraCurs[hand].fetch_end  = 1;
    OraCurs[hand].fetch_cnt  = 0;


    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraPLexec --
 *    Implements the oraplexec command:
 *    usage: oraplexec cur_handle pl_block [ :varname value ]  [ .... ]
 *	                
 *    results:
 *	return parms in tcl list form
 *      TCL_OK - proc executed
 *      TCL_ERROR - wrong # args, or handle not opened,
 */

Tcl_OraPLexec (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{

    int         hand;
    int         parm_cnt;
    ColBufs    *new_col_head;
    ColBufs    *new_col;
    ColBufs    *last_col;

    /* prologue */
    if ((hand = ora_prologue(interp, argc, argv, 3, 
	 " cur_handle pl_block [ :varname value ] ...")) == -1) {
	return TCL_ERROR;
    }

    /* cancel any pending usage of cursor */
    ocan(OraCurs[hand].cda);

    /* free existing columns, if any */
    free_cols(OraCurs[hand].col_list);
    OraCurs[hand].col_list   = NULL;
    OraCurs[hand].fetch_end  = 1;
    OraCurs[hand].fetch_cnt  = 0;
    OraCurs[hand].cache_size = ORA_CACHE;
    OraCurs[hand].row_num    = ORA_CACHE;

    /* parse pl_block */
    oparse(OraCurs[hand].cda, argv[2], ((long) -1), DEF_FLAG, LNG_FLAG);

    if (OraCurs[hand].cda->rc != 0) {
	get_ora_err(interp, hand);
        Tcl_AppendResult(interp, argv[0], " oparse failed ",
	                (char *) NULL);
	return TCL_ERROR;
    }

    /* bind :colon_parms following arg, alloc large enough return space (?)*/
    parm_cnt = 3;      

    /* only alloc new col if one or more valid arg for :var */
    if (argc > parm_cnt + 1) {        
        new_col_head = alloc_col();
        if (new_col_head == NULL) {
            Tcl_AppendResult(interp, argv[0], " cannot malloc new col ",
			     (char *) NULL);
	    return TCL_ERROR;
        }
        new_col      = new_col_head;
        last_col     = new_col_head;
    } else {
	new_col_head = NULL;
    }

    while (argc > parm_cnt + 1) {       /* always in pairs of two */

	/* make sure bind variable name has a leading ':' */
	if (*argv[parm_cnt] != ':') {
	    free_cols(new_col_head);
	    OraCurs[hand].col_list = NULL;
	    Tcl_AppendResult(interp, argv[0], " bind variable ", argv[parm_cnt],
			     " does not begin with ':' ", (char *) NULL);
	    return TCL_ERROR;
	}

	new_col->disp_size  = ORA_BUFF_SIZE;
	new_col->col_data   = ckalloc(new_col->disp_size + 1);
	if (new_col->col_data == NULL) {
	    free_cols(new_col_head);
	    OraCurs[hand].col_list = NULL;
            Tcl_AppendResult(interp, argv[0], 
			     " cannot malloc new col data for ", argv[parm_cnt],
			     (char *) NULL);
	    return TCL_ERROR;
	} else {
	    strncpy(new_col->col_data, argv[parm_cnt + 1], ORA_BUFF_SIZE - 1);
	}

	new_col->rlen[0] = 0;
	new_col->rcode[0] = 0;

        /* if oracols is called after the exec, give it some info */
	new_col->dbtype = 1;   /* internal type char/varchar*/
	strncpy(new_col->dbname,argv[parm_cnt],(sizeof new_col->dbname) -1);


        obndrv( OraCurs[hand].cda, argv[parm_cnt], -1, new_col->col_data,
		ORA_BUFF_SIZE, EXT_STRING_TYPE, -1, NULL, NULL, -1, -1);
       

	if (OraCurs[hand].cda->rc != 0) {
	    get_ora_err(interp, hand);
	    free_cols(new_col_head);
	    OraCurs[hand].col_list = NULL;
            Tcl_AppendResult(interp, argv[0], " bind failed for ",
			     argv[parm_cnt], (char *) NULL);
	    return TCL_ERROR;
	}

	parm_cnt += 2;

        if (argc > parm_cnt + 1) {        /* more args? alloc new colbufs */
	    last_col           = new_col;
	    new_col            = alloc_col();
	    if (new_col == NULL) {
		free_cols(new_col_head);
	        OraCurs[hand].col_list = NULL;
		Tcl_AppendResult(interp, argv[0], " cannot malloc new col ",
				 (char *) NULL);
		return TCL_ERROR;
	    }
	    last_col->next_buf = new_col;
	}

    }
     
    OraCurs[hand].col_list  = new_col_head;
    OraCurs[hand].fetch_end = 0;        /* let append_cols work */
    OraCurs[hand].fetch_cnt = 0;        
    OraCurs[hand].row_num   = 0;

    /* exec cursor */
    oexec(OraCurs[hand].cda);

    get_ora_err(interp, hand);

    /* get results , same as orafetch */

    append_cols(interp, hand);
   
    /* set cursor data */
    OraCurs[hand].fetch_end  = 1;
    OraCurs[hand].fetch_cnt  = 0;
    OraCurs[hand].cache_size = ORA_CACHE;
    OraCurs[hand].row_num    = 0;

    /* don't free col_list yet; allow oracols to fetch info */

    return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraCommit --
 *    Implements the oracommit command:
 *    usage: oracommit lda_handle
 *	                
 *    results:
 *	null string
 *      TCL_OK - transactions commited
 *      TCL_ERROR - wrong # args, or handle not opened,
 */

Tcl_OraCommit (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;
    char    buf[ORA_MSG_SIZE];

    /* can't use ora_prologue, oracommit just uses an lda handle */

    clear_msg(interp);

    if (argc < 2) {
	Tcl_AppendResult (interp, "wrong # args: ", argv[0],
			  " lda_handle ", (char *) NULL);
	return TCL_ERROR;
    }

    if ((hand = get_ora_lda_handle(argv[1])) == -1) {
	Tcl_AppendResult (interp, argv[0], ": lda_handle ", argv[1],
			  " not valid ", (char *) NULL);
	return TCL_ERROR;
    }

    Tcl_SetVar2(interp, OraMsgArray, "handle",  argv[1], TCL_GLOBAL_ONLY);

    ocom(OraLdas[hand].lda); 

    get_lda_err(interp, hand);

    return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraRoll --
 *    Implements the oraroll command:
 *    usage: oraroll lda_handle
 *	                
 *    results:
 *	null string
 *      TCL_OK - transactions rolled back
 *      TCL_ERROR - wrong # args, or handle not opened,
 */

Tcl_OraRoll (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;
    char    buf[ORA_MSG_SIZE];

    /* can't use ora_prologue, oraroll just uses an lda handle */

    clear_msg(interp);

    if (argc < 2) {
	Tcl_AppendResult (interp, "wrong # args: ", argv[0],
			  " lda_handle ", (char *) NULL);
	return TCL_ERROR;
    }

    if ((hand = get_ora_lda_handle(argv[1])) == -1) {
	Tcl_AppendResult (interp, argv[0], ": lda_handle ", argv[1],
			  " not valid ", (char *) NULL);
	return TCL_ERROR;
    }

    Tcl_SetVar2(interp, OraMsgArray, "handle",  argv[1], TCL_GLOBAL_ONLY);

    orol(OraLdas[hand].lda); 

    get_lda_err(interp, hand);


    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraAutocom --
 *    Implements the oraautocom command:
 *    usage: oraroll lda_handle on|off
 *	                
 *    results:
 *	null string
 *      TCL_OK - auto commit set on or off
 *      TCL_ERROR - wrong # args, or handle not opened,
 */

Tcl_OraAutocom (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;
    char    buf[ORA_MSG_SIZE];

    /* can't use ora_prologue, oraautocom just uses an lda handle */

    clear_msg(interp);

    if (argc < 3) {
	Tcl_AppendResult (interp, "wrong # args: ", argv[0],
			  " lda_handle on|off ", (char *) NULL);
	return TCL_ERROR;
    }

    if ((hand = get_ora_lda_handle(argv[1])) == -1) {
	Tcl_AppendResult (interp, argv[0], ": lda_handle ", argv[1],
			  " not valid ", (char *) NULL);
	return TCL_ERROR;
    }


    Tcl_SetVar2(interp, OraMsgArray, "handle",  argv[1], TCL_GLOBAL_ONLY);

    if (strncasecmp(argv[2],"on",2) == 0) {   /* case insensitive compare */
        ocon(OraLdas[hand].lda); 
    } else {
        ocof(OraLdas[hand].lda); 
    }

    get_lda_err(interp, hand);

    return TCL_OK;
}




/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraWrlong --
 *    Implements the orawritelong command:
 *    usage: orawritelong cur_handle rowid table column file
 *      where: cur_handle = open cursor handle
 *             rowid  = rowid of row to retrieve
 *             table  = table name
 *             column = column name of long or long raw
 *             file   = file in which to get text 
 *
 *    orawritelong should be preceded by a select that returns a rowid
 *	                
 *    results:
 *	null string
 *      TCL_OK - handle is closed
 *      TCL_ERROR - wrong # args, or handle not opened, file not found,
 */


Tcl_OraWrlong (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;
    int     fd;
    long    filesize;
    char    buf[ORA_MSG_SIZE];
    char   *data_ptr;
    struct  stat stat_buf;
    ColBufs *new_col;

    if ((hand = ora_prologue(interp, argc, argv, 6, 
	    " cur_handle rowid table column filename ")) == -1) {
	return TCL_ERROR;
    }


    fd = open(argv[5], O_RDONLY, 0);
    if (fd < 0) {
        Tcl_AppendResult (interp, argv[0], ": file ", argv[5],
		     " cannot be opened for reading ", (char *) NULL);
        return TCL_ERROR;
    }

    fstat(fd, &stat_buf);
    filesize = stat_buf.st_size;

    if (filesize == 0) {	/* if zero, no need to do anything */
	close(fd);
        Tcl_SetResult (interp,"0",TCL_VOLATILE);
        return TCL_OK;
    }

    /* alloc memory for file, and read whole file */
    /* this sucks, oracle should have a "write long chunk" oci call */
    if ((data_ptr = ckalloc(filesize)) == NULL) {
	close(fd);
        Tcl_AppendResult (interp, " cannot malloc memory for file ", 
				 (char *) NULL);
        return TCL_ERROR;
    }

    if (filesize != read(fd,data_ptr,filesize)) {
	close(fd);
	ckfree(data_ptr);
        Tcl_AppendResult (interp, " cannot read file into memory ", 
				 (char *) NULL);
        return TCL_ERROR;
    }
    close(fd);

    /* cancel any pending usage of cursor */
    ocan(OraCurs[hand].cda);

    /* free existing columns, if any */
    free_cols(OraCurs[hand].col_list);
    OraCurs[hand].col_list   = NULL;
    OraCurs[hand].fetch_end  = 1;
    OraCurs[hand].fetch_end  = 0;
    OraCurs[hand].cache_size = ORA_CACHE;
    OraCurs[hand].row_num    = ORA_CACHE;

    /* compose select stmt */

    sprintf(buf,"select %s from %s where rowid = '%s'",argv[4],argv[3],argv[2]);

    /* parse sql stmt */
    oparse(OraCurs[hand].cda, buf, ((long) -1), DEF_FLAG, LNG_FLAG);

    if (OraCurs[hand].cda->rc != 0) {
	ckfree(data_ptr);
	get_ora_err(interp, hand);
        Tcl_AppendResult(interp, argv[0], " oparse failed for long select",
	                (char *) NULL);
	return TCL_ERROR;
    }

    /* get description of column so we know which datatype we're dealing with */

    new_col = alloc_col();
    if (new_col == NULL) {
	ckfree(data_ptr);
        Tcl_AppendResult(interp, argv[0], " cannot malloc new col ",
                         (char *) NULL);
        return TCL_ERROR;
    }

    new_col->disp_size  = 0;

    new_col->rlen[0] = 0;

    new_col->rcode[0] = 0;


    odescr(OraCurs[hand].cda, 1, &(new_col->dbsize),&(new_col->dbtype),
          &(new_col->dbname[0]), &(new_col->dbname_len), &(new_col->disp_size),
          &(new_col->prec), &(new_col->scale), &(new_col->nullok)   );


    if (OraCurs[hand].cda->rc != 0) {
	ckfree(data_ptr);
	free_cols(new_col);
	get_ora_err(interp, hand);
        Tcl_AppendResult(interp, argv[0], " odescr error ",
	                (char *) NULL);
	return TCL_ERROR;
    }


    /* compose update stmt */

    sprintf(buf,"update %s set %s = :1 where rowid = '%s' ",
		 argv[3],argv[4],argv[2]);

    /* parse sql stmt */
    oparse(OraCurs[hand].cda, buf, ((long) -1), DEF_FLAG, LNG_FLAG);

    if (OraCurs[hand].cda->rc != 0) {
	ckfree(data_ptr);
	free_cols(new_col);
	get_ora_err(interp, hand);
        Tcl_AppendResult(interp, argv[0], " oparse failed for long update",
	                (char *) NULL);
	return TCL_ERROR;
    }

    /* bind filearea to column */

    obndrn( OraCurs[hand].cda, 1, data_ptr,
		filesize, new_col->dbtype, -1, NULL, NULL, -1, -1);
       
    /* exec stmt */

    oexec(OraCurs[hand].cda);
    if (OraCurs[hand].cda->rc != 0) {  /* oops, problem */
	ckfree(data_ptr);
	free_cols(new_col);
	get_ora_err(interp, hand);
        Tcl_AppendResult(interp, argv[0], " oexec error ",
	                (char *) NULL);
	return TCL_ERROR;
    }
    if (OraCurs[hand].cda->rpc != 1) {  /* not found, row process != 1 */
	ckfree(data_ptr);
	free_cols(new_col);
	get_ora_err(interp, hand);
        Tcl_AppendResult(interp, argv[0], " row not found ",
	                (char *) NULL);
	return TCL_ERROR;
    }

    /* free mem */
    ckfree(data_ptr);
    free_cols(new_col);

    get_ora_err(interp, hand);

    /* return total bytes sent */
    sprintf(buf,"%d",filesize);
    Tcl_SetResult (interp,buf,TCL_VOLATILE);

    return TCL_OK;

}




/*
 *----------------------------------------------------------------------
 *
 * Tcl_OraRdlong --
 *    Implements the orareadlong command:
 *    usage: orareadlong cur_handle rowid table column file 
 *
 *    results:
 *	null string
 *      TCL_OK - handle is closed
 *      TCL_ERROR - wrong # args, or handle not opened, can't open file,
 *                  or other error in text/image handling
 */


Tcl_OraRdlong (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;
    int     fd;
    int     s;
    int     total_bytes = 0;
    char    buf[ORA_MSG_SIZE];
    char    buf2[ORA_MSG_SIZE];
    char   *data_ptr;
    long    long_size = MAX_LONG_SIZE;
    unsigned long bytes_fetched = 0;
    long     offset = 0;
    ColBufs *new_col;
    unsigned short   fsize;
    short   fcode;
    int     rc;


    if ((hand = ora_prologue(interp, argc, argv, 6, 
	    " cur_handle rowid table column filename")) == -1) {
	return TCL_ERROR;
    }


    fd = open(argv[5], O_WRONLY | O_TRUNC | O_CREAT, 0644);
    if (fd == -1) {
        Tcl_AppendResult (interp, argv[0], ": file ", argv[5],
		     " could not be opened for writing ", (char *) NULL);
        return TCL_ERROR;
    }

    /* compose select stmt */

    sprintf(buf,"select %s from %s where rowid = '%s'",argv[4],argv[3],argv[2]);

    /* parse sql stmt */
    oparse(OraCurs[hand].cda, buf, ((long) -1), DEF_FLAG, LNG_FLAG);

    if (OraCurs[hand].cda->rc != 0) {
	close(fd);
	get_ora_err(interp, hand);
        Tcl_AppendResult(interp, argv[0], " oparse failed for select long",
	                (char *) NULL);
	return TCL_ERROR;
    }

    /* get description of column so we know which datatype we're dealing with */

    new_col = alloc_col();
    if (new_col == NULL) {
	close(fd);
        Tcl_AppendResult(interp, argv[0], " cannot malloc new col ",
                         (char *) NULL);
        return TCL_ERROR;
    }

    new_col->rlen[0] = 0;

    new_col->rcode[0] = 0;

    odescr(OraCurs[hand].cda, 1, &(new_col->dbsize),&(new_col->dbtype),
          &(new_col->dbname[0]), &(new_col->dbname_len), &(new_col->disp_size),
          &(new_col->prec), &(new_col->scale), &(new_col->nullok)   );

    if (OraCurs[hand].cda->rc != 0) {
	close(fd);
	free_cols(new_col);
	get_ora_err(interp, hand);
        Tcl_AppendResult(interp, argv[0], " odescr error ",
	                (char *) NULL);
	return TCL_ERROR;
    }

    /* define a one byte fetch buffer now, dbsize is erroneous until */
    /* first fetch is done, then do the whole process again */
    odefin(OraCurs[hand].cda, 1, buf2, 
	       (int) 1, new_col->dbtype, 0, (short *) 0, 
	       NULL, 0, 0, new_col->rlen, new_col->rcode);

    /* exec and fetch to find out actual length of column */
    oexec(OraCurs[hand].cda);

    ofetch(OraCurs[hand].cda);

    if (OraCurs[hand].cda->rc == NO_DATA_FOUND) {  /* not found */
	close(fd);
	free_cols(new_col);
	get_ora_err(interp, hand);
        Tcl_AppendResult(interp, argv[0], " row not found for long",
	                (char *) NULL);
	return TCL_ERROR;
    }

    if (OraCurs[hand].cda->rc != COL_TRUNCATED &&
                  OraCurs[hand].cda->rc != 0) {          /* problem */
	close(fd);
	free_cols(new_col);
	get_ora_err(interp, hand);
        Tcl_AppendResult(interp, argv[0], " oexec/ofetch failed",
	                (char *) NULL);
	return TCL_ERROR;
    }

#ifdef VERSION7
    
    /* at this point, the version 7 oflng() can be used since we */
    /* have retrieved the long column */

    if ((data_ptr = ckalloc(LONG_BUFF_SIZE)) == NULL) {
	close(fd);
	free_cols(new_col);
        Tcl_AppendResult (interp, " cannot malloc memory for column fetch", 
				 (char *) NULL);
        return TCL_ERROR;
    }

    rc = oflng(OraCurs[hand].cda, 1, data_ptr, LONG_BUFF_SIZE, 
	       new_col->dbtype, &bytes_fetched, offset);	
    
    while (bytes_fetched > 0 && rc == 0) {

      write(fd, data_ptr, bytes_fetched);

      offset += bytes_fetched;

      rc = oflng(OraCurs[hand].cda, 1, data_ptr, LONG_BUFF_SIZE, 
	         new_col->dbtype, &bytes_fetched, offset);	
    }

    /* save total number of bytes fetched */
    long_size = offset;

    /* end of v7 specific code */
    
#else 

    /* continue with v6 code */


    /* describe again to find actual size of long column returned */

    odsc(OraCurs[hand].cda, 1, &(new_col->dbsize),
	 &fsize, &fcode, &(new_col->dbtype),
         &(new_col->dbname[0]), &(new_col->dbname_len), &(new_col->disp_size));

    /* malloc an area to hold the column */
    long_size = fsize;

    if ((data_ptr = ckalloc(long_size)) == NULL) {
	close(fd);
	free_cols(new_col);
        Tcl_AppendResult (interp, " cannot malloc memory for column fetch", 
				 (char *) NULL);
        return TCL_ERROR;
    }

    /* now defin, exec, fetch again */

    odefin(OraCurs[hand].cda, 1, data_ptr, 
	       (int) long_size, new_col->dbtype, 0, (short *) 0,
	       NULL, 0, 0, new_col->rlen, new_col->rcode);

    oexec(OraCurs[hand].cda);

    ofetch(OraCurs[hand].cda);
	
    if (long_size != write(fd,data_ptr,long_size)) {
	ckfree(data_ptr);
	close(fd);
	free_cols(new_col);
	get_ora_err(interp, hand);
        Tcl_AppendResult(interp, argv[0], " could not write data ",
	                (char *) NULL);
	return TCL_ERROR;
    }

    /* end of version 6 code */

#endif

    /* common cleanup & return code */

    close(fd);
    ckfree(data_ptr);
    free_cols(new_col);

    /* return total bytes sent */
    sprintf(buf,"%ld",long_size);
    Tcl_SetResult (interp,buf,TCL_VOLATILE);

    return TCL_OK;

}


/* finis */
