/* 
 * tkCanvVport.c --
 *
 *	This file implements 3-D viewport items for canvas widgets.
 *
 * Adapted from tkCanvPoly.c by Paul Mackerras (paulus@cs.anu.edu.au).
 *
 * Copyright 1992 The Australian National University.
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, provided that the above copyright
 * notice appears in all copies.  This software is provided without any
 * warranty, express or implied. The Australian National University
 * makes no representations about the suitability of this software for
 * any purpose.
 *
 * Copyright 1991-1992 Regents of the University of California.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#ifndef lint
static char rcsid[] = "$Header: /localdisks/kaffa/cap/CaVis/src/tk2.3/RCS/tkCanvVport.c,v 1.2 1992/09/17 01:34:58 paulus Exp paulus $";
#endif

#include <stdio.h>
#include <math.h>
#include "tkInt.h"
#include "tkCanvas.h"
#include "tkConfig.h"
#include "tkCamera.h"
#include <stdlib.h>

/*
 * The structure below defines the record for each viewport item.
 */

typedef struct ViewportItem  {
    Tk_Item	header;		/* Generic stuff that's the same for all
				 * types.  MUST BE FIRST IN STRUCTURE. */
    Tk_3DItem	items;		/* Head of list of items in this viewport. */
    double	x1, y1;		/* upper-left corner of viewport,
				 * in canvas coordinates */
    double	x2, y2;		/* lower-right corner of viewport */

    Camera	camera;		/* definition of 3D view point */
} ViewportItem;

/*
 * Information used for parsing configuration specs:
 */

int		ParseVector _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, Tk_Window tkwin, char *value,
			    char *widgRec, int offset));
char *		PrintVector _ANSI_ARGS_((ClientData clientData,
			    Tk_Window tkwin, char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));

static Tk_CustomOption vectorOption = {
    ParseVector,
    PrintVector,
    NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_DOUBLE, "-angle", NULL, NULL, "30",
	 Tk_Offset(ViewportItem, camera.view_angle), 0},
    {TK_CONFIG_CUSTOM, "-at", NULL, NULL, "0 0 0",
	 Tk_Offset(ViewportItem, camera.at[0]), 0, &vectorOption},
    /* N.B. [0] is to shut the stupid compiler up */
    {TK_CONFIG_DOUBLE, "-distance", NULL, NULL, "4",
	 Tk_Offset(ViewportItem, camera.view_distance), 0},
    {TK_CONFIG_BOOLEAN, "-perspective", NULL, NULL, "true",
	 Tk_Offset(ViewportItem, camera.perspective), 0},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tkCanvasTagsOption},
    {TK_CONFIG_CUSTOM, "-up", NULL, NULL, "0 1 0",
	 Tk_Offset(ViewportItem, camera.up[0]), 0, &vectorOption},
    {TK_CONFIG_CUSTOM, "-view", NULL, NULL, "0 0 1",
	 Tk_Offset(ViewportItem, camera.view[0]), 0, &vectorOption},
    {TK_CONFIG_DOUBLE, "-zmin", NULL, NULL, "0",
	 Tk_Offset(ViewportItem, camera.zmin), 0},
    {TK_CONFIG_DOUBLE, "-zmax", NULL, NULL, "0",
	 Tk_Offset(ViewportItem, camera.zmax), 0},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

/*
 * Prototypes for procedures defined in this file:
 */

static void		CalculateTransform _ANSI_ARGS_((
			    ViewportItem *vportPtr));
static void		ComputeViewportBbox _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    ViewportItem *vportPtr));
static int		ConfigureViewport _ANSI_ARGS_((
			    Tk_Canvas *canvasPtr, Tk_Item *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateViewport _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    struct Tk_Item *itemPtr, int argc, char **argv));
static void		Delete3DItem _ANSI_ARGS_((Tk_Item *item));
static void		DeleteViewport _ANSI_ARGS_((Tk_Item *itemPtr));
static void		DisplayViewport _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, Drawable dst));
static int		ViewportCoords _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, int argc, char **argv));
static int		ViewportToArea _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *rectPtr));
static double		ViewportToPoint _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double *pointPtr));
static int		VportCommands _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, int argc, char **argv));
static void		ScaleViewport _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
static void		TranslateViewport _ANSI_ARGS_((Tk_Canvas *canvasPtr,
			    Tk_Item *itemPtr, double deltaX, double deltaY));

/*
 * The structures below defines the viewport item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_ItemType TkViewportType = {
    "viewport",				/* name */
    sizeof(ViewportItem),		/* itemSize */
    CreateViewport,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureViewport,			/* configureProc */
    ViewportCoords,			/* coordProc */
    DeleteViewport,			/* deleteProc */
    DisplayViewport,			/* displayProc */
    0,					/* alwaysRedraw */
    ViewportToPoint,			/* pointProc */
    ViewportToArea,			/* areaProc */
    (Tk_ItemPostscriptProc *) NULL,	/* postscriptProc */
    ScaleViewport,			/* scaleProc */
    TranslateViewport,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* cursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,		/* insertProc */
    (Tk_ItemDCharsProc *) NULL,		/* dTextProc */
    (Tk_ItemType *) NULL		/* nextPtr */
};

void viewportinstall() {
	Tk_CreateItemType(&TkViewportType);
	}


/*
 * Static data, shared by all viewports.
 */

static int nextUid;

#define PI 3.14159265358979323846

/*
 * Structure used to access the Tk_3DItem header in
 * an arbitrary 3-D item's data structure.
 */

typedef struct {
    Tk_Item	header;
    Tk_3DItem	hdr3d;
} Basic3DItem;

#define HDR3D(item)	(((Basic3DItem *)(item))->hdr3d)

/*
 *--------------------------------------------------------------
 *
 * CreateViewport --
 *
 *	This procedure is invoked to create a new viewport item in
 *	a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	canvasPtr->interp->result;  in this case itemPtr is
 *	left uninitialized, so it can be safely freed by the
 *	caller.
 *
 * Side effects:
 *	A new viewport item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateViewport(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing viewport. */
{
    register ViewportItem *vportPtr = (ViewportItem *) itemPtr;
    char *cname, *pref, suff[24];

    if (argc < 4) {
	Tcl_AppendResult(canvasPtr->interp, "wrong # args:  should be \"",
		Tk_PathName(canvasPtr->tkwin),
		" create x1 y1 x2 y2 ?options?\"",
		(char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Carry out initialization that is needed in order to clean
     * up after errors during the the remainder of this procedure.
     */

    vportPtr->items.next = vportPtr->items.prev = itemPtr;

    /*
     * Parse the canvas coords defining the extent of the viewport,
     * and process the configuration options given.
     */

    if( ViewportCoords(canvasPtr, itemPtr, 4, argv) != TCL_OK
       || ConfigureViewport(canvasPtr, itemPtr, argc-4, argv+4, 0) != TCL_OK ){
	DeleteViewport(itemPtr);
	return TCL_ERROR;
    }

    /*
     * Create a command for accessing the camera information,
     * etc., more directly.
     */

    sprintf(suff, "-%d", itemPtr->id);
    pref = Tk_PathName(canvasPtr->tkwin);
    cname = ckalloc(strlen(pref) + strlen(suff) + 1);
    strcpy(cname, pref);
    strcat(cname, suff);
    Tcl_CreateCommand(canvasPtr->interp, cname, VportCommands,
		      (ClientData) itemPtr, (void (*)()) NULL);

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ViewportCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on viewports.  See the user documentation for details
 *	on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets canvasPtr->interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
ViewportCoords(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
    register ViewportItem *vportPtr = (ViewportItem *) itemPtr;
    double t;
    char buffer[128];

    if( argc == 0 ){
	sprintf(buffer, "%g %g %g %g", vportPtr->x1, vportPtr->y1,
		vportPtr->x2, vportPtr->y2);
	Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);

    } else if( argc == 4 ){
	if (TkGetCanvasCoord(canvasPtr, argv[0], &vportPtr->x1) != TCL_OK
	    || TkGetCanvasCoord(canvasPtr, argv[1], &vportPtr->y1) != TCL_OK
	    || TkGetCanvasCoord(canvasPtr, argv[2], &vportPtr->x2) != TCL_OK
	    || TkGetCanvasCoord(canvasPtr, argv[3], &vportPtr->y2) != TCL_OK)
	    return TCL_ERROR;

	/*
	 * Make sure our viewport is the right way around.
	 */
	if( vportPtr->x1 > vportPtr->x2 ){
	    t = vportPtr->x1;
	    vportPtr->x1 = vportPtr->x2;
	    vportPtr->x2 = t;
	}
	if( vportPtr->y1 > vportPtr->y2 ){
	    t = vportPtr->y1;
	    vportPtr->y1 = vportPtr->y2;
	    vportPtr->y2 = t;
	}

	ComputeViewportBbox(canvasPtr, vportPtr);

    } else {
	Tcl_AppendResult(canvasPtr->interp, "wrong number of coordinates",
			 " for viewport: must have 4", (char *) NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureViewport --
 *
 *	This procedure is invoked to configure various aspects
 *	of a viewport item such as the at, view, up vectors, etc.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in canvasPtr->interp->result.
 *
 * Side effects:
 *	Configuration information, such as the view
 *	transformation, may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureViewport(canvasPtr, itemPtr, argc, argv, flags)
    Tk_Canvas *canvasPtr;	/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Viewport item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    register ViewportItem *vportPtr = (ViewportItem *) itemPtr;
    XGCValues gcValues;
    GC newGC;
    unsigned long mask;

    if (Tk_ConfigureWidget(canvasPtr->interp, canvasPtr->tkwin,
	    configSpecs, argc, argv, (char *) vportPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * Re-calculate the transformation matrix etc.
     */

    CalculateTransform(vportPtr);

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteViewport --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a viewport item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteViewport(itemPtr)
    Tk_Item *itemPtr;			/* Item that is being deleted. */
{
    register ViewportItem *vportPtr = (ViewportItem *) itemPtr;

    /*
    while( vportPtr->items.next != itemPtr )
	Delete3DItem(vportPtr->items.next);
     */
}

/*
 *--------------------------------------------------------------
 *
 * ComputeViewportBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a viewport.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

static void
ComputeViewportBbox(canvasPtr, vportPtr)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    ViewportItem *vportPtr;		/* Item whose bbox is to be updated */
{

    vportPtr->header.x1 = (int) vportPtr->x1;
    vportPtr->header.x2 = (int) vportPtr->x2 + 1;
    vportPtr->header.y1 = (int) vportPtr->y1;
    vportPtr->header.y2 = (int) vportPtr->y2 + 1;

    /*
     * Add one more pixel of fudge factor just to be safe (e.g.
     * X may round differently than we do).
     */

    vportPtr->header.x1 -= 1;
    vportPtr->header.x2 += 1;
    vportPtr->header.y1 -= 1;
    vportPtr->header.y2 += 1;
}

/*
 *--------------------------------------------------------------
 *
 * DisplayViewport --
 *
 *	This procedure is invoked to draw a viewport item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None: a viewport is not itself directly visible
 *	(at least at present).
 *
 *--------------------------------------------------------------
 */

static void
DisplayViewport(canvasPtr, itemPtr, drawable)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
{
}

/*
 *--------------------------------------------------------------
 *
 * ViewportToPoint --
 *
 *	Computes the distance from a given point to a given
 *	viewport, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are pointPtr[0] and pointPtr[1] is inside the viewport.
 *	If the point isn't inside the viewport then the return value
 *	is the distance from the point to the viewport.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
ViewportToPoint(canvasPtr, itemPtr, pointPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to coordinates of point. */
{
    ViewportItem *vportPtr = (ViewportItem *) itemPtr;
    double dx, dy;

    if( pointPtr[0] < vportPtr->x1 )
	dx = vportPtr->x1 - pointPtr[0];
    else if( pointPtr[0] < vportPtr->x2 )
	dx = 0;
    else
	dx = pointPtr[0] - vportPtr->x2;

    if( pointPtr[1] < vportPtr->y1 )
	dy = vportPtr->y1 - pointPtr[1];
    else if( pointPtr[1] < vportPtr->y2 )
	dy = 0;
    else
	dy = pointPtr[1] - vportPtr->y2;

    return hypot(dx, dy);
}

/*
 *--------------------------------------------------------------
 *
 * ViewportToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangular area.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the
 *	area, 0 if it overlaps, and 1 if it is entirely
 *	insde the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
ViewportToArea(canvasPtr, itemPtr, rectPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against viewport. */
    double *rectPtr;		/* Pointer to array of four coordinates
				 * (x1, y1, x2, y2) describing rectangular
				 * area.  */
{
    ViewportItem *vportPtr = (ViewportItem *) itemPtr;

    if( rectPtr[0] >= vportPtr->x2 || rectPtr[2] < vportPtr->x1
       || rectPtr[1] >= vportPtr->y2 || rectPtr[3] < vportPtr->y1 )
	return -1;		/* viewport is entirely outside */

    if( rectPtr[0] <= vportPtr->x1 && rectPtr[2] >= vportPtr->x2
       && rectPtr[1] <= vportPtr->y1 && rectPtr[3] >= vportPtr->y2 )
	return 1;		/* entirely inside */

    return 0;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleViewport --
 *
 *	This procedure is invoked to rescale a viewport item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The viewport referred to by itemPtr is rescaled so that the
 *	following transformation is applied to all point
 *	coordinates:
 *		x' = originX + scaleX*(x-originX)
 *		y' = originY + scaleY*(y-originY)
 *
 *--------------------------------------------------------------
 */

static void
ScaleViewport(canvasPtr, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas *canvasPtr;		/* Canvas containing viewport. */
    Tk_Item *itemPtr;			/* Viewport to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
    ViewportItem *vportPtr = (ViewportItem *) itemPtr;

    vportPtr->x1 = originX + scaleX * (vportPtr->x1 - originX);
    vportPtr->y1 = originY + scaleY * (vportPtr->y1 - originY);
    vportPtr->x2 = originX + scaleX * (vportPtr->x2 - originX);
    vportPtr->y2 = originY + scaleY * (vportPtr->y2 - originY);
    CalculateTransform(vportPtr);
    ComputeViewportBbox(canvasPtr, vportPtr);
}

/*
 *--------------------------------------------------------------
 *
 * TranslateViewport --
 *
 *	This procedure is called to move a viewport by a given
 *	amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the viewport is offset by (xDelta, yDelta),
 *	and the bounding box is updated in the generic part of the
 *	item structure.
 *
 *--------------------------------------------------------------
 */

static void
TranslateViewport(canvasPtr, itemPtr, deltaX, deltaY)
    Tk_Canvas *canvasPtr;		/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    ViewportItem *vportPtr = (ViewportItem *) itemPtr;

    vportPtr->x1 += deltaX;
    vportPtr->y1 += deltaY;
    vportPtr->x2 += deltaX;
    vportPtr->y2 += deltaY;
    CalculateTransform(vportPtr);
    ComputeViewportBbox(canvasPtr, vportPtr);
}

/*
 *--------------------------------------------------------------
 *
 * CalculateTransform --
 *
 *	This procedure is called to calculate the transform
 *	matrix and other values in a Camera structure.
 *
 * Results:
 *	None.
 *	
 * Side effects:
 *	Various fields in vportPtr->camera are updated.
 *
 *--------------------------------------------------------------
 */

/* First some macros for vector maths. */
#define Dot(a, b)	((a)[0] * (b)[0] + (a)[1] * (b)[1] + (a)[2] * (b)[2])
#define VecLength(a)	(sqrt(Dot((a), (a))))
#define VecCopy(d, s)	((d)[0] = (s)[0], (d)[1] = (s)[1], (d)[2] = (s)[2])
#define VecScale(d, s, v)	\
		((d)[0] = (s)[0] * (v),		\
		 (d)[1] = (s)[1] * (v),		\
		 (d)[2] = (s)[2] * (v))
#define VecAdd(d, a, b)		\
		((d)[0] = (a)[0] + (b)[0],	\
		 (d)[1] = (a)[1] + (b)[1],	\
		 (d)[2] = (a)[2] + (b)[2])
#define VecAddS(d, a, b, s)		\
		((d)[0] = (a)[0] + (b)[0] * (s),	\
		 (d)[1] = (a)[1] + (b)[1] * (s),	\
		 (d)[2] = (a)[2] + (b)[2] * (s))
#define VecSub(d, a, b)		\
		((d)[0] = (a)[0] - (b)[0],	\
		 (d)[1] = (a)[1] - (b)[1],	\
		 (d)[2] = (a)[2] - (b)[2])
#define VecSubS(d, a, b, s)		\
		((d)[0] = (a)[0] - (b)[0] * (s),	\
		 (d)[1] = (a)[1] - (b)[1] * (s),	\
		 (d)[2] = (a)[2] - (b)[2] * (s))
#define VecCross(d, a, b)	\
		((d)[0] = (a)[1] * (b)[2] - (a)[2] * (b)[1],	\
		 (d)[1] = (a)[2] * (b)[0] - (a)[0] * (b)[2],	\
		 (d)[2] = (a)[0] * (b)[1] - (a)[1] * (b)[0])
#define RotateVector(d, m, s)	\
		((d)[0] = Dot((m)[0], (s)),	\
		 (d)[1] = Dot((m)[1], (s)),	\
		 (d)[2] = Dot((m)[2], (s)))

static void
CalculateTransform(vportPtr)
    ViewportItem *vportPtr;
{
    register Camera *cp;
    Tk_Item *item;
    int i;
    double l, xpixels, ypixels, minpix, vmax, xyscale;
    vector u, v, w;

    cp = &vportPtr->camera;

    /*
     * First compute the u, v, w vectors (up, view, right directions)
     */
    l = VecLength(cp->view);
    if( l == 0 ){
	cp->view[2] = 1;	/* default to view = z */
	l = 1;
    }
    VecScale(v, cp->view, 1/l);
    l = Dot(cp->up, v);
    VecSubS(u, cp->up, v, l);
    l = VecLength(u);
    if( l == 0 ){
	/* degenerate up vector - now what?? */
	/* make an arbitrary perpendicular vector */
	if( fabs(v[2]) < 0.5 ){
	    u[0] = v[1];
	    u[1] = -v[0];
	    u[2] = 0;
	} else {
	    u[0] = v[2];
	    u[1] = 0;
	    u[2] = -v[0];
	}
	l = VecLength(u);
    }
    VecScale(u, u, 1/l);
    VecCross(w, u, v);

    /*
     * Compute the rotation matrix to rotate
     * w, u, v to x, -y, -z respectively.
     * Include a translation part so `at' gets transformed
     * to view_distance * z.
     */
    for( i = 0; i < 3; ++i ){
	cp->xmat[0][i] = w[i];
	cp->xmat[1][i] = - u[i];
	cp->xmat[2][i] = - v[i];
    }
    cp->xmat[0][3] = - Dot(cp->xmat[0], cp->at);
    cp->xmat[1][3] = - Dot(cp->xmat[1], cp->at);
    cp->xmat[2][3] = cp->view_distance - Dot(cp->xmat[2], cp->at);

    /*
     * Scale factor to get us to pixels in one transformation.
     */
    if( cp->view_angle <= 0 )
	cp->view_angle = 1e-6;
    xpixels = vportPtr->x2 - vportPtr->x1;
    ypixels = vportPtr->y2 - vportPtr->y1;
    minpix = xpixels < ypixels? xpixels: ypixels;
    if( minpix < 1 )
	minpix = 1;
    vmax = tan(cp->view_angle * PI / 360);
    xyscale = minpix / (2 * vmax);
    cp->xyscale = xyscale;
    for( i = 0; i < 4; ++i ){
	cp->xmat[0][i] *= xyscale;
	cp->xmat[1][i] *= xyscale;
    }

    /*
     * Work out the slopes of the sides of the view pyramid
     * and the canvas coords of the center of the viewport.
     */
    cp->xmax = xpixels / 2;
    cp->ymax = ypixels / 2;
    cp->xcenter = (vportPtr->x1 + vportPtr->x2) / 2;
    cp->ycenter = (vportPtr->y1 + vportPtr->y2) / 2;

    /*
     * Update the ID of this camera, so clients will know they
     * need to re-transform.
     */
    cp->uid = ++nextUid;

    /*
     * Set all of our clients' bounding boxes to the bounding box
     * of the whole viewport.  This is a conservative assumption
     * until the individual widgets get invoked to update their
     * bounding boxes more specifically.
     */
    for( item = vportPtr->items.next; item != (Tk_Item *) vportPtr;
	 item = HDR3D(item).next ){
	item->x1 = vportPtr->header.x1;
	item->y1 = vportPtr->header.y1;
	item->x2 = vportPtr->header.x2;
	item->y2 = vportPtr->header.y2;
    }

}


/*
 *--------------------------------------------------------------
 *
 * Tk_AttachToViewport --
 *
 *	This procedure is called to specify that a given 3-D item
 *	should be rendered in the viewport (identified by its 
 *	canvas and ID number).
 *
 * Results:
 *	A standard Tcl return value.
 *	
 * Side effects:
 *	The Tk_3DItem header in the item's data structure
 *	is updated, unless the given ID doesn't correspond
 *	to a viewport item, in which case an error message is
 *	left in interp->result.
 *
 *--------------------------------------------------------------
 */

/* A little id -> viewport cache */
static int lastID = -1;
static ViewportItem *lastVport;

int
Tk_AttachToViewport(interp, canvas, vport_id, item)
    Tcl_Interp *interp;
    Tk_Canvas *canvas;
    int vport_id;
    Tk_Item *item;
{
    Tk_3DItem *item3d;
    ViewportItem *vport;
    Tk_Item *ip;
    char ascid[16];

    if( vport_id < 0 )
	vport = NULL;
    if( vport_id == lastID )
	vport = lastVport;
    else {
	/* look up the viewport by its item number. */
	for( ip = canvas->firstItemPtr; ip != NULL; ip = ip->nextPtr )
	    if( ip->id == vport_id )
		break;
	if( ip == NULL ){
	    sprintf(ascid, "%d", vport_id);
	    Tcl_AppendResult(interp, "item \"", ascid, "\" doesn't exist",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	if( ip->typePtr != &TkViewportType ){
	    sprintf(ascid, "%d", vport_id);
	    Tcl_AppendResult(interp, "item \"", ascid, "\" is not a viewport",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	vport = (ViewportItem *) ip;
	lastID = vport_id;
	lastVport = vport;
    }

    item3d = & HDR3D(item);
    if( item3d->next != NULL ){
	if( vport != NULL && item3d->camera == &vport->camera )
	    return TCL_OK;
	Delete3DItem(item);	/* remove from existing list */
    }

    if( vport != NULL ){
	/* link into list for the viewport */
	item3d->next = vport->items.next;
	item3d->prev = (Tk_Item *) vport;
	HDR3D(item3d->next).prev = item;
	vport->items.next = item;
	
	/* set the item's viewport field */
	item3d->camera = &vport->camera;
    }

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Delete3DItem --
 *
 *	This procedure is called to remove a 3D item from the
 *	list for a viewport.
 *
 * Results:
 *	None.
 *	
 * Side effects:
 *	The Tk_3DItem header in the item's data structure
 *	is updated.
 *
 *--------------------------------------------------------------
 */

static void
Delete3DItem(item)
    Tk_Item *item;
{
    Tk_3DItem *ip3;

    ip3 = & HDR3D(item);
    if( ip3->next != NULL ){
	HDR3D(ip3->prev).next = ip3->next;
	HDR3D(ip3->next).prev = ip3->prev;
	ip3->next = NULL;
	ip3->camera = NULL;
    }
}

/*
 *--------------------------------------------------------------
 *
 * ParseVector --
 *
 *	This procedure is called to parse a vector-valued
 *	configuration option.
 *
 * Results:
 *	A standard Tcl result, indicating the success or failure
 *	of the conversion.
 *	
 * Side effects:
 *	If the conversion succeeds, a vector value (3 doubles)
 *	is placed at the address given by widgRec + offset.
 *	If the conversion fails, an error message is left in
 *	interp->result.
 *
 *--------------------------------------------------------------
 */

int
ParseVector(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;
    Tcl_Interp *interp;
    Tk_Window tkwin;
    char *value;
    char *widgRec;
    int offset;
{
    double *cPtr, v[3];
    char *p, *q;
    int i;

    p = value;
    for( i = 0; i < 3; ++i ){
	v[i] = strtod(p, &q);
	if( q == p ){
	    Tcl_AppendResult(interp, "can't parse \"", value, "\" as a vector",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	p = q;
    }

    for( ; *p != 0; ++p )
	if( !isspace(*p) ){
	    Tcl_AppendResult(interp, "garbage at end of vector value \"",
			     value, "\"", (char *) NULL);
	    return TCL_ERROR;
	}

    cPtr = (double *) (widgRec + offset);
    for( i = 0; i < 3; ++i )
	*cPtr++ = v[i];
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * PrintVector --
 *
 *	This procedure is called to produce a string representation
 *	for a vector.
 *
 * Results:
 *	This procedure returns a pointer to a static string
 *	containing the string representation.
 *	
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

char *
PrintVector(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;
    Tk_Window tkwin;
    char *widgRec;
    int offset;
    Tcl_FreeProc **freeProcPtr;
{
    double *cPtr;
    static char str[72];

    cPtr = (double *) (widgRec + offset);
    sprintf(str, "%g %g %g", cPtr[0], cPtr[1], cPtr[2]);
    return str;
}

/*
 *--------------------------------------------------------------
 *
 * VportCommands --
 *
 *	This procedure implements the widget command which is
 *	created for each viewport.
 *
 * Results:
 *	A standard Tcl result.
 *	
 * Side effects:
 *	Depend on the command.
 *
 *--------------------------------------------------------------
 */
static int
VportCommands(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    ViewportItem *vport;
    char *option;
    int j, c, length, result;
    int x, y;
    vector wp, sp, from, to;
    double dx, dy, dz, rad, t;
    char str[72];
    double mat[3][3];

    vport = (ViewportItem *) clientData;

    if( argc < 2 ){
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " option ?arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }

    option = argv[1];
    c = option[0];
    length = strlen(option);
    result = TCL_OK;

    if( c == 'p' && strncmp(option, "project", length) == 0 ){
	/* report the projection of a world coord point */
	if( argc != 3 ){
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
			     argv[0], " project {x y z}\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if( ParseVector(NULL, interp, NULL, argv[2], wp, 0) != TCL_OK )
	    return TCL_ERROR;
	for( j = 0; j < 3; ++j )
	    sp[j] = vport->camera.xmat[j][3] + Dot(vport->camera.xmat[j], wp);
	dz = vport->camera.perspective? sp[2]: vport->camera.view_distance;
	x = (int)(sp[0] / dz + vport->camera.xcenter + 0.5);
	y = (int)(sp[1] / dz + vport->camera.ycenter + 0.5);
	sprintf(str, "%d %d %g", x, y, sp[2]);
	Tcl_AppendResult(interp, str, (char *) NULL);

    } else if( c == 'r' && strncmp(option, "roll", length) == 0 ){
	/* canv-vport roll from to
	   returns a string giving the value of the view and up vectors
	   rotated by the rotation from FROM to TO (wrt `at' point). */
	if( argc != 4 ){
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
			     argv[0], " roll from-vector to-vector\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	if( ParseVector(NULL, interp, NULL, argv[2], from, 0) != TCL_OK
	   || ParseVector(NULL, interp, NULL, argv[3], to, 0) != TCL_OK )
	    return TCL_ERROR;
	VecSub(from, from, vport->camera.at);
	VecSub(to, to, vport->camera.at);
	MakeRotationMatrix(from, to, mat);
	RotateVector(wp, mat, vport->camera.view);
	sprintf(str, "{%g %g %g}", wp[0], wp[1], wp[2]);
	Tcl_AppendResult(interp, "-view ", str, (char *) NULL);
	RotateVector(wp, mat, vport->camera.up);
	sprintf(str, "{%g %g %g}", wp[0], wp[1], wp[2]);
	Tcl_AppendResult(interp, " -up ", str, (char *) NULL);

    } else if( c == 's' && strncmp(option, "sphere", length) == 0 ){
	/* canv-vport sphere x y ?radius? */
	/* returns world coords of intersection of ray through
	   pixel (x,y) with a sphere of radius `radius' centered
	   at the `at' point. */
	if( argc < 4 || argc > 5 ){
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
			     argv[0], " sphere x y ?radius?\"", (char *) NULL);
	    return TCL_ERROR;
	}
	rad = 1;
	if( Tcl_GetDouble(interp, argv[2], &dx) != TCL_OK
	   || Tcl_GetDouble(interp, argv[3], &dy) != TCL_OK
	   || argc > 4 && Tcl_GetDouble(interp, argv[4], &rad) != TCL_OK )
	    return TCL_ERROR;
	RaySphereIx(vport, dx, dy, rad, wp);
	sprintf(str, "%g %g %g", wp[0], wp[1], wp[2]);
	Tcl_AppendResult(interp, str, (char *) NULL);

    } else if( c == 'w' && strncmp(option, "world", length) == 0 ){
	/* canv-vport world screen-x screen-y ?depth?
	   returns world coords of point at that depth */
	if( argc < 4 || argc > 5 ){
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
			     argv[0], " world screen-x screen-y ?depth?\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	dz = 1;
	if( Tcl_GetDouble(interp, argv[2], &dx) != TCL_OK
	   || Tcl_GetDouble(interp, argv[3], &dy) != TCL_OK
	   || argc > 4 && Tcl_GetDouble(interp, argv[4], &dz) != TCL_OK )
	    return TCL_ERROR;
	t = vport->camera.perspective? dz: vport->camera.view_distance;
	t /= vport->camera.xyscale * vport->camera.xyscale;
	dx = (dx - vport->camera.xcenter) * t;
	dy = (dy - vport->camera.ycenter) * t;
	dz -= vport->camera.view_distance;
	/* Transform to world coords */
	VecCopy(wp, vport->camera.at);
	VecAddS(wp, wp, vport->camera.xmat[0], dx);
	VecAddS(wp, wp, vport->camera.xmat[1], dy);
	VecAddS(wp, wp, vport->camera.xmat[2], dz);
	sprintf(str, "%g %g %g", wp[0], wp[1], wp[2]);
	Tcl_AppendResult(interp, str, (char *) NULL);

    } else {
	Tcl_AppendResult(interp, "bad option \"", option,
			 "\": must be project, roll, sphere or world",
			 (char *) NULL);
	result = TCL_ERROR;
    }
    return result;
}

/*
 *--------------------------------------------------------------
 *
 * RaySphereIx --
 *
 *	This procedure calculates the world coordinates of the
 *	intersection between a ray through the viewpoint and
 *	a sphere of radius RAD centered on the `at' point.
 *	The ray passes through pixel (DX, DY).
 *	If the ray does not intersect the sphere, it calculates
 *	instead the nearest point on the sphere to the ray.
 *
 * Results:
 *	IXPT gets the calculated world coordinates.
 *	
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
RaySphereIx(vport, dx, dy, rad, ixpt)
    ViewportItem *vport;
    double dx, dy, rad;
    vector ixpt;
{
    double d, t, sinth, costh, tanth, dz;
    double rsp, rcp;
    vector u;

    /* Scale dx and dy to z=1 in translated & rotated coords */
    dx = (dx - vport->camera.xcenter) / vport->camera.xyscale;
    dy = (dy - vport->camera.ycenter) / vport->camera.xyscale;

    tanth = hypot(dx, dy);	/* tan of theta = angle to -v vector */
    if( tanth == 0 ){
	dz = vport->camera.view_distance < rad? rad: -rad;
    } else {
	if( vport->camera.perspective ){
	    t = hypot(1.0, tanth);
	    costh = 1 / t;		/* cos theta */
	    sinth = tanth / t;	/* sin theta */
	    d = vport->camera.view_distance;
	    t = sinth * d;
	    t = rad * rad - t * t;
	    if( t < 0 ){
		/* doesn't intersect sphere: take point on outline
		   of projection of sphere. */
		sinth = rad / d;
		costh = sqrt(1 - sinth * sinth);
		rsp = rad * costh;
		rcp = rad * sinth;
	    } else {
		t = sqrt(t);
		costh *= d;
		if( t <= costh )	/* choose the closer intersection */
		    t = costh - t;
		else
		    t = costh + t;
		rsp = t * sinth;
		rcp = (rad * rad + d * d - t * t) / (2 * d);
	    }

	} else {
	    /* orthogonal projection */
	    rsp = tanth * vport->camera.view_distance;
	    t = rad * rad - rsp * rsp;
	    if( t <= 0.0 ){
		rsp = 1.0;
		rcp = 0.0;
	    } else
		rcp = sqrt(t);
	}

	/* Intersection coords in rotated space */
	dx *= rsp / tanth;
	dy *= rsp / tanth;
	dz = - rcp;
    }

    /* Transform back to world coords */
    VecCopy(ixpt, vport->camera.at);
    VecAddS(ixpt, ixpt, vport->camera.xmat[0], dx / vport->camera.xyscale);
    VecAddS(ixpt, ixpt, vport->camera.xmat[1], dy / vport->camera.xyscale);
    VecAddS(ixpt, ixpt, vport->camera.xmat[2], dz);
}

/*
 *--------------------------------------------------------------
 *
 * MakeRotationMatrix --
 *
 *	This procedure calculates the rotation matrix to rotate
 *	vector FROM to vector TO.
 *
 * Results:
 *	MAT gets the calculated world coordinates.
 *	
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
MakeRotationMatrix(from, to, mat)
    vector from, to;
    double mat[3][3];
{
    vector a, b, c;
    double k, s, l1, l2, mk;

    l1 = VecLength(from);
    l2 = VecLength(to);
    if( l1 == 0 || l2 == 0 ){
	/* Degenerate vectors: return identity matrix */
	k = 1;
	s = 0;
	c[0] = c[1] = c[2] = 0;
    } else {
	/* Make `a' a unit vector in the direction of `from',
	   `b' a unit vector perpendicular to `a' in the plane
	   of `from' and `to'. */
	k = Dot(from, to) / (l1 * l2);	/* cos of (theta = rotation angle) */
	VecScale(a, from, 1/l1);
	VecSubS(b, to, a, k * l2);
	l2 = VecLength(b);
	if( l2 != 0 )
	    VecScale(b, b, 1/l2);
	/* `c' is the vector about which we rotate */
	VecCross(c, a, b);
    }

    /* Set up the matrix */
    s = sqrt(1 - k*k);	/* sin theta */
    mk = 1 - k;
    mat[0][0] = mk * c[0] * c[0] + k;
    mat[0][1] = mk * c[0] * c[1] + s * c[2];
    mat[0][2] = mk * c[0] * c[2] - s * c[1];
    mat[1][0] = mk * c[1] * c[0] - s * c[2];
    mat[1][1] = mk * c[1] * c[1] + k;
    mat[1][2] = mk * c[1] * c[2] + s * c[0];
    mat[2][0] = mk * c[2] * c[0] + s * c[1];
    mat[2][1] = mk * c[2] * c[1] - s * c[0];
    mat[2][2] = mk * c[2] * c[2] + k;
}
