class TCL_INTERPRETER
	-- Encapsulation of a tcl interpreter.
	-- Only standard tcl features are supported.

creation
    make

feature 
   -- status report of the interpreter:

   -- result codes from tcl.h:
    TCL_OK       : INTEGER is 0;
    TCL_ERROR    : INTEGER is 1;
    TCL_RETURN   : INTEGER is 2;
    TCL_BREAK    : INTEGER is 3;
    TCL_CONTINUE : INTEGER is 4;

    tcl_result : INTEGER;
	-- tcl result code of last tcl command.

    tcl_result_string : STRING is
	-- result or error message of last tcl command.
	do
	    !!Result.make(0);
	    Result.from_c(c_tcl_result_string (interpreter));
	end; -- tcl_result_string
    
feature {TCL_COMMAND}
   -- status setting:
    set_tcl_result (res : STRING) is
	-- set result of current tcl command (case of no error).
	local c_string : ANY;
 	do
	    c_string := res.to_c;
	    c_set_tcl_result(interpreter, TCL_OK, $c_string);
	end; -- set_tcl_result;
		
    set_tcl_error (res : STRING) is
	-- set error condition and error message of current tcl command.
	local c_string : ANY;
 	do
	    c_string := res.to_c;
	    c_set_tcl_result(interpreter, TCL_ERROR, $c_string);
	end; -- set_tcl_result;
		
feature
   -- command- and file-evaluation:

    eval (tcl_cmd : STRING) is
	-- evaluate a tcl command in the current interpreter.
	local cmd_string : ANY;
    	do
	    cmd_string := tcl_cmd.to_c;		
    	    tcl_result := c_tcl_eval (interpreter, $cmd_string);
    	end; -- eval

    eval_file (filename : STRING) is
    	-- evaluate a script file in the current interpreter.
	local file_string : ANY;
   	do
	    file_string := filename.to_c;
    	    tcl_result := c_tcl_eval_file (interpreter, 
					   $file_string);
    	end; -- eval_file

feature
   -- creation:

    make is
    	do
    	    interpreter := c_create_tcl_interpreter;
    	end; -- interpreter

feature
   -- variable transfer to and from tcl:

    set_var (name, value :STRING) is
	-- make variable `name' known to tcl with value `value'.
	local c_name, c_value :ANY;
	do
	    c_name := name.to_c;
	    c_value := value.to_c;
            c_set_tcl_var(interpreter, $c_name, $c_value);
	end; -- set_var

    get_var (name:STRING) : STRING is
	-- get the value of a tcl variable with name `name'.
	-- return `Void' if variable doesn't exist.
	local c_name, c_value :ANY;
	do
	    c_name := name.to_c;
            c_value := c_get_tcl_var(interpreter, $c_name, c_value);
	    if c_value /= Void then
	      !!Result.make(0);
	      Result.from_c(c_value);
	    end;
	end; -- get_var

feature {TCL_INTERPRETER, TCL_COMMAND}
   -- implementation:

    interpreter : POINTER;
	-- reference to a C structure used as handle to the
	-- actual interpreter.
 
feature {TCL_INTERPRETER}
   -- C implementation:

    c_create_tcl_interpreter : POINTER is
    	external "C"
    	end;
	
    c_tcl_result_string (interp : POINTER) : ANY is
	external "C"
	end;

    c_set_tcl_result   (interp : POINTER; 
			code : INTEGER;
			res : ANY) is
	external "C"
	end; 

    c_tcl_eval_file (interp : POINTER; 
		     filename : ANY) : INTEGER is
    	external "C"
    	end; 

    c_tcl_eval (interp : POINTER; 
		cmd : ANY) : INTEGER is
    	external "C"
    	end;

    c_set_tcl_var (interp :POINTER; name, value :ANY) is
	external "C"
	end;
	
   	
    c_get_tcl_var (interp :POINTER; name :ANY; Voidref : ANY) : ANY is
	external "C"
	end;

end -- class TCL_INTERPRETER
