#include "tkgsWin.h"
#include "tkgsWinInt.h"

/* FIXME: include our own? We need this for strcasecmp */
#include <tkPort.h>

/*
 * Global data.
 */

static Tcl_ThreadDataKey dataKey;

typedef struct ThreadSpecificData {
    Tcl_HashTable fontFamilyTable;
				/* The list of font families that are 
				 * currently loaded.  As screen fonts
				 * are loaded, this list grows to hold 
				 * information about what characters
				 * exist in each font family. */
} ThreadSpecificData;

/*
 * Information cached about the system at startup time.
 */

static Tcl_Encoding unicodeEncoding;

/*
 * Prototypes for procedures defined later in this file:
 */

static TkGS_FreeInternalRepProc		FreeWinFontIntRep;
static TkGS_SetFromAnyProc		SetWinFontFromAny;

static TkGSFreeSubFontProc		FreeWinSubFont;
static TkGSFreeFontFamilyProc		FreeWinFontFamily;


static TkGSFontFamily   AllocFontFamily _ANSI_ARGS_((HDC hdc, HFONT hFont));
static HFONT		CreateClosestFont _ANSI_ARGS_((TkGS_Drawable d,
			    CONST TkGS_FontAttributes *faPtr));
static TkGSSubFont	GetDrawableSubFont _ANSI_ARGS_((TkGS_Drawable d,
			    TkGS_InternalRep *intRepPtr));
static void		InitFont _ANSI_ARGS_((TkGS_Font font, 
			    WinFont *winFontPtr, HDC hdc, HFONT hFont));
static void		InitSubFont _ANSI_ARGS_((TkGS_Font font, 
			    WinSubFont *winSubFontPtr, HDC hdc, HFONT hFont));
static int		LoadFontRanges(HDC hdc, HFONT hFont, 
			    USHORT **startCount, USHORT **endCount,
			    int *symbolPtr);
static void		SwapLong(PULONG p);
static void		SwapShort(USHORT *p);


/*
 * Win32 Font Obj type
 */

/* Object type */
TkGS_ObjType WinFontType = {
    "WinFont",
    NULL,				/* Base type, initialized at runtime */

    FreeWinFontIntRep,			/* freeIntRepProc */
    SetWinFontFromAny			/* setFromAnyProc */
};
TkGS_ObjType *WinFontTypePtr = &WinFontType;


/* Object type procs */

static void
FreeWinFontFamily(fontFamily)
    TkGSFontFamily fontFamily;
{
    register WinFontFamily *familyPtr = (WinFontFamily *) fontFamily;
    if (familyPtr->startCount != NULL) {
	ckfree((char *) familyPtr->startCount);
    }
    if (familyPtr->endCount != NULL) {
	ckfree((char *) familyPtr->endCount);
    }

    /* 
     * Delete from table. 
     */

    Tcl_DeleteHashEntry(familyPtr->hashPtr);
}

static void
FreeWinSubFont(font, subFont)
    TkGSMultiFont font;
    TkGSSubFont subFont;
{
    TkGSFreeSubFont(subFont, FreeWinFontFamily);

    DeleteObject(WinSubFont_HFONT(subFont));
}

static void
FreeWinFontIntRep(intRepPtr)
    TkGS_InternalRep *intRepPtr;
{
    TkGSFreeMultiFont((TkGSMultiFont) WinFont_InternalRep(intRepPtr), 
	sizeof(WinSubFont), FreeWinSubFont);

    /* Free the allocated memory */
    ckfree((char *) WinFont_InternalRep(intRepPtr));
}

static int
SetWinFontFromAny(interp, objPtr, intRepPtr)
    Tcl_Interp *interp;
    TkGS_Obj *objPtr;
    TkGS_InternalRep *intRepPtr;
{
    register WinFont *winFontPtr;

    /* 
     * Free old internal rep
     */

    if (   intRepPtr->typePtr != NULL
	&& intRepPtr->typePtr->freeIntRepProc != NULL) {
	intRepPtr->typePtr->freeIntRepProc(intRepPtr);
    }

    /* 
     * Initialize internal rep.
     */

    winFontPtr = (WinFont *) ckalloc(sizeof(WinFont));
    memset(winFontPtr, 0, sizeof(WinFont));
    WinFont_InternalRep(intRepPtr) = winFontPtr;

    /* Generic multi-font initialization. */
    TkGSInitMultiFont((TkGSMultiFont) winFontPtr);

    /* HFONT lazily computed. */
    WinFont_HFONT(intRepPtr) = NULL;

    /* Change type */
    intRepPtr->typePtr = WinFontTypePtr;

    return TKGS_OK;
}

WinFont *
WinTkGS_GetWinFontFromFont(interp, d, font)
    Tcl_Interp *interp;
    TkGS_Drawable d;
    TkGS_Font font;
{
    DECLAREDRAWINGVARS(d, dIntRepPtr, hdc)
    register TkGS_InternalRep 
	*intRepPtr = TkGS_FindInternalRep((TkGS_Obj *) font, &WinFontType);
    HFONT hFont;
    TkGS_FontAttributes *faPtr;

    /*
     * Get a WinFont from TkGS Font's internal rep
     */

    if (intRepPtr == NULL) {
	intRepPtr = TkGS_AddNewInternalRep(interp, (TkGS_Obj *) font, 
					   &WinFontType);
	if (!intRepPtr) {
	    return NULL;
	}
    }

    if (WinFont_HFONT(intRepPtr) == NULL) {

	/*
	 * Get the closest font to the name requested.
	 */

	faPtr = &TkGSFont_Attributes(font);
	hFont = CreateClosestFont(d, faPtr);
	InitFont(font, WinFont_InternalRep(intRepPtr), hdc, hFont);

	WinFont_HFONT(intRepPtr) = hFont;
    }

    return WinFont_InternalRep(intRepPtr);
}

/*
 *---------------------------------------------------------------------------
 *
 * WinTkGSGetScreenFont --
 *
 *	Given the name and other attributes, construct an HFONT.
 *	This is where all the alias and fallback substitution bottoms
 *	out.
 *
 * Results:
 *	The screen font that corresponds to the attributes.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */

HFONT 
WinTkGSGetScreenFontUni(hdc, faPtr, faceName)
    HDC hdc;
    CONST TkGS_FontAttributes *faPtr;
				/* Desired font attributes for new HFONT. */
    CONST char *faceName;	/* Font family to use. */
{
    Tcl_DString ds;
    HFONT hFont;
    LOGFONTW lf;
    long size;

	/* TODO: better pixel <-> point conversion */
    if (faPtr->size < 0) {
	/* Pixel size */
	size = -faPtr->size;
    } else {
	/* Point size */
	size = -MulDiv(faPtr->size, GetDeviceCaps(hdc, LOGPIXELSY), 72);
    }

    memset(&lf, 0, sizeof(lf));
    lf.lfHeight		= -size;
    lf.lfWidth		= 0;
    lf.lfEscapement	= 0;
    lf.lfOrientation	= 0;
    lf.lfWeight		= (faPtr->weight == TKGS_FW_NORMAL) ? FW_NORMAL : FW_BOLD;
    lf.lfItalic		= faPtr->slant;
    lf.lfUnderline	= faPtr->underline;
    lf.lfStrikeOut	= faPtr->overstrike;
    lf.lfCharSet	= DEFAULT_CHARSET;
    lf.lfOutPrecision	= OUT_TT_PRECIS;
    lf.lfClipPrecision	= CLIP_DEFAULT_PRECIS;
    lf.lfQuality	= DEFAULT_QUALITY;
    lf.lfPitchAndFamily = DEFAULT_PITCH | FF_DONTCARE;

    /*
     * We can only store up to LF_FACESIZE wide characters
     */

    Tcl_DStringInit(&ds);
    Tcl_UtfToUniCharDString(faceName, -1, &ds);
    if (Tcl_DStringLength(&ds) >= (LF_FACESIZE * sizeof(WCHAR))) {
	Tcl_DStringSetLength(&ds, (LF_FACESIZE * sizeof(WCHAR)));
    }
    wcscpy(lf.lfFaceName, (WCHAR *) Tcl_DStringValue(&ds));
    hFont = CreateFontIndirectW(&lf);
    Tcl_DStringFree(&ds);
    return hFont;
}

HFONT 
WinTkGSGetScreenFontSys(hdc, faPtr, faceName)
    HDC hdc;
    CONST TkGS_FontAttributes *faPtr;
				/* Desired font attributes for new HFONT. */
    CONST char *faceName;	/* Font family to use. */
{
    Tcl_DString ds;
    HFONT hFont;
    LOGFONTA lf;
    long size;

    if (faPtr->size < 0) {
	/* Pixel size */
	size = -faPtr->size;
    } else {
	/* Point size */
	/* TODO: compute right point -> pixel conversion */
	size = -MulDiv(faPtr->size, GetDeviceCaps(hdc, LOGPIXELSY), 72);
    }

    memset(&lf, 0, sizeof(lf));
    lf.lfHeight		= -size;
    lf.lfWidth		= 0;
    lf.lfEscapement	= 0;
    lf.lfOrientation	= 0;
    lf.lfWeight		= (faPtr->weight == TKGS_FW_NORMAL) ? FW_NORMAL : FW_BOLD;
    lf.lfItalic		= faPtr->slant;
    lf.lfUnderline	= faPtr->underline;
    lf.lfStrikeOut	= faPtr->overstrike;
    lf.lfCharSet	= DEFAULT_CHARSET;
    lf.lfOutPrecision	= OUT_TT_PRECIS;
    lf.lfClipPrecision	= CLIP_DEFAULT_PRECIS;
    lf.lfQuality	= DEFAULT_QUALITY;
    lf.lfPitchAndFamily = DEFAULT_PITCH | FF_DONTCARE;

    /*
     * We can only store up to LF_FACESIZE characters
     */

    Tcl_DStringInit(&ds);
    Tcl_UtfToExternalDString(NULL /* system encoding */, faceName, -1, &ds);
    if (Tcl_DStringLength(&ds) >= LF_FACESIZE) {
	Tcl_DStringSetLength(&ds, LF_FACESIZE);
    }
    strcpy(lf.lfFaceName, Tcl_DStringValue(&ds));
    hFont = CreateFontIndirectA(&lf);
    Tcl_DStringFree(&ds);
    return hFont;
}


void
WinTkGSGetFaceNameUni(hdc, namePtr)
    HDC hdc;
    char *namePtr;
{
    Tcl_DString faceString;
    WCHAR buf[LF_FACESIZE];

    /*
     * On any version of NT, there may be fonts with international names.  
     * Use the NT-only Unicode version of GetTextFace to get the font's 
     * name.  If we used the ANSI version on a non-internationalized 
     * version of NT, we would get a font name with '?' replacing all 
     * the international characters.
     *
     * On a non-internationalized verson of 95, fonts with international
     * names are not allowed, so the ANSI version of GetTextFace will work.
     * On an internationalized version of 95, there may be fonts with 
     * international names; the ANSI version will work, fetching the 
     * name in the international system code page.  Can't use the Unicode 
     * version of GetTextFace because it only exists under NT.
     */

    Tcl_DStringInit(&faceString);
    GetTextFaceW(hdc, LF_FACESIZE, (WCHAR *) buf);
    Tcl_UniCharToUtfDString(buf, -1, &faceString);
    strcpy(namePtr, Tcl_DStringValue(&faceString));
    Tcl_DStringFree(&faceString);
}

void
WinTkGSGetFaceNameSys(hdc, namePtr)
    HDC hdc;
    char *namePtr;
{
    Tcl_DString faceString;
    char buf[LF_FACESIZE];

    /*
     * On any version of NT, there may be fonts with international names.  
     * Use the NT-only Unicode version of GetTextFace to get the font's 
     * name.  If we used the ANSI version on a non-internationalized 
     * version of NT, we would get a font name with '?' replacing all 
     * the international characters.
     *
     * On a non-internationalized verson of 95, fonts with international
     * names are not allowed, so the ANSI version of GetTextFace will work.
     * On an internationalized version of 95, there may be fonts with 
     * international names; the ANSI version will work, fetching the 
     * name in the international system code page.  Can't use the Unicode 
     * version of GetTextFace because it only exists under NT.
     */

    Tcl_DStringInit(&faceString);
    GetTextFaceA(hdc, LF_FACESIZE, (char *) buf);
    Tcl_ExternalToUtfDString(NULL /* system encoding */, buf, -1, &faceString);
    strcpy(namePtr, Tcl_DStringValue(&faceString));
    Tcl_DStringFree(&faceString);
}

/*
 *-------------------------------------------------------------------------
 *
 * GetDrawableSubFont --
 *
 *	Helper proc for getting the current subfont.
 *
 * Results:
 *	Return value is the TkGSSubFont that is currently selected in the
 *	drawable. It is never null.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */

static TkGSSubFont
GetDrawableSubFont(d, intRepPtr)
    TkGS_Drawable d;			/* Drawable to get the subfont from. */
    TkGS_InternalRep *intRepPtr;	/* Win32-specific internal rep of the
					 * above drawable. */
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    WinFont *winFontPtr = WinTkGS_GetWinFontFromFont(NULL, d, gcValues->font);

    return (TkGSSubFont) WinTkGSGetSubFontFromId(winFontPtr, WinDrawable_SubFontId(intRepPtr));
}

/*
 *-------------------------------------------------------------------------
 *
 * CreateClosestFont --
 *
 *	Given a set of font attributes, construct a close HFONT.
 *	If requested face name is not available, automatically
 *	substitutes an alias for requested face name.  For all 
 *	other attributes, if the requested value was not available, 
 *	the appropriate "close" value will be used.
 *
 * Results:
 *	Return value is the HFONT that best matched the requested 
 *	attributes.  The return value is never NULL; some
 *	font will always be returned.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */

static HFONT
CreateClosestFont(d, faPtr)
    TkGS_Drawable d;
    CONST TkGS_FontAttributes *faPtr;
{
    DECLAREDRAWINGVARS(d, dIntRepPtr, hdc)
    char *faceName, *actualName;

    /*
     * Get the closest font to the name requested.
     */

    /* 
     * TODO: manage multiple font family specifiers. For now only take 
     * the first.
     */

    faceName = TkGSFontFaces_Names(faPtr->faces)[0];
    actualName = TkGSGetClosestFaceName(d, faceName);
    if (actualName != NULL) {
	faceName = actualName;
    }

    return WinTkGSGetScreenFont(hdc, faPtr, faceName);
}

/*
 *---------------------------------------------------------------------------
 *
 * InitFont --
 *
 *	Helper for WinTkGS_GetWinFontFromFont().
 *	Initializes the memory for a new font.
 *
 * Results:
 *	Fills the font structure.
 *
 * Side effects:
 *	Memory allocated.
 *
 *---------------------------------------------------------------------------
 */ 

static void
InitFont(font, winFontPtr, hdc, hFont)
    TkGS_Font font;
    WinFont *winFontPtr;
    HDC hdc;
    HFONT hFont;
{
    HFONT oldFont;
    TEXTMETRIC tm;
    TkGS_FontMetrics *fmPtr;
    Tcl_Encoding encoding;
    TkGS_FontAttributes *faPtr;

    /*
     * Initialize base font (first subfont).
     */

    winFontPtr->header.numSubFonts++;
    winFontPtr->header.subFontArray = 
	(TkGSSubFont_ *) ckrealloc((char *) winFontPtr->header.subFontArray,
				   sizeof(WinSubFont));
    InitSubFont(font, WinTkGSGetSubFontFromIndex(winFontPtr, 0), hdc, hFont);

    oldFont = SelectObject(hdc, hFont);

    /*
     * Get real font family.
     */

/*FIXME    winFontPtr->family = WinTkGSGetFaceName(hdc);*/

    /*
     * Compute font attributes.
     */

    GetTextMetrics(hdc, &tm);

    faPtr		= &winFontPtr->fa;
    faPtr->faces	= NULL;
    faPtr->size		= -(tm.tmHeight - tm.tmInternalLeading);
    faPtr->weight	= (tm.tmWeight > FW_MEDIUM) ? TKGS_FW_BOLD : TKGS_FW_NORMAL;
    faPtr->slant	= (tm.tmItalic != 0) ? TKGS_FS_ITALIC : TKGS_FS_ROMAN;
    faPtr->underline	= (tm.tmUnderlined != 0) ? 1 : 0;
    faPtr->overstrike	= font->attributes.overstrike;
    
    /*
     * Compute font metrics.
     */

    fmPtr		= &winFontPtr->fm;
    fmPtr->ascent	= tm.tmAscent;
    fmPtr->descent	= tm.tmDescent;
    fmPtr->maxWidth	= tm.tmMaxCharWidth;
    fmPtr->fixed	= !(tm.tmPitchAndFamily & TMPF_FIXED_PITCH);

    /*
     * Measure chars.
     */

    encoding = WinTkGSGetSubFontFromIndex(winFontPtr, 0)->header.family->encoding;
    if (encoding == unicodeEncoding) {
	GetCharWidthW(hdc, 0, WIN_BASE_CHARS - 1, winFontPtr->widths);
    } else {
	GetCharWidthA(hdc, 0, WIN_BASE_CHARS - 1, winFontPtr->widths);
    } 

    fmPtr->tabWidth	= winFontPtr->widths['\t'];
    if (fmPtr->tabWidth == 0) {
	fmPtr->tabWidth = fmPtr->maxWidth;
    }
    fmPtr->tabWidth *= 8;

    /*
     * Make sure the tab width isn't zero (some fonts may not have enough
     * information to set a reasonable tab width).
     */

    if (fmPtr->tabWidth == 0) {
	fmPtr->tabWidth = 1;
    }

    /*
     * Get information used for drawing underlines in generic code on a
     * non-underlined font.
     */
    
    fmPtr->underlinePos = fmPtr->descent / 2;
    fmPtr->underlineHeight = faPtr->size / 10;
    if (fmPtr->underlineHeight == 0) {
	fmPtr->underlineHeight = 1;
    }
    if (fmPtr->underlinePos + fmPtr->underlineHeight > fmPtr->descent) {
	/*
	 * If this set of values would cause the bottom of the underline
	 * bar to stick below the descent of the font, jack the underline
	 * up a bit higher.
	 */

	fmPtr->underlineHeight = fmPtr->descent - fmPtr->underlinePos;
	if (fmPtr->underlineHeight == 0) {
	    fmPtr->underlinePos--;
	    fmPtr->underlineHeight = 1;
	}
    }
    
    SelectObject(hdc, oldFont);
}

/*
 *-------------------------------------------------------------------------
 *
 * InitSubFont --
 *
 *	Wrap a screen font and load the FontFamily that represents
 *	it.  Used to prepare a SubFont so that characters can be mapped
 *	from UTF-8 to the charset of the font.
 *
 * Results:
 *	The subFontPtr is filled with information about the font.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */

static void
InitSubFont(font, winSubFontPtr, hdc, hFont)
    TkGS_Font font;
    WinSubFont *winSubFontPtr;
    HDC hdc;
    HFONT hFont;
{
    /* Generic subfont initialization. */
    TkGSInitSubFont((TkGSSubFont) winSubFontPtr, 
	AllocFontFamily(hdc, hFont));

    /* Windows-specific subfont initialization. */
    winSubFontPtr->hFont = hFont;
}

/*
 *-------------------------------------------------------------------------
 *
 * AllocFontFamily --
 *
 *	Find the font family structure associated with the given font
 *	name.  The information should be stored by the caller in a 
 *	subfont and used when determining if that SubFont supports a 
 *	character.
 *
 *	Cannot use the string name used to construct the font as the 
 *	key, because the capitalization may not be canonical.  Therefore
 *	use the face name actually retrieved from the font metrics as
 *	the key.
 *
 * Results:
 *	A pointer to a font family.  The reference count in the font family
 *	is automatically incremented.  When the subfont is released, the
 *	reference count is decremented.  When no subfont is using this
 *	font family, it may be deleted.
 *
 * Side effects:
 *	A new font family structure will be allocated if this font family
 *	has not been seen.  TrueType character existence metrics are
 *	loaded into the font family structure.
 *
 *-------------------------------------------------------------------------
 */

static TkGSFontFamily
AllocFontFamily(hdc, hFont)
    HDC hdc;
    HFONT hFont;
{
    WinFontFamily *familyPtr;
    ThreadSpecificData *tsdPtr = (ThreadSpecificData *) 
            Tcl_GetThreadData(&dataKey, sizeof(ThreadSpecificData));
    char faceName[256];
    Tcl_Encoding encoding;
    int bNew;
    Tcl_HashEntry *entryPtr;
    HFONT oldFont;
    int isSymbolFont, isWideFont;
    int segCount;
    USHORT *startCount, *endCount;
    
    /* 
     * Set key for this FontFamily. 
     */

    oldFont = SelectObject(hdc, hFont);
    WinTkGSGetFaceName(hdc, faceName);
    SelectObject(hdc, oldFont);

    /*
     * Find corresponding entry in family table or create one.
     */

    entryPtr = Tcl_CreateHashEntry(&tsdPtr->fontFamilyTable, faceName, &bNew);
    if (!bNew) {
	/* 
	 * Already existing font family.
	 */

	familyPtr = (WinFontFamily *) Tcl_GetHashValue(entryPtr);
	familyPtr->header.refCount++;
	return (TkGSFontFamily) familyPtr;
    }

    /*
     * Create new font family.
     */

    segCount = LoadFontRanges(hdc, hFont, &startCount, &endCount, 
	    &isSymbolFont);

    encoding = NULL;
    if (isSymbolFont != 0) {
	/*
	 * Symbol fonts are handled specially.  For instance, Unicode 0393
	 * (GREEK CAPITAL GAMMA) must be mapped to Symbol character 0047
	 * (GREEK CAPITAL GAMMA), because the Symbol font doesn't have a
	 * GREEK CAPITAL GAMMA at location 0393.  If Tk interpreted the
	 * Symbol font using the Unicode encoding, it would decide that
	 * the Symbol font has no GREEK CAPITAL GAMMA, because the Symbol
	 * encoding (of course) reports that character 0393 doesn't exist.  
	 *
	 * With non-symbol Windows fonts, such as Times New Roman, if the
	 * font has a GREEK CAPITAL GAMMA, it will be found in the correct
	 * Unicode location (0393); the GREEK CAPITAL GAMMA will not be off
	 * hiding at some other location.
	 */

	encoding = Tcl_GetEncoding(NULL, faceName);
    }

    if (encoding == NULL) {
	encoding = Tcl_GetEncoding(NULL, "unicode");
	isWideFont = 1;
    } else {
	isWideFont = 0;
    } 


    familyPtr = (WinFontFamily *) ckalloc(sizeof(WinFontFamily));

    /* Generic font family initialization. */
    TkGSInitFontFamily((TkGSFontFamily) familyPtr, encoding);
    familyPtr->header.refCount++;

    /* Windows-specific font family initialization. */
    Tcl_SetHashValue(entryPtr, (ClientData) familyPtr);
    familyPtr->hashPtr      = entryPtr;
    familyPtr->isSymbolFont = isSymbolFont;
    familyPtr->isWideFont   = isWideFont;
    familyPtr->segCount     = segCount;
    familyPtr->startCount   = startCount;
    familyPtr->endCount     = endCount;

    return (TkGSFontFamily) familyPtr;
}

/*
 *-------------------------------------------------------------------------
 *
 * WinTkGSCanUseFallback --
 *
 *	If the specified screen font has not already been loaded into 
 *	the font object, determine if it can display the given character.
 *
 * Results:
 *	The return value is the id of a newly allocated subfont, owned
 *	by the font object.  This subfont can be used to display the given
 *	character.  The subfont represents the screen font with the base set 
 *	of font attributes from the font object, but using the specified 
 *	font name.  NULL is returned if the font object already holds
 *	a reference to the specified physical font or if the specified 
 *	font doesn't exist or cannot display the given character.
 *
 * Side effects:				       
 *	The font object's subFontArray is updated to contain a reference
 *	to the newly allocated SubFont.
 *
 *-------------------------------------------------------------------------
 */

TkGSSubFontId
WinTkGSCanUseFallback(d, font, multiFont, faceName, ch)
    TkGS_Drawable d;
    TkGS_Font font;
    TkGSMultiFont multiFont;	/* The font object that will own the new
				 * screen font. */
    CONST char *faceName;	/* Desired face name for new screen font. */
    TkGS_UniChar ch;		/* The Unicode character that the new
				 * screen font must be able to display. */
{
    DECLAREDRAWINGVARS(d, dIntRepPtr, hdc)
    ThreadSpecificData *tsdPtr = (ThreadSpecificData *) 
            Tcl_GetThreadData(&dataKey, sizeof(ThreadSpecificData));
    WinFont *fontPtr = (WinFont *) multiFont;
    WinFontFamily *familyPtr;
    int i;
    WinSubFont subFont;
    HFONT hFont;

    if (TkGS_FontFamilyExists(d, faceName) == 0) {
	return 0;
    }

    /* 
     * Skip all fonts we've already used.
     */
     
    for (i = 0; i < multiFont->numSubFonts; i++) {
	familyPtr = (WinFontFamily *) 
	    WinTkGSGetSubFontFromIndex(fontPtr, i)->header.family;

	if (strcasecmp(faceName, 
		Tcl_GetHashKey(&tsdPtr->fontFamilyTable, familyPtr->hashPtr))
	    == 0) {
	    return 0;
	}
    }

    /*
     * Load this font and see if it has the desired character.
     */

    hFont = WinTkGSGetScreenFont(hdc, &TkGSFont_Attributes(font), faceName);
    InitSubFont(font, &subFont, hdc, hFont);
    familyPtr = (WinFontFamily *) subFont.header.family;
    if (((ch < 256) && (familyPtr->isSymbolFont)) 
	    || (TkGSFontMapLookup(d, (TkGSSubFont) &subFont, ch) == 0)) {
	/*
	 * Don't use a symbol font as a fallback font for characters below
	 * 256.
	 */

	FreeWinSubFont(multiFont,(TkGSSubFont) &subFont);
	return 0;
    }

    /*
     * Add subfont.
     */

    fontPtr->header.subFontArray = (TkGSSubFont) ckrealloc(
				(char *) fontPtr->header.subFontArray, 
				sizeof(WinSubFont) 
				* (fontPtr->header.numSubFonts + 1));

    *WinTkGSGetSubFontFromIndex(fontPtr, fontPtr->header.numSubFonts) = subFont;
    fontPtr->header.numSubFonts++;
    return WinTkGSGetSubFontId(fontPtr->header.numSubFonts - 1);
}

/*
 * The following data structures are used when querying a TrueType font file
 * to determine which characters the font supports.
 */

#pragma pack(1)			/* Structures are byte aligned in file. */

#define CMAPHEX  0x636d6170	/* Key for character map resource. */

typedef struct CMAPTABLE {
    USHORT version;		/* Table version number (0). */
    USHORT numTables;		/* Number of encoding tables following. */
} CMAPTABLE;

typedef struct ENCODINGTABLE {
    USHORT platform;		/* Platform for which data is targeted.  
				 * 3 means data is for Windows. */
    USHORT encoding;		/* How characters in font are encoded.  
				 * 1 means that the following subtable is 
				 * keyed based on Unicode. */
    ULONG offset;		/* Byte offset from beginning of CMAPTABLE 
				 * to the subtable for this encoding. */
} ENCODINGTABLE;

typedef struct ANYTABLE {
    USHORT format;		/* Format number. */
    USHORT length;		/* The actual length in bytes of this 
				 * subtable. */
    USHORT version;		/* Version number (starts at 0). */
} ANYTABLE;

typedef struct BYTETABLE {
    USHORT format;		/* Format number is set to 0. */
    USHORT length;		/* The actual length in bytes of this 
				 * subtable. */
    USHORT version;		/* Version number (starts at 0). */
    BYTE glyphIdArray[256];	/* Array that maps up to 256 single-byte char
				 * codes to glyph indices. */
} BYTETABLE;

typedef struct SUBHEADER {
    USHORT firstCode;		/* First valid low byte for subHeader. */
    USHORT entryCount;		/* Number valid low bytes for subHeader. */
    SHORT idDelta;		/* Constant adder to get base glyph index. */
    USHORT idRangeOffset;	/* Byte offset from here to appropriate 
				 * glyphIndexArray. */
} SUBHEADER;

typedef struct HIBYTETABLE {
    USHORT format;  		/* Format number is set to 2. */
    USHORT length;		/* The actual length in bytes of this
				 * subtable. */
    USHORT version;		/* Version number (starts at 0). */
    USHORT subHeaderKeys[256];	/* Maps high bytes to subHeaders: value is 
				 * subHeader index * 8. */
#if 0
    SUBHEADER subHeaders[];	/* Variable-length array of SUBHEADERs. */
    USHORT glyphIndexArray[];	/* Variable-length array containing subarrays 
				 * used for mapping the low byte of 2-byte 
				 * characters. */
#endif
} HIBYTETABLE;

typedef struct SEGMENTTABLE {
    USHORT format;		/* Format number is set to 4. */
    USHORT length;		/* The actual length in bytes of this
				 * subtable. */
    USHORT version;		/* Version number (starts at 0). */
    USHORT segCountX2;		/* 2 x segCount. */
    USHORT searchRange;		/* 2 x (2**floor(log2(segCount))). */
    USHORT entrySelector;	/* log2(searchRange/2). */
    USHORT rangeShift;		/* 2 x segCount - searchRange. */
#if 0
    USHORT endCount[segCount]	/* End characterCode for each segment. */
    USHORT reservedPad;		/* Set to 0. */
    USHORT startCount[segCount];/* Start character code for each segment. */
    USHORT idDelta[segCount];	/* Delta for all character in segment. */
    USHORT idRangeOffset[segCount]; /* Offsets into glyphIdArray or 0. */
    USHORT glyphIdArray[]	/* Glyph index array. */
#endif
} SEGMENTTABLE;

typedef struct TRIMMEDTABLE {
    USHORT format;		/* Format number is set to 6. */
    USHORT length;		/* The actual length in bytes of this
				 * subtable. */
    USHORT version;		/* Version number (starts at 0). */
    USHORT firstCode;		/* First character code of subrange. */
    USHORT entryCount;		/* Number of character codes in subrange. */
#if 0
    USHORT glyphIdArray[];	/* Array of glyph index values for 
				        character codes in the range. */
#endif
} TRIMMEDTABLE;

typedef union SUBTABLE {
    ANYTABLE any;
    BYTETABLE byte;
    HIBYTETABLE hiByte;
    SEGMENTTABLE segment;
    TRIMMEDTABLE trimmed;
} SUBTABLE;

#pragma pack()

/*
 *-------------------------------------------------------------------------
 *
 * LoadFontRanges --
 *
 *	Given an HFONT, get the information about the characters that 
 *	this font can display.
 *
 * Results:
 *	If the font has no Unicode character information, the return value
 *	is 0 and *startCountPtr and *endCountPtr are filled with NULL.  
 *	Otherwise, *startCountPtr and *endCountPtr are set to pointers to 
 *	arrays of TrueType character existence information and the return 
 *	value is the length of the arrays (the two arrays are always the 
 *	same length as each other).
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */

static int
LoadFontRanges(
    HDC hdc,			/* HDC into which font can be selected. */
    HFONT hFont,		/* HFONT to query. */
    USHORT **startCountPtr,	/* Filled with malloced pointer to 
				 * character range information. */
    USHORT **endCountPtr,	/* Filled with malloced pointer to 
				 * character range information. */
    int *symbolPtr)
 {
    int n, i, swapped, offset, cbData, segCount;
    DWORD cmapKey;
    USHORT *startCount, *endCount;
    CMAPTABLE cmapTable;
    ENCODINGTABLE encTable;
    SUBTABLE subTable;
    char *s;

    segCount = 0;
    startCount = NULL;
    endCount = NULL;
    *symbolPtr = 0;

    hFont = SelectObject(hdc, hFont);

    i = 0;
    s = (char *) &i;
    *s = '\1';
    swapped = 0;

    if (i == 1) {
	swapped = 1;
    }

    cmapKey = CMAPHEX;
    if (swapped) {
	SwapLong(&cmapKey);
    }

    n = GetFontData(hdc, cmapKey, 0, &cmapTable, sizeof(cmapTable));
    if (n != GDI_ERROR) {
	if (swapped) {
	    SwapShort(&cmapTable.numTables);
	}
	for (i = 0; i < cmapTable.numTables; i++) {
	    offset = sizeof(cmapTable) + i * sizeof(encTable);
	    GetFontData(hdc, cmapKey, offset, &encTable, sizeof(encTable));
	    if (swapped) {
		SwapShort(&encTable.platform);
		SwapShort(&encTable.encoding);
		SwapLong(&encTable.offset);
	    }
	    if (encTable.platform != 3) {
		/* 
		 * Not Microsoft encoding.
		 */

		continue;
	    }
	    if (encTable.encoding == 0) {
		*symbolPtr = 1;
	    } else if (encTable.encoding != 1) {
		continue;
	    }

	    GetFontData(hdc, cmapKey, encTable.offset, &subTable, 
		    sizeof(subTable));
	    if (swapped) {
		SwapShort(&subTable.any.format);
	    }
	    if (subTable.any.format == 4) {
		if (swapped) {
		    SwapShort(&subTable.segment.segCountX2);
		}
		segCount = subTable.segment.segCountX2 / 2;
		cbData = segCount * sizeof(USHORT);

		startCount = (USHORT *) ckalloc(cbData);
		endCount = (USHORT *) ckalloc(cbData);

		offset = encTable.offset + sizeof(subTable.segment);
		GetFontData(hdc, cmapKey, offset, endCount, cbData);
		offset += cbData + sizeof(USHORT);
		GetFontData(hdc, cmapKey, offset, startCount, cbData);
		if (swapped) {
		    for (i = 0; i < segCount; i++) {
			SwapShort(&endCount[i]);
			SwapShort(&startCount[i]);
		    }
		}
		if (*symbolPtr != 0) {
		    /*
		     * Empirically determined:  When a symbol font is
		     * loaded, the character existence metrics obtained
		     * from the system are mildly wrong.  If the real range
		     * of the symbol font is from 0020 to 00FE, then the
		     * metrics are reported as F020 to F0FE.  When we load
		     * a symbol font, we must fix the character existence
		     * metrics.
		     *
		     * Symbol fonts should only use the symbol encoding
		     * for 8-bit characters [note Bug: 2406]
		     */

		    for (i = 0; i < segCount; i++) {
			if (((startCount[i] & 0xff00) == 0xf000)
				&& ((endCount[i] & 0xff00) == 0xf000)) {
			    startCount[i] &= 0xff;
			    endCount[i] &= 0xff;
			}
		    }
		}
	    }
	}
    } else if (GetTextCharset(hdc) == ANSI_CHARSET) {
	/*
	 * Bitmap font.  We should also support ranges for the other
	 * *_CHARSET values.
	 */
	segCount = 1;
	cbData = segCount * sizeof(USHORT);
	startCount = (USHORT *) ckalloc(cbData);
	endCount = (USHORT *) ckalloc(cbData);
	startCount[0] = 0x0000;
	endCount[0] = 0x00ff;
    }
    SelectObject(hdc, hFont);

    *startCountPtr = startCount;
    *endCountPtr = endCount;
    return segCount;
}

/*
 *-------------------------------------------------------------------------
 * 
 * SwapShort, SwapLong --
 *
 *	Helper functions to convert the data loaded from TrueType font
 *	files to Intel byte ordering.
 *
 * Results:
 *	Bytes of input value are swapped and stored back in argument.
 *
 * Side effects:
 *	None.
 *
 *-------------------------------------------------------------------------
 */

static void
SwapShort(PUSHORT p)
{
    *p = (SHORT)(HIBYTE(*p) + (LOBYTE(*p) << 8));
}

static void 
SwapLong(PULONG p)
{					     
    ULONG temp;

    temp = (LONG) ((BYTE) *p);
    temp <<= 8;
    *p >>=8;

    temp += (LONG) ((BYTE) *p);
    temp <<= 8;
    *p >>=8;

    temp += (LONG) ((BYTE) *p);
    temp <<= 8;
    *p >>=8;

    temp += (LONG) ((BYTE) *p);
    *p = temp;
}


/*
 * Driver font primitives.
 */



typedef struct FontFamiliesEnumParam {
    TkGS_Drawable		d;
    TkGS_FontFamiliesEnumProc	*enumProc;
    Tcl_Encoding		encoding;
    ClientData			clientData;
} FontFamiliesEnumParam;

static int CALLBACK
FontFamiliesEnumProc(
    ENUMLOGFONT *lfPtr,		/* Logical-font data. */
    NEWTEXTMETRIC *tmPtr,	/* Physical-font data (not used). */
    int fontType,		/* Type of font (not used). */
    LPARAM lParam)		/* Result object to hold result. */
{
    FontFamiliesEnumParam *paramPtr = (FontFamiliesEnumParam *) lParam;
    Tcl_DString buf;
    int result;

    Tcl_DStringInit(&buf);
    Tcl_ExternalToUtfDString(paramPtr->encoding, 
	lfPtr->elfLogFont.lfFaceName, -1, &buf);
    result = (paramPtr->enumProc)(paramPtr->d, paramPtr->clientData, 
		Tcl_DStringValue(&buf));
    Tcl_DStringFree(&buf);

    return result;
}

void
WinTkGS_EnumerateFontFamiliesUni(d, name, enumProc, clientData)
    TkGS_Drawable		d;
    CONST char			*name;
    TkGS_FontFamiliesEnumProc	*enumProc;
    ClientData			clientData;
{
    DECLAREDRAWINGVARS(d, intRepPtr, hdc)
    FontFamiliesEnumParam param;

    param.d = d;
    param.enumProc = enumProc;
    param.encoding = unicodeEncoding;
    param.clientData = clientData;

    if (name) {
	Tcl_DString buf;

	Tcl_DStringInit(&buf);
	Tcl_UtfToUniCharDString(name, -1, &buf);
	EnumFontFamiliesW(hdc, (WCHAR *) Tcl_DStringValue(&buf), 
	    (FONTENUMPROCW) FontFamiliesEnumProc, (LPARAM) &param);
	Tcl_DStringFree(&buf);
    } else {
	EnumFontFamiliesW(hdc, NULL, 
	    (FONTENUMPROCW) FontFamiliesEnumProc, (LPARAM) &param);
    }
}

void
WinTkGS_EnumerateFontFamiliesSys(d, name, enumProc, clientData)
    TkGS_Drawable		d;
    CONST char			*name;
    TkGS_FontFamiliesEnumProc	*enumProc;
    ClientData			clientData;
{
    DECLAREDRAWINGVARS(d, intRepPtr, hdc)
    FontFamiliesEnumParam param;

    param.d = d;
    param.enumProc = enumProc;
    param.encoding = NULL; /* system-specific encoding */
    param.clientData = clientData;

    if (name) {
	Tcl_DString buf;

	Tcl_DStringInit(&buf);
	Tcl_UtfToExternalDString(NULL /* system-specific encoding */, 
	    name, -1, &buf);
	EnumFontFamiliesA(hdc, Tcl_DStringValue(&buf), 
	    (FONTENUMPROCA) FontFamiliesEnumProc, (LPARAM) &param);
	Tcl_DStringFree(&buf);
    } else {
	EnumFontFamiliesA(hdc, NULL, 
	    (FONTENUMPROCA) FontFamiliesEnumProc, (LPARAM) &param);
    }
}


static int CALLBACK
FontFamilyExistsProc(
    ENUMLOGFONT *lfPtr,		/* Logical-font data. */
    NEWTEXTMETRIC *tmPtr,	/* Physical-font data (not used). */
    int fontType,		/* Type of font (not used). */
    LPARAM lParam)		/* Result object to hold result. */
{
    int *result = (int *) lParam;
    *result = 1;
    return 0;
}

int 
WinTkGS_FontFamilyExistsUni(d, name)
    TkGS_Drawable		d;
    CONST char			*name;
{
    DECLAREDRAWINGVARS(d, intRepPtr, hdc)
    Tcl_DString buf;
    int result=0;

    /*
     * Just immediately rule out the following fonts, because they look so
     * ugly on windows.  The caller's fallback mechanism will cause the
     * corresponding appropriate TrueType fonts to be selected.
     */

    if (strcasecmp(name, "Courier") == 0) {
	return 0;
    }
    if (strcasecmp(name, "Times") == 0) {
	return 0;
    }
    if (strcasecmp(name, "Helvetica") == 0) {
	return 0;
    }


    Tcl_DStringInit(&buf);
    Tcl_UtfToUniCharDString(name, -1, &buf);
    EnumFontFamiliesW(hdc, (WCHAR *) Tcl_DStringValue(&buf), 
	(FONTENUMPROCW) FontFamilyExistsProc, (LPARAM) &result);
    Tcl_DStringFree(&buf);

    return result;
}

int 
WinTkGS_FontFamilyExistsSys(d, name)
    TkGS_Drawable		d;
    CONST char			*name;
{
    DECLAREDRAWINGVARS(d, intRepPtr, hdc)
    Tcl_DString buf;
    int result=0;

    /*
     * Just immediately rule out the following fonts, because they look so
     * ugly on windows.  The caller's fallback mechanism will cause the
     * corresponding appropriate TrueType fonts to be selected.
     */

    if (strcasecmp(name, "Courier") == 0) {
	return 0;
    }
    if (strcasecmp(name, "Times") == 0) {
	return 0;
    }
    if (strcasecmp(name, "Helvetica") == 0) {
	return 0;
    }

    
    Tcl_DStringInit(&buf);
    Tcl_UtfToExternalDString(NULL /* system-specific encoding */, 
	name, -1, &buf);
    EnumFontFamiliesA(hdc, Tcl_DStringValue(&buf), 
	(FONTENUMPROCA) FontFamilyExistsProc, (LPARAM) &result);
    Tcl_DStringFree(&buf);

    return result;
}

void 
WinTkGS_GetFontMetrics(d, font, fmPtr)
    TkGS_Drawable d;
    TkGS_Font font;
    TkGS_FontMetrics *fmPtr;
{
    WinFont *winFontPtr = WinTkGS_GetWinFontFromFont(NULL, d, font);
    if (!winFontPtr) {
	/* Failed! */
	memset(fmPtr, 0, sizeof(TkGS_FontMetrics));
    	return;
    }

    *fmPtr = winFontPtr->fm;
}


void 
WinTkGS_GetActualFontAttributes(d, font, faPtr)
    TkGS_Drawable d;
    TkGS_Font font;
    TkGS_FontAttributes *faPtr;
{
    WinFont *winFontPtr = WinTkGS_GetWinFontFromFont(NULL, d, font);
    if (!winFontPtr) {
	/* Failed! */
	memset(faPtr, 0, sizeof(TkGS_FontAttributes));
	return;
    }

    *faPtr = winFontPtr->fa;
}

/*
 * Unicode font primitives.
 */

int
WinTkGS_MeasureCharsUni(d, string, length, maxPixels, flags, lengthPtr)
    TkGS_Drawable	d;
    CONST Tcl_UniChar	*string;
    int			length;
    int			maxPixels;
    int			flags;
    int			*lengthPtr;
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    /* TODO: handle case where we use the default font. */
    register TkGSMultiFont multiFont = (TkGSMultiFont) WinTkGS_GetWinFontFromFont(NULL, d, gcValues->font);

    return TkGSMultiFontMeasureCharsUni(d, gcValues->font, multiFont, 
	string, length, maxPixels, flags, lengthPtr);
}

void
WinTkGS_DrawCharsUni(d, string, length, x, y, widthPtr)
    TkGS_Drawable	d;
    CONST Tcl_UniChar	*string;
    int			length;
    int			x;
    int			y;
    int			*widthPtr;
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    /* TODO: handle case where we use the default font. */
    register TkGSMultiFont multiFont = (TkGSMultiFont) WinTkGS_GetWinFontFromFont(NULL, d, gcValues->font);

    TkGSMultiFontDrawCharsUni(d, gcValues->font, multiFont, 
	string, length, x, y, widthPtr);
}


/*
 * UTF-8 font primitives.
 */

int
WinTkGS_MeasureCharsUtf(d, string, length, maxPixels, flags, lengthPtr)
    TkGS_Drawable	d;
    CONST char		*string;
    int			length;
    int			maxPixels;
    int			flags;
    int			*lengthPtr;
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    /* TODO: handle case where we use the default font. */
    register TkGSMultiFont multiFont = (TkGSMultiFont) WinTkGS_GetWinFontFromFont(NULL, d, gcValues->font);

    return TkGSMultiFontMeasureCharsUtf(d, gcValues->font, multiFont, 
	string, length, maxPixels, flags, lengthPtr);
}

void
WinTkGS_DrawCharsUtf(d, string, length, x, y, widthPtr)
    TkGS_Drawable	d;
    CONST char		*string;
    int			length;
    int			x;
    int			y;
    int			*widthPtr;
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    /* TODO: handle case where we use the default font. */
    register TkGSMultiFont multiFont = (TkGSMultiFont) WinTkGS_GetWinFontFromFont(NULL, d, gcValues->font);

    TkGSMultiFontDrawCharsUtf(d, gcValues->font, multiFont, 
	string, length, x, y, widthPtr);
}


/*
 * System-specific font primitives.
 */

int
WinTkGS_TextWidthSys(d, string, length)
    TkGS_Drawable	d;
    CONST char		*string;
    int			length;
{
    DECLAREDRAWINGVARS(d, intRepPtr, hdc)
    SIZE size;

    TkGSSetDrawableSubFont(d, TKGS_BASE_SUBFONT);
    GetTextExtentPoint32A(hdc, string, length, &size);
    return size.cx;
}


void
WinTkGS_DrawCharsSys(d, string, length, x, y, widthPtr)
    TkGS_Drawable	d;
    CONST char		*string;
    int			length;
    int			x;
    int			y;
    int			*widthPtr;
{
    DECLAREDRAWINGVARS(d, intRepPtr, hdc)

    TkGSSetDrawableSubFont(d, TKGS_BASE_SUBFONT);
    TextOutA(hdc, x, y, string, length);
    if (widthPtr) {
	SIZE size;
	GetTextExtentPoint32A(hdc, string, length, &size);
	*widthPtr = size.cx;
    }
}

/*
 * Multi-font Unicode primitives.
 */

TkGSSubFont 
WinTkGSGetSubFont(d, multiFont, subFontId)
    TkGS_Drawable d;
    TkGSMultiFont multiFont;
    TkGSSubFontId subFontId;
{
    return (TkGSSubFont) WinTkGSGetSubFontFromId(multiFont, subFontId);
}


void 
WinTkGSSetDrawableSubFont(d, subFontId)
    TkGS_Drawable d;
    TkGSSubFontId subFontId;
{
    DECLAREDRAWINGVARS(d, intRepPtr, hdc)
    WinSubFont *winSubFontPtr;

    if (WinDrawable_SubFontId(intRepPtr) == subFontId)
	return;

    WinDrawable_SubFontId(intRepPtr) = subFontId;

    /* Select subfont's HFONT. */
    winSubFontPtr = (WinSubFont *) GetDrawableSubFont(d, intRepPtr);
    SelectObject(hdc, winSubFontPtr->hFont);
}

TkGSSubFontId 
WinTkGSDefaultSubFont(d, font, multiFont) 
    TkGS_Drawable d;
    TkGS_Font font;
    TkGSMultiFont multiFont;
{
    return TKGS_BASE_SUBFONT;
}

void 
WinTkGSFontMapLoadPage(d, subFont, row)
    TkGS_Drawable d;
    TkGSSubFont subFont;	/* Contains font mapping cache to be 
				 * updated. */
    int row;			/* Index of the page to be loaded into 
				 * the cache. */
{
    WinFontFamily *familyPtr;
    Tcl_Encoding encoding;
    char src[TCL_UTF_MAX], buf[16];
    USHORT *startCount, *endCount;
    int i, j, bitOffset, end, segCount;

    familyPtr = (WinFontFamily *) subFont->family;
    encoding = familyPtr->header.encoding;

    if (encoding == unicodeEncoding) {
	/*
	 * Font is Unicode.  Few fonts are going to have all characters, so 
	 * examine the TrueType character existence metrics to determine 
	 * what characters actually exist in this font.
	 */

	segCount    = familyPtr->segCount;
	startCount  = familyPtr->startCount;
	endCount    = familyPtr->endCount;

	j = 0;
	end = (row + 1) << FONTMAP_SHIFT;
	for (i = row << FONTMAP_SHIFT; i < end; i++) {
	    for ( ; j < segCount; j++) {
		if (endCount[j] >= i) {
		    if (startCount[j] <= i) {
			bitOffset = i & (FONTMAP_BITSPERPAGE - 1);
			subFont->fontMap[row][bitOffset >> 3] |= 1 << (bitOffset & 7);
		    }
		    break;
		}
	    }
	}
    } else if (familyPtr->isSymbolFont) {
	/*
	 * Assume that a symbol font with a known encoding has all the 
	 * characters that its encoding claims it supports.  
	 *	 
	 * The test for "encoding == unicodeEncoding"
	 * must occur before this case, to catch all symbol fonts (such 
	 * as {Comic Sans MS} or Wingdings) for which we don't have 
	 * encoding information; those symbol fonts are treated as if
	 * they were in the Unicode encoding and their symbolic
	 * character existence metrics are treated as if they were Unicode
	 * character existence metrics.  This way, although we don't know
	 * the proper Unicode -> symbol font mapping, we can install the
	 * symbol font as the base font and access its glyphs.
	 */

        end = (row + 1) << FONTMAP_SHIFT;
	for (i = row << FONTMAP_SHIFT; i < end; i++) {
	    if (Tcl_UtfToExternal(NULL, encoding, src, 
		    Tcl_UniCharToUtf(i, src), TCL_ENCODING_STOPONERROR, NULL, 
		    buf, sizeof(buf), NULL, NULL, NULL) != TCL_OK) {
		continue;
	    }
	    bitOffset = i & (FONTMAP_BITSPERPAGE - 1);
	    subFont->fontMap[row][bitOffset >> 3] |= 1 << (bitOffset & 7);
	}
    }
}


TkGSSubFontId
WinTkGSCanUseFont(d, font, multiFont, ch) 
    TkGS_Drawable d;
    TkGS_Font font;
    TkGSMultiFont multiFont;
    TkGS_UniChar ch;
{
    int i;
    
    if (ch < WIN_BASE_CHARS) {
	return TKGS_BASE_SUBFONT;
    }
    for (i = 0; i < multiFont->numSubFonts; i++) {
	if (TkGSFontMapLookup(d, (TkGSSubFont) WinTkGSGetSubFontFromIndex(multiFont, i), ch)) {
	    return WinTkGSGetSubFontId(i);
	}
    }
    return 0;
}


/*  - Unicode version */

int
WinTkGSSubFontMeasureCharsUni(d, string, length, maxPixels, flags, lengthPtr)
    TkGS_Drawable	d;
    CONST Tcl_UniChar	*string;
    int			length;
    int			maxPixels;
    int			flags;
    int			*lengthPtr;
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    WinFont *winFontPtr = WinTkGS_GetWinFontFromFont(NULL, d, gcValues->font);
    Tcl_UniChar ch;
    CONST Tcl_UniChar *p, *end, *next;
    CONST Tcl_UniChar *term;
    int newX, termX, sawNonSpace;
    int curX, curByte;

    /*
     * This first version may be inefficient because it measures
     * every character individually. There is a function call that
     * can measure multiple characters at once and return the
     * offset of each of them, but it only works on NT, even though
     * the documentation claims it works for 95.
     * TODO: verify that GetTextExtentExPoint is still broken in '95, and
     * possibly use it for NT anyway since it should be much faster and
     * more accurate.
     */

    ch = *string;
    next = string + 1;
    newX = curX = termX = 0;
    
    term = string;
    end = string + length;

    sawNonSpace = (ch > 255) || !isspace(ch);
    for (p = string; ; ) {
	if (ch < WIN_BASE_CHARS) {
	    newX += winFontPtr->widths[ch];
	} else {
	    newX += TkGSSubFontTextWidthUni(d, p, next - p);
	}
	if (newX > maxPixels) {
	    break;
	}
	curX = newX;
	p = next;
	if (p >= end) {
	    term = end;
	    termX = curX;
	    break;
	}

	ch = *next;
	next++;
	if ((ch < 256) && isspace(ch)) {
	    if (sawNonSpace) {
		term = p;
		termX = curX;
		sawNonSpace = 0;
	    }
	} else {
	    sawNonSpace = 1;
	}
    }

    /*
     * P points to the first character that doesn't fit in the desired
     * span.  Use the flags to figure out what to return.
     */

    if ((flags & TK_PARTIAL_OK) && (p < end) && (curX < maxPixels)) {
	/*
	 * Include the first character that didn't quite fit in the desired
	 * span.  The width returned will include the width of that extra
	 * character.
	 */

	curX = newX;
	p++;
    }
    if ((flags & TK_AT_LEAST_ONE) && (term == string) && (p < end)) {
	term = p;
	termX = curX;
	if (term == string) {
	    term++;
	    termX = newX;
	}
    } else if ((p >= end) || !(flags & TK_WHOLE_WORDS)) {
	term = p;
	termX = curX;
    }

    curX = termX;
    curByte = term - string;

    *lengthPtr = curX;
    if (term >= end) {
	return length;
    } else {
	return term-string;
    }
}



int
WinTkGSSubFontTextWidthUni(d, string, length)
    TkGS_Drawable	d;
    CONST Tcl_UniChar	*string;
    int			length;
{
    DECLAREDRAWINGVARS(d, intRepPtr, hdc)
    TkGSSubFont subFont = GetDrawableSubFont(d, intRepPtr);
    WinFontFamily *familyPtr = (WinFontFamily *) subFont->family;
    SIZE size;

    if (familyPtr->isWideFont) {
	/* Unicode encoded font. */
	GetTextExtentPoint32W(hdc, (WCHAR *) string, length, &size);
    } else {
	/* Font-specific encoding. */
	Tcl_DString dsUtf, ds;

	Tcl_DStringInit(&dsUtf);
	Tcl_DStringInit(&ds);
	Tcl_UniCharToUtfDString(string, length, &dsUtf);
        Tcl_UtfToExternalDString(familyPtr->header.encoding, 
		Tcl_DStringValue(&dsUtf), Tcl_DStringLength(&dsUtf), &ds);
	GetTextExtentPoint32A(hdc, Tcl_DStringValue(&ds), 
		Tcl_DStringLength(&ds), &size);
	Tcl_DStringFree(&ds);
	Tcl_DStringFree(&dsUtf);
    }
    return size.cx;
}

void
WinTkGSSubFontDrawCharsUni(d, string, length, x, y, widthPtr)
    TkGS_Drawable	d;
    CONST Tcl_UniChar	*string;
    int			length;
    int			x;
    int			y;
    int			*widthPtr;
{
    DECLAREDRAWINGVARS(d, intRepPtr, hdc)
    TkGSSubFont subFont = GetDrawableSubFont(d, intRepPtr);
    WinFontFamily *familyPtr = (WinFontFamily *) subFont->family;
    SIZE size;

    if (familyPtr->isWideFont) {
	/* Unicode encoded font. */
	TextOutW(hdc, x, y, (WCHAR *) string, length);
	if (widthPtr) {
	    GetTextExtentPoint32W(hdc, (WCHAR *) string, length, &size);
	    *widthPtr = size.cx;
	}
    } else {
	/* Font-specific encoding. */
	Tcl_DString dsUtf, ds;

	Tcl_DStringInit(&dsUtf);
	Tcl_DStringInit(&ds);
	Tcl_UniCharToUtfDString(string, length, &dsUtf);
        Tcl_UtfToExternalDString(familyPtr->header.encoding, 
		Tcl_DStringValue(&dsUtf), Tcl_DStringLength(&dsUtf), &ds);
	TextOutA(hdc, x, y, Tcl_DStringValue(&ds), Tcl_DStringLength(&ds));
	if (widthPtr) {
	    GetTextExtentPoint32A(hdc, Tcl_DStringValue(&ds), 
		    Tcl_DStringLength(&ds), &size);
	    *widthPtr = size.cx;
	}
	Tcl_DStringFree(&ds);
	Tcl_DStringFree(&dsUtf);
    }
}


/*  - UTF-8 version */

int
WinTkGSSubFontMeasureCharsUtf(d, string, length, maxPixels, flags, lengthPtr)
    TkGS_Drawable	d;
    CONST char		*string;
    int			length;
    int			maxPixels;
    int			flags;
    int			*lengthPtr;
{
    register TkGS_GCValues *gcValues = &TkGSDrawable_GCValues(d);
    WinFont *winFontPtr = WinTkGS_GetWinFontFromFont(NULL, d, gcValues->font);
    Tcl_UniChar ch;
    CONST char *p, *end, *next;
    CONST char *term;
    int newX, termX, sawNonSpace;
    int curX, curByte;

    /*
     * This first version may be inefficient because it measures
     * every character individually. There is a function call that
     * can measure multiple characters at once and return the
     * offset of each of them, but it only works on NT, even though
     * the documentation claims it works for 95.
     * TODO: verify that GetTextExtentExPoint is still broken in '95, and
     * possibly use it for NT anyway since it should be much faster and
     * more accurate.
     */

    next = string + Tcl_UtfToUniChar(string, &ch);
    newX = curX = termX = 0;
    
    term = string;
    end = string + length;

    sawNonSpace = (ch > 255) || !isspace(ch);
    for (p = string; ; ) {
	if (ch < WIN_BASE_CHARS) {
	    newX += winFontPtr->widths[ch];
	} else {
	    newX += TkGSSubFontTextWidthUtf(d, p, next - p);
	}
	if (newX > maxPixels) {
	    break;
	}
	curX = newX;
	p = next;
	if (p >= end) {
	    term = end;
	    termX = curX;
	    break;
	}

	next += Tcl_UtfToUniChar(next, &ch);
	if ((ch < 256) && isspace(ch)) {
	    if (sawNonSpace) {
		term = p;
		termX = curX;
		sawNonSpace = 0;
	    }
	} else {
	    sawNonSpace = 1;
	}
    }

    /*
     * P points to the first character that doesn't fit in the desired
     * span.  Use the flags to figure out what to return.
     */

    if ((flags & TK_PARTIAL_OK) && (p < end) && (curX < maxPixels)) {
	/*
	 * Include the first character that didn't quite fit in the desired
	 * span.  The width returned will include the width of that extra
	 * character.
	 */

	curX = newX;
	p += Tcl_UtfToUniChar(p, &ch);
    }
    if ((flags & TK_AT_LEAST_ONE) && (term == string) && (p < end)) {
	term = p;
	termX = curX;
	if (term == string) {
	    term += Tcl_UtfToUniChar(term, &ch);
	    termX = newX;
	}
    } else if ((p >= end) || !(flags & TK_WHOLE_WORDS)) {
	term = p;
	termX = curX;
    }

    curX = termX;
    curByte = term - string;

    *lengthPtr = curX;
    if (term >= end) {
	return length;
    } else {
	return term-string;
    }
}

int
WinTkGSSubFontTextWidthUtf(d, string, length)
    TkGS_Drawable	d;
    CONST char		*string;
    int			length;
{
    DECLAREDRAWINGVARS(d, intRepPtr, hdc)
    TkGSSubFont subFont = GetDrawableSubFont(d, intRepPtr);
    WinFontFamily *familyPtr = (WinFontFamily *) subFont->family;
    SIZE size;
    Tcl_DString ds;

    Tcl_DStringInit(&ds);
    if (familyPtr->isWideFont) {
	/* Unicode encoded font. */
	Tcl_UtfToUniCharDString(string, length, &ds);
	GetTextExtentPoint32W(hdc, (WCHAR *) Tcl_DStringValue(&ds), 
		Tcl_DStringLength(&ds) / sizeof(WCHAR), &size);
    } else {
	/* Font-specific encoding. */
        Tcl_UtfToExternalDString(familyPtr->header.encoding, 
		string, length, &ds);
	GetTextExtentPoint32A(hdc, Tcl_DStringValue(&ds), 
		Tcl_DStringLength(&ds), &size);
    }
    Tcl_DStringFree(&ds);
    return size.cx;
}

void
WinTkGSSubFontDrawCharsUtf(d, string, length, x, y, widthPtr)
    TkGS_Drawable	d;
    CONST char		*string;
    int			length;
    int			x;
    int			y;
    int			*widthPtr;
{
    DECLAREDRAWINGVARS(d, intRepPtr, hdc)
    TkGSSubFont subFont = GetDrawableSubFont(d, intRepPtr);
    WinFontFamily *familyPtr = (WinFontFamily *) subFont->family;
    SIZE size;
    Tcl_DString ds;

    Tcl_DStringInit(&ds);
    if (familyPtr->isWideFont) {
	/* Unicode encoded font. */
	Tcl_UtfToUniCharDString(string, length, &ds);
	TextOutW(hdc, x, y, (WCHAR *) Tcl_DStringValue(&ds), 
		    Tcl_DStringLength(&ds) / sizeof(WCHAR));
	if (widthPtr) {
	    GetTextExtentPoint32W(hdc, (WCHAR *) Tcl_DStringValue(&ds), 
		    Tcl_DStringLength(&ds) / sizeof(WCHAR), &size);
	    *widthPtr = size.cx;
	}
    } else {
	/* Font-specific encoding. */
        Tcl_UtfToExternalDString(familyPtr->header.encoding, 
		string, length, &ds);
	TextOutA(hdc, x, y, Tcl_DStringValue(&ds), Tcl_DStringLength(&ds));
	if (widthPtr) {
	    GetTextExtentPoint32A(hdc, Tcl_DStringValue(&ds), 
		    Tcl_DStringLength(&ds), &size);
	    *widthPtr = size.cx;
	}
    }
    Tcl_DStringFree(&ds);
}

/*
 *-------------------------------------------------------------------------
 * 
 * WinTkGSInitFontPackage --
 *
 *	This procedure is called when an application loads the device 
 *	driver.  It initializes all the structures that are used by the 
 *	platform-dependent code on a per application basis.
 *
 * Results:
 *	None.  
 *
 * Side effects:
 *	See comments below.
 *
 *-------------------------------------------------------------------------
 */

void
WinTkGSInitFontPackage() {
    ThreadSpecificData *tsdPtr = (ThreadSpecificData *) 
            Tcl_GetThreadData(&dataKey, sizeof(ThreadSpecificData));

    /* Font families are uniquely identified by their face name. */
    Tcl_InitHashTable(&tsdPtr->fontFamilyTable, TCL_STRING_KEYS);

    unicodeEncoding = Tcl_GetEncoding(NULL, "unicode");
}

