/*
 * Copyright 1990,1991 the Regents of the University of California.  All
 * rights reserved.  Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without fee is
 * hereby granted, provided that this copyright notice appear in all
 * copies.  See the file copyright.h for more information.
 *
 */

#include <stdio.h>
#include "xwrap.h"

#include "debug.h"
#include "global.h"
#include "util.h"
#include "patchlevel.h"
#include "object.h"

Tcl_Interp *mainInterp;
void InterpretTheCommand();
char currentcommand[MAXLINE];

/* the following routines are for tcl create command.  Arguements are:
     char *clientData;      unused
     Tcl_Interp *interp;    interpreter to place error messages
     int argc;              number of arguements
     char **argv;           arguement list */


static int ParseQuit(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char usage[MAXLINE];
	entering("ParseQuit");
	sprintf(usage,"%s",argv[0]);
	if (BadUsage(argc,1,usage,interp)) {
		myreturn TCL_ERROR;
	}
	Quit();
	myreturn TCL_OK;
}

static int ParseEmptyEventLoop(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char usage[MAXLINE];
	entering("ParseEmptyEventLoop");
	sprintf(usage,"%s",argv[0]);
	if (BadUsage(argc,1,usage,interp)) {
		myreturn TCL_ERROR;
	}
	EmptyEventLoop();
	myreturn TCL_OK;
}

static int ParseLowercase(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
     /* returns a lowercase version of argv[1] */
{
	char usage[MAXLINE];
	char *p;
	entering("ParseLowercase");
	sprintf(usage,"%s string",argv[0]);
	if ((BadUsage(argc,2,usage,interp))) {
		myreturn TCL_ERROR;
	}
	p = argv[1];
	while (*p) {
		if ((*p>='A') && (*p <= 'Z')) *p = *p + ' ';
		p++;
	}
	Tcl_Return(interp,argv[1],TCL_VOLATILE);
	myreturn TCL_OK;
}

static int ParseIconify(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char usage[MAXLINE];
	entering("ParseIconify");
	sprintf(usage,"%s",argv[0]);
	if (BadUsage(argc,1,usage,interp)) {
		myreturn TCL_ERROR;
	}
	XIconifyWindow(XtDisplay(TopLevel),XtWindow(TopLevel),
		       XScreenNumberOfScreen(XtScreen(TopLevel)));
	myreturn TCL_OK;
}

static int ParseUniconify(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char usage[MAXLINE];
	entering("ParseUniconify");
	sprintf(usage,"%s",argv[0]);
	if (BadUsage(argc,1,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if (XtIsRealized(TopLevel)) {
		XtMapWidget(TopLevel);
	}
	myreturn TCL_OK;
}

static int ParseDebuglevel(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char usage[MAXLINE];
	entering("ParseDebuglevel");
	sprintf(usage,"%s #",argv[0]);
	if (BadUsage(argc,2,usage,interp)) {
		myreturn TCL_ERROR;
	}
	set_debuglevel(atoi(argv[1]));
	myreturn TCL_OK;
}

static int ParseArgc(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char result[MAXLINE],usage[MAXLINE];
	entering("ParseArgc");
	sprintf(usage,"%s",argv[0]);
	if (BadUsage(argc,1,usage,interp)) {
		myreturn TCL_ERROR;
	}
	sprintf(result,"%d",GlobalArgc);
	Tcl_Return(interp,result,TCL_VOLATILE);
	myreturn TCL_OK;
}

static int ParseArgv(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char result[MAXLINE],usage[MAXLINE];
	entering("ParseArgv");
	sprintf(usage,"%s #",argv[0]);
	if (BadUsage(argc,2,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if ((atoi(argv[1])<GlobalArgc)&&(atoi(argv[1])>=0)) {
		sprintf(result,"%s",GlobalArgv[atoi(argv[1])]);
	} else {
		sprintf(result," ");
	}
	Tcl_Return(interp,result,TCL_VOLATILE);
	myreturn TCL_OK;
}

long random();

static int ParseRandom(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
     /* Returns a random number between 0 and argv[1]-1, inclusive */
{
	char result[MAXLINE],usage[MAXLINE];
	entering("ParseRandom");
	sprintf(usage,"%s number",argv[0]);
	if (BadUsage(argc,2,usage,interp)) {
		myreturn TCL_ERROR;
	}
	sprintf(result,"%d",(random()&32767)*abs(atoi(argv[1]))/32767);
	Tcl_Return(interp,result,TCL_VOLATILE);
	myreturn TCL_OK;
}

static int ParsePaste(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
     /* puts the contents of cut_buffer 0 in interp->result or
	puts the contents of argv[1] in cut_buffer 0 */
{
	String contents;
	int n;
        char usage[MAXLINE];
	entering("ParsePaste");
	sprintf(usage,"%s [string]",argv[0]);
	if (BadUsageOptional(argc,2,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if (argc==1) {
		contents = XFetchBuffer(XtDisplay(TopLevel),&n,0);
		if (n>0) {
			Tcl_Return(interp,contents,TCL_VOLATILE);
		}
	}
	else
	  XStoreBuffer(XtDisplay(TopLevel),(argv[1]),strlen(argv[1]),0);
	myreturn TCL_OK;
}

static int ParseSetupfile(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
     /* returns the name of the setup file orginially read in.
	This is useful for determining some sort of default directory */
{
	char usage[MAXLINE];
	entering("ParseSetupfile");
	sprintf(usage,"%s",argv[0]);
	if (BadUsage(argc,1,usage,interp)) {
		myreturn TCL_ERROR;
	}
	Tcl_Return(interp,MyResources.file,TCL_VOLATILE);
	myreturn TCL_OK;
}

static int ParseLocateFile(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
     /* returns the setupPath resource */
{
	char *file, *dirlist, result[MAXLINE],usage[MAXLINE];

	entering("ParseLocateFile");
	sprintf(usage,"%s file [directory-list]",argv[0]);
	if (BadUsageOptional(argc,3,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if (argc == 3)
		dirlist = argv[2];
	else {
		dirlist = Tcl_GetVar(interp, "setuppath", 1);
		if (!dirlist) {
			sprintf(result, "Variable \"setuppath\" not defined!");
			Tcl_Return(interp,result,TCL_VOLATILE);
			myreturn TCL_ERROR;
		}
	}
	file = LocateFile(argv[1], dirlist);
	if (file) {
		Tcl_Return(interp,file,TCL_VOLATILE);
	} else {
		sprintf(result,"");
		Tcl_Return(interp,result,TCL_VOLATILE);
	}
	myreturn TCL_OK;
}

static int ParseListFiles(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
     /* usage is 'locatefile filename [directory-list]'.  If
	directory-list is not specified, then search the global tcl
	variable called setuppath. */
{
	char **files;
	char *result;
	char *ptr;
	int numfiles;
	char usage[MAXLINE];

	entering("ParseListFiles");
	sprintf(usage,"%s directory-name",argv[0]);
	if (BadUsage(argc,2, usage,interp)) {
		myreturn TCL_ERROR;
	}
	numfiles = ListFiles(argv[1],&files);
	if (numfiles>0) {
		qsort((char*)files,numfiles,sizeof(char*),ptrstrcmp);
		result = Tcl_Merge(numfiles,files);
		for (ptr = (*files); ptr==NULL; ptr=(*files++)) {
			XtFree(ptr);
		}
		if (*files) {XtFree(files);}
		Tcl_Return(interp,result,TCL_VOLATILE);
	        if (*result) {XtFree(result);};
	} else {
		result=NULL;
		Tcl_Return(interp,NULL,TCL_STATIC);
	}
	/* tell tcl to free the result since Tcl_Merge mallocs it */
	myreturn TCL_OK;
}

static int ParseWait(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
     /* waits for argv[1] seconds */
{
	char usage[MAXLINE];
	entering("ParseWait");
	sprintf(usage,"%s #",argv[0]);
	if (BadUsage(argc,2,usage,interp)) {
		myreturn TCL_ERROR;
	}
	EmptyEventLoop();
	sleep((unsigned)abs(atoi(argv[1])));
	EmptyEventLoop();
	myreturn TCL_OK;
}

static int ParseVersion(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char result[MAXLINE],usage[MAXLINE];
	entering("ParseVersion");
	sprintf(usage,"%s",argv[0]);
	if (BadUsage(argc,1,usage,interp)) {
		myreturn TCL_ERROR;
	}
	sprintf(result,"%s rev. %s",VERSION,PATCHLEVEL);
	Tcl_Return(interp,result,TCL_VOLATILE);
	myreturn TCL_OK;
}

int InitializeInterpreters()
{
	Tcl_Interp *interp;
	entering("InitializeInterpreters");
	srandom(getpid());  /* initialize random number generator for
			       random command */
	mainInterp = Tcl_CreateInterp();
	interp = mainInterp;  /* copy needed because the CREATE macro uses 
				 the word 'interp' */
	CREATE("quit",ParseQuit);                    /* quit */
	CREATE("emptyeventloop",ParseEmptyEventLoop);/* emptyeventloop */
	CREATE("lowercase",ParseLowercase);          /* lowercase word */
	CREATE("iconify",ParseIconify);              /* iconify */
	CREATE("uniconify",ParseUniconify);          /* uniconify */
	CREATE("debuglevel",ParseDebuglevel);        /* debuglevel # */
	CREATE("argc",ParseArgc);                    /* argc */
	CREATE("argv",ParseArgv);                    /* argv # */
	CREATE("random",ParseRandom);                /* random # */
	CREATE("paste",ParsePaste);                  /* paste */
	CREATE("setupfile",ParseSetupfile);          /* setupfile */
	CREATE("locatefile",ParseLocateFile);        /* setuppath */
	CREATE("listfiles",ParseListFiles);          /* listfiles dir-name */
	CREATE("wait",ParseWait);                    /* wait */
	CREATE("version",ParseVersion);              /* version */
	AddWidgetHandlingCommands(mainInterp);       /* found in object.c */
	myreturn;
}

int InterpretCommand(cmd)
     char *cmd;
     /* this is the main command interpreter.  Everything calls this
	routine for interpreting xcbi commands. Sometimes this routine is
	called recursively so it must return a TCL error value. */
{
	int result;
	entering("InterpretCommand");
	dprint(cmd);
	if (!cmd) {
		myreturn TCL_OK;
	}
	/* initialize tcl results to NULL */
	Tcl_Return(mainInterp,NULL,TCL_STATIC);
	result = Tcl_RecordAndEval(mainInterp,cmd,0);
	if (result==TCL_OK) {
		if (*mainInterp->result != 0) {
			printf("%s\n",mainInterp->result);
		}
	} else {
		printf("Error");
		if (*mainInterp->result != 0) {
			printf(": %s\n",mainInterp->result);
		} else {
			printf("\n");
		}
	}
	myreturn result;
}
