/* object.c
 *
 * Copyright 1990,1991 the Regents of the University of California.  All
 * rights reserved.  Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without fee is
 * hereby granted, provided that this copyright notice appear in all
 * copies.  See the file copyright.h for more information.
 *
 */

/*
 * contains general object manipulation commands.
 *
 */

#include <stdio.h>
#include "xwrap.h"
#include <X11/IntrinsicP.h>
#include <X11/ObjectP.h>         /* Sun Open Windows IntrinsicP.h doesn't
				    include this so include by hand */
#include <X11/Xaw/TextP.h>       /* to get XtNScrollMode, etc. */
#include <X11/Xmu/Converters.h>  /* for converting from string to ...  */

#include "debug.h"
#include "global.h"
#include "util.h"
#include "object.h"

Widget DebugWidget=NULL;

Widget WidgetExists(name,interp) 
     char *name;
     Tcl_Interp *interp;
     /* tests to see if widget named 'name' exists.  If it doesn't,
	then put a result string in interp.  Returns the widget or NULL.
	A bad widget name could be very long so need to malloc space for 
	the result. */
{
	char *result,*newname;
	Widget w;
	entering("WidgetExists");
	/* skip any widget name that begins with top. */
	if (strncmp(name,"top.",4)==0) {
		newname = &name[4];
	} else {
		newname = name;
	}
	if (!(w=MyFindWidgetByName(newname))) {
		result = XtMalloc(strlen(currentcommand)+strlen(newname)+50);
		sprintf(result,
			"%s: There is no widget called: %s\n",
			currentcommand,newname);
		Tcl_Return(interp,result,TCL_VOLATILE);
		XtFree(result);
	}
	myreturn w;
}

Widget WidgetSiblingOrParent(w,name,interp) 
     Widget w;
     char *name;
     Tcl_Interp *interp;
     /* tests to see if widget named 'name' is a sibling or parent of w.  
	If it isn't then put a result string in interp.  
	Returns the widget or NULL */
{
	char result[MAXLINE];
	Widget child;
	entering("WidgetSiblingOrParent");
	if (mystrcmp(XtName(XtParent(w)),name)==0) {
		myreturn XtParent(w);
	}
	if (!(child=XtNameToWidget(XtParent(w),name))) {
		sprintf(result,
			"%s: '%s' is not a parent or child of '%s'.\n",
			name,currentcommand,XtName(w));
		Tcl_Return(interp,result,TCL_VOLATILE);
	}
	myreturn child;
}

int InvalidType(name,type,interp)
     char *name;          /* name of object */
     WidgetClass type;      /* type to test */
     Tcl_Interp *interp;  /* interpreter to place error message */
     /* Returns True if object named 'name' is not of class
	'type'.  Writes an error message to interp->result. */
{
	char result[MAXLINE];
	Widget w;
	entering("InvalidType");
	if (!(w=WidgetExists(name,interp))) {
		myreturn True;
	}
	if (XtClass(w)!=type) {
		sprintf(result,
			"%s: Object '%s' is not of the correct type.\n",
			currentcommand,name);
		Tcl_Return(interp,result,TCL_VOLATILE);
		myreturn True;
	}
	myreturn False;
}

void SetCurrentCommand(msg)
     char *msg;               /* full command message */
     /* sets the global variable, currentcommand, to the first word of msg.
	and the global tcl variable usagecommand to the msg */
{
	char *r;
	char space;
	space = ' ';
	if (r=index(msg,space)) {
		strncpy(currentcommand,msg,strlen(msg)-strlen(r));
		currentcommand[strlen(msg)-strlen(r)]=NULL;
	} else {
		strcpy(currentcommand,msg);
	}
	Tcl_SetVar(mainInterp,"currentcommand",currentcommand,0);
	Tcl_SetVar(mainInterp,"usagecommand",msg,0);
}

int BadUsage(actual,correct,msg,interp)
     int actual;              /* actual number of args */
     int correct;             /* correct number of args */
     char *msg;               /* error message */
     Tcl_Interp *interp;      /* interpreter to place error messages */
     /* if m != n, then set interp->result to hold the message
	"usage is 'msg'." and return True.  Else return False. */
{
	char result[MAXLINE];
	entering("BadUsage");
	SetCurrentCommand(msg);
	if (actual == correct) {
		myreturn False;
	}
	if (actual < correct) {
		sprintf (result,"Too few arguments to %s.\nUsage is '%s'",
			 currentcommand,msg);
		Tcl_Return(interp,result,TCL_VOLATILE);
		myreturn True;
	}
	if (actual > correct) {
		sprintf (result,"Too many arguments to %s.\nUsage is '%s'",
			 currentcommand,msg);
		Tcl_Return(interp,result,TCL_VOLATILE);
		myreturn True;
	}
	myreturn False;
}

int BadUsageOptional(actual,correct,msg,interp)
     int actual;              /* actual number of args */
     int correct;             /* correct number of args */
     char *msg;               /* error message */
     Tcl_Interp *interp;      /* interpreter to place error messages */
     /* Similar to BadUsage but allows for the last argument[s] to be 
	optional */
{
	char result[MAXLINE];
	entering("BadUsageOptional");
	SetCurrentCommand(msg);
	if (actual == correct) {
		myreturn False;
	}
	if (actual < (correct-1)) {
		sprintf (result,"Too few arguments to %s.\nUsage is '%s'",
			 currentcommand,msg);
		Tcl_Return(interp,result,TCL_VOLATILE);
		myreturn True;
	}
	if (actual > correct) {
		myreturn False;
	}
	myreturn False;
}

int InvalidParent(name,parent,interp)
     char *name;
     Widget *parent;
     Tcl_Interp *interp;
     /* checks to make sure there exists a widget named name. */
{
	char result[MAXLINE];
	entering("InvalidParent");
 	if (mystrcmp(name,"top")==0) {
		*parent = TopLevel;
		myreturn False;
	}
	if (!(*parent=WidgetExists(name,interp))) {
		myreturn True;
	}
	/* menu buttons can have a menu popup shell as children. */
	if (XtIsShell((*parent))||XtIsComposite(*parent)||
	    XtIsConstraint(*parent)||
	    (XtClass(*parent)==menuButtonWidgetClass)) {
		myreturn False;
	} else {
		sprintf(result,"Widget '%s' cannot have children.\n",name);
		Tcl_Return(interp,result,TCL_VOLATILE);
		myreturn True;
	}
}


int ParseOperationNoOptions(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
     /* useful for commands of the form 'operation object-name'. */
{
	char usage[MAXLINE],result[MAXLINE];
	Widget w;
	entering("ParseOperationNoOptions");
	sprintf(usage,"%s object-name",argv[0]);
	if (BadUsage(argc,2,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if (!(w=WidgetExists(argv[1],interp))) {
		myreturn TCL_ERROR;
	}
	if (mystrcmp(argv[0],"destroy")==0) {
		XtDestroyWidget(w);
	} else if (mystrcmp(argv[0],"manage")==0) {
		XtManageChild(w);
	} else if (mystrcmp(argv[0],"clear")==0) {
		if ((!InvalidType(argv[1],scToolWidgetClass,interp))&&
		    (!InvalidType(argv[1],asciiTextWidgetClass,interp))) {
			myreturn TCL_ERROR;
		}
		ScClearText(w);
	} else if (mystrcmp(argv[0],"press")==0) {
		if ((!InvalidType(argv[1],commandWidgetClass,interp))&&
		    (!InvalidType(argv[1],listWidgetClass,interp))&&
		    (!InvalidType(argv[1],gripWidgetClass,interp))&&
		    (!InvalidType(argv[1],toggleWidgetClass,interp))&&
		    (!InvalidType(argv[1],menuButtonWidgetClass,interp))) {
			myreturn TCL_ERROR;
		}
		XtCallCallbacks(w,XtNcallback,NULL);
		EmptyEventLoop();
	} else if (mystrcmp(argv[0],"popdown")==0) {
		if (InvalidType(argv[1],transientShellWidgetClass,interp)) {
			myreturn TCL_ERROR;
		}
		XtPopdown(w);
		EmptyEventLoop();
	} else {
		sprintf (result,"Don't know how to apply '%s' to object %s\n.",
			 argv[0],argv[1]);
		Tcl_Return(interp,result,TCL_VOLATILE);
	}
	myreturn TCL_OK;
}

static char *ConstructName(parentname,w)
     char *parentname;
     Widget w;
     /* returns the name of w preceeded by "parentname.".  If parentname
	is NULL, then only the name of w is returned.  The caller is
	responsible for freeing the returned value. */
{
	/* make a name parent.name */
	char *wname,*fullname;
	entering("ConstructName");
	wname = XtName(w);
	if (parentname[0]) {
		fullname = (char*)XtMalloc(sizeof(char)*
					   (strlen(parentname)
					    +strlen(wname)+2));
		sprintf(fullname,"%s.%s",parentname,wname);
	} else {
		/* return only name */
		fullname=XtNewString(wname);
	}
	myreturn fullname;
}

static char *ConcatAndFree(word)
     char **word;
     /* concats word[0] and word[1] and then frees both of them */
{
	char *result;
	int i;
	entering("ConcatAndFree");
	/* if the first word is "" then return word[1] */
	if ((word[0])&&(*word[0])) {
		/* if both words exist, then cat them and free them */
		if ((word[1])&&(*word[1])) {
			result = XtMalloc(strlen(word[0])+strlen(word[1])+2);
			sprintf(result,"%s %s",word[0],word[1]);
			XtFree(word[0]);
			XtFree(word[1]);
		} else {
			result = NULL;
		}
	} else {
		/* else just one word exists so return the second word */
		result = XtNewString(word[1]);
		XtFree(word[1]);
	}
	myreturn result;
}

static char *GetChildNames();

static char *GetChildNames(w,namefromtop,argc,argv)
     Widget w;
     char *namefromtop;
     int argc;
     char **argv;
     /* returns a list of names of w, the popups of w, the children of w
	and all their popups and children which are of the types listed in
	argv.  The namefromtop is used to keep track of all the names to w.
	Form is {namefromtop.w w.popups w.child}. Recursively calls itself. */
{
	int found;
	char *newnamefromtop,*type;
	Widget child,p;
	CompositeWidget c;
	char *names[2],*result;
	int i;
	
	entering("GetChildNames");
	/* start with namefromtop.name */
	if (w==TopLevel) {
		/* the toplevel name is not included in the result */
		names[0]="";
		newnamefromtop="";
	} else if (argc>0) {
		found=False;
		i=0;
		type = ObjectTypeString(w);
		while((i<argc)&&(!found)) {
			found=(mystrcmp(argv[i++],type)==0);
		}
		XtFree(type);
		if (found) {
			names[0]=ConstructName(namefromtop,w);
			newnamefromtop=XtNewString(names[0]);
		} else {
			names[0]="";
			newnamefromtop=ConstructName(namefromtop,w);
		}
	} else {
		names[0]=ConstructName(namefromtop,w);
		newnamefromtop=XtNewString(names[0]);
	}
	/* Gadgets must be handled differently.
	   For now, only worry about smeBSB or smeLine ObjectClass types. */
	if ((XtClass(w)==smeBSBObjectClass)||
	    (XtClass(w)==smeLineObjectClass)) {
		if (*newnamefromtop)
		  XtFree(newnamefromtop);
		myreturn names[0];
	}
	/* append the popup names to result */
	
	for (i=0; i<w->core.num_popups; i++) {
		p = w->core.popup_list[i];
		if (p) {
			names[1]=GetChildNames(p,newnamefromtop,argc,argv);
			result = ConcatAndFree(names);
			names[0]=result;  /* need to keep track of 
					     the previous popup */
		}
	}
	result=names[0];
	/* append the child names to result */
	if (XtIsComposite(w)) {
		c = (CompositeWidget)w;
		for (i=0; i<c->composite.num_children; i++) {
			child=c->composite.children[i];
			if (c) {
				names[1]=GetChildNames(child,
						       newnamefromtop,
						       argc,argv);
				result = ConcatAndFree(names);
				names[0]=result;  /* need to keep track of 
						     the previous children */
			}
		}
	} else {
		result=names[0];
	}
	if (*newnamefromtop)
	  XtFree(newnamefromtop);
	myreturn result;
}
	
static int ParseListNames(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char usage[MAXLINE],*result;
	sprintf(usage,"%s [type1 type2...]",argv[0]);
	entering("ParseListNames");
	if ((BadUsageOptional(argc,2,usage,interp))) {
		myreturn TCL_ERROR;
	}
	result = GetChildNames(TopLevel,"",argc-1,argv+1);
	Tcl_Return(interp,result,TCL_VOLATILE);
	if (*result) {
		XtFree(result);
	}
	myreturn TCL_OK;
}

static char* ExtractParentName(name)
     char *name;
     /* returns a copy of the parent part of name.  If there is no parent,
	then, "top" is returned.  The caller is responsible for freeing
	the memory returned. */
{
	char *endindex,*parentname;
	int length;
	entering("ExtractParentName");
	endindex = rindex(name,'.');
	if (endindex) {
		length = strlen(name)-strlen(endindex);
		parentname = (char *)XtMalloc(sizeof(char)*(length+1));
				
		strncpy(parentname,name,length);
		parentname[length]=NULL;
		myreturn parentname;
	} else {
		myreturn XtNewString("top");
	}
}

char* ObjectTypeString(w)
     Widget w;
     /* returns a string representation of the object type. The
	caller is responsible for freeing the memory. */
{
	WidgetClass c;
	entering("ObjectTypeString");
	c=XtClass(w);
	if (c==transientShellWidgetClass) {
		myreturn(XtNewString("popup"));
	} else if (c==boxWidgetClass) {
		myreturn(XtNewString("box"));
	} else if (c==commandWidgetClass) {
		myreturn(XtNewString("button"));
	} else if (c==scToolWidgetClass) {
		myreturn(XtNewString("text"));
	} else if (c==asciiTextWidgetClass) {
		myreturn(XtNewString("textonly"));
	} else if (c==tableWidgetClass) {
		myreturn(XtNewString("table"));
	} else if (c==gripWidgetClass) {
		myreturn(XtNewString("grip"));
	} else if (c==labelWidgetClass) {
		myreturn(XtNewString("label"));
	} else if (c==formWidgetClass) {
		myreturn(XtNewString("form"));
	} else if (c==simpleMenuWidgetClass) {
		myreturn(XtNewString("menu"));
	} else if (c==menuButtonWidgetClass) {
		myreturn(XtNewString("menubutton"));
	} else if (c==toggleWidgetClass) {
		myreturn(XtNewString("toggle"));
	} else if (c==stripChartWidgetClass) {
		myreturn(XtNewString("stripchart"));
	} else if (c==viewportWidgetClass) {
		myreturn(XtNewString("viewport"));
	} else if (c==dialogWidgetClass) {
		myreturn(XtNewString("dialog"));
	} else if (c==panedWidgetClass) {
		myreturn(XtNewString("pane"));
	} else if (c==listWidgetClass) {
		myreturn(XtNewString("list"));
	} else if (c==smeBSBObjectClass) {
		myreturn(XtNewString("menuitem"));
	} else if (c==smeLineObjectClass) {
		myreturn(XtNewString("menuline"));
	} else {
		myreturn(XtNewString("Unknown"));
	}
}

ClassType ObjectType(w)
     Widget w;
     /* returns an enumeration representing the object type */
{
	WidgetClass c;
	entering("ObjectType");
	c=XtClass(w);
	if (c==transientShellWidgetClass) {
		myreturn(popup);
	} else if (c==boxWidgetClass) {
		myreturn(box);
	} else if (c==commandWidgetClass) {
		myreturn(button);
	} else if (c==scToolWidgetClass) {
		myreturn(scTool);
	} else if (c==tableWidgetClass) {
		myreturn(table);
	} else if (c==gripWidgetClass) {
		myreturn(grip);
	} else if (c==labelWidgetClass) {
		myreturn(label);
	} else if (c==formWidgetClass) {
		myreturn(form);
	} else if (c==simpleMenuWidgetClass) {
		myreturn(menu);
	} else if (c==menuButtonWidgetClass) {
		myreturn(menuButton);
	} else if (c==toggleWidgetClass) {
		myreturn(toggle);
	} else if (c==stripChartWidgetClass) {
		myreturn(stripChart);
	} else if (c==viewportWidgetClass) {
		myreturn(viewport);
	} else if (c==dialogWidgetClass) {
		myreturn(dialog);
	} else if (c==panedWidgetClass) {
		myreturn(pane);
	} else if (c==listWidgetClass) {
		myreturn(list);
	} else if (c==smeBSBObjectClass) {
		myreturn(menuItem);
	} else if (c==smeLineObjectClass) {
		myreturn(menuLine);
	} else {
		myreturn(invalid);
	}
}

static int ParseSaveObject(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char usage[MAXLINE],*result;
	char *parentname,nameargs[MAXMAXLINE],*otype,*value,*menuname;
	char *p;
	Widget w;
	sprintf(usage,"%s name",argv[0]);
	entering("ParseSaveObject");
	if (BadUsage(argc,2,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if (!(w=WidgetExists(argv[1],interp))) {
		myreturn TCL_ERROR;
	}
	parentname = ExtractParentName(argv[1]);
	sprintf(nameargs,"%s",XtName(w));
	otype = ObjectTypeString(w);
	switch(ObjectType(w)) {
	      case box:
		break;
	      case pane:
		break;
	      case viewport:
		break;
	      case form:
		break;
	      case dialog:
		break;
	      case scTool:
		break;
	      case button:
		XtVaGetValues(w,XtNlabel,&value,NULL);
		/* get the callback data by doing a 'fake' press of the
		   button which sets the global callbackdata */
		XtCallCallbacks(w,XtNcallback,True);
		sprintf(nameargs,"%s {%s} {%s}",XtName(w),value,callbackdata);
		XtFree(callbackdata);
		break;
	      case label:
		XtVaGetValues(w,XtNlabel,&value,NULL);
		sprintf(nameargs,"%s {%s}",XtName(w),value);
		break;
	      case popup:
		break;
	      case toggle:
		XtVaGetValues(w,XtNlabel,&value,NULL);
		/* get the callback data by doing a 'fake' press of the
		   button which sets the global callbackdata */
		XtCallCallbacks(w,XtNcallback,True);
		sprintf(nameargs,"%s {%s} {%s}",XtName(w),value,callbackdata);
		XtFree(callbackdata);
		break;
	      case list:
		XtCallCallbacks(w,XtNcallback,True);
		sprintf(nameargs,"%s {%s}",XtName(w),callbackdata);
		XtFree(callbackdata);
		break;
	      case stripChart:

		break;
	      case menuButton:
		XtVaGetValues(w,XtNlabel,&value,XtNmenuName,&menuname,NULL);
		sprintf(nameargs,"%s {%s} %s",XtName(w),value,menuname);
		break;
	      case menu:
		break;
	      case menuItem:
		XtVaGetValues(w,XtNlabel,&value,NULL);
		/* get the callback data by doing a 'fake' press of the
		   button which sets the global callbackdata */
		XtCallCallbacks(w,XtNcallback,True);
		sprintf(nameargs,"%s {%s} {%s}",XtName(w),value,callbackdata);
		XtFree(callbackdata);
		break;
	      case menuLine:
		break;
	      default:
		XtFree(parentname);
		XtFree(otype);
		myreturn TCL_OK;
		break;
	}
	p = otype;
	while (*p) {
		if ((*p>='A') && (*p <= 'Z')) *p = *p + ' ';
		p++;
	}
	if (ObjectType(w)==popup) {
		result = XtMalloc(strlen(otype)+strlen(nameargs)+5);
		sprintf(result,"new%s %s",otype,nameargs);
	} else {
		result = XtMalloc(strlen(otype)+
				  strlen(nameargs)+
				  strlen(parentname)+6);
		sprintf(result,"new%s %s %s",otype,parentname,nameargs);
	}
	Tcl_Return(interp,result,TCL_VOLATILE);
	XtFree(result);
	XtFree(parentname);
	XtFree(otype);
	myreturn TCL_OK;
}

static int ParseObjectOption(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	Widget w;
	char *result,usage[MAXLINE];
	entering("ParseObjectOption");
	sprintf(usage,"%s option object-name",argv[0]);
	if (BadUsage(argc,3,usage,interp)) {
		myreturn TCL_ERROR;
	}
	w=WidgetExists(argv[2],interp);
	if (mystrcmp(argv[1],"exists")==0) {
		if (w) {
			result=XtNewString("1");
		} else {
			result=XtNewString("0");
		}
		Tcl_Return(interp,result,TCL_VOLATILE);
		XtFree(result);
		myreturn TCL_OK;
	} else if (mystrcmp(argv[1],"type")==0) {
		if (!WidgetExists(argv[2],interp)) {
			myreturn TCL_ERROR;
		}
		result = ObjectTypeString(w);
		Tcl_Return(interp,result,TCL_VOLATILE);
		XtFree(result);
		myreturn TCL_OK;
	} else {
		Tcl_Return(interp,"Object options are 'exists' or 'type'",
			   TCL_STATIC);
		myreturn TCL_ERROR;
	}
}

static int ParseSetDebug(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
     /* sets a global variable debugwidget to the specified widget.
	This is useful when running dbx. */
{
	Widget w;
	char usage[MAXLINE];
	entering("ParseSetDebug");
	sprintf (usage,"%s object-name",argv[0]);
	if (BadUsage(argc,2,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if (w=WidgetExists(argv[1],interp)) {
		DebugWidget = w;
		myreturn TCL_OK;
	} else {
		myreturn TCL_ERROR;
	}
}

static int ParseMapObject(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char usage[MAXLINE];
	Widget w;
	entering("ParseMapObject");
	sprintf(usage,"%s object-name",argv[0]);
	if (BadUsage(argc,2,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if (w=WidgetExists(argv[1],interp)) {
		/* map semantics for menu items means to make selectable */
		if (XtClass(w)==smeBSBObjectClass) {
			XtVaSetValues(w,XtNsensitive,True,NULL);
		} else {
			if (XtIsRealized(w))
			  XtMapWidget(w);
		}
		myreturn TCL_OK;
	} else {
		myreturn TCL_ERROR;
	}
}

static int ParseUnMapObject(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char usage[MAXLINE];
	Widget w;
	entering("ParseUnMapObject");
	sprintf (usage,"%s object-name",argv[0]);
	if (BadUsage(argc,2,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if (w=WidgetExists(argv[1],interp)) {
		/* map semantics for menu items means to make selectable */
		if (XtClass(w)==smeBSBObjectClass) {
			XtVaSetValues(w,XtNsensitive,False,NULL);
		} else {
			if (XtIsRealized(w))
			  XtUnmapWidget(w);
		}
		myreturn TCL_OK;
	} else {
		myreturn TCL_ERROR;
	}
}

static int ParseMovePointer(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	int x,y;
	char usage[MAXLINE];
	Widget w;
	entering("ParseMovePointer");
	sprintf (usage,"%s object-name [xoffset yoffset]",argv[0]);
	if (BadUsageOptional(argc,3,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if (w=WidgetExists(argv[1],interp)) {
		x=0; y=0;
		if (argc>2) {
			x = atoi(argv[2]);
		}
		if (argc>3) {
			y = atoi(argv[3]);
		}
		if (x<0) x=0;
		if (y<0) y=0;
		XWarpPointer(XtDisplay(TopLevel),None,
			     XtWindow(w),
			     0,0,0,0,
			     x,y);
		myreturn TCL_OK;
	} else {
		myreturn TCL_ERROR;
	}
}

static int ParseQueryPointer(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	int x,y;
	char usage[MAXLINE],result[MAXLINE];
	Window root,child;
	int root_x,root_y;
	int win_x,win_y;
	unsigned int keys_buttons;
	Widget w;
	entering("ParseQueryPointer");
	sprintf (usage,"%s object-name",argv[0]);
	if (BadUsage(argc,2,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if (w=WidgetExists(argv[1],interp)) {
		XQueryPointer(XtDisplay(TopLevel),XtWindow(w),
			      &root,&child,&root_x,&root_y,
			      &win_x,&win_y,&keys_buttons);
		sprintf(result,"%d %d",win_x,win_y);
		Tcl_Return(interp,result,TCL_VOLATILE);
		myreturn TCL_OK;
	} else {
		myreturn TCL_ERROR;
	}
}

static void xcbiCommand(widget, event, params, num_params)
     Widget widget;
     XEvent *event;
     String *params;
     Cardinal *num_params;
     /* Invokes the first parameter as an xcbi command */
{
	entering("xcbiCommand");
	InterpretCommand(params[0]);
	myreturn;
}

static XtActionsRec actionTable[] = {
    {"xcbiCommand", xcbiCommand},
};

void AddWidgetHandlingCommands(interp)
     Tcl_Interp *interp;
{
	entering("AddWidgetHandlingCommands");

	/* add xcbiCommand action and other actions */

	XtAppAddActions(AppCon, actionTable, XtNumber(actionTable));

	AddWidgetCreateCommands(interp);
	AddWidgetSetCommands(interp);
	AddWidgetGetCommands(interp);

	/* other widget manipulation commands */

	CREATE("destroy",ParseOperationNoOptions);   /* destroy name */
	CREATE("manage",ParseOperationNoOptions);    /* manage objectname */
	CREATE("listnames",ParseListNames);          /* listnames */
	CREATE("saveobject",ParseSaveObject);        /* saveobject name */
	CREATE("object",ParseObjectOption);          /* object option name */
	CREATE("setdebugwidget",ParseSetDebug);      /* setdebugwidget name */
	CREATE("map",ParseMapObject);                /* map name */
	CREATE("unmap",ParseUnMapObject);            /* unmap name */
	CREATE("movepointer",ParseMovePointer);      /* movepointer name 
							[x y] */
	CREATE("querypointer",ParseQueryPointer);    /* querypointer name */

	myreturn;
}
