/* create.c
 *
 * Copyright 1990,1991 the Regents of the University of California.  All
 * rights reserved.  Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without fee is
 * hereby granted, provided that this copyright notice appear in all
 * copies.  See the file copyright.h for more information.
 *
 */

/*
 * Contains commands for creating new objects and a few commands which
 * apply to only a specific object. 
 */

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>            /* needed for IsExecutable() */
#include "xwrap.h"
#include <X11/Xmu/Converters.h>  /* for converting from string to ...  */
#include <X11/IntrinsicP.h>
#include <X11/Xaw/FormP.h>       /* need this for form...num_children */
#include "debug.h"
#include "global.h"
#include "util.h"
#include "object.h"

char *callbackdata;               /* global needed for PressCallback */
Boolean gettingcallback=FALSE;    /* flag to say that trying to get the
					 callback data. */
void PressCallback(w,client_data,call_data)
     Widget w;            /* button,list,grip or menuitem widget */
     caddr_t client_data; /* string to send to interpret command */
     caddr_t call_data;

     /* called when the button or menuitem widget is pressed. 
	Calls interpret command using the client_data of the widget w. */
{
	int calllength,clientlength;
	char *msg;
	GripCallData data;
	entering("PressCallback");
	/* if getting callback via getcallback command, then return the
	   client data */
	
	if (gettingcallback) {
		if (client_data) {
			callbackdata = XtNewString((char*)client_data);
		} else {
			callbackdata = XtNewString("");
		}
		myreturn;
	}
		
	/* note: the call_data for toggle Widgets is undefined and the grip
	   widget always has calldata. So the grip and toggle are a 
	   special cases. */
	if (XtClass(w)==toggleWidgetClass) {
		calllength = strlen("Pressed toggle: w= \nclient=.");
		msg = XtMalloc(calllength+strlen(client_data)+XtName(w)+2);
		sprintf(msg,"Pressed toggle: w=%s\nclient=%s.",XtName(w),
			client_data);
		dprint(msg);
		XtFree(msg);
		InterpretCommand(XtNewString((char*)client_data));
	} else if (XtClass(w)==gripWidgetClass) {
		GripCallData data;
		data = (GripCallData)call_data;
		if (!data) {
			myreturn;
		} else
		  InterpretCommand(XtNewString((char*)client_data));
	} else {
		if (call_data)
		  calllength = strlen(call_data);
		else
		  calllength = 6;   /* represents "(NULL)" */
		if (client_data)
		  clientlength = strlen(client_data);
		else
		  clientlength = 6;   /* represents "(NULL)" */
		msg = XtMalloc(calllength+clientlength+strlen(XtName(w))+55);
		sprintf(msg,"In PressCallback: w=%s\ncall=%d,client=%s",
			XtName(w),call_data,client_data);
		dprint(msg);
		XtFree(msg);
		XtSetSensitive(w,FALSE);
		InterpretCommand(XtNewString((char*)client_data));
		XtSetSensitive(w,TRUE);
	}
	myreturn;
}

static void SafeDestroyCallback(w,client_data,call_data)
     Widget w;
     caddr_t client_data;
     caddr_t call_data;
     /* callback for destroying anything that the object needs.
	Form items get their fromvert and fromhoriz set correct. */
{
	Widget parent;
	entering("SafeDestroyCallback");
	parent=XtParent(w);
	if (XtClass(parent)==formWidgetClass) {
		FormWidget form;
		Widget horiz,vert,child_horiz,child_vert;
		int i;
		XtVaGetValues(w,XtNfromVert,&vert,XtNfromHoriz,&horiz,NULL);
		form = (FormWidget)parent;
		/* change the fromvert or fromhoriz for any child that
		   references the widget being deleted. */
		for (i=0; i<form->composite.num_children; i++) {
			XtVaGetValues(form->composite.children[i],
				      XtNfromVert,&child_vert,
				      XtNfromHoriz,&child_horiz,NULL);
			if (child_vert==w)
			  XtVaSetValues(form->composite.children[i],
					XtNfromVert,vert,NULL);
			if (child_horiz==w)
			  XtVaSetValues(form->composite.children[i],
					XtNfromHoriz,horiz,NULL);
		}
	}
	myreturn;
}


static int ParseNewObject(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
     /* useful for operations of the form 'newobject object-name' */
{
	Widget w,parent;
	char usage[MAXLINE],result[MAXLINE];
	WidgetClass class;
	entering("ParseNewObject");
	sprintf(usage,"%s parent object-name [unmanaged]",argv[0]);
	if ((BadUsageOptional(argc,4,usage,interp)) ||
	    (InvalidParent(argv[1],&parent,interp))) {
		myreturn TCL_ERROR;
	}
	if (mystrcmp(argv[0],"newbox")==0)
	  class=boxWidgetClass;
	else if (mystrcmp(argv[0],"newviewport")==0)
	  class=viewportWidgetClass;
	else if (mystrcmp(argv[0],"newpane")==0)
	  class=panedWidgetClass;
	else if (mystrcmp(argv[0],"newform")==0)
	  class=formWidgetClass;
	else if (mystrcmp(argv[0],"newtable")==0)
	  class=tableWidgetClass;
	else if (mystrcmp(argv[0],"newgrip")==0)
	  class=gripWidgetClass;
	else if (mystrcmp(argv[0],"newdialog")==0)
	  class=dialogWidgetClass;
	else if (mystrcmp(argv[0],"newmenu")==0) {
		XtCreatePopupShell(argv[2],simpleMenuWidgetClass,
				   parent,NULL,ZERO);
		myreturn TCL_OK;
	}
	else if (mystrcmp(argv[0],"newmenuline")==0)
	  class=smeLineObjectClass;
	else
	  class=NULL;
	if (class) {
		if (argc<4)
		  w = XtVaCreateManagedWidget(argv[2],class,parent,NULL);
		else {
			w=XtVaCreateWidget(argv[2],class,parent,NULL);
		}
		XtAddCallback(w,XtNdestroyCallback,SafeDestroyCallback,NULL);
		
		/* special cases for various objects */

		if (class==gripWidgetClass) {
			char translation[MAXLINE];
			sprintf(translation,"<Btn1Down>: GripAction()");
			XtOverrideTranslations(w,
			       XtParseTranslationTable(translation));
		} else if (class==dialogWidgetClass) {
			/* add ok, cancel, and clear buttons and set
			   return translation in value to press the ok
			   button. */
			char translation[MAXLINE];
			Widget ok,cancel,clear;
			ok = XtVaCreateManagedWidget("ok",commandWidgetClass,w,
						NULL);
			cancel = XtVaCreateManagedWidget("cancel",
							 commandWidgetClass,w,
							 NULL);
			clear = XtVaCreateManagedWidget("clear",
							commandWidgetClass,w,
							NULL);
			sprintf(result,"setvalue %s.%s {}",argv[1],argv[2]);
			XtAddCallback(clear,XtNcallback,PressCallback,
				      XtNewString(result));
			/* need to set the value to tell the dialog to create
			   the value widget before assigning a tranlation
			   to it */
			XtVaSetValues(w,XtNvalue,"",NULL);
			sprintf(translation,
				"<Key>Return: xcbiCommand(\"press %s.%s.ok\")",
			        argv[1],argv[2]);
			XtOverrideTranslations(XtNameToWidget(w,"value"),
			       XtParseTranslationTable(translation));
		}
		myreturn TCL_OK;
	} else {
		sprintf (result,"Don't know how to create a %s\n.",
			 argv[0]);
		Tcl_Return(interp,result,TCL_VOLATILE);
		myreturn TCL_ERROR;
	}
}

#define XMASK	(0111)
/* Little function to check executable bits in file status */
static int
IsExecutable(path)
  char	*path;
{
  struct stat	FileStatus;

  if (stat(path,&FileStatus)) {
    return 0;
  } else {
    return (FileStatus.st_mode & XMASK) != 0;
  }
}

#undef XMASK

char *getenv();
/*
** GetExe(name,path)
**
** This procedure searches for the pathname of the executable in the
** name itself and then in the user's path
*/
static int
GetExe(name,path)
     char	*name;
     char	*path;
{
  int	i;
  char	*UserPath;
  char	*p;

  strcpy(path,name);
  if (IsExecutable(name)) { 
	  return True;
  }
  /* Try each directory in the user path */
  UserPath = getenv("PATH");
  if (UserPath) {
	  p = UserPath;
	  while(*p) {
		  for(i=0;*p && *p != ':';p++) path[i++] = *p;
		  path[i++] = '/';
		  strcpy(path+i,name);
		  if (IsExecutable(path)) { 
			  return True;
		  }
		  if (*p) p++;		/* Skip the colon */
	  }
  }
  
  /* Have to assume its just the default name */
  strcpy(path,name);
  return False;
}

static void SetTextWidthAndHeight(w,columns,rows)
     Widget w;
     int columns, rows;
     /* sets the width and height of the text widget to columns and rows
	measured in characters. Default is 24x80. */
{
	Widget sink;
	XFontStruct *font;
	int height, width;
	entering("SetTextWidthAndHeight");
	/* default rows and columns to 24x80 */
	if (rows<=0) rows = 24;
	if (columns<=0) columns = 80;
	XtVaGetValues(w,XtNtextSink,&sink,XtNfont,&font,NULL);
	width=(font->max_bounds.rbearing - font->min_bounds.lbearing)*columns;
	height = (font->max_bounds.ascent + font->max_bounds.descent)*rows;
	XtVaSetValues(w,XtNheight,height,XtNwidth,width,NULL);
	/* don't let text windows grow less than one line */
	height = XawTextSinkMaxHeight(sink,3);
	XtVaSetValues(w,XtNmin,height,NULL);
	myreturn;
}

void SendToInterpreter(w,client_data,call_data)
     Widget w;             /* the ScTool widget (text window) */
     caddr_t client_data;  /* the tcl procedure to call with the line
			      as its argument.  This is only called
			      when the callback is called using 
			      CallCallbacks */
     caddr_t call_data;    /* contains the line from the ScTool widget */
     /* callback for text window <CR>.  Sends line to InterpretCommand */
{
	char* line,*argv[1],*args,*fullcmd;
	int argc;
	entering("SendToInterpreter");
	line = (char *)call_data;
	if (client_data) {
		argc = 1;
		argv[0] = line;
		args = Tcl_Merge(argc,argv);
		fullcmd = XtMalloc(strlen(args)+
				   strlen((char *)client_data)+2);
		sprintf(fullcmd,"%s %s",(char *)client_data,args);
		InterpretCommand(fullcmd);
		XtFree(args);
		XtFree(fullcmd);
	} else {
		InterpretCommand(line);
	}
	myreturn;
}

static void DestroyTextCallback(w,client_data,call_data)
     Widget w;
     caddr_t client_data;
     caddr_t call_data;
     /* callback for destroying the text object */
{
	entering("DestroyTextCallback");
	ScToolKill(w);
	myreturn;
}

static int ParseNewText(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	Widget w,parent;
	int rows,columns;
	char usage[MAXLINE];
	XtCallbackRec callbacks[2];
	entering("ParseNewText");
	callbacks[0].callback = SendToInterpreter;
	callbacks[0].closure = NULL;
	callbacks[1].callback = NULL;
	callbacks[1].closure = NULL;
	sprintf(usage,"%s parent window-name [callback] [columns] [rows]",
		argv[0]);
	if ((BadUsageOptional(argc,4,usage,interp)) ||
	    (InvalidParent(argv[1],&parent,interp))) {
		myreturn TCL_ERROR;
	}
	rows = 0; columns = 0;
	if (argc >4)
	  columns = atoi(argv[4]);
	if (argc > 5)
	  rows = atoi(argv[5]);
	if ((argc>3)&&(*argv[3])) {
		callbacks[0].closure = XtNewString(argv[3]);
	}
	w =  XtVaCreateManagedWidget(argv[2],
				     scToolWidgetClass,
				     parent,
				     XtNcommandCallback,callbacks,
				     XtNexitCallback,callbacks,
				     XtNtype,XawAsciiString,
				     XtNeditType, XawtextEdit,
				     XtNdataCompression,True,
				     XtNdisplayNonprinting,True,
				     NULL);
	XtAddCallback(w,XtNdestroyCallback,DestroyTextCallback,NULL);
	SetTextWidthAndHeight(w,columns,rows);
	myreturn TCL_OK;
}

static int ParseNewTextOnly(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	Widget w,parent;
	int rows,columns;
	char usage[MAXLINE];
	entering("ParseNewTextOnly");
	sprintf(usage,"%s parent window-name [columns] [rows]",
		argv[0]);
	if ((BadUsageOptional(argc,4,usage,interp)) ||
	    (InvalidParent(argv[1],&parent,interp))) {
		myreturn TCL_ERROR;
	}
	rows = 0; columns = 0;
	if (argc > 3)
	  columns = atoi(argv[3]);
	if (argc > 4)
	  rows = atoi(argv[4]);
	w =  XtVaCreateManagedWidget(argv[2],
				     asciiTextWidgetClass,
				     parent,
				     XtNtype,XawAsciiString,
				     XtNeditType, XawtextEdit,
				     XtNdataCompression,True,
				     XtNdisplayNonprinting,True,
				     NULL);
	SetTextWidthAndHeight(w,columns,rows);
	myreturn TCL_OK;
}

static int ParseGotoLine(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	XEvent event;
	Widget w;
	char usage[MAXLINE];
	XawTextPosition pos,lastpos;
	entering("ParseGotoLine");
	sprintf(usage,"%s text-object line-number",argv[0]);
	if (BadUsage(argc,3,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if ((!InvalidType(argv[1],scToolWidgetClass,interp))&&
	    (!InvalidType(argv[1],asciiTextWidgetClass,interp))) {
		myreturn TCL_ERROR;
	}
	if (!(w = WidgetExists(argv[1],interp))) {
		myreturn TCL_ERROR;
	}
	Tcl_Return(interp,NULL,TCL_VOLATILE);
	XawTextDisableRedisplay(w);
	XtCallActionProc(w,"beginning-of-file",&event,NULL,ZERO);
	if (atoi(argv[2])>0) {
		XtCallActionProc(w,"multiply",&event,&argv[2],ONE);
		XtCallActionProc(w,"next-line",&event,NULL,ZERO);
	}
	XawTextEnableRedisplay(w);
	XawTextDisplay(w);
	XawTextUnsetSelection(w);
	XtCallActionProc(w,"beginning-of-line",&event,NULL,ZERO);
	pos = XawTextGetInsertionPoint(w);
	XtCallActionProc(w,"end-of-line",&event,NULL,ZERO);
	lastpos = XawTextGetInsertionPoint(w);
	XawTextSetSelection(w,pos,lastpos);
	XawTextSetInsertionPoint(w,pos);
	myreturn TCL_OK;
}

static int ParseCurrentLine(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	XEvent event;
	XawTextPosition pos,newpos,lastpos;
	Widget w;
	int n;
	char result[MAXLINE],usage[MAXLINE];
	entering("ParseCurrentLine");
	sprintf(usage,"%s text-object",argv[0]);
	if (BadUsage(argc,2,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if ((!InvalidType(argv[1],scToolWidgetClass,interp))&&
	    (!InvalidType(argv[1],asciiTextWidgetClass,interp))) {
		myreturn TCL_ERROR;
	}
	if (!(w = WidgetExists(argv[1],interp))) {
		myreturn TCL_ERROR;
	}
	XawTextDisableRedisplay(w);
	pos = XawTextGetInsertionPoint(w);
	XtCallActionProc(w,"end-of-file",&event,NULL,ZERO);
	lastpos = XawTextGetInsertionPoint(w);
	XtCallActionProc(w,"beginning-of-file",&event,NULL,ZERO);
	newpos = XawTextGetInsertionPoint(w);
	n = 0;
	while ((newpos < pos)&&(newpos < lastpos)) {
		n++;
		XtCallActionProc(w,"next-line",&event,NULL,ZERO);
		newpos = XawTextGetInsertionPoint(w);
	}
	XawTextSetInsertionPoint(w,pos);
	XawTextEnableRedisplay(w);
	XawTextDisplay(w);
	sprintf(result,"%d",n);
	Tcl_Return(interp,result,TCL_VOLATILE);
	myreturn TCL_OK;
}

static int ParseTextSearch(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	XEvent event;
	Widget w;
	char usage[MAXLINE];
	String search[2];
	entering("ParseTextSearch");
	sprintf(usage,"%s text-object [direction]",argv[0]);
	if (BadUsageOptional(argc,3,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if ((!InvalidType(argv[1],scToolWidgetClass,interp))&&
	    (!InvalidType(argv[1],asciiTextWidgetClass,interp))) {
		myreturn TCL_ERROR;
	}
	if (!(w = WidgetExists(argv[1],interp))) {
		myreturn TCL_ERROR;
	}
	Tcl_Return(interp,NULL,TCL_VOLATILE);
	search[0] = "forward";
	if (argc>2) {
		if (argv[2][0]=='b') {
			search[0]="backward";
		}
	}
	XtCallActionProc(w,"search",&event,search,ONE);
	myreturn TCL_OK;
}

static int ParseUseFilter(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	Widget w;
	int b;
	char result[MAXLINE],usage[MAXLINE];
	XrmValue from,to;   /* for converting string to... */
	entering("ParseUseFilter");
	sprintf(usage,"%s text-object [True|False]",argv[0]);
	if ((BadUsageOptional(argc,3,usage,interp)) ||
	    (InvalidType(argv[1],scToolWidgetClass,interp))) {
		myreturn TCL_ERROR;
	}
	if (!(w = WidgetExists(argv[1],interp))) {
		myreturn TCL_ERROR;
	}
	if (argc>2) {
		from.size = strlen(from.addr=argv[2]);
		XtConvert(w,XtRString,&from,XtRBoolean,&to);
		if (to.addr==NULL) {
			b=True;
		} else {
			b = *((Boolean*)to.addr);
		}
	} else {
		b=True;
	}
	ScAlwaysCallback(w,b);
	Tcl_Return(interp,NULL,TCL_VOLATILE);
	myreturn TCL_OK;
}

static int ParseDisplayCaret(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	Widget w;
	int b;
	char result[MAXLINE],usage[MAXLINE];
	XrmValue from,to;   /* for converting string to... */
	entering("ParseDisplayCaret");
	sprintf(usage,"%s text-object [True|False]",argv[0]);
	if (BadUsageOptional(argc,3,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if ((!InvalidType(argv[1],scToolWidgetClass,interp))&&
	    (!InvalidType(argv[1],asciiTextWidgetClass,interp))) {
		myreturn TCL_ERROR;
	}
	if (!(w = WidgetExists(argv[1],interp))) {
		myreturn TCL_ERROR;
	}
	if (argc>2) {
		from.size = strlen(from.addr=argv[2]);
		XtConvert(w,XtRString,&from,XtRBoolean,&to);
		if (to.addr==NULL) {
			b=True;
		} else {
			b = *((Boolean*)to.addr);
		}
	} else {
		b=True;
	}
	XawTextDisplayCaret(w,b);
	Tcl_Return(interp,NULL,TCL_VOLATILE);
	myreturn TCL_OK;
}

static int ParseRun(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	Widget w;
	int pid;
	char result[MAXLINE],usage[MAXLINE];
	char dummypath[MAXLINE];
	entering("ParseRun");
	sprintf(usage,"%s window pathname [args]",argv[0]);
	if ((BadUsageOptional(argc,4,usage,interp)) ||
	    (InvalidType(argv[1],scToolWidgetClass,interp))) {
		myreturn TCL_ERROR;
	}
	if (!GetExe(argv[2],dummypath)) {
		Tcl_Return(interp,"0",TCL_VOLATILE);
		myreturn TCL_OK;
	}
	if (!(w = WidgetExists(argv[1],interp))) {
		myreturn TCL_ERROR;
	}
	if (w) {
		if ((pid = ScToolStart(w,dummypath,argc-2,argv+2)>0));
		sprintf(result,"%d",pid);
	} else {
		sprintf(result,"-1");
	}
	Tcl_Return(interp,result,TCL_VOLATILE);
	myreturn TCL_OK;
}

static int ParseWrite(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	XawTextBlock text;
	XawTextPosition pos;
	XEvent event;
	Widget w;
	char usage[MAXLINE],result[1];
	String prog;
	entering("ParseWrite");
	sprintf(usage,"%s text-window string",argv[0]);
	if (BadUsage(argc,3,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if ((!InvalidType(argv[1],scToolWidgetClass,interp))&&
	    (!InvalidType(argv[1],asciiTextWidgetClass,interp))) {
		myreturn TCL_ERROR;
	}
	if (!(w=WidgetExists(argv[1],interp))) {
		myreturn TCL_ERROR;
	}
	if (XtClass(w)==scToolWidgetClass) {
		XtVaGetValues(w,XtNprogram,&prog,NULL);
		if (prog) {
			ScWriteText(w,argv[2],1);
		} else {
			ScWriteText(w,argv[2],0);
		}
		EmptyEventLoop();
	} else if (XtClass(w)==asciiTextWidgetClass) {
		XtCallActionProc(w,"end-of-file",&event,NULL,ZERO);
		text.firstPos = 0;
		text.format = FMT8BIT;
		text.length = strlen(argv[2]);
		text.ptr = argv[2];
		pos = XawTextGetInsertionPoint(w);
		XawTextReplace(w,pos,pos,&text);
		EmptyEventLoop();
	}
	result[0]=NULL;
	Tcl_Return(interp,result,TCL_VOLATILE);
	myreturn TCL_OK;
}

static int ParseWriteUser(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
     /* like write command, but always writes as if the user typed it */
{
	XawTextBlock text;
	XawTextPosition pos;
	XEvent event;
	Widget w;
	char usage[MAXLINE];
	entering("ParseWriteUser");
	sprintf(usage,"%s text-window string",argv[0]);
	if (BadUsage(argc,3,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if ((!InvalidType(argv[1],scToolWidgetClass,interp))&&
	    (!InvalidType(argv[1],asciiTextWidgetClass,interp))) {
		myreturn TCL_ERROR;
	}
	if (!(w =WidgetExists(argv[1],interp))) {
		myreturn TCL_ERROR;
	}
	if (XtClass(w)==scToolWidgetClass) {
		ScWriteText(w,argv[2],1);
		EmptyEventLoop();
	} else if (XtClass(w)==asciiTextWidgetClass) {
		XtCallActionProc(w,"end-of-file",&event,NULL,ZERO);
		text.firstPos = 0;
		text.format = FMT8BIT;
		text.length = strlen(argv[2]);
		text.ptr = argv[2];
		pos = XawTextGetInsertionPoint(w);
		XawTextReplace(w,pos,pos,&text);
		EmptyEventLoop();
	}
	Tcl_Return(interp,NULL,TCL_VOLATILE);
	myreturn TCL_OK;
}

static int ParseNewPressItem(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	Widget w,parent;
	char usage[MAXLINE],result[MAXLINE];
	entering("ParseNewPressItem");
	sprintf(usage,"%s parent object-name label callback|menuname",argv[0]);
	if ((BadUsage(argc,5,usage,interp)) ||
	    (InvalidParent(argv[1],&parent,interp))) {
		myreturn TCL_ERROR;
	}
	if (mystrcmp(argv[0],"newbutton")==0) {
		w = XtVaCreateManagedWidget(argv[2],commandWidgetClass,parent,
					    XtNlabel,argv[3],NULL);
		XtAddCallback(w,XtNcallback,PressCallback,
			      XtNewString(argv[4]));
	} else if (mystrcmp(argv[0],"newtoggle")==0) {
		w = XtVaCreateManagedWidget(argv[2],toggleWidgetClass,parent,
					    XtNlabel,argv[3],NULL);
		XtAddCallback(w,XtNcallback,PressCallback,
			      XtNewString(argv[4]));
	} else if (mystrcmp(argv[0],"newmenubutton")==0) {
		w = XtVaCreateManagedWidget(argv[2],menuButtonWidgetClass,
					    parent,
					    XtNlabel,argv[3],
					    XtNmenuName,XtNewString(argv[4]),
					    NULL);
		/* create the associated menu */
		XtCreatePopupShell(argv[4],simpleMenuWidgetClass,
				   w,NULL,ZERO);
	} else if (mystrcmp(argv[0],"newmenuitem")==0) {
		w = XtVaCreateManagedWidget(argv[2],smeBSBObjectClass,parent,
					    XtNlabel,argv[3],NULL);
		XtAddCallback(w,XtNcallback,PressCallback,
			      XtNewString(argv[4]));
	} else {
		sprintf (result,"Don't know how to create a %s\n.",
			 argv[0]);
		Tcl_Return(interp,result,TCL_VOLATILE);
		myreturn TCL_ERROR;
	}
	myreturn TCL_OK;
}

static void SetListSelected(w,data)
     XawListReturnStruct *data;
     Widget w;
{
	String cmd;
	entering("SetListSelected");
/* 	XtSetSensitive(w,FALSE); */
	if (XtClass(w)==listWidgetClass) {
		if (data->list_index!=XAW_LIST_NONE) {
			cmd = data->string;
		} else {
			cmd = "";
		}
		Tcl_SetVar(mainInterp,"listselected",cmd,0);
	}
/* 	XtSetSensitive(w,TRUE); */
	myreturn;
}
	
static int ParseNewCheckBox(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
     /* this is a routine because there is a problem with the form layout
	and it's too messy to do as a tcl script. */
{
	char usage[MAXLINE];
	Widget w,parent,toggle,button;
	entering("ParseNewCheckBox");
	sprintf(usage,"%s parent name label togglecallback [buttoncallback]",
		argv[0]);
	if ((BadUsageOptional(argc,6,usage,interp)) ||
	    (InvalidParent(argv[1],&parent,interp))) {
		myreturn TCL_ERROR;
	}
	w = XtVaCreateManagedWidget(argv[2],boxWidgetClass,parent,
				    XtNresizable,True,
				    XtNorientation,"horizontal",NULL);
	toggle = XtVaCreateManagedWidget("toggle",toggleWidgetClass,w,
					 XtNlabel," ",
					 XtNresizable,True,NULL);
	XtAddCallback(toggle,XtNcallback,PressCallback,XtNewString(argv[4]));
	button = XtVaCreateManagedWidget("button",commandWidgetClass,w,
					XtNlabel,argv[3],
					XtNborderWidth,0,
					XtNresizable,True,NULL);
	if (argc>5) {
		XtAddCallback(button,XtNcallback,PressCallback,
			      XtNewString(argv[5]));
	}
	myreturn TCL_OK;
}

static int ParseNewTableTextLine(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
     /* this is a routine because it takes a little too long to repeatedly
	create this in a tcl script.  Creates a line in a table looking like:
	'   label text    ' */
{
	char *labelname,usage[MAXLINE];
	Widget label,text,parent;
	int rows,columns,x,y;
	entering("ParseNewTableTextLine");
	sprintf(usage,"%s parent name label x y [columns rows]",
		argv[0]);
	if ((BadUsageOptional(argc,7,usage,interp)) ||
	    (InvalidParent(argv[1],&parent,interp))) {
		myreturn TCL_ERROR;
	}
	if (InvalidType(argv[1],tableWidgetClass,interp)) {
		myreturn TCL_ERROR;
	}
	labelname = XtMalloc(strlen(argv[2])+6);
	sprintf(labelname,"%slabel",argv[2]);
	label = XtVaCreateManagedWidget(labelname,labelWidgetClass,parent,
					XtNborderWidth,0,
					XtNlabel,argv[3],NULL);
	text = XtVaCreateManagedWidget(argv[2],asciiTextWidgetClass,parent,
				       XtNtype,XawAsciiString,
				       XtNeditType, XawtextEdit,
				       XtNdataCompression,True,
				       XtNdisplayNonprinting,True,
				       NULL);
	x = atoi(argv[4]); y = atoi(argv[5]);
	rows = 0; columns = 0;
	if (argc > 6)
	  columns = atoi(argv[6]);
	if (argc > 7)
	  rows = atoi(argv[7]);
	SetTextWidthAndHeight(text,columns,rows);
	XtTblPosition(label,x,y);
	XtTblPosition(text,x+1,y);
	XtFree(labelname);
	myreturn TCL_OK;
}

static int ParseNewLabel(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	int dojustify;
	char usage[MAXLINE];
	Widget w,parent;
	XrmValue from,to;   /* for converting string to justify */
	entering("ParseNewLabel");
	sprintf(usage,"%s parent name string [justification]",argv[0]);
	if ((BadUsageOptional(argc,5,usage,interp)) ||
	    (InvalidParent(argv[1],&parent,interp))) {
		myreturn TCL_ERROR;
	}
	dojustify = False;
	if (argc>4) {
		from.size = strlen(from.addr=argv[4]);
		XtConvert(parent,XtRString,&from,XtRJustify,&to);
		if (to.addr!=NULL) {
			dojustify = True;
		}
	}
	/* need to do the justification during the CreateWidget because
	   otherwise some of the default resources (like width) get reset */
	if (dojustify) {
		w = XtVaCreateManagedWidget(argv[2],labelWidgetClass,parent,
					    XtNlabel,argv[3],
					    XtNjustify,*(XtJustify*)to.addr,
					    NULL);
	} else {
		w = XtVaCreateManagedWidget(argv[2],labelWidgetClass,parent,
					    XtNlabel,argv[3],NULL);
	}
	myreturn TCL_OK;
}

static void DoubleClickList(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
     /* Translation for double click in the list widget.  Sets the
	variable listselected to the selected item then calls
	InterpretCommand on the first parameter. */
{
	SetListSelected(w,XawListShowCurrent(w));
	InterpretCommand(params[0]);
	XtSetSensitive(w,TRUE);
	myreturn;
}

static int ParseNewList(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char *clicktranslation,*tempstr;
	char **listargv;
	int listargc;
	Widget w,parent;
	char usage[MAXLINE];
	entering("ParseNewList");
	sprintf(usage,"%s parent object-name callback",argv[0]);
	if ((BadUsage(argc,4,usage,interp)) ||
	    (InvalidParent(argv[1],&parent,interp))) {
		myreturn TCL_ERROR;
	}
	listargc=1;
	listargv=(char**)XtMalloc(2*sizeof(char*));
	listargv[0] = XtNewString(argv[1]);
	listargv[1] = NULL;
	w = XtVaCreateManagedWidget(argv[2],listWidgetClass,parent,
				    XtNlist,listargv,
				    XtNnumberStrings,listargc,
				    NULL);
	tempstr = XtNewString("<Btn1Up>(2): DoubleClickList");
	clicktranslation = XtMalloc(strlen(argv[3])+strlen(tempstr)+5);
	sprintf(clicktranslation,"%s(\"%s\")",tempstr,argv[3]);
	XtOverrideTranslations(w,XtParseTranslationTable(clicktranslation));
	XtFree(clicktranslation);
	XtFree(tempstr);
	/* this callback is needed so that I can access my callback
	   data that is sent to DoubleClickList */
	XtAddCallback(w,XtNcallback,PressCallback,XtNewString(argv[3]));
	myreturn TCL_OK;
}

static int ParseChangeList(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char **listargv;
	char **oldlist;
	int listargc,columns;
	Widget w;
	char usage[MAXLINE];
	entering("ParseChangeList");
	sprintf(usage,"%s list-name newlist [number of columns]",argv[0]);
	if ((BadUsageOptional(argc,4,usage,interp)) ||
	    (InvalidType(argv[1],listWidgetClass,interp))) {
		myreturn TCL_ERROR;
	}
	listargc = 0;
	if (Tcl_SplitList(interp,argv[2],&listargc,&listargv)==TCL_ERROR) {
		myreturn TCL_ERROR;
	}
	if (!(w = WidgetExists(argv[1],interp))) {
		myreturn TCL_ERROR;
	}
	XtVaGetValues(w,XtNlist,&oldlist,NULL);
	if (*oldlist) {
		XtFree((char*)oldlist);
	}
	/* note: XawListChange resets the highlight.  
	   XtVaSetValues does not :-( */
	if (listargc==0) {
		listargv[0] = XtNewString("No items");
		listargc = 1;
	}
	XawListChange(w,listargv,listargc,0,True);
	if ((argc>3)&&((columns=atoi(argv[3]))>0)) {
		XtVaSetValues(w,XtNdefaultColumns,columns,
			      XtNforceColumns,True,NULL);
	}
	myreturn TCL_OK;
}

static int ParseNewPopup(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	char usage[MAXLINE];
	entering("ParseNewPopup");
	sprintf(usage,"%s name",argv[0]);
	if (BadUsage(argc,2,usage,interp)) {
		myreturn TCL_ERROR;
	}
	if (mystrcmp(argv[0],"newpopup")==0) {
		XtVaCreatePopupShell(argv[1],transientShellWidgetClass,
				     TopLevel,XtNallowShellResize,True,
				     NULL);
	} else if (mystrcmp(argv[0],"newtoplevel")==0) {
		XtVaAppCreateShell(GlobalArgv[0],"Xcbi",
				   topLevelShellWidgetClass,
				   XtDisplay(TopLevel),NULL);
	}
	myreturn TCL_OK;
}

static int ParsePopUp(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	Widget w,gotowidget;
	char usage[MAXLINE];
	entering("ParsePopUp");
	sprintf(usage,"%s popup-name [goto-widget]",argv[0]);
	if ((BadUsageOptional(argc,3,
			      usage,interp))||
	    (InvalidType(argv[1],transientShellWidgetClass,interp))) {
		myreturn TCL_ERROR;
	}
	if (!(w = WidgetExists(argv[1],interp))) {
		myreturn TCL_ERROR;
	}
	/* place the pointer in the gotowidget */
	if (argc>2) {
		if (!(gotowidget=WidgetExists(argv[2],interp))) {
			gotowidget = w;
		}
	} else {
		gotowidget = w;
	}
	SafePopupWidget(w,gotowidget);
	EmptyEventLoop();
	myreturn TCL_OK;
}


static int ParsePopUpOnly(clientData,interp,argc,argv)
     char *clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
	Widget w;
	char usage[MAXLINE];
	entering("ParsePopUpOnly");
	sprintf(usage,"%s popup-name",argv[0]);
	if ((BadUsage(argc,2,usage,interp))||
	    (InvalidType(argv[1],transientShellWidgetClass,interp))) {
		myreturn TCL_ERROR;
	}
	if (!(w = WidgetExists(argv[1],interp))) {
		myreturn TCL_ERROR;
	}
	XtPopup(w,XtGrabNone);
	EmptyEventLoop();
	myreturn TCL_OK;
}


static XtActionsRec actionTable[] = {
    {"DoubleClickList", DoubleClickList},
};

void AddWidgetCreateCommands(interp)
     Tcl_Interp *interp;
{
	entering("AddWidgetCreateCommands");

	/* add action for double clicking on lists */

	XtAppAddActions(AppCon, actionTable, XtNumber(actionTable));

	/* create objects that are of the form:
	   "newobject parent name" */

	CREATE("newbox",ParseNewObject);
	CREATE("newpane",ParseNewObject);
	CREATE("newviewport",ParseNewObject);
	CREATE("newform",ParseNewObject);
	CREATE("newtable",ParseNewObject);
	CREATE("newgrip",ParseNewObject);
	CREATE("newdialog",ParseNewObject);
	CREATE("newmenu",ParseNewObject);
	CREATE("newmenuline",ParseNewObject);

	/* create objects that have a label and a callback */

	CREATE("newbutton",ParseNewPressItem);
	CREATE("newtoggle",ParseNewPressItem);
	CREATE("newmenubutton",ParseNewPressItem);
	CREATE("newmenuitem",ParseNewPressItem);

	/* create other objects */

	CREATE("newcheckbox",ParseNewCheckBox);      /* newcheckbox parent
							name label callback */
	CREATE("newlabel",ParseNewLabel);            /* newlabel parent
                                                        labelname string
							[justification] */
	CREATE("newlist",ParseNewList);              /* newllist parent
							listname callback */
	CREATE("changelist",ParseChangeList);        /* changelist listname
							newlist [#columns] */
	CREATE("newpopup",ParseNewPopup);            /* newpopup name */
	CREATE("popdown",ParseOperationNoOptions);   /* popdown name */
	CREATE("popup",ParsePopUp);                  /* popup name [place] */
	CREATE("popuponly",ParsePopUpOnly);          /* popuponly name */
	CREATE("newtext",ParseNewText);              /* newtext parent
							text-name args */
	CREATE("newtextonly",ParseNewTextOnly);      /* newtextonly parent
							text-name args */
	CREATE("newtabletextline",ParseNewTableTextLine); /* newtextonly parent
							text-name args */
	CREATE("gotoline",ParseGotoLine);            /* gotoline
							text-name 
							line-number */
	CREATE("currentlinenumber",ParseCurrentLine);/* currentline
							text-name */
	CREATE("usefilter",ParseUseFilter);          /* usefilter
							text-name true|false */
	CREATE("textsearch",ParseTextSearch);        /* textsearch
							text-name [for|back] */
	CREATE("displaycaret",ParseDisplayCaret);    /* displaycaret
							text-name true|false */
	CREATE("run",ParseRun);                      /* run parent pathname
							args */
	CREATE("write",ParseWrite);                  /* write textname msg */
	CREATE("writeuser",ParseWriteUser);          /* writeuser textname 
							msg */
	CREATE("clear",ParseOperationNoOptions);     /* clear textname */
	CREATE("press",ParseOperationNoOptions);     /* press buttonname */
}
