/* yift.c
   Routines for translating the if1 (or if2) to a stack based C program.
   The C program looks like:

   Type Table information
   Forward Declarations
   Mark Vector Pool
   Edge Name Pool
   Edge Name Info types
   Literal Pool
   Type Pool
   Node Table
   Code for Graphs

   Call If2Process() to build the tables and print the C program.
*/

/* $Header: /u0/uejio/city/debug/yift/RCS/if2write.c,v 1.14 91/05/08 15:06:19 uejio Exp Locker: uejio $ */
/* $Log:	if2write.c,v $
 *
 * Revision 1.18  92/03/26  08:00 Cedeno
 * Replace temporary id.s T#.# to create unique identifier.
 *
 * Revision 1.17  92/02/26  14:35 Cedeno
 * Fixed a bug in CountSubNodesAndExpr calculating number of expressions and
 * a bug in CalculateEdgeOffsets to include edges with Fanout > 1.
 *
 * Revision 1.16  92/02/12  14:35 Cedeno
 * Corrected an infinite recursive call when marking data types in 
 * recursive union types.
 *
 * Revision 1.15  92/01/16  13:00:00  cedeno
 * Corrected the bugs in calls to MyAlloc where the end of string character
 * was not computed as part of the string size.
 *
 * Revision 1.14  91/05/08  15:06:19  uejio
 * Changed the edgename pool to allocate space for edgenames then output
 * the array.  The EdgeNamePool is output as a series of names and then
 * an array of pointers to the names. The pointers are formed by a macro.
 * 
 * Revision 1.13  91/04/22  15:19:57  uejio
 * Added a routine to change '.' to '_' in nested function names and changed
 * the name of unnamed edges to node#_port#.  Changed Makefile to link -lm.
 * 
 * Revision 1.12  91/04/02  14:30:25  uejio
 * Finished association list for compounds.  Yift should be finished :-)
 * 
 * Revision 1.11  91/03/20  10:54:27  uejio
 * Changed the literals format and added a module name.
 * Fixed a bug in MakeUpName.  Forgot to strcpy after malloc.
 * 
 * Revision 1.10  90/12/06  16:35:04  uejio
 * Fixed bug with LocalOutputs and calculating subexpressions.
 * Added the D to E conversion and fixed the error literals.
 * 
 * Revision 1.9  90/12/03  16:14:00  uejio
 * Fixed bug with SR vectors.
 * Fixed literals to allow for error values.
 * Fixed double literals to use 'E' instead of 'D' notation.
 * 
 * Revision 1.8  90/11/30  17:05:12  uejio
 * Fixed bug in records and unions.
 * Changed WriteTypes to write types nested in arrays and records.
 * 
 * Revision 1.7  90/10/23  16:15:15  uejio
 * Fixed a minor bug in CopyToGraph.  changed e->eport to e->iport on the
 * second argument.
 * 
 * Revision 1.6  90/10/18  14:59:03  uejio
 * Added edge name pool.
 * 
 * Revision 1.5  90/10/09  09:39:36  uejio
 * Added the %sr marks and a pool for marks.  Still need to optimize by
 * adding an edge name pool, and need to fix DoubleLiteral to be 1e0 instead
 * of 1d0.
 * 
 * Revision 1.4  90/10/01  08:53:41  uejio
 * Added an ApplyToAllNodes function to traverse the node tree and apply 
 * a function to every node.  Finished Node Table except for % marks
 * 
 * Revision 1.3  90/09/25  09:11:36  uejio
 * Added a literal pool and changed the edges to be listed by port numbers.
 * 
 * Revision 1.2  90/09/24  10:05:35  uejio
 * This is really the initial revision
 *
 *  */

#include <strings.h>
#include <ctype.h>
#include "world.h"

static int MaxNodes=0;              /* total number of nodes in tree */

static PNODE CurrentGraph;          /* the current graph being examined */

static char *SymbolTableTypes[19];  /* types such as MULTIPLE, INTEGER, etc. */
typedef struct _NeededTypesType {
	int used;
} NeededTypesType;

static NeededTypesType *NeededTypes; /* array to check if types are 
					used in edges and for node info */

static int TypePoolSize;	/* size of type pool */
static int TypePoolHighest=0;	/* active size of type pool */
static int TypePoolSizeFactor=16; /* TypePoolSize=numnodes*factor*2  */
static int *TypePool;		/* type pool for edges */

static int MaxLiterals=0;            /* maximum number of literals. */
typedef struct _LiteralPoolType {
	char *value;
	int type;
	int edgetype;      /* needed by function literal for the type */
} LiteralPoolType;

static LiteralPoolType *LiteralPool; /* array of the e->constant,e->info->type. */

static char **EdgeNamePool;          /* pool of edgenames */
static int EdgeNamePoolSize=500;     /* initial size of edgename pool */
                                     /* note: EdgeNamePool starts at 0 so
					that there will always be an offset
					and FindEdgeName can return 0 for
					not found */

static int GraphNameLength=0;        /* length of longest graph name */
typedef void	((*Function)());     /* needed to pass functions */

/**************************************************************************/
/* LOCAL *************** UppercaseName      ***************************/
/**************************************************************************/
/* Purpose: Converts the name pointed to by p to uppercase                */
/**************************************************************************/

static void UppercaseName(p)
     char *p;
{
	if (!p) {
		return;
	}
	while (*p) {
		if ((*p>='a') && (*p <= 'z')) *p = *p - ' ';
		p++;
	}
}
       
/**************************************************************************/
/* LOCAL ***************   UppercaseGraphName   ***************************/
/**************************************************************************/
/* Purpose: Converts the name pointed to by p to uppercase and changes    */
/*          any '.' to '_'.                                               */
/**************************************************************************/

static void UppercaseGraphName(p)
     char *p;
{
	if (!p) {
		return;
	}
	while (*p) {
		if ((*p>='a') && (*p <= 'z')) *p = *p - ' ';
		if (*p=='.') *p='x';
		p++;
	}
}
       
/**************************************************************************/
/* LOCAL ***************    ApplyToEachNode     ***************************/
/**************************************************************************/
/* Purpose: Applys the function, funct, to all nodes starting from n      */
/**************************************************************************/

static void ApplyToEachNode(n,funct)
     PNODE n;
     Function funct;
{
	PNODE sg;
	funct(n);
	if (IsCompound(n))
	  for (sg = n->C_SUBS; sg!=NULL; sg=sg->gsucc)
	    ApplyToEachNode(sg,funct);
	if (IsGraph(n)) {
		CurrentGraph = n;
		for (sg = n->G_NODES; sg!=NULL; sg = sg->nsucc)
		  ApplyToEachNode(sg,funct);
	}
}

/**************************************************************************/
/* LOCAL ***************    ApplyToAllNodes     ***************************/
/**************************************************************************/
/* Purpose: Applys the function,funct, to all nodes starting from glstop  */
/**************************************************************************/

static void ApplyToAllNodes(funct)
     Function funct;
{
	PNODE f;
	for (f=glstop->gsucc; f!=NULL; f=f->gsucc) {
		CurrentGraph = f;
		ApplyToEachNode(f,funct);
	}
}

/**************************************************************************/
/* LOCAL ***************   PostApplyToEachNode  ***************************/
/**************************************************************************/
/* Purpose: Post Applys the function, funct, to all nodes starting from n */
/**************************************************************************/

static void PostApplyToEachNode(n,funct)
     PNODE n;
     Function funct;
{
	PNODE sg;
	if (IsCompound(n)) 
	  for (sg = n->C_SUBS; sg!=NULL; sg=sg->gsucc) {
		  ApplyToEachNode(sg,funct);
	  }
	if (IsGraph(n))
	  for (sg = n->G_NODES; sg!=NULL; sg = sg->nsucc) {
		  ApplyToEachNode(sg,funct);
	  }
	funct(n);
}

/**************************************************************************/
/* LOCAL ***************  PostApplyToAllNodes   ***************************/
/**************************************************************************/
/* Purpose: Post applys the function to all nodes starting from glstop    */
/**************************************************************************/

static void PostApplyToAllNodes(funct)
     Function funct;
{
	PNODE f;
	for (f=glstop->gsucc; f!=NULL; f=f->gsucc) {
		PostApplyToEachNode(f,funct);
	}
}

/**************************************************************************/
/* LOCAL  **************    HighestTypeNumber      ************************/
/**************************************************************************/
/* PURPOSE: Calculates the highest type number in the ihead structure     */
/**************************************************************************/

static int HighestTypeNumber()
{
	PINFO i;
	int high;
	high=0;
	for ( i = ihead; i != NULL; i = i->next ) {
		if (i->label > high)
		  high=i->label;
	}
	return high;
}

/**************************************************************************/
/* LOCAL  *************      BuildTypeStruct       ************************/
/**************************************************************************/
/* PURPOSE: Builds the SymbolTableTypes and NeededTypes structures        */
/**************************************************************************/

static void BuildTypeStruct()
{
	int highesttype,j;
	highesttype = HighestTypeNumber();
	NeededTypes = (NeededTypesType*)(MyAlloc((highesttype+1)*
						 sizeof(NeededTypesType)));
	for (j=0; j<=highesttype; j++) {
		NeededTypes[j].used = -1;
	}
	SymbolTableTypes[0] = "Array";
	SymbolTableTypes[1] = "Basic";
	SymbolTableTypes[2] = "Field";
	SymbolTableTypes[3] = "Function";
	SymbolTableTypes[4] = "Multiple";
	SymbolTableTypes[5] = "Record";
	SymbolTableTypes[6] = "Stream";
	SymbolTableTypes[7] = "Tag";
	SymbolTableTypes[8] = "Tuple";
	SymbolTableTypes[9] = "Union";
	SymbolTableTypes[10] = "Unknown";
	SymbolTableTypes[11] = "Buffer";
	SymbolTableTypes[12] = "Boolean";
	SymbolTableTypes[13] = "Character";
	SymbolTableTypes[14] = "Double";
	SymbolTableTypes[15] = "Integer";
	SymbolTableTypes[16] = "Null";
	SymbolTableTypes[17] = "Real";
	SymbolTableTypes[18] = "Nontype";
}

/**************************************************************************/
/* LOCAL  **************  WriteRecordUnionInfo     ************************/
/**************************************************************************/
/* PURPOSE: Writes the info node for Records and unions                   */
/**************************************************************************/

static void WriteRecordUnionInfo(i)
     PINFO i;
{
	PINFO field;
	int max;
	fprintf(output,"static TypeD TypeList%d[] = {",i->label);
	max = 0;
	if (i->R_FIRST) {
		max = 1;
		fprintf(output,"TP(Type%d)",i->R_FIRST->L_SUB->label);
	}
	for (field=i->R_FIRST->L_NEXT; field!=NULL; field = field->L_NEXT) {
		max++;
		fprintf(output,",TP(Type%d)",field->info2->label);
	}
	fprintf(output,"};\n");
	fprintf(output,"Define%s(Type%d,{\"%s\"},%d,TypeList%d);\n",
		SymbolTableTypes[i->type],i->label,i->name,max,i->label);
}

/**************************************************************************/
/* LOCAL  **************     MarkBaseType          ************************/
/**************************************************************************/
/* PURPOSE: Marks the base type of composite types in the                 */
/* NeededTypes.used array.  Must recursively mark records and unions.     */
/**************************************************************************/

static void MarkBaseType(i)
     PINFO i;
{
  PINFO field;
  int t;
  t = i->type;
  if (t>=BASIC_TYPE_START) t = BASIC_TYPE_START;
  switch(t) {
    case IF_RECORD:
    case IF_UNION:
      if ( NeededTypes[i->label].used == 1 ) break;
      for (field=i->R_FIRST; field!=NULL; field=field->L_NEXT) {
	MarkBaseType(field->L_SUB);
      }
      break;
    default:
      NeededTypes[i->label].used=1;
      break;
    }
}

/**************************************************************************/
/* LOCAL  **************         WriteTypes        ************************/
/**************************************************************************/
/* PURPOSE: WRITE THE INFO NODE LIST HEADED BY ihead TO OUTPUT.           */
/**************************************************************************/

static void WriteTypes()
{
	PINFO	i,field;
	int	t,n;

	fprintf( output, "/* Type Table */\n");

	/* loop thru the types and mark the base types as used. */
	for (i=ihead; i!=NULL; i=i->next) {
		t = i->type;
		if (t>=BASIC_TYPE_START) t = BASIC_TYPE_START;
		switch(t) {
		      case IF_STREAM:
		      case IF_ARRAY:
		      case IF_MULTPLE:
			NeededTypes[i->A_ELEM->label].used=1;
			break;
		      case IF_RECORD:
		      case IF_UNION:
                        NeededTypes[i->label].used=1;
			for (field=i->R_FIRST; field!=NULL; 
			     field=field->L_NEXT) {
				MarkBaseType(field->L_SUB);
			}
			break;
		      default:
			/* do nothing */
			break;
		}
	}
	for ( i = ihead; i != NULL; i = i->next )
	  if (NeededTypes[i->label].used>0) {
		  t = i->type;
		  if (t>=BASIC_TYPE_START) t=1;  /* basic type */
		  fprintf(output,"Forward%s(Type%d);\n",
			  SymbolTableTypes[t],i->label);
	  }
	for ( i = ihead; i != NULL; i = i->next )
	  if (NeededTypes[i->label].used>0) {
	    n = i->label;
	    if ( i->name==NULL ) {
		    i->name = (char*)MyAlloc((int)(log10((double)i->label)+3));
		    sprintf(i->name,"t%d",n);
	    }
	    t = i->type;
	    if (t>=BASIC_TYPE_START) t = BASIC_TYPE_START;
	    switch(t) {
		  case IF_FUNCTION:
		  case IF_TUPLE:
		  case IF_FIELD:
		  case IF_TAG:
		    fprintf(output,
			    "Define%s(Type%d,{\"%s\"});\n",
			    SymbolTableTypes[t],n,i->name);
		    break;
		  case IF_STREAM:
		  case IF_ARRAY:
		  case IF_MULTPLE:
		  case IF_UNKNOWN:
		    fprintf(output,
			    "Define%s(Type%d,{\"%s\"},Type%d);\n",
			    SymbolTableTypes[t],n,i->name,i->info1->label);
		    break;
		  case BASIC_TYPE_START:
		    fprintf(output,
			    "DefineBasic(Type%d,{\"%s\"},IF1%s);\n",
			    n,i->name,SymbolTableTypes[i->type]);
		    break;
		  case IF_RECORD:
		  case IF_UNION:
		    WriteRecordUnionInfo(i);
		    break;
		  default:
		    fprintf(output,
			    "DefineUnsupported(Type%d,{\"%s\"},Unknown);\n",
			    SymbolTableTypes[t],n,i->name);
		    break;
	    }	    
          }
}

/**************************************************************************/
/* LOCAL  **************     CountImportPorts      ************************/
/**************************************************************************/
/* PURPOSE: Returns number of edges in the import edgelist                */
/**************************************************************************/

static int CountImportPorts(edge)
     PEDGE edge;
{
	PEDGE e;
	int i;
	i = 0;
	for (e = edge; e!=NULL; e = e->isucc)
	  if (e->iport>i) i=e->iport;
	return i;
}

/**************************************************************************/
/* LOCAL  **************     CountExportPorts      ************************/
/**************************************************************************/
/* PURPOSE: Returns number of edges in the export edgelist                */
/**************************************************************************/

static int CountExportPorts(edge)
     PEDGE edge;
{
	PEDGE e;
	int i;
	i = 0;
	for (e = edge; e!=NULL; e = e->esucc)
	  if (e->eport>i) i=e->eport;
	return i;
}

/**************************************************************************/
/* LOCAL  **************        CountPorts         ************************/
/**************************************************************************/
/* PURPOSE: Counts the number of import and export ports and returns the  */
/* max number.                                                            */
/**************************************************************************/

static int CountPorts(n,max)
     PNODE n;
     int max;
{
	int numimp, numexp;
	PNODE sg;
	numimp = 0; numexp = 0;
	if (n->imp) numimp = CountImportPorts(n->imp);
	if (n->exp) numexp = CountExportPorts(n->exp);
	if (numimp>max) max = numimp;
	if (numexp>max) max = numexp;
	if (IsCompound(n)) {
		for (sg=n->C_SUBS; sg!=NULL; sg=sg->gsucc) {
			max = CountPorts(sg,max);
		}
	} else if (IsGraph(n)) {
		for (sg = n->G_NODES; sg!=NULL; sg = sg->nsucc) {
			max = CountPorts(sg,max);
		}
	}
	/* printf ("CountPorts: looking at node %d, max is %d\n",
		n->label,max); */
	return max;
}

/**************************************************************************/
/* LOCAL  **************      WriteTypeList        ************************/
/**************************************************************************/
/* PURPOSE: Writes an int type list                                       */
/**************************************************************************/

static void WriteTypeList(size,TypeList)
     int size;
     int *TypeList;
{
	int i;
	if (TypeList[0]== 0 )
	  fprintf(output,"{NULL");
	else
	  fprintf(output,"{TP(Type%d)",TypeList[0]);
	for (i=1; i<size; i++) {
		if (TypeList[i]>=0) {
			if ((i%5)==0)
			  fprintf(output,",\n\t");
			else
			  fprintf(output,",");
			if (TypeList[i]==0)
			  fprintf(output,"NULL");
			else
			  fprintf(output,"TP(Type%d)",TypeList[i]);
		}
	}
	fprintf(output,"};\n");
}

/**************************************************************************/
/* LOCAL  **************        FindIntItem        ************************/
/**************************************************************************/
/* PURPOSE: Finds an item in the int pool.                                */
/* Returns the position of the item in the int pool or -1 if not found    */
/**************************************************************************/

static int FindIntItem(pool,length,item)
     int *pool;
     int length;
     int *item;
{
	int i,j,first,found,highest;
	i = 0; j = 0; first = 0;
	found = 0;
	for (highest = 0; pool[highest]>=0; highest++);
	/* if length of item less than size of remaining pool and not found 
	   then keep searching. */
	while ((length<=(highest-first+1))&&(!found)) {
		if (pool[i]==item[j]) {
			i++;
			j++;
		} else {
			first++;
			i=first;
			j=0;
		}
		if (j>=length) found = 1;
	}
	if (!found)
	  return -1;
	else
	  return first;
}

/**************************************************************************/
/* LOCAL  **************        AddIntItem         ************************/
/**************************************************************************/
/* PURPOSE: Adds a set of items to the int pool                           */
/**************************************************************************/

static void AddIntItem(poolptr,highestptr,maxptr,length,item)
     int **poolptr;
     int *highestptr;
     int *maxptr;
     int length;
     int *item;
{
  int *pool = (*poolptr);
  int *newpool;
  int highest = (*highestptr);
  int i;

  /* ------------------------------------------------------------ */
  /* Must extend and recopy the pool if there's not enough room*/
  if ( highest+length > *maxptr ) {
    *maxptr *= 2;
    newpool = (int*)(MyAlloc(((*maxptr)+1)*sizeof(int)));
    for(i=0;i<highest;i++) newpool[i] = pool[i];
    for(;i<=*maxptr;i++) newpool[i] = -1;
    free((char*)pool);
    *poolptr = newpool;
    pool = newpool;
  }

  for (i=0; i<length; i++) {
    pool[highest++] = item[i]; 
  }

  *highestptr = highest;
}

/**************************************************************************/
/* LOCAL  **************        WriteNewType       ************************/
/**************************************************************************/
/* PURPOSE: Writes a new type for node n (import or export)               */
/**************************************************************************/

static void WriteNewType(n,edgetype,max,itemlist)
     int n;
     char* edgetype;
     int max;
     int *itemlist;
{
  fprintf(output,"static TypeD %sN%dType[] = ",edgetype,n);
  WriteTypeList(max,itemlist);
}

/**************************************************************************/
/* LOCAL  **************        AddEdgeTypes       ************************/
/**************************************************************************/
/* PURPOSE: Fills the TypePool with all the edges                         */
/**************************************************************************/

static void AddEdgeTypes(n)
     PNODE n;
{
	PEDGE e;
	int max,i;
	int *newitems;

	if (n->imp) {
		max = CountImportPorts(n->imp);
		newitems = (int*)(MyAlloc((max+1)*sizeof(int)));
		for (i=0; i<max; i++)
		  newitems[i] = 0;
		for (e=n->imp; e!=NULL; e=e->isucc) {
			newitems[e->iport-1]=e->info->label;
		}
		if (FindIntItem(TypePool,max,newitems)<0)
		  if (max>TypePoolSizeFactor)
		    WriteNewType(n->label,"Imp",max,newitems);
		  else
		    AddIntItem(&TypePool,&TypePoolHighest,&TypePoolSize,
			       max,newitems);
		free((char*)newitems);
	}
	if (n->exp) {
		max = CountExportPorts(n->exp);
		newitems = (int*)(MyAlloc((max+1)*sizeof(int)));
		for (i=0; i<max; i++)
		  newitems[i] = 0;
		for (e=n->exp; e!=NULL; e=e->esucc) {
			newitems[e->eport-1]=e->info->label;
		}
		if (FindIntItem(TypePool,max,newitems)<0)
		  if (max>TypePoolSizeFactor)
		    WriteNewType(n->label,"Exp",max,newitems);
		  else
		    AddIntItem(&TypePool,&TypePoolHighest,&TypePoolSize,
			       max,newitems);
		free((char*)newitems);
	}
}

/**************************************************************************/
/* LOCAL  **************   InitializeEdgePools     ************************/
/**************************************************************************/
/* PURPOSE: Initializes the edge type pool and edge name pool             */
/**************************************************************************/

static void InitializeEdgePools()
{
	int max,i;
	PNODE f;
	max = 0;
	for (f = glstop->gsucc; f!=NULL; f=f->gsucc) {
		max = CountPorts(f,max);
	}
	TypePoolSize = max*2*TypePoolSizeFactor;
	TypePool = (int*)(MyAlloc((TypePoolSize+1)*sizeof(int)));
	for (i=0; i<=TypePoolSize; i++) {
		TypePool[i] = -1;
	}
	TypePool[TypePoolSize] = -2; /* FENCE */
	/* use size + 2 because EdgeNamePool starts from 0. */
	EdgeNamePool = (char**)(MyAlloc((EdgeNamePoolSize+2)*sizeof(char*)));
	for (i=0; i<=EdgeNamePoolSize; i++) {
		EdgeNamePool[i] = NULL;
	}
}

/**************************************************************************/
/* LOCAL  ************** WriteVectorOfOffsetAddr   ************************/
/**************************************************************************/
/* PURPOSE: Writes                                                        */
/* "static <name><num>[]={<basename>+<listitem>,...};"                    */
/**************************************************************************/

static void WriteVectorOfOffsetAddr(name,nodenum,basename,numitems,list)
     char *name;
     int nodenum;
     char *basename;
     int numitems;
     int *list;
{
	int i;
	if (numitems>0) {
		fprintf(output,"static %s%d[] = {",name,nodenum);
		if (list[0])
		  fprintf(output,"%s+%d",basename,list[0]);
		else
		  fprintf(output,"NULL");
		for (i=1; i<numitems; i++) {
			if ((i%3)==0)
			  fprintf(output,",\n\t");
			else
			  fprintf(output,",");
			if (list[i])
			  fprintf(output,"%s+%d",basename,list[i]);
			else
			  fprintf(output,"NULL");
		}
		fprintf(output,"};\n");
	}
}

/**************************************************************************/
/* LOCAL  **************  CountSubNodesAndExpr     ************************/
/**************************************************************************/
/* PURPOSE: Counts the number of subexprs and subnodes in graph n         */
/**************************************************************************/

static void CountSubNodesAndExpr(n,numnodes,numexpr)
     PNODE n;
     int *numnodes, *numexpr;
{
	PNODE sub;
	int i,maxout,*portfound,Fanout();

	*numexpr = 0; *numnodes = 0;
	if (n->nsucc) 
	{
	  for (sub = n->nsucc; sub!=NULL; sub=sub->nsucc) {
		  PEDGE e;
		  *numnodes = *numnodes+1;
		  maxout = CountExportPorts(sub->exp);
		  portfound = (int *)MyAlloc((maxout+1)*sizeof(int));
		  for (i=0; i<=maxout; i++) portfound[i] = 0;
		  for (e = sub->exp; e!=NULL; e=e->esucc)
		    if (!portfound[e->eport]) {
			    portfound[e->eport] = 1;
			    if ((e->dst!=n)||(e->sr>1)||(Fanout(e)>1))
			      *numexpr = *numexpr+1;
		    }
		  free((char*)portfound);
	  }
	}
}

/**************************************************************************/
/* LOCAL  **************      FindEdgeName         ************************/
/**************************************************************************/
/* PURPOSE: Returns the position of the name in EdgeNamePool              */
/**************************************************************************/

static int FindEdgeName(name)
     char *name;
{
	int i;
	for (i=1; EdgeNamePool[i]!=NULL; i++)
	  if (strcmp(EdgeNamePool[i],name)==0)
	    return i;
	return 0;
}

/**************************************************************************/
/* LOCAL  **************    WriteInfoForward       ************************/
/**************************************************************************/
/* PURPOSE: Write the types needed for edgenames, compounds, and graphs   */
/**************************************************************************/

static void WriteInfoForward(n)
     PNODE n;
{
	int	i, *intlist;
	PNODE	sg;

	/* write the special types required by nodes */
	AddEdgeTypes(n);

	if (IsCompound(n)) {
		PALIST alist;
		intlist=(int*)
		  (MyAlloc((n->scnt+1)*sizeof(char*)));
		/* create a vector of children for compounds */
		i=0;
		for (alist = n->alst; alist!=NULL; alist=alist->next) {
			int j;
			j=0;
			for (sg=n->gsucc; (sg!=NULL)&&(j++<alist->datum); 
			     sg=sg->gsucc)
			  ;
			intlist[i++] = sg->label;
		}
		WriteVectorOfOffsetAddr("NodeInfo Children",n->label,
					"NodeTable",n->scnt,intlist);
		free((char*)intlist);
	}
}

/**************************************************************************/
/* LOCAL ***************    UppercaseEdgeName   ***************************/
/**************************************************************************/
/* PURPOSE: Converts the name in edge e->name to uppercase                */
/* Also replaces all dashes with spaces                                   */
/**************************************************************************/

static void UppercaseEdgeName(e)
     PEDGE e;
{
	char *p;
	p = e->name;
	if (!p) {
		return;
	}
	while (*p) {
		if ((*p>='a') && (*p <= 'z')) *p = *p - ' ';
		if (*p=='-') *p = ' ';
		p++;
	}
}
       
/**************************************************************************/
/* LOCAL ***************       RenumberNodes    ***************************/
/**************************************************************************/
/* PURPOSE: Renumber all the nodes from n                                 */
/* Increments the global MaxNodes                                         */
/**************************************************************************/

static void RenumberNodes(n)
     PNODE n;
{
  if (!IsGraph(n)) {
    MaxNodes++;
    n->label = MaxNodes;
  }
}

/**************************************************************************/
/* LOCAL ***************      RenumberGraphs    ***************************/
/**************************************************************************/
/* PURPOSE: Renumber all the graphs from n                                */
/* Increments the global MaxNodes                                         */
/**************************************************************************/

static void RenumberGraphs(n)
     PNODE n;
{
  if (IsGraph(n)) {
    MaxNodes++;
    n->label = MaxNodes;
  }
}

/**************************************************************************/
/* LOCAL  **************     InsertEdgeName        ************************/
/**************************************************************************/
/* PURPOSE: Inserts a name into the EdgeNamePool.                         */
/**************************************************************************/

static void InsertEdgeName(name)
     char *name;
{
	int i,found;
	char **NewPool;
	static int factor=1;
	found = 0;
	for (i = 1; EdgeNamePool[i]!=NULL; i++) {
		if (strcmp(EdgeNamePool[i],name)==0) {
			found = 1;
			break;
		}
	}
	/* if reached size of pool then increase the pool size
	   by EdgeNamePoolSize and copy the pool */
	if (i>=(factor*EdgeNamePoolSize)) {
		factor = factor+1;
		/* use size + 2 because EdgeNamePool starts from 0. */
		NewPool = (char**)
		  (MyAlloc((EdgeNamePoolSize*factor+2)*sizeof(char*)));
		for (i=0; i<=(EdgeNamePoolSize*factor); i++)
		  NewPool[i] = NULL;
		for (i=1; EdgeNamePool[i]!=NULL; i++)
		  NewPool[i] = EdgeNamePool[i];
		free((char*)EdgeNamePool);
		EdgeNamePool = NewPool;
	}
	if (!found) {
		EdgeNamePool[i] = name;
	}
}

/**************************************************************************/
/* LOCAL  **************        MakeUpName         ************************/
/**************************************************************************/
/* PURPOSE: Makes up a name for edge e and stores it in e->name           */
/**************************************************************************/

static void MakeUpName(e)
     PEDGE e;
{
	if (!e->src) {
		if (!e->constant) /* could be an error value */
		  e->name = ERROR_CONSTANT;
		else {
			e->name = (char*)MyAlloc(strlen(e->constant)*
						 (sizeof(char))+1);
			strcpy(e->name,e->constant);
		}
		
	} else {
		e->name = MyAlloc((int)(log10((double)e->iport))
				  +(int)(log10((double)e->eport))
				  +(int)(log10((double)e->src->label))
				  +(int)(log10((double)e->dst->label))
				  +12+4);
		sprintf(e->name,"EDGE_%d.%d_TO_%d.%d",e->src->label,e->eport,
			e->dst->label,e->iport);
		/* printf("e->name=%s\n",e->name); */
	}
}

/**************************************************************************/
/* LOCAL ***************     RenameEdgesEtc     ***************************/
/**************************************************************************/
/* PURPOSE: Renames all the edges using all caps and names like           */
/* EDGE x PORT y, modifies NeededTypes array, calculates MaxLiterals,     */
/* calculates edge marks, sets GraphNameLength (for WriteGraphNames),     */
/* writes forward declarations for graphs, and inserts edge names and,    */
/* types into their respective pools.                                     */
/**************************************************************************/

static void RenameEdgesEtc(n)
     PNODE n;
{
	PEDGE e;
	int numexp,*expports,i,length;
	/* calculate GraphNameLength and 
	   write forward declarations for graphs */
	if (IsGraph(n)) {
		if (n->gname) {
			UppercaseGraphName(n->gname);
			length = strlen(n->gname)+1;
		} else {
			length = (int)(log10((double)n->label)+3); /* Gr%d */
		}
		if (length>GraphNameLength) {
			GraphNameLength = length;
		}
		if ((n->gname)&&(IsXGraph(n))) {
			fprintf(output,"void %s_();\n",n->gname);
			fprintf(output,"void %s_Setup();\n",n->gname);
		} else if ((n->gname)&&(IsIGraph(n))) {
			fprintf(output,"extern void %s_();\n",n->gname);
		} else if (n->gname) {
			fprintf(output,"static void %s_();\n",n->gname);
			fprintf(output,"static void %s_Setup();\n",n->gname);
		} else {
			fprintf(output,"static void Gr%d();\n",n->label);
			fprintf(output,"static void Gr%dSetup();\n",n->label);
		}
	}
	/* allocate storage for port array and zero the array */
	numexp = CountExportPorts(n->exp);
	expports = (int*)(MyAlloc((numexp+1)*sizeof(int)));
	for (i=0; i<=numexp; i++)
	  expports[i] = 0;

	/* calculate the fan out on a port basis */
	for (e=n->exp; e!=NULL; e=e->esucc) {
		expports[e->eport]++;
		/* printf("node %d, e->eport %d, expports[eport] %d\n",
		       n->label,e->eport,expports[e->eport]); */
	}
	/* rename import edges */
	for (e=n->imp; e!=NULL; e=e->isucc) {
		if ((!e->name)||(!e->src)) {
			MakeUpName(e);
		}
		UppercaseEdgeName(e);
		NeededTypes[e->info->label].used = 1;
		if (!e->src) {
			/* printf("found a literal on edge %s, node %d\n",
			       e->name,n->label); */
			MaxLiterals++;
		}
		InsertEdgeName(e->name);
	}

	/* rename export edges and set the %sr pragma by port */
	for (e=n->exp; e!=NULL; e=e->esucc) {
		if ((!e->name)||(!e->src)) {
			MakeUpName(e);
		}
		UppercaseEdgeName(e);
		NeededTypes[e->info->label].used = 1;
		if (!e->src) {
			/* printf("found a literal on edge %s, node %d\n",
			       e->name,n->label); */
			MaxLiterals++;
		}

		InsertEdgeName(e->name);
	}
	MaxLiterals++;
	free((char*)expports);
}

/**************************************************************************/
/* LOCAL ***************  CalculateEdgeOffsets  ***************************/
/**************************************************************************/
/* PURPOSE: Calculates the edge offsets in the e->offset field for all    */
/* the subnodes of a graph                                                */
/**************************************************************************/

static void CalculateEdgeOffsets(n)
     PNODE n;
{
	PNODE sub;
	PEDGE e;
	int offset;
	int numexp,*expports,i;
	int Fanout();
	       
	if (IsGraph(n)) {
		offset = 1;
		for (sub=n->nsucc; sub!=NULL; sub=sub->nsucc) {
			/* allocate storage for port array and zero the 
			   array */
			numexp = CountExportPorts(sub->exp);
			expports = (int*)(MyAlloc((numexp+1)*sizeof(int)));
			for (i=0; i<=numexp; i++)
			  expports[i] = 0;
		
			/* mark which ports are used */
			for (e=sub->exp; e!=NULL; e=e->esucc) {
				/* offsets are needed for edges which are
				   completely internal or which have a fan 
				   out */
				if ((e->dst!=n)||(e->sr>1)||(Fanout(e)>1))
				  expports[e->eport]=1;
			}
			/* set offsets on a port basis 
			   (ports are numbered from 1) */
			for (i=1; i<=numexp; i++) {
				if (expports[i])
				  expports[i]=offset++;
			}
			/* set e->offset to the corresponding port offset */
			for (e=sub->exp; e!=NULL; e=e->esucc) {
				e->offset = expports[e->eport];
			}
			free((char*)expports);
		}
	}
}

/**************************************************************************/
/* LOCAL ***************    WriteEdgeNamePool   ***************************/
/**************************************************************************/
/* PURPOSE: Writes a pool for the edge names                              */
/**************************************************************************/

static void WriteEdgeNamePool()
{
	int max,i,highest,longest;
	PNODE f;
	max = 0;
	for (f = glstop->gsucc; f!=NULL; f=f->gsucc) {
		max = CountPorts(f,max);
	}
	fprintf(output,"/* Edge Name Pool */\n");
	longest = 0;
	for (highest = 1; EdgeNamePool[highest]!=NULL; highest++) {
		if (strlen(EdgeNamePool[highest])>longest) 
		  longest = strlen(EdgeNamePool[highest]);
	}
	/* the first entry is always a NULL entry */
	fprintf(output,"static char ENP[][%d] = {\n\t\"NULL\"",longest+1);
	for (i=1; i<highest; i++) {
		if ((i%4)==0)
		  fprintf(output,",\n\t");
		else
		  fprintf(output,",");
		fprintf(output,"\"%s\"",EdgeNamePool[i]);
	}
	fprintf(output,"\n};\n");
}

/**************************************************************************/
/* LOCAL  **************        WriteEdgeDesc      ************************/
/**************************************************************************/
/* PURPOSE: Prints a description of the edges for this graph or node      */
/**************************************************************************/

static WriteEdgeDesc(n)
     PNODE n;
{
	char **impnames,**expnames;
	PEDGE e;
	int max,i,offset,numimp,numexp;
	int *newitems;

	/* INARITY,OUTARITY */
	numimp = CountImportPorts(n->imp);
	numexp = CountExportPorts(n->exp);
	fprintf(output,",%d,%d",numimp,numexp);

	/* InTYPES */
	impnames = (char**)(MyAlloc((numimp+1)*sizeof(char*)));
	expnames = (char**)(MyAlloc((numexp+1)*sizeof(char*)));
	if (n->imp) {
	  /* Get a list of input types */
		max = numimp;
		newitems = (int*)(MyAlloc((max+1)*sizeof(int)));
		for (i=0; i<max; i++)
		  newitems[i] = 0;
		for (e=n->imp; e!=NULL; e=e->isucc) {
			newitems[e->iport-1]=e->info->label;
		}

		/* If we can find it in the pool, then use its offset, */
		/* otherwise, a input type array was prepared separately. */
		offset = FindIntItem(TypePool,max,newitems);
		if (offset>=0) {
			fprintf(output,",NTypes+%d",offset);
		} else {
			fprintf(output,",ImpN%dType",n->label);
		}
		free((char*)newitems);
	} else {
		fprintf(output,",NULL");
	}

	/* OutTypes */
	if (n->exp) {
	  /* Get a list of output types */
		max = numexp;
		newitems = (int*)(MyAlloc((max+1)*sizeof(int)));
		for (i=0; i<max; i++)
		  newitems[i] = 0;
		for (e=n->exp; e!=NULL; e=e->esucc) {
			newitems[e->eport-1]=e->info->label;
		}

		/* If we can find it, use the offset in the type pool. */
		/* If not, then use the separately created array. */
		offset = FindIntItem(TypePool,max,newitems);
		if (offset>=0) {
			fprintf(output,",NTypes+%d",offset);
		} else {
			fprintf(output,",ExpN%dType",n->label);
		}
		/* WriteTypeList(max,newitems); */
		free((char*)newitems);
	} else {
		fprintf(output,",NULL");
	}
	free((char*)impnames);
	free((char*)expnames);

	/* EdgeNames */
	fprintf(output,",NULL");
}

/**************************************************************************/
/* LOCAL ***************    WriteOneGraphName   ***************************/
/**************************************************************************/
/* PURPOSE: Writes the name of one graph                                  */
/**************************************************************************/

static void WriteOneGraphName(n)
     PNODE n;
{
	if (IsGraph(n)) {
		if (n->gname) {
			fprintf(output,"\n\t\"%s_\",",n->gname);
		} else {
			fprintf(output,"\n\t\"Gr%d\",",n->label);
		}
	}
}

/**************************************************************************/
/* LOCAL ***************    WriteGraphNames     ***************************/
/**************************************************************************/
/* PURPOSE: Writes the graph names to an array                            */
/**************************************************************************/

static void WriteGraphNames()
{
	fprintf(output,"\nstatic char GN[][%d] = {",GraphNameLength+1);
	ApplyToAllNodes(WriteOneGraphName);
	fprintf(output,"\n};\n\n");
}

/**************************************************************************/
/* LOCAL  **************      WriteGraphNode       ************************/
/**************************************************************************/
/* PURPOSE: Print the node table entry for this graph node.               */
/**************************************************************************/

static void WriteGraphNode( n )
PNODE n;
{
	static	gp_counter = 0;
	int	numexpr,numnodes;
	char	*name;

	if (IsGraph(n)) {
		fprintf(output,",\n\t{%d",n->label);
		WriteEdgeDesc(n);
		/* Write out the parent field for subgraphs from */
		/* compounds.  The parent of function graphs is */
		/* itself. */
		if ( n->G_DAD ) {
		  fprintf(output,",NodeTable+%d,\n\t  ",n->G_DAD->label);
		} else {
		  fprintf(output,",NodeTable+%d,\n\t  ",gp_counter+1);
		}
		fprintf(output,"Graph,%s",IsXGraph(n)?"TRUE":"FALSE");
		CurrentGraph = n;
		if (n->gname) {
			name=(char*)MyAlloc(sizeof(char)*(strlen(n->gname)+2));
			(void)sprintf(name,"%s_",n->gname);
		} else {
			name=(char*)MyAlloc(sizeof(char)*11);
			(void)sprintf(name,"Gr%d",n->label);
		}
		CountSubNodesAndExpr(n,&numnodes,&numexpr);
		fprintf(output,",{%d,%d,",numexpr,numnodes);
		fprintf(output,"NULL,");
		fprintf(output,"NULL,");
		fprintf(output,
			IsIGraph(n)?"GP(GN+%d),GP(%s),NULL}"
			:"GP(GN+%d),GP(%s),GP(%sSetup)}",
			gp_counter++,name,name);
		free((char*)name);
		fprintf(output,",{%d}}",n->line);
	}
}

/**************************************************************************/
/* LOCAL  **************  InitializeLiteralPool    ************************/
/**************************************************************************/
/* PURPOSE: Initializes the literal pool to all NULLs                     */
/**************************************************************************/

static void InitializeLiteralPool()
{
	int i;
	LiteralPool =(LiteralPoolType*)MyAlloc((MaxLiterals+1)*
					       sizeof(LiteralPoolType));
	for (i=0; i<=MaxLiterals; i++) {
		LiteralPool[i].value = NULL;
		LiteralPool[i].type = -1;
		LiteralPool[i].edgetype = -1;
	}
}

/**************************************************************************/
/* LOCAL  **************       FindLiteral         ************************/
/**************************************************************************/
/* PURPOSE: Finds a string in the literal pool                            */
/**************************************************************************/

static int FindLiteral(value,type,edgetype)
     char* value;
     int type;
     int edgetype;
{
	int i;
	for (i=1; i<=MaxLiterals; i++) {
		if ((!value)&&(!LiteralPool[i].value)&&
		    (LiteralPool[i].type==type)&&
		    (LiteralPool[i].edgetype==edgetype)) /* an error value */
		  return i;
		if ((LiteralPool[i].value)&&(value)&&
		    (strcmp(LiteralPool[i].value,value)==0)&&
		    (LiteralPool[i].type==type)&&
		    (LiteralPool[i].edgetype==edgetype))
		  return i;
	}
	return 0;
}

/**************************************************************************/
/* LOCAL  **************       AddLiteral          ************************/
/**************************************************************************/
/* PURPOSE: Adds a literal to the literal pool                            */
/**************************************************************************/

static int AddLiteral(value,type,edgetype)
     char* value;
     int type;
     int edgetype;
{
	int i,Pos;

	Pos = FindLiteral(value,type,edgetype);

	if (!Pos) {
		for (i=1; i<=MaxLiterals; i++)
		  if ((LiteralPool[i].value==NULL)&&(LiteralPool[i].type<0)&&
		      (LiteralPool[i].edgetype<0)) {
			  LiteralPool[i].value = value;
			  LiteralPool[i].type = type;
			  LiteralPool[i].edgetype = edgetype;
			  return i;
		  }
		/* actually this is an error and should be handled 
		   by a message or increasing the size of the pool? */
		LiteralPool[i].value = value;
		return i;
	}

	return Pos;

}
			  
/**************************************************************************/
/* LOCAL  **************    WriteNodeLiterals      ************************/
/**************************************************************************/
/* PURPOSE: Writes the literal values for all the import                  */
/* edges.  Also builds the LiteralPool.                                   */
/**************************************************************************/

static void WriteNodeLiterals(n)
     PNODE n;
{
	PEDGE e;

	for (e=n->imp; e!=NULL; e=e->isucc) {
		if (!e->src) {
			if ((e->iport==1)&&(n->type==IFCall)) {
			  UppercaseName(e->constant);
			}
			if (!FindLiteral(e->constant,e->info->type,
					 e->info->label)) {
				(void)AddLiteral(e->constant,e->info->type,
						 e->info->label);
			}
		}
	}
}

/**************************************************************************/
/* LOCAL  **************     QuoteString           ************************/
/**************************************************************************/
/* PURPOSE: returns a quoted string                                       */
/**************************************************************************/

static void QuoteString(s,qs)
     char	*s;
     char	*qs;
{
  for(;*s;s++) {
    switch (*s) {
     case '"':
      *qs++ = '\\';
      *qs++ = '"';
      break;
     case '\\':
      *qs++ = '\\';
      *qs++ = '\\';
      break;
     default:
      *qs++ = *s;
    }
  }
  *qs = '\0';
}

/**************************************************************************/
/* LOCAL  **************     WriteLiterals         ************************/
/**************************************************************************/
/* PURPOSE: Prints the literal pool                                       */
/**************************************************************************/

static int WriteLiterals()
{
	int i,numliterals,found,longest;
	char *quoted;

	InitializeLiteralPool();
	if (MaxLiterals>0) {
		ApplyToAllNodes(WriteNodeLiterals);
	}
	found = 0;
	for (i=1; ((LiteralPool[i].type>=0)&&(LiteralPool[i].edgetype>=0));
	     i++)
	  if (LiteralPool[i].value) {
		  found = 1;
		  break;
	  }
	if (!found) {
		fprintf(output,"NoLiterals\n");
		return 0;
	}
	fprintf(output,"/* Literal Pool */\n");
	longest = 0;
	for (i=1; ((LiteralPool[i].type>=0)&&(LiteralPool[i].edgetype>=0));
	     i++) {
	  int		len;	/* Working length of entry */
	  if (LiteralPool[i].value) {
	    len = strlen(LiteralPool[i].value);
	  } else if (LiteralPool[i].type >= 0) {
	    len = 5;		/* Length of the word ``error'' */
	  }
	  if (len>longest) longest = len;
	}

	fprintf(output,"static char LP[][%d] = {\n",longest+1);
	for (i=1; ((LiteralPool[i].type>=0)&&(LiteralPool[i].edgetype>=0));
	     i++) {
	  if (LiteralPool[i].value) {
	    quoted=(char*)MyAlloc((strlen(LiteralPool[i].value)+1)
				   *2*sizeof(char));
	    QuoteString(LiteralPool[i].value,quoted);
	    fprintf(output,"\t\"%s\",\t/* %d */\n",quoted,i);
	    free((char*)quoted);
	  } else if ( LiteralPool[i].type >= 0 ) {
	    fprintf(output,"\t\"ERROR\",\t/* %d */\n",i);
	  }
	}
	fprintf(output,"};\n");
	numliterals = 0;
	fprintf(output,"LiteralPool{\n");
	for (i=1; ((LiteralPool[i].type>=0)&&(LiteralPool[i].edgetype>=0));
	     i++) {
	  if (LiteralPool[i].value || LiteralPool[i].type >= 0) {
	    fprintf(output,"\tLiteral(Type%d,LP+%d),\n",
		    LiteralPool[i].edgetype,numliterals);
	    numliterals++;
	  }
	} 
	fprintf(output,"};\n");
	return numliterals;
}

/**************************************************************************/
/* LOCAL  **************   WriteEnterGraphLine     ************************/
/**************************************************************************/
/* PURPOSE: Prints the EnterGraph(...) line.                              */
/**************************************************************************/

static void WriteEnterGraphLine(n)
     PNODE n;
{
	int maxin, maxout, temp, numlocals, numnodes;
	PNODE sub;
	/* find the max in/out of all the subnodes and
	   count the subnodes and expressions */
	maxin = 0; maxout = 0;
	for (sub=n->nsucc; sub!=NULL; sub=sub->nsucc) {
		temp = CountImportPorts(sub->imp);
		if (temp>maxin) maxin = temp;
		temp = CountExportPorts(sub->exp);
		if (temp>maxout) maxout = temp;
	}
	CountSubNodesAndExpr(n,&numnodes,&numlocals);
	
	fprintf(output,") \{\n\tEnterGraph(%d,%d,%d,%d);\n\n",
		n->label,numlocals,maxin,maxout);
}

/**************************************************************************/
/* LOCAL  **************   Fanout     ************************/
/**************************************************************************/
/* PURPOSE: Find out how many edges in the output chain share this port */
/**************************************************************************/

static int Fanout(E)
     PEDGE	E;
{
  PNODE		N;
  int		Port;
  int		PortCount;

  N = E->src;
  Port = E->eport;

  for(PortCount=0,E=N->exp; E; E=E->esucc) {
    if ( E->eport == Port ) PortCount++;
  }

  return PortCount;  
}
/**************************************************************************/
/* LOCAL  **************   GenerateOutputCalls     ************************/
/**************************************************************************/
/* PURPOSE: Prints the ArgOutput, Local output or null output lines.      */
/**************************************************************************/

static void GenerateOutputCalls(parent,n)
     PNODE parent,n;
{
  PEDGE e;
  int maxout,i,*portfound;
  /* allocate storage for array to keep track of
     used export ports */
  maxout = CountExportPorts(n->exp);
  portfound = (int *)MyAlloc((maxout+1)*sizeof(int));
  for (i=0; i<=maxout; i++) portfound[i] = 0;
	
  /* generate ArgOutput or LocalOutput calls */
  for (e=n->exp; e!=NULL; e=e->esucc)
    /* only process each port once. */
    if (!portfound[e->eport]) {
      i = FindEdgeName(e->name);
      portfound[e->eport]=1;
      if (e->dst==parent && Fanout(e)==1) {
	fprintf(output,
		"\tArgOutput(%d,%d,CP(ENP+%d)); /* %s */\n",
		e->eport,e->iport,i,e->name);
      } else {
	fprintf(output,
		"\tLocalOutput(%d,%d,CP(ENP+%d)); /* %s */\n",
		e->eport,e->offset,i,e->name);
      }
    }
  /* dump out null outputs */
  for (i=1; i<=maxout; i++)
    if (!portfound[i])
      fprintf(output,"\tNullOutput(%d);\n",i);
  free((char*)portfound);
}
		
/**************************************************************************/
/* LOCAL  **************       WriteNodeLine       ************************/
/**************************************************************************/
/* PURPOSE: Write the NodeExecute line                                    */
/**************************************************************************/

static void WriteNodeLine( n )
PNODE n;
{
  if (IsCompound(n)) {
    fprintf(output,"\tCNodeExecute(%d",n->label);
    WriteEdgeDesc(n);
    /* parent */
    fprintf(output,",\n\t\tNodeTable+%d,",CurrentGraph->label);
    fprintf(output,"Compound,FALSE");
    fprintf(output,",{%d,%d,GP(Children%d)}", n->type,n->scnt,n->label);
  } else if (IsGraph(n)) {
    ; /* do nothing */
  } else if (IsSimple(n)) {
    fprintf(output,"\tNodeExecute(%d",n->label);
    WriteEdgeDesc(n);
    /* parent */
    fprintf(output,",\n\t\tNodeTable+%d,",CurrentGraph->label);
    fprintf(output,"Simple,FALSE");
    fprintf(output,",{%d}",n->type);
  } else { /* is function ??? */
    fprintf(output,"Unknown,");
    fprintf(output,"{%d,%d,%s}",n->label,n->label,n->gname);
  }
  fprintf(output,",{%d});\n",n->line);
}

/**************************************************************************/
/* LOCAL  **************    WriteTheSubNodes       ************************/
/**************************************************************************/
/* PURPOSE: Writes the information for the subnodes                       */
/**************************************************************************/

static void WriteTheSubNodes(n)
     PNODE n;
{
  int maxin, i, *portfound;
  PEDGE e;
  PNODE sub;

  /* loop thru all the subnodes */
  for (sub=n->nsucc; sub!=NULL; sub=sub->nsucc) {
		
    /* allocate storage for array to keep track of used import ports */
    maxin = CountImportPorts(sub->imp);
    portfound = (int *)MyAlloc((maxin+1)*sizeof(int));
    for (i=0; i<=maxin; i++) portfound[i] = 0;
		
    /* generate input calls */
    for (e=sub->imp; e!=NULL; e=e->isucc) {
      portfound[e->iport]=1;
      if (e->src==n) {
	i = FindEdgeName(e->name);
	fprintf(output,"\tArgInput(%d,%d,CP(ENP+%d));/* %s */\n",
		e->iport,e->eport,i,e->name);
      } else if (e->src) {
	i = FindEdgeName(e->name);
	fprintf(output,"\tLocalInput(%d,%d,CP(ENP+%d));/* %s */\n",
		e->iport,e->offset,i,e->name);
      } else {
	fprintf(output,"\tLiteralInput(%d,%d);/* %s */\n",
		e->iport,FindLiteral(e->constant,e->info->type,e->info->label),
		(e->constant)?(e->constant):"Error"
		);
      }
    }

    /* dump out null inputs */
    for (i=1; i<=maxin; i++) {
      if (!portfound[i]) fprintf(output,"\tNullInput(%d);\n",i);
    }
    free((char*)portfound);

    GenerateOutputCalls(n,sub);

    /* print the node execute */
    WriteNodeLine(sub);

    /* print a CopyToGraph for any nodes that go to the graph and also */
    /* fan out */
    for (e=sub->exp; e!=NULL; e=e->esucc) {
      if ( e->dst == n && Fanout(e) > 1 ) {
	fprintf(output,"\tCopyToGraph(%d,%d);\n", e->offset,e->iport);
      }
    }
    fprintf(output,"\n");
  }
}

/**************************************************************************/
/* LOCAL  **************       WriteCode           ************************/
/**************************************************************************/
/* PURPOSE: Prints the actual code.                                       */
/**************************************************************************/
static int SetupMode = 0;
static void WriteCode(n)
     PNODE n;
{
  PEDGE e;
  /* only generate code for graphs */
  if (IsGraph(n)) {
		
    /* determine if global, local, or imported function */
    switch ( n->type ) {
     case IFXGraph:		/* GLOBAL FOR EXPORT */
      fprintf(output,"GlobalFunction("); break;

     case IFIGraph:		/* IMPORTED FROM OTHER MODULE */
      return;			/* nothing to generate */

     default:
      fprintf(output,"LocalFunction(");
    }

    if (n->gname) {
      fprintf(output,"%s_%s",n->gname,SetupMode?"Setup":"");
    } else {
      fprintf(output,"Gr%d%s",n->label,SetupMode?"Setup":"");
    }

    WriteEnterGraphLine(n);

    CurrentGraph = n;
    WriteTheSubNodes(n);

    /* write out values that aren't produced by nodes */
    fprintf(output,"\n");
    for (e=n->imp; e!=NULL; e=e->isucc) {
      if (!e->src)
	fprintf(output,"\tCopyLiteral(%d,%d);\n",
		FindLiteral(e->constant,e->info->type,
			    e->info->label),
		e->iport);
      else if (e->src==n)
	fprintf(output,"\tCopyForward(%d,%d);\n",
		e->eport,e->iport);
    }
    fprintf(output,"\tExitGraph(%d);\n\}\n",n->label);
  }
}

/**************************************************************************/
/* GLOBAL **************         If2Process          ************************/
/**************************************************************************/
/* PURPOSE: Writes the converted C file for the if1 file                  */
/**************************************************************************/

void If2Process()
{
	int NumLiterals,highest,NumGraphs;
	char ModuleName[1024],*p;

	/* Get the root name of the output file to create the unique */
	/* nodetable name. */
	strcpy(ModuleName,(sfile)?sfile:"");

	p = rindex(ModuleName,'.');
	if ( p ) *p = '\0';	/* Remove the extension (if any) */
	for( p = rindex(ModuleName,'.'); p; p = rindex(ModuleName,'.') )
	    *p = '_';

/*	for(p=ModuleName,q = ModuleName; *p; p++) if (isalnum(*p)) *q++ = *p;
	*q = '\0'; */

	fprintf(output,"#include \"SC.h\"\n");
	fprintf(output,"#define NodeTable %s_NT\n",ModuleName);

	BuildTypeStruct();
	InitializeEdgePools();
	/* need to renumber graphs first because node table consists of only
	   graphs, but is built by traversing the entire node structure */
	ApplyToAllNodes(RenumberGraphs);
	NumGraphs = MaxNodes;
	ApplyToAllNodes(RenumberNodes);

	fprintf(output,"\n/* Forward Declarations */\n");
	fprintf(output,"extern NodeStuff NodeTable[];\n");
	ApplyToAllNodes(RenameEdgesEtc);      /* also fills in the 
						 NeededTypes.used array, 
						 counts the number of 
						 literals */
	fprintf(output,"\n");

	WriteTypes();
	ApplyToAllNodes(CalculateEdgeOffsets);
	fprintf(output,"\n");

	WriteEdgeNamePool();
	fprintf(output,"\n");

	ApplyToAllNodes(WriteInfoForward);
	NumLiterals = WriteLiterals();
	fprintf(output,"\n");

	fprintf(output,"/* Type Pool */\n");
	for (highest = 0; TypePool[highest]>=0; highest++);
	if ( highest > TypePoolSize ) {
	  fprintf(stderr,"Blew type pool out of the water!\n");
	  exit(1);
	}
	fprintf(output,"static TypeD NTypes[] = ");
	WriteTypeList(highest,TypePool);
	fprintf(output,"\n");

	fprintf(output,"\n/* Node Table */\n");
	WriteGraphNames();
	fprintf(output,"#if Debugger\n");
	fprintf(output,"# define TableSize (%d)\n",MaxNodes+1);
	fprintf(output,"#else\n");
	fprintf(output,"# define TableSize /* default */\n");
	fprintf(output,"#endif\n");
	fprintf(output,
		"NodeStuff NodeTable[TableSize] = {\n\tEntryInformation(%d,%d,\"%s\")",
		NumGraphs,NumLiterals,ModuleName);
	CurrentGraph=glstop->gsucc;
	ApplyToAllNodes(WriteGraphNode);
	fprintf(output,"};\n\n");


	SetupMode = 0;
	fprintf(output,"\n/* Code */\n\n");
	PostApplyToAllNodes(WriteCode);

	SetupMode = 1;
	fprintf(output,"\n/* Debug Code */\n");
	fprintf(output,"#include \"SCDebug.h\"\n\n");
	PostApplyToAllNodes(WriteCode);
}
