#include "SC.h"
#include "SCLib.h"
#include "Debug.h"

#define MaxDebugLevel	(10)

/* ------------------------------------------------------------ */
/* Exports */
PackageHookType	CurrentHook;
int	   	DebugLevel	 = -1;
SisalBoolean	StartInDebug;
SisalBoolean	DisplayByName;
SisalBoolean	NodeStep;
SisalBoolean	LineStep;

int		BreakOnError	= 0;
SisalBoolean	ErrorDummy;

int		LastLine 	= 0;
int		CurrentLine 	= 0;
int		LastModule	= 0;
int		CurrentModule	= 0;
int		Reasons[MaxBreakPoints];
int		ReasonCount;
/* ------------------------------------------------------------ */

#define CurrentSourceLine()	(123)
#define CurrentModuleNumber()	(123)

static void
InitBreakError()
{
  if ( BreakOnError ) {
    RemoveBreakPoint(LocateBreakpoint(BreakOnError));
  } else {
    (void)BreakError("",0);	/* set the break on error flag */
  }
}

static Parameter PackageParms[] = {
  { "-debug", ToggleParm,ParmAddr(StartInDebug),ParmAddr(ParmFalse),NULL,
      "Start execution in debug mode" },
  { "-ss", ToggleParm,ParmAddr(NodeStep),ParmAddr(ParmFalse),NULL,
      "Step IF1 nodes one at a time" },
  { "-ls", ToggleParm,ParmAddr(LineStep),ParmAddr(ParmFalse),NULL,
      "Step one Sisal line at a time" },
  { "-be", ToggleParm,ParmAddr(ErrorDummy),ParmAddr(ParmFalse),
      InitBreakError,
      "Break every time an error value is produced" },
  { "-names", ToggleParm,ParmAddr(DisplayByName),ParmAddr(ParmFalse),NULL,
      "Display debug values by name on break" },
  {NULL, StringParm, NULL, NULL, NULL}
};

char	DebugPrompt[20] = "debug-%d> ";
char    MorePrompt[20]  = ".....-%d> ";
DebugInputBuffer	DebugBuffer = {NULL,"",0};

void
DebugMessage(s)
     char	*s;
{
  fputs(s,stdout);
}

void
DebugExit(s)
     char	*s;
{
  DebugMessage(s);
  exit(0);
}

int
GetCurrentLine()
{
  Environment	*Env;
  int		Line;

  Env = TopOfEnvironmentStack();
  Line = (Env)?(Env->INFO->DebugInfo.SL):0;
  
  return (Line)?(Line):(CurrentLine);
}

int
GetCurrentModule()
{
  Environment	*Env;
  NodeInfo	ModuleAddress;
  int		Module;
  int		NT;

  Env = TopOfEnvironmentStack();
  if ( Env && InfoKind(Env->INFO) != Constructed ) {
    /* Get the module address from the environment */
    ModuleAddress = Env->INFO - NodeIDOf(Env->INFO);

    /* Search for it in the linkage table */
    Module = 0;
    for(NT=0;Linkage[NT];NT++) {
      if (Linkage[NT] == ModuleAddress) { Module = NT; break; }
    }
  } else {
    Module = 0;
  }

  return Module;
}

/* ------------------------------------------------------------ */
int
Handler(signal,code,scp)
     int	signal;
     int	code;		/* ARGSUSED */
     struct sigcontext	*scp;
{
  unsigned	OldMask,NewMask;
  static int	BeenHere = 0;
  int		Location;

  DebugLevel++;

  if ( DebugLevel >= MaxDebugLevel ) {
    DebugExit("Maximum debug depth exceeded\n");
  }

  /* Allow the handler to be re-entrant on new signals */
  OldMask = (unsigned)(sigblock(0));
  NewMask = OldMask & ~(sigmask(SIGINT) | sigmask(SIGUSR1));
  (void)(sigsetmask((int)NewMask));

  if (!BeenHere) {
    BeenHere = 1;
    DebugMessage("Entering debug mode... enter Q; to exit ?; for help\n");
    DebugMessage("Remember... All commands are semi-colon terminated\n");
  }

  /* Locate the breakpoint causing the halt.  If its a trace, continue */
  /* without invoking the command interface. */
  Location = (scp)?(-1):LocateBreakpoint(signal);
  if ( Location < 0 || SomeReasonIsABreakPoint() ) {
    DebugDisplay((scp)?0:signal); /* Show where we are */
    DisplayViewVars();		/* Show trace vars */
    DebugCommand(TRUE);		/* Parse commands (with prompt) */
  } else {
    DisplayViewVars();		/* Just show trace vars */
  }

  DebugLevel--;
}

void
DebugEmitPrompt(MoreCmd)
     int MoreCmd;
{
  char	message[100];

  if ( MoreCmd )
    (void)sprintf(message,MorePrompt,DebugLevel);
  else
    (void)sprintf(message,DebugPrompt,DebugLevel);
  DebugMessage(message);
}
  
void
EnableDebug()
{
  (void)signal(SIGUSR1,Handler);
  (void)signal(SIGINT,Handler);
}

void
EnterDebugMode()
{
  Handler(Reasons[0],0,(struct sigcontext*)NULL);
}

/* ------------------------------------------------------------ */
void
DebugPreTestForBreak()
{
  ReasonCount = 0;

  /* Check for an infinite loop flag set by a loop or recursion op */
  if ( InfiniteLoopFlag ) {
    DebugMessage("Warning: Iteration count has exceeded user limit.\n");
    Reasons[ReasonCount++] = 0;
    InfiniteLoopFlag = FALSE;	/* Reset so we can continue if possible */
  }

  /* Keep filling the reason list with nodal and nodeless breakpoints */
  ReasonCount += NodalBreakPointSetHere(Reasons+ReasonCount,
				       MaxBreakPoints-ReasonCount);
  ReasonCount += NodelessBreakPointSetHere(Reasons+ReasonCount,
					   MaxBreakPoints-ReasonCount);

  if ( ReasonCount ) EnterDebugMode();
}

/* ------------------------------------------------------------ */
void
DebugPostTestForBreak()
{
  ReasonCount = PostNodelessBreakPointSetHere(Reasons,
					      MaxBreakPoints);
  ReasonCount += PostNodalBreakPointSetHere(Reasons+ReasonCount,
					    MaxBreakPoints-ReasonCount);
  if ( ReasonCount ) EnterDebugMode();
}
/* ------------------------------------------------------------ */
void
DebugPostGraphTestForBreak()
{
  ReasonCount = PostNodalBreakPointSetHere(Reasons,MaxBreakPoints);
  if ( ReasonCount ) EnterDebugMode();
}
/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
void
PACKAGE_HOOK(hook)
     PackageHookType	hook;
{
  /* Find out where we are */
  CurrentLine	= GetCurrentLine();
  CurrentModule	= GetCurrentModule();

  /* Set the hook so we can examine where where are when looking at breaks */
  CurrentHook = hook;

  switch( hook ) {
    /* When we start the program, we must include debugger parameters */
    /* for option parsing, enable the debugger, and clear out the */
    /* table of breakpoints. */
   case PackagePreStart:
    FindLocalNames();
    InsertParmList(PackageParms,ParmList);
    EnableDebug();
    InitBreakPointTable();
    AliasInit();
    break;

    /* When entering a graph or node, we must check to see if we need */
    /* to stop here. */
   case PackageGraph:
   case PackagePreNode:
    DebugPreTestForBreak();
    break;

    /* When exitting a node, we must check again */
   case PackagePostNode:
    DebugPostTestForBreak();
    break;

    /* When exiting a graph, also check for an end of graph check */
   case PackageEndGraph:
    DebugPostGraphTestForBreak();
    break;

    /* If we must start in the debug mode, just call the handler and */
    /* exit when the user quits (i.e. don't continue with the program. */
   case PackageStart:
    if ( StartInDebug ) {
      Handler(0,0,(struct sigcontext*)NULL);
      DebugExit("Complete\n");
    }
    break;

   case PackageError:  
    break;

   case PackageForall:
    break;

   case PackageFunction:
    break;

   case PackageComplete:
    break;
  }

  /* Remember where we were for next time */
  LastLine	= CurrentLine;
  LastModule	= CurrentModule;
}
