#include "SC.h"
#include "SCLib.h"
#include "Debug.h"

static int	BreakCount = 0;
static int	TotalBreakCount = 0;

BreakPointEntry BreakPointTable[MaxBreakPoints];
char *BreakPointTypeName[] = {
  "empty",
  "On Line",
  "On Function",
  "On Error",
  "On Operation",
  "At End Of Graph",
  "On Name"
  };

/* ------------------------------------------------------------ */
int
SomeReasonIsABreakPoint()
{
  int		i,BP;

  for(i=0;i<ReasonCount;i++) {
    BP = LocateBreakpoint(Reasons[i]);
    if ( BP >= 0 && !BreakPointTable[BP].IsTrace ) return 1;
  }

  return 0;
}

/* ------------------------------------------------------------ */
int
BreakpointApplies(ID)
     int	ID;
{
  int		i;

  for(i=0;i<ReasonCount;i++) {
    if ( ID == Reasons[i] ) return 1;
  }

  return 0;
}
/* ------------------------------------------------------------ */
int
AddAndHashBreakpoint(Node,Type,BreakOrTrace,Err,Desc)
     NodeInfo		Node;
     BreakPointType	Type;
     int		BreakOrTrace;
     char		*Err;
     char               *Desc;
{
  int	Identifier = 0;		/* Assume no identifier found */
  int	BreakPoint;

  if ( Node ) {
    /* Select a breakpoint */ 
      BreakPoint = FirstFreeBreakPoint();

    /* Make sure its a legal breakpoint */
    if ( BreakPoint < 0 ) {
      DebugMessage("No space in breakpoint table\n");
    } else {
      Identifier = AddBreakPoint(BreakPoint,Type,BreakOrTrace,Desc);
      
      /* Add the node to the hash table */
      AddHash(Node,BreakPoint);
    }
  } else if ( Err ) {
    DebugMessage(Err);
  }

  return Identifier;
}
/* ------------------------------------------------------------ */
void
SplitModuleLine(S,delimiter,ModPtr,ArgPtr)
     char	*S;
     char	delimiter;
     int	*ModPtr;
     int	*ArgPtr;
{
  char		*DelPtr;

  DelPtr = index(S,delimiter);
  if ( DelPtr ) {
    *ArgPtr = atoi(DelPtr+1);
    *DelPtr = '\0';
    *ModPtr = GetModuleOffset(S);
  } else {
    *ArgPtr = atoi(S);
    *ModPtr = CurrentModule;
  }
}
/* ------------------------------------------------------------ */
int
IsALineNumber(args)
     char	*args;
{
  char	*ColonPtr;

  /* Short test.  If the args start with a digit, its a line number */
  return isdigit(*args) ||
    /* Otherwise, look for a colon followed by a digit */
    (ColonPtr = index(args,':'),ColonPtr && isdigit(*(ColonPtr+1)));
}

/* ------------------------------------------------------------ */
NodeInfo
FindFirstNodeAtLine(args,ModPtr,LinePtr)
     char	*args;
     int	*ModPtr;
     int	*LinePtr;
{
  NodeInfo	Node;
  int		Line;
  int		ModuleOffset;
  char		message[MaxCommentSize];

  /* Assume no first node exists */
  Node = NULL;

  /* Break apart the line description */
  SplitModuleLine(args,':',&ModuleOffset,&Line);

  /* Make sure we have a line number */
  if ( Line <= 0 ) return NULL;

  if ( ModuleOffset < 0 ) {
    DebugMessage("No such module\n");
  } else {
    /* Find the first node at the line */
    Node = FindNode(Linkage[ModuleOffset],Line,1);

    /* Make sure it matches the line we want */
    if ( (!Node) ) {
      DebugMessage("No nodes in module\n");

    } else if ( Node->DebugInfo.SL != Line ) {
      (void)sprintf(message,"No code at line %d, try %d\n",
		    Line,Node->DebugInfo.SL);
      DebugMessage(message);

      Node = NULL;		/* No node really exists */
    }
  }

  *ModPtr  = ModuleOffset;
  *LinePtr = Line;
  return Node;
}
/* ------------------------------------------------------------ */
void
InitBreakPointTable()
{
  int	i;

  /* Initialize the hash table */
  InitHash();

  /* Clear out the entries in the table */
  BreakCount = 0;
  for(i=0;i<MaxBreakPoints;i++) {
    BreakPointTable[i].Type = EmptyBreak;
  }
}

/* ------------------------------------------------------------ */
int
LowestBreakPoint()
{
  int	Low = TotalBreakCount+1; /* Higher than any */
  int	i;

  for(i=0;i<MaxBreakPoints;i++) {
    if (BreakPointTable[i].Type != EmptyBreak &&
	BreakPointTable[i].Number < Low) {
      Low = BreakPointTable[i].Number;
    }
  }

  return Low;
}

/* ------------------------------------------------------------ */
int
HighestBreakPoint()
{
  int	High = -1;		/* Lower than any */
  int	i;

  for(i=0;i<MaxBreakPoints;i++) {
    if (BreakPointTable[i].Type != EmptyBreak &&
	BreakPointTable[i].Number > High) {
      High = BreakPointTable[i].Number;
    }
  }

  return High;
}

/* ------------------------------------------------------------ */
void
DisplayBreakPoint(number)
     int	number;
{
  int		Location;
  char		message[MaxCommentSize];

  Location = LocateBreakpoint(number);
  
  if ( Location >= 0 && BreakPointTable[Location].Type != EmptyBreak ) {
    (void)sprintf(message,"[%d]\t%s%s\t%s\n",
		  number,
		  BreakPointTable[Location].IsTrace?"(Trace) ":"",
		  BreakPointTypeName[(int)(BreakPointTable[Location].Type)],
		  BreakPointTable[Location].Desc);
    DebugMessage(message);
  }
}

/* ------------------------------------------------------------ */
int
FirstFreeBreakPoint()
{
  int	i;

  for(i=0;i<MaxBreakPoints;i++) {
    if (BreakPointTable[i].Type == EmptyBreak) return i;
  }

  return -1;
}

/* ------------------------------------------------------------ */
int
AddBreakPoint(Location,BType,BreakOrTrace,Desc)
     int		Location;
     BreakPointType	BType;
     int		BreakOrTrace;
     char               *Desc;
{
  /* Keep track of breakpoints in table */
  BreakCount++;

  BreakPointTable[Location].Type	= BType;
  BreakPointTable[Location].IsTrace	= BreakOrTrace;
  BreakPointTable[Location].Number	= ++TotalBreakCount;
  (void)strncpy(BreakPointTable[Location].Desc,Desc,BreakDescSize-1);

  return BreakPointTable[Location].Number;
}

/* ------------------------------------------------------------ */
int
LocateBreakpoint(Number)
     int	Number;
{
  int		i;
  for(i=0;i<MaxBreakPoints;i++) {
    if ( BreakPointTable[i].Type != EmptyBreak &&
	BreakPointTable[i].Number == Number ) return i;
  }

  return -1;
}

/* ------------------------------------------------------------ */
void
RemoveBreakPoint(Location)
     int		Location;
{
  if ( Location >= 0 && Location < MaxBreakPoints ) {
    switch( BreakPointTable[Location].Type ) {
     case EmptyBreak:
      break;

     case LineBreak:
     case GraphBreak:
     case OperationBreak:
     case EndBreak:
     case NameBreak:
      BreakCount--;
      DeleteID(Location);	/* Remove from hash lookup table */
      break;

     case ErrorBreak:
      BreakCount--;
      BreakOnError = 0;		/* Don't break on error anymore */
      break;

     default:
      DebugExit("Bad break delete\n");
    }

    BreakPointTable[Location].Type = EmptyBreak;
  }
}

/* ------------------------------------------------------------ */
int
NodalBreakPointSetHere(BPList,maxn)
     int	BPList[];
     int	maxn;
{
  Environment	*Env;
  int		IDList[MaxBreakAtPoint];
  NodeInfo	Node;
  int		i;
  int		n;
  int		FoundCount = 0;

  /* If we've never set any breakpoints, just say no */
  if ( !BreakCount ) return 0;

  /* Check the current state of the stack */
  Env = TopOfEnvironmentStack();

  /* Get a list of breakpoints set at this node */
  Node = Env->INFO;
  n = NodalBreakPointList(Node,IDList,MaxBreakAtPoint);

  /* Check if any breakpoints in the list are matched */
  for(i=0;i<n;i++) {
    switch( BreakPointTable[IDList[i]].Type ) {
      /* For line breaks, do break if this is the first time on this line */
     case LineBreak:
      if ( CurrentLine != LastLine || CurrentModule != LastModule) {
	if ( FoundCount >= maxn ) return FoundCount;
	BPList[FoundCount++] = BreakPointTable[IDList[i]].Number;
      }
      break;

      /* If this is a break on graph, or its a break on operation then */
      /* it always matches */
     case GraphBreak:
     case OperationBreak:
      if ( FoundCount >= maxn ) return FoundCount;
      BPList[FoundCount++] = BreakPointTable[IDList[i]].Number;
      break;

     default:
      ;
    }
  }

  return FoundCount;
}

/* ------------------------------------------------------------ */
int
NodelessBreakPointSetHere(BPList,maxn) /* ARGSUSED */
     int	BPList[];
     int	maxn;
{
  /* Check Node Step */
  if (NodeStep) return 1;

  /* Check Line Step */
  if ( LineStep &&
      (CurrentLine != LastLine || CurrentModule != LastModule)) {
    BPList[0] = 0;		/* Just the one reason with no ID */
    return 1;
  }

  return 0;
}

/* ------------------------------------------------------------ */
int
PostNodelessBreakPointSetHere(BPList,maxn) /* ARGSUSED */
     int	BPList[];
     int	maxn;
{
  Environment		*Env;

  /* Check errors */
  if ( BreakOnError &&
      (Env=TopOfEnvironmentStack(), Env && 
       AnyErrorValues(Env->OUT,OUTARITY(Env->INFO))) ) {
    BPList[0] = BreakOnError;
    return 1;
  }

  return 0;
}

/* ------------------------------------------------------------ */
int
PostNodalBreakPointSetHere(BPList,maxn)
     int	BPList[];
     int	maxn;
{
  Environment		*Env;
  int		IDList[MaxBreakAtPoint];
  NodeInfo	Node;
  int		i;
  int		n;
  int		FoundCount = 0;

  /* If we've never set any breakpoints, just say no */
  if ( !BreakCount ) return 0;

  /* Check the current state of the stack */
  Env = TopOfEnvironmentStack();

  /* Get a list of breakpoints set at this node */
  Node = Env->INFO;
  n = NodalBreakPointList(Node,IDList,MaxBreakAtPoint);

  /* Check if any breakpoints in the list are matched */
  for(i=0;i<n;i++) {
    switch( BreakPointTable[IDList[i]].Type ) {
      /* If this is a break on end of graph or on name, it always matches */
     case NameBreak:
     case EndBreak:
      if ( FoundCount >= maxn ) return FoundCount;
      BPList[FoundCount++] = BreakPointTable[IDList[i]].Number;
      break;

     default:
      ;
    }
  }

  return FoundCount;
}

/* ------------------------------------------------------------ */
int
GetModuleOffset(name)
     char	*name;
{
  int		NT;
  for(NT=0;Linkage[NT];NT++) {
    if (StringEq(name,ModuleNameOf(Linkage[NT]))) return NT;
  }
  return -1;
}

/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
int
BreakLine(args,BreakOrTrace)
     char	*args;
     int	BreakOrTrace;
{
  NodeInfo	Node;
  int		Line;
  int		ModuleOffset;
  int		BreakPoint;
  int		Identifier;

  /* Assume that no breakpoint will be set */
  Identifier = 0;
  
  if ( !*args ) {
    DebugMessage("Break at line requires at least one line number\n");
  } else
    /* Process the breakpoint in the command */
    if ( (Node = FindFirstNodeAtLine(args,&ModuleOffset,&Line)) ) {

      /* Prepare a breakpoint entry in the table */
      BreakPoint = FirstFreeBreakPoint();
      if ( BreakPoint < 0 ) {
	DebugMessage("No space in breakpoint table\n");
      } else {
	Identifier = AddBreakPoint(BreakPoint,LineBreak,BreakOrTrace,args);
      
	/* Keep adding nodes at this line */
	do {
	  AddHash(Node,BreakPoint);
	  Node = FindNode(Linkage[ModuleOffset],Line,NodeIDOf(Node)+1);
	} while ( Node && Node->DebugInfo.SL == Line );
      }
    }
  
  return Identifier;
}

/* ------------------------------------------------------------ */
int
BreakFunction(args,BreakOrTrace)
     char	*args;
     int	BreakOrTrace;
{
  NodeInfo	Node;
  int		Identifier;
  char          message[MaxCommentSize];
  
  /* Prepare error message */
  (void) sprintf(message,"No such function %s\n", args);
  
  /* Find the function */
  Node = FindGraph(args,NULL,FALSE);
  Identifier = AddAndHashBreakpoint(Node,
				    GraphBreak,
				    BreakOrTrace,
				    message, args);
  
  return Identifier;
}

/* ------------------------------------------------------------ */
int
BreakError(args,BreakOrTrace)
     char	*args;		/* ARGSUSED */
     int	BreakOrTrace;
{
  int		BreakPoint;
  int		Identifier;

  /* Assume we can't set... */
  Identifier = 0;

  if ( !BreakOnError ) {	/* Don't reset if already done */
    BreakPoint = FirstFreeBreakPoint();
    if ( BreakPoint < 0 ) {
      DebugMessage("No space in breakpoint table\n");
    } else {
      Identifier = AddBreakPoint(BreakPoint,ErrorBreak,BreakOrTrace,args);
      BreakOnError = Identifier;
    }
  }
  return Identifier;
}

/* ------------------------------------------------------------ */
int
BreakOperation(args,BreakOrTrace)
     char	*args;
     int	BreakOrTrace;
{
  NodeInfo	Node;
  int		ID;
  int		ModuleOffset;
  int		Identifier;

  /* Assume that no breakpoint will be set */
  Identifier = 0;
  
  if ( !*args ) {
    DebugMessage("Break at ID requires ID number\n");
  } else {
    SplitModuleLine(args,'.',&ModuleOffset,&ID);

    /* Make sure the module exists */
    if ( ModuleOffset < 0 ) {
      DebugMessage("No such module\n");
    } else if ( ID < 1 || ID > ModuleSizeOf(Linkage[ModuleOffset]) ) {
      DebugMessage("No such operation\n");
    } else {
      Node = Linkage[ModuleOffset]+ID;
      Identifier = AddAndHashBreakpoint(Node,
					OperationBreak,
					BreakOrTrace,
					(char*)NULL,args);
    }
  }

  return Identifier;
}

/* ------------------------------------------------------------ */
int
BreakGraphEnd(args,BreakOrTrace)
     char	*args;
     int	BreakOrTrace;
{
  int		Identifier;
  NodeInfo	Node;
  int		Line;
  int		ModuleOffset;

  /* Assume that no breakpoint will be set */
  Identifier = 0;

  if ( !*args ) {
    DebugMessage("Break at graphend requires line number or name\n");
  } else if ( IsALineNumber(args) ) {
    /* Find a node at the requested line number */
    Node = FindFirstNodeAtLine(args,&ModuleOffset,&Line);

    /* If this is a graph, great.  Otherwise use the Node's parent */
    if ( Node &&!IsGraph(Node) ) Node = ParentOf(Node);
    Identifier = AddAndHashBreakpoint(Node,
				      EndBreak,
				      BreakOrTrace,
				      (char*)NULL,args);
  } else {
    /* ------------------------------------------------------------ */
    /* It must be a function */

    Node = FindGraph(args,NULL,FALSE);
    Identifier = AddAndHashBreakpoint(Node,
				      EndBreak,
				      BreakOrTrace,
				      "No such function",args);
  }

  return Identifier;
}

/* ------------------------------------------------------------ */
void
DeleteBreak(args)
     char	*args;
{
  char		*parse[2];
  int		i;
  int		BreakPoint;

  if ( *args )  {
    parse[1] = args;
    while(*parse[1]) {
      /* Chop off the first breakpoint id */
      ParseList(parse[1],parse,1);

      if ( *parse[0] == '*' ) {
	for(i=0;i<MaxBreakPoints;i++) RemoveBreakPoint(i);
      } else {
	/* Delete the specified breakpoint */
	BreakPoint = atoi(parse[0]);
	if (BreakPoint >= 0) RemoveBreakPoint(LocateBreakpoint(BreakPoint));
      }

    }
  } else {
    DebugMessage("Delete requires a breakpoint identifier or wildcard *\n");
  }
}
