#include <stdio.h>
#include <ctype.h>
#include "alias.h"

static char LastExpandedAlias[2*ALIASLENGTH];
AliasListStructure	AliasList[MAXALIASES];

/* ------------------------------------------------------------ */
/* Count the number of wildcards */
static int
DollarCount(n)
     int	n;
{
  char		*p;
  int		Count = 0;

  for(p=AliasList[n].Pattern;*p;p++) if (*p == '$') Count++;

  return Count;
}

/* ------------------------------------------------------------ */
/* Expand the $1-$9 parameters ($0 == $, $x = $x) */
static void
ExpandAlias(Expansion,Dollars,DollarCount,Buffer)
     char	*Expansion;
     char	Dollars[][ALIASLENGTH];
     int	DollarCount;
     char	*Buffer;
{
  char		*p,*q;
  int		WhichDollar;

  /* Copy the expansion into the buffer expanding dollar vars */
  for(p=Expansion;*p;p++) {
    if ( *p == '$' ) {
      /* If its an escape, see which one (1-9) to use.. */
      /* $0 is just a dollar sign as is $A or something like that */
      WhichDollar = 0;
      if ( *(p+1) && isdigit(*(p+1)) ) {
	WhichDollar = ((char)(*(p+1)))-'0';
	p++;			/* Consume the digit on the next for pass */
      }

      /* If the WhichDollar is set, then insert the binding (if set) */
      if ( WhichDollar > 0 ) {
	if ( WhichDollar <= DollarCount ) {
	  for(q=Dollars[WhichDollar-1];*q;q++) *Buffer++ = *q;
	}
      } else {
	/* Just insert the dollar sign */
	*Buffer++ = '$';
      }
    } else {
      *Buffer++ = *p;
    }
  }

  *Buffer = NULL;
}

/* ------------------------------------------------------------ */
/* Make the WorkingAliases table full of empty entries and the Alias */
/* list empty */
void
AliasInit()
{
  int	i;

  LastExpandedAlias[0] = '\0';
  for(i=0;i<MAXALIASES;i++) AliasList[i].ID = 0;
}

/* ------------------------------------------------------------ */
/* Add an alias pattern to the AliasList table */
int
AddAlias(pattern,expansion)
     char	*pattern,*expansion;
{
  int		Entry;
  static int	AliasID = 0;

  /* Make sure the first character is alphanumeric */
  if ( !(*pattern && (isalnum(*pattern) || *pattern == '_')) ) {
    DebugMessage("Bad alias, use only alphanumeric and underscore chars\n");
    return(0);
  }

  /* Find an empty entry in the alias list */
  for(Entry=0;Entry<MAXALIASES;Entry++) if (!AliasList[Entry].ID) break;
  if ( Entry >= MAXALIASES ) { 
    DebugMessage("No more alias room\n");
    return(0);
  }
  
  /* Assign it an unique ID and mark it*/
  AliasList[Entry].ID = ++AliasID;	/* Get a new one */

  /* Put the pattern and result into the list */
  (void)strncpy(AliasList[Entry].Pattern,pattern,ALIASLENGTH-1);
  (void)strncpy(AliasList[Entry].Expansion,expansion,ALIASLENGTH-2);
  (void)strcat(AliasList[Entry].Expansion,";");

  return( AliasID );
  
}

/* ------------------------------------------------------------ */
/* Match a line of input to the patterns in AliasList */
char
*MatchAlias(line)
     char	*line;
{
  char		DollarList[9][ALIASLENGTH]; 
  char		MatchList[9][ALIASLENGTH];
  char		*p,*s,*q;
  int		Dollar,i,Entry,Match = -1;
  int		MatchDollars,EntryDollars;
  int	        Quoted;
  int	        DQuoted;

  /* Empty the dollar list */
  for(i=0;i<9;i++) DollarList[i][0] = NULL;
  Dollar = 0;

  /* Try to make a binding between each of the patterns and the line */
  DQuoted = 0;
  Quoted = 0;
  for(Entry=0;Entry<MAXALIASES;Entry++) {
    if ( AliasList[Entry].ID ) { /* Only check current entries */
      /* Try a character by character matching to get a dollar binding */
      for(p = AliasList[Entry].Pattern, s = line;
	  *p && *s;
	  ) {
	if ( *p == '$' ) {
	  if ( Dollar == 9 ) { 
	    DebugMessage("Too many bindings\n");
	    return( (char*) 0 );
	  }
	  p++;			/* Skip the dollar */
	  /* Get the parameter for the dollar */
	  for(q=DollarList[Dollar++];*s && (!isspace(*s) || Quoted || DQuoted);
	      q++,s++)
	  {
	    *q = *s;
	    /* Check to see if we're entering a quoted section */
	    if ( !Quoted && *(q-1) != '\\' && *q == '"' ) DQuoted = !DQuoted;
	    if ( !DQuoted && *(q-1) != '\\' && *q == '\'' ) Quoted = !Quoted;
	  }
	  *q = NULL;
	} else if ( isspace(*p) && isspace(*s) ) { /* Skip whitespace  */
	  while(*p && isspace(*p)) p++;
	  while(*s && isspace(*s)) s++;
	} else if ( *p != *s ) {
	  Dollar = 0;
	  break;		/* Pattern mismatch */
	} else {
	  p++;			/* Characters match */
	  s++;
	}
      }

      /* Allow whitespace at the end of the line... */
      while(*s && isspace(*s)) s++;

      /* If the list is empty, then the alias matches. */
      /* Keep the entry with the fewest dollar signs or the */
      /* highest ID'd entry for expansion */
      if ( *p == NULL && *s == NULL ) {
	EntryDollars = DollarCount(Entry);
	if ((Match < 0) ||
	    ( EntryDollars < MatchDollars ) ||
	    ( EntryDollars == MatchDollars &&
	     AliasList[Entry].ID > AliasList[Match].ID)
	    ) {
	  Match = Entry;
	  MatchDollars = EntryDollars;
	  for(i=0;i<MatchDollars;i++) 
	    strncpy(MatchList[i],DollarList[i],ALIASLENGTH-1);
	}
      }
    }
  }

  /* Use the alias matched above (if any) */
  if ( Match >= 0 ) {
    ExpandAlias(AliasList[Match].Expansion,MatchList,MatchDollars,
		LastExpandedAlias);
    return LastExpandedAlias;
  }
  
  return ((char *) 0);
}

/* ------------------------------------------------------------ */
/* Delete the aliases from the table */
void
DeleteAlias(args)
     char	*args;
{
  int		Entry;
  char          *parse[2];
  int           id;
  
  if ( *args )
  {
    parse[1] = args;
    while(*parse[1]) {
      /* Chop off the first alias id */
      ParseList(parse[1],parse,1);
      id = atoi(parse[0]);
      
      /* Find the entry in the alias list and mark it open */
      for(Entry=0;Entry<MAXALIASES;Entry++) {
	if (AliasList[Entry].ID == id) {
	  AliasList[Entry].ID = 0;
	  break;
	}
      }
    }
  } else
    DebugMessage("Delete requires an alias identifier\n");
}

/* ------------------------------------------------------------ */
/* List the aliases by unique ID, pattern, and expansion */
void
ListAliases()
{
  int		IDList[MAXALIASES],IDCount = 0;
  int		i,j,T;

  /* Get all offsets that contain a valid ID */
  for(i=0;i<MAXALIASES;i++) {
    if ( AliasList[i].ID ) IDList[IDCount++] = i;
  }

  /* Yeah, its a bubble sort of the ID's, but so what... */
  for(i=0;i<IDCount-1;i++) {
    for(j=i+1;j<IDCount;j++) {
      if ( AliasList[IDList[i]].ID > AliasList[IDList[j]].ID ) {
	T = IDList[i]; IDList[i] = IDList[j]; IDList[j] = T;
      }
    }
  }

  /* Display the ID's  */
  for(i=0;i<IDCount;i++) {
    printf("[%d] %s ==> %s\n",
	   AliasList[IDList[i]].ID,AliasList[IDList[i]].Pattern,
	   AliasList[IDList[i]].Expansion);
  }
}
