#include "world.h"


/**************************************************************************/
/* GLOBAL **************        FindSource         ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE EDGE WHOSE SOURCE NODE GENERATES THE DATA VALUE    */
/*          REFERENCED BY EDGE i (A NON-SUBGRAPH NODE).                   */
/**************************************************************************/

PEDGE FindSource( i )
register PEDGE i;
{
    register PEDGE ii;
    register PNODE owner;

    for ( ;; ) {
	if ( IsConst( i ) )
	    return( i );

        if ( !IsSGraph( i->src ) )
	    return( i );

        owner = i->src->G_DAD;

	/* TAGCASE VALUE REFERENCE?  */
	if ( IsTagCase( owner ) && (i->eport == 1) )
	    return( i );

	/* A K-PORT VALUE REFERENCE? */
        if ( (ii = FindImport( owner, i->eport )) == NULL )
	    return( i );

        i = ii;
	}
}


/**************************************************************************/
/* GLOBAL **************       AreEdgesEqual       ************************/
/**************************************************************************/
/* PURPOSE: RETURNS TRUE IF THE EDGES e1 AND e2 ARE EQUAL.  TO BE EQUAL,  */
/*          THEY MUST EITHER BOTH DEFINE THE SAME CONSTANTS OR ADDRESS    */
/*          THE SAME SOURCE NODE AND HAVE THE SAME EXPORT PORT NUMBER.    */
/*          IF THEY ARE NOT EQUAL, FALSE IS RETURNED. IF NECESSARY, GRAPH */
/*          BOUNDARIES ARE CROSSED TO FIND AN EDGE'S SOURCE NODE.         */
/**************************************************************************/

int AreEdgesEqual( e1, e2 )
PEDGE e1;
PEDGE e2;
{
    e1 = FindSource( e1 );
    e2 = FindSource( e2 );

    if ( (e1->src == e2->src) && (e1->eport == e2->eport) ) {
        if ( IsConst( e1 ) )
	    return( AreConstsEqual( e1, e2 ) );

	return( TRUE );
        }

    return( FALSE );
}


/**************************************************************************/
/* LOVAL   **************        IsInvariant        ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF NODE n IS LOOP INVARIANT. RETURN AND GENERATE  */
/*          NODES ARE NEVER CONSIDERED INVARIANT.  NODE n IS INVARIANT IF */
/*          ITS IMPORTS ARE CONSTANTS OR IMPORTS TO ITS OWNING COMPOUND   */
/*          NODE.                                                         */
/**************************************************************************/

static int IsInvariant( n )
PNODE n;
{
    register PEDGE i;

    if ( IsReturn( n ) || IsGenerate( n ) )
	return( FALSE );

    for ( i = n->imp; i != NULL; i = i->isucc ) {
	if ( IsConst( i ) )
	    continue;

        if ( !IsSGraph( i->src ) )
	    return( FALSE );

        if ( !IsImport( i->src->G_DAD, i->eport ) )
	    return( FALSE );
	}

    return( TRUE );
}


/**************************************************************************/
/* LOCAL  **************      IsRetNormalized      ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF THE RETURN NODE OF LOOP l IS NORMALIZED:       */
/*          CONTAINS ONLY RETURN NODES.                                   */
/**************************************************************************/

static int IsRetNormalized( l )
PNODE l;
{
    register PNODE n;

    if ( IsForall( l ) )
	n = l->F_RET->G_NODES;
    else
	n = l->L_RET->G_NODES;

    for ( /* EMPTY */; n != NULL; n = n->nsucc )
	if ( !IsReturn( n ) )
	    return( FALSE );

    return( TRUE );
}


/**************************************************************************/
/* LOCAL  **************      IsGenNormalized        **********************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF THE GENERATOR SUBGRAPH OF FORALL NODE f IS     */
/*          NORMALIZED: CONTAINS ONLY SCATTER AND RANGEGENERATE NODES.    */
/**************************************************************************/

static int IsGenNormalized( f )
PNODE f;
{
    register PNODE n;

    for ( n = f->F_GEN->G_NODES; n != NULL; n = n->nsucc )
	if ( !IsGenerate( n ) )
	    return( FALSE );

    return( TRUE );
}


/**************************************************************************/
/* LOCAL  **************      IsTestNormalized       **********************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF THE TEST SUBGRAPH OF LOOPA OR LOOPB NODE l IS  */
/*          NORMALIZED: CONTAINS ONLY BOOLEAN OPERATION NODES.            */
/**************************************************************************/

static int IsTestNormalized( l )
PNODE l;
{
    register PNODE n;

    for ( n = l->L_TEST->G_NODES; n != NULL; n = n->nsucc )
	switch ( n->type ) {
	    case IFNot:
	    case IFLess:
	    case IFLessEqual:
	    case IFEqual:
	    case IFNotEqual:
                break;

	    case IFPlus:
	    case IFTimes:
	        if ( IsBoolean( n->exp->info ) )
		    break;

	    default:
		return( FALSE );
	    }

    return( TRUE );
}


/**************************************************************************/
/* LOCAL  **************       CheckNodeError      ************************/
/**************************************************************************/
/* PURPOSE: PRINT MESSAGE msg, ASSOCIATED WITH NODE n, TO output.         */
/**************************************************************************/

static void CheckNodeError( n, msg )
PNODE  n;
char  *msg;
{
    fprintf( output, "%s: NODE(", program );

    if ( IsSimple( n ) )
        fprintf( output, "%s,", snames[ n->type - IFAAddH ] );
    else if ( IsCompound( n ) )
        fprintf( output, "%s,", cnames[ n->type ] );
    else
        fprintf( output, "%s,", atnames[ n->type - IFAAddLAT ] );

    fprintf( output, "%d): %s\n", n->if1line, msg );
}


/**************************************************************************/
/* LOCAL  **************       CheckEdgeError      ************************/
/**************************************************************************/
/* PURPOSE: PRINT MESSAGE msg, ASSOCIATED WITH EDGE e, TO output.         */
/**************************************************************************/

static void CheckEdgeError( e, msg )
PEDGE  e;
char  *msg;
{
    fprintf( output, "%s: EDGE(%d): %s\n", program, e->if1line, msg );
}


/**************************************************************************/
/* LOCAL  **************        FloatType          ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE FLOATING POINT TYPE OF CONSTANT c: 'd' FOR DOUBLES,*/
/*          'e' FOR REALS, AND ' ' FOR UNKNOWNS.                          */
/**************************************************************************/

static char FloatType( c )
register char *c;
{
    while ( *c != '\0' ) {
	if ( (*c == 'd') || (*c == 'D') )
	    return( 'd' );

	if ( (*c == 'e') || (*c == 'E') )
	    return( 'e' );
        
	c++;
	}

    return( ' ' );
}


/**************************************************************************/
/* LOCAL  **************   CheckConstantImports    ************************/
/**************************************************************************/
/* PURPOSE: CHECK THE IMPORTS OF NODE n FOR ILLEGAL FLOATING CONSTANTS,   */
/*          AND SIGNED ARITHMETIC CONSTANTS.  IF ALL OF n'S IMPORTS ARE   */
/*          CONSTANTS, A MESSAGE IS PRINTED.                              */
/**************************************************************************/

static void CheckConstantImports( n )
PNODE n;
{
    register PEDGE i;

    for ( i = n->imp; i != NULL; i = i->isucc ) {
	if ( !IsConst( i ) || i->CoNsT == NULL )
            continue;

	switch ( i->info->type ) {
            case IF_DOUBLE:
	        if ( FloatType( i->CoNsT ) == 'e' )
		    CheckEdgeError( i, "CONSTANT FORMAT ERROR" );

		if ( i->CoNsT[0] == '-' )
	            CheckEdgeError( i, "SIGNED DOUBLE CONSTANT" );

		break;

            case IF_REAL:
		if ( FloatType( i->CoNsT ) == 'd' )
	            CheckEdgeError( i, "CONSTANT FORMAT ERROR" );
		    
		if ( i->CoNsT[0] == '-' )
	            CheckEdgeError( i, "SIGNED REAL CONSTANT" );

		break;

            case IF_INTEGER:
		if ( i->CoNsT[0] == '-' )
	            CheckEdgeError( i, "SIGNED INTEGER CONSTANT" );

		break;

            default:
		break;
            }
	}

    for ( i = n->imp; i != NULL; i = i->isucc )
	if ( (!IsConst( i )) || (i->CoNsT == NULL) )
            break;

    if ( (i == NULL) &&  (!IsReturn( n )) && (!IsCall( n )) &&
	 (!IsGenerate( n )) && (IsSimple( n )))
	CheckNodeError( n, "CONSTANT NODE NOT FOLDED" );
}


/**************************************************************************/
/* LOCAL  **************     AreKportsCombined     ************************/
/**************************************************************************/
/* PURPOSE: CHECK FOR REDUNDANT IMPORTS TO COMPOUND NODE c.               */
/**************************************************************************/

static int AreKportsCombined( c )
PNODE c;
{
    register PEDGE i1;
    register PEDGE i2;

    for ( i1 = c->imp; i1 != NULL; i1 = i1->isucc ) 
	for ( i2 = i1->isucc; i2 != NULL; i2 = i2->isucc ) 
	    if ( AreEdgesEqual( i1, i2 ) )
		return( FALSE );

    return( TRUE );
}


/**************************************************************************/
/* LOCAL  **************        AreSameType        ************************/
/**************************************************************************/
/* PURPOSE: CHECK IF THE TYPES i1 AND i2 ARE THE SAME. MULTIPLE TYPES ARE */
/*          TRANSPARENT. NOTE, A COMPLETE CHECK OF COMPLEX TYPES IS NOT   */
/*          DONE.                                                         */
/**************************************************************************/

static int AreSameType( i1, i2 )
PINFO i1;
PINFO i2;
{
    if ( IsMultiple( i1 ) )
	i1 = i1->A_ELEM;

    if ( IsMultiple( i2 ) )
	i2 = i2->A_ELEM;

    return( i1->type == i2->type );
}


/**************************************************************************/
/* LOCAL  **************      CheckIfDFOrdered     ************************/
/**************************************************************************/
/* PURPOSE: CHECK IF THE NODES IN THE NODE LIST OF GRAPH g ARE DFOrdered: */
/*          IF NODE b DEPENDS ON A VALUE GENERATED BY NODE a THEN THE     */
/*          LABEL(b) > LABEL(a).                                          */
/**************************************************************************/

static void CheckIfDFOrdered( g )
PNODE g;
{
    register PNODE n;
    register PEDGE i;

    for ( n = g->G_NODES; n != NULL; n = n->nsucc )
	if ( n->nsucc != NULL )
	    if ( n->nsucc->label <= n->label )
		CheckNodeError( n->nsucc, "NODE NOT DFOrdered" );

    for ( n = g->G_NODES; n != NULL; n = n->nsucc )
	for ( i = n->imp; i != NULL; i = i->isucc )
	    if ( !IsConst( i ) )
		if ( i->src->label >= n->label )
		    CheckNodeError( n, "NODE NOT DFOrdered" );
}


/**************************************************************************/
/* LOCAL  **************      CheckPortNumbers     ************************/
/**************************************************************************/
/* PURPOSE: CHECK IF THE IMPORT PORT NUMBERS OF g'S IMPORTS ARE IN        */
/*          SEQUENCE BEGINNING WITH p.  THE NEXT EXPECTED PORT NUMBER IS  */
/*          RETURNED.  IF gg IS NOT NULL, THEN AN IMPORT IS ONLY CHECKED  */
/*          IF IT IS NOT ALSO AN IMPORT TO gg: ONLY IF IT IS A LOOPA T    */
/*          PORT NUMBER.                                                  */
/**************************************************************************/

static int CheckPortNumbers( p, g, gg )
int   p;
PNODE g;
PNODE gg;
{
    register PEDGE i;

    for ( i = g->imp; i != NULL; i = i->isucc ) {
	if ( gg != NULL )
	    if ( IsImport( gg, i->iport ) )
		continue;

        if ( i->iport != p++ )
	    CheckEdgeError( i, "ILLEGAL IMPORT PORT NUMBER" );
        }

    return( p );
}


/**************************************************************************/
/* LOCAL  **************           IsUsed          ************************/
/**************************************************************************/
/* PURPOSE: CHECK THAT ALL USAGES OF i->iport IN GRAPH NODE g ARE OF THE  */
/*          SAME TYPE. IF i->iport IS USED, TRUE IS RETURNED, ELSE FALSE. */
/**************************************************************************/

static int IsUsed( g, i )
PNODE g;
PEDGE i;
{
    register PEDGE e;
    register int   u = FALSE;

    for ( e = g->exp; e != NULL; e = e->esucc )
	if ( e->eport == i->iport ) {
	    u = TRUE;

	    if ( !AreSameType( e->info, i->info ) ) {
		CheckEdgeError( i, "** USAGE TYPE MISMATCH" );
		CheckEdgeError( e, "   USAGE TYPE MISMATCH" );
		}
	    }

    return( u );
}


/**************************************************************************/
/* LOCAL  **************    CheckForUnusedKports   ************************/
/**************************************************************************/
/* PURPOSE: CHECK THAT ALL IMPORTS TO COMPOUND NODE c ARE USED.           */
/**************************************************************************/

static void CheckForUnusedKports( c )
PNODE c;
{
    register PEDGE i;
    register int   u;
    register PNODE g;

    for ( i = (IsTagCase(c))? c->imp->isucc : c->imp; i != NULL; i = i->isucc) {
	for ( u = FALSE, g = c->C_SUBS; g != NULL; g = g->gsucc )
            u |= IsUsed( g, i );

        if ( !u )
	    CheckEdgeError( i, "K IMPORT NOT USED" );
	}
}


/**************************************************************************/
/* LOCAL  **************   CheckForUnusedRports    ************************/
/**************************************************************************/
/* PURPOSE: CHECK THAT ALL IMPORTS OF SUBGRAPH g ARE EXPORTS OF THE       */
/*          COMPOUND NODE DEFINING g.                                     */
/**************************************************************************/

static void CheckForUnusedRports( g )
PNODE g;
{
    register PEDGE i;

    for ( i = g->imp; i != NULL; i = i->isucc )
        if ( !IsUsed( g->G_DAD, i ) )
	    CheckEdgeError( i, "R EXPORT NOT USED" );
}


/**************************************************************************/
/* LOCAL  **************    CheckForUnusedTports   ************************/
/**************************************************************************/
/* PURPOSE: CHECK THAT ALL IMPORTS TO FORALL f'S BODY SUBGRAPH ARE USED.  */
/**************************************************************************/

static void CheckForUnusedTports( f )
PNODE f;
{
    register PEDGE i;

    for ( i = f->F_BODY->imp; i != NULL; i = i->isucc )
        if ( !IsUsed( f->F_RET, i ) )
	    CheckEdgeError( i, "T IMPORT NOT USED" );
}


/**************************************************************************/
/* LOCAL  **************   CheckForUnusedLTports   ************************/
/**************************************************************************/
/* PURPOSE: CHECK THAT THE T IMPORTS OF LOOPA l'S BODY ARE USED AND CHECK */
/*          THAT THE L IMPORTS OF LOOP l'S INITIAL SUBGRAPH ARE USED.     */
/**************************************************************************/

static void CheckForUnusedLTports( l )
PNODE l;
{
    register PEDGE i;

    for ( i = l->L_BODY->imp; i != NULL; i = i->isucc ) {
	if ( IsUsed( l->L_TEST, i ) || 
             IsUsed( l->L_BODY, i ) ||
             IsUsed( l->L_RET,  i )  )
            continue;

	CheckEdgeError( i, "T IMPORT NOT USED" );
	}

    for ( i = l->L_INIT->imp; i != NULL; i = i->isucc ) {
	if ( IsUsed( l->L_TEST, i ) || 
             IsUsed( l->L_BODY, i ) ||
             IsUsed( l->L_RET,  i )   )
            continue;

	CheckEdgeError( i, "L IMPORT NOT USED" );
        }
}


/**************************************************************************/
/* LOCAL  **************    CheckForUnnecEdges     ************************/
/**************************************************************************/
/* PURPOSE: CHECK FOR UNNECESSARY EDGES AND LITERALS IN THE SUBGRAPHS OF  */
/*          LOOP l. THE FOLLOWING ARE CONSIDERED UNNECESSARY:             */
/*                                                                        */
/*          1. "T := constant, M, OR K" (FORALL BODY)                     */
/*          2. "L := constant OR K", L NOT REDEFINED IN BODY              */
/*          3. "L := OLD L"                                               */
/*          4. "T := constant, K, OR NOT REDEFINED L (LOOPA BODY)         */
/**************************************************************************/

static void CheckForUnnecEdges( l )
PNODE l;
{
    register PEDGE i;

    switch ( l->type ) {
        case IFForall:
	    for ( i = l->F_BODY->imp; i != NULL; i = i->isucc )
		if ( IsConst( i ) )
		    CheckEdgeError( i, "UNNECESSARY LITERAL: T := const" );
		else if ( IsSGraph( i->src ) )
		    CheckEdgeError( i, "UNNECESSARY EDGE: T := M OR K" );

	    break;

        case IFLoopA:
	case IFLoopB:
	    for ( i = l->L_BODY->imp; i != NULL; i = i->isucc ) {
		if ( IsConst( i ) )
		    continue;

		if ( IsSGraph( i->src ) && ( i->iport == i->eport ) )
	            CheckEdgeError( i, "UNNECESSARY EDGE: L := OLD L" );
                }

	    for ( i = l->L_INIT->imp; i != NULL; i = i->isucc ) {
		if ( IsImport( l->L_BODY, i->iport ) )
		    continue;

		if ( IsConst( i ) )
                    CheckEdgeError( i, "UNNECESSARY EDGE: L NR := const" );
                else if ( IsSGraph( i->src ) )
	            CheckEdgeError( i, "UNNECESSARY EDGE: L NR := K" );
                }

            if ( !IsLoopA( l ) )
		break;

            for ( i = l->L_BODY->imp; i != NULL; i = i->isucc ) {
		if ( IsImport( l->L_INIT, i->iport ) )
	            continue;

	        if ( IsConst( i ) )
	            CheckEdgeError( i, "UNNECESSARY EDGE: T := const" );
                else if ( IsSGraph( i->src ) )
	            if ( !IsImport( l->L_BODY, i->eport ) )
			    CheckEdgeError( i, 
					"UNNECESSARY EDGE: T := K OR L NR" );
                }

	    break;

	default:
	    break;
        }
}


/**************************************************************************/
/* LOCAL  **************     CheckCompoundNode     ************************/
/**************************************************************************/
/* PURPOSE: FOR COMPOUND NODE c, CHECK THAT ALL M, T, L, K, AND R PORT    */
/*          VALUES ARE USED; CHECK FOR UNNECESSARY EDGES AND LITERALS;    */
/*          CHECK THAT ALL PORT NUMBERS ARE CORRECTLY ASSIGNED; CHECK     */
/*          THAT TEST, INITIAL, AND RETURN SUBGRAPHS ARE NORMALIZED; AND  */
/*          MAKE SURE ALL REFERENCED SUBGRAPH EXPORTS ARE DEFINED.        */
/**************************************************************************/

static void CheckCompoundNode( c )
PNODE c;
{
    register PNODE g;
    register int   p;
    register PEDGE e;

    CheckForUnnecEdges( c );

    switch( c->type ) {
	case IFSelect:
	    if ( !IsNodeListEmpty( c->S_TEST ) )
		CheckNodeError( c->S_TEST, "SELECT TEST NOT NORMALIZED" );

            CheckForUnusedRports( c->S_ALT  );
            CheckForUnusedRports( c->S_CONS );

            CheckPortNumbers( 1, c, NULL );                              /* K */

	    for ( g = c->C_SUBS; g != NULL; g = g->gsucc )
		CheckPortNumbers( 1, g, NULL );                          /* R */

            for ( g = c->C_SUBS; g != NULL; g = g->gsucc )
		for ( e = g->exp; e != NULL; e = e->esucc )
		    if ( !IsImport( c, e->eport ) )
			CheckEdgeError( e, "SUBGRAPH EXPORT NOT DEFINED" );

	    break;

        case IFTagCase:
            CheckPortNumbers( 1, c, NULL );                              /* K */

	    for ( g = c->C_SUBS; g != NULL; g = g->gsucc ) {
		CheckForUnusedRports( g );
		CheckPortNumbers( 1, g, NULL );                          /* R */
		}

            for ( g = c->C_SUBS; g != NULL; g = g->gsucc )
		for ( e = g->exp; e != NULL; e = e->esucc )
		    if ( !IsImport( c, e->eport ) )
			CheckEdgeError( e, "SUBGRAPH EXPORT NOT DEFINED" );

	    break;

        case IFForall:
	    if ( !IsGenNormalized( c ) )
		CheckNodeError( c->F_GEN, "FORALL GEN NOT NORMALIZED" );

	    if ( !IsRetNormalized( c ) )
		CheckNodeError( c->F_GEN, "FORALL RETURN NOT NORMALIZED" );

	    CheckForUnusedRports( c->F_RET );
	    CheckForUnusedTports( c );

	    p = CheckPortNumbers( 1, c, NULL );                          /* K */
	    p = CheckPortNumbers( p, c->F_GEN, NULL );                   /* M */

	    CheckPortNumbers( p, c->F_BODY, NULL );                      /* T */
	    CheckPortNumbers( 1, c->F_RET,  NULL );                      /* R */

            for ( e = c->F_GEN->exp; e != NULL; e = e->esucc )
		if ( !IsImport( c, e->eport ) )
	            CheckEdgeError( e, "SUBGRAPH EXPORT NOT DEFINED" );

            for ( e = c->F_BODY->exp; e != NULL; e = e->esucc )
		if ( (!IsImport( c,        e->eport )) && 
		     (!IsImport( c->F_GEN, e->eport ))   )
	            CheckEdgeError( e, "SUBGRAPH EXPORT NOT DEFINED" );

            for ( e = c->F_RET->exp; e != NULL; e = e->esucc )
		if ( (!IsImport( c,         e->eport )) &&
		     (!IsImport( c->F_GEN,  e->eport )) &&
		     (!IsImport( c->F_BODY, e->eport ))   )
	            CheckEdgeError( e, "SUBGRAPH EXPORT NOT DEFINED" );

	    break;

        case IFLoopA:
	case IFLoopB:
	    if ( !IsNodeListEmpty( c->L_INIT ) )
		CheckNodeError( c->L_INIT, "LOOP INIT NOT NORMALIZED" );

	    if ( !IsTestNormalized( c ) )
		CheckNodeError( c->L_TEST, "LOOP TEST NOT NORMALIZED" );

	    if ( !IsRetNormalized( c ) )
		CheckNodeError( c->L_RET, "LOOP RETURN NOT NORMALIZED" );

	    CheckForUnusedRports( c->L_RET );
	    CheckForUnusedLTports( c );

	    p = CheckPortNumbers( 1, c, NULL );                          /* K */
	    p = CheckPortNumbers( p, c->L_INIT, NULL );                  /* L */

	    if ( IsLoopA( c ) )
		CheckPortNumbers( p, c->L_BODY, c->L_INIT );             /* T */

            CheckPortNumbers( 1, c->L_RET, NULL );                       /* R */

            for ( e = c->L_INIT->exp; e != NULL; e = e->esucc )
		if ( !IsImport( c, e->eport ) )
	            CheckEdgeError( e, "SUBGRAPH EXPORT NOT DEFINED" );

            for ( e = c->L_BODY->exp; e != NULL; e = e->esucc )
		if ( (!IsImport( c,         e->eport )) &&
		     (!IsImport( c->L_INIT, e->eport ))   )
	            CheckEdgeError( e, "SUBGRAPH EXPORT NOT DEFINED" );

            for ( e = c->L_TEST->exp; e != NULL; e = e->esucc )
		if ( (!IsImport( c,         e->eport )) &&
		     (!IsImport( c->L_INIT, e->eport )) &&
		     (!IsImport( c->L_BODY, e->eport ))   )
	            CheckEdgeError( e, "SUBGRAPH EXPORT NOT DEFINED" );

            for ( e = c->L_RET->exp; e != NULL; e = e->esucc )
		if ( (!IsImport( c,         e->eport )) &&
		     (!IsImport( c->L_INIT, e->eport ))   )
	            CheckEdgeError( e, "SUBGRAPH EXPORT NOT DEFINED" );

	    break;
	}

    CheckForUnusedKports( c );
}


/**************************************************************************/
/* LOCAL  **************         CheckNode         ************************/
/**************************************************************************/
/* PURPOSE: CHECK THE STRUCTURE AND SEMANTICS OF ALL NODES IN GRAPH g.    */
/*          BESIDES THE CHECKS PERFORMED BY CheckCompoundNode, DATA FLOW  */
/*          ORDERING IS CHECKED; CONSTANT PROPAGATION IS CHECKED; A       */
/*          CHECK IS MADE FOR REDUNDANT NOT PAIRS; A CHECK IS MADE FOR    */
/*          REDUNDANT IMPORTS TO ALL COMPOUND NODES; A CHECK IS MADE FOR  */
/*          UNDEFINED AND DEAD NODES; AND THE FORMAT OF ALL CONSTANT      */
/*          IMPORTS IS CHECKED. A CHECK IS MADE FOR LOOP INVARIANTS AND   */
/*          NODES THAT SHOULD BE MIGRATED INTO SELECT OR COMPOUND NODES.  */
/**************************************************************************/

static void CheckNode( g )
register PNODE g;
{
    register PEDGE i;
    register PEDGE e;
    register PNODE n;
    register PNODE nd;
    register int   f;

    if ( IsIGraph( g ) )
	return;

    CheckIfDFOrdered( g );
    CheckConstantImports( g );

    for ( n = g->G_NODES; n != NULL; n = n->nsucc ) {
	CheckConstantImports( n );

	if ( n->exp == NULL )
	    CheckNodeError( n, "DEAD NODE" );

        for ( e = n->exp; e != NULL; e = e->esucc )
	    if ( IsUndefined( e->dst ) )
		CheckEdgeError( e, "DESTINATION NODE UNDEFINED" );

        for ( i = n->imp; i != NULL; i = i->isucc )
	    if ( IsConst( i ) )
		continue;
            else if ( IsUndefined( i->src ) )
		CheckEdgeError( i, "SOURCE NODE UNDEFINED" );

        if ( IsCompound( n ) ) {
	    if ( !AreKportsCombined( n ) )
		CheckNodeError( n, "REDUNDANT K IMPORTS" );

	    for ( i = n->imp; i != NULL; i = i->isucc )
		if ( IsConst( i ) )
		    CheckEdgeError( i, "CONSTANT NOT PROPAGATED" );

	    for ( g = n->C_SUBS; g != NULL; g = g->gsucc )
		CheckNode( g );

	    switch ( n->type ) {
	        case IFLoopA:
	        case IFLoopB:
		    for ( nd = n->L_BODY->G_NODES; nd != NULL; nd = nd->nsucc )
	                if ( IsInvariant( nd ) )
	                    CheckNodeError( nd, "LOOP INVARIANT" );

		    break;

	        case IFForall:
		    for ( nd = n->F_BODY->G_NODES; nd != NULL; nd = nd->nsucc )
	                if ( IsInvariant( nd ) )
	                    CheckNodeError( nd, "LOOP INVARIANT" );

		    break;

		case IFSelect:
		    for ( i = n->imp; i != NULL; i = i->isucc ) {
			if ( IsConst( i ) )
			    continue;

			if ( IsExport( n->S_TEST, i->iport ) )
			    continue;

			if ( !(IsExport( n->S_ALT, i->iport ) ^
			       IsExport( n->S_CONS, i->iport)) )
			    continue;

			if ( IsSimple( i->src ) || IsCompound( i->src ) ) {
	                    f = TRUE;

		            for ( e = i->src->exp; e != NULL; e = e->esucc )
                                if ( e->dst != n )
			            f = FALSE;

                            if ( f )
			        CheckNodeError( i->src, "MIGRATION ERROR");
		            }
			}

		    break;

		case IFTagCase:
		    for ( i = n->imp->isucc; i != NULL; i = i->isucc ) {
			if ( IsConst( i ) )
			    continue;

			f = 0;

			for ( g = n->C_SUBS; g != NULL; g = g->gsucc )
			    if ( IsExport( g, i->iport ) )
				f++;

			if ( f != 1 )
			    continue;

			if ( IsSimple( i->src ) || IsCompound( i->src ) ) {
		            f = TRUE;

		            for ( e = i->src->exp; e != NULL; e = e->esucc )
                                if ( e->dst != n )
			            f = FALSE;

                            if ( f )
		                CheckNodeError( i->src, "MIGRATION ERROR");
		            }
			}

		    break;
		}

	    CheckCompoundNode( n );
	    continue;
	    }

	if ( IsNot( n ) )
	    if ( (n->exp->esucc == NULL) && (IsNot( n->exp->dst )) )
		CheckNodeError( n, "REDUNDANT NOT NODE PAIRS" ); 
        }
}


/**************************************************************************/
/* GLOBAL **************          If1Check         ************************/
/**************************************************************************/
/* PURPOSE: CHECK THE STRUCTURE AND SEMANTICS OF ALL FUNCTION GRAPHS.     */
/**************************************************************************/

void If1Check()
{
    register PNODE f;

    for ( f = glstop->gsucc; f != NULL; f = f->gsucc )
	CheckNode( cfunct = f );
}
