/*  $Header: in.c,v 3.2 89/03/03 09:59:55 sylvia Exp $ */
/*
 *  This file is part of the Amsterdam SGML Parser.
 *
 *  Copyright: Faculteit Wiskunde en Informatica
 *             Department of Mathematics and Computer Science
 *             Vrije Universiteit Amsterdam
 *             The Netherlands
 *
 *  Authors:   Sylvia van Egmond
 *             Jos Warmer
 */
#define  ENDOFSTREAM  (-1)
#include "types.h"
#include "entity.h"
#include "file_in.h"
#include "group.h"
#include "in.h"
#include "str_in.h"
#include "token_in.h"

#define FILE_INPUT    101
#define STRING_INPUT  102
#define TOKEN_INPUT   103
#define ENTITY_INPUT   104

typedef struct Stream_struct {
    int    type;
    String name;
    int    nr_read;
    union {
	P_Filestream   f;
	P_StringStream s;
	P_TokenStream  t;
    } state;
} Stream;

/*
 *  Define local variables.
 */
static P_Group           input_group;
static Stream           *input_top;
static P_Iterator        input_iter;
static Stream           *input=0;   	/* current input stream */
static VoidFunctionInt   real_pushback_ch;
static IntFunction       real_next_ch;
static IntFunction       real_current_ch;
static IntFunction       real_current_ch_value;
static int               decrement_line = 0;
static String            given_filename = 0;

Bool is_end_of_stream(ch)
int ch;
{
    return( ((input->type==FILE_INPUT)and(ch==(-1)))
			      or
	((input->type != FILE_INPUT) and (ch==0)) );
}

#ifdef DEBUG
static Bool           debug = FALSE;

void print_stream(file, s)
FILE     *file;
Stream   *s;
{
    if( s == 0 ){ fprintf(file, "(nil-stream)"); return ;}
    switch( s->type ){
    case FILE_INPUT   : fprintf(file,"File_input %s:", s->name);
			break;
    case ENTITY_INPUT : fprintf(file,"Entity_input %s:", s->name);
			break;
    case STRING_INPUT : fprintf(file,"String_input: ");
			break;
    case TOKEN_INPUT  : fprintf(file,"Token_input %s:", s->name);
			break;
    default : report(NO_LABEL, FATAL, 0, 0, s->type, "print_stream");
	      break;
    }
}

void print_input()
{
    Stream    *s;
    P_Iterator it;

    DEB("top = "); print_stream(fpdg, input_top);
    DEB("; input = "); print_stream(fpdg, input);
    s = (Stream*)iterator_elem(input_iter);
    DEB("; iter = "); print_stream(fpdg, s);
    DEB("\n");
    it = group_iterator(input_group);
    while( s = (Stream*)iterator_next(it) ){
	print_stream(fpdg, s);
	fprintf(fpdg," ; ");
    }
    DEB("\n");
}

#endif

void delete_stream(s)
Stream *s;
{
    switch(s->type){
	case FILE_INPUT   : delete_file_stream(s->state.f);
			    CFREE(s->name);
			    break;
	case STRING_INPUT : delete_string_stream(s->state.s);
			    CFREE(s->name);
			    break;
	case ENTITY_INPUT : delete_token_stream(s->state.t);
			    break;
	case TOKEN_INPUT  : if( streq(s->name, "TOK_RS") ){
				decrement_line = 0;
			    }
			    delete_token_stream(s->state.t);
			    CFREE(s->name);
			    break;
	default : report(NO_LABEL, FATAL, 0, 0, s->type, "delete_stream");
		  break;
    }
    FREE(s, sizeof(Stream));
}

static void switch_input(stream)
Stream   *stream;
{
    input = stream;

#ifdef DEBUG
    if( debug ){
	fprintf(fpdg,"switch input to %d\n", input->type);
	print_input();
    }
#endif
    switch( input->type ){
    case FILE_INPUT:
	real_next_ch      = file_next_ch;
	real_pushback_ch  = file_pushback_ch;
	real_current_ch   = file_current_ch  ;
	real_current_ch_value   = file_current_ch  ;
	file_stream(stream->state.f);
	break;
    case STRING_INPUT:
	real_next_ch      = string_next_ch;
	real_pushback_ch  = string_pushback_ch;
	real_current_ch   = string_current_ch  ;
	real_current_ch_value   = string_current_ch  ;
	string_stream(stream->state.s);
	break;
    case TOKEN_INPUT:
    case ENTITY_INPUT:
	real_next_ch      = token_next_ch;
	real_pushback_ch  = token_pushback_ch;
	real_current_ch   = token_current_ch  ;
	real_current_ch_value   = token_current_ch_value  ;
	token_stream(stream->state.t);
	break;
    }
}

void done_ch()
{
    P_Iterator iter;
    Stream    *tmp_in;
    Stream    *d;

    DEB("done_ch:\n");
    if( input != input_top ){
        iter = group_iterator(input_group);
        tmp_in = (Stream*)iterator_previous(iter);
        while( tmp_in != input ){
	    DEB("delete input : ");
	    d = (Stream*)iterator_elem(iter);
	    if( d->type == ENTITY_INPUT ){
		leave_entity(d->name);
	    }
#ifdef DEBUG
	    if( debug ){ print_stream(fpdg, d); }
#endif
	    DEB("\n");
            group_delete_previous(input_group, iter, delete_stream);
	    tmp_in = (Stream*)iterator_elem(iter);
        }
    }
    input->nr_read = 0;
    input_iter = group_iterator(input_group);
    iterator_previous(input_iter);
    input_top = input;
}

int  current_ch  ()
{
    int ch;

    ch = (*real_current_ch  )();
    if( is_end_of_stream(ch) ){
	return ENDOFSTREAM;
    }
    return ch;
}

int  current_ch_value  ()
{
    int ch;

    ch = (*real_current_ch_value  )();
    if( is_end_of_stream(ch) ){
	return ENDOFSTREAM;
    }
    return ch;
}

int next_ch()
{
    int     ch;
    Stream *old_input;
 
    ch = (*real_next_ch)();
    if( is_end_of_stream(ch) ){
        old_input = (Stream*)iterator_previous(input_iter);
        if( old_input == 0 ){
	    input->nr_read++;
	    iterator_next(input_iter);
            return ENDOFSTREAM;
        } else {
	    DEB("next_ch: switch to old input\n");
	    (*real_pushback_ch)(1);
            switch_input(old_input);
            return( next_ch() );
        }
    } else {
        input->nr_read++;
    }
    return ch;
}

void pushback_ch(l)
int l;
{
    Stream *old_input;

    if( l == 0 ){ return; }
    DEB1("pushback_ch %d\n", l);
    if( input->nr_read == 0 ){
	report(SYSTEM_PB, FATAL, 0, 0, l, input->nr_read);
        return;
    }
    while( l > input->nr_read ){
        (*real_pushback_ch)(input->nr_read);
        l -= input->nr_read;
        input->nr_read = 0;
	old_input = (Stream*)iterator_next(input_iter);
	if( old_input != 0 ){
	    DEB("pushback_ch switch to old_input\n");
	    switch_input( old_input );
	} else {
	    report(SYSTEM_PB, FATAL, 0, 0, l, 0);
	    iterator_previous(input_iter);
	}
    }

    if( l > 0 ){
	input->nr_read -= l;
	(*real_pushback_ch)(l);
    }
    if( input->nr_read == 0 ){
	old_input = (Stream*)iterator_next(input_iter);
	if( old_input != 0 ){
	    if( old_input->nr_read > 0 ){
		DEB("pushback_ch switch on 0 to old_input\n");
		switch_input( old_input );
		/* (*real_pushback_ch)(1); */
	    }
	} else {
	    iterator_previous( input_iter );
	}
    }
}


static Stream*  new_stream_file(file, name)
FILE   *file;
String  name;
{
    Stream *stream;

    stream          = (Stream*)MALLOC( (unsigned)( sizeof(Stream) ) );
    stream->type    = FILE_INPUT;
    stream->nr_read = 0;
    stream->name    = strsave(name);
    stream->state.f = new_file_stream(file, name);
    return stream;
}

static Stream*  new_stream_entity(entity_name, tokens, values)
String entity_name;
Token *tokens;
int*   values;
{
    Stream *stream;

    stream          = (Stream*)MALLOC( (unsigned)( sizeof(Stream) ) );
    stream->type    = ENTITY_INPUT;
    stream->nr_read = 0;
    stream->name    = entity_name;
    stream->state.t = new_token_stream(tokens, values);
    return stream;
}

static Stream*  new_stream_string(string)
String  string;
{
    Stream *stream;

    stream          = (Stream*)MALLOC( (unsigned)( sizeof(Stream) ) );
    stream->type    = STRING_INPUT;
    stream->nr_read = 0;
    stream->name    = strsave("string_input");
    stream->state.s = new_string_stream(string);
    return stream;
}

static Stream*  new_stream_token(name, tokens, values)
String name;
int   *tokens;
int   *values;
{
    Stream *stream;

    stream          = (Stream*)MALLOC( (unsigned)( sizeof(Stream) ) );
    stream->type    = TOKEN_INPUT;
    stream->nr_read = 0;
    stream->name    = strsave(name);
    stream->state.t = new_token_stream(tokens, values);
    return stream;
}

void new_input(stream)
Stream  *stream;
{
    group_add(input_group, stream);
    input_top = stream;
    input_iter = group_iterator(input_group);
    iterator_previous(input_iter);
    switch_input(stream);
}

void entity_input(entity_name, tokens, values)
String entity_name;
Token *tokens;
int   *values;
{
    Stream *stream;

    stream = new_stream_entity(entity_name, tokens, values);
    DEB("entity_input :\n");
    new_input(stream);
}

void string_input(string)
String string;
{
    Stream *stream;

    stream = new_stream_string(string);
    DEB("string_input :\n");
    new_input(stream);
}

void token_input(name, tokens, values)
String name;
int   *tokens;
int   *values;
{
    Stream *stream;

    stream = new_stream_token(name, tokens, values);
    DEB("token_input :\n");
    new_input(stream);
    if( streq(name, "TOK_RS") ){
	decrement_line = 1;
    }
}

Bool file_input(file_name)
String file_name;
{
    char    st_name[10];
    Stream *stream;
    FILE   *file;

    if (file_name == (String) 0) {
	file = stdin;
        strcpy(st_name, "stdin");
	file_name = st_name;
    }
    else {
        file = fopen(file_name, "r");
    }
    if (file == 0) return(FALSE);
    stream = new_stream_file(file, file_name);
    DEB("file_input :\n");
    new_input(stream);
    return(TRUE);
}

void init_input(file_name, given_name)
String file_name;
String given_name;
{
    input_group = group_create();
    if (strcmp(file_name, "-") == 0) {
	file_name = (String) 0;
    }
    if ( not file_input(file_name) ){
	report(FILE_OPEN, FATAL, 0, 0, file_name);
    }
    if (given_name) given_filename = strsave(given_name);
}

String input_location()
{
    static String location = 0;
	   String filenm   = 0;
    static int    length   = 0;
           int    new_length;

    if (given_filename) filenm = given_filename;
    else filenm = file_name();

    new_length = 9 + strlen(filenm) + 12 + 1;

    if( length < new_length ){
	if( location != (String)0 ){
	    FREE(location, length);
	}
	length = new_length;
	location = (String) CALLOC( length, sizeof(char) );
    }

    sprintf(location,"\"%s\": line %d", filenm,
		       file_line_nr() - decrement_line);
    return location;
}

#ifdef DEBUG
void debug_in(b)
Bool b;
{
    debug = b;
}
#endif
