/* $Header: context.c,v 3.0 88/04/13 15:48:17 jos Exp $ */
/*
 *  This file is part of the Amsterdam SGML Parser.
 *
 *  Copyright: Faculteit Wiskunde en Informatica
 *             Department of Mathematics and Computer Science
 *             Vrije Universiteit Amsterdam
 *             The Netherlands
 *
 *  Authors:   Sylvia van Egmond
 *             Jos Warmer
 */
/*
 *  Deze functies berekenen de 'optional' of 'required' status.
 *  Zie 'contextual_required' voor een uitleg van de
 *  verschillende soorten statussen.
 *  Wellicht is een andere opdeling handiger, zo ja dan pas
 *  ik die wel aan.
 */
#include "types.h"
#include "node.h"
#include "context.h"

/*  returns TRUE if 'node' is inherently optional.
 *  returns FALSE if it is not inherenrtly optional.
 *
 *  See 4.159, page 12, Standard.
 *
 *  Determing wether a node is inherently optional is possible after the
 *  DTD is read.
 *  This function walks through the tree from which it is the root.
 */
static int  inherent_optional(node)
P_Node node;
{
    P_Iterator  iterator;
    int         optional;
    int         tmp;
    
    switch( node_type(node) ){
    case CONN_OR :
        optional = FALSE;
        iterator = group_iterator(node_group(node));
	while( iterator_next(iterator) ){
	    tmp = inherent_optional(iterator_elem(iterator));
            optional = optional or tmp;
        }
	node_set_status(node, (optional ? INHERENTLY_OPT : UNDEFINED) );
        break;
    case CONN_AND :
    case CONN_SEQ :                                               /* 4.159 c */
        optional = TRUE;
        iterator = group_iterator(node_group(node));
	while( iterator_next(iterator) ){
	    tmp = inherent_optional(iterator_elem(iterator));
            optional = optional and tmp;
        }
	node_set_status(node,  (optional ? INHERENTLY_OPT : UNDEFINED) );
        break;
    case GI :
    case KEY :
        node_set_status(node, UNDEFINED);
        break;
    default :
	report(NO_LABEL, FATAL, 0, 0, node_type(node), "inherent_optional");
        break;
    } /* end switch */

    /*  Overrule foregoing decision if the occurrence indicator is OR, REP
     */
    if( (node_occ(node) == OCC_OPT) or (node_occ(node)==OCC_REP) ){/* 4.159 a */
        node_set_status(node, INHERENTLY_OPT);
    }

    return  (node_status(node) == INHERENTLY_OPT);
}

/*  Determines wether a node is contextually required.
 *  Also dertermines the dependencies:
 *  ALWAYS_REQUIRED  : is always required. Code generated:
 *                     <elem>? elem </elem>
 *  CONTEXT_REQUIRED : is required only when a token of the group which the
 *                     element is a part of is satisfied.
 *                     Code generated:
 *                     [%if( s ) <elem>? | <elem> { s = s(parent) = TRUE; } ]
 *                     elem </elem>
 *
 *  For all members of a CONN_SEQ group, that are not *REQUIRED*,
 *  code generated is:
 *      [ <elem> { s = s(parent) = TRUE; } ] elem </elem>
 *
 *  See 4.62, page 8, Standard.
 *
 *  Determing wether an element is contextually required is possible when
 *  the element is encountered in the document.
 *  Only SEQ groups are of interest, so 'node' must be a SEQ group.
 */
static void  contextual_required(node)
P_Node node;
{
    P_Node      child;
    P_Iterator  iterator;
    Bool        first_non_opt = TRUE;
    Bool        seen_opt      = FALSE;
    
    switch( node_status(node) ){
        case UNDEFINED    :
        case INHERENTLY_OPT :
            iterator = group_iterator(node_group(node));
	    while( child  = (P_Node)iterator_next(iterator) ){
                if( node_status(child) != INHERENTLY_OPT ){
		    if( first_non_opt ){
			first_non_opt = FALSE;
			if( seen_opt ){
			    node_set_status(child, CONTEXT_REQUIRED);
			} else {
			    node_set_status(child, UNDEFINED);
			}
		    } else {
			node_set_status(child, ALWAYS_REQUIRED);
		    }
		} else {
		    seen_opt = TRUE;
		}
	    }
	    break;
        case ALWAYS_REQUIRED :
            iterator = group_iterator(node_group(node));
	    while( child  = (P_Node)iterator_next(iterator) ){
                if( node_status(child) != INHERENTLY_OPT ){
		    node_set_status(child, ALWAYS_REQUIRED);
		}
	    }
	    break;
	case CONTEXT_REQUIRED :
            iterator = group_iterator(node_group(node));
	    while( child  = (P_Node)iterator_next(iterator) ){
                if( node_status(child) != INHERENTLY_OPT ){
		    if( first_non_opt ){
			first_non_opt = FALSE;
			node_set_status(child, CONTEXT_REQUIRED);
		    } else {
			node_set_status(child, ALWAYS_REQUIRED);
		    }
		}
	    }
	    break;
        default :
	    report(NO_LABEL, FATAL, 0, 0, node_status(node),
		   "context_rewuired");
	    break;
    } /* switch */
}

/*  Depth first traversal, calling 'contextual_required' at every SEQ node.
 *  Sets level.
 */
static void walk(node, level)
P_Node  node;
int     level;
{
    int          t;
    P_Iterator   iterator;
    
    /*   node->level = level; */
    t = node_type(node);
    if( t == CONN_SEQ ){
        contextual_required(node);
    }
    if( t==CONN_SEQ or t==CONN_AND or t==CONN_OR ){
        iterator = group_iterator(node_group(node));
	while( node  = (P_Node)iterator_next(iterator) ){
            walk(node, level+1);
	}
    }
}

/*  Calculates the status of 'node', which is the root of an element definition.
 */
void  calculate_status(node)
P_Node   node;
{
    inherent_optional(node);

    if( node_status(node) != INHERENTLY_OPT ){                  /* 4.62a */
        node_set_status(node, ALWAYS_REQUIRED);
    }
    walk(node, 1);
}
