;;; -*-Scheme-*-
;;;
;;; $Id: mitutil.sc,v 1.9 1993/03/11 17:42:53 cph Exp $
;;;
;;; Copyright (c) 1993 Massachusetts Institute of Technology
;;;
;;; This material was developed by the Scheme project at the
;;; Massachusetts Institute of Technology, Department of Electrical
;;; Engineering and Computer Science.  Permission to copy this
;;; software, to redistribute it, and to use it for any purpose is
;;; granted, subject to the following restrictions and understandings.
;;;
;;; 1. Any copy made of this software must include this copyright
;;; notice in full.
;;;
;;; 2. Users of this software agree to make their best efforts (a) to
;;; return to the MIT Scheme project any improvements or extensions
;;; that they make, so that these may be included in future releases;
;;; and (b) to inform MIT of noteworthy uses of this software.
;;;
;;; 3. All materials developed as a consequence of the use of this
;;; software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4. MIT has made no warrantee or representation that the operation
;;; of this software will be error-free, and MIT is under no
;;; obligation to provide any services, by way of maintenance, update,
;;; or otherwise.
;;;
;;; 5. In conjunction with products arising from the use of this
;;; material, there shall be no use of the name of the Massachusetts
;;; Institute of Technology nor of any adaptation thereof in any
;;; advertising, promotional, or sales literature without prior
;;; written consent from MIT in each case.

;;;; Utilities from MIT Scheme

(module mitutil
	(top-level
	 error:wrong-type-argument
	 error:wrong-number-of-arguments
	 vector-copy
	 list-copy
	 sort
	 append-map
	 for-all?
	 symbol-append
	 append!
	 reverse!
	 write-string
	 call-with-values
	 values
	 make-%defstruct-methods
	 set-%defstruct-methods-print-procedure!
	 make-%defstruct-%record
	 %defstruct-%record-description))
(include "mitutil.sch")

(define (error:wrong-type-argument object name procedure)
  (error procedure
	 "The object ~S is not ~A ~A."
	 object
	 (if (char-vowel? (string-ref name 0)) "an" "a")
	 name))

(define (error:wrong-number-of-arguments procedure arity args)
  (error (if (symbol? procedure) procedure 'UNKNOWN)
	 "Wrong number of arguments, expected ~S, got ~S."
	 arity
	 (length args)))

(define (serror procedure message . objects)
  (apply error
	 procedure
	 (let loop ((message message) (objects objects))
	   (if (null? objects)
	       message
	       (loop (string-append message " ~S") (cdr objects))))
	 objects))

(define (char-vowel? char)
  (or (char-ci=? char #\a)
      (char-ci=? char #\e)
      (char-ci=? char #\i)
      (char-ci=? char #\o)
      (char-ci=? char #\u)))

(define (vector-copy vector)
  (let ((n (vector-length vector)))
    (let ((result (make-vector n)))
      (do ((i 0 (+ i 1)))
	  ((= i n))
	(vector-set! result i (vector-ref vector i)))
      result)))

(define (list-copy list)
  (if (null? list)
      '()
      (cons (car list) (list-copy (cdr list)))))

(define (sort l pred)
  (letrec
      ((loop
	(lambda (l)
	  (if (and (pair? l) (pair? (cdr l)))
	      (split l '() '())
	      l)))
       (split
	(lambda (l one two)
	  (if (pair? l)
	      (split (cdr l) two (cons (car l) one))
	      (merge (loop one) (loop two)))))
       (merge
	(lambda (one two)
	  (cond ((null? one) two)
		((pred (car two) (car one))
		 (cons (car two)
		       (merge (cdr two) one)))
		(else
		 (cons (car one)
		       (merge (cdr one) two)))))))
    (loop l)))

(define (append-map procedure list . lists)
  (if (null? lists)
      (let loop ((list list))
	(cond ((pair? list)
	       (append (procedure (car list))
		       (loop (cdr list))))
	      ((null? list)
	       '())
	      (else
	       (error 'APPEND-MAP "Argument not a list: ~S" list))))
      (let loop ((lists (cons list lists)))
	(let parse-cars
	    ((lists lists)
	     (receiver
	      (lambda (cars cdrs)
		(append (apply procedure cars)
			(loop cdrs)))))
	  (cond ((null? lists)
		 (receiver '() '()))
		((pair? (car lists))
		 (parse-cars (cdr lists)
			     (lambda (cars cdrs)
			       (receiver
				(cons (car (car lists)) cars)
				(cons (cdr (car lists)) cdrs)))))
		((null? (car lists))
		 '())
		(else
		 (error 'APPEND-MAP
			"Argument not a list: ~S"
			(car lists))))))))

(define (for-all? list predicate)
  (let loop ((list list))
    (or (null? list)
	(and (predicate (car list))
	     (loop (cdr list))))))

(define (symbol-append . symbols)
  (string->symbol (apply string-append (map symbol->string symbols))))

(define (append! . lists)
  (apply append lists))

(define (reverse! list)
  (reverse list))

(define (write-string string . port)
  (apply display string port))

(define (call-with-values generator consumer)
  (apply consumer (generator)))

(define (values . x)
  x)

;;;; Support for DEFINE-STRUCTURE macro.

(define (make-%defstruct-methods type name field-names field-indexes
				 print-procedure)
  `((%DEFSTRUCT-TYPE . ,(vector type name field-names field-indexes))
    (WRITE-RECORD . ,print-procedure)
    ,@%defstruct-default-methods))

(define %defstruct-default-methods
  (let ((%defstruct-%to-write
	 (lambda (record port indent wlevel wlength circle)
	   (write-record record port))))
    `((%TO-WRITE . ,%defstruct-%to-write)
      (%TO-DISPLAY . ,%defstruct-%to-write))))

(define (write-record record port)
  (let ((method (%record-lookup-method record 'WRITE-RECORD)))
    (if method
	(method record port)
	(begin
	  (write-string "#*" port)
	  (write (let ((type (%record-lookup-method record '%DEFSTRUCT-TYPE)))
		   (if type
		       (vector-ref type 1)
		       'RECORD))
		 port)
	  (write-char #\* port)))))

(define (set-%defstruct-methods-print-procedure! methods print-procedure)
  (set-cdr! (or (assq 'WRITE-RECORD methods)
		(error:wrong-type-argument
		 methods "DEFINE-STRUCTURE methods"
		 'SET-%DEFSTRUCT-METHODS-PRINT-PROCEDURE!))
	    print-procedure))

(define (make-%defstruct-%record methods . fields)
  (let ((record (list->%record fields)))
    (%record-methods-set! record methods)
    record))

(define (%defstruct-%record-description record)
  (let ((type
	 (or (%record-lookup-method record '%DEFSTRUCT-TYPE)
	     (error:wrong-type-argument record "DEFINE-STRUCTURE record"
					'%DEFSTRUCT-%RECORD-DESCRIPTION))))
    (map (lambda (name index)
	   (list name
		 (%record-ref record index)))
	 (vector-ref type 2)
	 (vector-ref type 3))))

;;; (define (ppr record . port)
;;;   (let ((port (if (null? port) (current-output-port) (car port))))
;;;     (write record port)
;;;     (newline port)
;;;     (for-each (lambda (entry)
;;; 		(pp entry port)
;;; 		(newline port))
;;; 	      (%defstruct-description record))))
