;;; menu.scm -- Display a menu and allow users to pick one.
;;;
;;; Surely there is a better way?

(require 'logical)
(require "scm_dir:smgdef.scm")
(require "scm_dir:readstring.scm")


;;; CHOICES is a vector containing the choices.  PBD is the pasteboard.
;;; KBD is the keyboard.  NUMBER-OF-ROWS is the height of the menu.
;;; NUMBER-OF-COLS is the width of the menu.  MENU-ROW and MENU-COL
;;; are the Y and X coordinates of the top left corner of the menu.
;;; OPTIONS are two optional arguments: START-ROW, the choice to start
;;; on, and LABEL-TEXT, the text to label the menu with.

(define (menu choices pbd kbd
	      number-of-rows number-of-cols
	      menu-row menu-col
	      .
	      options)
  (let* ((noselect-keys (list SMG$K_TRM_CTRLZ SMG$K_TRM_F10 SMG$K_TRM_F20))
	 (select-keys   (list SMG$K_TRM_CTRLJ SMG$K_TRM_CTRLM
			      SMG$K_TRM_ENTER SMG$K_TRM_SELECT))
	 (exit-keys (append select-keys noselect-keys))
	 (choices-len (vector-length choices))
	 (mdpy (vector-ref (smg$create_virtual_display
			    ;; Make sure it is at least as many rows as user
			    ;; asked for: viewports can't be bigger than the
			    ;; display.
			    (max choices-len number-of-rows)
			    number-of-cols SMG$M_BORDER) 1))
	 (edpy (vector-ref (smg$create_virtual_display 1 80) 1))
	 (more-width 14)
	 (more-column (+ menu-col number-of-cols (- more-width))) 
	 (tdpy (vector-ref (smg$create_virtual_display 1 more-width
						       #f SMG$M_REVERSE) 1))
	 (bdpy (vector-ref (smg$create_virtual_display 1 more-width
						       #f SMG$M_REVERSE) 1))
	 (mvpt-height number-of-rows)
	 ;; local functions
	 (get-key (lambda ()
		    (vector-ref (smg$read_keystroke kbd) 1)))
	 (message (lambda (str . options)
		    (if str
			(smg$put_chars edpy str 1 1 SMG$M_ERASE_TO_EOL))
		    (if (member 'bell options)
			(smg$ring_bell edpy))
		    (if (member 'get-key options)
			(get-key))
		    (if (not (member 'sticky options))
			(smg$erase_display edpy))))
	 (finish (lambda (key)
		   ;; We subtract one because vectors run from 0 up and
		   ;; display coordinates run from 1 up.
		   (let ((row (- (vector-ref (smg$cursor_row mdpy) 0) 1)))
		     (smg$delete_viewport mdpy)
		     (smg$delete_virtual_display mdpy)
		     (smg$delete_virtual_display edpy)
		     (smg$delete_virtual_display tdpy)
		     (smg$delete_virtual_display bdpy)
		     (if (member key select-keys) row #f))))
	 (display-more-boxes
	  (lambda ()
	    (let* ((vc (smg$get_viewport_char mdpy))
		   (row (vector-ref (smg$cursor_row mdpy) 0))
		   (vrs (vector-ref vc 1))
		   (vnr (vector-ref vc 3))
		   (vre (+ vrs vnr -1)))
	      (smg$begin_pasteboard_update pbd)
	      (if (> vrs 1)
		  (smg$put_chars tdpy (string-append " "
						     (number->string (- vrs 1))
						     " more ^")
				 1 1 SMG$M_ERASE_TO_EOL)
		  (smg$put_chars tdpy " No more ^" 1 1 SMG$M_ERASE_TO_EOL))
	      (if (< vre choices-len)
		  (smg$put_chars bdpy (string-append " " (number->string
							  (- choices-len vre))
						     " more v")
				 1 1 SMG$M_ERASE_TO_EOL)
		  (smg$put_chars bdpy " No more v" 1 1 SMG$M_ERASE_TO_EOL))
	      (smg$end_pasteboard_update pbd))))
	 (adjust-menu
	  (lambda ()
	    (let* ((vc (smg$get_viewport_char mdpy))
		   (row (vector-ref (smg$cursor_row mdpy) 0))
		   (vrs (vector-ref vc 1))
		   (vnr (vector-ref vc 3))
		   (vre (+ vrs vnr -1)))
	      (smg$begin_pasteboard_update pbd)
	      (if (> row vre)
		  (smg$scroll_viewport mdpy SMG$M_UP (- row vre))
		  (if (< row vrs)
		      (smg$scroll_viewport mdpy SMG$M_DOWN (- vrs row))))
	      (display-more-boxes)
	      (smg$end_pasteboard_update pbd)
	      (smg$set_cursor_abs mdpy row))))
	 (scroll-menu
	  (lambda (direction)
	    (let* ((vc (smg$get_viewport_char mdpy))
		   (row (vector-ref (smg$cursor_row mdpy) 0))
		   (vrs (vector-ref vc 1))
		   (vnr (vector-ref vc 3))
		   (vre (+ vrs vnr -1))
		   (wanted-row (if (eq? direction 'down)
				   (max 1 (- row vnr))
				   (min choices-len (+ row vnr))))
		   (delta (min (abs (- wanted-row row))
			       (if (eq? direction 'down)
				   (- vrs 1)
				   (- choices-len vre))))
		   )

	      (smg$begin_pasteboard_update pbd)
	      (smg$scroll_viewport mdpy (if (eq? direction 'down)
					    SMG$M_DOWN SMG$M_UP)
				   delta)
	      (display-more-boxes)
	      (smg$end_pasteboard_update pbd)
	      (smg$set_cursor_abs mdpy wanted-row))))
	 )
    
    (smg$create_viewport mdpy 1 1 mvpt-height number-of-cols)
    (do ((i 0 (+ i 1)))
	((>= i choices-len))
      (smg$put_chars mdpy (vector-ref choices i) (+ i 1) 1))
    (smg$set_cursor_abs mdpy 1 1)
    (if (not (null? options))
	(begin
	  (if (car options)
	      (begin
		(smg$set_cursor_abs mdpy (min (max (+ 1 (car options)) 1)
					      choices-len) 1)
		(adjust-menu)))
	  (if (> (length options) 1)
	      (smg$label_border mdpy (cadr options) #f 2 SMG$M_REVERSE))))
    (smg$begin_pasteboard_update pbd)
    (smg$paste_virtual_display edpy pbd 24 1)
    (smg$paste_virtual_display mdpy pbd menu-row menu-col)
    (smg$paste_virtual_display tdpy pbd (- menu-row 1) more-column)
    (smg$paste_virtual_display bdpy pbd
			       (+ menu-row
				  (vector-ref (smg$get_viewport_char mdpy) 3))
			       more-column)
    (display-more-boxes)
    (smg$end_pasteboard_update pbd)
    (smg$set_cursor_abs mdpy)
    (do ((key (get-key) (get-key)))
	((member key exit-keys) (finish key))
      (cond
       ((= key SMG$K_TRM_UP)
	(smg$set_cursor_rel mdpy -1)
	(adjust-menu)
	)
       ((= key SMG$K_TRM_DOWN)
	;; Don't move below choices if choices are fewer than the height
	;; of the menu.
	(if (< (vector-ref (smg$cursor_row mdpy) 0) choices-len)
	    (smg$set_cursor_rel mdpy 1))
	(adjust-menu)
	)
       ((= key SMG$K_TRM_PREV_SCREEN)
	(cond
	 (#f
	  (smg$set_cursor_rel
	   mdpy (- (min (- (vector-ref (smg$cursor_row mdpy) 0) 1)
			(- mvpt-height 1))))
	  (adjust-menu))
	 (else
	  (scroll-menu 'down)))
	)
       ((= key SMG$K_TRM_NEXT_SCREEN)
	(cond
	 (#f
	  (smg$set_cursor_rel mdpy
			      (min (- choices-len
				      (vector-ref (smg$cursor_row mdpy) 0))
				   (- mvpt-height 1)))
	  (adjust-menu))
	 (else
	  (scroll-menu 'up)))
	)
       ((= key SMG$K_TRM_CTRLA)
	(smg$set_cursor_abs mdpy 1)
	(adjust-menu)
	)
       ((= key SMG$K_TRM_CTRLE)
	(smg$set_cursor_abs mdpy choices-len)
	(adjust-menu)
	)
       (else
	(message (string-append "Unknown key: "
				(vector-ref (smg$keycode_to_name key) 1)
				" (" (number->string key) ").  Press RETURN.")
		 'bell 'get-key)
	(smg$set_cursor_abs mdpy))))))
    
