;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;;
;;; boxes.sc -- TeX-like boxes and glue for widget placement.
;;;

;;; $Id: boxes.sc,v 1.1 91/09/15 00:58:19 johani Exp $

(module stoxbox)

(include "../include/scix.sch")
(include "../macros/extsyntax.sc")
(include "../macros/oos.sc")

(define-class (hglue . args)
  (locals
   (width (match-arg 'width 0 args))
   (stretch (match-arg 'stretch 0 args))
   (shrink (match-arg 'shrink 0 args)) )
  (methods
   (width (lambda () width))
   (border-width (lambda () 0))
   (height (lambda () 0))
   (stretch (lambda () stretch))
   (shrink (lambda () shrink)) ))

(define-class (hbox-aux . args)
  (locals
   (contents (match-arg 'contents (list (hfill)) args))
   (width (match-arg 'width
		     (apply + (map (lambda (o) (total-width o)) contents))
		     args))
   (height (match-arg 'height
		       (apply max (map (lambda (o) (total-height o)) contents))
		       args))
   (x (match-arg 'x 0 args))
   (y (match-arg 'y 0 args)) )
  (methods
   (width (lambda () width))
   (border-width (lambda () 0))
   (height (lambda () height))
   (natural-width (lambda ()
		    (apply + (map (lambda (o) (total-width o)) contents))))
   (natural-height (lambda () height))
   (set-width! (lambda (new-width) (set! width new-width)))
   (set-height! (lambda (new-height)
		  (set! height new-height)
		  (for-each (lambda (o)
			      (if (eq? (o 'object-class) 'vbox-aux)
				  (o 'set-height! new-height) ))
			    contents)))
   (move (lambda (new-x new-y)
	   (set! x new-x)
	   (set! y new-y)
	   (me 'stox-arrange) ))
   (stox-arrange
    (lambda ()
      (cond ((> width (me 'natural-width))
	     (let ((stretchability (map (lambda (o)
					  (if (eq? (o 'object-class) 'hglue)
					      (o 'stretch)
					      0))
					contents)))
	       (if (and (not (memq 'inf stretchability))
			(zero? (apply + stretchability)) )
		   (begin
		     (set-cdr! (last-pair stretchability) (list 'inf))
		     (set-cdr! (last-pair contents)
			       (list (hglue 'make 'stretch 'inf)))
		     (scix-msg
		      "HBOX-AUX: Warning: underfull hbox, glue inserted.~%")))
	       (if (memq 'inf stretchability)
		   (let ((inf-stretches
			  (length (remq #f (map (lambda (el)
						  (if (symbol? el) el #f))
						stretchability)))))
		     (let loop ((ls contents) (current-x x))
		       (if (not (null? ls))
			   (if (eq? ((car ls) 'object-class) 'hglue)
			       (loop (cdr ls)
				     (+ current-x
					(if (eq? ((car ls) 'stretch) 'inf)
					    (+ ((car ls) 'width)
					       (inexact->exact
						(round
						 (/ (- width
						       (me 'natural-width))
						    inf-stretches))))
					    ((car ls) 'width) )))
			       (begin
				 ((car ls) 'move current-x y)
				 (loop (cdr ls)
				       (+ current-x
					  (total-width (car ls)) )))))))
		   (let ((stretch-ratio (/ (- width
					      (me 'natural-width))
					   (apply + stretchability))))
		     (let loop ((ls contents) (current-x x))
		       (if (not (null? ls))
			   (if (eq? ((car ls) 'object-class) 'hglue)
			       (loop (cdr ls)
				     (+ current-x ((car ls) 'width)
					(inexact->exact
					 (round
					  (* ((car ls) 'stretch)
					     stretch-ratio) ))))
			       (begin
				 ((car ls) 'move current-x y)
				 (loop (cdr ls)
				       (+ current-x
					  (total-width (car ls)) ))))))))))
	    ((= width (me 'natural-width))
	     (let loop ((ls contents) (current-x x))
	       (if (not (null? ls))
		   (begin
		     (if (not (eq? ((car ls) 'object-class) 'hglue))
			 ((car ls) 'move current-x y) )
		     (loop (cdr ls)
			   (+ current-x (total-width (car ls))) )))))
	    (else
	     (let ((shrinkability (map (lambda (o)
					 (if (eq? (o 'object-class) 'hglue)
					      (o 'shrink)
					      0)) 
				       contents)))
	       (if (and (not (memq 'inf shrinkability))
			(zero? (apply + shrinkability)) )
		   (begin
		     (set-cdr! (last-pair shrinkability) (list 'inf))
		     (set-cdr! (last-pair contents)
			       (list (hglue 'make 'shrink 'inf)))
		     (scix-msg
		      "HBOX-AUX: Warning: overfull hbox, glue inserted.~%")))
	       (if (memq 'inf shrinkability)
		   (let ((inf-shrinks
			  (length (remq #f (map (lambda (el)
						  (if (symbol? el) el #f))
						shrinkability)))))
		     (let loop ((ls contents) (current-x x))
		       (if (not (null? ls))
			   (if (eq? ((car ls) 'object-class) 'hglue)
			       (loop (cdr ls)
				     (+ current-x
					(if (eq? ((car ls) 'shrink) 'inf)
					    (- ((car ls) 'width)
					       (inexact->exact
						(round
						 (/ (- (me 'natural-width)
						       width)
						    inf-shrinks))))
					    ((car ls) 'width) )))
			       (begin
				 ((car ls) 'move current-x y)
				 (loop (cdr ls)
				       (+ current-x
					  (total-width (car ls)))) )))))
		   (let ((shrink-ratio (/ (- (me 'natural-width) width)
					  (apply + shrinkability))))
		     (let loop ((ls contents) (current-x x))
		       (if (not (null? ls))
			   (if (eq? ((car ls) 'object-class) 'hglue)
			       (loop (cdr ls)
				     (+ current-x
					(- ((car ls) 'width)
					   (inexact->exact
					    (round
					     (* ((car ls) 'shrink)
						shrink-ratio) )))))
			       (begin
				 ((car ls) 'move current-x y)
				 (loop (cdr ls)
				       (+ current-x
					  (total-width
					   (car ls)))) )))))))))))))

(define-class (vglue . args)
  (locals
   (height (match-arg 'height 0 args))
   (stretch (match-arg 'stretch 0 args))
   (shrink (match-arg 'shrink 0 args)) )
  (methods
   (width (lambda () 0))
   (border-width (lambda () 0))
   (height (lambda () height))
   (stretch (lambda () stretch))
   (shrink (lambda () shrink)) ))

(define-class (vbox-aux . args)
  (locals
   (contents (match-arg 'contents (list (vfill)) args))
   (width (match-arg 'width
		     (apply max (map (lambda (o) (total-width o)) contents))
		     args))
   (height (match-arg 'height
		      (apply + (map (lambda (o) (total-height o)) contents))
		      args))
   (x (match-arg 'x 0 args))
   (y (match-arg 'y 0 args)) )
  (methods
   (width (lambda () width))
   (border-width (lambda () 0))
   (height (lambda () height))
   (natural-width (lambda () width))
   (natural-height (lambda ()
		     (apply + (map (lambda (o) (total-height o)) contents))))
   (set-width! (lambda (new-width)
		 (set! width new-width)
		 (for-each (lambda (o)
			     (if (eq? (o 'object-class) 'hbox-aux)
				 (o 'set-width! new-width) ))
			   contents)))
   (set-height! (lambda (new-height) (set! height new-height)))
   (move (lambda (new-x new-y)
	   (set! x new-x)
	   (set! y new-y)
	   (me 'stox-arrange) ))
   (stox-arrange
    (lambda ()
      (cond ((> height (me 'natural-height))
	     (let ((stretchability (map (lambda (o)
					  (if (eq? (o 'object-class) 'vglue)
					      (o 'stretch)
					      0))
					contents)))
	       (if (and (not (memq 'inf stretchability))
			(zero? (apply + stretchability)) )
		   (begin
		     (set-cdr! (last-pair stretchability) (list 'inf))
		     (set-cdr! (last-pair contents)
			       (list (vglue 'make 'stretch 'inf)))
		     (scix-msg
		      "VBOX-AUX: Warning: underfull vbox, glue inserted.~%")))
	       (if (memq 'inf stretchability)
		   (let ((inf-stretches
			  (length (remq #f (map (lambda (el)
						  (if (symbol? el) el #f))
						stretchability)))))
		     (let loop ((ls contents) (current-y y))
		       (if (not (null? ls))
			   (if (eq? ((car ls) 'object-class) 'vglue)
			       (loop (cdr ls)
				     (+ current-y
					(if (eq? ((car ls) 'stretch) 'inf)
					    (+ ((car ls) 'height)
					       (inexact->exact
						(round
						 (/ (- height
						       (me 'natural-height))
						    inf-stretches))))
					    ((car ls) 'height) )))
			       (begin
				 ((car ls) 'move x current-y)
				 (loop (cdr ls)
				       (+ current-y
					  (total-height (car ls)))) )))))
		   (let ((stretch-ratio (/ (- height (me 'natural-height))
					   (apply + stretchability))))
		     (let loop ((ls contents) (current-y y))
		       (if (not (null? ls))
			   (if (eq? ((car ls) 'object-class) 'vglue)
			       (loop (cdr ls)
				     (+ current-y ((car ls) 'height)
					(inexact->exact
					 (round
					  (* ((car ls) 'stretch)
					     stretch-ratio) ))))
			       (begin
				 ((car ls) 'move x current-y)
				 (loop (cdr ls)
				       (+ current-y
					  (total-height (car ls)))) ))))))))
	    ((= height (me 'natural-height))
	     (let loop ((ls contents) (current-y y))
	       (if (not (null? ls))
		   (begin
		     (if (not (eq? ((car ls) 'object-class) 'vglue))
			 ((car ls) 'move x current-y) )
		     (loop (cdr ls)
			   (+ current-y
			      (total-height (car ls))) )))))
	    (else
	     (let ((shrinkability (map (lambda (o)
					 (if (eq? (o 'object-class) 'vglue)
					      (o 'shrink)
					      0))
				       contents)))
	       (if (and (not (memq 'inf shrinkability))
			(zero? (apply + shrinkability)) )
		   (begin
		     (set-cdr! (last-pair shrinkability) (list 'inf))
		     (set-cdr! (last-pair contents)
			       (list (vglue 'make 'shrink 'inf)))
		     (scix-msg
		      "VBOX-AUX: Warning: overfull vbox, glue inserted.~%")))
	       (if (memq 'inf shrinkability)
		   (let ((inf-shrinks
			  (length (remq #f (map (lambda (el)
						  (if (symbol? el) el #f))
						shrinkability)))))
		     (let loop ((ls contents) (current-y y))
		       (if (not (null? ls))
			   (if (eq? ((car ls) 'object-class) 'vglue)
			       (loop (cdr ls)
				     (+ current-y
					(if (eq? ((car ls) 'shrink) 'inf)
					    (- ((car ls) 'height)
					       (inexact->exact
						(round
						 (/ (- (me 'natural-height)
						       height)
						    inf-shrinks))))
					    ((car ls) 'height) )))
			       (begin
				 ((car ls) 'move x current-y)
				 (loop (cdr ls)
				       (+ current-y
					  (total-height (car ls)))) )))))
		   (let ((shrink-ratio (/ (- (me 'natural-height) height)
					  (apply + shrinkability))))
		     (let loop ((ls contents) (current-y y))
		       (if (not (null? ls))
			   (if (eq? ((car ls) 'object-class) 'vglue)
			       (loop (cdr ls)
				     (+ current-y
					(- ((car ls) 'height)
					   (inexact->exact
					    (round
					     (* ((car ls) 'shrink)
						shrink-ratio) )))))
			       (begin
				 ((car ls) 'move x current-y)
				 (loop (cdr ls)
				       (+ current-y
					  (total-height
					   (car ls)))) )))))))))))))

(define (hfill)
  (hglue 'make 'stretch 'inf) )

(define (hss)
  (hglue 'make 'stretch 'inf 'shrink 'inf) )

(define (hskip amount)
  (hglue 'make 'width amount) )

(define (vfill)
  (vglue 'make 'stretch 'inf) )

(define (vss)
  (vglue 'make 'stretch 'inf 'shrink 'inf) )

(define (vskip amount)
  (vglue 'make 'height amount) )

(define (hbox . contents)
  (if (null? contents)
      (hbox-aux 'make)
      (hbox-aux 'make 'contents contents) ))

(define (hbox-to width . contents)
  (if (null? contents)
      (hbox-aux 'make 'width width)
      (hbox-aux 'make 'width width 'contents contents) ))

(define (vbox . contents)
  (if (null? contents)
      (vbox-aux 'make)
      (vbox-aux 'make 'contents contents) ))

(define (vbox-to height . contents)
  (if (null? contents)
      (vbox-aux 'make 'height height)
      (vbox-aux 'make 'height height 'contents contents) ))

(define (total-width obj)
  (+ (obj 'width) (* 2 (obj 'border-width))) )

(define (total-height obj)
  (+ (obj 'height) (* 2 (obj 'border-width))) )
