;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;; $Id: requests1.sc,v 1.4 91/09/15 01:13:27 johani Exp $

;;; requests1.sc -- the code that constructs the individual requests, i e here
;;;                 we have the translation from the OO-level to the X protocol
;;;                 level.

(module scixreq1); (top-level))

(include "../include/types.sch")
(include "../include/util.sch")
(include "../include/lowlevel.sch")
(include "../include/msgutil.sch")

(define-external create-event scixce)

(define-constant X-major-version 11)
(define-constant X-minor-version 0)

;;; Request #0: InitConnection (new format)
(define send-initConnection
  (let ((fmt (vector a-card8 a-card8 a-card16 a-card16
		     a-card16 a-card16 a-card16 a-string8 a-string8))
	(dta (vector #f 0 X-major-version X-minor-version #f #f 0 #f #f))
	(namelen #f) (name #f) (datalen #f) (data #f) )
    (lambda (dpy . rest)
      ;; Authorization stuff.
      (set! namelen (authnamelen))
      (set! name (make-string namelen))
      (set! datalen (authdatalen))
      (set! data (make-string datalen))
      ;; Copy the raw C data into scheme strings.
      (let ((c-name (authname)) (c-data (authdata)))
	(let loop ((i 0))
	  (if (< i namelen)
	      (begin
		(string-set! name i (integer->char (c-byte-ref c-name i)))
		(loop (+ i 1)) )))
	(let loop ((i 0))
	  (if (< i datalen)
	      (begin
		(string-set! data i (integer->char (c-byte-ref c-data i)))
		(loop (+ i 1)) ))))
      (if (positive? (string-length name))
	  (scix-msg "xdisplay: Using authorization protocol ~a~%" name)
	  (scix-msg "xdisplay: No authorization mechanism found.~%") )
      (vector-set! dta 0 (byte-order))
      (vector-set! dta 4 namelen)
      (vector-set! dta 5 datalen)
      (vector-set! dta 7 name)
      (vector-set! dta 8 data)
      (dpy 'scix-xas 8 fmt dta rest)
      (dpy 'flush!)			    ; Flush pending connection request
  (let loop ((str (readfromserver dpy)))
    (cond ((null? str)
	   (loop (readfromserver dpy)) )
	  ((or (< (c-input-string-length str) 8)
	       (< (c-input-string-length str)     ; Partial message read.
		  (+ 8 (* 4 (c-shortunsigned-ref
			     (c-input-string-string str) 6)))))
	   (loop (readmorefromserver dpy str)) )
	  ((zero? (get-next-byte! str))	; Check out the initial byte
	   ;; Return a pair instead of a reply to indicate a failed
	   ;; connection attempt to the display object.
	   (list
	    (make-reply `((reason-len   . ,d-card8) ; Connection refused
			  (prot-major   . ,d-card16)
			  (prot-minor   . ,d-card16)
			  (add-data-len . ,d-card16)
			  (reason       . ,(lambda (str dpy)
					     (d-string8 (backtrack 'reason-len)
							str dpy) )))
			str dpy)))	    
	  (else
	   (let* ((visualtype-format `((id    . ,d-visualid)
				       (class . ,(d-const #t d-card8
							'((0 . StaticGray)
							  (1 . GrayScale)
							  (2 . StaticColor)
							  (3 . PseudoColor)
							  (4 . TrueColor)
							  (5 . DirectColor) )))
				       (bits-per-rgb-value . ,d-card8)
				       (colormap-entries   . ,d-card16)
				       (red-mask           . ,d-card32)
				       (green-mask         . ,d-card32)
				       (blue-mask          . ,d-card32)
				       (unused             . ,d-card32) ))
		  
		  (format-format `((depth          . ,d-card8)
				   (bits-per-pixel . ,d-card8)
				   (scanline-pad   . ,d-card8)
				   (unused         . ,(d-unused 5)) ))

		  (depth-format `((depth              . ,d-card8)
				  (unused             . ,d-card8)
				   (nr-of-visualtypes . ,d-card16)
				   (unused            . ,d-card32)
				   (visuals . ,(lambda (str dpy)
					       ((d-list (lambda (str dpy)
							  (make-reply
							   visualtype-format
							   str
							   dpy))
							(backtrack
							 'nr-of-visualtypes) )
						str dpy)))))

		  ;; d-rootwindow a kludge? Yes, definitely. See dtypes.tmpl.
		  (screen-format `((root . ,(d-object d-rootwindow))
				   (default-colormap . ,(d-object d-colormap))
				   (white-pixel      . ,d-card32)
				   (black-pixel      . ,d-card32)
				   (current-input-masks . ,(d-object
							    d-setofevent))
				   (width-in-pixels  . ,d-card16)
				   (height-in-pixels . ,d-card16)
				   (width-in-mm      . ,d-card16)
				   (height-in-mm     . ,d-card16)
				   (min-installed-maps . ,d-card16)
				   (max-installed-maps . ,d-card16)
				   (root-visual      . ,d-visualid)
				   (backing-stores   . ,(d-const #t d-card8 
							     '((0 . Never)
							       (1 . WhenMapped)
							       (2 . Always) )))
				   (save-unders      . ,d-bool)
				   (root-depth       . ,d-card8)
				   (nr-of-depths     . ,d-card8)
				   (allowed-depths   . ,(lambda (str dpy)
							  ((d-list
							    (lambda (str dpy)
							      (make-reply
							       depth-format
							       str dpy))
							    (backtrack
							     'nr-of-depths))
							   str dpy))))) )
						  
	     ;; Connection accepted
	     (make-reply `((unused             . ,d-card8)
			   (prot-major         . ,d-card16)
			   (prot-minor         . ,d-card16)
			   (add-data-len       . ,d-card16)
			   (release-number     . ,d-card32)
			   (resource-id-base   . ,d-card32)
			   (resource-id-mask   . ,d-card32)
			   (motion-buffer-size . ,d-card32)
			   (vendor-len         . ,d-card16)
			   (max-request-len    . ,d-card16)
			   (nr-of-screens      . ,d-card8)
			   (nr-of-pixmap-formats . ,d-card8)
			   (image-byte-order   . ,(d-const #t d-card8
							   '((0 . LSBFirst)
							     (1 . MSBFirst) )))
			   (bitmap-format-bit-order . ,(d-const #t d-card8 
					            '((0 . LeastSignificant)
						      (1 . MostSignificant) )))
			   (bitmap-format-scanline-unit . ,d-card8)
			   (bitmap-format-scanline-pad  . ,d-card8)
			   (min-keycode             . ,d-keycode)
			   (max-keycode             . ,d-keycode)
			   (unused                  . ,d-card32)
			   (vendor                  . ,(lambda (str dpy)
							 (d-string8
							  (backtrack
							   'vendor-len)
							  str dpy) ))
			   (pixmap-formats . ,(lambda (str dpy)
					      ((d-list 
						(lambda (str dpy)
						  (make-reply format-format
							      str
							      dpy))
						(backtrack
						 'nr-of-pixmap-formats))
					       str dpy)))
			   (roots . ,(lambda (str dpy)
				     ((d-list
				       (lambda (str dpy)
					 (make-reply screen-format str dpy))
				       (backtrack 'nr-of-screens))
				      str dpy))))
			 str dpy)) )))) ))

;;; Request #1: CreateWindow (new format)
(define send-CreateWindow
  (let ((fmt (vector a-request a-card8 a-card16 a-window a-window
		     a-int16 a-int16 a-card16 a-card16 a-card16 a-card16
		     a-visualid a-bitmask a-listofvalue))
	(dta (vector 1 #f #f #f #f #f #f #f #f #f #f #f #f #f)))
    (lambda (w data scr . rest)
      (vector-set! dta 1 (w 'depth))
      (vector-set! dta 2 (+ 8 (data 'length)))
      (vector-set! dta 3 w)
      (vector-set! dta 4 (w 'parent))
      (vector-set! dta 5 (w 'x))
      (vector-set! dta 6 (w 'y))
      (vector-set! dta 7 (w 'width))
      (vector-set! dta 8 (w 'height))
      (vector-set! dta 9 (w 'border-width))
      (vector-set! dta 10 (lookup-constant (w 'class) '((CopyFromParent . 0)
							(InputOutput    . 1)
							(InputOnly     . 2) )))
      (vector-set! dta 11 (lookup-constant (w 'visual)
					   '((CopyFromParent . 0))))
      (vector-set! dta 12 data)
      (vector-set! dta 13 data)
      (scr 'scix-xas 13 fmt dta rest))))

;;; Request #2: ChangeWindowAttributes (new format)
(define send-ChangeWindowattributes
  (let ((fmt (vector a-request a-card8 a-card16
		     a-window a-bitmask a-listofvalue))
	(dta (vector 2 0 #f #f #f #f)) )
    (lambda (w data scr . rest)
      (vector-set! dta 2 (+ 3 (data 'length)))
      (vector-set! dta 3 w)
      (vector-set! dta 4 data)
      (vector-set! dta 5 data)
      (scr 'scix-xas 5 fmt dta rest))))

;;; Request #3: GetWindowAttributes (new format)
(define send-GetWindowattributes
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector 3 0 2 #f)) )
    (lambda (w scr . rest)
      (vector-set! dta 3 w)
      (msg-handler 'scix-wait-for-reply
		 (scr 'scix-xas 3 fmt dta rest)
		 `((reply-name            . ,(d-identity 'GetWindowAttributes))
		   (backing-store . ,(d-const #t d-card8 '((0 . NotUseful)
							   (1 . WhenMapped)
							   (2 . Always) )))
		   (seq-nr                . ,d-card16)
		   (unused                . ,(d-unused 4)) ; Reply length = 3
		   (visual                . ,d-visualid)
		   (class . ,(d-const #t d-card16 '((1 . InputOutput)
						    (2 . InputOnly) )))
		   (bit-gravity           . ,d-bitgravity)
		   (win-gravity           . ,d-wingravity)
		   (backing-planes        . ,d-card32)
		   (backing-pixel         . ,d-card32)
		   (save-under            . ,d-bool)
		   (map-is-installed      . ,d-bool)
		   (map-state . ,(d-const #t d-card8 '((0 . Unmapped)
						       (1 . Unviewable)
						       (2 . Viewable) )))
		   (override-redirect     . ,d-bool)
		   (colormap . ,(d-const #f d-colormap '((0 . None))))
		   (all-event-masks       . ,d-setofevent)
		   (your-event-mask       . ,d-setofevent)
		   (do-not-propagate-mask . ,d-setofdeviceevent)
		   (unused                . ,d-card16) )
		 scr) )))

;;; Request #4: DestroyWindow (new format)
(define send-DestroyWindow
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector 4 0 2 #f)) )
    (lambda (w scr . rest)
      (vector-set! dta 3 w)
      (scr 'scix-xas 3 fmt dta rest))))

;;; Request #5: DestroySubWindows (new format)
(define send-DestroySubWindows
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector 5 0 2 #f)) )
    (lambda (w scr . rest)
      (vector-set! dta 3 w)
      (scr 'scix-xas 3 fmt dta rest))))

;;; Request #6: ChangeSaveSet (new format)
(define send-ChangeSaveSet
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector 6 #f 2 #f)) )
    (lambda (w mode scr . rest)
      (vector-set! dta 1 (lookup-constant mode '((Insert . 0)
						 (Delete . 1) )))
      (vector-set! dta 3 w)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #7: ReparentWindow (new format)
(define send-ReparentWindow
  (let ((fmt (vector a-request a-card8 a-card16
		     a-window a-window a-int16 a-int16))
	(dta (vector 7 0 4 #f #f #f #f)) )
    (lambda (w parent scr . rest)
      (vector-set! dta 3 w)
      (vector-set! dta 4 parent)
      (vector-set! dta 5 (w 'x))
      (vector-set! dta 6 (w 'y))
      (scr 'scix-xas 6 fmt dta rest) )))

;;; Request #8: MapWindow (new format)
(define send-MapWindow
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector 8 0 2 #f)) )
    (lambda (w scr . rest)
      (vector-set! dta 3 w)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #9: MapSubWindows (new format)
(define send-MapSubWindows
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector 9 0 2 #f)) )
    (lambda (w scr . rest)
      (vector-set! dta 3 w)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #10: UnMapWindow (new format)
(define send-UnMapWindow
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector 10 0 2 #f)) )
    (lambda (w scr . rest)
      (vector-set! dta 3 w)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #11: UnMapSubWindows (new format)
(define send-UnMapSubWindows
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector 11 0 2 #f)) )
    (lambda (w scr . rest)
      (vector-set! dta 3 w)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #12: ConfigureWindow (new format)
(define send-ConfigureWindow
  (let ((fmt (vector a-request a-card8 a-card16 a-window a-bitmask16
		     a-card16 a-listofvalue))
	(dta (vector 12 0 #f #f #f 0 #f)) )
    (lambda (w data scr . rest)
      (vector-set! dta 2 (+ 3 (data 'length)))	; Length of request
      (vector-set! dta 3 w)
      (vector-set! dta 4 data)
      (vector-set! dta 6 data)
      (scr 'scix-xas 6 fmt dta rest) )))

;;; Request #13: CirculateWindow (new format)
(define send-CirculateWindow
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector 13 #f 2 #f)) )
    (lambda (w direction scr . rest)
      (vector-set! dta 1 (lookup-constant direction
					  '((RaiseLowest  . 0)
					    (LowerHighest . 1) )))
      (vector-set! dta 3 w)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #14: GetGeometry (new format)
(define send-GetGeometry
  (let ((fmt (vector a-request a-card8 a-card16 a-drawable))
	(dta (vector 14 0 2 #f)) )
    (lambda (drawable scr . rest)
      (vector-set! dta 3 drawable)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 3 fmt dta rest)
		   `((reply-name   . ,(d-identity 'GetGeometry))
		     (depth        . ,d-card8)
		     (seq-nr       . ,d-card16)
		     (unused       . ,d-card32) ; Reply length = 0
		     (root         . ,d-window)
		     (x            . ,d-int16)
		     (y            . ,d-int16)
		     (width        . ,d-card16)
		     (height       . ,d-card16)
		     (border-width . ,d-card16)
		     (unused       . ,(d-unused 10)) )
		   scr) )))

;;; Request #15: QueryTree (new format)
(define send-QueryTree
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector 15 0 2 #f)) )
    (lambda (w scr . rest)
      (vector-set! dta 3 w)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 3 fmt dta rest)
		   `((reply-name . ,(d-identity 'QueryTree))
		     (unused     . ,d-card8)
		     (seq-nr     . ,d-card16)
		     (unused     . ,d-card32) ; Reply length not used
		     (root       . ,d-window)
		     (parent     . ,(d-const #f d-window '((0 . None))))
		     (nrofwins   . ,d-card16)
		     (unused     . ,(d-unused 14))
		     (children   . ,(lambda (str dpy)
				      ((d-list d-window (backtrack 'nrofwins) )
				       str dpy))))
		   scr) )))

;;; Request #16: InternAtom (new format)
(define send-InternAtom
  (let ((fmt (vector a-request a-bool a-card16 a-card16 a-card16 a-string8))
	(dta (vector 16 #f #f #f 0 #f)) )
    (lambda (only-if-exists name scr . rest)
      (let ((n (string-length name)))
	(vector-set! dta 1 only-if-exists)
	(vector-set! dta 2 (+ 2 (/ (+ n (pad n)) 4)))
	(vector-set! dta 3 n)
	(vector-set! dta 5 name)
	(msg-handler 'scix-wait-for-reply 
		     (scr 'scix-xas 5 fmt dta rest)
		     `((reply-name . ,(d-identity 'InternAtom))
		       (unused     . ,d-card8)
		       (seq-nr     . ,d-card16)
		       (unused     . ,d-card32) ; Reply length = 0
		       (atom       . ,(d-const #f d-atom '((0 . None))))
		       (unused     . ,(d-unused 20)) )
		     scr) ))))

;;; Request #17: GetAtomName (new format)
(define send-GetAtomName
  (let ((fmt (vector a-request a-card8 a-card16 a-atom))
	(dta (vector 17 0 2 #f)) )
    (lambda (atom scr . rest)
      (vector-set! dta 3 atom)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 3 fmt dta rest)
		   `((reply-name . ,(d-identity 'GetAtomName))
		     (unused     . ,d-card8)
		     (seq-nr     . ,d-card16)
		     (unused     . ,(d-unused 4)) ; Reply length not used
		     (name-len   . ,d-card16)
		     (unused     . ,(d-unused 22))
		     (name       . ,(lambda (str dpy)
				      (d-string8 (backtrack 'name-len)
						 str dpy) )))
		   scr) )))

;;; Request #18: ChangeProperty (new format)
;;; In order to implement ICCCM in SCIX, the following protocol is used:
;;; <type> may be a symbol or a triple (<symbol> <a-type> <n>) where
;;; <a-type> is an assembler type procedure and <n> is the length of the
;;; data in bytes
(define send-ChangeProperty
  (let ((fmt (vector a-request a-card8 a-card16 a-window a-atom a-atom
		     a-card8 a-card8 a-card16 a-card32 a-listofbyte))
	(dta (vector 18 #f #f #f #f #f #f 0 0 #f #f))
	(atombox #f) (type #f) (a-type #f) (n #f) )
    (lambda (w mode prop type format data scr . rest)
      (set! atombox (scr 'atombox))
      (set! type (if (pair? type) (car type) type))
      (set! a-type (if (pair? type) (cadr type) #f))
      (set! n (if (pair? type)
		  (caddr type)
		  (if (pair? data)
		      (length data)
		      (string-length data))))
      (vector-set! dta 1 (lookup-constant mode '((Replace . 0)
						 (Prepend . 1)
						 (Append  . 2) )))
      (vector-set! dta 2 (+ 6 (/ (+ n (pad n)) 4)))
      (vector-set! dta 3 w)
      (vector-set! dta 4 (atombox 'lookup-id prop))
      (vector-set! dta 5 (atombox 'lookup-id type))
      (vector-set! dta 6 format)  ; Format = { 8 | 16 | 32 }
      (vector-set! dta 9 (/ (* 8 n) format))  ; Length of data in fmt units
      (vector-set! dta 10 data)
      (vector-set! fmt 10 (if a-type a-type a-listofbyte))
      (scr 'scix-xas 10 fmt dta rest) )))

;;; Request #19: DeleteProperty (new format)
(define send-DeleteProperty
  (let ((fmt (vector a-request a-card8 a-card16 a-window a-atom))
	(dta (vector 19 0 3 #f #f)) )
    (lambda (w prop scr . rest)
      (vector-set! dta 3 w)
      (vector-set! dta 4 ((scr 'atombox) 'lookup-id prop))
      (scr 'scix-xas 4 fmt dta rest) )))

;;; Request #20: GetProperty (new format)
;;; In order to implement ICCCM in SCIX, the following protocol is used:
;;; <type> may be a symbol or a tuple (<symbol> <proc>) where <proc> is
;;; a procedure of 5 parameters <reply-len>, <type>, <bytes-after>,
;;; <len-in-format-units>, and <default-proc> is the default disassembler type
;;; procedure. The procedure returns a disassembler type procedure.
(define send-GetProperty
  (let ((fmt (vector a-request a-card16 a-bool a-window a-atom a-atom
		     a-card32 a-card32))
	(dta (vector 20 #f 6 #f #f #f #f #f))
	(atombox #f) (req-type #f) (proc #f))
    (lambda (delete w prop type long-offset long-len scr . rest)
      (set! atombox (scr 'atombox))
      (set! req-type (if (pair? type) (car type) type))
      (set! proc (if (pair? type) (cadr type) #f))
      (vector-set! dta 1 delete)
      (vector-set! dta 3 w)
      (vector-set! dta 4 (atombox 'lookup-id prop))
      (vector-set! dta 5 (if (eq? req-type 'AnyProperty)
			     0
			     (atombox 'lookup-id req-type) ))
      (vector-set! dta 6 long-offset)
      (vector-set! dta 7 long-len)
      (msg-handler
       'scix-wait-for-reply
       (scr 'scix-xas 7 fmt dta rest)
       `((reply-name          . ,(d-identity 'GetProperty))
	 (format              . ,d-card8)
	 (seq-nr              . ,d-card16)
	 (reply-len           . ,d-card32)
	 (type                . ,(d-const #f d-atom '((0 . None))))
	 (bytes-after         . ,d-card32)
	 (len-in-format-units . ,d-card32)
	 (unused              . ,(d-unused 12))
	 (value . ,(lambda (str dpy)
		     (let* ((reply-len (backtrack 'reply-len))
			    (type (backtrack 'type))
			    (bytes-after (backtrack 'bytes-after))
			    (len-in-format-units (backtrack
						  'len-in-format-units))
			    (default-proc (d-list d-card8 (* 4 reply-len))))
		       (if (or (not proc) (> bytes-after 0))
			   (default-proc str dpy)
			   ((proc reply-len type bytes-after
				  len-in-format-units default-proc) str dpy)))
		     )))
       scr)) ))

;;; Request #21: ListProperties (new format)
(define send-ListProperties
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector 21 0 2 #f)) )
    (lambda (w scr . rest)
      (vector-set! dta 3 w)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 3 fmt dta rest)
		   `((reply-name  . ,(d-identity 'ListProperties))
		     (unused      . ,d-card8)
		     (seq-nr      . ,d-card16)
		     (unused      . ,d-card32) ; Reply length not used
		     (nr-of-atoms . ,d-card16)
		     (unused      . ,(d-unused 22))
		     (atoms       . ,(lambda (str dpy)
				       ((d-list d-atom
						(backtrack 'nr-of-atoms) )
					str dpy))))
		   scr) )))

;;; Request #22: SetSelectionOwner (new format)
(define send-SetSelectionOwner
  (let ((fmt (vector a-request a-card8 a-card16 a-window-or-none
		     a-atom a-timestamp))
	(dta (vector 22 0 4 #f #f #f)) )
    (lambda (owner selection time scr . rest)
      (vector-set! dta 3 owner)
      (vector-set! dta 4 ((scr 'atombox) 'lookup-id selection))
      (vector-set! dta 5 time)
      (scr 'scix-xas 5 fmt dta rest) )))

;;; Request #23: GetSelectionOwner (new format)
(define send-GetSelectionOwner
  (let ((fmt (vector a-request a-card8 a-card16 a-atom))
	(dta (vector 23 0 2 #f)) )
    (lambda (selection scr . rest)
      (vector-set! dta 3 ((scr 'atombox) 'lookup-id selection))
      (msg-handler 'scix-wait-for-reply 
		   (scr 'scix-xas 3 fmt dta rest)
		   `((reply-name . ,(d-identity 'GetSelectionOwner))
		     (unused     . ,d-card8)
		     (seq-nr     . ,d-card16)
		     (unused     . ,d-card32)
		     (owner      . ,(d-const #f d-window '((0 . None))))
		     (unused     . ,(d-unused 20)) )
		   scr) )))

;;; Request #24: ConvertSelection (new format)
(define send-ConvertSelection
  (let ((fmt (vector a-request a-card8 a-card16 a-window a-atom a-atom
		     a-atom a-timestamp))
	(dta (vector 24 0 6 #f #f #f #f #f))
	(atombox #f) )
    (lambda (requestor selection target prop time scr . rest)
      (set! atombox (scr 'atombox))
      (vector-set! dta 3 requestor)
      (vector-set! dta 4 (atombox 'lookup-id selection))
      (vector-set! dta 5 (atombox 'lookup-id target))
      (vector-set! dta 6 (if (eq? prop 'None) 0 (atombox 'lookup-id prop)))
      (vector-set! dta 7 time)
      (scr 'scix-xas 7 fmt dta rest) )))

;;;; Request #25: SendEvent (new format)
(define send-SendEvent
  (let ((fmt (vector a-request a-bool a-card16 a-window
		     a-setofevent a-intlist))
	(dta (vector 25 #f 11 #f #f #f))
	(win #f)
	(evmask #f)
	(ev #f) )
    (lambda (propagate w event-mask event-name event-data scr . rest)
      (set! win (lookup-constant w '((PointerWindow . 0)
				     (InputFocus . 1) )))
      (set! evmask (event-mask 'mask))
      (set! ev (create-event event-name event-data))
      (vector-set! dta 1 propagate)
      (vector-set! dta 3 win)
      (vector-set! dta 4 evmask)
      (vector-set! dta 5 ev)
      (scr 'scix-xas 5 fmt dta rest) )))

;;;; Request #26: GrabPointer (new format)
(define send-GrabPointer
  (let ((fmt (vector a-request a-bool a-card16 a-window a-setofpointerevent
		     a-card8 a-card8 a-window-or-none a-cursor-or-none
		     a-timestamp))
	(dta (vector 26 #f 6 #f #f #f #f #f #f #f))
	(alist '((Synchronous . 0) (Asynchronous . 1)))
	(pm #f) (km #f) )
    (lambda (owner-events grab-window event-mask pntr-mode keybd-mode
			  confine-to cursor time scr . rest)
      (set! pm (lookup-constant pntr-mode alist))
      (set! km (lookup-constant keybd-mode alist))
      (vector-set! dta 1 owner-events)
      (vector-set! dta 3 grab-window)
      (vector-set! dta 4 (event-mask 'mask))
      (vector-set! dta 5 pm)
      (vector-set! dta 6 km)
      (vector-set! dta 7 confine-to)
      (vector-set! dta 8 cursor)
      (vector-set! dta 9 time)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 9 fmt dta rest)
		   `((reply-name . ,(d-identity 'GrabPointer))
		     (status     . ,(d-const #t d-card8 '((0 . Success)
							  (1 . AlreadyGrabbed)
							  (2 . InvalidTime)
							  (3 . NotViewable)
							  (4 . Frozen) )))
		     (seq-nr     . ,d-card16)
		     (unused     . ,(d-unused 28)) ) ; Reply length = 0
		   scr) )))

;;; Request #27: UngrabPointer (new format)
(define send-UngrabPointer
  (let ((fmt (vector a-request a-card8 a-card16 a-timestamp))
	(dta (vector 27 0 2 #f)) )
    (lambda (time scr . rest)
      (vector-set! dta 3 time)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #28: GrabButton (new format)
(define send-GrabButton
  (let ((fmt (vector a-request a-bool a-card16 a-window a-setofpointerevent
		     a-card8 a-card8 a-window-or-none a-cursor-or-none a-button
		     a-card8 a-setofkeymask))
	(dta (vector 28 #f 6 #f #f #f #f #f #f #f 0 #f))
	(alist '((Synchronous . 0) (Asynchronous . 1))) )
    (lambda (owner-ev grab-win event-mask ptr-mode kbd-mode
		      confine-to cursor button modifiers scr . rest)
      (vector-set! dta 1 owner-ev)
      (vector-set! dta 3 grab-win)
    (vector-set! dta 4 (event-mask 'mask))
    (vector-set! dta 5 (lookup-constant ptr-mode alist))
    (vector-set! dta 6 (lookup-constant kbd-mode alist))
    (vector-set! dta 7 confine-to)
    (vector-set! dta 8 cursor)
    (vector-set! dta 9 (if (eq? button 'AnyButton)
			   0
			   button))
    (vector-set! dta 11 (if (eq? modifiers 'AnyModifier)
			    #x8000
			    (modifiers 'mask) ))
    (scr 'scix-xas 11 fmt dta rest) )))

;;; Request #29: UngrabButton (new format)
(define send-UngrabButton
  (let ((fmt (vector a-request a-button a-card16
		     a-window a-setofkeymask a-card16))
	(dta (vector 29 #f 3 #f #f 0)) )
    (lambda (button grab-win modifiers scr . rest)
      (vector-set! dta 1 (if (eq? button 'AnyButton)
			     0
			     button))
      (vector-set! dta 3 grab-win)
      (vector-set! dta 4 (if (eq? modifiers 'AnyModifier)
			     #x8000
			     (modifiers 'mask) ))
      (scr 'scix-xas 5 fmt dta rest) )))

;;; Request #30: ChangeActivePointerGrab (new format)
(define send-ChangeActivePointerGrab
  (let ((fmt (vector a-request a-card8 a-card16 a-cursor-or-none
		     a-timestamp a-setofpointerevent a-card16))
	(dta (vector 30 0 4 #f #f #f 0)) )
    (lambda (cursor time event-mask scr . rest)
      (vector-set! dta 3 cursor)
      (vector-set! dta 4 time)
      (vector-set! dta 5 (event-mask 'mask))
      (scr 'scix-xas 6 fmt dta rest) )))

;;; Request #31: GrabKeyboard (new format)
(define send-GrabKeyboard
  (let ((fmt (vector a-request a-bool a-card16 a-window a-timestamp
		     a-card8 a-card8 a-card16))
	(dta (vector 31 #f 4 #f #f #f #f 0))
	(alist '((Synchronous . 0) (Asynchronous . 1))) )
    (lambda (owner-events grab-window time pntr-mode keybd-mode scr . rest)
      (vector-set! dta 1 owner-events)
      (vector-set! dta 3 grab-window)
      (vector-set! dta 4 time)
      (vector-set! dta 5 (lookup-constant pntr-mode alist))
      (vector-set! dta 6 (lookup-constant keybd-mode alist))
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 7 fmt dta rest)
		   `((reply-name . ,(d-identity 'GrabKeyboard))
		     (status     . ,(d-const #t d-card8 '((0 . Success)
							  (1 . AlreadyGrabbed)
							  (2 . InvalidTime)
							  (3 . NotViewable)
							  (4 . Frozen) )))
		     (seq-nr     . ,d-card16)
		     (unused     . ,(d-unused 28)) ) ; Reply length = 0
		   scr) )))

;;; Request #32: UngrabKeyboard (new format)
(define send-UngrabKeyboard
  (let ((fmt (vector a-request a-card8 a-card16 a-timestamp))
	(dta (vector 32 0 2 #f)) )
    (lambda (time scr . rest)
      (vector-set! dta 3 time)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #33: GrabKey (new format)
(define send-GrabKey
  (let ((fmt (vector a-request a-bool a-card16 a-window a-setofkeymask
		     a-keycode a-card8 a-card8 a-card8 a-card16))
	(dta (vector 33 #f 4 #f #f #f #f #f 0 0))
	(alist '((Synchronous . 0) (Asynchronous . 1))) )
    (lambda (owner-ev grab-win modifiers key pntr-mode kbd-mode scr . rest)
      (vector-set! dta 1 owner-ev)
      (vector-set! dta 3 grab-win)
      (vector-set! dta 4 (if (eq? modifiers 'AnyModifier)
			     #x8000
			     (modifiers 'mask) ))
      (vector-set! dta 5 (lookup-constant key '((AnyKey . 0))))
      (vector-set! dta 6 (lookup-constant pntr-mode alist))
      (vector-set! dta 7 (lookup-constant kbd-mode alist))
      (scr 'scix-xas 9 fmt dta rest) )))

;;; Request #34: UngrabKey (new format)
(define send-UngrabKey
  (let ((fmt (vector a-request a-keycode a-card16 a-window a-setofkeymask
		     a-card16))
	(dta (vector 34 #f 3 #f #f 0)) )
    (lambda (key grab-win modifiers scr . rest)
      (vector-set! dta 1 (lookup-constant key '((AnyKey . 0))))
      (vector-set! dta 3 grab-win)
      (vector-set! dta 4 (if (eq? modifiers 'AnyModifier)
			      #x8000
			      (modifiers 'mask) ))
      (scr 'scix-xas 5 dta fmt rest) )))

;;; Request #35: AllowEvents (new format)
(define send-AllowEvents
  (let ((fmt (vector a-request a-card8 a-card16 a-timestamp))
	(dta (vector 35 #f 2 #f)) )
    (lambda (mode time scr . rest)
      (vector-set! dta 1 (lookup-constant mode '((AsyncPointer   . 0)
						 (SyncPointer    . 1)
						 (ReplayPointer  . 2)
						 (AsyncKeyboard  . 3)
						 (SyncKeyboard   . 4)
						 (ReplayKeyboard . 5)
						 (AsyncBoth      . 6)
						 (SyncBoth       . 7) )))
      (vector-set! dta 3 time)
      (scr 'scix-xas 3 dta fmt rest) )))

;;; Request #36: GrabServer (new format)
(define send-GrabServer
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 36 0 1)) )
    (lambda (scr . rest)
      (scr 'scix-xas 2 fmt dta rest) )))

;;; Request #37: UngrabServer (new format)
(define send-UngrabServer
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 37 0 1)) )
    (lambda (scr . rest)
      (scr 'scix-xas 2 fmt dta rest) )))

;;; Request #38: QueryPointer (new format)
(define send-QueryPointer
  (let ((fmt (vector a-request a-card8 a-card16 a-window))
	(dta (vector 38 0 2 #f)) )
    (lambda (w scr . rest)
      (vector-set! dta 3 w)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 3 fmt dta rest)
		   `((reply-name  . ,(d-identity 'QueryPointer))
		     (same-screen . ,d-bool)
		     (seq-nr      . ,d-card16)
		     (unused      . ,d-card32) ; Reply length = 0
		     (root        . ,d-window)
		     (child       . ,(d-const #f d-window '((0 . None))))
		     (root-x      . ,d-int16)
		     (root-y      . ,d-int16)
		     (win-x       . ,d-int16)
		     (win-y       . ,d-int16)
		     (mask        . ,d-setofkeybutmask)
		     (unused      . ,(d-unused 6)) )
		   scr) )))

;;;; Request #39: GetMotionEvents (new format)
(define send-GetMotionEvents
  (let ((fmt (vector a-request a-card8 a-card16 a-window
		     a-timestamp a-timestamp))
	(dta (vector 39 0 4 #f #f #f)) )
    (lambda (w start stop scr . rest)
      (vector-set! dta 3 w)
      (vector-set! dta 4 start)
      (vector-set! dta 5 stop)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 5 fmt dta rest)
		   `((reply-name       . ,(d-identity 'GetMotionEvents))
		     (unused           . ,d-card8)
		     (seq-nr           . ,d-card16)
		     (unused           . ,d-card32) ; Reply length not used
		     (nr-of-timecoords . ,d-card32)
		     (unused           . ,(d-unused 20))
		     (events . ,(lambda (str dpy)
				  ((d-list
				    (lambda (str dpy)
				      (let* ((time (d-timestamp str dpy))
					     (x (d-card16 str dpy))
					     (y (d-card16 str dpy)))
					(list time x y) ))
				    (backtrack 'nr-of-timecoords) )
				   str dpy))))
		   scr)) ))

;;; Request #40: TranslateCoordinates (new format)
(define send-TranslateCoordinates
  (let ((fmt (vector a-request a-card8 a-card16
		     a-window a-window a-int16 a-int16))
	(dta (vector 40 0 4 #f #f #f #f)) )
    (lambda (src-win dst-win src-x src-y scr . rest)
      (vector-set! dta 3 src-win)
      (vector-set! dta 4 dst-win)
      (vector-set! dta 5 src-x)
      (vector-set! dta 6 src-y)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 6 fmt dta rest)
		   `((reply-name  . ,(d-identity 'TranslateCoordinates))
		     (same-screen . ,d-bool)
		     (seq-nr      . ,d-card16)
		     (unused      . ,d-card32) ; Reply length = 0
		     (child       . ,(d-const #f d-window '((0 . None))))
		     (dst-x       . ,d-int16)
		     (dst-y       . ,d-int16)
		     (unused      . ,(d-unused 16)) )
		   scr) )))

;;; Request #41: WarpPointer (new format)
(define send-WarpPointer
  (let ((fmt (vector a-request a-card8 a-card16
		     a-window-or-none a-window-or-none a-int16
		     a-int16 a-card16 a-card16 a-int16 a-int16))
	(dta (vector 41 0 6 #f #f #f #f #f #f #f #f)) )
    (lambda (src-win dst-win src-x src-y
		     src-width src-height dst-x dst-y scr . rest)
      (vector-set! dta 3 src-win)
      (vector-set! dta 4 dst-win)
      (vector-set! dta 5 src-x)
      (vector-set! dta 6 src-y)
      (vector-set! dta 7 src-width)
      (vector-set! dta 8 src-height)
      (vector-set! dta 9 dst-x)
      (vector-set! dta 10 dst-y)
      (scr 'scix-xas 10 fmt dta rest) )))

;;;; Request #42: SetInputFocus (new format)
(define send-SetInputFocus
  (let ((fmt (vector a-request a-card8 a-card16 a-window a-timestamp))
	(dta (vector 42 #f 3 #f #f)) )
    (lambda (revert-to focus time scr . rest)
      (vector-set! dta 1 (lookup-constant revert-to '((None        . 0)
						      (PointerRoot . 1)
						      (Parent      . 2) )))
      (vector-set! dta 3 (lookup-constant focus '((None        . 0)
						  (PointerRoot . 1) )))
      (vector-set! dta 4 time)
      (scr 'scix-xas 4 fmt dta rest) )))

;;; Request #43: GetInputFocus (new format)
(define send-GetInputFocus
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 43 0 1)) )
    (lambda (scr . rest)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 2 fmt dta rest)
		   `((reply-name . ,(d-identity 'GetInputFocus))
		     (revert-to  . ,(d-const #t d-card8 '((0 . None)
							  (1 . PointerRoot)
							  (2 . Parent) )))
		     (seq-nr     . ,d-card16)
		     (unused     . ,d-card32) ; Reply length = 0
		     (focus      . ,(d-const #f d-window
					     '((0 . None)
					       (1 . PointerRoot) )))
		     (unused     . ,(d-unused 20)) )
		   scr) )))

;;; Request #44: QueryKeymap (new format)
(define send-QueryKeymap
  (let ((fmt (vector a-request a-card8 a-card16))
	(dta (vector 44 0 1)) )
    (lambda (scr . rest)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 2 fmt dta rest)
		   `((reply-name . ,(d-identity 'QueryKeymap))
		     (unused     . ,d-card8)
		     (seq-nr     . ,d-card16)
		     (unused     . ,d-card32) ; Reply length = 2
		     (keys       . ,(d-list d-card8 32)))
		   scr) )))

;;; Request #45: OpenFont (new format)
(define send-OpenFont
  (let ((fmt (vector a-request a-card8 a-card16
		     a-font a-card16 a-card16 a-string8))
	(dta (vector 45 0 #f #f #f 0 #f))
	(name-len 0) )
    (lambda (font name scr . rest)
      (set! name-len (string-length name))
      (vector-set! dta 2 (+ 3 (/ (+ name-len (pad name-len)) 4)))
      (vector-set! dta 3 font)
      (vector-set! dta 4 name-len)
      (vector-set! dta 6 name)
      (scr 'scix-xas 6 fmt dta rest) )))

;;; Request #46: CloseFont (new format)
(define send-CloseFont
  (let ((fmt (vector a-request a-card8 a-card16 a-font))
	(dta (vector 46 0 2 #f)) )
    (lambda (font scr . rest)
      (vector-set! dta 3 font)
      (scr 'scix-xas 3 fmt dta rest) )))

;;; Request #47: QueryFont (new format)
(define send-QueryFont
  (let ((fmt (vector a-request a-card8 a-card16 a-fontable))
	(dta (vector 47 0 2 #f))
	(charinfo `((left-side-bearing  . ,d-int16)
		    (right-side-bearing . ,d-int16)
		    (character-width    . ,d-int16)
		    (ascent             . ,d-int16)
		    (descent            . ,d-int16)
		    (attributes         . ,d-card16) )))
    (lambda (fontable scr . rest)
      (vector-set! dta 3 fontable)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 3 fmt dta rest)
		   `((reply-name      . ,(d-identity 'QueryFont))
		     (unused          . ,d-card8)
		     (seq-nr          . ,d-card16)
		     (unused          . ,d-card32) ; Reply length not used
		     (min-bounds      . ,(lambda (str dpy)
					   (make-reply charinfo str dpy) ))
		     (unused          . ,d-card32)
		     (max-bounds      . ,(lambda (str dpy)
					   (make-reply charinfo str dpy) ))
		     (unused          . ,d-card32)
		     (min-char-or-byte2 . ,d-card16)
		     (max-char-or-byte2 . ,d-card16)
		     (default-char    . ,d-card16)
		     (nr-of-fontprops . ,d-card16)
		     (draw-direction  . ,(d-const #t d-card8
						  '((0 . LeftToRight)
						    (1 . RightToLeft) )))
		     (min-byte1       . ,d-card8)
		     (max-byte1       . ,d-card8)
		     (all-chars-exist . ,d-bool)
		     (font-ascent     . ,d-int16)
		     (font-descent    . ,d-int16)
		     (nr-of-charinfos . ,d-card32)
		     (properties . ,(lambda (str dpy)
				      ((d-list (lambda (str dpy)
						 (let* ((name (d-atom str dpy))
							(val (d-card32
							      str dpy)) )
						   (cons name val) ))
					       (backtrack 'nr-of-fontprops) )
				       str dpy)))
		     (charinfos . ,(lambda (str dpy)
				     ((d-list (lambda (str dpy)
						(make-reply charinfo str dpy))
					      (backtrack 'nr-of-charinfos) )
				      str dpy))))
		   scr) )))

;;; Request #48: QueryTextExtents (new format)
(define send-QueryTextExtents
  (let ((fmt (vector a-request a-bool a-card16 a-fontable a-string16))
	(dta (vector 48 #f #f #f #f))
	(p #f) )
    (lambda (fontable string scr . rest)
      (set! p (string-length string))
      (vector-set! dta 1 (not (zero? (pad p))))
      (vector-set! dta 2 (+ 2 (/ (+ p (pad p)) 4)))
      (vector-set! dta 3 fontable)
      (vector-set! dta 4 string)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 4 fmt dta rest)
		   `((reply-name      . ,(d-identity 'QueryTextExtents))
		     (draw-direction  . ,(d-const #t d-card8
						  '((0 . LeftToRight)
						    (1 . RightToLeft) )))
		     (seq-nr          . ,d-card16)
		     (unused          . ,d-card32) ; Reply length = 0
		     (font-ascent     . ,d-int16)
		     (font-descent    . ,d-int16)
		     (overall-ascent  . ,d-int16)
		     (overall-descent . ,d-int16)
		     (overall-width   . ,d-int32)
		     (overall-left    . ,d-int32)
		     (overall-right   . ,d-int32)
		     (unused          . ,d-card32) )
		   scr) )))

;;; Request #49: ListFonts (new format)
(define send-ListFonts
  (let ((fmt (vector a-request a-card8 a-card16 a-card16 a-card16 a-string8))
	(dta (vector 49 0 #f #f #f #f))
	(n #f) )
    (lambda (maxnames pattern scr . rest)
      (set! n (string-length pattern))
      (vector-set! dta 2 (+ 2 (/ (+ n (pad n)) 4)))
      (vector-set! dta 3 maxnames)
      (vector-set! dta 4 n)
      (vector-set! dta 5 pattern)
      (msg-handler 'scix-wait-for-reply
		   (scr 'scix-xas 5 fmt dta rest)
		   `((reply-name . ,(d-identity 'ListFonts))
		     (unused     . ,(d-unused 1))
		     (seq-nr     . ,d-card16)
		     (unused     . ,(d-unused 4)) ; Reply length not used
		     (nr-of-strs . ,d-card16)
		     (unused     . ,(d-unused 22))
		     (names . ,(lambda (str dpy)
				 (let loop ((nr (backtrack 'nr-of-strs))
					    (total-length 0) (result '()))
				   (if (zero? nr)
				       (begin
					 ((d-unused (pad total-length))
					  str dpy)
					 result)
				       (let* ((this-str (list (d-str str dpy)))
					      (this-len
					       (+ 1 (string-length
						     (car this-str) ))))
					 (loop (- nr 1)
					       (+ total-length this-len)
					       (append result this-str) )))))))
		   scr) )))

;;;; Request #50: ListFontsWithInfo (new format)
(define send-ListFontsWithInfo
  (let* ((fmt (vector a-request a-card8 a-card16 a-card16 a-card16 a-string8))
	 (dta (vector 50 0 #f #f #f #f))
	 (charinfo-format `((left-side-bearing  . ,d-int16)
			    (right-side-bearing . ,d-int16)
			    (character-width    . ,d-int16)
			    (ascent             . ,d-int16)
			    (descent            . ,d-int16)
			    (attributes         . ,d-card16) ))
	 (reply-format `((reply-name        . ,(d-identity 'ListFontsWithInfo))
			 (length-of-name    . ,d-card8)
			 (seq-nr            . ,d-card16)
			 (unused            . ,d-card32) ; Reply len not used
			 (min-bounds        . ,(lambda (str dpy)
						 (make-reply charinfo-format
							     str dpy) ))
			 (unused            . ,d-card32)
			 (max-bounds        . ,(lambda (str dpy)
						 (make-reply charinfo-format
							     str dpy) ))
			 (unused            . ,d-card32)
			 (min-char-or-byte2 . ,d-card16)
			 (max-char-or-byte2 . ,d-card16)
			 (default-char      . ,d-card16)
			 (nr-of-fontprops   . ,d-card16)
			 (draw-direction    . ,(lambda (scr dpy)
						 ((d-const
						   #t d-card8
						   '((0 . LeftToRight)
						     (1 . RightToLeft) ))
						  scr dpy)))
			 (min-byte1         . ,d-card8)
			 (max-byte1         . ,d-card8)
			 (all-chars-exist   . ,d-bool)
			 (font-ascent       . ,d-int16)
			 (font-descent      . ,d-int16)
			 (replies-hint      . ,d-card32)
			 (properties . ,(lambda (str dpy)
					  ((d-list (lambda (str dpy)
						     (let* ((name
							     (d-atom str dpy))
							    (val (d-card32
								  str dpy)))
						      (cons name val) ))
						   (if (zero?
							(backtrack
							 'length-of-name))
						       0
						       (backtrack
							'nr-of-fontprops)) )
					   str dpy)))
			 (name              . ,(lambda (str dpy)
						 (d-string8 (backtrack
							     'length-of-name)
							    str dpy) ))))
	 (n #f) (seq-nr #f) (reply #f) )
    (lambda (maxnames pattern scr . rest)
      (set! n (string-length pattern))
      (vector-set! dta 2 (+ 2 (/ (+ n (pad n)) 4)))
      (vector-set! dta 3 maxnames)
      (vector-set! dta 4 n)
      (vector-set! dta 5 pattern)
      (set! seq-nr (scr 'scix-xas 5 fmt dta rest))
      (set! reply (msg-handler 'scix-wait-for-reply seq-nr reply-format scr))
      (if (x-error? reply)
	  reply
	  (let loop ((repl reply) (result '()))
	    (if (zero? (repl 'length-of-name))
		result
		(loop (msg-handler 'scix-wait-for-reply
				   seq-nr
				   reply-format
				   scr) 
		      (append result (list repl)) )))))))

;;; Request #51: SetFontPath (new format)
;;; Note: The contraption below has not been tested.
(define send-SetFontPath
(let ((fmt (vector a-request a-card8 a-card16 a-card16 a-card16 a-string8))
      (dta (vector 51 0 #f #f 0 #f))
      (nr-of-strings #f)
      (tot-str-l #f) )
  (lambda (list-of-str scr . rest)
    (set! nr-of-strings (length list-of-str))
    (set! tot-str-l (+ nr-of-strings
		       (apply + (map string-length list-of-str)) ))
    (vector-set! dta 2 (+ 2 (/ (+ tot-str-l (pad tot-str-l)) 4)))
    (vector-set! dta 3 nr-of-strings)
    (vector-set! dta 5 (apply string-append
			      (apply append
				     (map (lambda (s)
					    (list (list->string
						   (list (integer->char
							  (string-length s))))
						  s))
					  list-of-str))))
    (scr 'scix-xas 5 fmt dta rest) )))
