;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;; $Id: requests1.sc,v 1.3 90/06/26 09:53:53 johani Exp $

;;; requests1.sc -- the code that constructs the individual requests, i e here
;;;                 we have the translation from the OO-level to the X protocol
;;;                 level.

(module scixreq1); (top-level))

(include "../include/opcodes.sch")
(include "../include/types.sch")
(include "../include/util.sch")
(include "../include/lowlevel.sch")
(include "../include/msgutil.sch")

(define-external create-event scixce)
(define-external msg-handler scixmh)

(define-constant X-major-version 11)
(define-constant X-minor-version 0)

;;; send-initConnection -- the initial handshaking between the client and the
;;;                        server. 
(define (send-initConnection dpy . rest)
  (let ((prot-major X-major-version)
	(prot-minor X-minor-version)
	(auth-prot-name "")
	(auth-prot-data "") )
    (let ((n (string-length auth-prot-name))
	  (d (string-length auth-prot-name)) )
      (dpy 'scix-xas `((,a-card8   . ,(byte-order))
		       (,a-card8   . 0)			; 1 unused byte
		       (,a-card16  . ,prot-major)
		       (,a-card16  . ,prot-minor)
		       (,a-card16  . ,n)
		       (,a-card16  . ,d)
		       (,a-card16  . 0)			; 2 unused bytes
		       (,a-string8 . ,auth-prot-name)
		       (,a-string8 . ,auth-prot-data) )
	   rest)))
  (dpy 'flush!)				; Flush pending connection request
  (let loop ((str (readfromserver dpy)))
    (cond ((null? str)
	   (loop (readfromserver dpy)) )
	  ((or (< (c-input-string-length str) 8)
	       (< (c-input-string-length str)     ; Partial message read.
		  (+ 8 (* 4 (c-shortunsigned-ref
			     (c-input-string-string str) 6)))))
	   (loop (readmorefromserver dpy str)) )
	  ((zero? (get-next-byte! str))	; Check out the initial byte
	   (make-reply `((reason-len   . ,d-card8) ; Connection refused
			 (prot-major   . ,d-card16)
			 (prot-minor   . ,d-card16)
			 (add-data-len . ,d-card16)
			 (reason       . ,(lambda (str dpy)
					    (d-string8 (backtrack 'reason-len)
						       str dpy) )))
		       str dpy))
	  (else
	   (let* ((visualtype-format `((id    . ,d-visualid)
				       (class . ,(d-const #t d-card8
							'((0 . StaticGray)
							  (1 . GrayScale)
							  (2 . StaticColor)
							  (3 . PseudoColor)
							  (4 . TrueColor)
							  (5 . DirectColor) )))
				       (bits-per-rgb-value . ,d-card8)
				       (colormap-entries   . ,d-card16)
				       (red-mask           . ,d-card32)
				       (green-mask         . ,d-card32)
				       (blue-mask          . ,d-card32)
				       (unused             . ,d-card32) ))
		  
		  (format-format `((depth          . ,d-card8)
				   (bits-per-pixel . ,d-card8)
				   (scanline-pad   . ,d-card8)
				   (unused         . ,(d-unused 5)) ))

		  (depth-format `((depth              . ,d-card8)
				  (unused             . ,d-card8)
				   (nr-of-visualtypes . ,d-card16)
				   (unused            . ,d-card32)
				   (visuals . ,(lambda (str dpy)
					       ((d-list (lambda (str dpy)
							  (make-reply
							   visualtype-format
							   str
							   dpy))
							(backtrack
							 'nr-of-visualtypes) )
						str dpy)))))

		  (screen-format `((root             . ,(d-object d-window))
				   (default-colormap . ,(d-object d-colormap))
				   (white-pixel      . ,d-card32)
				   (black-pixel      . ,d-card32)
				   (current-input-masks . ,(d-object
							    d-setofevent))
				   (width-in-pixels  . ,d-card16)
				   (height-in-pixels . ,d-card16)
				   (width-in-mm      . ,d-card16)
				   (height-in-mm     . ,d-card16)
				   (min-installed-maps . ,d-card16)
				   (max-installed-maps . ,d-card16)
				   (root-visual      . ,d-visualid)
				   (backing-stores   . ,(d-const #t d-card8 
							     '((0 . Never)
							       (1 . WhenMapped)
							       (2 . Always) )))
				   (save-unders      . ,d-bool)
				   (root-depth       . ,d-card8)
				   (nr-of-depths     . ,d-card8)
				   (allowed-depths   . ,(lambda (str dpy)
							  ((d-list
							    (lambda (str dpy)
							      (make-reply
							       depth-format
							       str dpy))
							    (backtrack
							     'nr-of-depths))
							   str dpy))))) )
						  
	     ;; Connection accepted
	     (make-reply `((unused             . ,d-card8)
			   (prot-major         . ,d-card16)
			   (prot-minor         . ,d-card16)
			   (add-data-len       . ,d-card16)
			   (release-number     . ,d-card32)
			   (resource-id-base   . ,d-card32)
			   (resource-id-mask   . ,d-card32)
			   (motion-buffer-size . ,d-card32)
			   (vendor-len         . ,d-card16)
			   (max-request-len    . ,d-card16)
			   (nr-of-screens      . ,d-card8)
			   (nr-of-pixmap-formats . ,d-card8)
			   (image-byte-order   . ,(d-const #t d-card8
							   '((0 . LSBFirst)
							     (1 . MSBFirst) )))
			   (bitmap-format-bit-order . ,(d-const #t d-card8 
					            '((0 . LeastSignificant)
						      (1 . MostSignificant) )))
			   (bitmap-format-scanline-unit . ,d-card8)
			   (bitmap-format-scanline-pad  . ,d-card8)
			   (min-keycode             . ,d-keycode)
			   (max-keycode             . ,d-keycode)
			   (unused                  . ,d-card32)
			   (vendor                  . ,(lambda (str dpy)
							 (d-string8
							  (backtrack
							   'vendor-len)
							  str dpy) ))
			   (pixmap-formats . ,(lambda (str dpy)
					      ((d-list 
						(lambda (str dpy)
						  (make-reply format-format
							      str
							      dpy))
						(backtrack
						 'nr-of-pixmap-formats))
					       str dpy)))
			   (roots . ,(lambda (str dpy)
				     ((d-list
				       (lambda (str dpy)
					 (make-reply screen-format str dpy))
				       (backtrack 'nr-of-screens))
				      str dpy))))
			 str dpy)) ))))

;;; Request #1: CreateWindow
(define (send-CreateWindow w data scr . rest)
  (let ((cl (lookup-constant (w 'class) '((CopyFromParent . 0)
					  (InputOutput    . 1)
					  (InputOnly      . 2) )))
	(vid (lookup-constant (w 'visual) '((CopyFromParent . 0)))) )
    (scr 'scix-xas `((,a-request     . ,CreateWindow)
		     (,a-card8       . ,(w 'depth))
		     (,a-card16      . ,(+ 8 (data 'length)))
		     (,a-window      . ,w)
		     (,a-window      . ,(w 'parent))
		     (,a-int16       . ,(w 'x))
		     (,a-int16       . ,(w 'y))
		     (,a-card16      . ,(w 'width))
		     (,a-card16      . ,(w 'height))
		     (,a-card16      . ,(w 'border-width))
		     (,a-card16      . ,cl)
		     (,a-visualid    . ,vid)
		     (,a-bitmask     . ,data)
		     (,a-listofvalue . ,data) )
	 rest)))

;;; Request #2: ChangeWindowAttributes
(define (send-ChangeWindowattributes w data scr . rest)
  (scr 'scix-xas `((,a-request     . ,ChangeWindowAttributes)
		   (,a-card8       . 0)        		; 1 unused byte
		   (,a-card16      . ,(+ 3 (data 'length)))
		   (,a-window      . ,w)
		   (,a-bitmask     . ,data)
		   (,a-listofvalue . ,data) )
       rest))

;;; Request #3: GetWindowAttributes
(define (send-GetWindowattributes w scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,GetWindowAttributes)
				 (,a-card8   . 0)             ; 1 unused byte
				 (,a-card16  . 2)             ; Request length
				 (,a-window  . ,w) )
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name            . ,(d-identity 'GetWindowAttributes))
		   (backing-store . ,(d-const #t d-card8 '((0 . NotUseful)
							   (1 . WhenMapped)
							   (2 . Always) )))
		   (seq-nr                . ,d-card16)
		   (unused                . ,(d-unused 4)) ; Reply length = 3
		   (visual                . ,d-visualid)
		   (class . ,(d-const #t d-card16 '((1 . InputOutput)
						    (2 . InputOnly) )))
		   (bit-gravity           . ,d-bitgravity)
		   (win-gravity           . ,d-wingravity)
		   (backing-planes        . ,d-card32)
		   (backing-pixel         . ,d-card32)
		   (save-under            . ,d-bool)
		   (map-is-installed      . ,d-bool)
		   (map-state . ,(d-const #t d-card8 '((0 . Unmapped)
						       (1 . Unviewable)
						       (2 . Viewable) )))
		   (override-redirect     . ,d-bool)
		   (colormap . ,(d-const #f d-colormap '((0 . None))))
		   (all-event-masks       . ,d-setofevent)
		   (your-event-mask       . ,d-setofevent)
		   (do-not-propagate-mask . ,d-setofdeviceevent)
		   (unused                . ,d-card16) )
		 scr) ))

;;; Request #4: DestroyWindow
(define (send-DestroyWindow w scr . rest)
  (scr 'scix-xas `((,a-request . ,DestroyWindow )
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 2)			; length of request
		   (,a-window  . ,w) )
       rest))

;;; Request #5: DestroySubWindows
(define (send-DestroySubWindows w scr . rest)
  (scr 'scix-xas `((,a-request . ,DestroySubWindows)
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 2)			; length of request
		   (,a-window  . ,w) )
       rest))

;;; Request #6: ChangeSaveSet
(define (send-ChangeSaveSet w mode scr . rest)
  (scr 'scix-xas `((,a-request . ,ChangeSaveSet)
		   (,a-card8   . ,(lookup-constant mode '((Insert . 0)
							  (Delete . 1) )))
		   (,a-card16  . 2)			; length of request
		   (,a-window  . ,w) )
       rest))

;;; Request #7: ReparentWindow
(define (send-ReparentWindow w parent scr . rest)
  (scr 'scix-xas `((,a-request . ,ReparentWindow )
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 4)			; length of request
		   (,a-window  . ,w)
		   (,a-window  . ,parent)
		   (,a-int16   . ,(w 'x))
		   (,a-int16   . ,(w 'y)) )
       rest))

;;; Request #8: MapWindow
(define (send-MapWindow w scr . rest)
  (scr 'scix-xas `((,a-request . ,MapWindow )
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 2)			; length of request
		   (,a-window  . ,w) )
       rest))

;;; Request #9: MapSubWindows
(define (send-MapSubWindows w scr . rest)
  (scr 'scix-xas `((,a-request . ,MapSubWindows )
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 2)			; length of request
		   (,a-window  . ,w) )
       rest))

;;; Request #10: UnMapWindow
(define (send-UnMapWindow w scr . rest)
  (scr 'scix-xas `((,a-request . ,UnMapWindow)
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 2)			; length of request
		   (,a-window  . ,w) )
       rest))

;;; Request #11: UnMapSubWindows
(define (send-UnMapSubWindows w scr . rest)
  (scr 'scix-xas `((,a-request . ,UnMapSubWindows)
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 2)			; length of request
		   (,a-window  . ,w) )
       rest))

;;; Request #12: ConfigureWindow
(define (send-ConfigureWindow w data scr . rest)
  (scr 'scix-xas `((,a-request     . ,ConfigureWindow)
		   (,a-card8       . 0)		            ; 1 unused byte
		   (,a-card16      . ,(+ 3 (data 'length))) ; length of request
		   (,a-window      . ,w)
		   (,a-bitmask16   . ,data)
		   (,a-card16      . 0)	                ; 2 unused bytes
		   (,a-listofvalue . ,data) )
       rest))

;;; Request #13: CirculateWindow
(define (send-CirculateWindow w direction scr . rest)
  (scr 'scix-xas `((,a-request . ,CirculateWindow)
		   (,a-card8   . ,(lookup-constant direction
						   '((RaiseLowest  . 0)
						     (LowerHighest . 1) )))
		   (,a-card16  . 2)			; length of request
		   (,a-window  . ,w) )
       rest))

;;; Request #14: GetGeometry
(define (send-GetGeometry drawable scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request  . ,GetGeometry)
				 (,a-card8    . 0)     	; 1 unused byte
				 (,a-card16   . 2)     	; length of request
				 (,a-drawable . ,drawable) )
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name   . ,(d-identity 'GetGeometry))
		   (depth        . ,d-card8)
		   (seq-nr       . ,d-card16)
		   (unused       . ,d-card32) ; Reply length = 0
		   (root         . ,d-window)
		   (x            . ,d-int16)
		   (y            . ,d-int16)
		   (width        . ,d-card16)
		   (height       . ,d-card16)
		   (border-width . ,d-card16)
		   (unused       . ,(d-unused 10)) )
		 scr) ))

;;; Request #15: QueryTree
(define (send-QueryTree w scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,QueryTree)
				 (,a-card8   . 0)             ; 1 unused byte
				 (,a-card16  . 2)             ; request length
				 (,a-window  . ,w) )
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name . ,(d-identity 'QueryTree))
		   (unused     . ,d-card8)
		   (seq-nr     . ,d-card16)
		   (unused     . ,d-card32) ; Reply length not used
		   (root       . ,d-window)
		   (parent     . ,(d-const #f d-window '((0 . None))))
		   (nrofwins   . ,d-card16)
		   (unused     . ,(d-unused 14))
		   (children   . ,(lambda (str dpy)
				    ((d-list d-window (backtrack 'nrofwins) )
				     str dpy))))
		 scr) ))

;;; Request #16: InternAtom
(define (send-InternAtom only-if-exists name scr . rest)
  (let* ((n (string-length name))
	 (req-len (+ 2 (/ (+ n (pad n)) 4)))
	 (seq-nr (scr 'scix-xas `((,a-request . ,InternAtom)
				  (,a-bool    . ,only-if-exists)
				  (,a-card16  . ,req-len) ; length of request
				  (,a-card16  . ,n)
				  (,a-card16  . 0)        ; 2 unused bytes
				  (,a-string8 . ,name) )
		      rest)))
    (msg-handler 'scix-wait-for-reply 
		 seq-nr
		 `((reply-name . ,(d-identity 'InternAtom))
		   (unused     . ,d-card8)
		   (seq-nr     . ,d-card16)
		   (unused     . ,d-card32) ; Reply length = 0
		   (atom       . ,(d-const #f d-atom '((0 . None))))
		   (unused     . ,(d-unused 20)) )
		 scr) ))

;;; Request #17: GetAtomName
(define (send-GetAtomName atom scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,GetAtomName)
				 (,a-card8   . 0)	; 1 unused byte
				 (,a-card16  . 2)	; length of request
				 (,a-atom    . ,atom) )
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
 		 `((reply-name . ,(d-identity 'GetAtomName))
 		   (unused     . ,d-card8)
 		   (seq-nr     . ,d-card16)
 		   (unusedn    . ,(d-unused 4)) ; Reply length not used
 		   (name-len   . ,d-card16)
 		   (unused     . ,(d-unused 22))
 		   (name       . ,(lambda (str dpy)
				    (d-string8 (backtrack 'name-len)
					       str dpy) )))
		 scr) ))

;;; Request #18: ChangeProperty
(define (send-ChangeProperty w mode prop type format data scr . rest)
  (let ((m (lookup-constant mode '((Replace . 0)
				   (Prepend . 1)
				   (Append  . 2) )))
	(atombox (scr 'atombox))
	(n (if (pair? data)
	       (length data)
	       (string-length data))) )
    (let ((fu-len (/ (* 8 n) format))
	  (req-len (+ 6 (/ (+ n (pad n)) 4))) )
      (scr 'scix-xas `((,a-request    . ,ChangeProperty)
		       (,a-card8      . ,m)		
		       (,a-card16     . ,req-len)
		       (,a-window     . ,w)
		       (,a-atom       . ,(atombox 'lookup-id prop))
		       (,a-atom       . ,(atombox 'lookup-id type))
		       (,a-card8      . ,format)  ; Format = { 8 | 16 | 32 }
		       (,a-card8      . 0)	  ; 3 unused bytes
		       (,a-card16     . 0)
		       (,a-card32     . ,fu-len)  ; Length of data in fmt units
		       (,a-listofbyte . ,data) )
	   rest))))

;;; Request #19: DeleteProperty
(define (send-DeleteProperty w prop scr . rest)
  (scr 'scix-xas `((,a-request . ,DeleteProperty)
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 3)			; Request length
		   (,a-window  . ,w)
		   (,a-atom    . ,((scr 'atombox) 'lookup-id prop)) )
       rest))

;;; Request #20: GetProperty
(define (send-GetProperty delete w prop type long-offset long-len scr . rest)
  (let* ((atombox (scr 'atombox))
	 (seq-nr (scr 'scix-xas `((,a-request . ,GetProperty)
				  (,a-bool    . ,delete)
				  (,a-card16  . 6)	; Request length
				  (,a-window  . ,w)
				  (,a-atom    . ,(atombox 'lookup-id prop))
				  (,a-atom   . ,(if (eq? type 'AnyPropertyType)
						    0
						    (atombox 'lookup-id type)))
				  (,a-card32 . ,long-offset)
				  (,a-card32 . ,long-len) )
		      rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name          . ,(d-identity 'GetProperty))
		   (format              . ,d-card8)
		   (seq-nr              . ,d-card16)
		   (reply-len           . ,d-card32)
		   (type                . ,(d-const #f d-atom '((0 . None))))
		   (bytes-after         . ,d-card32)
		   (len-in-format-units . ,d-card32)
		   (unused              . ,(d-unused 12))
		   (value               . ,(lambda (str dpy)
					     ((d-list d-card8 
						      (* 4 (backtrack
							    'reply-len)) )
					      str dpy))))
		 scr) ))

;;; Request #21: ListProperties
(define (send-ListProperties w scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,ListProperties)
				 (,a-card8   . 0)	; 1 unused byte
				 (,a-card16  . 2)	; Request length
				 (,a-window  . ,w) )
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name  . ,(d-identity 'ListProperties))
		   (unused      . ,d-card8)
		   (seq-nr      . ,d-card16)
		   (unused      . ,d-card32) ; Reply length not used
		   (nr-of-atoms . ,d-card16)
		   (unused      . ,(d-unused 22))
		   (atoms       . ,(lambda (str dpy)
				     ((d-list d-atom (backtrack 'nr-of-atoms) )
				      str dpy))))
		 scr) ))

;;; Request #22: SetSelectionOwner
(define (send-SetSelectionOwner owner selection time scr . rest)
  (scr 'scix-xas `((,a-request        . ,SetSelectionOwner)
		   (,a-card8          . 0)		; 1 unused byte
		   (,a-card16         . 4)		; Request length
		   (,a-window-or-none . ,owner)
		   (,a-atom           . ,((scr 'atombox) 'lookup-id selection))
		   (,a-timestamp      . ,time) )
       rest))

;;; Request #23: GetSelectionOwner
(define (send-GetSelectionOwner selection scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,GetSelectionOwner)
				 (,a-card8   . 0)	; 1 unused byte
				 (,a-card16  . 2)	; Request length
				 (,a-atom    . ,((scr 'atombox) 'lookup-id
								selection) ))
		     rest)))
    (msg-handler 'scix-wait-for-reply 
		 seq-nr
		 `((reply-name . ,(d-identity 'GetSelectionOwner))
		   (unused     . ,d-card8)
		   (seq-nr     . ,d-card16)
		   (unused     . ,d-card32)
		   (owner      . ,(d-const #f d-window '((0 . None))))
		   (unused     . ,(d-unused 20)) )
		 scr) ))

;;; Request #24: ConvertSelection
(define (send-ConvertSelection requestor selection target prop time scr . rest)
  (let ((atombox (scr 'atombox)))
    (scr 'scix-xas `((,a-request   . ,ConvertSelection)
		     (,a-card8     . 0)			; 1 unused byte
		     (,a-card16    . 6)			; Request length
		     (,a-window    . ,requestor)
		     (,a-atom      . ,(atombox 'lookup-id selection))
		     (,a-atom      . ,(atombox 'lookup-id target))
		     (,a-atom      . ,(if (eq? prop 'None)
					  0
					  (atombox 'lookup-id prop) ))
		     (,a-timestamp . ,time) )
	 rest)))

;;; Request #25: SendEvent
(define (send-SendEvent propagate w event-mask
			event-name event-data scr . rest)
  (let ((win (lookup-constant w '((PointerWindow . 0)
				  (InputFocus . 1) )))
	(evmask (event-mask 'mask))
	(ev (create-event event-name event-data)) )
    (scr 'scix-xas `((,a-request    . ,SendEvent)
		     (,a-bool       . ,propagate)
		     (,a-card16     . 11)		; Request length
		     (,a-window     . ,win)
		     (,a-setofevent . ,evmask)
		     (,a-intlist    . ,ev) )	; Johan: Should use strings...
	 rest)))

;;; Request #26: GrabPointer
(define (send-GrabPointer owner-events grab-window event-mask pntr-mode
			  keybd-mode confine-to cursor time scr . rest)
  (let* ((alist '((Synchronous . 0) (Asynchronous . 1)))
	 (pm (lookup-constant pntr-mode alist))
	 (km (lookup-constant keybd-mode alist))
	 (seq-nr (scr 'scix-xas `((,a-request           . ,GrabPointer)
				  (,a-bool              . ,owner-events)
				  (,a-card16            . 6) ; Request length
				  (,a-window            . ,grab-window)
				  (,a-setofpointerevent . ,(event-mask 'mask))
				  (,a-card8             . ,pm)
				  (,a-card8             . ,km)
				  (,a-window-or-none    . ,confine-to)
				  (,a-cursor-or-none    . ,cursor)
				  (,a-timestamp         . ,time) )
		      rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name . ,(d-identity 'GrabPointer))
		   (status     . ,(d-const #t d-card8 '((0 . Success)
							(1 . AlreadyGrabbed)
							(2 . InvalidTime)
							(3 . NotViewable)
							(4 . Frozen) )))
		   (seq-nr     . ,d-card16)
		   (unused     . ,(d-unused 28)) ) ; Reply length = 0
		 scr) ))

;;; Request #27: UngrabPointer
(define (send-UngrabPointer time scr . rest)
  (scr 'scix-xas `((,a-request   . ,UngrabPointer)
		   (,a-card8     . 0)			; 1 unused byte
		   (,a-card16    . 2)			; Request length
		   (,a-timestamp . ,time) )
       rest))

;;; Request #28: GrabButton
(define (send-GrabButton owner-ev grab-win event-mask ptr-mode kbd-mode
			 confine-to cursor button modifiers scr . rest)
  (let* ((alist '((Synchronous . 0) (Asynchronous . 1)))
	 (mod (if (eq? modifiers 'AnyModifier)
		  #x8000
		  (modifiers 'mask) )))
    (scr 'scix-xas `((,a-request           . ,GrabButton)
		     (,a-bool              . ,owner-ev)
		     (,a-card16            . 6)		; Request length
		     (,a-window            . ,grab-win)
		     (,a-setofpointerevent . ,(event-mask 'mask))
		     (,a-card8             . ,(lookup-constant ptr-mode alist))
		     (,a-card8             . ,(lookup-constant kbd-mode alist))
		     (,a-window-or-none    . ,confine-to)
		     (,a-cursor-or-none    . ,cursor)
		     (,a-button            . ,(if (eq? button 'AnyButton)
						  0
						  button))
		     (,a-card8             . 0)		; 1 unused byte
		     (,a-setofkeymask      . ,mod) )
	 rest)))

;;; Request #29: UngrabButton
(define (send-UngrabButton button grab-win modifiers scr . rest)
  (let ((mod (if (eq? modifiers 'AnyModifier)
		 #x8000
		 (modifiers 'mask) )))
    (scr 'scix-xas `((,a-request      . ,UngrabButton)
		     (,a-button       . ,(if (eq? button 'AnyButton)
					     0
					     button))
		     (,a-card16       . 3)		; Request length
		     (,a-window       . ,grab-win)
		     (,a-setofkeymask . ,mod)
		     (,a-card16       . 0) )  		; 2 unused bytes
	 rest)))

;;; Request #30: ChangeActivePointerGrab
(define (send-ChangeActivePointerGrab cursor time event-mask scr . rest)
  (scr 'scix-xas `((,a-request           . ,ChangeActivePointerGrab)
		   (,a-card8             . 0)		; 1 unused byte
		   (,a-card16            . 4)		; Request length
		   (,a-cursor-or-none    . ,cursor)
		   (,a-timestamp         . ,time)
		   (,a-setofpointerevent . ,(event-mask 'mask))
		   (,a-card16            . 0) )		; 2 unused bytes
       rest))

;;; Request #31: GrabKeyboard
(define (send-GrabKeyboard owner-events grab-window time
			   pntr-mode keybd-mode scr . rest)
  (let* ((alist '((Synchronous . 0) (Asynchronous . 1)))
	 (seq-nr (scr 'scix-xas `((,a-request   . ,GrabKeyboard)
				  (,a-bool      . ,owner-events)
				  (,a-card16    . 4)	; Request length
				  (,a-window    . ,grab-window)
				  (,a-timestamp . ,time)
				  (,a-card8     . ,(lookup-constant pntr-mode
								    alist))
				  (,a-card8     . ,(lookup-constant keybd-mode
								    alist))
				  (,a-card16    . 0) )	; 2 unused bytes
		      rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name . ,(d-identity 'GrabKeyboard))
		   (status     . ,(d-const #t d-card8 '((0 . Success)
							(1 . AlreadyGrabbed)
							(2 . InvalidTime)
							(3 . NotViewable)
							(4 . Frozen) )))
		   (seq-nr     . ,d-card16)
		   (unused     . ,(d-unused 28)) ) ; Reply length = 0
		 scr) ))

;;; Request #32: UngrabKeyboard
(define (send-UngrabKeyboard time scr . rest)
  (scr 'scix-xas `((,a-request   . ,UngrabKeyboard)
		   (,a-card8     . 0)			; 1 unused byte
		   (,a-card16    . 2)			; Request length
		   (,a-timestamp . ,time) )
       rest))

;;; Request #33: GrabKey
(define (send-GrabKey owner-ev grab-win modifiers key
		      pntr-mode kbd-mode scr . rest)
  (let* ((alist '((Synchronous . 0) (Asynchronous . 1)))
	 (mod (if (eq? modifiers 'AnyModifier)
		  #x8000
		  (modifiers 'mask) )))
    (scr 'scix-xas `((,a-request      . ,GrabKey)
		     (,a-bool         . ,owner-ev)
		     (,a-card16       . 4)		; Request length
		     (,a-window       . ,grab-win)
		     (,a-setofkeymask . ,mod)
		     (,a-keycode      . ,(lookup-constant key '((AnyKey . 0))))
		     (,a-card8        . ,(lookup-constant pntr-mode alist))
		     (,a-card8        . ,(lookup-constant kbd-mode alist))
		     (,a-card8        . 0)		; 1 unused byte
		     (,a-card16       . 0) )   		; and 2 more
	 rest)))

;;; Request #34: UngrabKey
(define (send-UngrabKey key grab-win modifiers scr . rest)
  (let ((mod (if (eq? modifiers 'AnyModifier)
		 #x8000
		 (modifiers 'mask) )))
    (scr 'scix-xas `((,a-request      . ,UngrabKey)
		     (,a-keycode      . ,(lookup-constant key '((AnyKey . 0))))
		     (,a-card16       . 3)		; Request length
		     (,a-window       . ,grab-win)
		     (,a-setofkeymask . ,mod)
		     (,a-card16       . 0) )   		; 2 unused bytes
	 rest)))

;;; Request #35: AllowEvents
(define (send-AllowEvents mode time scr . rest)
  (let ((md (lookup-constant mode '((AsyncPointer   . 0)
				    (SyncPointer    . 1)
				    (ReplayPointer  . 2)
				    (AsyncKeyboard  . 3)
				    (SyncKeyboard   . 4)
				    (ReplayKeyboard . 5)
				    (AsyncBoth      . 6)
				    (SyncBoth       . 7) ))))
    (scr 'scix-xas `((,a-request   . ,AllowEvents)
		     (,a-card8     . ,md)
		     (,a-card16    . 2)			; Request length
		     (,a-timestamp . ,time) )
	 rest)))

;;; Request #36: GrabServer
(define (send-GrabServer scr . rest)
  (scr 'scix-xas `((,a-request . ,GrabServer)
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 1) )   		; Request length
       rest))

;;; Request #37: UngrabServer
(define (send-UngrabServer scr . rest)
  (scr 'scix-xas `((,a-request . ,UngrabServer)
		   (,a-card8   . 0)     		; 1 unused byte
		   (,a-card16  . 1) )  	        	; Request length
       rest))

;;; Request #38: QueryPointer
(define (send-QueryPointer w scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,QueryPointer)
				 (,a-card8   . 0)	; 1 unused byte
				 (,a-card16  . 2)	; Request length
				 (,a-window  . ,w) )
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name  . ,(d-identity 'QueryPointer))
		   (same-screen . ,d-bool)
		   (seq-nr      . ,d-card16)
		   (unused      . ,d-card32) ; Reply length = 0
		   (root        . ,d-window)
		   (child       . ,(d-const #f d-window '((0 . None))))
		   (root-x      . ,d-int16)
		   (root-y      . ,d-int16)
		   (win-x       . ,d-int16)
		   (win-y       . ,d-int16)
		   (mask        . ,d-setofkeybutmask)
		   (unused      . ,(d-unused 6)) )
		 scr) ))

;;; Request #39: GetMotionEvents
(define (send-GetMotionEvents w start stop scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request   . ,GetMotionEvents)
				 (,a-card8     . 0)	; 1 unused byte
				 (,a-card16    . 4)	; Request length
				 (,a-window    . ,w)
				 (,a-timestamp . ,start)
				 (,a-timestamp . ,stop) )
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr 
		 `((reply-name       . ,(d-identity 'GetMotionEvents))
		   (unused           . ,d-card8)
		   (seq-nr           . ,d-card16)
		   (unused           . ,d-card32) ; Reply length not used
		   (nr-of-timecoords . ,d-card32)
		   (unused           . ,(d-unused 20))
		   (events . ,(lambda (str dpy)
			      ((d-list (lambda (str dpy)
					 (let* ((time (d-timestamp str dpy))
						(x (d-card16 str dpy))
						(y (d-card16 str dpy)))
					   (list time x y) ))
				       (backtrack 'nr-of-timecoords) )
			       str dpy))))
		 scr)) )

;;; Request #40: TranslateCoordinates
(define (send-TranslateCoordinates src-win dst-win src-x src-y scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,TranslateCoordinates)
				 (,a-card8   . 0)	; 1 unused byte
				 (,a-card16  . 4)	; Request length
				 (,a-window  . ,src-win)
				 (,a-window  . ,dst-win)
				 (,a-int16   . ,src-x)
				 (,a-int16   . ,src-y) )
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name  . ,(d-identity 'TranslateCoordinates))
		   (same-screen . ,d-bool)
		   (seq-nr      . ,d-card16)
		   (unused      . ,d-card32) ; Reply length = 0
		   (child       . ,(d-const #f d-window '((0 . None))))
		   (dst-x       . ,d-int16)
		   (dst-y       . ,d-int16)
		   (unused      . ,(d-unused 16)) )
		 scr) ))

;;; Request #41: WarpPointer
(define (send-WarpPointer src-win dst-win src-x src-y
			  src-width src-height dst-x dst-y scr . rest)
  (scr 'scix-xas `((,a-request        . ,WarpPointer)
		   (,a-card8          . 0)		; 1 unused byte
		   (,a-card16         . 6)		; Request length
		   (,a-window-or-none . ,src-win)
		   (,a-window-or-none . ,dst-win)
		   (,a-int16          . ,src-x)
		   (,a-int16          . ,src-y)
		   (,a-card16         . ,src-width)
		   (,a-card16         . ,src-height)
		   (,a-int16          . ,dst-x)
		   (,a-int16          . ,dst-y) )
       rest))

;;; Request #42: SetInputFocus
(define (send-SetInputFocus revert-to focus time scr . rest)
  (let ((rt (lookup-constant revert-to '((None        . 0)
					 (PointerRoot . 1)
					 (Parent      . 2) )))
	(win (lookup-constant focus '((None        . 0)
				      (PointerRoot . 1) ))))
    (scr 'scix-xas `((,a-request   . ,SetInputFocus)
		     (,a-card8     . ,rt)
		     (,a-card16    . 3)		; Request length
		     (,a-window    . ,win)
		     (,a-timestamp . ,time) )
	 rest)))

;;; Request #43: GetInputFocus
(define (send-GetInputFocus scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,GetInputFocus)
				 (,a-card8   . 0)	 ; 1 unused byte
				 (,a-card16  . 1) )     ; Request length
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr 
		 `((reply-name . ,(d-identity 'GetInputFocus))
		   (revert-to  . ,(d-const #t d-card8 '((0 . None)
							(1 . PointerRoot)
							(2 . Parent) )))
		   (seq-nr     . ,d-card16)
		   (unused     . ,d-card32) ; Reply length = 0
		   (focus      . ,(d-const #f d-window '((0 . None)
							 (1 . PointerRoot) )))
		   (unused     . ,(d-unused 20)) )
		 scr) ))

;;; Request #44: QueryKeymap
(define (send-QueryKeymap scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request . ,QueryKeymap)
				 (,a-card8   . 0)	 ; 1 unused byte
				 (,a-card16  . 1) )     ; Request length
		     rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name . ,(d-identity 'QueryKeymap))
		   (unused     . ,d-card8)
		   (seq-nr     . ,d-card16)
		   (unused     . ,d-card32) ; Reply length = 2
		   (keys       . ,(d-list d-card8 32)))
		 scr) ))

;;; Request #45: OpenFont
(define (send-OpenFont font name scr . rest)
  (let ((name-len (string-length name)))
    (scr 'scix-xas `((,a-request . ,OpenFont)
		     (,a-card8   . 0)			; 1 unused byte
		     (,a-card16  . ,(+ 3 (/ (+ name-len (pad name-len)) 4)))
		     (,a-font    . ,font)
		     (,a-card16  . ,name-len)
		     (,a-card16  . 0)			; 2 unused bytes
		     (,a-string8 . ,name) )
	 rest)))

;;; Request #46: CloseFont
(define (send-CloseFont font scr . rest)
  (scr 'scix-xas `((,a-request . ,CloseFont)
		   (,a-card8   . 0)			; 1 unused byte
		   (,a-card16  . 2)			; Request length
		   (,a-font    . ,font) )
       rest))

;;; Request #47: QueryFont
(define (send-QueryFont fontable scr . rest)
  (let ((seq-nr (scr 'scix-xas `((,a-request  . ,QueryFont)
				 (,a-card8    . 0)	; 1 unused byte
				 (,a-card16   . 2)	; Request length
				 (,a-fontable . ,fontable) )
		     rest))
	(charinfo `((left-side-bearing  . ,d-int16)
		    (right-side-bearing . ,d-int16)
		    (character-width    . ,d-int16)
		    (ascent             . ,d-int16)
		    (descent            . ,d-int16)
		    (attributes         . ,d-card16) )))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name      . ,(d-identity 'QueryFont))
		   (unused          . ,d-card8)
		   (seq-nr          . ,d-card16)
		   (unused          . ,d-card32) ; Reply length not used
		   (min-bounds      . ,(lambda (str dpy)
					 (make-reply charinfo str dpy) ))
		   (unused          . ,d-card32)
		   (max-bounds      . ,(lambda (str dpy)
					 (make-reply charinfo str dpy) ))
		   (unused          . ,d-card32)
		   (min-char-or-byte2 . ,d-card16)
		   (max-char-or-byte2 . ,d-card16)
		   (default-char    . ,d-card16)
		   (nr-of-fontprops . ,d-card16)
		   (draw-direction  . ,(d-const #t d-card8 '((0 . LeftToRight)
							  (1 . RightToLeft) )))
		   (min-byte1       . ,d-card8)
		   (max-byte1       . ,d-card8)
		   (all-chars-exist . ,d-bool)
		   (font-ascent     . ,d-int16)
		   (font-descent    . ,d-int16)
		   (nr-of-charinfos . ,d-card32)
		   (properties . ,(lambda (str dpy)
				  ((d-list (lambda (str dpy)
					     (let* ((name (d-atom str dpy))
						    (val (d-card32
							  str dpy)) )
					       (cons name val) ))
					   (backtrack 'nr-of-fontprops) )
				   str dpy)))
		   (charinfos . ,(lambda (str dpy)
				 ((d-list (lambda (str dpy)
					    (make-reply charinfo str dpy))
					  (backtrack 'nr-of-charinfos) )
				  str dpy))))
		 scr) ))

;;; Request #48: QueryTextExtents
(define (send-QueryTextExtents fontable string scr . rest)
  (let* ((p (string-length string))
	 (odd (if (zero? (pad p)) #f #t))
	 (seq-nr (scr 'scix-xas `((,a-request  . ,QueryTextExtents)
				  (,a-bool     . ,odd)
				  (,a-card16   . ,(+ 2 (/ (+ p (pad p)) 4)))
				  (,a-fontable . ,fontable)
				  (,a-string16 . ,string) )
		      rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name      . ,(d-identity 'QueryTextExtents))
		   (draw-direction  . ,(d-const #t d-card8 '((0 . LeftToRight)
							  (1 . RightToLeft) )))
		   (seq-nr          . ,d-card16)
		   (unused          . ,d-card32) ; Reply length = 0
		   (font-ascent     . ,d-int16)
		   (font-descent    . ,d-int16)
		   (overall-ascent  . ,d-int16)
		   (overall-descent . ,d-int16)
		   (overall-width   . ,d-int32)
		   (overall-left    . ,d-int32)
		   (overall-right   . ,d-int32)
		   (unused          . ,d-card32) )
		 scr) ))

;;; Request #49: ListFonts
(define (send-ListFonts  maxnames pattern scr . rest)
  (let* ((n (string-length pattern))
	 (seq-nr (scr 'scix-xas `((,a-request . ,ListFonts)
				  (,a-card8   . 0)	; 1 unused byte
				  (,a-card16  . ,(+ 2 (/ (+ n (pad n)) 4)))
				  (,a-card16  . ,maxnames)
				  (,a-card16  . ,n)
				  (,a-string8 . ,pattern) )
		      rest)))
    (msg-handler 'scix-wait-for-reply
		 seq-nr
		 `((reply-name . ,(d-identity 'ListFonts))
		   (unused     . ,(d-unused 1))
		   (seq-nr     . ,d-card16)
		   (unused     . ,(d-unused 4)) ; Reply length not used
		   (nr-of-strs . ,d-card16)
		   (unused     . ,(d-unused 22))
		   (names . ,(lambda (str dpy)
			     (let loop ((nr (backtrack 'nr-of-strs))
					(total-length 0) (result '()))
			       (if (zero? nr)
				   (begin
				     ((d-unused (pad total-length)) str dpy)
				     result)
				   (let* ((this-str (list (d-str str dpy)))
					  (this-len (+ 1 (string-length
							  (car this-str) ))))
				     (loop (- nr 1)
					   (+ total-length this-len)
					   (append result this-str) )))))))
		 scr) ))

;;; Request #50: ListFontsWithInfo
(define (send-ListFontsWithInfo maxnames pattern scr . rest)
  (let* ((n (string-length pattern))
	 (seq-nr (scr 'scix-xas `((,a-request . ,ListFontsWithInfo)
				  (,a-card8   . 0)	; 1 unused byte
				  (,a-card16  . ,(+ 2 (/ (+ n (pad n)) 4)))
				  (,a-card16  . ,maxnames)
				  (,a-card16  . ,n)
				  (,a-string8 . ,pattern) )
		      rest))
	 (charinfo-format `((left-side-bearing  . ,d-int16)
			    (right-side-bearing . ,d-int16)
			    (character-width    . ,d-int16)
			    (ascent             . ,d-int16)
			    (descent            . ,d-int16)
			    (attributes         . ,d-card16) ))
	 (reply-format `((reply-name        . ,(d-identity 'ListFontsWithInfo))
			 (length-of-name    . ,d-card8)
			 (seq-nr            . ,d-card16)
			 (unused            . ,d-card32) ; Reply len not used
			 (min-bounds        . ,(lambda (str dpy)
						 (make-reply charinfo-format
							     str dpy) ))
			 (unused            . ,d-card32)
			 (max-bounds        . ,(lambda (str dpy)
						 (make-reply charinfo-format
							     str dpy) ))
			 (unused            . ,d-card32)
			 (min-char-or-byte2 . ,d-card16)
			 (max-char-or-byte2 . ,d-card16)
			 (default-char      . ,d-card16)
			 (nr-of-fontprops   . ,d-card16)
			 (draw-direction    . ,(lambda (scr dpy)
						 ((d-const #t d-card8
							 '((0 . LeftToRight)
							   (1 . RightToLeft) ))
						  scr dpy)))
			 (min-byte1         . ,d-card8)
			 (max-byte1         . ,d-card8)
			 (all-chars-exist   . ,d-bool)
			 (font-ascent       . ,d-int16)
			 (font-descent      . ,d-int16)
			 (replies-hint      . ,d-card32)
			 (properties . ,(lambda (str dpy)
					((d-list (lambda (str dpy)
						   (let* ((name
							   (d-atom str dpy))
							  (val (d-card32
								str dpy)))
						     (cons name val) ))
						 (if (zero?
						      (backtrack
						       'length-of-name))
						     0
						     (backtrack
						      'nr-of-fontprops)) )
					 str dpy)))
			 (name              . ,(lambda (str dpy)
						 (d-string8 (backtrack
							     'length-of-name)
							    str dpy) ))))

	 (reply (msg-handler 'scix-wait-for-reply seq-nr reply-format scr)) )
    (if (x-error? reply)
	reply
	(let loop ((repl reply) (result '()))
	  (if (zero? (repl 'length-of-name))
	      result
	      (loop (msg-handler 'scix-wait-for-reply
				 seq-nr
				 reply-format
				 scr) 
		    (append result (list repl)) ))))))

;;; Request #51: SetFontPath
;;; Note: The contraption below has not been tested.
(define (send-SetFontPath list-of-str scr . rest)
  (let* ((nr-of-strings (length list-of-str))
	 (tot-str-l (+ nr-of-strings
			 (apply + (map string-length list-of-str)) )))
    (scr 'scix-xas `((,a-request . ,SetFontPath)
		     (,a-card8   . 0)			; 1 unused byte
		     (,a-card16  . ,(+ 2 (/ (+ tot-str-l (pad tot-str-l)) 4)))
		     (,a-card16  . ,nr-of-strings)
		     (,a-card16  . 0)			; 2 unused bytes
		     (,a-string8 . ,(apply string-append
					   (apply append
						  (map (lambda (s)
							 (list
							  (list->string
							   (list
							    (integer->char
							     (string-length
							      s))))
							  s))
						       list-of-str)))) )
	 rest)))
