;;;
;;;              Copyright 1990 by Digital Equipment AB, Sweden
;;;
;;;                                  and
;;;
;;;                       Hakan Huss and Johan Ihren
;;;
;;;                           All Rights Reserved
;;;
;;;    Permission to use, copy, modify, and distribute this software and
;;;    its documentation for any purpose and without fee is hereby
;;;    granted, provided that the above copyright notice appear in all
;;;    copies and that both that copyright notice and this permis-
;;;    sion notice appear in supporting documentation, and that the
;;;    names of the copyright holders not be used in advertising in
;;;    publicity pertaining to distribution of the software without
;;;    specific, written prior permission. The copyright holders make no
;;;    representations about the suitability of this software for any
;;;    purpose. It is provided "as is" without express or implied warranty.
;;;
;;;    THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
;;;    THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANT-
;;;    ABILITY AND FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS
;;;    BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
;;;    ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;;;    PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
;;;    TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
;;;    OR PERFORMANCE OF THIS SOFTWARE.
;;;
;;;    Authors: Hakan Huss, KTH and Johan Ihren, KTH
;;;

;;;
;;; Some random thoughts on how to incorporate extensions into SCIX as
;;; applied to the sample SHAPE extension available in X11R4.
;;;
;;; $Id$

(module shapeextension)

(include "../include/opcodes.sch")
(include "../include/types.sch")
(include "../include/util.sch")
(include "../include/msg-hndl.sch")
(include "../include/lowlevel.sch")

(define-external create-event scixce)
(define-external x-protocol-atoms scixglobal) ; Note: Wrong place.

;;; Note1: It is obvious that the current format for the request encoding won't
;;;        work for extensions. It has to be conform for all fields in the
;;;        request so the current special handling of the opcode must go away.
;;; Note2: The 'info-extension method of the display/screen is thought to
;;;        cache the values major-opcode, first-event and first-error, in that
;;;        order, when it is first used. So (car (...)) should return the
;;;        major-opcode.

(define (a-pixmap-or-none p buffer)
  (if (eq? p 'none)
      (a-card32 0 buffer)
      (a-pixmap p buffer) ))

;;; shaped-window is a suggestion for how the Shape extension might be used.
;;; It is not thoroughly tested. Note that a shaped-window inherits an
;;; EXISTING window, rather than creating a new one.

(define-class (shaped-window window &optional id)
  (locals (scr (window 'screen)))
  (inherit window)
  (methods
   (shaperectangles (lambda (op kind order xoff yoff rec-list . rest)
		      (send-ShapeRectangles op kind order me
					    xoff yoff rec-list scr rest) )) 
   (shapemask (lambda (op kind xoff yoff mask . rest)
		(send-ShapeMask op kind me xoff yoff mask scr rest) ))
   (shapecombine (lambda () 'foo))
   (shapeoffset (lambda (dest destkind xoff yoff . rest)
		  (send-ShapeOffset dest xoff yoff scr rest) ))
   (shapequeryextents (lambda () 'foo))
   (shapeselectinput (lambda (enable . rest)
		       (send-ShapeSelectInput me enable scr rest) ))
   (shapeinputselected (lambda rest
			 (send-ShapeInputSelected me scr rest) ))
   (shapegetrectangles (lambda (kind . rest)
			 (send-ShapeGetRectangles me kind scr rest) )) )
  (init
    (me 'scix-denounce-id!)
    ((scr 'scix-id-vector) 'flush-deferred!)
    (me 'scix-announce-id! me) ))

;;; Request Shape-0: ShapeQueryVersion
(define (send-ShapeQueryVersion scr . rest)
  (let ((seq-nr (scr 'scix-xas
		     `((,a-card8  . ,(scr 'extension-major-opcode "SHAPE"))
		       (,a-card8  . ,0)	; ShapeQueryVersion
		       (,a-card16 . 1) )
		     rest)))
    (msg-handler 'wait-for-reply
		 seq-nr
		 `((reply-name    . ,(d-identity 'ShapeQueryVersion))
		   (unused        . ,d-card8)
		   (seq-nr        . ,d-card16)
		   (unused        . ,d-card32) ; Reply length not used
		   (major-version . ,d-card16)
		   (minor-version . ,d-card16)
		   (unused        . ,(d-unused 20)) )
		 scr) ))

;;; Request Shape-1: ShapeRectangles
(define (send-ShapeRectangles operation dstkind ordering
			      dstwin xoff yoff rectangle-list scr . rest)
  (scr 'scix-xas 
       `((,a-card8  . ,(scr 'extension-major-opcode "SHAPE"))
	 (,a-card8  . ,1) ; ShapeRectangles
	 (,a-card16 . ,(+ 4 (* 2 (length rectangle-list))))
	 (,a-card8  . ,(lookup-constant operation '((Set       . 0)
						    (Union     . 1)
						    (Intersect . 2)
						    (Subtract  . 3)
						    (Invert    . 4) )))
	 (,a-card8  . ,(lookup-constant dstkind '((Bounding . 0)
						  (Clip . 1) )))
	 (,a-card8  . ,(lookup-constant ordering '((UnSorted . 0)
						   (YSorted  . 1)
						   (YXSorted . 2)
						   (YXBanded . 3) )))
	 (,a-card8  . 0)		; 1 unused byte
	 (,a-window . ,dstwin)
	 (,a-int16  . ,xoff)
	 (,a-int16  . ,yoff)
	 (,(a-list a-rectangle) . ,rectangle-list) )
       rest))

;;; Request Shape-2: ShapeMask
(define (send-ShapeMask operation dstkind dstwin xoff yoff srcbmap scr . rest)
  (scr 'scix-xas
       `((,a-card8  . ,(scr 'extension-major-opcode "SHAPE"))
	 (,a-card8  . ,2) ; ShapeMask
	 (,a-card16 . 5)
	 (,a-card8  . ,(lookup-constant operation '((Set       . 0)
						    (Union     . 1)
						    (Intersect . 2)
						    (Subtract  . 3)
						    (Invert    . 4) )))
	 (,a-card8  . ,(lookup-constant dstkind '((Bounding . 0)
						  (Clip . 1) )))
	 (,a-card16 . 0)		; 2 unused bytes
	 (,a-window . , dstwin)
	 (,a-int16  . ,xoff)
	 (,a-int16  . ,yoff)
	 (,a-pixmap-or-none . ,srcbmap) )
       rest))

;;; Request Shape-3: ShapeCombine
(define (send-ShapeCombine operation dstkind srckind
			   dstwin xoff yoff srcwin scr . rest)
  (scr 'scix-xas
       `((,a-card8  . ,(scr 'extension-major-opcode "SHAPE"))
	 (,a-card8  . ,3) ; ShapeCombine
	 (,a-card16 . 5)
	 (,a-card8  . ,(lookup-constant operation '((Set       . 0)
						    (Union     . 1)
						    (Intersect . 2)
						    (Subtract  . 3)
						    (Invert    . 4) )))
	 (,a-card8  . ,(lookup-constant dstkind '((Bounding . 0)
						  (Clip . 1) )))
	 (,a-card8  . ,(lookup-constant srckind '((Bounding . 0)
						  (Clip . 1) )))
	 (,a-card8  . 0)		; 1 unused byte
	 (,a-window . ,dstwin)
	 (,a-int16  . ,xoff)
	 (,a-int16  . ,yoff)
	 (,a-window . ,srcwin) )
       rest))

;;; Request Shape-4: ShapeOffset
(define (send-ShapeOffset dstkind dstwin xoff yoff scr . rest)
  (scr 'scix-xas 
       `((,a-card8  . ,(scr 'extension-major-opcode "SHAPE"))
	 (,a-card8  . ,4) ; ShapeOffset
	 (,a-card16 . 4)
	 (,a-card8  . ,(lookup-constant dstkind '((Bounding . 0)
						  (Clip . 1) )))
	 (,a-card8  . 0)		; 3 unused
	 (,a-card16 . 0)		; bytes
	 (,a-window . ,dstwin)
	 (,a-int16  . ,xoff)
	 (,a-int16  . ,yoff) )
       rest))

;;; Request Shape-5: ShapeQueryExtents
(define (send-ShapeQueryExtents win scr . rest)
  (let ((seq-nr (scr 'scix-xas
		     `((,a-card8  . ,(scr 'extension-major-opcode "SHAPE"))
		       (,a-card8  . ,5) ; ShapeQueryExtents
		       (,a-card16 . 2)
		       (,a-window . ,win) )
		     rest)))
    (msg-handler 'wait-for-reply
		 seq-nr
		 `((reply-name . ,(d-identity 'ShapeQueryExtents))
		   (unused                        . ,card8)
		   (seq-nr                        . ,d-card16)
		   (unused                        . ,d-card32)
		   (bounding-shaped               . ,d-bool)
		   (clip-shaped                   . ,d-bool)
		   (unused                        . ,d-card16)
		   (bounding-shape-extents-x      . ,d-int16)
		   (bounding-shape-extents-y      . ,d-int16)
		   (bounding-shape-extents-width  . ,d-card16)
		   (bounding-shape-extents-height . ,d-card16)
		   (clip-shape-extents-x          . ,d-int16)
		   (clip-shape-extents-y          . ,d-int16)
		   (clip-shape-extents-width      . ,d-card16)
		   (clip-shape-extents-height     . ,d-card16)
		   (unused                        . ,d-card32) )
		 scr) ))

;;; Request Shape-6: ShapeSelectInput
(define (send-ShapeSelectInput dstwin enable scr . rest)
  (scr 'scix-xas 
       `((,a-card8  . ,(scr 'extension-major-opcode "SHAPE"))
	 (,a-card8  . ,6) ; ShapeSelectInput
	 (,a-card16 . 3)
	 (,a-window . ,dstwin)
	 (,a-bool   . ,enable)
	 (,a-card8  . 0)		; 3 unused
	 (,a-card16 . 0) )		; bytes
       rest))

;;; Request Shape-7: ShapeInputSelected
(define (send-ShapeInputSelected dstwin scr . rest)
  (let ((seq-nr (scr 'scix-xas
		     `((,a-card8  . ,(scr 'extension-major-opcode "SHAPE"))
		       (,a-card8  . ,7) ; ShapeInputSelected
		       (,a-card16 . 2)
		       (,a-window . ,dstwin))
		     rest)))
    (msg-handler 'wait-for-reply 
		 seq-nr
		 `((reply-name . ,(d-identity 'ShapeInputSelected))
		   (enabled    . ,d-bool)
		   (seq-nr     . ,d-card16)
		   (unused     . ,(d-unused 24)) )
		 scr) ))
    
;;; Request Shape-8: ShapeGetRectangles
(define (send-ShapeGetRectangles win srckind scr . rest)
  (let ((seq-nr (scr 'scix-xas
		     `((,a-card8  . ,(scr 'extension-major-opcode "SHAPE"))
		       (,a-card8  . ,8) ; ShapeGetRectangles
		       (,a-card16 . 3)
		       (,a-window . ,win)
		       (,a-card8  . ,(lookup-constant srckind
						      '((Bounding . 0)
							(Clip . 1) )))
		       (,a-card8  . 0)     ; 3 unused 
		       (,a-card16 . 0) )  ; bytes
		     rest)))
    (msg-handler 'wait-for-reply
		 seq-nr
		 `((reply-name   . ,(d-identity 'ShapeGetRectangles))
		   (ordering     . ,(d-const #t d-card8 '((0 . UnSorted)
							  (1 . YSorted)
							  (2 . YXSorted)
							  (3 . YXBanded) )))
		   (seq-nr       . ,d-card16)
		   (reply-length . ,d-card16)
		   (nrects       . ,d-card32)
		   (unused       . ,(d-unused 20))
		   (rectangles   . ,(lambda (str dpy)
				      ((d-list d-rectangle
					       (backtrack 'nrects))
				       str dpy))) )
		 scr) ))

(define (activate-shape-extension dpy)
  (if (not (dpy 'activate-extension
		"SHAPE"
		`(
;;; Event Shape #0: ShapeNotify
		  ((event-name . ,(d-identity 'ShapeNotify))
		   (shape-kind . ,(d-const #f d-card8 '((0 . Bounding)
							(1 . Clip) )))
		   (seq-nr             . ,d-card16)
		   (window             . ,d-window)
		   (x-value-of-extents . ,d-int16)
		   (y-value-of-extents . ,d-int16)
		   (width-of-extents   . ,d-card16)
		   (height-of-extents  . ,d-card16)
		   (server-time        . ,d-timestamp)
		   (shaped             . ,d-bool)
		   (unused             . ,(d-unused 11)) ))
		'() ; No errors in the shape extension.
		))
      (error 'activate-shape-extension
	     "SHAPE extension not present in server") ))