;;;;
;;;; runtime.scm 1.15
;;;;
;;;; psd -- a portable Scheme debugger, version 1.0
;;;; Copyright (C) 1992 Pertti Kellomaki, pk@cs.tut.fi

;;;; This program is free software; you can redistribute it and/or modify
;;;; it under the terms of the GNU General Public License as published by
;;;; the Free Software Foundation; either version 1, or (at your option)
;;;; any later version.

;;;; This program is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;;;; GNU General Public License for more details.

;;;; You should have received a copy of the GNU General Public License
;;;; along with this program; if not, write to the Free Software
;;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;; See file COPYING in the psd distribution.

;;;; 
;;;; Written by Pertti Kellomaki, pk@cs.tut.fi
;;;;
;;;; This file contains part of the runtime support for psd. The parts
;;;; that have to know about the primitive procedures of the
;;;; implementation are in the file "primitives.scm".

;;;
;;; List of breakpoints (file, line number)
;;;

(define *psd-breakpoints* '())

;;; Some state variables for the runtime. These have to be settable
;;; from the outside, so they are now here. A better solution would be
;;; to make the debugger main loop a closure that would keep track of
;;; these, and change them when requested.

(define *psd-break?* #f)            ; used for stepping thru evaluation
(define *psd-coming-from-line* #f)  ; used for stepping line by line
(define *psd-stepping-by-line* #f)
(define *psd-current-line* #f)      ; used for triggering breakpoints only
(define *psd-hits-this-line* 0)     ; once per line

;;;
;;; Reset the runtime state.
;;;

(define (psd-reset)
  (set! *psd-breakpoints* '())
  (set! *psd-break?* #f)
  (set! *psd-coming-from-line* #f)
  (set! *psd-stepping-by-line* #f)
  (set! *psd-current-line* #f)
  (set! *psd-hits-this-line* 0)
  'ok)

;;;
;;; Set a breakpoint.
;;;

(define psd-set-breakpoint 
  (let ((list list) (member member) (cons cons)
		    (string-append string-append)
		    (number->string number->string))

    (lambda (file line)
      (let ((this (list file line)))
	(if (member this *psd-breakpoints*)
	    #f
	    (set! *psd-breakpoints*
		  (cons this
			*psd-breakpoints*))))
      (string-append "Breakpoint at "
		     file
		     ":"
		     (number->string line)))))

;;;
;;; The debugger command interpreter.
;;;

(define psd-debug 

  ;; just to make sure...
  (let ((+ +) (- -) (< <) (= =) (apply apply) (boolean? boolean?)
	      (caddr caddr) (cadr cadr) (car car) (cdr cdr) (char? char?)
	      (display display) (eq? eq?) (equal? equal?) (for-each for-each)
	      (force-output force-output) (input-port? input-port?) (list list)
	      (map map) (member member) (min min) (newline newline) (not not)
	      (null? null?) (number->string number->string) (number? number?)
	      (output-port? output-port?) (pair? pair?) (procedure? procedure?)
	      (quotient quotient) (read read) (reverse reverse)
	      (string-length string-length) (string? string?) (substring substring)
	      (symbol->string symbol->string) (symbol? symbol?)
	      (vector-length vector-length) (vector-ref vector-ref)
	      (vector? vector?) (write write))


    (lambda (val set context place
		 file-index start-line
		 end-line continuation)
    
;;; qp is taken from slib. I have put it here so that people who do
;;; not use slib can still use psd. I also took out printing the newline.
;;;  -pk-
;;; 
;;; Qp writes its arguments, separated by spaces to
;;; (current-output-port).  Qp compresses printing by substituting
;;; `...'  for substructure it does not have sufficient room to print.
;;; *qp-width* is the largest number of characters that qp uses.  Qp
;;; outputs a newline before returning.

      (define *psd-qp-width* 80)

      (define psd-qp
	(let
	    ((+ +) (- -) (< <) (= =) (apply apply) (boolean? boolean?)
		   (car car) (cdr cdr) (char? char?) (display display) (eq? eq?)
		   (for-each for-each) (input-port? input-port?) (newline newline)
		   (not not) (null? null?) (number->string number->string)
		   (number? number?) (output-port? output-port?)
		   (procedure? procedure?) (string-length string-length)
		   (string? string?) (substring substring)
		   (symbol->string symbol->string) (symbol? symbol?)
		   (vector-length vector-length) (vector-ref vector-ref)
		   (vector? vector?) (write write))
	  (letrec
	      ((num-cdrs
		(lambda (pairs max-cdrs)
		  (cond
		   ((null? pairs) 0)
		   ((< max-cdrs 1) 1)
		   ((pair? pairs) (+ 1 (num-cdrs (cdr pairs) (- max-cdrs 1))))
		   (else 1))))
	 
	       (l-elt-room
		(lambda (room pairs)
		  (quotient room (num-cdrs pairs (quotient room 8)))))

	       (qp-pairs
		(lambda (cdrs room)
		  (cond
		   ((null? cdrs) 0)
		   ((not (pair? cdrs))
		    (display " . ")
		    (+ 3 (qp-obj cdrs (l-elt-room (- room 3) cdrs))))
		   ((< 11 room)
		    (display #\ )
		    ((lambda (used)
		       (+ (qp-pairs (cdr cdrs) (- room used)) used))
		     (+ 1 (qp-obj (car cdrs) (l-elt-room (- room 1) cdrs)))))
		   (else
		    (display " ...") 4))))

	       (v-elt-room
		(lambda (room vleft)
		  (quotient room (min vleft (quotient room 8)))))

	       (qp-vect
		(lambda (vect i room)
		  (cond
		   ((= (vector-length vect) i) 0)
		   ((< 11 room)
		    (display #\ )
		    ((lambda (used)
		       (+ (qp-vect vect (+ i 1) (- room used)) used))
		     (+ 1 (qp-obj (vector-ref vect i)
				  (v-elt-room (- room 1)
					      (- (vector-length vect) i))))))
		   (else
		    (display " ...") 4))))

	       (qp-string
		(lambda (str room)
		  (cond
		   ((< (string-length str) room)
		    (display str)
		    (string-length str))
		   (else
		    (display (substring str 0 (- room 3)))
		    (display "...")
		    room))))

	       (qp-obj
		(lambda (obj room)
		  (cond
		   ((null? obj) (write obj) 2)
		   ((boolean? obj) (write obj) 2)
		   ((char? obj) (write obj) 8)
		   ((number? obj) (qp-string (number->string obj) room))
		   ((string? obj)
		    (display #\")
		    ((lambda (ans) (display #\") ans)
		     (+ 2 (qp-string obj (- room 2)))))
		   ((symbol? obj) (qp-string (symbol->string obj) room))
		   ((input-port? obj) (display "#[input]") 8)
		   ((output-port? obj) (display "#[output]") 9)
		   ((procedure? obj) (display "#[proc]") 7)
		   ((vector? obj)
		    (set! room (- room 3))
		    (display "#(")
		    ((lambda (used) (display #\)) (+ used 3))
		     (cond
		      ((= 0 (vector-length obj)) 0)
		      ((< room 8) (display "...") 3)
		      (else
		       ((lambda (used) (+ (qp-vect obj 1 (- room used)) used))
			(qp-obj (vector-ref obj 0)
				(v-elt-room room (vector-length obj))))))))
		   ((pair? obj) 
		    (set! room (- room 2))
		    (display #\()
		    ((lambda (used) (display #\)) (+ 2 used))
		     (if (< room 8) (begin (display "...") 3)
			 ((lambda (used)
			    (+ (qp-pairs (cdr obj) (- room used)) used))
			  (qp-obj (car obj) (l-elt-room room obj))))))
		   (else (display "#[unknown]") 10)))))

	    (lambda objs
	      (qp-pairs (cdr objs)
			(- *psd-qp-width*
			   (qp-obj (car objs) (l-elt-room *psd-qp-width*
							  objs))))))))




      ;; Convert the file index to a string
      (define file-name (psd-index->path file-index))

      ;; Prompt the user for commands
      (define (prompt)
	(display "psd> ")
	(force-output)
	(read))

      ;; Evaluator for simple procedure calls and set!
      (define (eval form)
	(cond ((and (pair? form)
		    (eq? 'set! (car form)))
	       (let ((value (eval (caddr form))))
		 (set (cadr form) value)
		 value))
	      ((pair? form)
	       (apply (eval (car form))
		      (map eval (cdr form))))
	      ((symbol? form)
	       (val form))
	      (else
	       form)))

      ;; Show the context as file name and a list of procedure names
      (define (show-context)
	(if (null? (context))
	    (display "Top level")
	    (begin
	      (display file-name)
	      (display ":")
	      (display (reverse (context)))))
	(newline))

      ;; Show the position in a format that Emacs can parse
      (define (show-position file line)
	(display psd:control-z)
	(display psd:control-z)
	(display file)
	(display ":")
	(display line)
	(newline))

      ;; Check if there is a breakpoint for this line
      (define (break-here? file line)
	(cond

	 ;; break only if there is a breakpoint for this line, and we
	 ;; have just come from somewhere else
	 ((and (member (list file line) *psd-breakpoints*)
	       (= *psd-hits-this-line* 0)))
	 (else #f)))

      ;; The top level loop. The top level loop returns either false, in
      ;; which case the program is continued, or a list whose contents
      ;; should be returned as the value of the current expression.
      (define (psd-top-level file-name line entering?)
	(show-position file-name line)
	(let loop ((command (prompt)))
	  (case command
	    ((val)
	     (display (val (read)))
	     (newline)
	     (loop (prompt)))
	    ((set!)
	     (let* ((sym (read))
		    (val (read)))
	       (set sym val))
	     (loop (prompt)))
	    ((w)
	     (show-context)
	     (loop (prompt)))
	    ((s)
	     (set! *psd-stepping-by-line* #f)
	     (set! *psd-break?* #t)
	     #f)
	    ((g)
	     (set! *psd-stepping-by-line* #f)
	     (set! *psd-break?* #f)
	     #f)
	    ((n)
	     (set! *psd-stepping-by-line* #t)
	     (set! *psd-break?* #f)
	     (set! *psd-coming-from-line* (list file-name line))
	     #f)
	    ((r)
	     (list (eval (read))))
	    (else
	     (if (pair? command)
		 (begin
		   (write (eval command))
		   (newline)
		   (loop (prompt)))
		 (begin
		   (display "Commands are:")(newline)
		   (display "val sym          give the value of sym")(newline)
		   (display "set! sym val     set the value of sym to val")(newline)
		   (display "g                run until the next break point")(newline)
		   (display "w                give the current context (file name and surrounding procedures)")(newline)
		   (display "s                step one step in the evaluation process")(newline)
		   (display "n                run until evaluation reaches another line")(newline)
		   (display "r expr           return expr as the value of current expression")(newline)
		   (display "                 expr can be a procedure call")(newline)
		   (newline)
		   (display "A list is taken to be a procedure call to be evaluated.")(newline)
		   (newline)
		   (loop (prompt))))))))


      ;; Body of psd-debug. First update the line information.
      (let ((position (list file-name start-line)))
	(if (equal? position
		    *psd-current-line*)
	    (set! *psd-hits-this-line*
		  (+ *psd-hits-this-line* 1))
	    (begin
	      (set! *psd-current-line* position)
	      (set! *psd-hits-this-line* 0))))
      
      (if (or *psd-break?*
	      (break-here? file-name start-line)
	      (and *psd-stepping-by-line*
		   (not (equal? (list file-name start-line)
				*psd-coming-from-line*))))

	  ;; Breakpoint or stepping
	  (begin
	    (psd-qp place)
	    (newline)
	    (let ((top-level-val
		   (psd-top-level file-name start-line #t)))
	      (if top-level-val

		  ;; the user wanted to return this value
		  (car top-level-val)

		  (let ((return-value (continuation)))
		    (if (or *psd-break?*
			    (break-here? file-name end-line)
			    (and *psd-stepping-by-line*
				 (not (equal? (list file-name end-line)
					      *psd-coming-from-line*))))
			(begin
			  (psd-qp place)
			  (display " ==> ")
			  (write return-value)
			  (newline)
			  (let ((top-level-val
				 (psd-top-level file-name end-line #f)))
			    (if top-level-val
				(car top-level-val)
				return-value)))

			return-value)))))

	  ;; Check if we were called from psd-apply with a #f continuation.
	  (if continuation
	      (continuation)
	      (let loop ((val
			  (psd-top-level file-name start-line #t)))
		(if val
		    (car val)
		    (begin
		      (display "This expression can not be evaluated normally.")(newline)
		      (display "You have to specify a return value, if you want to continue execution.")
		      (newline)
		      (loop 
		       (psd-top-level file-name start-line #f))))))))))


;;;
;;; Top level definitions of psd-val, psd-set! and psd-context. Each
;;; time a file is instrumented, two procedures that map the names of
;;; all top level definitions in the file are written into the file
;;; that contains the instrumented procedures. When the file is
;;; loaded, these procedures are added to the lists
;;; psd-global-symbol-accessors and psd-global-symbol-setters. If the
;;; user wants to access a global variable, the procedures in the
;;; appropriate list are called one by one. If none of them has access
;;; to the variable, the user is notified. The accessor procedures
;;; return either a list containing the result or #f (I would much
;;; rather return two values here... sigh!). The setter
;;; procedures return #t or #f.
;;;

(define psd-val
  (let ((null? null?) (display display) (newline newline)
		      (car car) (cdr cdr))
    (lambda (sym)
      (let loop ((procs psd-global-symbol-accessors))
	(if (null? procs)
	    (begin (display "Symbol is not visible to psd.")
		   (newline)
		   #f)
	    (let ((result ((car procs) sym)))
	      (if result
		  (car result)
		  (loop (cdr procs)))))))))

(define psd-set!
  (let ((null? null?) (display display) (newline newline)
		      (car car) (cdr cdr))
    (lambda (sym val)
      (let loop ((procs psd-global-symbol-setters))
	(if (null? procs)
	    (begin (display "Symbol is not visible to psd.")
		   (newline)
		   #f)
	    (if ((car procs) sym val)
		#f
		(loop (cdr procs))))))))

