(herald (assembler n32forms t 90)
        (env tsys (assembler as_open))
        )

;;; ---------------- Jump ops
;;; Names are machine-independent; values are machine-dependent

(define-constant jump-op/j=             0)
(define-constant jump-op/jn=            1)
(define-constant jump-op/uj>            4)
(define-constant jump-op/uj<=           5) 
(define-constant jump-op/j>             6)
(define-constant jump-op/j<=            7)
(define-constant jump-op/uj<           10)
(define-constant jump-op/uj>=          11) 
(define-constant jump-op/j<            12)
(define-constant jump-op/j>=           13)
(define-constant jump-op/jabs          14)
(define-constant jump-op/carry_set      2) 
(define-constant jump-op/carry_cleared  3) 
(define-constant jump-op/flag_set       8) 
(define-constant jump-op/flag_cleared   9) 
                                                                   
(define (n32-jump-op-name jump-op)
  (vref '#(eq ne cs cc hi ls gt le fs fc lo hs lt ge r) jump-op))

(define (n32-reverse-jump jump-op)
  (fixnum-logxor jump-op #b0001))

;;; ---------------- Various symbol format converters

(define n32-convert-reglist
  (let ((r7-first (make-symbolic-set-converter '(R7 R6 R5 R4 R3 R2 R1 R0)))
        (r0-first (make-symbolic-set-converter '(R0 R1 R2 R3 R4 R5 R6 R7))))
    (lambda (rl r0-first?)
      ((if r0-first? r0-first r7-first) rl))))

(define (n32-convert-uwb uwb)
  (fx+ (if (memq 'b uwb) 1 0)
       (cond ((memq 'u uwb) 6)
             (else (if (memq 'w uwb) 2 0)))))

(define (n32-format-i i-size)
  (if (fx= i-size 0) #\B (if (fx= i-size 1) #\W #\D)))

(define (n32-format-f f-size)
  (if (fx= f-size 0) #\L #\F))

(define (n32-mnem port mnem size)
  (let ((len (string-length mnem)))
    (iterate loop ((i 0))
      (cond ((fx= i len) 'done)
            (else
             (let ((c (string-elt mnem i)))
               (cond ((char= c #\i)
                      (write-char port (n32-format-i size)))
                     ((char= c #\f)
                      (write-char port (n32-format-f size)))
                     (else
                      (write-char port c)))
               (loop (fx+ i 1))))))))

;;; ----------------------------------------------------------------
;;; Combinators that converts an fg-conser into a similar fg-conser
;;; except that an additional field is tacked on the end of the fg consed

(define (n32-append-displacement-0 fg-conser)
  (lambda (tag)
    (n32-displacement-appendix (fg-conser) tag)))

(define (n32-append-displacement-1 fg-conser)
  (lambda (a1 tag)
    (n32-displacement-appendix (fg-conser a1) tag)))

(define (n32-append-displacement-5 fg-conser)
  (lambda (a1 a2 a3 a4 a5 tag)
    (n32-displacement-appendix (fg-conser a1 a2 a3 a4 a5) tag)))

(define-fg (n32-displacement-appendix body tag)
  (printer "~g,~g" body tag)
  (local start)
  (fields
   (mark start)
   (subfield body)
   (subfield (n32-minimum-displacement start tag))))

(define-fg (n32-field-appendix body offset length)
  (printer "~g,~s,~s" body offset length)
  (fields
   (subfield body)
   (fixed 3 offset) (fixed 5 (- length 1))))

;;; ----------------------------------------------------------------
;;; The formats themselves

(define-fg (n32-format-0 mnem jop disp)
  (printer "~a~a~a     ~g" mnem (n32-jump-op-name jop)
                                (if (eq? jop 'r) " " "") disp)
  (local start)
  (fields
   (mark start)
   (fixed 4 jop) (1 0 1 0)
   (subfield (n32-minimum-displacement start disp))
   ))

;;; ----------------

(define make-n32-format-1
  (let ((f1-fg (fg-template (n32-format-1 mnem opc)
                 (print (p) (format p "~a" mnem))
                 (fields
                  (fixed 8 opc)
                  ))))
    (lambda (mnem opc)
      (lambda () (f1-fg mnem opc)))))

(define make-n32-format-1rl
  (let ((f1rl-fg (fg-template (n32-format-1rl mnem opc rl r0-first?)
                   (print (p) (format p "~a ~s" mnem rl))
                   (fields
                    (fixed 8 opc)
                    (fixed 8 (n32-convert-reglist rl r0-first?))
                    ))))
    (lambda (mnem opc r0-first?)
      (lambda (rl) (f1rl-fg mnem opc rl r0-first?)))))

(define (make-n32-format-1d mnem opc)
  (n32-append-displacement-0 (make-n32-format-1 mnem opc)))

(define (make-n32-format-1rld mnem opc r0-first?)
  (n32-append-displacement-1 (make-n32-format-1rl mnem opc r0-first?)))

;;; ----------------

(define make-n32-format-2
  (let ((f2-fg (fg-template (n32-format-2 mnem size opc (n32-quick quick) gen)
                 (print (p)
                        (n32-mnem p mnem size)
                        (format p "#~s,~g" quick gen))
                 (local start ext)
                 (set-context-item ext (general (size size) (start start) type))
                 (fields
                  (mark start)
                  (group 
                   (subfield (n32-operand-mode gen))
                   (fixed 4 quick) (fixed 3 opc) (1 1) (fixed 2 size))
                  (subfield (n32-operand-index-byte gen)) 
                  (subfield ext (n32-operand-extension gen))
                  ))))
    (lambda (mnem opc)
      (lambda (size quick gen)
        (f2-fg mnem size opc quick gen)))))

(define (make-n32-format-2d mnem opc)
  (n32-append-displacement-5 (make-n32-format-2 mnem opc)))

;;; ----------------

(define make-n32-format-3
  (let ((f3-fg (fg-template (n32-format-3 mnem size opc gen)
                 (print (p) (n32-mnem p mnem size) (format p "~g" gen))
                 (local start ext)
                 (set-context-item ext (general (size size) (start start) type))
                 (fields
                  (mark start)
                  (group
                   (subfield (n32-operand-mode gen)) 
                   (fixed 3 opc) (0 1 1 1 1 1)
                   (fixed 2 size))
                  (subfield (n32-operand-index-byte gen)) 
                  (subfield ext (n32-operand-extension gen))))))
    (lambda (mnem opc)
      (lambda (size gen)
        (f3-fg mnem size opc gen)))))

(define (make-n32-format-3' mnem opc)
  (let ((conser (make-n32-format-3 mnem opc)))
    (lambda (gen)
      (conser #b11 gen))))

;;; ----------------

(define make-n32-format-4
  (let ((f4-fg
         (fg-template (n32-format-4 mnem size opc src dst)
           (print (p) (n32-mnem p mnem size) (format p "~g,~g" src dst))
           (local start s-ext d-ext)
           (set-context-item s-ext (general (size size) (start start) type))
           (set-context-item d-ext (general (size size) (start start) type))
           (fields
            (mark start)
            (group
             (subfield (n32-operand-mode src))
             (subfield (n32-operand-mode dst))
             (fixed 4 opc)
             (fixed 2 size))
            (subfield (n32-index-bytes src dst))
            (subfield s-ext (n32-operand-extension src)) 
            (subfield d-ext (n32-operand-extension dst)) 
            ))))
    (lambda (mnem opc)
      (lambda (size src dst)
        (f4-fg mnem size opc src dst)))))

(define (make-n32-format-4' mnem opc)
  (let ((conser (make-n32-format-4 mnem opc)))
    (lambda (src dst)
      (conser #b11 src dst))))

;;; ----------------

(define make-n32-format-5
  (let ((f5-fg
         (fg-template (n32-format-5 mnem size opc uwb)
           (print (p) (n32-mnem p mnem size) (format p "~s,~s" ...))
           (fields
            (group (0 0 0 0 0)
                   (fixed 2 uwb)
                   (fixed 6 opc)
                   (fixed 2 size)
                   (0 0 0 0 1 1 1 0))))))
    (lambda (mnem opc)
      (lambda (size uwb)
        (f5-fg mnem size opc uwb)))))

(define (make-n32-format-5t mnem opc)
  (let ((conser (make-n32-format-4 mnem opc)))
    (lambda (uwb)
      (conser #b00 uwb))))

;;; ----------------

(define make-n32-format-6/7
  (let ((f67-fg
         (fg-template (n32-format-6/7 mnem size opc src dst opc-b)
           (print (p) (n32-mnem p mnem size) (format p "~g,~g" src dst))
           (local start s-ext d-ext)
           (set-context-item s-ext (general (size size) (start start) type))
           (set-context-item d-ext (general (size size) (start start) type))
           (fields
            (mark start)
            (group
             (subfield (n32-operand-mode src)) 
             (subfield (n32-operand-mode dst)) 
             (fixed 4 opc) (fixed 2 size) (fixed 8 opc-b))
            (subfield (n32-index-bytes src dst))
            (subfield s-ext (n32-operand-extension src)) 
            (subfield d-ext (n32-operand-extension dst)) 
            ))))
    (lambda (mnem opc opc-b)
      (lambda (size src dst)
        (f67-fg mnem size opc src dst opc-b)))))
        
;;; this is gross and temporary.  Hopefully Norm can do better.

(define make-n32-format-6'
  (let ((f67-fg
         (fg-template (n32-format-6/7 mnem size opc src dst opc-b)
           (print (p) (n32-mnem p mnem size) (format p "~g,~g" src dst))
           (local start s-ext d-ext)
           (set-context-item s-ext (general (size size) (start start) (type 'format-6')))
           (set-context-item d-ext (general (size size) (start start) type))
           (fields
            (mark start)
            (group
             (subfield (n32-operand-mode src)) 
             (subfield (n32-operand-mode dst)) 
             (fixed 4 opc) (fixed 2 size) (fixed 8 opc-b))
            (subfield (n32-index-bytes src dst))
            (subfield s-ext (n32-operand-extension src)) 
            (subfield d-ext (n32-operand-extension dst)) 
            ))))
    (lambda (mnem opc opc-b)
      (lambda (size src dst)
        (f67-fg mnem size opc src dst opc-b)))))

(define (make-n32-format-7d mnem opc)
  (let ((conser (make-n32-format-6/7 mnem opc #xCE)))
    (lambda (size src dst iter)
      (n32-displacement-appendix (conser size src dst) (* (- iter 1) size)))))

(define (make-n32-format-7' mnem opc)
  (let ((conser (make-n32-format-6/7 mnem opc #xCE)))
    (lambda (size src dst offset length)
      (n32-field-appendix (conser size src dst) offset length))))

;;; ----------------

(define make-n32-format-8
  (let ((f8-fg
         (fg-template
          (n32-format-8 mnem size opc-a opc-b (n32-regnum reg) src dst)
          (print (p) (n32-mnem p mnem size) (format p "r~s,~g,~g" reg src dst))
          (local start s-ext d-ext)
          (set-context-item s-ext (general (size size) (start start) type))
          (set-context-item d-ext (general (size size) (start start) type))
          (fields
           (mark start)
           (group   
            (subfield (n32-operand-mode src)) 
            (subfield (n32-operand-mode dst))
            (fixed 3 reg) (fixed 1 opc-a)
            (fixed 2 size)
            (fixed 2 opc-b)
            (1 0 1 1  1 0))
           (subfield (n32-index-bytes src dst))
           (subfield s-ext (n32-operand-extension src)) 
           (subfield d-ext (n32-operand-extension dst)) 
           ))))
    (lambda (mnem opc-a opc-b)
      (lambda (size reg src dst)
        (f8-fg mnem size opc-a opc-b reg src dst)))))

(define (make-n32-format-8a mnem opc-a opc-b reg)
  (let ((conser (make-n32-format-8 mnem opc-a opc-b)))
    (lambda (size src dst)
      (conser size reg src dst))))

(define (make-n32-format-8b mnem opc-a opc-b)
  (let ((conser (make-n32-format-8 mnem opc-a opc-b)))
    (lambda (reg src dst)
      (conser #b11 reg src dst))))

(define (make-n32-format-8d mnem opc-a opc-b)
  (let ((conser (make-n32-format-8 mnem opc-a opc-b)))
    (lambda (size reg src dst displ)
      (n32-displacement-appendix (conser size reg src dst) displ))))

;;; ----------------

#|

(define make-n32-format-9if
  (let ((f9if-fg
         (fg-template (n32-format-9if mnem i-size f-size opc src dst)
           (print (p)
                  (n32-mnem-if p mnem i-size f-size)
                  (format p "~g,~g" src dst))
           (local start s-ext d-ext)
           (set-context-item s-ext (general (size i-size) (start start) type))
           (set-context-item d-ext (general (size f-size) (start start)
                                            (type 'float)))
           (fields
            (mark start)
            (group
             (subfield (n32-operand-mode src))
             (subfield (n32-operand-mode dst))
             (fixed 3 opc)  
             (fixed 1 f-size)
             (fixed 2 i-size)
             (0 0 1 1 1 1 1 0))
            (subfield (n32-index-bytes src dst))
            (subfield s-ext (n32-operand-extension src)) 
            (subfield d-ext (n32-operand-extension dst)) 
            ))))
    (lambda (mnem opc)
      (lambda (i-size f-size src dst)
        (f9if-fg mnem i-size f-size opc src dst)))))

(define make-n32-format-9fi
  (let ((f9fi-fg
         (fg-template (n32-format-9fi mnem i-size f-size opc src dst)
           (print (p)
                  (n32-mnem-fi p mnem f-size i-size)
                  (format p "~g,~g" src dst))
           (local start s-ext d-ext)
           (set-context-item s-ext (general (size f-size) (start start) type
                                            (type 'float)))
           (set-context-item d-ext (general (size i-size) (start start) type))
           (fields
            (mark start)
            (group
             (subfield (n32-operand-mode src))
             (subfield (n32-operand-mode dst))
             (fixed 3 opc)  
             (fixed 1 f-size)
             (fixed 2 i-size)
             (0 0 1 1 1 1 1 0))
            (subfield (n32-index-bytes src dst))
            (subfield s-ext (n32-operand-extension src)) 
            (subfield d-ext (n32-operand-extension dst)) 
            ))))
    (lambda (mnem opc)
      (lambda (i-size f-size src dst)
        (f9fi-fg mnem i-size f-size opc src dst))))

;;; ----------------

(define make-n32-format-11
  (let ((f11-fg
         (fg-template (n32-format-11 mnem size opc src dst)
           (print (p) (n32-mnem p mnem size) (format p "~g,~g" src dst))
           (local start s-ext d-ext)
           (set-context-item s-ext (general (size size) (start start)
                                            (type 'float)))
           (set-context-item d-ext (general (size size) (start start)
                                            (type 'float)))
           (fields
            (mark start)
            (group
             (subfield (n32-operand-mode src)) 
             (subfield (n32-operand-mode dst)) 
             (fixed 4 opc) (0) (fixed 1 size) (1 0 1 1  1 1 1 0))
            (subfield (n32-index-bytes src dst))
            (subfield s-ext (n32-operand-extension src)) 
            (subfield d-ext (n32-operand-extension dst)) 
            ))))
    (lambda (mnem opc)
      (lambda (size src dst)
        (f11-fg mnem size opc src dst)))))

|#

;;; ---------------- Displacements for branches

;;; If the tag is an immediate, then the displacement is its value.
;;; If the tag is the null fg, then make the whole displacment field null. ?
;;; otherwise, the tag should be an IB

(define (n32-minimum-displacement start tag)
  (cond ((eq? tag n32-null-fg) n32-null-fg)
;;;;        ((fixnum? tag) (n32-displacement tag))
        ((n32-imm? tag)
         (n32-displacement (fg-argref (n32-operand-extension tag) 0)))
        (else
         (n32-minimum-displacement-fg start tag))))

(define-fg (n32-minimum-displacement-fg start tag)
  (print (p) (format p "displacement to ~s" tag))
  (local displ width)
  (fields
   (variable (disp start tag) 
             (n32-choose-displacement (width 8) displ)
             (make-n32-displacement width displ))))
             
(define (make-n32-displacement width displ)
  (cond ((fx= width 8)  (7-bit-displacement (fixnum-ashr displ 3)))
        ((fx= width 16) (14-bit-displacement (fixnum-ashr displ 3)))
        ((fx= width 32) (30-bit-displacement (fixnum-ashr displ 3)))))

(define (n32-choose-displacement current-width displ)
  (cond ((fx< displ 0)
         (cond ((7bit-in-bits? displ) (return 8 displ))
               ((14bit-in-bits? displ) (return 16 displ))
               ((30bit-in-bits? displ) (return 32 displ))
               (else (error " displacement bigger than 30 bits"))))
        (else
         (let ((base-displ (fx- displ current-width)))
           (cond ((7bit-in-bits? (fx+ base-displ 8))
                  (return 8 (fx+ base-displ 8)))
                 ((14bit-in-bits? (fx+ base-displ 16))
                  (return 16 (fx+ base-displ 16)))
                 ((30bit-in-bits? (fx+ base-displ 32))
                  (return 32 (fx+ base-displ 32)))
                 (else (error " displacement bigger than 30 bits")))))))

;;; ----------------

(set (machine-cond-branch n32)
     (lambda (jop tag)
       ((machine-operation n32 bcc) jop tag)))

(set (machine-uncond-branch n32)
     (lambda (tag) ((machine-operation n32 bcc) jump-op/jabs tag)))

(set (machine-reverse-jump n32) n32-reverse-jump)
