/* gsx.c, Tool to examine Gambit Scheme event logs on an X11 display or Mac */

#ifdef THINK_C
#define MAC
#else
#define X11
#endif


/*--------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>

#ifdef X11
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#endif

#define WHITE 0
#define BLACK 1
#define OTHER 2

#define CENTER 0
#define LEFT   1
#define RIGHT  2

#define MAX_NB_COLORS 18
#define INVERT (MAX_NB_COLORS-1)

typedef struct {
  char *name;
  int w, h;
  FILE *ps_file;
#ifdef X11
  Window win;
  XFontStruct *font;
  GC col[MAX_NB_COLORS];
#endif
#ifdef MAC
  WindowPtr win;
#endif
  void (*redraw)();
  void (*point)();
  void (*click)();
  int (*key)();
  void (*resize)();
  } *window;

int graph_display = 1;

int graph_black_and_white = 0;

int graph_screen_width = 1024, graph_screen_height = 768;


/*--------------------------------------------------------------------------*/

#ifdef X11

char *colors[] =
{ "White",
  "Black",
  "LightSlateBlue",
  "Lavender",
  "Pink",
  "Black",
  "PaleGreen",
  "Gold",
  "Tomato",
  "purple",
  "gray",
  "brown",
  "yellow",
  "cornsilk",
  "magenta",
  "cyan",
  "ivory"
};

#define NB_COLORS (sizeof(colors)/sizeof(char *))

#define STIPPLE_W 8
#define STIPPLE_H 8

char stipple[][STIPPLE_H] = {
{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff },
{ 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
{ 0x00, 0x80, 0x00, 0x08, 0x00, 0x80, 0x00, 0x08 },
{ 0x11, 0x22, 0x44, 0x88, 0x11, 0x22, 0x44, 0x88 },
{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff },
{ 0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22 },
{ 0x00, 0xaa, 0x00, 0xaa, 0x00, 0xaa, 0x00, 0xaa },
{ 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55 },
{ 0xee, 0xdd, 0xbb, 0x77, 0xee, 0xdd, 0xbb, 0x77 },
{ 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77 },
{ 0xdd, 0xff, 0x77, 0xff, 0xdd, 0xff, 0x77, 0xff },
{ 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff },
{ 0x80, 0x10, 0x02, 0x20, 0x01, 0x08, 0x40, 0x04 },
{ 0x88, 0xff, 0x88, 0x88, 0x88, 0xff, 0x88, 0x88 },
{ 0xb1, 0x30, 0x03, 0x1b, 0xd8, 0xc0, 0x0c, 0x8d },
{ 0x00, 0x80, 0x40, 0x20, 0x00, 0x02, 0x04, 0x08 }
};


Display *display;
int screen;


void g_init()
{ /* connect to X server */

  if ( (display=XOpenDisplay(NULL)) == NULL )
  { fprintf( stderr, "can't connect to X server %s\n",
                     XDisplayName(NULL) );
    exit( -1 );
  }

  /* get screen size */

  screen = DefaultScreen( display );

  graph_screen_width  = DisplayWidth( display, screen );
  graph_screen_height = DisplayHeight( display, screen );
}


void g_clear( win )
window win;
{ XClearWindow( display, win->win );
}


void g_box( win, color, x, y, w, h )
window win;
int color;
int x, y, w, h;
{ XFillRectangle( display, win->win, win->col[color], x, win->h - (y+h), w, h );
}


void g_text( win, color, where, x, y, str )
window win;
int color;
int where;
int x, y;
char *str;
{ int len, str_width, y_pos, x_pos;

  len = strlen( str );
  str_width = XTextWidth( win->font, str, len );
  y_pos = win->h - (y - win->font->max_bounds.ascent/2);

  switch (where)
  { case CENTER: x_pos = x - str_width/2; break;
    case LEFT:   x_pos = x;               break;
    case RIGHT:  x_pos = x - str_width;   break;
  }

  XDrawString( display, win->win, win->col[color], x_pos, y_pos, str, len );
}


window g_window( name, width, height, redraw, point, click, key, resize )
char *name;
int width, height;
void (*redraw)();
void (*point)();
void (*click)();
int (*key)();
void (*resize)();
{ int         i;
  XGCValues   values;
  Colormap    cmap;
  int         win_x, win_y, win_w, win_h, win_b;
  window      win;
  XEvent      report;

  win = (window)malloc( (long)sizeof(*win) );
  if (win == NULL)
  { fprintf( stderr, "can't allocate window\n" );
    exit( -1 );
  }

  win->name    = name;
  win->w       = width;
  win->h       = height;
  win->ps_file = NULL;
  win->redraw  = redraw;
  win->point   = point;
  win->click   = click;
  win->key     = key;
  win->resize  = resize;

  /* get color map */

  cmap = DefaultColormap( display, screen );

  /* create a window */

  win_x = 0;      win_y = 0;       /* position */
  win_w = win->w; win_h = win->h;  /* size     */
  win_b = 2;                       /* border   */

  win->win = XCreateSimpleWindow( display,
                                  RootWindow( display, screen ),
                                  win_x, win_y,
                                  win_w, win_h,
                                  win_b, BlackPixel( display, screen ),
                                  WhitePixel( display, screen ) );

  /* set graphic context */

  if ((win->font=XLoadQueryFont(display,"6x10")) == NULL)
    if ((win->font=XLoadQueryFont(display,"fixed")) == NULL)
    { fprintf( stderr, "can't open 6x10 or fixed font\n");
      exit( -1 );
    }

  for (i=0; i<NB_COLORS; i++)
  { if (graph_black_and_white || (DisplayPlanes( display, screen ) == 1))
    { values.stipple = XCreateBitmapFromData( display,
                                              RootWindow( display, screen ),
                                              stipple[i], STIPPLE_W, STIPPLE_H);
      values.fill_style = FillOpaqueStippled;
      values.foreground = BlackPixel( display, screen );
      values.background = WhitePixel( display, screen );
    }
    else
    { XColor def;

      if (!XParseColor( display, cmap, colors[i], &def ))
      { fprintf( stderr, "color name %s not in database\n", colors[i] );
        exit( -1 );
      }

      if (!XAllocColor( display, cmap, &def ))
      { fprintf( stderr, "can't allocate color\n" );
        exit( -1 );
      }

      values.stipple = XCreateBitmapFromData( display,
                                              RootWindow( display, screen ),
                                              stipple[0], STIPPLE_W, STIPPLE_H);
      values.fill_style = FillSolid;
      values.foreground = def.pixel;
      values.background = WhitePixel( display, screen );
    }

    win->col[i] = XCreateGC( display,
                             RootWindow( display, screen ),
                             (GCForeground | GCBackground | GCStipple | GCFillStyle),
                             &values );

    XSetFont( display, win->col[i], win->font->fid );
  }

  values.function   = GXxor;
  values.plane_mask = 1;
  values.foreground = 1;
  values.background = 0;
  win->col[INVERT]  = XCreateGC( display,
                                 RootWindow( display, screen ),
                                 (GCForeground | GCBackground | GCFunction | GCPlaneMask),
                                 &values );

  /* display window */

  XMapWindow( display, win->win );

  /* wait until window appears */

  XSelectInput( display, win->win, ExposureMask );
  XWindowEvent( display, win->win, ExposureMask, &report );

  return win;
}


void g_resize( win )
window win;
{ XResizeWindow( display, win->win, win->w, win->h );
}


void g_idle( win )
window win;
{ XEvent report;
  int but_x, but_y, but = 0;

  /* enable events */

  XSelectInput( display, win->win, ExposureMask | KeyPressMask |
                                   PointerMotionMask |
                                   ButtonPressMask | ButtonReleaseMask |
                                   StructureNotifyMask );

  /* handle events */

  again:

  XNextEvent( display, &report );
  switch (report.type)
  { case Expose:
      while (XCheckTypedEvent( display, Expose, &report ));
      (*win->redraw)( win );
      break;

    case ButtonPress:
      but_x = report.xbutton.x;
      but_y = win->h - report.xbutton.y;
      but = report.xbutton.button;
      break;

    case ButtonRelease:
      if (report.xbutton.button == but)
        (*win->click)( win, but_x, but_y, report.xbutton.x, win->h - report.xbutton.y, but );
      break;

    case KeyPress:
    { KeySym sym; XComposeStatus comp; char buf[1];
      buf[0] = '\0';
      XLookupString( &report, buf, 1, &sym, &comp );
      if ((*win->key)( win, buf[0] )) goto done;
      break;
    }

    case MotionNotify:
    { int x, y;
      x = report.xmotion.x;
      y = report.xmotion.y;
      (*win->point)( win, x, win->h - y );
      break;
    }

    case ConfigureNotify:
      win->w = report.xconfigure.width;
      win->h = report.xconfigure.height;
      (*win->resize)( win );
      break;

    default: ;
  }

  goto again;

  done:

  /* finalize X */

  XCloseDisplay( display );
}

#endif


/*--------------------------------------------------------------------------*/

#ifdef MAC

#define OFFS_H 0
#define OFFS_V 0

char pattern[][8] = {
{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff },
{ 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa },
{ 0x00, 0x80, 0x00, 0x08, 0x00, 0x80, 0x00, 0x08 },
{ 0x11, 0x22, 0x44, 0x88, 0x11, 0x22, 0x44, 0x88 },
{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff },
{ 0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22 },
{ 0x00, 0xaa, 0x00, 0xaa, 0x00, 0xaa, 0x00, 0xaa },
{ 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55 },
{ 0xee, 0xdd, 0xbb, 0x77, 0xee, 0xdd, 0xbb, 0x77 },
{ 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77 },
{ 0xdd, 0xff, 0x77, 0xff, 0xdd, 0xff, 0x77, 0xff },
{ 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff },
{ 0x80, 0x10, 0x02, 0x20, 0x01, 0x08, 0x40, 0x04 },
{ 0x88, 0xff, 0x88, 0x88, 0x88, 0xff, 0x88, 0x88 },
{ 0xb1, 0x30, 0x03, 0x1b, 0xd8, 0xc0, 0x0c, 0x8d },
{ 0x00, 0x80, 0x40, 0x20, 0x00, 0x02, 0x04, 0x08 }
};


void g_init()
{ /* get screen size */

  graph_screen_width  = screenBits.bounds.right;
  graph_screen_height = screenBits.bounds.bottom;
}


void g_clear( win )
window win;
{ Rect r;

  r.left   = 0;
  r.right  = win->w;
  r.top    = 0;
  r.bottom = win->h;

  SetPort( win->win );
  FillRect( &r, &pattern[WHITE] );
}


void g_box( win, color, x, y, w, h )
window win;
int color;
int x, y, w, h;
{ Rect r;

  r.left   = x+OFFS_H;
  r.right  = x+OFFS_H + w;
  r.top    = win->h - (y+OFFS_H+h);
  r.bottom = win->h - (y+OFFS_H);

  SetPort( win->win );
  if (color == INVERT)
    InvertRect( &r );
  else
    FillRect( &r, &pattern[color] );
}


void g_text( win, color, where, x, y, str )
window win;
int color;
int where;
int x, y;
char *str;
{ int len = strlen(str);
  int str_width = TextWidth( str, 0, len );
  int y_pos = win->h - (y - 4);
  int x_pos;

  switch (where)
  { case CENTER: x_pos = x - str_width/2 + 1; break;
    case LEFT:   x_pos = x;                   break;
    case RIGHT:  x_pos = x - str_width;       break;
  }

  MoveTo( x_pos+OFFS_H, y_pos+OFFS_V );
  DrawText( str, 0, len );
}



void setup_window( win )
window win;
{ Rect r;

  r.left   = 0;
  r.right  = r.left + win->w;
  r.top    = 40;
  r.bottom = r.top + win->h;

  if (r.bottom > screenBits.bounds.bottom)
  { r.bottom = screenBits.bounds.bottom; r.top = r.bottom - win->h; }

  if (r.top < 0)
  { r.top = 0; r.bottom = r.top + win->h; }

  win->win = NewWindow(NULL, &r, "\pgsx", 1, noGrowDocProc, -1L, 1, 0);
  if (win->win == NULL)
  { fprintf( stderr, "can't allocate window\n" );
    exit( -1 );
  }

  SetPort( win->win );
  TextFont( monaco );
  TextSize( 9 );
}


window g_window( name, width, height, redraw, point, click, key, resize )
char *name;
int width, height;
void (*redraw)();
void (*point)();
void (*click)();
int (*key)();
void (*resize)();
{ window win;

  win = (window)malloc( (long)sizeof(*win) );
  if (win == NULL)
  { fprintf( stderr, "can't allocate window\n" );
    exit( -1 );
  }

  win->name    = name;
  win->w       = width;
  win->h       = height;
  win->ps_file = NULL;
  win->redraw  = redraw;
  win->point   = point;
  win->click   = click;
  win->key     = key;
  win->resize  = resize;

  setup_window( win );

  return win;
}


void g_resize( win )
window win;
{ DisposeWindow( win->win );
  setup_window( win );
  (*win->resize)( win );
  (*win->redraw)( win );
}


void g_idle( win )
window win;
{ int ok;
  int x1 = 0, y1 = 0;
  EventRecord event;
  Point last_pt, pt;

  GetMouse( &last_pt );

  again:

  SystemTask ();  /* Handle desk accessories */

  ok = GetNextEvent(everyEvent, &event);
  if (ok)
  { switch (event.what)
    { case mouseDown:
      case mouseUp:
      { WindowPtr w;
        if ((FindWindow( event.where, &w ) == inContent) && (w == win->win))
        { SetPort( win->win );
          GlobalToLocal( &event.where );
          { int x = event.where.h;
            int y = win->h - event.where.v;
            if (event.what == mouseDown)
            { x1 = x; y1 = y; }
            else
              (*win->click)( win, x1, y1, x, y, 0 );
          }
        }
        break;
      }
      case keyDown: 
      case autoKey:
        if ((event.modifiers & cmdKey) == 0)
          if ((*win->key)( win, (char)(event.message & charCodeMask) )) goto done;
        break;
    }
    GetMouse( &pt );
    if ((pt.h != last_pt.h) || (pt.v != last_pt.v))
    { last_pt = pt;
      (*win->point)( win, pt.h, win->h - pt.v );
    }
  }

  goto again;

  done:

  DisposeWindow( win->win );
}

#endif


/*--------------------------------------------------------------------------*/

int last_point_x = -1, last_point_y = -1;

int shade[] = { 100, 0, 50, 99, 83, 0, 92, 65, 33, 10, 25, 15, 60, 75, 55, 40, 95 };


void graph_postscript_begin( win, name, for_printing )
window win;
char *name;
int for_printing;
{ if (win->ps_file == NULL)
  { win->ps_file = fopen( name, "w" );
    if (win->ps_file == NULL)
      fprintf( stderr, "couldn't create postscript file\n" );
    else
    { fprintf( win->ps_file, "%%!PS-Adobe-1.0\n" );
      fprintf( win->ps_file, "%%%%Title: %s\n", name );
      fprintf( win->ps_file, "%%%%Creator: gsx\n" );
      fprintf( win->ps_file, "%%%%BoundingBox: 0 0 %d %d\n", win->w, win->h );
      fprintf( win->ps_file, "%%%%EndComments\n" );
      fprintf( win->ps_file, "/$GSXDict 32 dict def $GSXDict begin\n" );
      if (for_printing)
        fprintf( win->ps_file, "/$GSXEnd {showpage end} def [ 0 -1 1 0 0 0 ] concat -750 17 translate 0.72 0.72 scale\n" );
      else
        fprintf( win->ps_file, "/$GSXEnd {$GSXState restore end} def /$GSXState save def\n" );
      fprintf( win->ps_file, "/$GSXBox {setgray 0.75 add moveto dup 0 exch rlineto exch 0 rlineto neg 0 exch rlineto closepath fill} def\n" );
      fprintf( win->ps_file, "/$GSXLeft {setgray 2.25 sub moveto show} def\n" );
      fprintf( win->ps_file, "/$GSXRight {setgray 2.25 sub moveto dup stringwidth pop neg 0 rmoveto show} def\n" );
      fprintf( win->ps_file, "/$GSXCenter {setgray 2.25 sub moveto dup stringwidth pop 2 div neg 0 rmoveto show} def\n" );
      fprintf( win->ps_file, "/Courier findfont 9 scalefont setfont\n" );
    }
  }
}


void graph_postscript_end( win )
window win;
{ if (win->ps_file != NULL)
  { fprintf( win->ps_file, "$GSXEnd\n" );
    fclose( win->ps_file );
    win->ps_file = NULL;
  }
}


void graph_init()
{ if (graph_display) g_init();
}


void graph_box( win, color, x, y, w, h )
window win;
int color;
int x, y, w, h;
{ if (win->ps_file == NULL)
  { if (graph_display) g_box( win, color, x, y, w, h ); }
  else if (color < INVERT)
  { fprintf( win->ps_file, "%d %d %d %d ", w, h, x, y );
    fprintf( win->ps_file, "%d.%d%d $GSXBox\n", shade[color]/100, shade[color]/10%10, shade[color]%10 );
  }
}


void graph_text( win, color, where, x, y, str )
window win;
int color;
int where;
int x, y;
char *str;
{ if (win->ps_file == NULL)
  { if (graph_display) g_text( win, color, where, x, y, str ); }
  else
  { fprintf( win->ps_file, "(%s) %d %d ", str, x, y );
    fprintf( win->ps_file, "%d.%d%d ", shade[color]/100, shade[color]/10%10, shade[color]%10 );
    switch (where)
    { case CENTER: fprintf( win->ps_file, "$GSXCenter\n" ); break;
      case LEFT:   fprintf( win->ps_file, "$GSXLeft\n" );   break;
      case RIGHT:  fprintf( win->ps_file, "$GSXRight\n" );  break;
    }
  }
}


void graph_clear( win )
window win;
{ if ((win->ps_file == NULL) && (graph_display)) g_clear( win );
}


window graph_window( name, width, height, redraw, point, click, key, resize )
char *name;
int width, height;
void (*redraw)();
void (*point)();
void (*click)();
int (*key)();
void (*resize)();
{ if (graph_display)
    return g_window( name, width, height, redraw, point, click, key, resize );
  else
  { window      win;
    win = (window)malloc( (long)sizeof(*win) );
    if (win == NULL)
    { fprintf( stderr, "can't allocate window\n" );
      exit( -1 );
    }
    win->name    = name;
    win->w       = width;
    win->h       = height;
    win->ps_file = NULL;
    win->redraw  = redraw;
    win->point   = point;
    win->click   = click;
    win->key     = key;
    win->resize  = resize;
    return win;
  }
}


void graph_resize( win, w, h )
window win;
int w, h;
{ win->w = w;
  win->h = h;
  if (graph_display) g_resize( win );
}


void graph_idle( win )
window win;
{ if (graph_display) g_idle( win );
}


/*--------------------------------------------------------------------------*/

/* event log stuff */

#define MAX_NB_TRACES 128
#define MAX_NB_EVENT_TYPES 15
#define EVENT_NUM(x) ((x)>>24)
#define EVENT_TIME(x) ((x)&0xffffff)

long CLOCK_TICKS_PER_TIME_UNIT_NUM, CLOCK_TICKS_PER_TIME_UNIT_DEN;
char *time_unit_name;

char *log_filename;

char log_event[MAX_NB_EVENT_TYPES][100];

struct {
  int len;
  long *event;
  } log_trace[MAX_NB_TRACES];

struct {
  int len;
  long *dur;
  } events_of_type[MAX_NB_EVENT_TYPES];

int log_nb_events;
int log_nb_traces;

long log_max_time;

int idle_event_num;


#ifdef MAC

void cvt( val )
long *val;
{
}

#else

#include <netinet/in.h>

void cvt( val )
long *val;
{ *val = ntohl( *val );
}

#endif


char upcase( c )
char c;
{ if ((c >= 'a') && (c <= 'z')) return 'A'+(c-'a'); else return c;
}


int string_ci_equal( s1, s2 )
char *s1, *s2;
{ while ((upcase(*s1) == upcase(*s2)) && (*s1 != '\0')) { s1++; s2++; }
  return ((*s1 == '\0') && (*s2 == '\0'));
}


void read_log( name, show_log )
char *name;
int show_log;
{ FILE *input = fopen( name, "rb" );
  int first_event = 1;
  int clock_64bits = 0;
  long reference, time_compression;
  long buf[10], *p;
  long log_min_time = 0;
  long total_nb_events = 0;

  if (input == NULL)
  { fprintf( stderr, "can't open event log file %s\n", name );
    exit( -1 );
  }

  log_max_time = 0;

  log_nb_events = 0;
  log_nb_traces = 0;

  idle_event_num = 0;

  next:
  if (fread( buf, (long)sizeof(long), 2L, input ) == 2)
  { cvt( &buf[0] ); cvt( &buf[1] );
    switch ( buf[1] )
    { case 1: fread( buf, (long)sizeof(long), 2L, input );
              cvt( &buf[0] );
              clock_64bits = (buf[0] != 0x80000000);
              goto next;
      case 2: fread( &buf[2], (long)sizeof(long), 1L, input );
              fread( log_event[log_nb_events], (long)sizeof(long), (long)buf[0]/sizeof(long)-2, input );
              if (show_log) fprintf( stdout, "Event %d = %s\n", log_nb_events, log_event[log_nb_events] );
              if (string_ci_equal( log_event[log_nb_events], "idle" ))
                idle_event_num = log_nb_events;
              log_nb_events++;
              goto next;
      case 3: { int i, len = ((buf[0]/sizeof(long))-5)/(clock_64bits?4:3);
                if (show_log) fprintf( stdout, "P%d: %d events\n", log_nb_traces, len );
                log_trace[log_nb_traces].len = len;
                total_nb_events += len;
                p = (long *)malloc( (long)sizeof(long) * len );
                if (p == NULL)
                { fprintf( stderr, "memory overflow\n" );
                  exit( -1 );
                }
                log_trace[log_nb_traces].event = p;
                fread( &buf[2], (long)sizeof(long), 4L, input );
                for (i=0; i<len; i++)
                { long tim, evt, e;
                  fread( buf, (long)sizeof(long), clock_64bits?4L:3L, input );
                  tim = buf[clock_64bits?1:0]; evt = buf[clock_64bits?2:1];
                  cvt( &tim ); cvt( &evt );
                  if (first_event) { first_event = 0; reference = tim; }
                  tim -= reference;
                  e = EVENT_TIME(tim)+((evt-1)<<24);
                  if (show_log) fprintf( stdout, " %l07x", e );
                  log_trace[log_nb_traces].event[i] = e;
                  if (tim < log_min_time) log_min_time = tim;
                  if (tim > log_max_time) log_max_time = tim;
                  if ((evt-1 < 0) || (evt-1 >= log_nb_events))
                  { fprintf( stderr, "event log format error (event_num=0x%lx)\n", evt );
                    exit( -1 );
                  }
                }
                if (show_log) fprintf( stdout, "\n" );
                log_nb_traces++;
                goto next;
              }
      default:
       fprintf( stderr, "event log format error (code=0x%lx)\n", buf[1] );
       exit( -1 );
    }
  }

  fclose( input );

  { int i;
    for (i=0; i<MAX_NB_EVENT_TYPES; i++)
    { long *p = (long *)malloc( (long)sizeof(long) * total_nb_events );
      if (p == NULL)
      { fprintf( stderr, "memory overflow\n" );
        exit( -1 );
      }
      events_of_type[i].dur = p;
    }
  }

  if (log_min_time == log_max_time)
  { fprintf( stderr, "event log empty\n" );
    exit( -1 );
  }

  if (clock_64bits)
  { CLOCK_TICKS_PER_TIME_UNIT_NUM = 1;
    CLOCK_TICKS_PER_TIME_UNIT_DEN = 1;
    time_unit_name = "usec";
    time_compression = 10;
  }
  else
  { CLOCK_TICKS_PER_TIME_UNIT_NUM = 16;
    CLOCK_TICKS_PER_TIME_UNIT_DEN = 1;
    time_unit_name = "msec";
    time_compression = 1;
  }

  CLOCK_TICKS_PER_TIME_UNIT_DEN *= time_compression;

  { int i, j;
    for (i=0;i<log_nb_traces;i++)
      for (j=log_trace[i].len-1;j>=0;j--)
      { long evt = log_trace[i].event[j];
        log_trace[i].event[j] = EVENT_TIME(evt-log_min_time)/time_compression +
                                (EVENT_NUM(evt)<<24);
      }
    log_max_time = (log_max_time - log_min_time)/time_compression;
  }
}


/*--------------------------------------------------------------------------*/

/* plot stuff */

#ifdef MAC
#define DEFAULT_PLOT_HEIGHT 128
#else
#define DEFAULT_PLOT_HEIGHT 320
#endif
#define MAX_PLOT_WIDTH      960
#define MIN_PLOT_WIDTH      64
#define BORDER_LEFT         12
#define BORDER_RIGHT        40

#define LABEL_BORDER   5
#define LABEL_HEIGHT   17
#define HIST_BORDER    35
#define HIST_HEIGHT    64
#define HIST_BAR_WIDTH 4
#define PERC_BORDER    20
#define PERC_HEIGHT    11
#define PLOT_BORDER    20
#define BAR_BORDER     9
#define BAR_HEIGHT     bar_height
#define BAR_SPACING    bar_spacing
#define TITLE_HEIGHT   10
#define TITLE_BORDER   10

#define LABEL_THEIGHT  (LABEL_BORDER+LABEL_HEIGHT)
#define HIST_THEIGHT   ((show_hist) ? HIST_BORDER+HIST_HEIGHT : 0 )
#define HIST_NB_BARS   ((plot_width/nb_labels() - 20) / HIST_BAR_WIDTH)
#define PERC_THEIGHT   (PERC_BORDER+PERC_HEIGHT)
#define PLOT_THEIGHT   (PLOT_BORDER+plot_height)
#define TRACES_THEIGHT ((show_traces) ? BAR_BORDER+log_nb_traces*(BAR_HEIGHT+BAR_SPACING) : 0)
#define TITLE_THEIGHT  (TITLE_BORDER+TITLE_HEIGHT)
#define THEIGHT LABEL_THEIGHT+HIST_THEIGHT+PERC_THEIGHT+PLOT_THEIGHT+TRACES_THEIGHT+TITLE_THEIGHT

#define MAX_TICKS_PER_PLOT 300
#define MAX_TICKS_PER_TIME_UNIT 10

#define TICK_TEXT_OFFS 10
#define TICK_SHORT     2
#define TICK_LONG      5


int full_plot_width, plot_width, plot_height;
int show_hist = 0;
int show_traces = 0;
int smooth_passes;
int bar_height, bar_spacing;

long *count[MAX_NB_EVENT_TYPES];
long perc_width[MAX_NB_EVENT_TYPES];
long min_time, max_time, duration, last_end;
long hist[MAX_NB_EVENT_TYPES][MAX_PLOT_WIDTH/HIST_BAR_WIDTH];
long hist_span[MAX_NB_EVENT_TYPES];
long hist_sum[MAX_NB_EVENT_TYPES];


void for_each_event_of_trace( i, proc )
int i;
void (*proc)();
{ int j;
  int last_event_num = idle_event_num;
  long last_event_time = 0;
  last_end = -1;
  for (j=0; j<log_trace[i].len; j++)
  { int event_num = EVENT_NUM(log_trace[i].event[j]);
    long event_time = EVENT_TIME(log_trace[i].event[j]);
    (*proc)( last_event_num, last_event_time, event_time );
    last_event_num = event_num;
    last_event_time = event_time;
  }
  (*proc)( last_event_num, last_event_time, log_max_time );
}


int xlate[] = { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14 };


int nb_labels()
{ int i, n = 0;
  for (i=0; i<log_nb_events; i++)
  { int event_num = xlate[i];
    if (perc_width[event_num] != 0) n++;
  }
  return n;
}


void for_each_event( proc )
void (*proc)();
{ int i;
  for (i=0; i<log_nb_traces; i++)
    for_each_event_of_trace( i, proc );
}


void qs( a, n )
long a[], n;
{ long *lpos, *rpos;
  long pivot;
  if (n <= 2)
  { if ((n == 2) && (a[0] > a[1]))
    { long tmp = a[0]; a[0] = a[1]; a[1] = tmp; }
    return;
  }
  pivot = a[n/2]; lpos = a; rpos = a+(n-1);
  while (lpos < rpos)
  { while (*rpos >= pivot) { if (--rpos == lpos) goto end; }
    *lpos++ = *rpos;
    while (*lpos <= pivot) { if (++lpos == rpos) goto end; }
    *rpos-- = *lpos;
  }
  end:
  *lpos = pivot;
  qs( a, lpos-a );
  qs( lpos+1, n-(lpos-a)-1 );
}


void smooth( a, n, passes )
long a[];
int n;
int passes;
{ int i;
  while (passes-- > 0)
  { long *p = a;
    long prev;
    long temp = *p;
    *p = (temp+temp+temp+p[1]+2) >> 2;
    p++;
    for (i=1; i<n-1; i++)
    { prev = temp;
      temp = *p;
      *p = (prev+temp+temp+p[1]+2) >> 2;
      p++;
    }
    temp = *p;
    *p = (prev+temp+temp+temp+2) >> 2;
  }
}


void histogram( a, n, b, m, range_top, peak )
long a[];
int n;
long b[];
int m;
long range_top;
long peak;
{ int i, last, max_val;
  last = 0;
  for (i=0; i<m; i++)
  { int cur = last;
    while ((cur < n) && (a[cur] <= (i+1)*range_top/m)) cur++;
    b[i] = cur-last;
    last = cur;
  }
  max_val = 0;
  for (i=0; i<m; i++)
    if (b[i] > max_val) max_val = b[i];
  if (max_val > 0)
    for (i=0; i<m; i++) b[i] = b[i]*peak/max_val;
}


void add_event1( event_num, start, end )
int event_num;
long start, end;
{ int i;

  if ((start >= max_time) || (end <= min_time)) return;
  if (start < min_time) start = min_time;
  if (end > max_time) end = max_time;

  events_of_type[event_num].dur[events_of_type[event_num].len++] = end-start;
}


void compute_hist()
{ int i, j;
  for (i=0; i<log_nb_events; i++)
    events_of_type[i].len = 0;
  for_each_event( add_event1 );
  for (i=0; i<log_nb_events; i++)
  { long span = 0, sum = 0;
    if (events_of_type[i].len > 0)
    { for (j=0; j<events_of_type[i].len; j++) sum += events_of_type[i].dur[j];
      qs( events_of_type[i].dur, events_of_type[i].len );
      span = events_of_type[i].dur[events_of_type[i].len-1];
      histogram( events_of_type[i].dur, events_of_type[i].len,
                 hist[i], HIST_NB_BARS, span, HIST_HEIGHT );
    }
    hist_span[i] = span;
    hist_sum[i] = sum;
  }
}


void add_event2( event_num, start, end )
int event_num;
long start, end;
{ long ss, s, ee, e;
  int i;
  long *p;

  if ((start >= max_time) || (end <= min_time)) return;
  if (start < min_time) start = min_time;
  if (end > max_time) end = max_time;

  ss = (start-min_time)*plot_width;
  ee = (end-min_time)*plot_width;
  if (ss < last_end) ss = last_end;
  if (ee <= ss) ee = ss+plot_width/2;
  last_end = ee;

  s = ss/duration;
  e = ee/duration;
  if (e > plot_width) e = plot_width;

  p = &count[event_num][s];
  *p -= ss - s*duration;
  for (i=s; i<e; i++) *p++ += duration;
  *p++ += ee - e*duration;
}


void compute_plot( min_t, max_t )
long min_t, max_t;
{ int i, j;

  if (min_t < 0) min_t = 0;
  if (min_t > log_max_time) min_t = log_max_time;

  if (max_t < 0) max_t = 0;
  if (max_t > log_max_time) max_t = log_max_time;

  min_time = min_t;
  max_time = max_t;

  duration = max_time - min_time;

  for (i=0; i<full_plot_width; i++)
    for (j=0; j<log_nb_events; j++) count[j][i] = 0;

  for_each_event( add_event2 );

  for (j=0; j<log_nb_events; j++)
    smooth( count[j], plot_width, smooth_passes );

  for (i=0; i<plot_width; i++)
  { long h1 = 0, h2 = 0, tot_dur = 0;
    for (j=0; j<log_nb_events; j++) tot_dur += count[j][i];
    for (j=0; j<log_nb_events-1; j++)
    { int k;
      h1 += count[j][i];
      k = (plot_height*h1)/tot_dur;
      count[j][i] = k - h2;
      h2 = k;
    }
    count[j][i] = plot_height - h2;
  }

  { long tot = 0;
    for (j=0; j<log_nb_events-1; j++)
    { long sum = 0;
      for (i=0; i<plot_width; i++) sum += count[j][i];
      sum = sum*full_plot_width/((long)plot_width*plot_height);
      perc_width[j] = sum;
      tot += sum;
    }
    perc_width[j] = full_plot_width-tot;
  }
}


void flip_cross_hair( win )
window win;
{ int b = LABEL_THEIGHT+HIST_THEIGHT+PERC_THEIGHT;
  int t = b+PLOT_THEIGHT+TRACES_THEIGHT;
  if ((last_point_x >= BORDER_LEFT) && (last_point_x <= BORDER_LEFT+plot_width+10) &&
      (last_point_y >= b) && (last_point_y <= t))
  { graph_box( win, INVERT, last_point_x, b, 1, t-b+1 );
    graph_box( win, INVERT, BORDER_LEFT-1, last_point_y, plot_width+10, 1 );
  }
}


void draw_scale( win, x, y, min_time, max_time, width, height, max_ticks )
window win;
int x, y;
long min_time, max_time;
int width, height;
int max_ticks;
{ int i, j, k, n;
  long incr1, incr2;
  long time_span = max_time-min_time;
  incr1 = CLOCK_TICKS_PER_TIME_UNIT_NUM;
  incr2 = 1;
  while (time_span*CLOCK_TICKS_PER_TIME_UNIT_DEN*MAX_TICKS_PER_TIME_UNIT > max_ticks*incr1*incr2) { incr1 *= 5; incr2 *= 2; }

  i = ((long)min_time*CLOCK_TICKS_PER_TIME_UNIT_DEN*MAX_TICKS_PER_TIME_UNIT+incr1*incr2-1)/(incr1*incr2);
  n = ((long)max_time*CLOCK_TICKS_PER_TIME_UNIT_DEN*MAX_TICKS_PER_TIME_UNIT+incr1*incr2-1)/(incr1*incr2) - i;

  while (i*incr1*incr2 <= (long)max_time*CLOCK_TICKS_PER_TIME_UNIT_DEN*MAX_TICKS_PER_TIME_UNIT)
  { int k = x +
            (incr1*width*i-(long)width*min_time*CLOCK_TICKS_PER_TIME_UNIT_DEN*MAX_TICKS_PER_TIME_UNIT/incr2) /
            (time_span*CLOCK_TICKS_PER_TIME_UNIT_DEN*MAX_TICKS_PER_TIME_UNIT/incr2);
    if ((n < max_ticks/MAX_TICKS_PER_TIME_UNIT) && (n <= width/12)) /* time stamp every tick mark if small time span and enough space */
    { char num[10];
      long v = i*10*incr1*incr2/(CLOCK_TICKS_PER_TIME_UNIT_NUM*MAX_TICKS_PER_TIME_UNIT);
      if (v < 10)
        sprintf( num, ".%d", v );
      else
        sprintf( num, "%d.%d", v/10, v%10 );
      graph_text( win, BLACK, CENTER, k, y-TICK_TEXT_OFFS, num );
    }
    else if (i % 10 == 0) /* else time stamp every 10 tick marks */
    { char num[10];
      sprintf( num, "%d", (int)(i*incr1*incr2/(CLOCK_TICKS_PER_TIME_UNIT_NUM*MAX_TICKS_PER_TIME_UNIT)) );
      graph_text( win, BLACK, CENTER, k, y-TICK_TEXT_OFFS, num );
    }
    if (i % 5 == 0)
    { int offs = (i % 2 == 0) ? 3 : 4;
      int len  = (i % 2 == 0) ? 3 : 1;
      graph_box( win, BLACK, k, y-TICK_LONG-1, 1, TICK_LONG );
      for (j=0; j<height; j+=8)
        graph_box( win, BLACK, k, y+j+offs, 1, len );
    }
    else
      graph_box( win, BLACK, k, y-TICK_SHORT-1, 1, TICK_SHORT );
    i++;
  }
}


void draw_title( win )
window win;
{ int b = LABEL_THEIGHT+HIST_THEIGHT+PERC_THEIGHT+PLOT_THEIGHT+TRACES_THEIGHT;
  char title[256];
  sprintf( title, "File: \"%s\"    Processors: %d", log_filename, log_nb_traces );
  graph_text( win, BLACK, LEFT, BORDER_LEFT, b+TITLE_BORDER, title );
}


void draw_cross_hair_info( win )
window win;
{ int b = LABEL_THEIGHT+HIST_THEIGHT+PERC_THEIGHT;
  int t = b+PLOT_THEIGHT+TRACES_THEIGHT;
  graph_box( win, WHITE, BORDER_LEFT, t, plot_width, TITLE_THEIGHT );
  draw_title( win );
  if ((last_point_x >= BORDER_LEFT) && (last_point_x <= BORDER_LEFT+plot_width) &&
      (last_point_y >= b) && (last_point_y <= t))
  { char info[256];
    long ticks_times2 = 2*min_time+2*duration*(last_point_x-BORDER_LEFT)/plot_width;
    long temp1 = (min_time*plot_width+duration*(last_point_x-BORDER_LEFT))*CLOCK_TICKS_PER_TIME_UNIT_DEN;
    long temp2 = CLOCK_TICKS_PER_TIME_UNIT_NUM*plot_width;
    long t_int = temp1 / temp2;
    long t_frc = 100*(temp1 % temp2) / temp2;
    int trace_y = last_point_y - (b+PLOT_THEIGHT+BAR_BORDER);
    if ((trace_y >= 0) && (trace_y % (BAR_HEIGHT+BAR_SPACING) >= BAR_SPACING))
    { int i = trace_y / (BAR_HEIGHT+BAR_SPACING);
      int j, event_num;
      long event_time, t1_int, t1_frc, t2_int, t2_frc, t3_int, t3_frc;
      int last_event_num = idle_event_num;
      long last_event_time = 0;
      for (j=0; j<log_trace[i].len; j++)
      { event_num = EVENT_NUM(log_trace[i].event[j]);
        event_time = EVENT_TIME(log_trace[i].event[j]);
        if (event_time == last_event_time)
        { if (ticks_times2 < event_time*2+1) goto found; }
        else
        { if (ticks_times2 < event_time*2) goto found; }
        last_event_num = event_num;
        last_event_time = event_time;
      }
      event_time = log_max_time;
      found:

      temp1 = last_event_time*CLOCK_TICKS_PER_TIME_UNIT_DEN;
      temp2 = CLOCK_TICKS_PER_TIME_UNIT_NUM;
      t1_int = temp1 / temp2;
      t1_frc = 100*(temp1 % temp2) / temp2;
      temp1 = event_time*CLOCK_TICKS_PER_TIME_UNIT_DEN;
      temp2 = CLOCK_TICKS_PER_TIME_UNIT_NUM;
      t2_int = temp1 / temp2;
      t2_frc = 100*(temp1 % temp2) / temp2;
      temp1 = (event_time-last_event_time)*CLOCK_TICKS_PER_TIME_UNIT_DEN;
      temp2 = CLOCK_TICKS_PER_TIME_UNIT_NUM;
      t3_int = temp1 / temp2;
      t3_frc = 100*(temp1 % temp2) / temp2;

      sprintf( info, "P%d: %s %d.%d%d %s (%d.%d%d..%d.%d%d)  [%d.%d%d]", i, log_event[last_event_num],
                     t3_int, t3_frc/10, t3_frc%10, time_unit_name,
                     t1_int, t1_frc/10, t1_frc%10,
                     t2_int, t2_frc/10, t2_frc%10,
                     t_int, t_frc/10, t_frc%10 );
    }
    else
      sprintf( info, "[%d.%d%d]", t_int, t_frc/10, t_frc%10 );
    graph_text( win, BLACK, RIGHT, BORDER_LEFT+plot_width, t+TITLE_BORDER, info );
  }
}


void draw_labels( win )
window win;
{ int b = 0;
  int i, j, n;

  n = nb_labels();

  j = 0;
  for (i=0; i<log_nb_events; i++)
  { int event_num = xlate[i];
    if (perc_width[event_num] != 0)
    { int x_pos = BORDER_LEFT+(full_plot_width*j)/n-1;
      graph_box( win, OTHER+event_num, x_pos, b, LABEL_HEIGHT, LABEL_HEIGHT );
      graph_box( win, BLACK, x_pos, b, 1, LABEL_HEIGHT );
      graph_box( win, BLACK, x_pos, b, LABEL_HEIGHT, 1 );
      graph_box( win, BLACK, x_pos+LABEL_HEIGHT-1, b, 1, LABEL_HEIGHT );
      graph_box( win, BLACK, x_pos, LABEL_HEIGHT-1, b+LABEL_HEIGHT, 1 );
      graph_text( win, BLACK, LEFT, x_pos+LABEL_HEIGHT+4, b+LABEL_HEIGHT/2, log_event[event_num] );
      j++;
    }
  }
}


void draw_hist( win )
window win;
{ int b = LABEL_THEIGHT+HIST_BORDER;
  int i, j, k, n;

  n = nb_labels();

  j = 0;
  for (i=0; i<log_nb_events; i++)
  { int event_num = xlate[i];
    if (perc_width[event_num] != 0)
    { int x_pos = BORDER_LEFT+(full_plot_width*j)/n;
      if (hist_span[event_num] > 0)
      { int a = hist_sum[event_num]*HIST_NB_BARS*HIST_BAR_WIDTH/(events_of_type[event_num].len*hist_span[event_num]);
        int l;
        char num[10];
        long v = 1000*hist_sum[event_num]/(CLOCK_TICKS_PER_TIME_UNIT_NUM*events_of_type[event_num].len);
        draw_scale( win, x_pos, b, 0, hist_span[event_num], HIST_NB_BARS*HIST_BAR_WIDTH, 0, HIST_NB_BARS*HIST_BAR_WIDTH*2/3 );
        for (k=0; k<HIST_NB_BARS; k++)
          graph_box( win, BLACK, x_pos+k*HIST_BAR_WIDTH, b, HIST_BAR_WIDTH, hist[event_num][k] );
        for (l= -20; l<0; l += 2)
          graph_box( win, BLACK, x_pos+a, b+l, 1, 1 );
        if (v < 1000)
          if (v < 100)
            sprintf( num, ".0%d%d", v/10, v%10 );
          else
            sprintf( num, ".%d%d%d", v/100, (v/10)%10, v%10 );
        else
          sprintf( num, "%d.%d%d%d", v/1000, (v/10)%10, (v/100)%10, v%10 );
        graph_box( win, BLACK, x_pos-1, b-1, 1, HIST_HEIGHT+1 );
        graph_box( win, BLACK, x_pos, b-1, HIST_NB_BARS*HIST_BAR_WIDTH, 1 );
        graph_text( win, BLACK, CENTER, x_pos+a, b-15-TICK_TEXT_OFFS, num );
      }
      j++;
    }
  }

  graph_text( win, BLACK, CENTER, BORDER_LEFT + plot_width + 20, b-TICK_TEXT_OFFS, time_unit_name );
}


void draw_perc_plot_frame( win )
window win;
{ int b = LABEL_THEIGHT+HIST_THEIGHT+PERC_BORDER-1;
  int i;

  graph_box( win, BLACK, BORDER_LEFT-1, b-1, full_plot_width+2, 1 );
  graph_box( win, BLACK, BORDER_LEFT-1, b-1, 1, PERC_HEIGHT+2 );
  graph_box( win, BLACK, BORDER_LEFT+full_plot_width, b-1, 1, PERC_HEIGHT+2 );
  graph_box( win, BLACK, BORDER_LEFT-1, b+PERC_HEIGHT, full_plot_width+2, 1 );

  for (i=0; i<=100; i++)
  { int k = BORDER_LEFT + ((long)full_plot_width)*i/100;
    if (i % 10 == 0) /* time stamp every 10 tick marks */
    { char num[10];
      sprintf( num, "%d", i );
      graph_text( win, BLACK, CENTER, k, b-TICK_TEXT_OFFS, num );
    }
    if (i % 5 == 0)
      graph_box( win, BLACK, k, b-TICK_LONG-1, 1, TICK_LONG );
    else
      graph_box( win, BLACK, k, b-TICK_SHORT-1, 1, TICK_SHORT );
  }

  graph_text( win, BLACK, CENTER, BORDER_LEFT + full_plot_width + 20, b-TICK_TEXT_OFFS, "%" );
}


void draw_perc_plot_content( win )
window win;
{ int b = LABEL_THEIGHT+HIST_THEIGHT+PERC_BORDER-1;
  int i, j = 0;
  for (i=0; i<log_nb_events; i++)
  { int event_num = xlate[i];
    int k = perc_width[event_num];
    graph_box( win, OTHER+event_num, BORDER_LEFT+j, b, k, PERC_HEIGHT );
    j += k;
  }
}


void draw_perc_plot( win )
window win;
{ draw_perc_plot_content( win );
  draw_perc_plot_frame( win );
}


void draw_activ_plot_frame( win )
window win;
{ int b = LABEL_THEIGHT+HIST_THEIGHT+PERC_THEIGHT+PLOT_BORDER-1;
  int i, j;

  graph_box( win, BLACK, BORDER_LEFT-1, b-1, plot_width+2, 1 );
  graph_box( win, BLACK, BORDER_LEFT-1, b-1, 1, plot_height+2 );
  graph_box( win, BLACK, BORDER_LEFT+plot_width, b-1, 1, plot_height+2 );
  graph_box( win, BLACK, BORDER_LEFT-1, b+plot_height, plot_width+2, 1 );

  for (i=0; i<=10; i++)
  { int k = b+plot_height*i/10;
    int offs = (i % 2 == 0) ? 3 : 4;
    int len  = (i % 2 == 0) ? 3 : 1;
    graph_box( win, BLACK, BORDER_LEFT+plot_width, k, 5, 1 );
    if (i % 2 == 0)
    { char num[10];
      sprintf( num, "%d", i*10 );
      graph_text( win, BLACK, LEFT, BORDER_LEFT+plot_width+8, k, num );
    }
    if ((i>0) && (i<10))
      for (j=0; j<plot_width; j+=8)
        graph_box( win, BLACK, BORDER_LEFT+j+offs, k, len, 1 );
  }
  graph_text( win, BLACK, LEFT, BORDER_LEFT+plot_width+30, b+plot_height/2, "%" );

  draw_scale( win, BORDER_LEFT, b, min_time, max_time, plot_width, plot_height, MAX_TICKS_PER_PLOT );

  graph_text( win, BLACK, CENTER, BORDER_LEFT + plot_width + 20, b-TICK_TEXT_OFFS, time_unit_name );
}


void draw_activ_plot_between( win, start, end, level, bot )
window win;
int start, end, level, bot;
{ int b = LABEL_THEIGHT+HIST_THEIGHT+PERC_THEIGHT+PLOT_BORDER-1;
  if (level < log_nb_events)
  { int event_num = xlate[level];
    int i = start;
    while (i <= end)
    { int h = count[event_num][i];
      int j = i+1;
      while ((j <= end) && (count[event_num][j] == h)) j++;
      graph_box( win, OTHER+event_num, BORDER_LEFT+i, b+bot, j-i, h );
      draw_activ_plot_between( win, i, j-1, level+1, bot+h );
      i = j;
    }
  }
}


void draw_activ_plot( win )
window win;
{ draw_activ_plot_between( win, 0, plot_width-1, 0, 0 );
  draw_activ_plot_frame( win );
}


void draw_event( win, trace_num, event_num, start, end )
window win;
int trace_num;
int event_num;
long start, end;
{ int b = LABEL_THEIGHT+HIST_THEIGHT+PERC_THEIGHT+PLOT_THEIGHT+BAR_BORDER-1;
  long ss, ee;
  int s, e;
  int i;

  if ((start >= max_time) || (end <= min_time)) return;
  if (start < min_time) start = min_time;
  if (end > max_time) end = max_time;

  ss = (start-min_time)*plot_width;
  ee = (end-min_time)*plot_width;
  if (ss < last_end) ss = last_end;
  if (ee <= ss) ee = ss+plot_width/2;
  last_end = ee;

  s = ss/duration;
  e = ee/duration;
  if (e <= s) e = s+1; /* draw at least 1 pixel wide */
  if (e > plot_width) e = plot_width;

  graph_box( win, OTHER+event_num, BORDER_LEFT+s,b+trace_num*(BAR_HEIGHT+BAR_SPACING)+BAR_SPACING+1, e-s, BAR_HEIGHT );
}


void draw_traces( win )
window win;
{ int b = LABEL_THEIGHT+HIST_THEIGHT+PERC_THEIGHT+PLOT_THEIGHT+BAR_BORDER-1;
  int i, j;

  for (i=0; i<log_nb_traces; i++)
  { int last_event_num = idle_event_num;
    long last_event_time = 0;
    last_end = -1;
    for (j=0; j<log_trace[i].len; j++)
    { int event_num = EVENT_NUM(log_trace[i].event[j]);
      long event_time = EVENT_TIME(log_trace[i].event[j]);
      draw_event( win, i, last_event_num, last_event_time, event_time );
      last_event_num = event_num;
      last_event_time = event_time;
    }
    draw_event( win, i, last_event_num, last_event_time, log_max_time );
    if (i % 5 == 0)
    { char num[10];
      sprintf( num, "P%d", i );
      graph_text( win, BLACK, LEFT, BORDER_LEFT+plot_width+8, b+i*(BAR_HEIGHT+BAR_SPACING)+BAR_SPACING+BAR_HEIGHT/2+1, num );
      graph_box( win, BLACK, BORDER_LEFT+plot_width, b+i*(BAR_HEIGHT+BAR_SPACING)+BAR_SPACING+BAR_HEIGHT/2+1, TICK_LONG, 1 );
    }
  }
}


void draw_all( win )
window win;
{ graph_clear( win );
  draw_title( win );
  draw_labels( win );
  if (show_hist) draw_hist( win );
  draw_perc_plot( win );
  draw_activ_plot( win );
  if (show_traces) draw_traces( win );
  flip_cross_hair( win );
  draw_cross_hair_info( win );
}


void point( win, x, y )
window win;
int x, y;
{ flip_cross_hair( win );
  last_point_x = x;
  last_point_y = y;
  flip_cross_hair( win );
  draw_cross_hair_info( win );
}


void click( win, x1, y1, x2, y2, but )
window win;
int x1, y1, x2, y2;
int but;
{ int b = LABEL_THEIGHT+HIST_THEIGHT+PERC_THEIGHT;
  if (y1 > b)
  { int z1 = x1-BORDER_LEFT;
    int z2 = x2-BORDER_LEFT;
    if (z1<0) z1=0; else if (z1>plot_width) z1=plot_width;
    if (z2<0) z2=0; else if (z2>plot_width) z2=plot_width;
    if (z1>z2) { int temp = z1; z1 = z2; z2 = temp; }
    compute_plot( min_time+duration*z1/plot_width, min_time+duration*z2/plot_width+1 );
    compute_hist();
    draw_all( win );
  }
  else
  { int z1 = x1-BORDER_LEFT;
    int z2 = x2-BORDER_LEFT;
    if ((z1>=0) && (z1<=full_plot_width) && (z2>=0) && (z2<=full_plot_width))
    { int aa, aaa, bb, bbb, temp;
      int i, j, n;
      n = 0;
      for (i=0; i<log_nb_events; i++)
      { int event_num = xlate[i];
        if (perc_width[event_num] != 0) n++;
      }
      if (z1>z2) { temp = z1; z1 = z2; z2 = temp; }
      aa = (z1*n)/full_plot_width;
      bb = (z2*n)/full_plot_width;
      j = 0;
      for (i=0; i<log_nb_events; i++)
      { int event_num = xlate[i];
        if (perc_width[event_num] != 0)
        { if (aa == j) aaa = i;
          if (bb == j) bbb = i;
          j++;
        }
      }
      temp = xlate[aaa]; xlate[aaa] = xlate[bbb]; xlate[bbb] = temp;
      flip_cross_hair();
      graph_box( win, WHITE, 0, 0, BORDER_LEFT+BORDER_RIGHT+full_plot_width, b);
      draw_labels( win );
      if (show_hist) draw_hist( win );
      draw_perc_plot( win );
      flip_cross_hair();
      draw_cross_hair_info( win );
    }
  }
}


void print_plot( win, send_to_printer )
window win;
int send_to_printer;
{ char command[100];
  char postscript_file[100];
  char *p1, *p2;
  if (send_to_printer) p1 = "gsx"; else p1 = log_filename;
  p2 = postscript_file;
  while (*p1 != '\0') *p2++ = *p1++;
  p1 = ".ps";
  while (*p1 != '\0') *p2++ = *p1++;
  *p2 = '\0';
  graph_postscript_begin( win, postscript_file, send_to_printer );
  draw_all( win );
  graph_postscript_end( win );
  if (send_to_printer)
  { sprintf( command, "lpr %s", postscript_file );
    if (system( command ) < 0)
      fprintf( stderr, "print failed\n" );
  }
}


int key( win, c )
window win;
char c;
{ if ((c == 'q') || (c == 'Q')) return 1;
  else if ((c == 'r') || (c == 'R')) draw_all( win );
  else if ((c == 'o') || (c == 'O')) print_plot( win, 0 );
  else if ((c == 'p') || (c == 'P')) print_plot( win, 1 );
  else if ((c == 'h') || (c == 'H'))
  { show_hist = !show_hist;
    graph_resize( win, win->w, THEIGHT );
  }
  else if ((c == 't') || (c == 'T'))
  { show_traces = !show_traces;
    if (show_traces) plot_height /= 2; else plot_height *= 2;
    graph_resize( win, win->w, THEIGHT );
  }
  else if ((c == 'n') || (c == 'N'))
  { long time_spanned = CLOCK_TICKS_PER_TIME_UNIT_NUM;
    while (time_spanned < duration) time_spanned *= 2;
    plot_width = full_plot_width*duration/time_spanned;
    compute_plot( min_time, max_time );
    compute_hist();
    draw_all( win );
  }
  else if ((c >= '0') && (c <= '9'))
  { smooth_passes = c-'0';
    compute_plot( min_time, max_time );
    compute_hist();
    draw_all( win );
  }
  else if ((c == 'i') || (c == 'I'))
  { plot_width = full_plot_width;
    smooth_passes = 0;
    compute_plot( 0, log_max_time );
    compute_hist();
    draw_all( win );
  }
  else if ((c == 'z') || (c == 'Z'))
  { long middle = (min_time+max_time)/2;
    compute_plot( middle-duration, middle+duration );
    compute_hist();
    draw_all( win );
  }
  return 0;
}


void resize( win )
window win;
{ int new_width = win->w - BORDER_RIGHT - BORDER_LEFT;
  if (new_width < MIN_PLOT_WIDTH) new_width = MIN_PLOT_WIDTH;
  else if (new_width > MAX_PLOT_WIDTH) new_width = MAX_PLOT_WIDTH;
  if (new_width != full_plot_width)
  plot_width = new_width * plot_width / full_plot_width;
  full_plot_width = new_width;
  compute_plot( min_time, max_time );
  compute_hist();
}


void main( argc, argv )
int argc;
char **argv;
{ int i;
  window win;
  int arg;
  char *exec = "";
  int show_log = 0;

#ifdef MAC
  int ac = 2;
  char *av[] = { "gsx", "gsi.elog", NULL };
  char filename[256];
  SFReply reply;
  static Point where = { 90, 82 };
  MaxApplZone();
  InitGraf(&thePort);
  InitFonts();
  FlushEvents(everyEvent, 0);
  InitWindows();
  InitMenus();
  TEInit();
  InitDialogs(0L);
  InitCursor();
#else
  int ac = argc;
  char **av = argv;
#endif

  arg = 1;
  while ((arg < ac) && (av[arg][0] == '-'))
  { if (av[arg][1] == 'e')
      exec = &av[arg][2];
    else if (av[arg][1] == 'w')
      ;
    else if (av[arg][1] == 'h')
      ;
    else if (av[arg][1] == 'b')
      graph_black_and_white = 1;
    else if (av[arg][1] == 'l')
      show_log = 1;
    else if (av[arg][1] == 'p')
      graph_display = 0;
    arg++;
  }

  if (!(arg < ac))
  { fprintf( stderr, "usage: gsx [options] event_log_file\n\n" );
    fprintf( stderr, "OPTIONS: -b (b&w); -p (postscript only); -wNNN (width NNN); -hNNN (height NNN)\n" );
    fprintf( stderr, "         -eXXX (exec cmds XXX); -l (show log)\n\n" );
    fprintf( stderr, "COMMANDS:\n\n" );
    fprintf( stderr, "keyboard: q -- quit\n" );
    fprintf( stderr, "          r -- redraw window\n" );
    fprintf( stderr, "          o -- generate postscript on <event_log_file>.ps\n" );
    fprintf( stderr, "          p -- print window on postscript printer (with lpr)\n" );
    fprintf( stderr, "          n -- normalize time scale\n" );
    fprintf( stderr, "          0..9 -- pass through smoothing filter 'n' times\n" );
    fprintf( stderr, "          h -- toggle event duration histograms on/off\n" );
    fprintf( stderr, "          t -- toggle per processor traces on/off\n" );
    fprintf( stderr, "          i -- revert to initial display\n" );
    fprintf( stderr, "          z -- zoom out by factor of 2\n\n" );
    fprintf( stderr, "mouse: to zoom in, use click&drag to select section of interest\n" );
    fprintf( stderr, "       to change state ordering, use click&drag on state names to swap states\n" );
    exit( -1 );
  }

  graph_init();

  log_filename = av[arg];

  for (i=0; i<MAX_NB_EVENT_TYPES; i++)
  { long *p = (long *)malloc( (long)sizeof(long) * (MAX_PLOT_WIDTH+1) );
    if (p == NULL)
    { fprintf( stderr, "memory overflow\n" );
      exit( -1 );
    }
    count[i] = p;
  }

  full_plot_width = graph_screen_width - BORDER_RIGHT - BORDER_LEFT;
  plot_height = DEFAULT_PLOT_HEIGHT;

  arg = 1;
  while ((arg < ac) && (av[arg][0] == '-'))
  { if (av[arg][1] == 'w')
      full_plot_width = atoi( &av[arg][2] );
    else if (av[arg][1] == 'h')
      plot_height = (atoi( &av[arg][2] )/16)*16;
    arg++;
  }

  if (show_traces) plot_height /= 2;

  if (full_plot_width < MIN_PLOT_WIDTH) full_plot_width = MIN_PLOT_WIDTH;
  else if (full_plot_width > MAX_PLOT_WIDTH) full_plot_width = MAX_PLOT_WIDTH;

#ifdef MAC
  again:
  SFGetFile( where, "\pEvent log file", 0L, -1, 0L, 0L, &reply );
  if (reply.good)
  { int i, len;
    len = reply.fName[0];
    for (i=0; i<len; i++) filename[i] = reply.fName[i+1];
    filename[i] = '\0';
    av[arg] = filename;
  }
  else
    exit( 0 );
#endif

  plot_width = full_plot_width;
  smooth_passes = 9;

  read_log( av[arg], show_log );

  if (log_nb_traces > 64)
  { bar_height  = 3;
    bar_spacing = 1;
  }
  else if (log_nb_traces > 32)
  { bar_height  = 6;
    bar_spacing = 2;
  }
  else
  { bar_height  = 9;
    bar_spacing = 3;
  }

  win = graph_window( log_filename,
                      full_plot_width+BORDER_RIGHT+BORDER_LEFT,
                      THEIGHT,
                      draw_all,
                      point,
                      click,
                      key,
                      resize );

  compute_plot( 0, log_max_time );
  compute_hist();

  draw_all( win );

  while (*exec != '\0') if (key( win, *exec++ )) exit( 0 );

  graph_idle( win );

#ifdef MAC
  goto again;
#endif

}
