/*  -*- Mode: C;  -*-
 * File: xvaux.c
 * Author: Heinz Schmidt (hws@ICSI.Berkeley.EDU)
 * Copyright (C) International Computer Science Institute, 1990, 1991
 **~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ** FUNCTION: Interface to Xview from Sather & Eiffel.
 **
 ** SUMMARY:  sxv_initialize, to_xv_, sxv_cb_adr 
 ** 
 ** RELATED PACKAGES:
 ** This is on open door to Xview. We export only a the few generic
 ** functions and all the properties.  In Sather/Eiffel there are ways
 ** to encapsulate and protect that. But we hope this way, later
 ** extensions can be made on the level of Sather.
 **
 ** cc -I$OPENWINHOME/include -c xvaux.c
 **
 ** HISTORY:
 ** Last edited: Jan  3 15:30 1992 (andrew)
 ** Created: Sun Oct 14 21:10:18 1990
 **~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 */

#include <X11/Xlib.h>

#include <xview/xview.h>
#include <xview/frame.h>
#include <xview/window.h>

#include <xview/canvas.h>
#include <xview/scrollbar.h>

#include <xview/panel.h>
#include <xview/notice.h>

#include <xview/openwin.h>
#include <xview/openmenu.h>

#include <xview/xv_xrect.h>
#include <xview/font.h>
#include <xview/fullscreen.h>
#include <xview/tty.h>
#include <xview/textsw.h> 
#include <xview/cursor.h>
 
#include <xview/defaults.h> 

#define MAIN_MENU_KEY     121
#define BUTTON_WINDOW_KEY 122
#define BUTTON_MENU_KEY   123

/*--------------------------------------------------------------------------
                                 Notices
--------------------------------------------------------------------------*/

/* extern char **to_c_strar(); */

extern char *to_c_str();


void sxv_confirm(msg, panel, event)
     char **msg;                                /* string array  */
     Xv_opaque panel;                           /* panel or frame  */
     Event *event;
{

  if ( event == NULL ) 
    { notice_prompt(panel, NULL,
		    NOTICE_MESSAGE_STRINGS_ARRAY_PTR, msg,
		    NOTICE_BUTTON, "OK", 100, 
		    NULL
		    );
    }
  else
    { notice_prompt(panel, NULL,
		    NOTICE_MESSAGE_STRINGS_ARRAY_PTR, msg,
		    NOTICE_FOCUS_XY,        event_x(event), event_y(event),
		    NOTICE_BUTTON, "OK", 100, 
		    NULL
		    );
    };
};

int sxv_yesorno(msg, panel, event)
     char **msg;                                /* string array    */
     Xv_opaque panel;                           /* panel or frame  */
     Event *event;
{
  int result;

  if ( event == NULL ) 
    { result = notice_prompt(panel, NULL,
			     NOTICE_MESSAGE_STRINGS_ARRAY_PTR, msg,
			     NOTICE_BUTTON_YES,      "Do It",
			     NOTICE_BUTTON_NO,       "Cancel", 
			     NULL
			     );
    }
  else
    { result = notice_prompt(panel, NULL,
			     NOTICE_MESSAGE_STRINGS_ARRAY_PTR, msg,
			     NOTICE_FOCUS_XY,        event_x(event), event_y(event),
			     NOTICE_BUTTON_YES,      "Do It",
			     NOTICE_BUTTON_NO,       "Cancel", 
			     NULL 
			     );
    };
  return result;
};

int sxv_query(msg, panel, event, butts)
     char **msg, **butts;                       /* string arrays */
     Xv_opaque panel;                           /* panel or frame  */
     Event *event;
{
  int result, butt[7], i;
  char *nxbut, *bs[7];
  
  nxbut = butts[0];
  
  if ( nxbut == NULL )           /* OOPS, show that the notice was setup in a strange way */
    { sxv_confirm(msg, panel, event);
      result=NOTICE_FAILED;
    }
  else
    {
      /* init butts; a NULL but will terminate the argument list to notice_prompt */
      for (i = 0; i == 7; i++) butt[i] = NULL;
      while ( nxbut != NULL )
	{ 
	  butt[i] = NOTICE_BUTTON;
	  bs[i] = nxbut; 
	  i++;
	  nxbut = butts[i];
	  };

      if ( event == NULL ) 
	{ result = notice_prompt(panel, NULL,
				 NOTICE_MESSAGE_STRINGS_ARRAY_PTR, msg,
				 butt[0], bs[0], 1,
				 butt[1], bs[1], 2,
				 butt[2], bs[2], 3,
				 butt[3], bs[3], 4,
				 butt[4], bs[4], 5,
				 butt[5], bs[5], 6,	
				 butt[6], bs[6], 7,
				 NULL
				 );
	}
      else
	{ result = notice_prompt(panel, NULL,
				 NOTICE_MESSAGE_STRINGS_ARRAY_PTR, msg,
				 NOTICE_FOCUS_XY, event_x(event), event_y(event),
				 butt[0], bs[0], 1,
				 butt[1], bs[1], 2,
				 butt[2], bs[2], 3,
				 butt[3], bs[3], 4,
				 butt[4], bs[4], 5,
				 butt[5], bs[5], 6,	
				 butt[6], bs[6], 7,
				 NULL 
				 );
	};
    };
  return result;
};

/*--------------------------------------------------------------------------
                            Call backs
--------------------------------------------------------------------------*/
/* Single call maybe simpler but requires merging the different parameter list 
 * and other funneling here and then additional dispatch on the Sather side.
 * So let Sather ask for the address of its own or an xvaux callback
 * when it has to register callbacks. Only the nececessary callbacks
 * need to go to Sather. Some can be handled by Xview default procedures
 * some can be handled here.
 *
 * The following table is used to initialize Sather constants that point to 
 * functions. The values will be passed in calls of xv_create and xv_set to
 * register callbacks. A table element can be a Sather routine (extern above) or
 * a routine here, that handles the event and then calls the corresponding 
 * Sather routine. This decision can be made without Sather noticing.
 * We use this technique for some callbacks that would otherwise have to interpret
 * events. We use event access macros here to pass the proper/interesting values to
 * the Sather callback in these cases.
 * 
 */

/* The callbacks that we expect Sather to have via .sather (c_name) option. */

/* Application error handling not supported, default behavior seems ok.
 *  In case we want to support: xv_init(XV_ERROR_PROC,s_error_proc,...)
 * 
 *extern int
 *        s_error_proc(),
 *        s_x_error_proc();
 */

void sxv_window_button_pop_window_proc();

extern void
        s_window_repaint_proc(),
        s_window_resize_proc(),
        s_frame_done_proc(),

        s_menu_done_proc(),

        s_split_window_proc(),
        s_split_destroy_window_proc(),

/*      s_menu_gen_proc(),         */  
        s_menu_notify_proc(),/* unify notify's maybe semantic via keys */
        s_menu_pin_proc(),   /* HANDLE completely here?   */
        s_menu_action_proc(),      

        s_panel_background_proc(),
/*      s_panel_repaint_proc(),    we use s_window_repaint_proc  */
    
        s_panel_action_proc(),
        s_panel_value_action_proc(),
        s_panel_list_notify_proc(), /* sorry have broken naming scheme, 
				     * xview's return parameters DON'T include the index of the selection,
				     * this is an annoying omission 
				     */

        s_scrollbar_compute_scrollbar_proc(),
        s_scrollbar_normalize_proc(),
   
        s_win_event_proc(),   
        s_win_notify_event_proc(),
        s_win_notify_immediate_event_proc(),

        s_pointer_button_proc(),
        s_pointer_drag_proc(),

        s_typein_key_proc(),
        
        exit();

/* Events we handle here */

void sxv_win_event_proc(),
     s_menu_pin_proc();

/* later ---**     sxv_menu_notify_proc(),
     sxv_panel_notify_proc();
*/

/* Callback table for Sather*/
void (*sather_call_back_[18])() = /* 17 => 18 by andrew */
{
        s_window_repaint_proc, 
        s_window_resize_proc,
        s_frame_done_proc, 

	s_menu_done_proc,
/*      s_menu_gen_proc,         NOT SUPPORTED */
        s_menu_notify_proc,      /* HANDLE here later */
        s_menu_pin_proc,         /* HANDLE here later */
        s_menu_action_proc,

        s_split_destroy_window_proc,

	s_split_window_proc,
   

        s_panel_background_proc,

	sxv_window_button_pop_window_proc,
        s_panel_action_proc,
        s_panel_value_action_proc,
        s_panel_list_notify_proc, /* added by andrew */

        s_scrollbar_compute_scrollbar_proc,
        s_scrollbar_normalize_proc,
   
        sxv_win_event_proc,   
        s_win_notify_event_proc,
        /* s_win_notify_safe_event_proc, */
        /* s_win_notify_immediate_event_proc, */

 /*     s_pointer_button_proc,
        s_pointer_drag_proc,
        s_typein_key_proc,
  */
	};    

int sxv_cb_adr_(cb)                            /* called during setup.           */
     int cb;                                   /* cb: code of callback procedure */
{ return (int)sather_call_back_[cb]; }

/*--------------------------------------------------------------------------*
 *                        Event handling here                               *
 *--------------------------------------------------------------------------*/

/*                         Event filter setup                               */

void s_menu_pin_proc(menu)
Xv_menu menu;
{
}

void sxv_win_consume_ascii_events(window)
Xv_window window;
{
  xv_set(window,	 
         WIN_EVENT_PROC,sxv_win_event_proc,
         WIN_CONSUME_EVENTS,WIN_ASCII_EVENTS,NULL,NULL);
}

void sxv_win_consume_mouse_events(window)
Xv_window window;
{
  xv_set(window,WIN_EVENT_PROC,sxv_win_event_proc,
	 WIN_CONSUME_EVENTS,LOC_DRAG,WIN_MOUSE_BUTTONS,NULL,NULL);
}

void sxv_win_consume_right_keys(window)
Xv_window window;
{
  xv_set(window,WIN_EVENT_PROC,sxv_win_event_proc,
                WIN_CONSUME_EVENTS,WIN_RIGHT_KEYS,NULL,NULL);
}

void sxv_win_ignore_up_ascii_events(window)
Xv_window window;
{
  xv_set(window,WIN_IGNORE_EVENTS,WIN_UP_ASCII_EVENTS,NULL,NULL);
}

void sxv_win_ignore_up_events(window)
Xv_window window;
{
  xv_set(window,WIN_IGNORE_EVENTS,WIN_UP_EVENTS,NULL,NULL);
}

/*--------------------------------------------------------------------------
                     Handling main menu and window buttons
--------------------------------------------------------------------------*/

void sxv_install_main_menu(window, menu)
     Xv_window window;
     Menu menu;
{
  xv_set(window,
         WIN_CONSUME_EVENTS,  ACTION_MENU, NULL,
         WIN_EVENT_PROC,      sxv_win_event_proc,
         XV_KEY_DATA,         MAIN_MENU_KEY, menu,
	 
	 NULL);  
};

void sxv_pop_main_menu_proc(window,event)
     Xv_window window;
     Event *event;
{
  Menu menu = (Menu)xv_get(window,XV_KEY_DATA,MAIN_MENU_KEY);
  if ( menu == 0 ) 
    { 
      /* printf ("XVAUX: No menu associated with window %i\n", (int) window); */
    }
  else
    menu_show(menu, window, event, NULL);
};

void sxv_window_button_pop_window_proc(button)
     Xv_opaque button;
{
  Frame f = (Frame)xv_get(button,XV_KEY_DATA,BUTTON_WINDOW_KEY,NULL);
  xv_set(f,XV_SHOW,TRUE,NULL);

};

/*--------------------------------------------------------------------------
                              Window Events
--------------------------------------------------------------------------*/
/* We rely on the fact that there is ONE pointing device from which pointer events
 * emanate. All of them go through here. We combine them to multi-clicks if appropriate.
 *
 * Before drag we need to send the button event, so that application can select the right
 * function.
 * 
 * Clicks are down-then-up events on the same position AND with time delta < delta_click
 * between down and up.
 * A Multi-click is a sequence of clicks on almost the same position AND with 
 * time delta < delta_click between successive clicks.
 * 
 * The following code integrates such event sequences in the following way:
 * it buffers down events. The next event determines the continuation where
 * 'flush' means to pass on the buffered event to the application, flush the buffer.
 *
 *   - drag :     flush
 *   - up   :     if second half of click, combine and buffer, else flush and send UP
 *   - down :     if 2n+1 half of multi-click, combine and buffer, else flush and buffer DOWN
 *
 * For simplicity no supports of button chords (simultaneous pressure of multiple buttons).
 *
 * The button state for a single button is passed on as int 0..n.
 * where 0 means down, 1 means up or single click, 2 means double-click and so on.
 * Click integration means that the application usually does not see down (0).
 * If all button changes need to be visible, the application can set delta_click to zero.
 * 
 */ 

#define BUTTON_DOWN 0
#define BUTTON_UP 1

#define MOUSE_LEFT 1
#define MOUSE_MIDDLE 2
#define MOUSE_RIGHT 3

#define SHIFT_MOD 1
#define CONTROL_MOD 2
#define META_MOD 4

int click_event_times;		/* number of times the mouse has been clicked */
struct timeval last_event_time;	/* time at which last event occured */
extern int multi_click_timeout;	/* application defined delta between successive events. */

#define TIME_DIFF(t1,t2) (((t1.tv_sec - t2.tv_sec)*1000000+(t1.tv_usec-t2.tv_usec))/1000)

void sxv_win_event_proc(window,event)
Xv_window window;
Event *event;
{
  int button_state;
  int modifier = 0;
  Menu menu;

  if event_ctrl_is_down(event) modifier+=CONTROL_MOD;
  if event_shift_is_down(event) modifier+=SHIFT_MOD;
  if event_meta_is_down(event) modifier+=META_MOD;
  
  button_state = event_is_up(event);  /* on change, change constants above ... */

/*  printf("Win=%i ->XID%i, paint_win=%i ->XID%i\n",window,xv_get(window,XV_XID),
          xv_get(window,CANVAS_VIEW_PAINT_WINDOW),
             xv_get(xv_get(window,CANVAS_VIEW_PAINT_WINDOW),XV_XID));
*/
  if ( event_is_ascii(event))
    s_ascii_key_proc(window,event_action(event),modifier,button_state,
		     event_x(event),event_y(event));

  else if ( event_action(event) == ACTION_MENU && ( modifier == 0 || modifier == CONTROL_MOD ))
    {   
        sxv_pop_main_menu_proc(window,event);
    }
  else if ( event_action(event) >= KEY_RIGHT(1) && event_action(event) <= KEY_RIGHT(15) )
    {
      s_right_key_proc(window,event_action(event)-KEY_RIGHT(0),modifier,button_state,
		     event_x(event),event_y(event));
    }
  else if ( event_action(event) >= KEY_TOP(1) && event_action(event) <= KEY_TOP(15) )
   {
      s_top_key_proc(window,event_action(event)-KEY_TOP(0),modifier,button_state,
		     event_x(event),event_y(event));
    }
  else
    { 
      switch (event_action(event)) {
      case LOC_DRAG:
      case LOC_MOVE:
	s_pointer_drag_proc(window,event_x(event),event_y(event),modifier);
	break;
      case ACTION_SELECT:
      case MS_LEFT:
	if event_is_down(event) 
	  {
	    if (TIME_DIFF((event_time(event)),last_event_time) < multi_click_timeout)
	      {
		click_event_times++;
		last_event_time = event_time(event);
	      }

            else 
	      {
		click_event_times = 0;
		last_event_time = event_time(event);
		s_pointer_button_proc(window,MOUSE_LEFT,event_x(event),event_y(event),
				      click_event_times,modifier);
	      }
	  }
	else
	  {
	    if (click_event_times >= 2)
	      s_pointer_button_proc(window,MOUSE_LEFT,event_x(event),event_y(event),
				    click_event_times,modifier);
	  }
	break;
      case ACTION_ADJUST:
      case MS_MIDDLE:
	if event_is_down(event) 
	  {
	    if (TIME_DIFF((event_time(event)),last_event_time) < multi_click_timeout)
	      {
		click_event_times++;
		last_event_time = event_time(event);
	      }
            else 
	      {
		click_event_times = 0;
		last_event_time = event_time(event);
		s_pointer_button_proc(window,MOUSE_MIDDLE,event_x(event),event_y(event),
				      click_event_times,modifier);
	      }
	  }
	else
	  {
	    if (click_event_times >= 2)
	      s_pointer_button_proc(window,MOUSE_MIDDLE,event_x(event),event_y(event),
				    click_event_times,modifier);
	  }
	break;
      case ACTION_MENU:
      case MS_RIGHT:
	if event_is_down(event) 
	  {
	    if (TIME_DIFF((event_time(event)),last_event_time) < multi_click_timeout)
	      {
		click_event_times++;
		last_event_time = event_time(event);
	      }
            else 
	      {
		click_event_times = 0;
		last_event_time = event_time(event);
		s_pointer_button_proc(window,MOUSE_RIGHT,event_x(event),event_y(event),
				      click_event_times,modifier);
	      }
	  }
	else
	  {
	    if (click_event_times >= 2)
	      s_pointer_button_proc(window,MOUSE_RIGHT,event_x(event),event_y(event),
				    click_event_times,modifier);
	  }
	break;
      case LOC_WINENTER:
	printf("Entered frame at %d,%d; not passed on to Sather yet\n",
	       event_x(event), event_y(event));
	break;
      case LOC_WINEXIT:
	printf("Exited window at %d,%d not passed on to Sather yet\n",
	       event_x(event), event_y(event));
	break;
      /* reaquire some of the rightpad keys */
      case ACTION_GO_DOCUMENT_START: /* R7 */
	s_right_key_proc(window,7,modifier,button_state,event_x(event),event_y(event));
	break;
      case ACTION_GO_COLUMN_BACKWARD: /* R8 */
	s_right_key_proc(window,8,modifier,button_state,event_x(event),event_y(event));
	break;
      case ACTION_GO_PAGE_BACKWARD: /* R9 */
	s_right_key_proc(window,9,modifier,button_state,event_x(event),event_y(event));
	break;      
      case ACTION_GO_CHAR_BACKWARD: /* R10 */
	s_right_key_proc(window,10,modifier,button_state,event_x(event),event_y(event));
	break;
      case ACTION_GO_LINE_FORWARD: /* R11 */
	s_right_key_proc(window,11,modifier,button_state,event_x(event),event_y(event));
	break;
      case ACTION_GO_CHAR_FORWARD: /* R12 */
	s_right_key_proc(window,12,modifier,button_state,event_x(event),event_y(event));
	break;
      case ACTION_GO_DOCUMENT_END: /* R13 */
	s_right_key_proc(window,13,modifier,button_state,event_x(event),event_y(event));
	break;
      case ACTION_GO_COLUMN_FORWARD: /* R14 */
	s_right_key_proc(window,14,modifier,button_state,event_x(event),event_y(event));
	break;
      case ACTION_GO_PAGE_FORWARD: /* R15 */
	s_right_key_proc(window,15,modifier,button_state,event_x(event),event_y(event));
	break;
      default: 
        /* printf("Unknown event: %d\n", (int)event_action(event)); */
	return;
      }
    };
};

/* This version integrates clicks. It is based on above. */

#define DOUBLE_CLICK 2
#define TRIPLE_CLICK 3
#define QUADRUPLE_CLICK 4

#define pass_on_buffered   s_pointer_button_proc(window, last_button, n_clicks,\
                                                 last_modifier, last_x, last_y) 
                           
int buffered = FALSE;
int n_clicks = 0;

Event *last_event;

int last_button = 0;        /* all of these are in last_event but why not speed up passing on */
int last_modifier = 0;
int last_x = 0;
int last_y = 0;
int last_time = 0;

int delta_click = 100;
int min_delta_between_clicks()
{
 return delta_click;
};
void set_min_delta_between_clicks(d)
int d;
{
 delta_click = d;
};

/* cf. todo txt and integrate when other stuff runs */


/* later ------* void sxv_menu_notify_proc(menu, menu_item)
Menu menu;
Menu_item menu_item;
{
    xv_set(frame,
        FRAME_LABEL,    xv_get(menu_item, MENU_STRING),
        NULL);
};

*/

/*--------------------------------------------------------------------------
                               XView and XLib Macros
--------------------------------------------------------------------------*/

void frame_fit_all_panes(frame)
Xv_opaque frame;
{
  frame_fit_all(frame);
};

GC sxv_default_gc(display) 
Display *display;
{
 return DefaultGC(display,DefaultScreen(display));
};

int xv_display_from_window(w)
Xv_window w;
{
 return((int)XV_DISPLAY_FROM_WINDOW(w));
};

int sxv_black_pixel(display)     /* code of black pixels */
Display *display;
{
  return(BlackPixel(display,DefaultScreen(display)));
};

int sxv_white_pixel(display)     /* code of white pixels */
Display *display;
{
  return(WhitePixel(display,DefaultScreen(display)));
};

/* non-square pixels? tell ratio of x / y */
float sxv_display_xy_ratio(display)
Display *display ;
{
  int scr=DefaultScreen(display);
  return((((double)DisplayHeight(display, scr))/DisplayHeightMM (display, scr))/
	 (((double)DisplayWidth(display, scr))/DisplayWidthMM (display, scr)));
};

/*--------------------------------------------------------------------------
                           Objects and Attributes
--------------------------------------------------------------------------*/
/*
 * Xview constants for class, attribute (name) and attribute value id. 
 * Sather xv constants are mapped at Sather startup to these constants.
 * They are independent of the values here; so that recompilation
 * of this file should do when the values change.
 */

/*
 **
 ** We enter all constants in a table for simplicity and compactness of our code.
 ** We had a two level scheme before with group + member offset = index but
 ** this seems to be more work to keep consistent. 
 ** The scheme now is simple: just have the constants here come in the same order
 ** as those in the xvlow.sa file. And make sure that the xvlow.sa constants
 ** have consecutive values starting from 0 incl.
 ** The Sather file has 'groups' so little changes do not require large edits.
 */

int xv_classes_and_attributes_[471] =  
     /* don't forget to change size when changing initialization...
      * Run  test LOW_ATTR_KEY_TST to check initialization and find size.
      */
{
/*-- Success and failure ------------------------------------*/
   
   XV_ERROR,
   XV_NULL,
   XV_OK,
   
   NULL,
   
/*-- Packages (Classes) -------------------------------------*/



(int)   CANVAS,
(int)   CANVAS_VIEW,
(int)   CURSOR, 
(int)   FONT,
(int)   FRAME,
(int)   FRAME_BASE,
(int)   FRAME_CMD,
(int)   FRAME_PROPS,
(int)   FULLSCREEN,
   
(int)   ICON,
   
(int)   MENU,
(int)   MENU_COMMAND_MENU,
(int)   MENU_CHOICE_MENU,
(int)   MENU_TOGGLE_MENU,
(int)   MENUITEM,

(int)   OPENWIN,
   
(int)   PANEL,
(int)   PANEL_BUTTON,
(int)   PANEL_CHOICE,
(int)   PANEL_ITEM,
(int)   PANEL_LIST,
(int)   PANEL_MESSAGE,
(int)   PANEL_NUMERIC_TEXT,
(int)   PANEL_SLIDER,
(int)   PANEL_GAUGE,
(int)   PANEL_TEXT,
(int)   PANEL_VIEW,
   
(int)   SCREEN,
   
(int)   SCROLLBAR,
   
(int)   SERVER,
(int)   SERVER_IMAGE,
(int)   TEXTSW,
(int)   TEXTSW_VIEW,
(int)   TTY,
(int)   TTY_VIEW,
(int)   WINDOW,
   
/*-- Attributes (XView 1.0 API pp.44--141 for details) ------
 *-- The prefixes tell to which XView class an attribute belongs.
 *  
 *-- Note: some of the attributes below are readonly or have 
 *-- values that would require creation and access routines in the C 
 * -- counterpart.
 */

/* -- Canvas attributes */


(int)   CANVAS_AUTO_EXPAND,
(int)   CANVAS_AUTO_SHRINK,
(int)   CANVAS_CMS_REPAINT,
(int)   CANVAS_FIXED_IMAGE,
(int)   CANVAS_HEIGHT,
(int)   CANVAS_MIN_PAINT_HEIGHT,
(int)   CANVAS_MIN_PAINT_WIDTH,
(int)   CANVAS_NO_CLIPPING,
(int)   CANVAS_NTH_PAINT_WINDOW,
(int)   CANVAS_PAINT_CANVAS_WINDOW,
(int)   CANVAS_PAINTWINDOW_ATTRS,
(int)   CANVAS_REPAINT_PROC,
(int)   CANVAS_RESIZE_PROC,
(int)   CANVAS_RETAINED,
(int)   CANVAS_VIEW_CANVAS_WINDOW,
(int)   CANVAS_VIEW_PAINT_WINDOW,
(int)   CANVAS_VIEWABLE_RECT,
(int)   CANVAS_WIDTH,
(int)   CANVAS_X_PAINT_WINDOW,
      
/* -- Common Attributes */
   


(int)   ATTR_LIST,
   

(int)   XV_BOTTOM_MARGIN,
(int)   XV_FONT,
(int)   XV_HEIGHT,
(int)   XV_LEFT_MARGIN,
(int)   XV_MARGIN,
(int)   XV_RECT,
(int)   XV_RIGHT_MARGIN,
(int)   XV_ROOT,
(int)   XV_SHOW,
(int)   XV_TOP_MARGIN,
(int)   XV_WIDTH,
(int)   XV_X,
(int)   XV_XID,
(int)   XV_Y,
   
/* -- Color Attributes --- skipped */



/* -- Cursor Attributes --- skipped */
   


/* -- Font Attributes --- skipped */
   


/* -- Frame Attributes */
/* PANEL_LAYOUT, */


   FRAME_BACKGROUND_COLOR,
   FRAME_BUSY,
   FRAME_FOREGROUND_COLOR,
   FRAME_INHERIT_COLORS,
   FRAME_LEFT_FOOTER,
   FRAME_NTH_SUBFRAME,
   FRAME_NTH_SUBWINDOW,
   FRAME_RIGHT_FOOTER,
   FRAME_SHOW_HEADER,
   FRAME_SHOW_FOOTER,
   FRAME_SHOW_LABEL,
   FRAME_SHOW_RESIZE_CORNER,
   XV_LABEL,
   FRAME_LABEL,

/* -- Frame Base Attributes */
   
   FRAME_CLOSED,
   FRAME_CLOSED_RECT,
   FRAME_ICON,
   FRAME_NO_CONFIRM,
   
/* -- Frame Command Attributes */
   
   FRAME_CMD_PANEL,
   FRAME_CMD_PUSHPIN_IN,
   /* FRAME_DEFAULT_DONE_PROC,    change not supported */
   FRAME_DONE_PROC,
   
/* -- Fullscreen Attributes --- skipped */
   


/* -- Generic Attributes */
   


   XV_AUTO_CREATE,
   XV_KEY_DATA,
   XV_HELP,               /* XV_HELP_DATA is a macro = XV_KEY_DATA, XV_HELP ! */

   XV_OWNER,
   XV_TYPE,
   XV_SCREEN,
   XV_DISPLAY,

   
/* -- Icon Attributes --- skipped */
   


/* -- Initialization Attributes --- DO NOT SUPPORT, sxv_initialize hides initialization*/
   


/* -- Menu Attributes*/
   



   MENU_ACTION_IMAGE,
   MENU_ACTION_ITEM,
   MENU_APPEND_ITEM,
   MENU_CLASS,
   MENU_COL_MAJOR,
   MENU_DEFAULT,
   MENU_DEFAULT_ITEM,
   MENU_DESCEND_FIRST,
   MENU_IMAGE_ITEM,
   MENU_IMAGES,
   MENU_INSERT,
   MENU_INSERT_ITEM,
   MENU_DONE_PROC,
   MENU_FIRST_EVENT,
   MENU_GEN_PIN_WINDOW,








   MENU_GEN_PROC,
   MENU_GEN_PULLRIGHT_IMAGE,
   MENU_GEN_PULLRIGHT_ITEM,
   MENU_ITEM,
   MENU_LAST_EVENT,
   MENU_NCOLS,
   MENU_NITEMS,
   MENU_NROWS,
   MENU_NOTIFY_PROC,
   MENU_NTH_ITEM,
   MENU_PIN,
   MENU_PIN_PROC,
   MENU_PIN_WINDOW,
   MENU_PULLRIGHT_IMAGE,
   MENU_PULLRIGHT_ITEM,
   MENU_REMOVE,
   MENU_REMOVE_ITEM,
   MENU_REPLACE,
   MENU_REPLACE_ITEM,
   MENU_STRINGS,
   MENU_TITLE_ITEM,
   MENU_TYPE,

   MENU_VALID_RESULT,

/* -- Menu Item Attributes */   



/* -- the ones dropped when compared to refmanual are also menu keys above */


   MENU_ACTION_PROC,
   MENU_FEEDBACK,

   MENU_GEN_PROC_IMAGE,
   MENU_GEN_PROC_ITEM,
   MENU_GEN_PULLRIGHT,


   MENU_IMAGE,

   MENU_INACTIVE,
   MENU_INVERT,


   MENU_PULLRIGHT,
   MENU_RELEASE,
   MENU_RELEASE_IMAGE,
   MENU_SELECTED,
   MENU_STRING,
   MENU_STRING_ITEM,
   MENU_TITLE,

   MENU_VALUE,
   
/*   Notice Attributes */
   


   NOTICE_BUTTON,
   NOTICE_BUTTON_NO,
   NOTICE_BUTTON_YES,
   NOTICE_FOCUS_XY,
   NOTICE_FONT,
   NOTICE_MESSAGE_STRINGS,
   NOTICE_MESSAGE_STRINGS_ARRAY_PTR,

   NOTICE_NO_BEEPING,
   NOTICE_TRIGGER,
   NOTICE_YES,
   NOTICE_NO,

/*   Openwin Attributes */



   OPENWIN_ADJUST_FOR_HORIZONTAL_SCROLLBAR,

   OPENWIN_ADJUST_FOR_VERTICAL_SCROLLBAR,

   OPENWIN_AUTO_CLEAR,
   OPENWIN_HORIZONTAL_SCROLLBAR,

   OPENWIN_NO_MARGIN,
   OPENWIN_NTH_VIEW,
   OPENWIN_NVIEWS,
   OPENWIN_SELECTED_VIEW,
   OPENWIN_SHOW_BORDERS,
   OPENWIN_SPLIT,
   OPENWIN_SPLIT_DESTROY_PROC,
   OPENWIN_SPLIT_DIRECTION,
   OPENWIN_SPLIT_INIT_PROC,
   OPENWIN_SPLIT_POSITION,
   OPENWIN_SPLIT_VIEW,
   OPENWIN_SPLIT_VIEW_START,
   OPENWIN_VERTICAL_SCROLLBAR,
   OPENWIN_VIEW_ATTRS,
   OPENWIN_VIEW_CLASS,

/*   Panel Area Attributes */



   PANEL_ACCEPT_KEYSTROKE,
   PANEL_BACKGROUND_PROC,
   PANEL_CARET_ITEM,
   PANEL_DEFAULT_ITEM,
   PANEL_DIRECTION,
   PANEL_EVENT_PROC,
   PANEL_EXTRA_PAINT_HEIGHT,
   PANEL_EXTRA_PAINT_WIDTH,
   PANEL_FIRST_ITEM,
   PANEL_ITEM_X,
   PANEL_ITEM_X_GAP,
   PANEL_ITEM_Y,
   PANEL_ITEM_Y_GAP,
   PANEL_LAYOUT,
   PANEL_REPAINT_PROC,

/*   Panel Button Item Attributes */

   PANEL_ITEM_MENU,

/*   Panel Message Item Attributes */

   PANEL_LABEL_BOLD, 
   PANEL_HORIZONTAL,
   PANEL_VERTICAL,
   PANEL_ALL,
   PANEL_CURRENT,
   PANEL_NONE,
   PANEL_INVERTED,
   PANEL_MARKED,

/*   Panel Item Generic Attributes */



    /*   panel_accept_keystroke -- cf above*/
    /*   panel_event_proc       -- cf above*/
   PANEL_INACTIVE,
   PANEL_ITEM_COLOR,
   PANEL_ITEM_CLASS,
   PANEL_ITEM_RECT,
   PANEL_LABEL_WIDTH,
   PANEL_LABEL_X,
   PANEL_LABEL_Y,
   PANEL_LABEL_IMAGE,
   PANEL_LABEL_STRING,

   PANEL_NEXT_COL,
   PANEL_NEXT_ITEM,
   PANEL_NEXT_ROW,
   PANEL_NOTIFY_PROC,
   PANEL_PARENT_PANEL,
   PANEL_PAINT,

   PANEL_VALUE_X,
   PANEL_VALUE_Y,
     /* -- xv_show, -- cf above */

/* -- Panel Choice and Toggle Item Attributes */



   PANEL_CHOOSE_NONE,
   PANEL_CHOOSE_ONE,
   PANEL_CLIENT_DATA,
   /* PANEL_CHOICE_FONT, */
   PANEL_CHOICE_IMAGE,
   PANEL_CHOICE_IMAGES,
   PANEL_CHOICE_STRING,
   PANEL_CHOICE_STRINGS,
   PANEL_DEFAULT_VALUE,
   PANEL_DISPLAY_LEVEL,
   PANEL_FEEDBACK,
   PANEL_TOGGLE_VALUE,
   PANEL_VALUE,


   PANEL_CHOICE_NROWS,
   PANEL_CHOICE_NCOLS,
   PANEL_CHOICE_X,
   PANEL_CHOICE_XS,
   PANEL_CHOICE_Y,
   PANEL_CHOICE_YS,


/* -- Panel List Attributes */
   
   /*  not supported unlike manual: PANEL_LIST_CHOOSE_ONE, PANEL_LIST_CHOOSE_NONE   */

  PANEL_LIST_CLIENT_DATA,
  PANEL_LIST_CLIENT_DATAS,
  PANEL_LIST_DELETE,
  PANEL_LIST_DISPLAY_ROWS,
  PANEL_LIST_FONT,
  PANEL_LIST_FONTS,
  PANEL_LIST_GLYPH,
  PANEL_LIST_GLYPHS,
  /* -- not supported unlike manual: PANEL_LIST_HEIGHT, */
  PANEL_LIST_INSERT,
  PANEL_LIST_NROWS,
  PANEL_LIST_ROW_HEIGHT,
  PANEL_LIST_SELECT,
  PANEL_LIST_SELECTED,
  PANEL_LIST_STRING,
  PANEL_LIST_STRINGS,
  PANEL_LIST_WIDTH,

/* panel list feedback */
  PANEL_LIST_OP_SELECT,
  PANEL_LIST_OP_DESELECT,
  PANEL_LIST_OP_VALIDATE,
  PANEL_LIST_OP_DELETE,
  
/* Panel Slider Attributes  */  



   PANEL_GAUGE_WIDTH,
   PANEL_MAX_VALUE,
   PANEL_MIN_VALUE,
   PANEL_SHOW_RANGE,
   PANEL_SHOW_VALUE,
   PANEL_SLIDER_WIDTH,
   PANEL_TICKS,
   PANEL_SLIDER_END_BOXES,


/* Panel Text Item Attributes  */
   


   PANEL_MASK_CHAR,
   PANEL_NOTIFY_LEVEL,
   PANEL_NOTIFY_STRING,
   PANEL_READ_ONLY,
   PANEL_VALUE_STORED_LENGTH,
   PANEL_VALUE_DISPLAY_LENGTH,

   PANEL_VALUE_UNDERLINED,

/* Panel Numeric Text Item Attributes  */
/* All inherited from other above  */




/* Screen Attributes --- skipped */



/* Scrollbar Attributes */
   


   SCROLLBAR_COMPUTE_SCROLL_PROC,

   SCROLLBAR_DIRECTION,
   SCROLLBAR_LAST_VIEW_START,
   SCROLLBAR_MENU,
   SCROLLBAR_NORMALIZE_PROC,
   SCROLLBAR_NOTIFY_CLIENT,
   SCROLLBAR_OBJECT_LENGTH,
   SCROLLBAR_PAGE_LENGTH,
   SCROLLBAR_PIXELS_PER_UNIT,
   SCROLLBAR_SPLITTABLE,
   SCROLLBAR_VIEW_LENGTH,
   SCROLLBAR_VIEW_START,
   SCROLLBAR_HORIZONTAL,
   SCROLLBAR_VERTICAL,

/* Selection Request Attributes --- skipped */
  

 
/* Advanced Selection Attributes --- skipped */
   


/* Server Attributes --- skipped */
   


/*   Server Image Attributes --- skipped */



/*   Text Subwindow Attributes  */
  
  TEXTSW_AGAIN_RECORDING,
  TEXTSW_AUTO_INDENT,
  TEXTSW_AUTO_SCROLL_BY,
  TEXTSW_BLINK_CARET,
  TEXTSW_BROWSING,
  TEXTSW_CHECKPOINT_FREQUENCY,
  TEXTSW_CLIENT_DATA,
  TEXTSW_CONFIRM_OVERWRITE,
  TEXTSW_CONTENTS,
  TEXTSW_CONTROL_CHARS_USE_FONT,
  TEXTSW_DISABLE_CD,
  TEXTSW_DISABLE_LOAD,
  TEXTSW_EDIT_COUNT,
  TEXTSW_FILE,
  TEXTSW_FILE_CONTENTS,
  TEXTSW_FIRST,
  TEXTSW_FIRST_LINE,
  TEXTSW_HISTORY_LIMIT,
  TEXTSW_IGNORE_LIMIT,
  TEXTSW_INSERT_FROM_FILE,
  TEXTSW_INSERT_MAKES_VISIBLE,
  TEXTSW_INSERTION_POINT,
  TEXTSW_LENGTH,
  TEXTSW_LINE_BREAK_ACTION,
  TEXTSW_LOWER_CONTEXT,
  TEXTSW_MEMORY_MAXIMUM,
  TEXTSW_MODIFIED,
  TEXTSW_MULTI_CLICK_SPACE,
  TEXTSW_MULTI_CLICK_TIMEOUT,
  TEXTSW_NOTIFY_PROC,
  TEXTSW_READ_ONLY,
  TEXTSW_STATUS,
  TEXTSW_STORE_CHANGES_FILE,
  TEXTSW_SUBMENU_EDIT,
  TEXTSW_SUBMENU_FILE,
  TEXTSW_SUBMENU_FIND,
  TEXTSW_SUBMENU_VIEW,
  TEXTSW_UPPER_CONTEXT,

/*   TTY Attributes --- skipped */ 



/*   Window Attributes */



/*   win_show -- cf above. */
   WIN_ALARM,
   WIN_ALARM_DATA,
   WIN_BACKGROUND_COLOR,
   WIN_BELOW,
   WIN_BIT_GRAVITY,
   WIN_CMD_LINE,
   WIN_CMS_DATA,
   WIN_CMS_NAME,
   WIN_COLUMN_GAP,
   WIN_COLUMN_WIDTH,
   WIN_COLUMNS,
   WIN_CONSUME_EVENT,
   WIN_CONSUME_EVENTS,
   WIN_CURSOR,
   WIN_DEPTH,
   WIN_DESIRED_HEIGHT,
   WIN_DESIRED_WIDTH,
   WIN_DYNAMIC_VISUAL,
   WIN_EVENT_PROC,
   WIN_FIT_HEIGHT,
   WIN_FIT_WIDTH,
   WIN_FOREGROUND_COLOR,
   WIN_FRAME,
   WIN_GRAB_ALL_INPUT,
   WIN_HORIZONTAL_SCROLLBAR,
   WIN_IGNORE_EVENT,
   WIN_INPUT_MASK,
   WIN_INPUT_ONLY,
   WIN_IS_CLIENT_PANE,
   WIN_KBD_FOCUS,
   WIN_MAP,
   WIN_MENU,
   WIN_MOUSE_XY,
   WIN_TOP_LEVEL_NO_DECOR,  /* different from Xview 1.0 */
   WIN_NO_CLIPPING,
   WIN_NOTIFY_EVENT_PROC,
   WIN_NOTIFY_SAFE_EVENT_PROC,
   WIN_NOTIFY_IMMEDIATE_EVENT_PROC,

   WIN_PARENT,
   WIN_PERCENT_HEIGHT,
   WIN_PERCENT_WIDTH,
   WIN_RECT,
   WIN_RETAINED,
   WIN_RIGHT_OF,
   WIN_ROW_GAP,
   WIN_ROW_HEIGHT,
   WIN_ROWS,
   WIN_SCREEN_RECT,
   WIN_SET_FOCUS,
   WIN_TOP_LEVEL,
   WIN_TRANSPARENT,
   WIN_VERTICAL_SCROLLBAR,
   WIN_WINDOW_GRAVITY,
   WIN_X_COLOR_INDICES,

/* Miscellaneous attribute values */



   NorthWestGravity,
   SouthWestGravity,

/* panel_item_paint values
 *  Xlib GC values
 *  some standard fonts using xv_find calls? 
 */
/* Sxv constants  */

  MAIN_MENU_KEY,
  BUTTON_WINDOW_KEY,
  BUTTON_MENU_KEY,

  BUTTON_DOWN, 
  BUTTON_UP,

  DOUBLE_CLICK,
  TRIPLE_CLICK,
  QUADRUPLE_CLICK,
 
  MOUSE_LEFT, 
  MOUSE_MIDDLE, 
  MOUSE_RIGHT, 


  SHIFT_MOD, 
  META_MOD, 
  CONTROL_MOD,

/* XLib constants */
/* graphics functions, as in GC.alu */

  GXclear,
  GXand,
  GXandReverse,
  GXcopy,
  GXandInverted,
  GXnoop,
  GXxor,
  GXor,
  GXnor,
  GXequiv,
  GXinvert,
  GXorReverse,
  GXcopyInverted,
  GXorInverted,
  GXnand,
  GXset,
  
  /* LineStyle */

  LineSolid,
  LineOnOffDash,
  LineDoubleDash,

  /* capStyle */

  CapNotLast,
  CapButt,
  CapRound,
  CapProjecting,

  /* joinStyle */

  JoinMiter,
  JoinRound,
  JoinBevel,
  
  /* fillStyle */

  FillSolid,
  FillTiled,
  FillStippled,
  FillOpaqueStippled,

  /* fillRule */

  EvenOddRule,
  WindingRule,

  /* subwindow mode */

  ClipByChildren,
  IncludeInferiors,
  
  /* SetClipRectangles ordering */

  Unsorted,
  YSorted,
  YXSorted,
  YXBanded,
  
  /* CoordinateMode for drawing routines */

  CoordModeOrigin,
  CoordModePrevious,
  
  /* Polygon shapes */

  Complex,
  Nonconvex,
  Convex,

  /* Arc modes for PolyFillArc */

  ArcChord,
  ArcPieSlice,
  
  /* GC components: masks used in CreateGC, CopyGC, ChangeGC, OR'ed into
   GC.stateChanges */

  GCFunction,
  GCPlaneMask,
  GCForeground,
  GCBackground,
  GCLineWidth,
  GCLineStyle,
  GCCapStyle,
  GCJoinStyle,
  GCFillStyle,
  GCFillRule,
  GCTile,
  GCStipple,
  GCTileStipXOrigin,
  GCTileStipYOrigin,
  GCFont,
  GCSubwindowMode,
  GCGraphicsExposures,
  GCClipXOrigin,
  GCClipYOrigin,
  GCClipMask,
  GCDashOffset,
  GCDashList,
  GCArcMode,
  
  GCLastBit

};

int to_xv_(key)
int key;
{ 
return xv_classes_and_attributes_[key];
};

/* Lowlevel test to figure whether attributes have the right position in table. */
/* Comment later */

void test_attr(name,there,here)
int here,there;
{
  printf(name);
  if ( here == there ) printf(" %d=%d ok\n",there,here);
  else 
    printf(" wrong %d != %d *****--------\n",there,here);
};

void test_some_keys(fullscreen,window,
		    canvas_auto_expand,canvas_x_paint_window,
		    xv_width,frame_label,
		    f_done_proc,
		    xv_auto_create,xv_type,
		    menu_default_item,menu_done_proc,menu_value,
		    notice_button_yes,notice_trigger,
		    openwin_nviews,panel_event_proc,panel_first_item,
		    panel_repaint_proc,panel_label_string,panel_value,
		    panel_choice_ys,panel_list_width,panel_ticks,
		    scrollbar_direction,scrollbar_view_start,
		    win_alarm,win_consume_events,win_parent,win_x_color_indices,
		    button_menu_key,
		    textsw_file,textsw_contents,textsw_insert_from_file)
int fullscreen,window,
		    canvas_auto_expand,canvas_x_paint_window,
		    xv_width,frame_label,
		    f_done_proc,
		    xv_auto_create,xv_type,
		    menu_default_item,menu_done_proc,menu_value,
		    notice_button_yes,notice_trigger,
		    openwin_nviews,panel_event_proc,panel_first_item,
		    panel_repaint_proc,panel_label_string,panel_value,
		    panel_choice_ys,panel_list_width,panel_ticks,
		    scrollbar_direction,scrollbar_view_start,
		    win_alarm,win_consume_events,win_parent,win_x_color_indices,
                    button_menu_key,
		    textsw_file,textsw_contents,textsw_insert_from_file;
{ 
  test_attr("fullscreen",fullscreen,FULLSCREEN);
  test_attr("window",window,WINDOW);
  test_attr("canvas_auto_expand",canvas_auto_expand,CANVAS_AUTO_EXPAND);
  test_attr("canvas_x_paint_window",canvas_x_paint_window,CANVAS_X_PAINT_WINDOW);
  test_attr("xv_width",xv_width,XV_WIDTH);
  test_attr("frame_label",frame_label,FRAME_LABEL);
  test_attr("frame_done_proc",f_done_proc,FRAME_DONE_PROC);
  test_attr("xv_auto_create",xv_auto_create,XV_AUTO_CREATE);
  test_attr("xv_type",xv_type,XV_TYPE);
  test_attr("menu_default_item",menu_default_item,MENU_DEFAULT_ITEM);
  test_attr("menu_done_proc",menu_done_proc,MENU_DONE_PROC);
  test_attr("menu_value",menu_value,MENU_VALUE);
  test_attr("notice_button_yes",notice_button_yes,NOTICE_BUTTON_YES);
  test_attr("notice_trigger",notice_trigger,NOTICE_TRIGGER);
  test_attr("openwin_nviews",openwin_nviews,OPENWIN_NVIEWS);
  test_attr("panel_repaint_proc",panel_repaint_proc,PANEL_REPAINT_PROC);
  test_attr("panel_event_proc",panel_event_proc,PANEL_EVENT_PROC);
  test_attr("panel_first_item",panel_first_item,PANEL_FIRST_ITEM);
  test_attr("panel_label_string",panel_label_string,PANEL_LABEL_STRING);
  test_attr("panel_value",panel_value,PANEL_VALUE);
  test_attr("panel_list_width",panel_list_width,PANEL_LIST_WIDTH);
  test_attr("panel_ticks",panel_ticks,PANEL_TICKS);
  test_attr("scrollbar_direction", scrollbar_direction, SCROLLBAR_DIRECTION);
  test_attr("scrollbar_view_start",scrollbar_view_start,SCROLLBAR_VIEW_START);
  test_attr("win_alarm",win_alarm,WIN_ALARM);
  test_attr("win_consume_events",win_consume_events,WIN_CONSUME_EVENTS);
  test_attr("win_parent",win_parent,WIN_PARENT);
  test_attr("win_x_color_indices",win_x_color_indices,WIN_X_COLOR_INDICES);
  test_attr("button_menu_key",button_menu_key,BUTTON_MENU_KEY);
  test_attr("textsw_file",textsw_file,TEXTSW_FILE);
  test_attr("textsw_contents",textsw_contents,TEXTSW_CONTENTS);
  test_attr("textsw_insert_from_file",textsw_insert_from_file,TEXTSW_INSERT_FROM_FILE);
  
};

void test_callbacks(repaint_action,split_window_action,menu_item_action,
		    panel_item_action,
		    win_notify_event_action)
int repaint_action,split_window_action,menu_item_action,panel_item_action,
		    win_notify_event_action;
{ 
  test_attr("repaint_action",repaint_action,(int)s_window_repaint_proc);
  test_attr("split_window_action",split_window_action,(int)s_split_window_proc);
  test_attr("menu_item_action",menu_item_action,(int)s_menu_action_proc);
  test_attr("panel_item_action",panel_item_action,(int)s_panel_action_proc);
  test_attr("win_notify_event_action",win_notify_event_action,(int)s_win_notify_event_proc);
};

