/* * Last edited: May 30 17:55 1991 (bilmes) */
/*
** COPYRIGHT NOTICE: This code is provided "AS IS" WITHOUT ANY WARRANTY
** and is subject to the terms of the SATHER LIBRARY GENERAL PUBLIC
** LICENSE contained in the file: "sather/doc/license.txt" of the Sathe
** distribution. The license is also available from ICSI, 1947 Center
** St., Suite 600, Berkeley CA 94704, USA.
** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
**
**  General C support routines for sdb.
**
*/

#include "defs.h"
#include "sdbSupport.h"
#include "param.h"
#include "frame.h"
#include "symtab.h"
#include "value.h"
#include "all_.h"


#include "command.h"

/* include the C names of sather routines we need to call */
#include "c_names_.h"

/* return !0 if i is a des of j in inferior */
extern int lookup_des_table(/* int, int */);
#define inf_IS_A_DES_OF_(i,j) \
  (((1<<(i&(IBITS-1))) & lookup_des_table(j,i/IBITS)) != 0)

/* external sather functions and shared variables */
extern ptr db_lastSatherFileFromCAccess;
extern int db_lastSatherFileLineFromCAccess;
extern ptr dbtable;

int didReadCNameFile; /* used by class CNAME_MAPPING in cname_mapping.sa */
extern int cname_mapping_readCNameFile();
extern ptr cname_mapping_cExp2cSat();

/* structure that may hold any one of a Sather type in the inferior process. */
union infVal {
  char c;
  int i;
  float f;
  double d;
  ptr p;
};



/* not zero if the function the selected frame is a Sather function */
char sather_current_function_p;
/* used to return current object integer type. Only valid after a valid return
 * value from checkIfSathercurrentFunc();
 */
int current_ob_type_;
/*
 * Pointer in inf of the current object for the function at the selected
 * frame. Only valid after a valid return value from checkIfSatherCurrentFunc()
 */
ptr current_ob_;

extern struct cmd_list_element *cmdlist;

extern char *xmalloc ();
extern char *xrealloc ();
extern void print_first_arg();
extern ptr inf_get_current_ob_();

/* sather function calls. */
extern void s_visit_command();
extern void s_leave_command();
extern void s_location_command();
extern void s_show_command();
extern void s_array_command();


/* sather function call hooks */
void
s_visit_command_hook(args,from_tty)
     char *args;
     int from_tty;
{
  int i,j;
  s_visit_command(0,makestr_(args));
}
void
s_leave_command_hook(args,from_tty)
     char *args;
     int from_tty;
{
  s_leave_command(0,makestr_(args));
}
void s_location_command_hook(args,from_tty)
     char *args;
     int from_tty;
{
  s_location_command(0,makestr_(args));
}

void
s_show_command_hook(args,from_tty)
     char *args;
     int from_tty;
{
  s_show_command(0,makestr_(args),0); /* give false flag to show routines */
}
void
s_showr_command_hook(args,from_tty)
     char *args;
     int from_tty;
{
  s_show_command(0,makestr_(args),1); /* give true flag to show Routines */
}
void
s_showc_command_hook(args,from_tty)
     char *args;
     int from_tty;
{
  s_show_command(0,makestr_(args),2); /* give true flag to show Routines */
}
void
s_showa_command_hook(args,from_tty)
     char *args;
     int from_tty;
{
  s_show_command(0,makestr_(args),3); /* give true flag to show Routines */
}

void
s_array_command_hook(args,from_tty)
     char *args;
     int from_tty;
{
  s_array_command(0,makestr_(args));
}
void
s_assign_command_hook(args,from_tty)
     char *args;
     int from_tty;
{
  s_assign_command(0,makestr_(args));
}
void
s_sdvars_command_hook(args,from_tty)
     char *args;
     int from_tty;
{
  s_sdvars_command(0,makestr_(args));
}
void
s_descendent_command_hook(args,from_tty)
     char *args;
     int from_tty;
{
  s_descendent_command(0,makestr_(args));
}

void
sdbInit()
{

  add_cmd("sather", class_sather, 0, "Sather Commands.", &cmdlist);

  add_com("descendents",  class_sather, s_descendent_command_hook,
	  "Show descendent information.\n\
If an argument is given, show descendents of the class specified by the\n\
argument. Otherwise, show all classes and their descendents (this really\n\
can be a prodigious amount of text).");

  add_com("visit",  class_sather, s_visit_command_hook, "\
Visit the object given by the feature specified in the argument.\n\
The feature specified may be a feature of the current object (i.e.\n\
'visit foo' where 'foo' is a feature of 'self'), may be a feature of\n\
a differet object (i.e. 'visit SOMECLASS::bar' where bar is a feature\n\
of class 'SOMECLASS'), may be a feature of a feature (i.e. 'visit foo.bar'\n\
where 'bar' is a feature of 'foo' which is a feature of 'self'), may\n\
be an array spec of an array object (i.e. 'visit [3]' if self is an\n\
array object, or 'visit foo[3]' if 'foo' is an array object of 'self),\n\
or may be a non-basic non-void local variable or argument.");

  add_com("sdvars",  class_sather, s_sdvars_command_hook,
	  "List the current Sather object debugger variables.");

  add_com("leave",  class_sather, s_leave_command_hook, "\
Leave current object being visited.\n\
If a number argument is given, leave that many times.");

  add_com("location", class_sather, s_location_command_hook,
	  "Show the objects in current visit stack.");

  add_com("show", class_sather, s_show_command_hook,"\
Show the types and values of features.\n\
If no argument is given, then show the features of the current\n\
object (not including the array portion, constants, or routines). The current\n\
object may be changed by using the 'visit' and 'leave' commands or by\n\
changing the current stack frame using the 'up' and 'down' commands. If an\n\
argument is given, show the feature specified by the argument. The argument\n\
syntax is the same as for the 'visit' command with the inclusion that basic\n\
local variables or arguments may be shown.");

  add_com("rshow", class_sather, s_showr_command_hook,
	  "Like 'show' but also list routines when given no argument.");

  add_com("cshow", class_sather, s_showc_command_hook,
	  "Like 'show' but also list constants when given no argument.");

  add_com("ashow", class_sather, s_showa_command_hook,
	  "Like 'show' but list all features when given no argument.");

  add_com("array", class_sather, s_array_command_hook,"\
Display the array portion (if any) of the current object.\n\
If no argument is given, default elements of 0 through 1 will be printed\n\
for each dimension. If an argument is given, selective portions of the nD\n\
array may be specified as:\n\
      1D: [range]\n\
      2D: [range,range]\n\
      3D: [range,range,range]\n\
      4D: [range,range,range,range]\n\
      where range is defined by\n\
        range ->         -- default elements\n\
        range -> *       -- all elements\n\
        range -> n       -- element n\n\
        range -> n -     -- element n through the end\n\
        range -> - m     -- beginining element through m\n\
        range -> n-m     -- elements n through m\n\
        range -> range|range -- elements in both ranges.");

  add_com("assign", class_sather, s_assign_command_hook,"\
Assign a value to an attribute or array elements.\n\
The value given by the argument will be assigned to either a Sather attribute\n\
or the specified array elements of the current object. If the current object is\n\
an array, the value may be assigned to the array elements specified by the\n\
same array specification syntax as the 'array' command. For example:\n\
      assign foo 4            -- assign 4 to self.foo\n\
      assign [3,4-5] 3.14159  -- assign pi to the given array elements.");

}

/* call gdb fatal for Sather */
void
cfatal(str)
     ptr str;
{
  fatal(STR2C_(str));
}


/*
 * Lookup entry in the 2D Sather table given which is assumed to
 *  be an int*[];
 *
 */
int
lookup_2d_sather_table(table_addr,table_name,i,j)
     CORE_ADDR* table_addr;
     char *table_name;
     int i;
     int j;
{
  CORE_ADDR element_addr;
  int retVal;
  if (*table_addr == NULL) {
    struct symbol* table_symbol;
    table_symbol = lookup_symbol(table_name,
				 get_selected_block(),VAR_NAMESPACE,
				 0);
    if (table_symbol == NULL) {
      /* should look in misc_function_vector using lookup_misc_func in
	 symtab.c. If it is there, we can at least get the address. */
      int index;
      if ((index = lookup_misc_func (table_name)) == -1) 
	error("Can't find Sather table (%s) Are you sure this is a Sather program?",table_name);
      else {
	*table_addr = misc_function_vector[index].address;
      }
    } else {
      *table_addr = SYMBOL_VALUE(table_symbol);
    }
  }

  if ( read_memory(*table_addr + sizeof(int)*i,
		   &element_addr,
		   sizeof(element_addr))) {
    error("Bad Sather (%s) offset.",table_name);
  }
  if (element_addr == (CORE_ADDR)0) 
    return 0;

  if ( read_memory(element_addr + sizeof(int)*j,
		   &retVal,
		   sizeof(retVal))) {
    error("Bad Sather (%s) table element address offset.",table_name);
  }
  return retVal;
}

/*
 * lookup entry in the Sather attribute table (attr_table_) of the process
 * being debugged. The table is assumed to be an int*[].
 */
int
lookup_attr_table(i,j)
     int i;
     int j;
{
  static CORE_ADDR attr_table_addr = NULL;
  return lookup_2d_sather_table(&attr_table_addr,"attr_table_",i,j);
}

/*
 * Lookup entry in the Sather feature table which is assumed to
 *  be an int*[];
 *
 */
int
lookup_feat_table(i,j)
     int i;
     int j;
{
  static CORE_ADDR feat_table_addr = NULL;
  return lookup_2d_sather_table(&feat_table_addr,"feat_table_",i,j);
}


/*
 * Lookup entry in the Sather des table which is assumed to
 *  be an int*[];
 *
 */
int
lookup_des_table(i,j)
     int i;
     int j;
{
  static CORE_ADDR des_table_addr = NULL;
  return lookup_2d_sather_table(&des_table_addr,"des_table_",i,j);
}


/*
 * Lookup entry in the 1D Sather table given which is assumed to
 *  be an int[];
 *
 */
int
lookup_1d_sather_table(table_addr,table_name,i)
     CORE_ADDR* table_addr;
     char *table_name;
     int i;
{
  int retVal;
  if (*table_addr == NULL) {
    struct symbol* table_symbol;
    table_symbol = lookup_symbol(table_name,
				 get_selected_block(),VAR_NAMESPACE,
				 0);
    if (table_symbol == NULL) {
      /* should look in misc_function_vector using lookup_misc_func in
	 symtab.c. If it is there, we can at least get the address. */
      int index;
      if ((index = lookup_misc_func (table_name)) == -1) 
	error("Can't find Sather table (%s) Are you sure this is a Sather program?",table_name);
      else {
	*table_addr = misc_function_vector[index].address;
      }
    } else {
      *table_addr = SYMBOL_VALUE(table_symbol);
    }
  }

  if ( read_memory(*table_addr + sizeof(int)*i,
		   &retVal,
		   sizeof(retVal))) {
    error("Bad Sather (%s) offset.",table_name);
  }
  return retVal;
}

/*
 * Lookup entry in the Sather dispatch_table which is assumed to
 *  be an int[];
 *
 */
int
lookup_dispatch_table(i)
     int i;
{
  static CORE_ADDR dispatch_table_addr = NULL;
  return lookup_1d_sather_table(&dispatch_table_addr,"dispatch_table_",i);
}

/*
 * lookup the global symbol 'str' in the debugged program, and return the
 * integer value at its address.
 * This routine assumes that the symbol must exist for the debugger to
 * continue, and will exit the program if the symbol is not found.
 */
int
lookup_symbol_int(p)
     ptr p;
{
  struct symbol* sym = NULL;
  int retVal;
  char *str;
  str = STR2C_(p);
  sym = lookup_symbol(str,
		      get_selected_block(),VAR_NAMESPACE,
		      0);
  if (sym == NULL)
    fatal("Can't find Sather symbol (%s)",str);

  if ( read_memory(SYMBOL_VALUE(sym),
		   &retVal,
		   sizeof(retVal))) {
    char buff[1024];
    sprintf(buff,"Can't find Sather symbol (%s) at address (0x%X)",
	    str,SYMBOL_VALUE(sym));
    fatal(buff);
  }
  return retVal;
}


/*
 *  lookup the ctype Sather value at address given by argument.
 */
union infVal
lookup_ctype(vaddr,ctype)
     CORE_ADDR vaddr; /* the variable address */
     int ctype;
{
  union infVal retVal;
  void* addr;
  int size;

  switch (ctype) {
     case CTYPE_PTR_: 
       addr = &retVal.p;
       size = sizeof(ptr);
       break;
     case CTYPE_CHAR_:
       addr = &retVal.c;
       size = sizeof(char);
       break;
     case CTYPE_INT_:
       addr = &retVal.i;
       size = sizeof(int);
       break;
     case CTYPE_FLOAT_:
       addr = &retVal.f;
       size = sizeof(float);
       break;
     case CTYPE_DOUBLE_:
       addr = &retVal.d;
       size = sizeof(double);
       break;
     default:
       break;
  }

  if ( read_memory(vaddr,
		   addr,
		   size)) {
    fprintf(stderr,"Can't read mem address (0x%X)\n",(int)vaddr);
    return_to_top_level();
  }
  return retVal;
}


/*
 *  write the ctype Sather value at address given by argument.
 */
void
write_ctype(vaddr,ctype,retVal)
     CORE_ADDR vaddr; /* the variable address to write value*/
     int ctype; /* type of value */
     union infVal retVal;
{
  void* addr;
  int size;

  switch (ctype) {
     case CTYPE_PTR_: 
       addr = &retVal.p;
       size = sizeof(ptr);
       break;
     case CTYPE_CHAR_:
       addr = &retVal.c;
       size = sizeof(char);
       break;
     case CTYPE_INT_:
       addr = &retVal.i;
       size = sizeof(int);
       break;
     case CTYPE_FLOAT_:
       addr = &retVal.f;
       size = sizeof(float);
       break;
     case CTYPE_DOUBLE_:
       addr = &retVal.d;
       size = sizeof(double);
       break;
     default:
       break;
  }

  if (write_memory((int)vaddr,addr,size)) {
    fprintf(stderr,"Can't write mem address (0x%X)\n",(int)vaddr);
    return_to_top_level();
  }
}



/* return a freeable string starting at addr from the inferior process */
char *
getStrFromInf(addr) 
     CORE_ADDR addr;
{
  char *strVal,*strValp;
  int localSize = 32;
  int sizeSoFar=0;

  /* read memory in int chunks until we reach the end of the string */
  strValp = strVal = xmalloc(localSize);
  /* a similar loop to this is done in valprint.c, line 545 */
  while (1) {
    int i;
    if (read_memory(addr+sizeSoFar,strValp,sizeof(int))) {
      fprintf(stderr,"Can't read mem address for (0x%X)\n",addr);
      return_to_top_level();
    }

    /* find a NULL */
    for (i=0;i<sizeof(int);i++) {
      if (strValp[i] == '\0') {
	return strVal;
      }
    }
    sizeSoFar += sizeof(int);
    if (sizeSoFar >= localSize) {
      localSize *= 2;
      strVal = xrealloc(strVal,localSize);
      strValp = &strVal[sizeSoFar];
    } else
      strValp += sizeof(int);
  }
}


/*
 * lookup the global symbol 'str' in the debugged program, and return the
 * Sather string value at the address it points to.
 *
 */
ptr
lookup_symbol_str(p)
     ptr p;
{
  struct symbol* sym = NULL;
  int valueOfSymbol;
  char *str,*rc;
  ptr res;

  str = STR2C_(p);  /* get C string */

  sym = lookup_symbol(str,
		      get_selected_block(),VAR_NAMESPACE,
		      0);
  if (sym == NULL)
    fatal("Can't find Sather symbol (%s)",str);

  /* get value at symbol location */
  if ( read_memory(SYMBOL_VALUE(sym),
		   &valueOfSymbol,
		   sizeof(valueOfSymbol))) {
    char buff[1024];
    sprintf(buff,"Can't find Sather symbol (%s) at address (0x%X)",
	    str,SYMBOL_VALUE(sym));
    fatal(buff);
  }
  /* valueOfSymbol now contains address of beginnging of string */
  if (valueOfSymbol == NULL)
    return NULL;

  /* a similar loop to this is done in valprint.c, line 545 */
  rc = getStrFromInf(valueOfSymbol);
  res = makestr_(rc);
  free (rc);
  return res;

}


/*
 * Support routines for DMIRROR class:
 */

/* return array dimension size of ob */
int inf_arr_d_size_(p,d)
     ptr p;
     int d;
{
  int obDim;
  if (d < 1 || d > 4)
    return 0;
  obDim = inf_ob_arr_dim_(inf_TYPE_(p));
  if (d > obDim)
    return 0;
  switch (d) {
    case 1:
      return inf_ARRD1_(p);
      break;
    case 2:
      return inf_ARRD2_(p);
      break;
    case 3:
      return inf_ARRD3_(p);
      break;
    case 4:
      return inf_ARRD4_(p);
      break;
    default:
      break;
    }
  return 0;
}


/* given an array entry, return string form */
ptr inf_strVal_(p,iattr,lastIndex,typ)
     ptr p;
     int iattr,lastIndex;
     int typ;
{
  char buff[100];
  int saType = inf_cl_arr_satype_(typ);

  if (saType == CHAR_ici || inf_IS_A_DES_OF_(saType,CHAR_ici)) {
    char cvalue;
    cvalue = inf_CATT_(p,iattr+SC_*lastIndex);
    if (cvalue > ' ' && cvalue <=  '~') /* assuming ascii */
      sprintf(buff,"%c",cvalue);
    else
      sprintf(buff,"C%u",(unsigned int)cvalue);
  } else if (saType == INT_ici || inf_IS_A_DES_OF_(saType,INT_ici))
    sprintf(buff,"%d",inf_IATT_(p,iattr+SI_*lastIndex));
  else if (saType == BOOL_ici || inf_IS_A_DES_OF_(saType,BOOL_ici))
    sprintf(buff,"%s",inf_CATT_(p,iattr+SC_*lastIndex) ? "T" : "F");
  else if (saType == REAL_ici || inf_IS_A_DES_OF_(saType,REAL_ici))
    sprintf(buff,"%f",inf_FATT_(p,iattr+SF_*lastIndex));
  else if (saType == DOUBLE_ici || inf_IS_A_DES_OF_(saType,DOUBLE_ici))
    sprintf(buff,"%f",inf_DATT_(p,iattr+SD_*lastIndex));
  else if (inf_PATT_(p,iattr+SP_*lastIndex) == NULL)
    sprintf(buff,"<void>");
  else if (saType == STR_ici) {
    ptr tmp;
    char *str = getStrFromInf(inf_PATT_(p,iattr+SP_*lastIndex)+2*SI_);
    int len = strlen(str);
    /* strip off last new line */
    if (str[len-1] == '\n')
      str[len-1] = '\0';
    tmp = makestr_(str);
    free(str);
    return tmp;
  } else {
    ptr tmp = inf_PATT_(p,iattr+SP_*lastIndex);
    sprintf(buff,"<0x%X>",(int)tmp);
  }
  return makestr_(buff);
}

/* given an array entry, return its address */
ptr inf_arrAddr_(p,iattr,lastIndex,typ)
     ptr p;
     int iattr,lastIndex;
     int typ;
{
  char buff[100];
  int saType = inf_cl_arr_satype_(typ);

  if (saType == CHAR_ici || inf_IS_A_DES_OF_(saType,CHAR_ici))
    return p+iattr+SC_*lastIndex;
  else if (saType == INT_ici || inf_IS_A_DES_OF_(saType,INT_ici))
    return p+iattr+SI_*lastIndex;
  else if (saType == BOOL_ici || inf_IS_A_DES_OF_(saType,BOOL_ici))
    return p+iattr+SC_*lastIndex;
  else if (saType == REAL_ici || inf_IS_A_DES_OF_(saType,REAL_ici))
    return p+iattr+SF_*lastIndex;
  else if (saType == DOUBLE_ici || inf_IS_A_DES_OF_(saType,DOUBLE_ici))
    return p+iattr+SD_*lastIndex;
  else 
    return p+iattr+SP_*lastIndex;
}


/* return value of array in string form, p !MUST! point to a non void
   array object */
ptr inf_arr_str_val_(p,i,j,k,l)
     ptr p;
     int i,j,k,l;
{
  int type_p = inf_TYPE_(p);
  int dim;

  /*  checked by caller
  if (inf_ob_arr_dim_(type_p) == 0)
    return NULL; */
  switch (inf_ob_arr_dim_(type_p)) {
    case 1:
      return inf_strVal_(p,inf_ARR1_(p,i),i,type_p);
      break;
    case 2:
      return inf_strVal_(p,inf_ARR2_(p,i,j),j,type_p);
      break;
    case 3:
      return inf_strVal_(p,inf_ARR3_(p,i,j,k),k,type_p);
      break;
    case 4:
      return inf_strVal_(p,inf_ARR4_(p,i,j,k,l),l,type_p);
      break;
    default:
      break;
    }
  return NULL;
}


/* return value of array element in F_OB form if it is not basic.
   p *MUST* be a non void array object */
ptr inf_arr_val_(p,i,j,k,l)
     ptr p;
     int i,j,k,l;
{
  int type_p = inf_TYPE_(p);

  /* (checked by caller) 
  if (inf_arr_type_is_basic_(0,inf_TYPE_(p)))
      return NULL; */
  switch (inf_ob_arr_dim_(type_p)) {
    case 1:
      if (i<0 || i> inf_ARRD1_(p)) return NULL;
      return inf_PATT_(p,inf_ARR1_(p,i)+SP_*i);
      break;
    case 2:
      if (i<0 || j<0 || i> inf_ARRD1_(p) || j>inf_ARRD2_(p)) return NULL;
      return inf_PATT_(p,inf_ARR2_(p,i,j)+SP_*j);
      break;
    case 3:
      if (i<0 || j<0 || k<0 || i>inf_ARRD1_(p) || j>inf_ARRD2_(p) || k>inf_ARRD3_(p)) 
	return NULL;
      return inf_PATT_(p,inf_ARR3_(p,i,j,k)+SP_*k);
      break;
    case 4:
      if (i<0 || j<0 || k<0 || l<0 || 
	  i>inf_ARRD1_(p) || j>inf_ARRD2_(p) || k>inf_ARRD3_(p) || l>inf_ARRD4_(p))
	return NULL;
      return inf_PATT_(p,inf_ARR4_(p,i,j,k,l)+SP_*l);
      break;
    default:
      break;
    }
  return NULL;
}


/* return address of array element in F_OB form if it is not basic.
   p *MUST* be a non void array object */
ptr inf_arr_addr_(p,i,j,k,l)
     ptr p;
     int i,j,k,l;
{
  int type_p = inf_TYPE_(p);

  switch (inf_ob_arr_dim_(type_p)) {
    case 1:
      if (i<0 || i> inf_ARRD1_(p)) return NULL;
      return inf_arrAddr_(p,inf_ARR1_(p,i),i,type_p);
      break;
    case 2:
      if (i<0 || j<0 || i> inf_ARRD1_(p) || j>inf_ARRD2_(p)) return NULL;
      return inf_arrAddr_(p,inf_ARR2_(p,i,j),j,type_p);
      break;
    case 3:
      if (i<0 || j<0 || k<0 || i>inf_ARRD1_(p) || j>inf_ARRD2_(p) || k>inf_ARRD3_(p)) 
	return NULL;
      return inf_arrAddr_(p,inf_ARR3_(p,i,j,k),k,type_p);
      break;
    case 4:
      if (i<0 || j<0 || k<0 || l<0 || 
	  i>inf_ARRD1_(p) || j>inf_ARRD2_(p) || k>inf_ARRD3_(p) || l>inf_ARRD4_(p))
	return NULL;
      return inf_arrAddr_(p,inf_ARR4_(p,i,j,k,l),l,type_p);
      break;
    default:
      break;
    }
  return NULL;
}


/* given an array entry, convert string form to value */
/* array entries must be basic */
char inf_setStrVal_(p,iattr,lastIndex,str)
     ptr p;
     int iattr,lastIndex;
     char* str;
{
  union infVal allVal;
  int saType = inf_cl_arr_satype_(inf_TYPE_(p));

  if (saType == CHAR_ici || inf_IS_A_DES_OF_(saType,CHAR_ici)) {
    if (sscanf(str,"%c",&allVal.c) == 0)
      return 0;
    write_ctype(p+iattr+SC_*lastIndex,CTYPE_CHAR_,allVal);
  } else if (saType == INT_ici || inf_IS_A_DES_OF_(saType,INT_ici)) {
    if (sscanf(str,"%i",&allVal.i) == 0)
      return 0;
    write_ctype(p+iattr+SI_*lastIndex,CTYPE_INT_,allVal);
  } else if (saType == BOOL_ici || inf_IS_A_DES_OF_(saType,BOOL_ici)) {
    if (sscanf(str,"%c",&allVal.c) == 0)
      return 0;
    if (allVal.c == 'T')
      allVal.c = 1;
    else if (allVal.c == 'F')
      allVal.c = 0;
    else
      return 0;
    write_ctype(p+iattr+SC_*lastIndex,CTYPE_CHAR_,allVal);
  } else if (saType == REAL_ici || inf_IS_A_DES_OF_(saType,REAL_ici)) {
    if (sscanf(str,"%f",&allVal.f) == 0)
      return 0;
    write_ctype(p+iattr+SF_*lastIndex,CTYPE_FLOAT_,allVal);
  } else if (saType == DOUBLE_ici || inf_IS_A_DES_OF_(saType,DOUBLE_ici)) {
    if (sscanf(str,"%lf",&allVal.d) == 0)
      return 0;
    write_ctype(p+iattr+SD_*lastIndex,CTYPE_DOUBLE_,allVal);
  }
  return 1;
}



/* set array element to value given by object */
char 
inf_set_arr_val_(p,i,j,k,l,val)
     ptr p;
     int i,j,k,l;
     ptr val;
{
  int type_p = inf_TYPE_(p);
  int dim;
  CORE_ADDR addr;  

  if (inf_ob_arr_dim_(type_p) == 0)
    return 0;
  if (inf_arr_type_is_basic_(0,inf_TYPE_(p)))
      return 0;
  switch (inf_ob_arr_dim_(type_p)) {
    case 1:
      addr = inf_ARR1_(p,i)+SP_*i;
      break;
    case 2:
      addr = inf_ARR2_(p,i,j)+SP_*j;
      break;
    case 3:
      addr = inf_ARR3_(p,i,j,k)+SP_*k;
      break;
    case 4:
      addr = inf_ARR4_(p,i,j,k,l)+SP_*l;
      break;
    default:
      break;
    }
  write_ctype(p+addr,CTYPE_PTR_,val);
  return 1;
}


/* set array element to value given by string. p *MUST* be a 
   non void array object */
char inf_set_arr_str_val_(p,i,j,k,l,str)
     ptr p;
     int i,j,k,l;
     ptr str;
{
  int type_p = inf_TYPE_(p);
  int dim;
  /* (checked in caller)
  if (inf_ob_arr_dim_(type_p) == 0)
    return 0; */
  if (!inf_arr_type_is_basic_(0,inf_TYPE_(p)))
      return 0;
  switch (inf_ob_arr_dim_(type_p)) {
    case 1:
      return inf_setStrVal_(p,inf_ARR1_(p,i),i,STR2C_(str));
      break;
    case 2:
      return inf_setStrVal_(p,inf_ARR2_(p,i,j),j,STR2C_(str));
      break;
    case 3:
      return inf_setStrVal_(p,inf_ARR3_(p,i,j,k),k,STR2C_(str));
      break;
    case 4:
      return inf_setStrVal_(p,inf_ARR4_(p,i,j,k,l),l,STR2C_(str));
      break;
    default:
      break;
    }
  return 1;
}


/* return true if type given by 'ici' is basic type  */
char
type_is_basic_(ici)
     int ici;
{
  return (ici == CHAR_ici   || 
	  ici == INT_ici    || 
	  ici == BOOL_ici   || 
	  ici == REAL_ici   || 
	  ici == DOUBLE_ici ||
	  ici == FOB_ici ||      /* treat F_OB as basic so we won't deref it */
	  inf_IS_A_DES_OF_(ici,CHAR_ici) ||
	  inf_IS_A_DES_OF_(ici,INT_ici)  ||
	  inf_IS_A_DES_OF_(ici,BOOL_ici) ||
	  inf_IS_A_DES_OF_(ici,REAL_ici) ||
	  inf_IS_A_DES_OF_(ici,DOUBLE_ici) ||
	  inf_IS_A_DES_OF_(ici,FOB_ici));
}


/* return true if feature f of ci is a basic type */
char 
inf_f_basic_(ci,f)
     int ci,f;
{
  int ici = inf_cl_feat_satype_(ci,f);
  return type_is_basic_(ici);
}

/* return value of non-basic feature f of object in F_OB format.
 */
ptr inf_f_val_(p,type_p,f)
     ptr p;
     int type_p;
     int f;
{
  int loc;
  int category;
  if (inf_f_basic_(type_p,f))
    return NULL; /* can't be basic */

  if ((category = inf_cl_feat_cat_(type_p,f)) == F_ROUTINE)
    return NULL; /* can't be a routine */

  loc = inf_get_dispatch_(0,type_p,inf_cl_feat_name_(type_p,f));
  if (category == F_ATTRIBUTE) {
    if (p != NULL) {
      return inf_PATT_(p,loc); /* loc is offset */
    }else
      return NULL;
  } else {
    /* loc is address of global variable */
    union infVal val;
    val = lookup_ctype(loc,CTYPE_PTR_);
    return val.p;
  }
}

/* return core address of feature 'f' of object in F_OB format.
 */
ptr inf_f_addr_(p,type_p,f)
     ptr p;
     int type_p;
     int f;
{
  int loc;
  int category;

  if ((category = inf_cl_feat_cat_(type_p,f)) == F_ROUTINE)
    return NULL; /* can't be a routine */

  loc = inf_get_dispatch_(0,type_p,inf_cl_feat_name_(type_p,f));
  if (category == F_ATTRIBUTE) {
    if (p != NULL) {
      return (p+loc); /* loc is offset */
    } else
      return NULL;
  } else /* loc is address of global variable */
    return (ptr)loc;
}


/* return sather STR from object given by p.
   p could be the address of a basic object (i.e. an int) and if so,
   return a STR form of its value */
ptr inf_str_val_(p,saType)
     ptr p;
     int saType;
{
  char buff[100];
  union infVal val;
  if (saType == CHAR_ici || inf_IS_A_DES_OF_(saType,CHAR_ici)) {
    val = lookup_ctype(p,CTYPE_CHAR_);
    if (val.c > ' ' && val.c <=  '~') /* assuming ascii */
      sprintf(buff,"'%c'",val.c);
    else
      sprintf(buff,"CHAR%u",(unsigned int)val.c);
  } else if (saType == INT_ici || inf_IS_A_DES_OF_(saType,INT_ici)) {
    val = lookup_ctype(p,CTYPE_INT_);
    sprintf(buff,"%i",val.i);
  } else if (saType == BOOL_ici || inf_IS_A_DES_OF_(saType,BOOL_ici)) {
    val = lookup_ctype(p,CTYPE_CHAR_);
    sprintf(buff,"%c",val.c ? 'T' : 'F');
  } else if (saType == REAL_ici || inf_IS_A_DES_OF_(saType,REAL_ici)) {
    val = lookup_ctype(p,CTYPE_FLOAT_);
    sprintf(buff,"%f",val.f);
  } else if (saType == DOUBLE_ici || inf_IS_A_DES_OF_(saType,DOUBLE_ici)) {
    val = lookup_ctype(p,CTYPE_DOUBLE_);    
    sprintf(buff,"%f",val.d);
  } else if (saType == FOB_ici || inf_IS_A_DES_OF_(saType,FOB_ici)) {
    val = lookup_ctype(p,CTYPE_PTR_);    
    sprintf(buff,"0x%X",val.p);
  } else {
    /* check if null */
    if (p == NULL)
      sprintf(buff,"<void>");
    else if (saType == STR_ici) {
      char* tmp = getStrFromInf(& (((int*)p)[2]));
      int len = strlen(tmp);
      ptr p;
      /* strip off last new line */
      if (tmp[len-1] == '\n')
	tmp[len-1] = '\0';
      p = makestr_(tmp);
      free(tmp);
      return p;
    } else
      sprintf(buff,"<0x%X>",(int)p);
  }
  return makestr_(buff);
}


/* return sather STR form of attribute given by feature num i of p */
ptr inf_f_str_val_(p,type_p,i)
     ptr p;
     int type_p;
     int i;
{
  int loc;
  int category;
  void *tmp;
  char buff[100];
  int saType;
  union infVal val;
  
  if ((category = inf_cl_feat_cat_(type_p,i)) == F_ROUTINE)
    return makestr_("<NOVAL>"); /* can't be a routine */
  loc = inf_get_dispatch_(0,type_p,inf_cl_feat_name_(type_p,i));
  if (category == F_ATTRIBUTE) {
    if (p != NULL)
      tmp = (void*) (p+loc); /* loc is offset */
    else
      return makestr_("");
  } else
    tmp = (void*) loc; /* loc is address of global variable */

  saType = inf_cl_feat_satype_(type_p,i);
  if (saType == CHAR_ici || inf_IS_A_DES_OF_(saType,CHAR_ici)) {
    val = lookup_ctype(tmp,CTYPE_CHAR_);
    if (val.c > ' ' && val.c <=  '~') /* assuming ascii */
      sprintf(buff,"'%c'",val.c);
    else
      sprintf(buff,"CHAR%u",(unsigned int)val.c);
  } else if (saType == INT_ici || inf_IS_A_DES_OF_(saType,INT_ici)) {
    val = lookup_ctype(tmp,CTYPE_INT_);
    sprintf(buff,"%i",val.i);
  } else if (saType == BOOL_ici || inf_IS_A_DES_OF_(saType,BOOL_ici)) {
    val = lookup_ctype(tmp,CTYPE_CHAR_);
    sprintf(buff,"%c",val.c ? 'T' : 'F');
  } else if (saType == REAL_ici || inf_IS_A_DES_OF_(saType,REAL_ici)) {
    val = lookup_ctype(tmp,CTYPE_FLOAT_);
    sprintf(buff,"%f",val.f);
  } else if (saType == DOUBLE_ici || inf_IS_A_DES_OF_(saType,DOUBLE_ici)) {
    val = lookup_ctype(tmp,CTYPE_DOUBLE_);    
    sprintf(buff,"%f",val.d);
  } else {
    /* check if null */
    val = lookup_ctype(tmp,CTYPE_PTR_);
    if (val.p == NULL)
      sprintf(buff,"<void>");
    else if (saType == STR_ici) {
      char* tmptmp = getStrFromInf(val.p+2*SI_);
      int len = strlen(tmptmp);
      ptr p;
      /* strip off last new line */
      if (tmptmp[len-1] == '\n')
	tmptmp[len-1] = '\0';
      p = makestr_(tmptmp);
      free(tmptmp);
      return p;
    } else
      sprintf(buff,"<0x%X>",(int)val.p);
  }
  return makestr_(buff);
}

/* set feature given by feature number f of ob to value */
/* represented by val, or return false if bad format */
char inf_set_f_str_val_(p,type_p,f,val)
     ptr p;
     int type_p;
     int f;
     ptr val;
{
  int loc;
  int category;
  void *tmp;
  char buff[100];
  int saType;
  char *str = STR2C_(val);
  union infVal allVal;
  
  if (!inf_f_basic_(type_p,f))
    return 0; /* must be basic */
  if ((category = inf_cl_feat_cat_(type_p,f)) == F_ROUTINE)
    return 0; /* can't be a routine */
  loc = inf_get_dispatch_(0,type_p,inf_cl_feat_name_(type_p,f));
  if (category == F_ATTRIBUTE) {
    if (p != NULL)
      tmp = (void*) (p+loc); /* loc is offset */
    else
      return 0;
  } else
    tmp = (void*) loc; /* loc is address of global variable */

  saType = inf_cl_feat_satype_(type_p,f);
  if (saType == CHAR_ici || inf_IS_A_DES_OF_(saType,CHAR_ici)) {
    if (sscanf(str,"%c",&allVal.c) != 1)
      return 0;
    write_ctype(tmp,CTYPE_CHAR_,allVal);
  } else if (saType == INT_ici || inf_IS_A_DES_OF_(saType,INT_ici)) {
    if (sscanf(str,"%i",&allVal.i) != 1)
      return 0;
    write_ctype(tmp,CTYPE_INT_,allVal);
  } else if (saType == BOOL_ici || inf_IS_A_DES_OF_(saType,BOOL_ici)) {
    if (*str == 'T')
      allVal.c = 1;
    else if (*str == 'F')
      allVal.c = 0;
    else
      return 0;
    write_ctype(tmp,CTYPE_CHAR_,allVal);    
  } else if (saType == REAL_ici || inf_IS_A_DES_OF_(saType,REAL_ici)) {
    if (sscanf(str,"%f",&allVal.f) != 1)
      return 0;
    write_ctype(tmp,CTYPE_FLOAT_,allVal);    
  } else if (saType == DOUBLE_ici || inf_IS_A_DES_OF_(saType,DOUBLE_ici)) {
    if (sscanf(str,"%lf",&allVal.d) != 1)
      return 0;
    write_ctype(tmp,CTYPE_DOUBLE_,allVal);
  } else
    return 0;
}

/* if not basic, set feature f of ob to val */
char inf_set_f_val_(ob,type_p,f,val)
     ptr ob;
     int type_p;
     int f;
     ptr val;
{
  int category;
  ptr* tmp;
  int loc;

  if (inf_f_basic_(type_p,f))
    return 0; /* can't be basic */
  if ((category = inf_cl_feat_cat_(type_p,f)) == F_ROUTINE)
    return NULL; /* can't be a routine */
  loc = inf_get_dispatch_(0,type_p,inf_cl_feat_name_(type_p,f));
  if (category == F_ATTRIBUTE) {
    if (ob != NULL)
      write_ctype(ob+loc,CTYPE_PTR_,val);
    else
      return 0;
  } else
     write_ctype(loc,CTYPE_PTR_,val);
  return 1;
}


int
inf_ob_type_(p)
     ptr p;
{
  return inf_TYPE_(p);
}


/* return the sather class integer index given a string form of the
 * function, or return -1 if not the proper format.
 * This function assumes sather numbers start at the 4th position in the symbol.
 * NOTE: This indirectly uses the information in
 * the c_names.h file produced by the compiler since it redefines 
 * the sather created C names to user defined exported C names. 
 */
int
satherClassFromFunction(str)
     char *str;
{
  int len;
  char *p,*pp;
  int val;
  ptr sat_str;
  if (didReadCNameFile == 0)
    cname_mapping_readCNameFile();

  if (str == NULL || *str == '\0')
    return -1;

  if ((sat_str = cname_mapping_cExp2cSat(makestr_(str))) != NULL)
    str = STR2C_(sat_str); /* we've hit a name that's been mapped */

  if ( (len=strlen(str)) <= 3 )
    if ((sat_str = cname_mapping_cExp2cSat(makestr_(str))) != NULL)
      /* we've hit a name that's been mapped */
      return satherClassFromFunction(STR2C_(sat_str));
    else
      return -1;

  /* make sure last char is a '_' */
  if (str[len-1] != '_')
    return -1;

  p = &str[3];
  val = strtol(p,&pp,10);
  if (pp == p || *pp != '_') /* no integer can be formed or no '_' char */
    if ((sat_str = cname_mapping_cExp2cSat(makestr_(str))) != NULL)
      /* we've hit a name that's been mapped */
      return satherClassFromFunction(STR2C_(sat_str));
    else
      return -1;

  return val;
}

int
inf_get_current_ob_type_()
{
  return current_ob_type_;
}

/*
 * get the name and int value of the first argument (if any)
 * of the current function being debugged in the stack frame.
 */

ptr
inf_get_current_ob_()
{
  return current_ob_;
}

/*
** checkIfSatherCurrentFunc:
** Check if the function at the current stack frame is a sather function.
** If it is, set the global variables current_ob_ (taken from the self__) 
** argument of the Sather function, and current_ob_type_ (taken from
** the value of *((int*)curent_ob_) in the inf process core memory
**
*/
int checkIfSatherCurrentFunc()
{
  struct frame_info* fi;
  struct partial_symtab *pst;
  struct symbol* func;
  struct block* b;
  int nsyms = 0; /* number of arguments */
  struct symbol* sym;
  char *funname;
  value val;
  CORE_ADDR addr;
  int i;

  /*
    This is called at the end of up_command.
    print_stack_frame (selected_frame, selected_frame_level, 1);
  */
  sather_current_function_p = 0;
  current_ob_type_ = -1;
  current_ob_ = (ptr)NULL;

  if (!have_inferior_p () && !have_core_file_p ()) {
    return 0;
    /* error ("No inferior process or core file."); */
  }

  fi = get_frame_info(selected_frame);
  if (fi == NULL)
    return 0;
  pst = find_pc_psymtab(fi->pc);
  if (pst && !pst->readin) {
    /* error("No symbol table for function at current frame."); */
    return 0;
  }
  func = find_pc_function(fi->pc);
  if (func)
    {
      /* In certain pathological cases, the symtabs give the wrong
	 function (when we are in the first function in a file which
	 is compiled without debugging symbols, the previous function
	 is compiled with debugging symbols, and the "foo.o" symbol
	 that is supposed to tell us where the file with debugging symbols
	 ends has been truncated by ar because it is longer than 15
	 characters).

	 So lookin the misc_function_vector as well, and if it comes
	 up with a larger address for the function use that instead.
	 I don't think this can ever cause any problems;
	 there shouldn't be any
	 misc_function_vector symbols in the middle of a function.  */
      int misc_index = find_pc_misc_function (fi->pc);
      if (misc_index >= 0
	  && (misc_function_vector[misc_index].address
	      > BLOCK_START (SYMBOL_BLOCK_VALUE (func))))
	{
	  /* We also don't know anything about the function besides
	     its address and name. Therefore, can't get its argument. */
	  funname = misc_function_vector[misc_index].name;
	  /* error("Can't get arguments for current function."); */
	  return 0;
	}
      else
	funname = SYMBOL_NAME (func);
    }
  else
    {
      /* error("Can't find function at current frame."); */
      return 0;
    }

  b = SYMBOL_BLOCK_VALUE (func);
  nsyms = BLOCK_NSYMS (b);
  addr = FRAME_ARGS_ADDRESS(fi);

  for (i=0;i<nsyms;i++) {
    sym = BLOCK_SYM(b,i);
    if (SYMBOL_CLASS (sym) == LOC_REGPARM
		       || SYMBOL_CLASS (sym) == LOC_ARG
		       || SYMBOL_CLASS (sym) == LOC_REF_ARG)
      break; /* we've found first argument */
  }
  if (i==nsyms) {
    /* printf("No arguments for current function.\n"); */
    return 0;
  }

  val = read_var_value(sym,selected_frame);

  current_ob_type_ = satherClassFromFunction(funname);
  if (current_ob_type_ == -1 || strcmp(SYMBOL_NAME(sym),"self__")) {
    current_ob_type_ = -1; /* make sure it is invalid */
    /* error("Current function is not a Sather function."); */
    return 0;
  }

  /*    
  printf("Symbol name at function (%s) is %s\n",funname,SYMBOL_NAME(sym));
  printf("Hex value is 0x%X\n",*(int*)VALUE_CONTENTS(val));
  printf("class number = %d\n",current_ob_type_);
  */
  current_ob_ = (ptr)*(int*)VALUE_CONTENTS(val);

  sather_current_function_p = 1;
  return 1; /* the current function is a Sather */

}


/* Sather calls this routine */
int
inf_get_dispatch__(ci,name,inf_dispatch_table_size)
     int ci,name,inf_dispatch_table_size;
{
  unsigned int hsh,key;
  unsigned int firsthsh;

  key=(((unsigned int)name)<<14)+ci;
  hsh=(((key*key)%inf_dispatch_table_size)>>1)<<1;
  firsthsh = hsh;
  while(1)
    {
      if (lookup_dispatch_table(hsh)==0) 
	error("Bad inferior dispatch table.");
      if (lookup_dispatch_table(hsh)==key) 
	return(lookup_dispatch_table(hsh+1));
      hsh+=2;
      if(hsh>=inf_dispatch_table_size) 
	hsh=0;
      if (firsthsh == hsh) 
	error("Couldn't find key in inferior dispatch table");
    }
}

/* return a Sather string version of the hex value of p */
ptr
ptrHexVal(p)
     ptr p;
{
  char buff[1024];
  sprintf(buff,"0x%X",(int)p);
  return makestr_(buff);
}


/*
getLocalOrArgsVal:
 This routine returns the value of either a local variable
 or an argument for the current sdb frame. The name of the local or argument
 is given by the Sather string 'name'. If nothing matches 'name', or some
 other error occurs,  return (CORE_ADDR)-1. 
 If 'printIfBasic' is non-zero and we sucessfully printed
 a local or argument that is a basic type, return (CORE_ADDR)1.
 If the value of the non-basic local or arg is NULL, this routine returns 0. 
 It returns 0 for a NULL local or arg since, currently, the Sather compiler doesn't
 produce any type information for locals and args and there is no way to
 determine the type information.
 Otherwise, return the pointer value of the non-basic local or argument.
*/
CORE_ADDR
getLocalOrArgsVal(satherName,cName,printIfBasic)
   ptr satherName;
   ptr cName;
   int printIfBasic;
{
  struct block *block;
  int nsyms;
  struct symbol *func;
  int i;

  if (!sather_current_function_p)
    return (CORE_ADDR)-1;

  func = get_frame_function(selected_frame);
  if (func == 0)
    error ("No symbol table info available.\n");

  block = SYMBOL_BLOCK_VALUE (func);
  nsyms = BLOCK_NSYMS(block);
  
  for (i =0;i < nsyms; i++) {
    struct symbol *sym;
    sym = BLOCK_SYM(block,i);
    if (/* Symbol is arg */
           SYMBOL_CLASS (sym) == LOC_ARG
	|| SYMBOL_CLASS (sym) == LOC_REF_ARG
	|| SYMBOL_CLASS (sym) == LOC_REGPARM
        /* Symbol is local */
        || SYMBOL_CLASS (sym) == LOC_LOCAL
	|| SYMBOL_CLASS (sym) == LOC_REGISTER
	|| SYMBOL_CLASS (sym) == LOC_STATIC)
      {
	if (!strcmp(STR2C_(cName),SYMBOL_NAME(sym))) {
	  value val = read_var_value(sym,selected_frame);
	  if (TYPE_CODE (VALUE_TYPE (val)) == TYPE_CODE_PTR
	      || TYPE_CODE (VALUE_TYPE (val)) == TYPE_CODE_REF)
	    /* assume that it is a ptr type, dereference to get the type */
	    return *(CORE_ADDR*)VALUE_CONTENTS(val);
	  else if (printIfBasic) {
	    printf("%s",STR2C_(satherName));
	    printf(" = ");
	    value_print (val, stdout, 0, Val_no_prettyprint);
	    printf("\n");
	    return (CORE_ADDR)1;
	  }
	  else
	    break; /* found matching symbol but not 'ptr' so quit */
	}
      }
  }
  return (CORE_ADDR)-1;
}


/* 
  Print the frame arguments for func which is 
  *MUST* be a Sather function frame.
  This function was takend from print_frame_args (printcmd.c)
  and from value_print (valprint.c)
*/
void
printSatherFrameArgs (func, fi, num, stream)
     struct symbol *func;
     struct frame_info *fi;
     int num;
     FILE *stream;
{
  struct block *b;
  int nsyms = 0;
  int first = 1;
  register int i;
  register int last_regparm = 0;
  register struct symbol *lastsym, *sym, *nextsym;
  register value val;

  register CORE_ADDR addr = FRAME_ARGS_ADDRESS (fi);

  if (func)
    {
      b = SYMBOL_BLOCK_VALUE (func);
      nsyms = BLOCK_NSYMS (b);
    }

  for (i = 0; i < nsyms; i++)
    {
      char satherName[1024];
      int symbolNameLength;
      QUIT;
      sym = BLOCK_SYM (b, i);

      if (SYMBOL_CLASS (sym) != LOC_REGPARM
	  && SYMBOL_CLASS (sym) != LOC_ARG
	  && SYMBOL_CLASS (sym) != LOC_REF_ARG)
	continue;

      if (!strcmp(SYMBOL_NAME(sym),"self__"))
	continue; /* don't worry about self parameter */

      /* strip out the ending "__" in the sather argument */
      symbolNameLength = strlen(SYMBOL_NAME(sym));
      strcpy(satherName,SYMBOL_NAME(sym));
      satherName[symbolNameLength-2] = '\0'; /* kill first '_' char */

      /* Print the next arg.  */
      val = read_var_value(sym,FRAME_INFO_ID(fi));

      if (! first)
	fprintf_filtered (stream, "; ");
      else
	fprintf_filtered (stream, "(");

      fputs_filtered (satherName, stream);

      /* try to get the sather type of the local var */
      if (TYPE_CODE (VALUE_TYPE (val)) == TYPE_CODE_PTR
	  || TYPE_CODE (VALUE_TYPE (val)) == TYPE_CODE_REF) {
	/* assume that it is a ptr type, dereference to get the type */
	fputs_filtered (":", stream);
	globals_printSatherTypeAndObject(NULL,*(int*)VALUE_CONTENTS(val));
      } else {
	/* print the value */
	fputs_filtered ("=", stream);
	value_print (val, stream, 0, Val_no_prettyprint);
      }
      first = 0;
    }

  if (!first) /* then printed an argument other than self */
    fprintf_filtered (stream, ")");
}



/* return the value of self (as a value) for a sather function `func`,
   or return NULL if not a sather function.
*/
value
valueOfSelf(func,fi)
     struct symbol *func;
     struct frame_info *fi;
{
  struct block *block;
  int nsyms;
  int i;

  if (func == 0)
    return (value)NULL;

  if (satherClassFromFunction(SYMBOL_NAME(func)) == -1)
    return (value)NULL; /* not a sather function */

  block = SYMBOL_BLOCK_VALUE (func);
  nsyms = BLOCK_NSYMS(block);
  
  for (i =0;i < nsyms; i++) {
    struct symbol *sym;
    sym = BLOCK_SYM(block,i);
    if (/* Symbol is arg */
           SYMBOL_CLASS (sym) == LOC_ARG
	|| SYMBOL_CLASS (sym) == LOC_REF_ARG
	|| SYMBOL_CLASS (sym) == LOC_REGPARM
        /* Symbol is local */
        || SYMBOL_CLASS (sym) == LOC_LOCAL
	|| SYMBOL_CLASS (sym) == LOC_REGISTER
	|| SYMBOL_CLASS (sym) == LOC_STATIC)
      {
	if (!strcmp("self__",SYMBOL_NAME(sym))) {
	  return read_var_value(sym,FRAME_INFO_ID(fi));
	}
      }
  }
  return (value)NULL; /* not a sather function, shouldn't happen */
}

/* return the value of self (as a ptr) for a sather function `func`,
   or return (CORE_ADDR)(-1) if not a sather function.
*/
CORE_ADDR
addrValueOfSelf(func,fi)
     struct symbol *func;
     struct frame_info *fi;
{
  value val;
  val = valueOfSelf(func,fi);
  if (val == NULL)
    return (CORE_ADDR)(-1);
  return *(CORE_ADDR*)VALUE_CONTENTS(val);  
}


/* used by satherFileAndLineChanged and by step_1 to keep track of
   the sather file and line before the 'step' occured. When the newly
   computed satherfile,line is different from these, a sather step has
   happend.
 */
int old_db_lastSatherFileLineFromCAccess;
ptr old_db_lastSatherFileFromCAccess;

/* this function is called by step_1 (infcmd.c) and is used to see if
   the sather file name and or line number has changed. If so, it
   will be considered a successful sather step and will be counted,
   otherwise, stepping will continue. Return (1) if the condition is
   true, or if any abnormal condition occurs (like stepping into
   a function without any symbol information).
*/

int
satherFileAndLineChanged()
{
  struct frame_info *fi;
  struct partial_symtab *pst;
  struct symtab_and_line sal;

  if (db_lastSatherFileLineFromCAccess == -1)
    return 1; /* no reason to continue */
  fi = get_frame_info(selected_frame);
  if (!fi)
    return 1;
  pst = find_pc_psymtab (fi->pc);
  if (pst && !pst->readin)
    /* havn't read in symbol table yet, return 1 */
    return 1;
  sal = find_pc_line (fi->pc, fi->next_frame);
  if (!sal.symtab)
    return 1;
  dbtable_cFileLine2SatherFileLine(dbtable,makestr_(sal.symtab->filename),sal.line);
  if (old_db_lastSatherFileLineFromCAccess !=
          db_lastSatherFileLineFromCAccess ||
      old_db_lastSatherFileFromCAccess != 
      db_lastSatherFileFromCAccess)
    return 1;

  return 0;
}

extern CORE_ADDR step_range_end;

int
satherFileAndLineWillChange()
{
  struct symtab_and_line sal;
  struct partial_symtab *pst;

  if (db_lastSatherFileLineFromCAccess == -1)
    return 1; /* no reason to continue */
  pst = find_pc_psymtab (step_range_end);
  if (pst && !pst->readin)
    /* havn't read in symbol table yet, return 1 */
    return 1;
  sal = find_pc_line (step_range_end,0);
  if (!sal.symtab)
    return 1;
  dbtable_cFileLine2SatherFileLine(dbtable,makestr_(sal.symtab->filename),sal.line);
  if (old_db_lastSatherFileLineFromCAccess !=
          db_lastSatherFileLineFromCAccess ||
      old_db_lastSatherFileFromCAccess != 
      db_lastSatherFileFromCAccess)
    return 1;

  return 0;
}

