;;; -*- Mode: Emacs-Lisp;  -*-
;;; File: sac.el
;;; Author: Heinz Schmidt (hws@ICSI.Berkeley.EDU)
;;; Copyright (C) International Computer Science Institute, 1990, 1991
;;;
;;; COPYRIGHT NOTICE: This code is provided "AS IS" WITHOUT ANY WARRANTY.
;;; It is subject to the terms of the GNU EMACS GENERAL PUBLIC LICENSE
;;; described in a file COPYING in the GNU EMACS distribution or to be obtained
;;; from Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;* FUNCTION: generation of Sather C interface definitions from high-level
;;;*           specifications.
;;;*
;;;* HISTORY:
;;;* Last edited: May 24 00:28 1991 (hws)
;;;*  Apr 14 12:16 1991 (hws): inline no longer supported
;;;*  Apr 14 12:14 1991 (hws): remove dependency from mini-cl-macros
;;;* Created: Wed Dec 19 10:56:32 1990 (hws)
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(provide 'sac)
(require 'll1-sdts)
(require 'mini-cl)

;;;
;;; PARSING
;;;

;;; SAC Syntax directed translation scheme

(defparse sac-constant ("constant" (sac-decl) ";" -> sac-decl ))

(defparse sac-struct 
  ("struct" (sac-name) (sac-slots) "is" "end" ";"  -> (cons sac-name sac-slots)))

;;; "," and ";" separated slots, we preserve the layout for .sa
(defparse sac-slots 
  ( "(" (* slots sac-decl ";") ")" -> (list (normalize-slots slots) slots)))

(defparse sac-decl
  ( (* names sac-name "," 1) ":" type -> (list names type)))

(defparse sac-name 
  (sa-name (c-name-opt) -> (list sa-name (if (eq c-name-opt '+) sa-name c-name-opt))))

(defparse c-name-opt ("/" c-name -> c-name) ("" -> '+))

(defparse sac-function
  ("function" name (sac-slots) (sac-type-opt) "is" (sac-body) "end" ";" ->
   (list* name sac-type-opt sac-body sac-slots)))

(defparse sac-type-opt 
  (":" type -> type)
  ("" -> '+))

(defparse sac-body 
  ( "implemented" -> 'implemented )
  ( (* c-items sac-c-item "" 1) ";" -> c-items))

(defparse sac-c-item 
  ( name -> name )
  ( (lex item forward-string strip-quotes) -> item))

;;; We allow now "," and ";" separated slots like in Sather and normalize
;;; them when translating concrete into abstract syntax.
(defun normalize-slots (slots)
  (if (eq slots ':empty-list) slots
    (let (norm-slots)
      (dolist (slot slots)
	      (let ((names (car slot))
		    (type (cadr slot)))
		(dolist (name names)
			(push (list (car name) (cadr name) type) norm-slots))))
      (reverse norm-slots))))

		  
;;;
;;; Lex -- we rely on Emacs' syntax tables
;;;

(defun strip-quotes (string)
  (car (read-from-string string)))

(defvar sac-syntax-table nil)

(setq sac-syntax-table;; allow to fix it by reloading
      (let ((table (make-syntax-table)))
	(modify-syntax-entry ?/ "." table)
	(modify-syntax-entry ?\\ "\\" table)
	(modify-syntax-entry ?: "." table)
	(modify-syntax-entry ?\; "." table)
	(modify-syntax-entry ?\, "." table)			     
	(modify-syntax-entry ?* "." table)
	(modify-syntax-entry ?+ "." table)
	(modify-syntax-entry ?- "." table)
	(modify-syntax-entry ?= "." table)
	(modify-syntax-entry ?% "." table)
	(modify-syntax-entry ?< "." table)
	(modify-syntax-entry ?> "." table)
	(modify-syntax-entry ?& "." table)
	(modify-syntax-entry ?\| "." table)
	(modify-syntax-entry ?\' "\"" table)
	table))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;
;;; SAC TRANSFORMERS

;;; Some transformers need to position into class C section

(defun into-c-class ()
  (goto-char 
   (save-excursion 
     (cond ((and (re-search-backward "class C is -- constants and functions"
				     nil t)
		 (re-search-forward "end; -- class C" nil t))
	    (beginning-of-line 0)
	    (point))
	   (t (goto-char (point-max))
	      (insert "\nclass C is -- constants and functions\n\n\nend; -- class C\n")
	      (beginning-of-line -1)
	      (point))))))

;;; Collect the names of macros

(defvar *macro-names*)

;;; CONSTANTS

(defun sa-trafo-constant (names sather-type &optional comment)
  (into-c-class)
  (dolist (name names)
	  (let ((sather-name (car name))
		(c-name (cadr name)))
	    ;; produce a 0-ary (constant) function.
	    ;; we continue in the last 'class C' section or create one.
	    (format "  %s:%s is %send;\n" sather-name sather-type 
		    (comment-nl-or-empty comment)))))

(defun c-trafo-constant (names sather-type &optional comment)
  (let ((type (sather-to-c-type sather-type)))
    (dolist (name names)
	    (let ((sather-name (car name))
		  (c-name (cadr name)))
	      (format "%s %s() %s{ return((%s)%s);};\n" 
		      type sather-name (comment-nl-or-empty comment t) 
		      type c-name)))))

(defun macros-trafo-constant (names sather-type &optional ignore)
  (let ((type (sather-to-c-type sather-type)))
    (dolist (name names)
	    (let ((sather-name (car name))
		  (c-name (cadr name)))
	      (push sather-name *macro-names*)
	      (format "(c_macro) \"#define %s() ((%s)%s)\" %s\n"
		      sather-name type c-name sather-name)))))

;;;
;;; FUNCTION TRANSFORMER
;;;

(defun sa-trafo-function (name type body normalized-args args &optional comment)
  (into-c-class)
  (format "   %s(%s)%s is %send;\n" 
	  name 
	  (mapconcat (function format-sa-formal) args "; ")
	  (if (eq type '+) "" (format ":%s" type))
	  (comment-nl-or-empty comment)))

(defun c-trafo-function (name type body args ignore &optional comment) 
  ;;take normalized args
  (if (not (eq body 'implemented))
      (let ((type (if (not (eq type '+)) (sather-to-c-type type) 'void)))
	(list (format "%s %s(%s) %s\n" type name 
		      (mapconcat (function format-actual) args ",")
		      (comment-nl-or-empty comment t))
	      (format "%s%s"
		      (mapconcat (function format-c-formal) args ""))
	      (format "{\n%s;\n };\n"
		      (if (eq type 'void) ;procedure
		      
			  (format "  %s" (format-c-body body args nil))
			(format "  return((%s)(%s))" 
				type
				(format-c-body body args nil))))))))

(defun macros-trafo-function (name type body args ignore1 &optional ignore2) 
  ;;take normalized args 
  (if (not (eq body 'implemented))
      (let ((type (if (not (eq type '+)) (sather-to-c-type type) 'void)))
	(push name *macro-names*)
	(format "(c_macro) \"#define %s(%s) %s\" %s\n"
		name (mapconcat (function format-actual) args ",")
		(if (eq type 'void)
		    (format "{%s}" (format-c-body body args t))
		  (format "(%s)(%s)" type
			  (format-c-body body args t)))
		name))))

(defun format-c-body (body args macrop)
  (mapconcat (function 
	      (lambda (item)
		(cond ((stringp item) item)
		      (t;; substitute item and maybe cast
		       (let ((arg (assq item args)))
			 (if (null arg) (error "Variable %s in body is undefined." item)
			   (if macrop
			       (format "(%s)(%s)" 
				     (sather-to-c-type (caddr arg))
				     (car arg))
			     (format "%s" (car arg)))))))))
	     body ""))

  
;;;
;;; STRUCTS
;;;

(defun sa-trafo-struct (type-spec normalized-slot-spec slot-spec &optional comment)
  ;; produce a constructor and also readers and writers for each slot
  (let* ((sather-type (car type-spec))
	 (type-xfix (downcase (format "%s" sather-type)))
	 (c-type (cadr type-spec))
	 (c-creator (concat "create_" type-xfix))
	 (create-args (mapconcat (function format-sa-formal) slot-spec "; "))
	 (create-actuals (mapconcat (function format-actual) normalized-slot-spec
				 ",")))
    (goto-char (point-max))	
    (append 
     (list
      (format "\nclass %s is -- an overlay class for the foreign C structure %s.\n%s\n"
	      sather-type c-type 
	      (comment-nl-or-empty comment))
      (format "   COB; -- a %s is a C object.\n"
	      sather-type sather-type)
      (format "        -- Do not use new, copy or deep-copy here. The C functions generated\n")
      (format "        -- don't know what to do with Sather-created instances.\n")
      (format "\n-- Constructor --\n")
      (format "   create(%s):%s is\n" create-args "SELF_TYPE")
      (format "            res := C::%s(%s)\n   end;\n" c-creator create-actuals)
      (format "\n-- Readers --\n"))
     (mapcar (function (lambda (slot)
			 (format "   %s:%s is res := C::%s_%s(self) end;\n" 
				 (car slot) (caddr slot) type-xfix (car slot))))
	     normalized-slot-spec)
     (list (format "\n-- Writers --\n"))
     (mapcar (function (lambda (slot)
			 (format "   set_%s(x:%s) is C::set_%s_%s(self,x) end;\n"
				 (car slot) (caddr slot) type-xfix (car slot))))
	     normalized-slot-spec)
     (list
      (format "\nend; -- class %s\n" sather-type)
      (format "\nclass C is -- the supporting C ADT for class %s\n" sather-type)
      (format "\n-- Constructor --\n")
      (format "   %s(%s):$%s is end;\n" c-creator create-args sather-type)
      (format "\n-- Readers --\n"))
     (mapcar (function (lambda (slot)
			 (format "   %s_%s(cob:$%s):%s is end;\n" 
				 type-xfix (car slot) sather-type (caddr slot))))
	     normalized-slot-spec)
     (list (format "\n-- Writers --\n"))
     (mapcar (function (lambda (slot)
			 (format "   set_%s_%s(cob:$%s; x:%s) is end;\n" 
				 type-xfix (car slot) sather-type (caddr slot))))
	     normalized-slot-spec)
     (list "\nend; -- class C\n"))
    ))

(defun c-trafo-struct (type-spec slot-spec ignore &optional comment)
  ;; produce a constructor and also readers and writers for each slot of the 
  ;; normalized slot-spec
  (let* ((sather-type (car type-spec))
	 (type-xfix (downcase (format "%s" sather-type)))
	 (c-type (cadr type-spec))
	 (c-creator (concat "create_" type-xfix))
	 (create-actuals 
	  (mapconcat (function format-actual) slot-spec ",")))
    (if (not (assq sather-type this-file-sather-c-type-alist))
	(push (cons sather-type (format "struct %s *" c-type))
	      this-file-sather-c-type-alist))
    (append 
     (list
      (format "\n/*\n * C ADT for %s structure to go with Sather interpreter.\n */\n" c-type)      
      (comment-nl-or-empty comment t)
      "\n /* Constructor */\n"
      (format "struct %s *%s(%s)\n" c-type c-creator create-actuals)
      (mapconcat (function format-c-formal) slot-spec "")
      (format "{ struct %s *cob;\n" c-type)
      (format "  cob = (struct %s *) malloc(sizeof(*cob));\n" c-type))
     (mapcar (function (lambda (slot)
			 (format "  cob->%s = %s;\n" (cadr slot) (car slot))))
	     slot-spec)
     (list 
      "  return (cob);\n};\n" 
      "\n /* Readers */\n")
     (mapcar (function (lambda (slot)
			 (format
			  "%s %s_%s(cob) struct %s *cob; { return ((%s)(cob->%s));};\n" 
			  (sather-to-c-type (caddr slot)) type-xfix (car slot)
			  c-type (sather-to-c-type (caddr slot)) (cadr slot))))
	     slot-spec)
     (list (format "\n/* Writers */\n"))
     (mapcar (function 
	      (lambda (slot)
		(format "void set_%s_%s(cob,x) struct %s *cob; %s x; { cob->%s = x;};\n" 
			type-xfix (car slot) c-type (sather-to-c-type (caddr slot))
			(cadr slot))))
	     slot-spec)
     (list "\n\n"))))

(defun macros-trafo-struct (type-spec slot-spec ignore1 &optional ignore2)
  ;; produce readers and writers for each slot of the normalized slot-spec
  (let* ((sather-type (car type-spec))
	 (type-xfix (downcase (format "%s" sather-type)))
	 (c-type (cadr type-spec)))
    (append 
     (list
      (format "\n-- Access macros for struct %s\n"
	      c-type)
      (format "-- in support of the Sather overlay class %s.\n" sather-type)
      "-- No constructor macro! If required, include the C function file.\n"
      "-- Readers --\n")
     (mapcar (function 
	      (lambda (slot)
		(push (format "%s_%s" type-xfix (car slot)) *macro-names*)
		(format
		 "(c_macro) \"#define %s_%s(cob) ((%s)((struct %s *)cob->%s))\" %s_%s\n"
		 type-xfix (car slot) (sather-to-c-type (caddr slot))
		 c-type (cadr slot) type-xfix (car slot))))
	     slot-spec)
     (list (format "-- Writers --\n"))
     (mapcar (function 
	      (lambda (slot)
		(push (format "set_%s_%s" type-xfix (car slot)) *macro-names*)
		(format 
		 "(c_macro) \"#define set_%s_%s(cob,x) {(struct %s *)cob->%s = x;}\" set_%s_%s\n"
			type-xfix (car slot) c-type (cadr slot) type-xfix (car slot))))
	     slot-spec)
     (list "\n"))))


;;;
;;; AUXILIARIES for local transformers
;;; 

(defvar sather-c-type-alist 
  '((INT . int)
    (REAL . float)
    (DOUBLE . double)
    (CHAR . char)
    (BOOL . char)
    ;;(CSTR . "char *") ;; = ptr
    (CKEY . int)
    (CHAN . int)
    (CARRAY . "ptr *")
    (CSTRAY . "ptr *")
    (OTHERWISE . ptr)))

(defvar this-file-sather-c-type-alist nil)	; struct correspondences in this file

(defun sather-to-c-type (sather-type)
  (let ((name (symbol-name sather-type)))
    (when (= (aref name 0) ?$)
	  (setq sather-type (intern (substring name 1)))))
  (or (cdr (assq sather-type sather-c-type-alist))
      (cdr (assq sather-type this-file-sather-c-type-alist))
      (cdr (assq 'OTHERWISE sather-c-type-alist))))
;;; TEST: (sather-to-c-type 'INT) (sather-to-c-type 'int) 

(defun format-sa-formal (slot)
  (format "%s:%s" 
	  (mapconcat (function format-sa-name) (car slot) ",")
	  (cadr slot)))

(defun format-sa-name (name)
  (format "%s" (car name)))

(defun format-actual (slot)
  (format "%s" (car slot)))

(defun format-c-formal (slot)
  (format "%s %s;\n" (sather-to-c-type (caddr slot)) (car slot)))

;;; Inline comment processing

(defun comment-nl-or-empty (&optional string-list C)
  (if string-list
      (concat "\n     "
	      (if C "/* " "") 
	      (mapconcat (function identity) string-list "\n     ")
	      (if C " */" "") 
	      "\n   ")
    ""))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;
;;; SAC top level forms and transformers
;;;

(defvar top-level-parser-alist 
  '((struct . parse-sac-struct)
    (constant . parse-sac-constant)
    (function . parse-sac-function)
    (OTHERWISE . top-level-error)))

(defun sac-parse-next-definition ()
  (interactive)
  (unwind-protect
      (progn (set-syntax-table sac-syntax-table)
	     (when (not (eq (this-token) ':EOF))
		   (must-be
		    (funcall
		     (or (cdr (assq (this-token) top-level-parser-alist))
			 (cdr (assq 'OTHERWISE top-level-parser-alist)))))))
    (set-syntax-table sather-mode-syntax-table)))

(defun top-level-error ()
  (error "Cannot find SAC top-level form %s. Expected one of %s!"
	 ;;(delete-if-eq 'OTHERWISE (mapcar 'car top-level-parser-alist))
	 (this-token)
	 (let (collection)
	   (dolist (elt top-level-parser-alist)
	     (when (not (eq (car elt) 'OTHERWISE))
	       (push (car elt) collection)))
	   (reverse collection))))

;;; A list of transformer, file-type, global transformer
;;; A transformer is applied to the tree of a definition and
;;; adds to a the file of its file-type. For each file, the global transformer
;;; is run once at the end of the file conversion.
(defvar trafo-file-alist ;; a list of transformer, file-type, global transformer
  '((sa-trafo ".sa" nil)  
    (c-trafo ".c" include-c-file)
    (macros-trafo ".macros" include-macros-file)))

;;;
;;; Global transformers run after file processing
;;;

(defun include-c-file (name)
  (after-banner)
  (insert "\n#include <" name ".h> /* This file should contain all relevant includes. */\n\n"))

(defun include-macros-file (name)
  (after-banner)
  (insert "\n(c_macro) \"#include <" name ".h>\" -- This file should contain all relevant includes.\n")
  (mapcar (function
	  (lambda (name)
	    (insert (format " %s" name))))
	 *macro-names*)
  (insert "\n\n"))

;;; AUXILIARIES for global transformers

(defun after-banner ()
  (beginning-of-buffer)
  (re-search-forward "--+------------")
  (beginning-of-line 3))


;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;
;;; CONVERSION COMMANDS
;;;

(defvar header-comment)

(defun sac-convert-buffer (arg)
  "Convert the Sather C interface definitions in the current buffer. The
result consists of three parts, Sather code, C code and Sather c_macros. 
These pieces of code are inserted into the buffers named like the current 
buffer but ending in .sa, .c and .macros respectively.
With a prefix argument, the command prompts for the file name."
  (interactive "P")
  (let* ((this-file-sather-c-type-alist nil)
	 (curr-buffer (current-buffer))
	 (file-name (file-name-sans-versions 
		     (buffer-file-name curr-buffer)))
         (dir (file-name-directory file-name))
	 (name-type (file-name-nondirectory file-name))
	 (tpos (string-match "\\.sac" name-type))
	 (name (if tpos (substring name-type 0 tpos)
		 name-type))
	 (*macro-names* nil)
         (*sather-c-type-correspondence* nil)
	 (header-comment nil))
    ;; save current buffer if changed
    (if (and (buffer-modified-p curr-buffer)
	     (y-or-n-p (format "Save file %s? " file-name)))
	(progn (save-buffer) (message "")))
    (if arg
	(setq name (read-input "File base name: " name)))
    ;; expand
    (goto-char (point-min))
    (collect-header-comment)		; file-header
    ;; prepare buffers and insert banner
    (dolist (x trafo-file-alist)
	    (let* ((file-type (cadr x))
		   (buf-name (concat name file-type)))
	      (save-excursion (find-file buf-name)
			      (set-buffer buf-name)
			      (erase-buffer)
			      (insert-sac-banner 
			       (buffer-file-name curr-buffer)
			       name))))
    ;; expand local transformations
    (while (sac-expand-next-definition dir name))
    ;; expand global transformations
    (dolist (x trafo-file-alist)
	    (let* ((file-type (cadr x))
		   (buf-name (concat name file-type))
		   (global-trafo (caddr x)))
	      (save-excursion (find-file buf-name)
			      (set-buffer buf-name)
			      (when global-trafo (funcall global-trafo name)))))
    (dolist (x trafo-file-alist)
	    (let* ((file-type (cadr x))
		   (buf-name (concat name file-type)))
	      (save-excursion (set-buffer buf-name)
			      (save-buffer))))))

(defun sac-convert-definition (arg)
  "Convert the Sather C interface definition at point. The result consists
of three parts, Sather code, C code and Sather c_macros. These pieces of code
are inserted into the buffers named like the current buffer but ending in .sa, 
.c and .macros respectively.
With a prefix argument, the command prompts for the file name."
  (interactive "P")
  (let* ((curr-buffer (current-buffer))
	 (file-name (file-name-sans-versions 
		     (buffer-file-name curr-buffer)))
         (dir (file-name-directory file-name))
	 (name-type (file-name-nondirectory file-name))
	 (tpos (string-match "\\.sac" name-type))
	 (name (if tpos (substring name-type 0 tpos)
		 name-type)))
    ;; save current buffer if changed
    (if (and (buffer-modified-p curr-buffer)
	     (y-or-n-p (format "Save file %s? " file-name)))
	(progn (save-buffer) (message "")))
    ;; maybe don't take default file triple
    (if arg
	(setq name (read-input "File base name: " name)))
    ;; expand single form
    (or 
     (sac-expand-next-definition dir name)
     (error "There seems to be no following definition!"))))

;;; Extract the proper form and apply all the transformers.
(defun sac-expand-next-definition (dir name)
  (let (construct 
	def comment transformer trafo-args buffer-name prefix header-comment
	(file-msg (format "File %s%s .." dir name)))	
    (when preserve-comment
	  (collect-header-comment)
	  (setq preserve-comment t))	;definition header
    (setq construct (this-token))	;save construct to get transformer
					;skips layout
    (when 
     (not (eq construct ':EOF))
     (setq prefix
	   (buffer-substring 
	    (point) 
	    (save-excursion 
	      (forward-char (min 25 (- (point-max) (point)))) (point))))
     (message "%s%s ... (parsing)" file-msg prefix)
     (when (setq def (sac-parse-next-definition)) 
	   (message "%s%s ... (translating)" file-msg prefix)
	   (setq comment (when (consp preserve-comment)
			       (prog1 (list (cdr (reverse preserve-comment)))
				 (setq preserve-comment t)))
		 trafo-args (nconc def comment))
	   (dolist 
	    (trafo trafo-file-alist)
	    (let* ((transformer (what-name (car trafo) construct))
		   (file-type (cadr trafo))
		   (file-name (concat name file-type)))
	      ;;(message "%s %s" transformer prefix) ;;save for trace
	      (save-excursion
		(set-buffer file-name)	;select buffer
		(goto-char (point-max))	;goto buffer end
		(let ((expansion  
		       (if transformer (apply transformer trafo-args))))
		  ;; keep this order.
		  ;; a transformer may want to position for header comment or
		  ;; also use and transform header comment.
		  (when expansion
			(insert-header-comment)
			(if (stringp expansion)
			    (insert expansion)
			  (mapcar 'insert expansion)))))))
	   t))))

(defun insert-sac-banner (sac-file-name file-name)
  (let* ((comment-start (or comment-start "--"))
	(comment-end (or comment-end ""))
	(mapping sather-c-type-alist)
	(n 3) (i n))
    (insert 
     comment-start "+ Sather to C interface code generated "
     (current-time-string) " from " file-name ".\n"
     "--+\n"
     "--+ preserve-comment=" (format "%s" preserve-comment) ".\n"
     "--+ sather-c-type-alist=\n"
     "--+   (")
    (while mapping
      (insert (format "%s " (car mapping))) 
      (setq mapping (cdr mapping) i (1- i))
      (when (zerop i) (setq i n) (insert "\n--+    ")))
    (insert ")\n--+\n"
     "--+ The code consists of three parts.\n"
     "--+ " file-name ".sa contains Sather and C classes importing from C.\n"
     "--+ " file-name ".c contains the corresponding C functions to go with\n"
     "--+ the Sather interpreter.\n"
     "--+ " file-name ".macros contains macros to be included in the\n"
     "--+ .sather file. Use (include) " file-name ".macros there.\n"
     "--+ If you refer to C structures, the " file-name ".c file must always\n"
     "--+ be included, with the macros file, too.\n"
     "--+----------------------------------------------------------------------------\n"
     comment-end "\n")
    (when preserve-comment
	  (insert comment-start " PRESERVED FILE HEADER " comment-end "\n")
	  (insert-header-comment)
	  (insert comment-start " END FILE HEADER " comment-end "\n\n"))))

(defun collect-header-comment ()
  "Collect continguous lines of top-level comments into header-comment
if preserve-comment=t. "
  (when preserve-comment
	;; collect leading layout
	(let (done (count 0))
	  (while (not done)
	    (re-search-forward "[ \t]*") ;skip white space
	    (cond ((looking-at "\n") 
		   (setq count (1+ count))
		   (forward-char 1))
		  (t (setq done t)
		     (if (> count 0) 
			 (dotimes (i (1- count))
				  (push "\n" header-comment)))))))
	;; collect contiguous comment lines
	(while (looking-at "--")
	  (push (buffer-substring (point)
				  (progn (beginning-of-line 2) (point)))
		header-comment))
	(setq header-comment (reverse header-comment)))) 

(defun insert-header-comment ()
  (when header-comment
	(let ((comment-start (or comment-start "--"))
	      (comment-end (or comment-end ""))
	      (C (when (not (equal comment-start "--")) t))
	      line 
	      (lines header-comment))
	  (while lines
	    (setq line (car lines)
		  lines (cdr lines))
	    (cond ((equal line "\n")
		   (insert "\n"))
		  (t (when C (insert comment-start "\n"))
		     (insert line)
		     (mapcar (function insert) lines)
		     (setq lines nil)
		     (when C (insert comment-end "\n"))))))))

;;(defun mouse-execute-at-point (ignore) (interactive) (message (format "%s" (mouse-parse))))
;;(defun mouse-parse () (interactive) (sac-expand-next-definition))
;;(defun mouse-parse () (interactive) (let (header-comment) (collect-header-comment)))
;;(defun mouse-parse () (interactive) (this-token))
;;(defun mouse-parse () (interactive) (parse-sac-struct))
;;(defun mouse-parse () (interactive) (parse-sac-constant))



