/*
 * Copyright 1999-2000 Vizdom Software, Inc. All Rights Reserved.
 * 
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the same terms as the Perl Kit, namely, under 
 *  the terms of either:
 *
 *      a) the GNU General Public License as published by the Free
 *      Software Foundation; either version 1 of the License, or 
 *      (at your option) any later version, or
 *
 *      b) the "Artistic License" that comes with the Perl Kit.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See either
 *  the GNU General Public License or the Artistic License for more 
 *  details. 
 */

package com.vizdom.dbd.jdbc;

import com.vizdom.ber.*;
import java.io.UnsupportedEncodingException;

/**
 * A connection request.
 *
 * @author Gennis Emerson
 * @version $Revision: 1.9 $
 */
class ConnectRequest extends BerSequence
{
    /* ConnectRequest {
     *    OctetString url
     *    OctetString username
     *    OctetString password
     *    BerHash properties
     * }
     *
     * It probably wouldn't hurt to move the url/username/password into 
     * the hash.
     */
    /** This object's identifier. */
    static final BerIdentifier gIDENTIFIER = 
        new BerIdentifier(BerTypes.APPLICATION, BerTypes.CONSTRUCTED, 
        BerDbdModule.gCONNECT_REQUEST);

    /** The factory for this object. */
    static final BerObjectFactory gFACTORY = 
        new BerObjectFactory()
    {
        public boolean acceptsIdentifier(BerIdentifier anIdentifier)
        {
            return anIdentifier.equals(gIDENTIFIER);
        }
        
        public BerObject createBerObject()
        {
            return new ConnectRequest();
        }
    };

    /** The index of the url within the sequence. */
    private static final int sURL = 0;
    /** The index of the username within the sequence. */
    private static final int sUSERNAME = 1;
    /** The index of the password within the sequence. */
    private static final int sPASSWORD = 2;
    /** The index of the properties hash within the sequence. */
    private static final int sPROPERTIES = 3;

    /**
     * The decoding constructor.
     */
    private ConnectRequest()
    {
        super();
    }

    /**
     * Returns the url portion of the connection request.
     *
     * @return the url portion of the connection request
     */
    String getURL(String aCharacterEncoding) 
        throws UnsupportedEncodingException
    {
        return ((BerOctetString) mSequence[sURL]).toString(
            aCharacterEncoding);
    }

    /**
     * Returns the username portion of the connection request.
     *
     * @return the username portion of the connection request
     */
    String getUser(String aCharacterEncoding)
        throws UnsupportedEncodingException
    {
        return ((BerOctetString) mSequence[sUSERNAME]).toString(
            aCharacterEncoding);
    }

    /**
     * Returns the password portion of the connection request.
     *
     * @return the password portion of the connection request
     */
    String getPassword(String aCharacterEncoding)
        throws UnsupportedEncodingException
    {
        return ((BerOctetString) mSequence[sPASSWORD]).toString(
            aCharacterEncoding);
    }

    /**
     * Returns the name/value pairs sent by the client.
     *
     * @return the name/value pairs sent by the client, as a hashtable
     *      of strings
     */
    java.util.Hashtable getProperties()
    {
        return ((BerHash) mSequence[sPROPERTIES]).toHashtable();
    }

    /**
     * Returns the identifier for this BerObject. 
     *
     * @return the identifier for this BerObject
     */
     public BerIdentifier getIdentifier()
    {
        return gIDENTIFIER;
    }
    
    /**
     * Returns a string representation of this object.
     *
     * @return a string representation of this object
     */
    public String toString()
    {
        return "Connect";
    }
}
