#!/usr/bin/perl -w

#
# sisify.pl - package Perl scripts or Perl libraries into SIS files
#
# Copyright (c) 2004-2005 Nokia. All rights reserved.
# The sisify.pl utility is licensed under the same terms as Perl itself.
#


use strict;

use vars qw($VERSION);

$VERSION = '0.2';

use Getopt::Long;
use File::Temp qw/tempdir/;
use File::Find;
use File::Basename qw/basename dirname/;
use Cwd qw/getcwd/;

BEGIN {
  # This utility has been developed in Windows under cmd.exe with
  # the Series 60 2.6 SDK installed, but for the makesis utility
  # in UNIX/Linux, try e.g. one of the following:
  # http://gnupoc.sourceforge.net/
  # http://symbianos.org/~andreh/ You
  # will also need the 'uidcrc' utility.
  die "$0: Looks like Cygwin, aborting.\n" if exists %ENV{'!C:'};
}

sub die_with_usage {
  if (@_) {
    warn "$0: $_\n" for @_;
  }
  die <<__USAGE__;
$0: Usage:
$0 [--variant=S60|S80|UIQ] [ --uid=hhhhhhhh ] [ --version=a.b.c ] [ --library=x.y.z ] [ some.pl | Some.pm | somefile | dir ... ]
The uid is the Symbian app uid for the SIS.
The version is the version of the SIS.
The library is the version of Perl under which to install.  If using this,
only specify directories for packaging.
__USAGE__
}

my $SisUid;
my $SisVersion;
my $Library;
my @SisPl;
my @SisPm;
my @SisDir;
my @SisOther;
my $AppName;
my $Debug;
my $ShowPkg;
my $Variant;

my $SisUidDefault     = 0x0acebabe;
my $SisVersionDefault = '0.0.0';
my $VariantDefault    = 'S60';

my %Variant = qw(S60 1 S80 1 UIQ 1);

die_with_usage()
  unless GetOptions(
		    'variant=s'		=> \$Variant,
		    'uid=s'		=> \$SisUid,
		    'version=s'		=> \$SisVersion,
		    'debug'		=> \$Debug,
		    'showpkg'		=> \$ShowPkg,
		    'library=s'		=> \$Library,
		    'appname=s'		=> \$AppName,
		   );
die_with_usage("Need to specify what to sisify")
  unless @ARGV;

unless (defined $Variant) {
  warn "$0: Defaulting to $VariantDefault\n";
  $Variant = $VariantDefault;
}

unless (exists %Variant{$Variant}) {
  die "$0: Unknown variant '$Variant'\n";
}

for my $i (@ARGV) {
  if ($i =~ m/\.pl$/i) {
    push @SisPl, $i;
  } elsif ($i =~ m/\.pm$/i) {
    push @SisPm, $i;
  } elsif (-f $i) {
    push @SisOther, $i;
  } elsif (-d $i) {
    push @SisDir, $i;
  } else {
    die_with_usage("Unknown sisifiable '$i'");
  }
}

sub do_system {
    my $cwd = getcwd();
    print qq{\# system("@_") [cwd "$cwd"]\n};
    return system("@_") == 0;
}

die_with_usage("Must specify something to sisify")
  unless @SisPl || @SisPm || @SisOther || @SisDir;

die_with_usage("With the lib option set, specify only directories")
  if defined $Library && ((@SisPl || @SisPm || @SisOther) || @SisDir == 0);

die_with_usage("Lib must define the Perl 5 version as 5.x.y")
  if defined $Library && $Library !~ m/^5.\d+\.\d+$/;

die_with_usage("With the lib option unset, specify at least one .pl file")
  if (! defined $Library && @SisPl == 0);

if (!defined $AppName) {
  if (defined $Library) {
    $AppName = @SisDir[0];
    $AppName =~ tr!/!-!;
  } elsif (@SisPl +> 0 && @SisPl[0] =~ m/^(.+)\.pl$/i) {
    $AppName = basename($1);
  }
}

die_with_usage("Must either specify appname or at least one .pl file or the lib option")
  unless defined $AppName || defined $Library;

print "[app name '$AppName']\n" if $Debug;

unless (defined $SisUid) {
  $SisUid = $SisUidDefault;
  printf "[default app uid '0x\%08x']\n", $SisUid;
} elsif ($SisUid =~ m/^(?:0x)?([0-9a-f]{8})$/i) {
  $SisUid = hex($1);
} else {
  die_with_usage("Bad uid '$SisUid'");
}
$SisUid = sprintf "0x\%08x", $SisUid;

die_with_usage("Bad uid '$SisUid'")
  if $SisUid !~ m/^0x[0-9a-f]{8}$/i;

unless (defined $SisVersion) {
  $SisVersion = $SisVersionDefault;
  print "[default app version '$SisVersionDefault']\n";
} elsif ($SisVersion !~ m/^\d+\.\d+\.\d+$/) {
  die_with_usage("Bad version '$SisVersion'")
}

my $tempdir = tempdir( CLEANUP => 1 );

print "[temp directory '$tempdir']\n" if $Debug;

for my $file (@SisPl, @SisPm, @SisOther) {
  print "[copying file '$file']\n" if $Debug;
  die_with_usage("$0: File '$file': $!") unless -f $file;
  my $dir = dirname($file);
  do_system("mkdir $tempdir\\$dir") unless $dir eq '.';
  do_system("copy $file $tempdir");
}
if (@SisPl) {
    do_system("copy @SisPl[0] $tempdir\\default.pl")
	unless @SisPl[0] eq "default.pl";
}
for my $dir (@SisDir) {
  print "[copying directory '$dir']\n" if $Debug;
  do_system("copy $dir $tempdir");
}

my $SisVersionCommas = $SisVersion;

$SisVersionCommas =~ s/\./\,/g;

my @pkg;

push @pkg, qq[&EN;];
push @pkg, qq[#\{"$AppName"\},($SisUid),$SisVersionCommas];
push @pkg, qq[(0x101F6F88), 0, 0, 0, \{"Series60ProductID"\}];

my $OWD = getcwd();

$OWD =~ s!/!\\!g;

chdir($tempdir) or die "$0: chdir('$tempdir')\n";

if (@SisPl) {
  if (open(my $fi, "<", "default.pl")) {
    my $fn = "default.pl.new";
    if (open(my $fo, ">", "$fn")) {
      while ( ~< $fi) {
	last unless m/^\#/;
	print $fo $_;
      }
      print $fo "use lib qw(\\system\\apps\\$AppName \\system\\apps\\$AppName\\lib);\n";
      printf $fo qq[# \%d "@SisPl[0]"\n], $.;
      print $fo $_;
      while ( ~< $fi) {
	print $fo $_;
      }
      close($fo);
    } else {
      die "$0: open '>$fn': $!\n";
    }
    close($fi);
    rename($fn, "default.pl") or die "$0: rename $fn default.pl: $!\n";
    # system("cat -nvet default.pl");
  } else {
    die "$0: open 'default.pl': $!\n";
  }
}


my @c;
find(
     sub {
       if (-f $_) {
	 $File::Find::name =~ s!^\./!!;
	 push @c, $File::Find::name;
       }
     }
     ,
     ".");

for my $i (sort @c) {
  my $j = $i;
  $j =~ s!/!\\!g;
  push @pkg, defined $Library ? qq["$j"-"!:\\System\\Libs\\Perl\\siteperl\\$Library\\$j"] : qq["$j"-"!:\\system\\apps\\$AppName\\$j"];
}

sub hex2data {
  pack("H*", shift); # symbian\hexdump.pl to create the hexdumps.
}

my $APPHEX;
my $RSCHEX;

unless ($Library) {
  # If we package an application we will need both a launching native
  # Symbian application and a resource file for it.  The resource file
  # we can get easily from a stub but for the native app we need to
  # patch in the right Symbian app uids and executable checksums.

  &init_hex; # Initialized $APPHEX and $RSCHEX.

  die "$0: No app template found\n" unless defined $APPHEX && defined $RSCHEX;

  my $app = hex2data($APPHEX);
  my $uidcrc;
  my $uids = "0x10000079 0x100039CE $SisUid";

  my $cmd = "uidcrc $uids |";

  if (open(my $fh, "<", $cmd)) {
    my $line = ~< $fh;
    close($fh);
    # 0x10000079 0x100039ce 0x0acebabe 0xc82b1900
    $line =~ s/\r?\n$//;
    if ($line =~ m/^$uids (0x[0-9a-f]{8})$/i) {
      $uidcrc = hex($1);
    } else {
      die "$0: uidcrc returned '$line'\n";
    }
  } else {
    die qq[$0: open '$cmd' failed: $!\n];
  }

  my $uid    = hex($SisUid);

  my $oldchk = unpack('V', substr($app, 24, 4));
  my $newchk = ($uid + $oldchk) ^&^ 0xFFFFFFFF;

  # printf "# uid    = 0x%08x\n", $uid;
  # printf "# uidcrc = 0x%08x\n", $uidcrc;
  # printf "# oldchk = 0x%08x\n", $oldchk;
  # printf "# newchk = 0x%08x\n", $newchk;

  substr($app,  8, 4, pack('V', $uid));
  substr($app, 12, 4, pack('V', $uidcrc));
  substr($app, 24, 4, pack('V', $newchk));
  
  my $UID_OFFSET = 0x0C7C; # This is where the uid is in the $app.
  substr($app, $UID_OFFSET, 4, substr($app, 8, 4)); # Copy the uid also here.

  if (open(my $fh, ">", "$AppName.app")) {
    binmode($fh);
    print $fh $app;
    close($fh);
  } else {
    die qq[$0: open '>$AppName.app' failed: $!\n];
  }

  push @pkg, qq["$AppName.app"-"!:\\system\\apps\\$AppName\\$AppName.app"];

  if (open(my $fh, ">", "$AppName.rsc")) {
    binmode($fh);
    print $fh hex2data($RSCHEX);
    close($fh);
  } else {
    die qq[$0: open '>$AppName.rsc' failed: $!\n];
  }
  push @pkg, qq["$AppName.rsc"-"!:\\system\\apps\\$AppName\\$AppName.rsc"];
}

if ($ShowPkg) {
  for my $l (@pkg) {
    print $l, "\r\n";
  }
} else {
  my $fn = "$AppName.pkg";
  if (open(my $fh, ">", "$fn")) {
    for my $l (@pkg) {
      print $fh "$l\r\n"; # Note CRLF!
    }
    close($fh);
  } else {
    die qq[$0: Failed to open "$fn" for writing: $!\n];
  }
  my $sis = "$AppName.SIS";
  unlink($sis);
  do_system("dir");
  do_system("makesis $fn");
  unless (-f $sis) {
    die qq[$0: failed to create "$sis"\n];
  }
  do_system("copy $AppName.sis $OWD");
  chdir($OWD);
  system("dir $sis");
  print "\n=== Now transfer $sis to your device ===\n";
}

exit(0);

# To create the hex: print unpack("H*", $data);

sub init_hex {
  # This is Symbian application executable skeleton.
  # You can create the ...\epoc32\release\thumb\urel\foo.app
  # by compiling the PerlApp.cpp with PerlMinSample defined in PerlApp.h.
  # The following executable has been compiled using the Series 60 SDK 2.6
  # for Visual C.
  # 'make sisify_hex' to create the perlappmin.hex for this hexdump.
  if ($Variant eq 'S60') {
      $APPHEX = <<__APP__;
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
__APP__
  }

  # This is Symbian application resource skeleton.
  # You can create the ...\epoc32\data\z\system\apps\PerlApp\PerlApp.rsc
  # by compiling the PerlApp.cpp.
  # The following resource has been compiled using the Series 60 SDK 2.6
  # for Visual C.
  # 'make sisify_hex' to create the perlrscmin.hex for this hexdump.
  if ($Variant eq 'S60') {
    $RSCHEX = <<__RSC__;
6b4a1f10000000005fde04001ca360de01b80010000400000001f0e54d0000000004f0e54d00000000000000001a00cc0800000000010005f0e54d000000000000ffffffff0000000000000000000f0500000400000000000000000000050541626f7574170000ffffffff00000000010400000000000000000000040454696d65170000ffffffff00000000020400000000000000000000030352756e170000ffffffff0000000003040000000000000000000008084f6e656c696e6572170000ffffffff000000000404000000000000000000000909436f707972696768740e0000ffffffff00000000000000000120000000000000001400cc0801006816000100000000000100000000ffffffff00ffffffff0000000000000000ffff000000000000000120000000000000002400cc0801006816000100000000000100000000ffffffff00ffffffff0000000000000000ffff000000000000004122000000000000001400cc08010069160000050000000001000000000000000001000000040007000800ff020100ffffffff00000000000000000000ffff000000000000004122000000000000001400cc08010074160007000000000054160000ffffffff000000000000ffff00000000000000000000000015001d001d0035004d00ef0026015d01a301d201d701
__RSC__
  }

  # This is Symbian application executable skeleton.
  # You can create the ...\epoc32\release\thumb\urel\foo.app
  # by compiling the PerlApp.cpp with PerlMinSample defined in PerlApp.h.
  # The following executable has been compiled using the Series 80 SDK 2.0
  # for Visual C.
  # 'make sisify_hex' to create the perlappmin.hex for this hexdump.
  if ($Variant eq 'S80') {
      $APPHEX = <<__APP__;
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
  \}

  # This is Symbian application resource skeleton.
  # You can create the ...\epoc32\data\z\system\apps\PerlApp\PerlApp.rsc
  # by compiling the PerlApp.cpp.
  # The following resource has been compiled using the Series 80 SDK 2.0
  # for Visual C.
  # 'make sisify_hex' to create the perlrscmin.hex for this hexdump.
  if ($Variant eq 'S80') \{
    $RSCHEX = <<__RSC__;
6b4a1f10000000005fde04001ca360de01b800780400000001f0e54d0000000005f0e54d000000000000000004f0e54d000000000010010000005000000000000400000204030352756e0900ffffffff00030408084f6e656c696e65720900ffffffff000504050541626f75740900ffffffff0000010404457869740500ffffffff0007010006f0e54d07074f7074696f6e73110000000000ffffffff0000000000000000000f0500000400000000000000000000050541626f7574170000ffffffff00000000010400000000000000000000040454696d65170000ffffffff00000000020400000000000000000000030352756e170000ffffffff0000000003040000000000000000000008084f6e656c696e6572170000ffffffff000000000404000000000000000000000909436f707972696768740e0000ffffffff00000000000000000003010005055469746c65110000000064b0f3000000000001000200060650726f6d70740f00080000000000000a00010080000014001c001c0034008200a50047017b01
__RSC__
  \}

  # This is Symbian application executable skeleton.
  # You can create the ...\epoc32\release\thumb\urel\foo.app
  # by compiling the PerlApp.cpp with PerlMinSample defined in PerlApp.h.
  # The following executable has been compiled using the UIQ SDK 2.1
  # for CodeWarrior.
  # 'make sisify_hex' to create the perlappmin.hex for this hexdump.
  if ($Variant eq 'UIQ') \{
      $APPHEX = <<__APP__;
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
__APP__
  }

  # This is Symbian application resource skeleton.
  # You can create the ...\epoc32\data\z\system\apps\PerlApp\PerlApp.rsc
  # by compiling the PerlApp.cpp.
  # The following resource has been compiled using the UIQ 2.1 SDK
  # for CodeWarrior.
  # 'make sisify_hex' to create the perlrscmin.hex for this hexdump.
  if ($Variant eq 'UIQ') {
    $RSCHEX = <<__RSC__;
6b4a1f10000000005fde04001ca360de019a00f8030400000001f0e54d0000000004f0e54d04f0e54d0000000000000000000000000007010005f0e54d07074f7074696f6e73110000000000ffffffff0000000000000000000f0400000400000000000000000000050541626f7574170000ffffffff00000000010400000000000000000000040454696d65170000ffffffff0000000003040000000000000000000008084f6e656c696e6572170000ffffffff000000000404000000000000000000000909436f707972696768740e0000ffffffff000000000000000000050100000005055469746c6511460014000000000064b0f30001000200060650726f6d707411000800000000000000000a0001008000000005000000000d0d4f4b202f2043616e63656c203f0e460014000000000008f0e54d00000015010000000200feff00000000000000000000000002024f6b1a00ffffffff000000000101ffff000000000000000000000000060643616e63656c0b00ffffffff0000000001010005000000000a0a596573202f204e6f203f0e46001400000000000af0e54d00000015010000000200feff00000000000000000000000003035965732900ffffffff000000000101ffff000000000000000000000000024e006f0000ffffffff00000000010115001d001d0035005800de0016013a018201a301e801
__RSC__
  }
}
