=head1 NAME

Imager::Cookbook - recipes working with Imager

=head1 DESCRIPTION

Various simple and not so simple ways to do things with Imager.

=head1 FILES

This is described in detail in L<Imager::Files>.

=head2 Reading an image from a file

  my $image = Imager->new;

  $image->read(file=>$filename) or die $image->errstr;

See L<Imager::Files>.

=head2 Writing an image to a file

  $image->write(file=>$filename) or die $image->errstr;

=head2 Write an animated gif.

  # build an array of images to use in the gif
  my  @images;
  # synthesize the images or read them from files, it doesn't matter
  ...

  # write the gif
  Imager->write_multi({ file=>$filename, type=>'gif' }, @images)
    or die Imager->errstr;

See L<Imager::Files/"Writing an animated GIF"> for a more detailed
example.

=head2 Reading multiple images from one file

Some formats, like GIF and TIFF support multiple images per file.  Use
the L<read_multi()|Imager::Files> method to read them:

  my @images = Imager->read_multi(file=>$filename)
    or die Imager->errstr;

=head1 IMAGE SYNTHESIS

=head2 Creating an image

To create a simple RGB image, supply the image width and height to the
new() method:

  my $rgb = Imager->new(xsize=>$width, ysize=>$height);

If you also want an alpha channel:

  my $rgb_alpha = Imager->new(xsize=>$width, ysize=>$height, channels=>4);

To make a grayscale image:

  my $gray = Imager->new(xsize=>$width, ysize=>$height, channels=>1);

and a grayscale image with an alpha channel:

  my $gray_alpha = Imager->new(xsize=>$width, ysize=>$height, channels=>2);

When a new image is created this way all samples are set to zero -
black for 1 or 3 channel images, transparent black for 2 or 4 channel
images.

You can also create paletted images and images with more than 8-bits
per channel, see L<Imager::ImageTypes> for more details.

=head2 Setting the background of a new image

To set the background of a new image to a solid color, use the box()
method with no limits, and C<< filled=>1 >>:

  $image->box(filled=>1, color=>$color);

As always, a color can be specified as an L<Imager::Color> object:

  my $white = Imager::Color->new(255, 255, 255);
  $image->box(filled=>1, color=>$white);

or you supply any single scalar that Imager::Color's new() method
accepts as a color description:

  $image->box(filled=>1, color=>'white');
  $image->box(filled=>1, color=>'#FF0000');
  $image->box(filled=>1, color=>[ 255, 255, 255 ]);

You can also fill the image with a fill object:

  use Imager::Fill;
  # create the fill object
  my $fill = Imager::Fill->new(hatch=>'check1x1')
  $image->box(fill=>$fill);

  # let Imager create one automatically
  $image->box(fill=>{ hatch=>'check1x1' });

See L<Imager::Fill> for information on Imager's fill objects.

=head1 WORLD WIDE WEB

As with any CGI script it's up to you to validate data and set limits
on any parameters supplied to Imager.

For example, if you allow the caller to set the size of an output
image you should limit the size to prevent the client from specifying
an image size that will consume all available memory.

This is beside any any other controls you need over access to data.

See L<CGI> for a module useful for processing CGI submitted data.

=head2 Returning an image from a CGI script

This is similar to writing to a file, but you also need to supply the
information needed by the web browser to identify the file format:

  my $img = ....; # create the image and generate the contents
  print "Content-Type: image/png\n\n";
  binmode STDOUT;
  $img->write(fd=>fileno(STDOUT), type=>'png')
    or die $img->errstr;

You need to set the Content-Type header depending on the file format
you send to the web browser.

If you want to supply a content-length header, write the image to a
scalar as a buffer:

  my $img = ....; # create the image and generate the contents
  my $data;
  $img->write(type=>'png', data=>\$data)
    or die $img->errstr;
  print "Content-Type: image/png\n";
  print "Content-Length: ",length($data),"\n\n";
  binmode STDOUT;
  print $data;

See C<samples/samp-scale.cgi> and C<samples/samp-image.cgi> for a
couple of simple examples of producing an image from CGI.

=head2 Inserting a CGI image in a page

There's occasionally confusion on how to display an image generated by
Imager in a page generated by a CGI.

Your web browser handles this process as two requests, one for the
HTML page, and another for the image itself.

Each request needs to perform validation since an attacker can control
the values supplied to both requests.

How you make the data available to the image generation code depends
on your application.

See C<samples/samp-form.cgi> and C<samples/samp-image.cgi> in the
Imager distribution for one approach.  The POD in C<samp-form.cgi>
also discusses some of the issues involved.

=head2 Parsing an image posted via CGI

C<WARNING>: file format attacks have become a common attack vector,
make sure you have up to date image file format libraries, otherwise
trying to parse uploaded files, whether with Imager or some other
tool, may result in a remote attacker being able to run their own code
on your system.  Currently Imager makes no attempt to place size
limits on a read image file.  This may result in consumption of large
amounts of memory.  Future versions of Imager may provide mechanisms
to limit the sizes of images read from files.

If your HTML form uses the correct magic, it can upload files to your
CGI script, in particular, you need to use C< method="post" > and
C<enctype="multipart/form-data"> in the C<form> tag, and use
C<type="file"> in the C<input>, for example:

  <form action="/cgi-bin/yourprogram" method="post" 
        enctype="multipart/form-data">
    <input type="file" name="myimage" />
    <input type="submit value="Upload Image" />
  </form>

To process the form:

=over

=item 1.

first check that the user supplied a file

=item 2.

get the file handle

=item 3.

have Imager read the image

=back

  # returns the client's name for the file, don't open this locally
  my $cgi = CGI->new;
  # 1. check the user supplied a file
  my $filename = $cgi->param('myimage');
  if ($filename) {
    # 2. get the file handle
    my $fh = $cgi->upload('myimage');
    if ($fh) {
      binmode $fh;
      
      # 3. have Imager read the image
      my $img = Imager->new;
      if ($img->read(fh=>$fh)) {
        # we can now process the image
      }
    }
    # else, you probably have an incorrect form or input tag
  }
  # else, the user didn't select a file

See C<samples/samp-scale.cgi> and C<samples/samp-tags.cgi> in the
Imager distribution for example code.

=head1 DRAWING

=head1 TEXT

=head2 Drawing text

=head2 Aligning text

=head2 Measuring text

=head2 Word wrapping text

=head1 METADATA

=head2 Image spatial resolution.

Keywords: DPI

=head1 AUTHOR

Tony Cook <tony@develop-help.com>

=head1 SEE ALSO

L<Imager>, L<Imager::Files>, L<Imager::Draw>.

=cut
