use strict;
use warnings;

use Shipwright;
use File::Temp qw/tempdir/;
use File::Copy;
use File::Copy::Recursive qw/dircopy/;
use File::Spec::Functions qw/catfile catdir updir/;
use File::Path qw/rmtree/;
use Cwd qw/getcwd abs_path/;

use Test::More tests => 10;
use Shipwright::Test;
Shipwright::Test->init;

SKIP: {
    skip "git: no git found or env SHIPWRIGHT_TEST_GIT not set", Test::More->builder->expected_tests
      if skip_git();

    my $cwd = getcwd;

    create_git_repo();

    my $repo = create_git_repo;

    my $shipwright = Shipwright->new(
        repository => "git:$repo",
        source => 'file:' . catfile( 't', 'hello', 'Foo-Bar-v0.01.tar.gz' ),
        follow => 0,
        log_level => 'FATAL',
        force => 1,
    );
    isa_ok( $shipwright->backend, 'Shipwright::Backend::Git' );

    # init
    $shipwright->backend->initialize();


    my $cloned_dir = $shipwright->backend->cloned_dir;
    my @dirs = sort `ls $cloned_dir`;
    chomp @dirs;
    is_deeply(
        [@dirs],
        [ '__default_builder_options', 'bin', 'etc', 'inc', 'scripts', 'shipwright', 'sources', 't' ],
        'initialize works'
    );

    # source
    my $source_dir = $shipwright->source->run();

    # import

    $shipwright->backend->import( name => 'hello', source => $source_dir );
    ok( grep( {/Makefile\.PL/} `ls $cloned_dir/sources/Foo-Bar/vendor` ),
        'imported ok' );

    my $script_dir = tempdir( 'shipwright_XXXXXX', CLEANUP => 1, TMPDIR => 1 );
    copy( catfile( 't', 'hello', 'scripts', 'build' ),       $script_dir );
    copy( catfile( 't', 'hello', 'scripts', 'require.yml' ), $script_dir );

    $shipwright->backend->import(
        name         => 'hello',
        source       => $source_dir,
        build_script => $script_dir,
    );
    ok( grep( {/Makefile\.PL/} `cat $cloned_dir/scripts/Foo-Bar/build` ),
        'build script ok' );

    # import another dist

    chdir $cwd;
    $shipwright = Shipwright->new(
        repository => "git:$repo",
        source => 'file:' . catfile( 't', 'hello', 'Foo-Bar-v0.01.tar.gz' ),
        name   => 'howdy',
        follow => 0,
        log_level => 'FATAL',
        force => 1,
    );

    $source_dir = $shipwright->source->run();
    like( $source_dir, qr/\bhowdy\b/, 'source name looks ok' );
    $shipwright->backend->import( name => 'hello', source => $source_dir );
    ok( grep( {/Makefile\.PL/} `ls $cloned_dir/sources/howdy/vendor` ),
        'imported ok' );
    $script_dir = tempdir( 'shipwright_XXXXXX', CLEANUP => 1, TMPDIR => 1 );
    copy( catfile( 't', 'hello', 'scripts', 'build' ), $script_dir );
    copy( catfile( 't', 'hello', 'scripts', 'howdy_require.yml' ),
        catfile( $script_dir, 'require.yml' ) );

    $shipwright->backend->import(
        name         => 'hello',
        source       => $source_dir,
        build_script => $script_dir,
    );
    ok( grep( {/Makefile\.PL/} `cat $cloned_dir/scripts/howdy/build` ),
        'build script ok' );

    my $tempdir = tempdir( 'shipwright_XXXXXX', CLEANUP => 1, TMPDIR => 1 );
    dircopy(
        catfile( 't',      'hello', 'shipwright' ),
        catfile( $tempdir, 'shipwright' )
    );

    # check to see if update_order works
    like(
        `cat $cloned_dir/shipwright/order.yml`,
        qr/Foo-Bar.*howdy/s,
        'original order is right'
    );

    system( 'cp -r ' . catfile( $tempdir, 'shipwright' ) . " $cloned_dir/" );
    $shipwright->backend->commit( comment => 'update shipwright/' );
    like(
        `cat $cloned_dir/shipwright/order.yml`,
        qr/howdy.*Foo-Bar/s,
        'imported wrong order works'
    );

    $shipwright->backend->update_order;
    like(
        `cat $cloned_dir/shipwright/order.yml`,
        qr/Foo-Bar.*howdy/s,
        'updated order works'
    );

}

