package Spp::Tools;

=head1 NAME

Spp::Tools - The perl interface for Spp

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.02';

=head1 SYNOPSIS

Tools gather some small reused function

    use Spp::Tools;

    my $first_element = first([1,2,3]);
    # 1

=head1 EXPORT

add_exprs all_is_array all_is_int all_is_str
all_is_sym apply_char array_index bool error fill_array
first get_atoms_type get_token_name in is_array
is_match_atom is_match_atoms is_case is_dot is_else is_elsif is_fail
is_false is_hash is_if is_in is_int is_lambda is_list
is_match is_nil is_perl_array is_func is_perl_hash
is_ref is_rule is_same is_str is_sym is_context is_true 
is_when len load_file perl_join perl_max perl_split perl_zip read_file
rest second see subarray tail to_str trim type uuid value
write_file

=cut

use Exporter;
our @ISA = qw(Exporter);
our @EXPORT = qw(
add_exprs
all_is_array
all_is_int
all_is_str
all_is_sym
apply_char
array_index
bool
error
fill_array
first
get_atoms_type
get_token_name
in
is_array
is_case
is_context
is_dot
is_else
is_elsif
is_fail
is_false
is_func
is_hash
is_if
is_in
is_int
is_lambda
is_list
is_match
is_match_atom
is_match_atoms
is_nil
is_perl_array
is_perl_hash
is_ref
is_rule
is_same
is_str
is_sym
is_true
is_when
len
load_file
perl_join
perl_max
perl_split
perl_substr
perl_zip
read_file
rest
second
see
subarray
tail
to_str
trim
type
uuid
value
write_file
);

use 5.020;
use Carp qw(croak);
use JSON qw(encode_json decode_json);
use List::Util qw(max);
use experimental qw(switch autoderef);
use List::MoreUtils qw(pairwise firstidx);

###################################################
#see add_exprs([1,2,3]);
#see add_exprs([[[1,2,3]]]);
sub add_exprs {
  my $atoms = shift;
  return first($atoms) if len($atoms) == 1;
  return ['exprs', $atoms];
}

sub all_is_array {
  my $args = shift;
  return 1 if get_atoms_type($args) eq 'array';
  return 0;
}

sub all_is_int {
  my $args = shift;
  return 1 if get_atoms_type($args) eq 'int';
  return 0;
}

sub all_is_str {
  my $args = shift;
  return 1 if get_atoms_type($args) eq 'str';
  return 0;
}

sub all_is_sym {
  my $args = shift;
  return 1 if get_atoms_type($args) eq 'sym';
  return 0;
}

sub apply_char {
  my ($len, $cursor) = @_;
  my $pos = $cursor->{POS};
  my $str = $cursor->{STR};
  return '' if $pos >= $cursor->{LEN};
  return substr($str, $pos, $len) if $len > 0;
  return substr($str, $pos + $len, abs($len)) if $len < 0;
}

sub array_index {
  my ($value, $array) = @_;
  return firstidx { is_same($_, $value) } @{$array};
}

sub bool {
  my $x = shift;
  return ['true'] if $x;
  return ['false'];
}

sub error { say @_; exit }

sub fill_array {
  my ($value, $len) = @_;
  my $fill_array = [];
  for my $x (1 .. $len) {
    push $fill_array, $value;
  }
  return $fill_array;
}

sub first {
  my $data = shift;
  return $data->[0] if is_perl_array($data);
  return substr($data, 0, 1) if is_str($data);
}

sub get_atoms_type {
  my $atoms = shift;
  my $type = type(first($atoms));
  my $types = [];
  for my $atom (values $atoms) {
    my $atom_type = type($atom);
    next if $atom_type eq $type;
    push $types, $atom_type;
  }
  return $types if len($types) > 0;
  return $type;
}

sub get_token_name {
   my $token_name = shift;
   if ( $token_name =~ /^\./ ) {
      return substr($token_name, 1);
   }
   return $token_name;
}

sub in {
  my ($element, $array) = @_;
  my $element_str = to_str($element);
  for my $x (@{$array}) {
    return 1 if to_str($x) eq $element_str;
  }
  return 0;
}

sub is_array {
  my $x = shift;
  return 1 if type($x) eq 'array';
  return 0;
}

sub is_case {
  my $x = shift;
  return 1 if is_sym($x) and value($x) eq 'case';
  return 0;
}

sub is_context {
  my $x = shift;
  return 1 if type($x) eq 'context';
  return 0;
}

sub is_dot {
  my $x = shift;
  return 1 if type($x) eq 'dot';
  return 0;
}

sub is_else {
  my $x = shift;
  return 1 if is_sym($x) and value($x) eq 'else';
  return 0;
}

sub is_elsif {
  my $x = shift;
  return 1 if is_sym($x) and value($x) eq 'elsif';
  return 0;
}

sub is_fail {
  my $x = shift;
  return 1 if is_false($x) or is_nil($x);
  return 0;
}

sub is_false {
  my $x = shift;
  return 1 if type($x) eq 'false';
  return 0;
}

sub is_func {
  my $x = shift;
  return 1 if ref($x) eq ref(sub {});
  return 0;
}

sub is_hash {
  my $x = shift;
  return 1 if type($x) eq 'hash';
  return 0;
}

sub is_if {
  my $x = shift;
  return 1 if is_sym($x) and value($x) eq 'if';
  return 0;
}

sub is_in {
  my $x = shift;
  return 1 if is_sym($x) and value($x) eq 'in';
  return 0;
}

sub is_int {
  my $x = shift;
  if (is_str($x)) {
    return 0 if $x ^ $x;
    return 0 if $x eq '';
    return 1;
  }
  return 0;
}

sub is_lambda {
  my $x = shift;
  return 1 if type($x) eq 'lambda';
  return 0;
}

sub is_list {
  my $x = shift;
  return 1 if type($x) eq 'list';
  return 0;
}

sub is_match {
  my $x = shift;
  return 1 if is_true($x) or is_str($x);
  return 0 if is_fail($x);
  return 1 if is_match_atom($x) or is_match_atoms($x);
  my $data_str = to_str($x);
  error("error match data: $data_str");
}

sub is_match_atom {
   my $x = shift;
   return 0 unless is_perl_array($x);
   return 0 unless len($x) > 1;
   return 0 unless is_str(first($x));
   return 1;
}

sub is_match_atoms {
  my $pairs = shift;
  if ( is_perl_array($pairs) ) {
    for my $pair (values $pairs) {
      return 0 unless is_match_atom($pair);
    }
  }
  return 1;
}

sub is_nil {
  my $x = shift;
  return 1 if type($x) eq 'nil';
  return 0;
}

sub is_perl_array {
   my $x = shift;
   return 1 if ref($x) eq ref([]);
   return 0;
}

sub is_perl_hash {
   my $x = shift;
   return 1 if ref($x) eq ref({});
   return 0;
}

sub is_ref {
  my $x = shift;
  return 1 if type($x) eq 'ref';
  return 0;
}

sub is_rule {
  my $x = shift;
  return 1 if type($x) eq 'rule';
  return 0;
}

sub is_same {
  my ($data_one, $data_two) = @_;
  if (is_str($data_one) and is_str($data_two)) {
    return ($data_one eq $data_two);
  }
  return 1 if to_str($data_one) eq to_str($data_two);
  return 0;
}

sub is_str {
   my $x = shift;
   return 1 if ref($x) eq ref('');
   return 0;
}

sub is_sym {
  my $x = shift;
  return 1 if type($x) eq 'sym';
  return 0;
}

sub is_true {
  my $x = shift;
  return 1 if type($x) eq 'true';
  return 0;
}

sub is_when {
  my $x = shift;
  return 1 if is_sym($x) and value($x) eq 'when';
  return 0;
}

sub len {
  my $data = shift;
  return scalar( @{$data} ) if is_perl_array($data);
  return $data if is_int($data);
  return length( $data ) if is_str($data);
}

sub load_file {
  my $file_name = shift;
  my $file_txt = read_file($file_name);
  return decode_json($file_txt);
}

sub perl_join {
  my ($array, $sep) = @_;
  if (defined $sep) {
    return join($sep, @$array);
  }
  return join('', @$array);
}

sub perl_max {
  my $args = shift;
  return max(@{$args});
}

sub perl_split {
  my ($str, $sep) = @_;
  if (defined $sep) {
    return [ split($sep, $str) ];
  }
  return [ split('', $str) ];
}

sub perl_substr {
  my ($str,$from,$len) = @_;
  return substr($str, $from, $len);
}

sub perl_zip {
  my ($a_one, $a_two) = @_;
  return [ pairwise { [$a, $b] } @$a_one, @$a_two ];
}

sub read_file {
  my $file = shift;
  local $/;
  open my ($fh), '<', $file or die $!;
  return <$fh>;
}

sub rest {
   my $data = shift;
   my $len_data = len($data);
   if (is_perl_array($data)) {
     return [ splice( [ @{$data} ], 1, $len_data ) ];
   }
   return substr($data, 1) if is_str($data);
   error("rest only could implement with str or array");
}

sub second {
  my $data = shift;
  return $data->[1] if is_perl_array($data);
  return substr($data, 1, 1) if is_str($data);
}

sub see {
  my $data = shift;
  say to_str($data);
  return ['true'];
}

sub subarray {
  my ($array, $from, $to) = @_;
  if (is_perl_array($array)) {
    my $list = [ @{$array} ];
    if ($to < 0) {
      my $len = len($list) + $to - $from + 1;
      my $sub_list = [ splice $list, $from, $len ];
      return $sub_list;
    }
    return [ splice $list, $from, $to ];
  }
  my $array_str = to_str($array);
  error("subarray only could process array: not $array_str");
}

sub tail {
  my $data = shift;
  return $data->[-1] if is_perl_array($data);
  return substr($data, -1) if is_str($data);
}

sub to_str {
  my $data = shift;
  return $data if is_str($data);
  return encode_json($data);
}

sub trim {
  my $str = shift;
  if (is_str($str)) {
    $str =~ s/^\s+|\s+$//g;
    return $str;
  }
  my $str_json = to_str($str);
  error("trim only could make string, not $str_json");
}

sub type {
  my $x = shift;
  return first($x) if is_perl_array($x);
  return 'int' if is_int($x);
  return 'str' if is_str($x);
  my $x_str = to_str($x);
  error("Could not get $x_str type");
}

sub uuid { return scalar(rand()) }

sub value {
  my $x = shift;
  return $x->[1] if is_perl_array($x);
  return $x if is_str($x);
  my $x_str = to_str($x);
  error("Could not get $x_str value");
}

sub write_file {
  my ($file, $str) = @_;
  open my ($fh), '>', $file or die $!;
  print {$fh} $str;
  return ['true'];
}

=head1 AUTHOR

Michael Song, C<< <10435916 at qq.com> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-spp at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Spp>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Spp::Tools

You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Spp>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Spp>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Spp>

=item * Search CPAN

L<http://search.cpan.org/dist/Spp/>

=back

=head1 ACKNOWLEDGEMENTS

=head1 LICENSE AND COPYRIGHT

Copyright 2015 Michael Song.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=cut

1; # End of Spp::Tools
